/* -*- mode: c++; indent-tabs-mode: nil -*- */
/** @file ql_json.qpp defines JSON functions */
/*
    lib/ql_json.qpp

    Qore JSON (JavaScript QoreObject Notation) functions

    Qore Programming Language

    Copyright 2003 - 2019 Qore Technologies, s.r.o.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "qore-json-module.h"

#include "ql_json.h"

#include <ctype.h>
#include <stdlib.h>
#include <cmath>

// RFC 4627 JSON specification
// qore only supports JSON with UTF-8

// returns 0 for OK
static int cmp_rest_token(const char*& p, const char* tok) {
   p++;
   while (*tok)
      if ((*(p++)) != (*(tok++)))
         return -1;
   if (!*p || *p == ',' || *p == ']' || *p == '}')
      return 0;
   if (isblank(*p) || (*p) == '\r' || (*p) == '\n') {
      ++p;
      return 0;
   }
   return -1;
}

static void skip_whitespace(const char*& buf, int& line_number) {
   while (*buf) {
      if (isblank(*buf) || (*buf) == '\r') {
         ++buf;
         continue;
      }
      if ((*buf) == '\n') {
         ++line_number;
         ++buf;
         continue;
      }
      break;
   }
}

// '"' has already been read and the buffer is set to this character
static int get_json_string_token(QoreString& str, const char*& buf, int& line_number, ExceptionSink* xsink) {
   // increment buffer to first character of string
   buf++;
   while (*buf) {
      if (*buf == '"') {
         buf++;
         return 0;
      }
      if (*buf == '\\') {
         buf++;
         if (*buf == '"' || *buf == '/' || *buf == '\\') {
            str.concat(*buf);
            buf++;
            continue;
         }
         if (*buf == 'a')
            str.concat('\a');
         else if (*buf == 'b')
            str.concat('\b');
         else if (*buf == 'f')
            str.concat('\f');
         else if (*buf == 'n')
            str.concat('\n');
         else if (*buf == 'r')
            str.concat('\r');
         else if (*buf == 't')
            str.concat('\t');
         else if (*buf == 'v')
            str.concat('\v');
         else if (*buf == 'u') { // expect a unicode character specification
            ++buf;
            // check for 4 hex digits
            if (isxdigit(*buf) && isxdigit(*(buf + 1)) && isxdigit(*(buf + 2)) && isxdigit(*(buf + 3))) {
               char unicode[5];
               strncpy(unicode, buf, 4);
               unicode[4] = '\0';
               unsigned code = strtoul(unicode, 0, 16);
               if (str.concatUnicode(code, xsink))
                  break;
               buf += 3;
            }
            else
               str.concat("\\u");
         }
         else { // otherwise just concatenate the characters
            str.concat('\\');
            str.concat(*buf);
         }
         ++buf;
         continue;
      }
      if (*buf == '\n')
         line_number++;
      str.concat(*buf);
      ++buf;
   }
   xsink->raiseException("JSON-PARSE-ERROR", "premature end of input at line %d while parsing JSON string", line_number);
   return -1;
}

static QoreValue get_json_value(const char*& buf, int& line_number, const QoreEncoding* enc, bool& empty, ExceptionSink* xsink);

// '{' has already been read and the buffer is set to this character
static QoreHashNode* get_json_object(const char*& buf, int& line_number, const QoreEncoding* enc, ExceptionSink* xsink) {
    // increment buffer to first character of object description
    buf++;
    ReferenceHolder<QoreHashNode> h(new QoreHashNode(autoTypeInfo), xsink);

    // get either string or '}'
    skip_whitespace(buf, line_number);

    if (*buf == '}') {
        buf++;
        return h.release();
    }

    while (*buf) {
        if (*buf != '"') {
            //printd(5, "*buf='%c'\n", *buf);
            if (h->size())
                xsink->raiseException("JSON-PARSE-ERROR", "unexpected text encountered at line %d while parsing JSON object (expecting '\"' for key string)", line_number);
            else
                xsink->raiseException("JSON-PARSE-ERROR", "unexpected text encountered at line %d while parsing JSON object (expecting '\" or '}'')", line_number);
            break;
        }

        // get key
        QoreString str(enc);
        if (get_json_string_token(str, buf, line_number, xsink))
            break;

        //printd(5, "get_json_object() key=%s\n", str.getBuffer());

        skip_whitespace(buf, line_number);
        if (*buf != ':') {
            //printd(5, "*buf='%c'\n", *buf);
            xsink->raiseException("JSON-PARSE-ERROR", "unexpected text encountered at line %d while parsing JSON object (expecting ':')", line_number);
            break;
        }
        buf++;
        skip_whitespace(buf, line_number);

        // get value
        bool empty = false;
        QoreValue val = get_json_value(buf, line_number, enc, empty, xsink);
        if (*xsink) {
            break;
        }
        if (empty) {
            xsink->raiseException("JSON-PARSE-ERROR", "premature end of input at line %d while parsing JSON object (expecting JSON value)", line_number);
            break;
        }
        h->setKeyValue(&str, val, xsink);

        skip_whitespace(buf, line_number);
        if (*buf == '}') {
            buf++;
            return h.release();
        }

        if (*buf != ',') {
            xsink->raiseException("JSON-PARSE-ERROR", "unexpected text encountered at line %d while parsing JSON object (expecting ',' or '}')", line_number);
            break;
        }
        buf++;

        skip_whitespace(buf, line_number);
    }
    return 0;
}

// '[' has already been read and the buffer is set to this character
static AbstractQoreNode* get_json_array(const char*& buf, int& line_number, const QoreEncoding* enc, ExceptionSink* xsink) {
    // increment buffer to first character of array description
    buf++;
    ReferenceHolder<QoreListNode> l(new QoreListNode(autoTypeInfo), xsink);

    skip_whitespace(buf, line_number);
    if (*buf == ']') {
        ++buf;
        return l.release();
    }

    while (*buf) {
        //printd(5, "before get_json_value() buf=%s\n", buf);
        bool empty = false;
        QoreValue val = get_json_value(buf, line_number, enc, empty, xsink);
        if (*xsink) {
            break;
        }
        if (empty) {
            xsink->raiseException("JSON-PARSE-ERROR", "premature end of input at line %d while parsing JSON array (expecting JSON value)", line_number);
            break;
        }
        //printd(5, "after get_json_value() buf=%s\n", buf);
        l->push(val, xsink);

        skip_whitespace(buf, line_number);
        if (*buf == ']') {
            buf++;
            return l.release();
        }

        if (*buf != ',') {
            //printd(5, "*buf='%c'\n", *buf);
            xsink->raiseException("JSON-PARSE-ERROR", "unexpected text encountered at line %d while parsing JSON array (expecting ',' or ']')", line_number);
            return 0;
        }
        buf++;

        skip_whitespace(buf, line_number);
    }
    return 0;
}

static QoreValue get_json_value(const char*& buf, int& line_number, const QoreEncoding* enc, bool& empty, ExceptionSink* xsink) {
    assert(!empty);
    // skip whitespace
    skip_whitespace(buf, line_number);
    if (!*buf) {
        empty = true;
        return QoreValue();
    }

    // can expect: 't'rue, 'f'alse, '{', '[', '"'string...", integer, '.'digits
    if (*buf == '{')
        return get_json_object(buf, line_number, enc, xsink);

    if (*buf == '[')
        return get_json_array(buf, line_number, enc, xsink);

    if (*buf == '"') {
        QoreStringNodeHolder str(new QoreStringNode(enc));
        return get_json_string_token(*(*str), buf, line_number, xsink) ? 0 : str.release();
    }

    // FIXME: implement parsing of JSON exponents
    if (isdigit(*buf) || (*buf) == '.' || (*buf) == '-') {
        // temporarily use a QoreString
        QoreString str;
        bool has_dot;
        if (*buf == '.') {
            // add a leading zero
            str.concat("0.");
            has_dot = true;
        } else {
            str.concat(*buf);
            has_dot = false;
        }
        ++buf;
        bool has_e = false;
        while (*buf) {
            if (*buf == '.') {
                if (has_dot) {
                    xsink->raiseException("JSON-PARSE-ERROR", "unexpected '.' in floating point number (too many '.' characters)");
                    return 0;
                }
                has_dot = true;
            }
            // if another token follows then break but do not increment buffer position
            else if (*buf == ',' || *buf == '}' || *buf == ']')
                break;
            // if whitespace follows then increment buffer position and break
            else if (isblank(*buf) || (*buf) == '\r') {
                ++buf;
                break;
            }
            // if a newline follows then  increment buffer position and line number and break
            else if ((*buf) == '\n') {
                ++buf;
                ++line_number;
                break;
            } else if (*buf == 'e' || *buf == 'E') {
                if (has_e) {
                    xsink->raiseException("JSON-PARSE-ERROR", "unexpected second exponent marker '%c' in number", *buf);
                    return QoreValue();
                }
                has_e = true;
            } else if (!isdigit(*buf)) {
                xsink->raiseException("JSON-PARSE-ERROR", "unexpected character '%c' in number", *buf);
                return QoreValue();
            }
            str.concat(*buf);
            ++buf;
        }
        if (has_dot || has_e)
            return QoreValue(q_strtod(str.c_str()));
        return strtoll(str.c_str(), 0, 10);
    }

    if ((*buf) == 't') {
        if (!cmp_rest_token(buf, "rue"))
            return true;
    }
    else if ((*buf) == 'f') {
        if (!cmp_rest_token(buf, "alse"))
            return false;
    }
    else if ((*buf) == 'n') {
        if (!cmp_rest_token(buf, "ull"))
            return nothing();
    }
    //printd(5, "buf=%s\n", buf);

    xsink->raiseException("JSON-PARSE-ERROR", "invalid input at line %d; unable to parse JSON value", line_number);
    return QoreValue();
}

#define JSF_THRESHOLD 20

static int do_json_value(QoreString* str, QoreValue v, int format, ExceptionSink* xsink);

static int do_json_list(ExceptionSink* xsink, QoreString* str, const QoreListNode* l, int format, unsigned offset = 0) {
    assert(l);
    str->concat("[");
    ConstListIterator li(l);
    QoreString tmp(str->getEncoding());
    while (li.next()) {
        if (li.index() < offset)
            continue;
        bool ind = tmp.strlen() > JSF_THRESHOLD;
        tmp.clear();
        if (do_json_value(&tmp, li.getValue(), format == -1 ? format : format + 2, xsink))
            return -1;

        if (format != -1 && (ind || tmp.strlen() > JSF_THRESHOLD)) {
            str->concat('\n');
            str->addch(' ', format + 2);
        }
        str->sprintf("%s", tmp.getBuffer());

        if (!li.last())
            str->concat(",");
    }
    str->concat("]");
    return 0;
}

static int do_json_string_intern(ExceptionSink* xsink, QoreString* str, const char* utf8_str, int format) {
   str->concat('"');
   qore_size_t i = str->size();
   str->concatEscape(utf8_str, '"', '\\');

   // http://tools.ietf.org/html/rfc4627
   // encode all control characters in the string concatenated
   while (i < str->size()) {
      // see if we have a single-byte character; do not try to look at bytes within multi-byte characters
      qore_size_t cl = q_get_char_len(str->getEncoding(), str->getBuffer() + i, str->size() - i, xsink);
      if (*xsink)
         return -1;
      if (cl > 1) {
         i += cl;
         continue;
      }

      unsigned char c = (*str)[i];
      if (c < 32) {
         switch (c) {
            case 7: str->replace(i, 1, "\\a"); ++i; break; // BEL
            case 8: str->replace(i, 1, "\\b"); ++i; break; // BS
            case 9: str->replace(i, 1, "\\t"); ++i; break; // HT
            case 10: str->replace(i, 1, "\\n"); ++i; break; // LF
            case 11: str->replace(i, 1, "\\v"); ++i; break; // VT
            case 12: str->replace(i, 1, "\\f"); ++i; break; // FF
            case 13: str->replace(i, 1, "\\r"); ++i; break; // CR
            default: {
               char buf[7];
               sprintf(buf, "\\u%04x", (int)c);
               str->replace(i, 1, buf); i += 5;
               break;
            }
         }
      }
      ++i;
   }

   str->concat('"');
   return 0;
}

static int do_json_string(ExceptionSink* xsink, QoreString* str, const QoreString* vstr, int format) {
   TempEncodingHelper t(vstr, str->getEncoding(), xsink);
   if (*xsink)
      return -1;

   return do_json_string_intern(xsink, str, t->c_str(), format);
}

static int do_json_value(QoreString* str, QoreValue v, int format, ExceptionSink* xsink) {
    if (v.isNullOrNothing()) {
        str->concat("null");
        return 0;
    }

    qore_type_t vtype = v.getType();

    if (vtype == NT_LIST)
        return do_json_list(xsink, str, v.get<const QoreListNode>(), format);

    if (vtype == NT_HASH) {
        const QoreHashNode* h = v.get<const QoreHashNode>();
        str->concat("{");
        ConstHashIterator hi(h);
        QoreString tmp(str->getEncoding());
        while (hi.next()) {
            bool ind = tmp.strlen() > JSF_THRESHOLD;
            tmp.clear();
            if (do_json_value(&tmp, hi.get(), format == -1 ? format : format + 2, xsink))
                return -1;

            if (format != -1 && (ind || tmp.strlen() > JSF_THRESHOLD)) {
                str->concat('\n');
                str->addch(' ', format + 2);
            }

            QoreString tmpkey(hi.getKey());
            if (do_json_string(xsink, str, &tmpkey, format))
                return -1;
            str->concat(": ");
            str->concat(tmp.c_str(), tmp.size());
            //str->sprintf("\"%s\": %s", hi.getKey(), tmp.getBuffer());
            if (!hi.last())
                str->concat(",");

        }
        str->concat("}");
        return 0;
    }

    if (vtype == NT_STRING)
        return do_json_string(xsink, str, v.get<const QoreStringNode>(), format);

    if (vtype == NT_INT) {
        str->sprintf("%lld", v.getAsBigInt());
        return 0;
    }

    if (vtype == NT_FLOAT) {
        double f = v.getAsFloat();
        // check for nan, +/-inf and serialize as null
        if (std::isnan(f) || std::isinf(f))
            str->concat("null");
        else {
            str->sprintf("%.25g", f);
            // apply noise reduction algorithm
            qore_apply_rounding_heuristic(*str, 6, 8);
        }
        return 0;
    }

    if (vtype == NT_NUMBER) {
        const QoreNumberNode* n = v.get<const QoreNumberNode>();
        if (!n->ordinary())
            str->concat("null");
        else
            n->getStringRepresentation(*str);
        return 0;
    }

    if (vtype == NT_BOOLEAN) {
        str->concat(v.getAsBool() ? "true" : "false");
        return 0;
    }

    if (vtype == NT_DATE) {
        const DateTimeNode* date = v.get<const DateTimeNode>();
        // ensure that all date/time values are reported in the current time zone
        // for simplicity's sake (particularly because json does not have a native date/time format)
        qore_tm info;
        date->getInfo(currentTZ(), info);

        // this will be serialized as a string
        str->concat('"');
        if (date->isRelative()) {
            str->concat('P');
            if (date->hasValue()) {
                if (info.year) {
                    str->sprintf("%dY", info.year);
                }
                if (info.month) {
                    str->sprintf("%dM", info.month);
                }
                if (info.day) {
                    str->sprintf("%dD", info.day);
                }

                bool has_t = false;
                if (info.hour) {
                    str->sprintf("T%dH", info.hour);
                    has_t = true;
                }
                if (info.minute) {
                    if (!has_t) {
                        str->concat('T');
                        has_t = true;
                    }
                    str->sprintf("%dM", info.minute);
                }
                if (info.second) {
                    if (!has_t) {
                        str->concat('T');
                        has_t = true;
                    }
                    str->sprintf("%dS", info.second);
                }
                if (info.us) {
                    if (!has_t) {
                        str->concat('T');
                        has_t = true;
                    }
                    str->sprintf("%du", info.us);
                }
            } else {
                str->concat("0D");
            }
        } else {
#ifndef SECS_PER_MINUTE
#define SECS_PER_MINUTE          60
#endif
#ifndef SECS_PER_HOUR
#define SECS_PER_HOUR            (SECS_PER_MINUTE * 60)
#endif
            // issue #2655 include a 'T' between the date and time as per ISO-8601
            str->sprintf("%04d-%02d-%02dT%02d:%02d:%02d.%06d", info.year, info.month, info.day, info.hour, info.minute, info.second, info.us);
            if (!info.utc_secs_east) {
                str->concat('Z');
            } else {
                // issue #2655 do not include a space before the UTC offset
                str->concat(info.utc_secs_east < 0 ? '-' : '+');
                if (info.utc_secs_east < 0)
                    info.utc_secs_east = -info.utc_secs_east;
                int h = info.utc_secs_east / SECS_PER_HOUR;
                // the remaining seconds after hours
                int r = info.utc_secs_east % SECS_PER_HOUR;
                // minutes
                int m = r / SECS_PER_MINUTE;
                // we have already output the hour sign above
                str->sprintf("%02d:%02d", h < 0 ? -h : h, m);
                // see if there are any seconds
                int s = info.utc_secs_east - h * SECS_PER_HOUR - m * SECS_PER_MINUTE;
                if (s) {
                    str->sprintf(":%02d", s);
                }
            }
        }
        str->concat('"');
        return 0;
    }

    if (vtype == NT_BINARY) {
        str->concat('\"');
        str->concatBase64(v.get<const BinaryNode>());
        str->concat('\"');
        return 0;
    }

    xsink->raiseException("JSON-SERIALIZATION-ERROR", "don't know how to serialize type '%s'", v.getTypeName());
    return -1;
}

QoreStringNode* make_jsonrpc_request_args(const std::string& version, const QoreListNode* args, QoreValue id, unsigned offset, ExceptionSink* xsink) {
   const QoreStringNode* p0 = HARD_QORE_VALUE_STRING(args, offset);

   const QoreValue p1 = get_param_value(args, offset + 1);

   QoreStringNodeHolder str(new QoreStringNode(QCS_UTF8));

   // write version key first
   if (version == "1.0")
      str->concat("{\"method\": ");
   else
      str->sprintf("{\"%s\": \"%s\",\"method\": ", version[0] == '1' ? "version" : "jsonrpc", version.c_str());
   if (do_json_value(*str, p0, -1, xsink))
      return 0;

   if (!id.isNullOrNothing()) {
      str->concat(",\"id\": ");
      if (do_json_value(*str, id, -1, xsink))
         return 0;
   }

   // params key should come last
   str->concat(",\"params\": ");
   if (!p1.isNullOrNothing()) {
      if (do_json_value(*str, p1, -1, xsink))
         return 0;
   }
   else
      str->concat("null");
   str->concat("}");
   return str.release();
}

QoreStringNode* make_jsonrpc_request(const std::string& version, const QoreListNode* args, QoreValue id, unsigned offset, ExceptionSink* xsink) {
   const QoreStringNode* p0 = HARD_QORE_VALUE_STRING(args, offset);

   QoreStringNodeHolder str(new QoreStringNode(QCS_UTF8));

   // write version key first
   if (version == "1.0")
      str->concat("{\"method\": ");
   else
      str->sprintf("{\"%s\": \"%s\",\"method\": ", version[0] == '1' ? "version" : "jsonrpc", version.c_str());
   if (do_json_value(*str, p0, -1, xsink))
      return 0;

   if (!id.isNullOrNothing()) {
      str->concat(",\"id\": ");
      if (do_json_value(*str, id, -1, xsink))
         return 0;
   }

   // params key should come last
   str->concat(",\"params\" : ");
   if (num_args(args) > (offset + 1)) {
      if (do_json_list(xsink, *str, args, 0, offset + 1))
         return 0;
   }
   else
      str->concat("null");
   str->concat("}");

   return str.release();
}

QoreValue parse_json(const QoreString* str, ExceptionSink* xsink) {
    int line_number = 1;
    const char* buf = str->getBuffer();
    const unsigned char* data = reinterpret_cast<const unsigned char*>(buf);
    if (str->size() >= 3 && data[0] == 0xEF && data[1] == 0xBB && data[2] == 0xBF) { // Skip UTF-8 BOM.
        buf = reinterpret_cast<const char*>(data+3);
    }
    else if (str->size() >= 2) { // Skip Unicode BOM.
        if ((data[0] == 0xFE && data[1] == 0xFF) || (data[0] == 0xFF && data[1] == 0xFE)) {
            buf = reinterpret_cast<const char*>(data+2);
        }
    }

    // ignore empty values; return NOTHIN in this case
    bool empty = false;
    ValueHolder rv(get_json_value(buf, line_number, str->getEncoding(), empty, xsink), xsink);
    if (rv && *buf) {
        // check for excess text after JSON data
        skip_whitespace(buf, line_number);
        if (*buf) {
            xsink->raiseException("JSON-PARSE-ERROR", "extra text after JSON data on line %d", line_number);
            return QoreValue();
        }
    }
    return rv.release();
}

// for JSON-RPC 1.1, version_name = "version"; for 2.0, version_name = "jsonrpc"
static QoreStringNode* make_jsonrpc_request(const QoreStringNode* method_name, const char* version_name, const char* version, QoreValue id, QoreValue request_msg, int format, ExceptionSink* xsink) {
   QoreStringNodeHolder str(new QoreStringNode(QCS_UTF8));

   bool fmt = format & JGF_ADD_FORMATTING;

   // write version key first if present
   if (version && *version) {
      str->sprintf("{%s\"%s\" : ", fmt ? "\n  " : "", version_name);
      if (do_json_string_intern(xsink, *str, version, fmt ? 2 : -1))
         return 0;
      str->concat(fmt ? ",\n  " : ",");
   }
   else
      str->concat(fmt ? "{\n  " : "{");

   str->concat("\"method\": ");
   if (do_json_value(*str, method_name, fmt ? 2 : -1, xsink))
      return 0;

   if (!id.isNullOrNothing()) {
      str->sprintf(",%s\"id\": ", fmt ? "\n  " : "");
      if (do_json_value(*str, id, fmt ? 2 : -1, xsink))
         return 0;
   }

   // params key should come last
   str->sprintf(",%s\"params\": ", fmt ? "\n  " : "");
   if (do_json_value(*str, request_msg, fmt ? 2 : -1, xsink))
      return 0;

   str->concat(fmt ? "\n}" : "}");
   return str.release();
}

// for JSON-RPC 1.1, version_name = "version"; for 2.0, version_name = "jsonrpc"
static QoreStringNode* make_jsonrpc_response(ExceptionSink* xsink, const char* version_name, const char* version, QoreValue id, QoreValue msg, const char* type = "result", int format = 0) {
   QoreStringNodeHolder str(new QoreStringNode(QCS_UTF8));

   bool fmt = format & JGF_ADD_FORMATTING;

   // write version key first if present
   if (version && *version) {
      str->sprintf("{%s\"%s\" : ", fmt ? "\n  " : "", version_name);
      if (do_json_string_intern(xsink, *str, version, fmt ? 2 : -1))
         return 0;
      str->concat(fmt ? ",\n  " : ",");
   }
   else
      str->concat(fmt ? "{\n  " : "{");

   if (!id.isNullOrNothing()) {
      str->concat("\"id\": ");
      if (do_json_value(*str, id, fmt ? 2 : -1, xsink))
         return 0;
      str->concat(fmt ? ",\n  " : ",");
   }

   // result/error key should come last
   str->sprintf("\"%s\": ", type);
   if (do_json_value(*str, msg, fmt ? 2 : -1, xsink))
      return 0;

   str->concat(fmt ? "\n}" : "}");
   return str.release();
}

static QoreStringNode* make_jsonrpc11_error(int64 code, const QoreStringNode* error, QoreValue id, QoreValue error_msg, int format, ExceptionSink* xsink) {
   if (code < 100 || code > 999) {
      xsink->raiseException("MAKE-JSONRPC11-ERROR-STRING-ERROR", "error code (first argument) must be between 100 and 999 inclusive (value passed: " QLLD ")", code);
      return 0;
   }

   if (error->empty()) {
      xsink->raiseException("MAKE-JSONRPC11-ERROR-STRING-ERROR", "empty error message string passed as second argument)");
      return 0;
   }

   QoreStringNodeHolder str(new QoreStringNode(QCS_UTF8));

   bool fmt = format & JGF_ADD_FORMATTING;

   str->concat(fmt ? "{\n  \"version\": \"1.1\",\n  " : "{\"version\": \"1.1\",");

   // get optional "id" value
   if (!id.isNullOrNothing()) {
      str->concat("\"id\": ");
      if (do_json_value(*str, id, fmt ? 2 : -1, xsink))
         return 0;
      str->concat(fmt ? ",\n  " : ",");
   }

   if (fmt)
      str->sprintf("\"error\":\n  {\n    \"name\": \"JSONRPCError\",\n    \"code\": %d,\n    \"message\": \"", (int)code);
   else
      str->sprintf("\"error\": {\"name\": \"JSONRPCError\",\"code\": %d,\"message\": \"", (int)code);
   // concat here so character encodings can be automatically converted if necessary
   str->concatEscape(error, '"', '\\', xsink);
   if (*xsink)
      return 0;

   str->concat('\"');

   // get optional "error" value
   if (!error_msg.isNullOrNothing()) {
      str->sprintf(",%s\"error\": ", fmt ? "\n   " : "");
      if (do_json_value(*str, error_msg, fmt ? 4 : -1, xsink))
         return 0;
   }
   str->concat(fmt ? "\n  }\n}" : "}}");
   return str.release();
}
# 808 "ql_json.qpp"
# 813 "ql_json.qpp"
# 840 "ql_json.qpp"
# 859 "ql_json.qpp"
# 878 "ql_json.qpp"
# 896 "ql_json.qpp"
# 902 "ql_json.qpp"
// string makeFormattedJSONString(any data, *string encoding) {}
static QoreValue f_makeFormattedJSONString_VaNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue data = get_param_value(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 872 "ql_json.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_UTF8;
   QoreStringNodeHolder str(new QoreStringNode(qe));
   return do_json_value(*str, data, 0, xsink) ? 0 : str.release();
}

// string makeJSONString(any data, *string encoding) {}
static QoreValue f_makeJSONString_VaNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue data = get_param_value(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 853 "ql_json.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_UTF8;
   QoreStringNodeHolder str(new QoreStringNode(qe));
   return do_json_value(*str, data, -1, xsink) ? 0 : str.release();
}

// string make_json(any data, *int format, *string encoding) {}
static QoreValue f_make_json_VaNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue data = get_param_value(args, 0);
    int64 format = HARD_QORE_VALUE_INT(args, 1);
    const QoreStringNode* encoding = get_param_value(args, 2).get<const QoreStringNode>();
# 834 "ql_json.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_UTF8;
   QoreStringNodeHolder str(new QoreStringNode(qe));
   return do_json_value(*str, data, format & JGF_ADD_FORMATTING ? 0 : -1, xsink) ? 0 : str.release();
}

// nothing parseJSON() {}
static QoreValue f_parseJSON(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 899 "ql_json.qpp"

    return QoreValue();
}

// auto parseJSON(string json_str) {}
static QoreValue f_parseJSON_Vs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* json_str = HARD_QORE_VALUE_STRING(args, 0);
# 913 "ql_json.qpp"
   return parse_json(json_str, xsink);
}

// auto parse_json(string json_str) {}
static QoreValue f_parse_json_Vs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* json_str = HARD_QORE_VALUE_STRING(args, 0);
# 892 "ql_json.qpp"
   return parse_json(json_str, xsink);
}

# 918 "ql_json.qpp"
# 958 "ql_json.qpp"
# 990 "ql_json.qpp"
# 1022 "ql_json.qpp"
# 1058 "ql_json.qpp"
# 1088 "ql_json.qpp"
# 1118 "ql_json.qpp"
# 1154 "ql_json.qpp"
# 1184 "ql_json.qpp"
# 1214 "ql_json.qpp"
# 1241 "ql_json.qpp"
# 1262 "ql_json.qpp"
// string makeFormattedJSONRPC11ErrorString(softint code, string error, any id, any error_msg) {}
static QoreValue f_makeFormattedJSONRPC11ErrorString_viVsVaVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    int64 code = HARD_QORE_VALUE_INT(args, 0);
    const QoreStringNode* error = HARD_QORE_VALUE_STRING(args, 1);
    QoreValue id = get_param_value(args, 2);
    QoreValue error_msg = get_param_value(args, 3);
# 1279 "ql_json.qpp"
   return make_jsonrpc11_error(code, error, id, error_msg, JGF_ADD_FORMATTING, xsink);
}

// string makeFormattedJSONRPCErrorString(any version, any id, any error_msg) {}
static QoreValue f_makeFormattedJSONRPCErrorString_VaVaVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue version = get_param_value(args, 0);
    QoreValue id = get_param_value(args, 1);
    QoreValue error_msg = get_param_value(args, 2);
# 1201 "ql_json.qpp"
   if (!version.isNullOrNothing()) {
      QoreStringValueHelper vstr(version, QCS_UTF8, xsink);
      if (*xsink)
         return QoreValue();

      if (**vstr != "1.0")
         return make_jsonrpc_response(xsink, (**vstr)[0] == '1' ? "version" : "jsonrpc", vstr->c_str(), id, error_msg, "error", JGF_ADD_FORMATTING);
   }

   return make_jsonrpc_response(xsink, nullptr, nullptr, id, error_msg, "error", JGF_ADD_FORMATTING);
}

// string makeFormattedJSONRPCRequestString(string method_name, any version, any id, any request_msg) {}
static QoreValue f_makeFormattedJSONRPCRequestString_VsVaVaVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* method_name = HARD_QORE_VALUE_STRING(args, 0);
    QoreValue version = get_param_value(args, 1);
    QoreValue id = get_param_value(args, 2);
    QoreValue request_msg = get_param_value(args, 3);
# 1009 "ql_json.qpp"
   if (!version.isNullOrNothing()) {
      QoreStringValueHelper vstr(version, QCS_UTF8, xsink);
      if (*xsink)
         return QoreValue();

      if (**vstr != "1.0")
         return make_jsonrpc_request(method_name, (**vstr)[0] == '1' ? "version" : "jsonrpc", vstr->c_str(), id, request_msg, JGF_ADD_FORMATTING, xsink);
   }

   return make_jsonrpc_request(method_name, nullptr, nullptr, id, request_msg, JGF_ADD_FORMATTING, xsink);
}

// string makeFormattedJSONRPCResponseString(any version, any id, any response_msg) {}
static QoreValue f_makeFormattedJSONRPCResponseString_VaVaVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue version = get_param_value(args, 0);
    QoreValue id = get_param_value(args, 1);
    QoreValue response_msg = get_param_value(args, 2);
# 1105 "ql_json.qpp"
   if (!version.isNullOrNothing()) {
      QoreStringValueHelper vstr(version, QCS_UTF8, xsink);
      if (*xsink)
         return QoreValue();

      if (**vstr != "1.0")
         return make_jsonrpc_response(xsink, (**vstr)[0] == '1' ? "version" : "jsonrpc", vstr->c_str(), id, response_msg, "result", JGF_ADD_FORMATTING);
   }

   return make_jsonrpc_response(xsink, nullptr, nullptr, id, response_msg, "result", JGF_ADD_FORMATTING);
}

// string makeJSONRPC11ErrorString(softint code, string error, any id, any error_msg) {}
static QoreValue f_makeJSONRPC11ErrorString_viVsVaVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    int64 code = HARD_QORE_VALUE_INT(args, 0);
    const QoreStringNode* error = HARD_QORE_VALUE_STRING(args, 1);
    QoreValue id = get_param_value(args, 2);
    QoreValue error_msg = get_param_value(args, 3);
# 1258 "ql_json.qpp"
   return make_jsonrpc11_error(code, error, id, error_msg, 0, xsink);
}

// string makeJSONRPCErrorString(any version, any id, any error_msg) {}
static QoreValue f_makeJSONRPCErrorString_VaVaVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue version = get_param_value(args, 0);
    QoreValue id = get_param_value(args, 1);
    QoreValue error_msg = get_param_value(args, 2);
# 1171 "ql_json.qpp"
   if (!version.isNullOrNothing()) {
      QoreStringValueHelper vstr(version, QCS_UTF8, xsink);
      if (*xsink)
         return QoreValue();

      if (**vstr != "1.0")
         return make_jsonrpc_response(xsink, (**vstr)[0] == '1' ? "version" : "jsonrpc", vstr->c_str(), id, error_msg, "error", 0);
   }

   return make_jsonrpc_response(xsink, nullptr, nullptr, id, error_msg, "error", 0);
}

// string makeJSONRPCRequestString(string method_name, any version, any id, any request_msg) {}
static QoreValue f_makeJSONRPCRequestString_VsVaVaVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* method_name = HARD_QORE_VALUE_STRING(args, 0);
    QoreValue version = get_param_value(args, 1);
    QoreValue id = get_param_value(args, 2);
    QoreValue request_msg = get_param_value(args, 3);
# 977 "ql_json.qpp"
   if (!version.isNullOrNothing()) {
      QoreStringValueHelper vstr(version, QCS_UTF8, xsink);
      if (*xsink)
         return QoreValue();

      if (**vstr != "1.0")
         return make_jsonrpc_request(method_name, (**vstr)[0] == '1' ? "version" : "jsonrpc", vstr->c_str(), id, request_msg, JGF_NONE, xsink);
   }

   return make_jsonrpc_request(method_name, nullptr, nullptr, id, request_msg, JGF_NONE, xsink);
}

// string makeJSONRPCResponseString(any version, any id, any response_msg) {}
static QoreValue f_makeJSONRPCResponseString_VaVaVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue version = get_param_value(args, 0);
    QoreValue id = get_param_value(args, 1);
    QoreValue response_msg = get_param_value(args, 2);
# 1075 "ql_json.qpp"
   if (!version.isNullOrNothing()) {
      QoreStringValueHelper vstr(version, QCS_UTF8, xsink);
      if (*xsink)
         return QoreValue();

      if (**vstr != "1.0")
         return make_jsonrpc_response(xsink, (**vstr)[0] == '1' ? "version" : "jsonrpc", vstr->c_str(), id, response_msg, "result", 0);
   }

   return make_jsonrpc_response(xsink, nullptr, nullptr, id, response_msg, "result", 0);
}

// string make_jsonrpc11_error(softint code, string error, any id, any error_msg, *int format) {}
static QoreValue f_make_jsonrpc11_error_viVsVaVaNi(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    int64 code = HARD_QORE_VALUE_INT(args, 0);
    const QoreStringNode* error = HARD_QORE_VALUE_STRING(args, 1);
    QoreValue id = get_param_value(args, 2);
    QoreValue error_msg = get_param_value(args, 3);
    int64 format = HARD_QORE_VALUE_INT(args, 4);
# 1237 "ql_json.qpp"
   return make_jsonrpc11_error(code, error, id, error_msg, format, xsink);
}

// string make_jsonrpc_error(any version, any id, any error_msg, *int format) {}
static QoreValue f_make_jsonrpc_error_VaVaVaNi(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue version = get_param_value(args, 0);
    QoreValue id = get_param_value(args, 1);
    QoreValue error_msg = get_param_value(args, 2);
    int64 format = HARD_QORE_VALUE_INT(args, 3);
# 1141 "ql_json.qpp"
   if (!version.isNullOrNothing()) {
      QoreStringValueHelper vstr(version, QCS_UTF8, xsink);
      if (*xsink)
         return QoreValue();

      if (**vstr != "1.0")
         return make_jsonrpc_response(xsink, (**vstr)[0] == '1' ? "version" : "jsonrpc", vstr->c_str(), id, error_msg, "error", format);
   }

   return make_jsonrpc_response(xsink, nullptr, nullptr, id, error_msg, "error", format);
}

// string make_jsonrpc_request(string method_name, any version, any id, any request_msg, *int format) {}
static QoreValue f_make_jsonrpc_request_VsVaVaVaNi(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* method_name = HARD_QORE_VALUE_STRING(args, 0);
    QoreValue version = get_param_value(args, 1);
    QoreValue id = get_param_value(args, 2);
    QoreValue request_msg = get_param_value(args, 3);
    int64 format = HARD_QORE_VALUE_INT(args, 4);
# 946 "ql_json.qpp"
   if (!version.isNullOrNothing()) {
      QoreStringValueHelper vstr(version, QCS_UTF8, xsink);
      if (*xsink)
         return QoreValue();
      if (**vstr != "1.0")
         return make_jsonrpc_request(method_name, (**vstr)[0] == '1' ? "version" : "jsonrpc", vstr->c_str(), id, request_msg, format, xsink);
   }

   return make_jsonrpc_request(method_name, nullptr, nullptr, id, request_msg, format, xsink);
}

// string make_jsonrpc_response(any version, any id, any response_msg, *int format) {}
static QoreValue f_make_jsonrpc_response_VaVaVaNi(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue version = get_param_value(args, 0);
    QoreValue id = get_param_value(args, 1);
    QoreValue response_msg = get_param_value(args, 2);
    int64 format = HARD_QORE_VALUE_INT(args, 3);
# 1045 "ql_json.qpp"
   if (!version.isNullOrNothing()) {
      QoreStringValueHelper vstr(version, QCS_UTF8, xsink);
      if (*xsink)
         return QoreValue();

      if (**vstr != "1.0")
         return make_jsonrpc_response(xsink, (**vstr)[0] == '1' ? "version" : "jsonrpc", vstr->c_str(), id, response_msg, "result", format);
   }

   return make_jsonrpc_response(xsink, nullptr, nullptr, id, response_msg, "result", format);
}


DLLLOCAL void init_json_functions(QoreNamespace& ns) {
    // string makeFormattedJSONString(any data, *string encoding) {}
    ns.addBuiltinVariant("makeFormattedJSONString", (q_func_n_t)f_makeFormattedJSONString_VaNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 2, anyTypeInfo, QORE_PARAM_NO_ARG, "data", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string makeJSONString(any data, *string encoding) {}
    ns.addBuiltinVariant("makeJSONString", (q_func_n_t)f_makeJSONString_VaNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 2, anyTypeInfo, QORE_PARAM_NO_ARG, "data", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string make_json(any data, *int format, *string encoding) {}
    ns.addBuiltinVariant("make_json", (q_func_n_t)f_make_json_VaNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo, 3, anyTypeInfo, QORE_PARAM_NO_ARG, "data", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "format", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // nothing parseJSON() {}
    ns.addBuiltinVariant("parseJSON", (q_func_n_t)f_parseJSON, QCF_RUNTIME_NOOP|QCF_DEPRECATED, QDOM_DEFAULT, nothingTypeInfo);
    // auto parseJSON(string json_str) {}
    ns.addBuiltinVariant("parseJSON", (q_func_n_t)f_parseJSON_Vs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, autoTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "json_str");
    // auto parse_json(string json_str) {}
    ns.addBuiltinVariant("parse_json", (q_func_n_t)f_parse_json_Vs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, autoTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "json_str");
    // string makeFormattedJSONRPC11ErrorString(softint code, string error, any id, any error_msg) {}
    ns.addBuiltinVariant("makeFormattedJSONRPC11ErrorString", (q_func_n_t)f_makeFormattedJSONRPC11ErrorString_viVsVaVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 4, softBigIntTypeInfo, QORE_PARAM_NO_ARG, "code", stringTypeInfo, QORE_PARAM_NO_ARG, "error", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "error_msg");
    // string makeFormattedJSONRPCErrorString(any version, any id, any error_msg) {}
    ns.addBuiltinVariant("makeFormattedJSONRPCErrorString", (q_func_n_t)f_makeFormattedJSONRPCErrorString_VaVaVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 3, anyTypeInfo, QORE_PARAM_NO_ARG, "version", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "error_msg");
    // string makeFormattedJSONRPCRequestString(string method_name, any version, any id, any request_msg) {}
    ns.addBuiltinVariant("makeFormattedJSONRPCRequestString", (q_func_n_t)f_makeFormattedJSONRPCRequestString_VsVaVaVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 4, stringTypeInfo, QORE_PARAM_NO_ARG, "method_name", anyTypeInfo, QORE_PARAM_NO_ARG, "version", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "request_msg");
    // string makeFormattedJSONRPCResponseString(any version, any id, any response_msg) {}
    ns.addBuiltinVariant("makeFormattedJSONRPCResponseString", (q_func_n_t)f_makeFormattedJSONRPCResponseString_VaVaVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 3, anyTypeInfo, QORE_PARAM_NO_ARG, "version", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "response_msg");
    // string makeJSONRPC11ErrorString(softint code, string error, any id, any error_msg) {}
    ns.addBuiltinVariant("makeJSONRPC11ErrorString", (q_func_n_t)f_makeJSONRPC11ErrorString_viVsVaVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 4, softBigIntTypeInfo, QORE_PARAM_NO_ARG, "code", stringTypeInfo, QORE_PARAM_NO_ARG, "error", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "error_msg");
    // string makeJSONRPCErrorString(any version, any id, any error_msg) {}
    ns.addBuiltinVariant("makeJSONRPCErrorString", (q_func_n_t)f_makeJSONRPCErrorString_VaVaVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 3, anyTypeInfo, QORE_PARAM_NO_ARG, "version", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "error_msg");
    // string makeJSONRPCRequestString(string method_name, any version, any id, any request_msg) {}
    ns.addBuiltinVariant("makeJSONRPCRequestString", (q_func_n_t)f_makeJSONRPCRequestString_VsVaVaVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 4, stringTypeInfo, QORE_PARAM_NO_ARG, "method_name", anyTypeInfo, QORE_PARAM_NO_ARG, "version", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "request_msg");
    // string makeJSONRPCResponseString(any version, any id, any response_msg) {}
    ns.addBuiltinVariant("makeJSONRPCResponseString", (q_func_n_t)f_makeJSONRPCResponseString_VaVaVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 3, anyTypeInfo, QORE_PARAM_NO_ARG, "version", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "response_msg");
    // string make_jsonrpc11_error(softint code, string error, any id, any error_msg, *int format) {}
    ns.addBuiltinVariant("make_jsonrpc11_error", (q_func_n_t)f_make_jsonrpc11_error_viVsVaVaNi, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 5, softBigIntTypeInfo, QORE_PARAM_NO_ARG, "code", stringTypeInfo, QORE_PARAM_NO_ARG, "error", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "error_msg", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "format");
    // string make_jsonrpc_error(any version, any id, any error_msg, *int format) {}
    ns.addBuiltinVariant("make_jsonrpc_error", (q_func_n_t)f_make_jsonrpc_error_VaVaVaNi, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo, 4, anyTypeInfo, QORE_PARAM_NO_ARG, "version", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "error_msg", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "format");
    // string make_jsonrpc_request(string method_name, any version, any id, any request_msg, *int format) {}
    ns.addBuiltinVariant("make_jsonrpc_request", (q_func_n_t)f_make_jsonrpc_request_VsVaVaVaNi, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo, 5, stringTypeInfo, QORE_PARAM_NO_ARG, "method_name", anyTypeInfo, QORE_PARAM_NO_ARG, "version", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "request_msg", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "format");
    // string make_jsonrpc_response(any version, any id, any response_msg, *int format) {}
    ns.addBuiltinVariant("make_jsonrpc_response", (q_func_n_t)f_make_jsonrpc_response_VaVaVaNi, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo, 4, anyTypeInfo, QORE_PARAM_NO_ARG, "version", anyTypeInfo, QORE_PARAM_NO_ARG, "id", anyTypeInfo, QORE_PARAM_NO_ARG, "response_msg", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "format");

}

DLLLOCAL void init_json_constants(QoreNamespace& ns) {
# 795 "ql_json.qpp"
# 801 "ql_json.qpp"
    ns.addConstant("JGF_ADD_FORMATTING", ((int64)JGF_ADD_FORMATTING));
    ns.addConstant("JGF_NONE", ((int64)JGF_NONE));
# 808 "ql_json.qpp"
# 813 "ql_json.qpp"
# 840 "ql_json.qpp"
# 859 "ql_json.qpp"
# 878 "ql_json.qpp"
# 896 "ql_json.qpp"
# 902 "ql_json.qpp"
# 918 "ql_json.qpp"
# 958 "ql_json.qpp"
# 990 "ql_json.qpp"
# 1022 "ql_json.qpp"
# 1058 "ql_json.qpp"
# 1088 "ql_json.qpp"
# 1118 "ql_json.qpp"
# 1154 "ql_json.qpp"
# 1184 "ql_json.qpp"
# 1214 "ql_json.qpp"
# 1241 "ql_json.qpp"
# 1262 "ql_json.qpp"

}
