#!/usr/bin/env python

# Copyright (C) Duncan Macleod (2014)
#
# This file is part of GWpy.
#
# GWpy is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# GWpy is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with GWpy.  If not, see <http://www.gnu.org/licenses/>.

"""Plotting an `EventTable` in a histogram

I would like to study the snr distribution of event triggers generated by
the ExcessPower gravitational-wave burst detection algorithm,
over a small stretch of data.

The data from which these events were generated contain a simulated
gravitational-wave signal, or hardware injection, used to validate
the performance of the LIGO detectors and downstream data analysis procedures.
"""

__author__ = "Duncan Macleod <duncan.macleod@ligo.org>"
__currentmodule__ = 'gwpy.table'

# First, we import the `EventTable` object and read in a set of events from
# a LIGO_LW-format XML file containing a
# :class:`sngl_burst <glue.ligolw.lsctables.SnglBurstTable>` table
from gwpy.table import EventTable
events = EventTable.read(
    'H1-LDAS_STRAIN-968654552-10.xml.gz', tablename='sngl_burst',
    columns=['peak', 'snr'])

# .. note::
#
#    Here we manually specify the `columns` to read in order to optimise
#    the `read()` operation to parse only the data we actually need.

# and can generate a new `~gwpy.plotter.HistogramPlot` using the
# :meth:`~EventTable.hist` instance method using `weights=1/10.`
# to convert the counts from the histogram into a rate in Hertz

plot = events.hist('snr', weights=1/10., logbins=True,
                   bins=50, histtype='stepfilled')
ax = plot.gca()
ax.set_xlabel('Signal-to-noise ratio (SNR)')
ax.set_ylabel('Rate [Hz]')
ax.set_title('LHO event triggers for HW100916')
ax.autoscale(axis='x', tight=True)
plot.show()
