# cloudflare-python

## Installation

Two methods are provided to install this software.
Use PyPi (see [package](https://pypi.python.org/pypi/cloudflare) details) or GitHub (see [package](https://github.com/cloudflare/python-cloudflare) details).

### Via PyPI

```bash
    $ sudo pip install cloudflare
    $
```

Yes - that simple! (the sudo may not be needed in some cases).

### Via github

```bash
    $ git clone https://github.com/cloudflare/python-cloudflare
    $ cd python-cloudflare
    $ ./setup.py build
    $ sudo ./setup.py install
    $
```

Or whatever variance of that you want to use.
There is a Makefile included.

## Cloudflare name change - dropping the capital F

In Sepember/October 2016 the company modified its company name and dropped the capital F.
However, for now (and for backward compatibility reasons) the class name stays the same.

## Cloudflare API version 4

The Cloudflare API can be found [here](https://api.cloudflare.com/).
Each API call is provided via a similarly named function within the **CloudFlare** class.
A full list is provided below.

## Example code

All example code is available on GitHub (see [package](https://github.com/cloudflare/python-cloudflare) in the [examples](https://github.com/cloudflare/python-cloudflare/tree/master/examples) folder).

## Blog

This package was initially introduced [here](https://blog.cloudflare.com/python-cloudflare/) via Cloudflare's [blog](https://blog.cloudflare.com/).

## Getting Started

A very simple listing of zones within your account; including the IPv6 status of the zone.

```python
import CloudFlare

def main():
    cf = CloudFlare.CloudFlare()
    zones = cf.zones.get()
    for zone in zones:
        zone_id = zone['id']
        zone_name = zone['name']
        print("zone_id=%s zone_name=%s" % (zone_id, zone_name))

if __name__ == '__main__':
    main()
```

This example works when there are less than 50 zones (50 is the default number of values returned from a query like this).

Now lets expand on that and add code to show the IPv6 and SSL status of the zones. Lets also query 100 zones.

```python
import CloudFlare

def main():
    cf = CloudFlare.CloudFlare()
    zones = cf.zones.get(params = {'per_page':100})
    for zone in zones:
        zone_id = zone['id']
        zone_name = zone['name']

        settings_ssl = cf.zones.settings.ssl.get(zone_id)
        ssl_status = settings_ssl['value']

        settings_ipv6 = cf.zones.settings.ipv6.get(zone_id)
        ipv6_status = settings_ipv6['value']

        print("zone_id=%s zone_name=%s" % (zone_id, zone_name))
        print("ssl_status=%s ipv6_status=%s" % (ssl_status, ipv6_status))

if __name__ == '__main__':
    main()
```

In order to query more than a single page of zones, we would have to use the raw mode (described more below).
We can loop over many get calls and pass the page parameter to facilitate the paging.

Raw mode is only needed when a get request has the possibility of returning many items.

```python
import CloudFlare

def main():
    cf = CloudFlare.CloudFlare(raw=True)
    page_number = 0
    while True:
        page_number += 1
        raw_results = cf.zones.get(params={'per_page':5,'page':page_number})
        zones = raw_results['result']

        for zone in zones:
            zone_id = zone['id']
            zone_name = zone['name']
            print("zone_id=%s zone_name=%s" % (zone_id, zone_name))

        total_pages = raw_results['result_info']['total_pages']
        if page_number == total_pages:
            break

if __name__ == '__main__':
    main()
```

A more complex example follows.

```python
import CloudFlare

def main():
    zone_name = 'example.com'

    cf = CloudFlare.CloudFlare()

    # query for the zone name and expect only one value back
    try:
        zones = cf.zones.get(params = {'name':zone_name,'per_page':1})
    except CloudFlare.exceptions.CloudFlareAPIError as e:
        exit('/zones.get %d %s - api call failed' % (e, e))
    except Exception as e:
        exit('/zones.get - %s - api call failed' % (e))

    if len(zones) == 0:
        exit('No zones found')

    # extract the zone_id which is needed to process that zone
    zone = zones[0]
    zone_id = zone['id']

    # request the DNS records from that zone
    try:
        dns_records = cf.zones.dns_records.get(zone_id)
    except CloudFlare.exceptions.CloudFlareAPIError as e:
        exit('/zones/dns_records.get %d %s - api call failed' % (e, e))

    # print the results - first the zone name
    print("zone_id=%s zone_name=%s" % (zone_id, zone_name))

    # then all the DNS records for that zone
    for dns_record in dns_records:
        r_name = dns_record['name']
        r_type = dns_record['type']
        r_value = dns_record['content']
        r_id = dns_record['id']
        print('\t', r_id, r_name, r_type, r_value)

    exit(0)

if __name__ == '__main__':
    main()
```

## Providing Cloudflare Username and API Key

When you create a **CloudFlare** class you can pass some combination of these four core parameters.

 * `email` - The account email (only if an API Key is being used)
 * `api` - The API Key (if coding prior to Issue-114 being merged)
 * `token` - The API Token (if coding after to Issue-114)
 * `certtoken` - Optional Origin-CA Certificate Token

This parameter controls how the data is returned from a successful call (see notes below).

 * `raw` - An optional Raw flag (True/False) - defaults to False

Timeouts (10s) and Retries (5) are configured by default. Should you wish to override them, use these settings:
* `global_request_timeout` - How long before each API call to Cloudflare should time out (in seconds)
* `max_requests_retries` - How many times to retry an API call when DNS lookups, socket connections, or connect timeouts occur.

> NOTE: `max_request_retries` is only available when `use_sessions` is not disabled.

The following paramaters are for debug and/or development usage

 * `debug` - An optional Debug flag (True/False) - defaults to False
 * `use_sessions` - An optional Use-Sessions flag (True/False) - defaults to True
 * `profile` - An optional Profile name (the default is `Cloudflare`)
 * `base_url` - An optional Base URL (only used for development)

email=None, key=None, token=None, certtoken=None, debug=False, raw=False, use_sessions=True, profile=None, base_url=None):

### Issue-114

After [Issue-114](https://github.com/cloudflare/python-cloudflare/issues/114) was coded and merged, the use of `token` and `key` changed; however, is backward compatible (amazingly!).

If you are using only the API Token, then don't include the API Email. If you are coding prior to Issue-114, then the API Key can also be used as an API Token if the API Email is not used.

### Python code to create class

```python
import CloudFlare

    # A minimal call - reading values from environment variables or configuration file
    cf = CloudFlare.CloudFlare()

    # A minimal call with debug enabled
    cf = CloudFlare.CloudFlare(debug=True)

    # An authenticated call using an API Token (note the missing email)
    cf = CloudFlare.CloudFlare(token='00000000000000000000000000000000')

    # An authenticated call using an API Email and API Key
    cf = CloudFlare.CloudFlare(email='user@example.com', key='00000000000000000000000000000000')

    # An authenticated call using an API Token and CA-Origin info
    cf = CloudFlare.CloudFlare(token='00000000000000000000000000000000', certtoken='v1.0-...')

    # An authenticated call using an API Email, API Key, and CA-Origin info
    cf = CloudFlare.CloudFlare(email='user@example.com', key='00000000000000000000000000000000', certtoken='v1.0-...')

    # An authenticated call using using a stored profile (see below)
    cf = CloudFlare.CloudFlare(profile="CompanyX"))
```

If the account email and API key are not passed when you create the class, then they are retrieved from either the users exported shell environment variables or the .cloudflare.cfg or ~/.cloudflare.cfg or ~/.cloudflare/cloudflare.cfg files, in that order.

If you're using an API Token, any `cloudflare.cfg` file must either not contain an `email` and `key` attribute (or they can be zero length strings) and the `CLOUDFLARE_EMAIL` `CLOUDFLARE_API_KEY` environment variable must be unset (or zero length strings), otherwise the token (`CLOUDFLARE_API_TOKEN` or `token` attribute) will not be used.

There is one call that presently doesn't need any email or token certification (the */ips* call); hence you can test without any values saved away.

### Using shell environment variables

Note (for latest version of code):

 * `CLOUDFLARE_EMAIL` has replaced `CF_API_EMAIL`.
 * `CLOUDFLARE_API_KEY` has replaced `CF_API_KEY`.
 * `CLOUDFLARE_API_TOKEN` has replaced `CF_API_TOKEN`.
 * `CLOUDFLARE_API_CERTKEY` has replaced `CF_API_CERTKEY`.

Additionally, these two variables are available for testing purposes:

 * `CLOUDFLARE_API_EXTRAS` has replaced `CF_API_EXTRAS`.
 * `CLOUDFLARE_API_URL` has replaced `CF_API_URL`.

The older environment variable names can still be used.

```bash
$ export CLOUDFLARE_EMAIL='user@example.com'
$ export CLOUDFLARE_API_KEY='00000000000000000000000000000000'
$ export CLOUDFLARE_API_CERTKEY='v1.0-...'
$
```

Or if using API Token.

```bash
$ export CLOUDFLARE_API_TOKEN='00000000000000000000000000000000'
$ export CLOUDFLARE_API_CERTKEY='v1.0-...'
$
```
These are optional environment variables; however, they do override the values set within a configuration file.

### Using configuration file to store email and keys

The default profile name is `Cloudflare` for obvious reasons.

```bash
$ cat ~/.cloudflare/cloudflare.cfg
[Cloudflare]
email = user@example.com # Do not set if using an API Token
key = 00000000000000000000000000000000
certtoken = v1.0-...
extras =
$
```

More than one profile can be stored within that file.
Here's an example for a work and home setup (in this example work has an API Token and home uses email/key).

```bash
$ cat ~/.cloudflare/cloudflare.cfg
[Work]
token = 00000000000000000000000000000000
[Home]
email = home@example.com
key = 00000000000000000000000000000000
$
```

To select a profile, use the `--profile profile-name` option for `cli4` command or use `profile="profile-name"` in the library call.

```bash
$ cli4 --profile Work /zones | jq '.[]|.name' | wc -l
      13
$

$ cli4 --profile Home /zones | jq '.[]|.name' | wc -l
       1
$
```

Here is the same in code.

```python
#!/usr/bin/env python

import CloudFlare

def main():
    cf = CloudFlare.CloudFlare(profile="Work")
    ...
```

### Passing your own HTTP headers to API calls

There are very specific case where a user of the library needs to add custom headers to all HTTP calls.
This is rarly needed.

The addition headers can be passed via the confuration file as follows:

```bash
$ cat ~/.cloudflare/cloudflare.cfg
...
http_headers =
        X-Header1:value
        X-Header2: value1 value2 value3
        X-Header3: "this is life as we know it"
        X-Header4: 'two single quotes'
        X-Header5:
...
$
```
Each line should have a header noun, a colon, and a verb.

You can also pass these via Python calls.
```python
    import CloudFlare

    http_headers = [
        'X-Header1:value',
        'X-Header2: value1 value2 value3',
        'X-Header3: "this is life as we know it"',
        'X-Header4: \'two single quotes\'',
        'X-Header5:',
    ]
    cf = CloudFlare.CloudFlare(http_headers=http_headers)
...
```

### Advanced use of configuration file for authentication based on method

The configuration file can have values that are both generic and specific to the method.
Here's an example where a project has a different API Token for reading and writing values.

```bash
$ cat ~/.cloudflare/cloudflare.cfg
[Work]
token = 0000000000000000000000000000000000000000
token.get = 0123456789012345678901234567890123456789
$
```

When a GET call is processed then the second token is used. For all other calls the first token is used.
Here's a more explict verion of that config:

```bash
$ cat ~/.cloudflare/cloudflare.cfg
[Work]
token.delete = 0000000000000000000000000000000000000000
token.get = 0123456789012345678901234567890123456789
token.patch = 0000000000000000000000000000000000000000
token.post = 0000000000000000000000000000000000000000
token.put = 0000000000000000000000000000000000000000
$
```

This can be used with email values also.

### About /certificates and certtoken

The *CLOUDFLARE_API_CERTKEY* or *certtoken* values are used for the Origin-CA */certificates* API calls.
You can leave *certtoken* in the configuration with a blank value (or omit the option variable fully).

The *extras* values are used when adding API calls outside of the core codebase.
Technically, this is only useful for internal testing within Cloudflare.
You can leave *extras* in the configuration with a blank value (or omit the option variable fully).

## Exceptions and return values

### Response data

The response is build from the JSON in the API call.
It contains the **results** values; but does not contain the paging values.

You can return all the paging values by calling the class with raw=True. Here's an example without paging.

```python
#!/usr/bin/env python

import json
import CloudFlare

def main():
    cf = CloudFlare.CloudFlare()
    zones = cf.zones.get(params={'per_page':5})
    print("len=%d" % (zones.length()))

if __name__ == '__main__':
    main()
```

The results are as follows.

```
5
```

When you add the raw option; the APIs full structure is returned. This means the paging values can be seen.

```python
#!/usr/bin/env python

import json
import CloudFlare

def main():
    cf = CloudFlare.CloudFlare(raw=True)
    zones = cf.zones.get(params={'per_page':5})
    print("len=%d" % (zones.length()))
    print(json.dumps(zones, indent=4, sort_keys=True))

if __name__ == '__main__':
    main()
```

This produces.

```
5
{
    "result": [
        ...
    ],
    "result_info": {
        "count": 5,
        "page": 1,
        "per_page": 5,
        "total_count": 31,
        "total_pages": 7
    }
}
```

A full example of paging is provided below.

### Exceptions

The library will raise **CloudFlareAPIError** when the API call fails.
The exception returns both an integer and textual message in one value.

```python
import CloudFlare

    ...
    try
        r = ...
    except CloudFlare.exceptions.CloudFlareAPIError as e:
        exit('api error: %d %s' % (e, e))
    ...
```

The other raised response is **CloudFlareInternalError** which can happen when calling an invalid method.

In some cases more than one error is returned. In this case the return value `e` is also an array.
You can iterate over that array to see the additional error.

```python
import sys
import CloudFlare

    ...
    try
        r = ...
    except CloudFlare.exceptions.CloudFlareAPIError as e:
        if len(e) > 0:
            sys.stderr.write('api error - more than one error value returned!\n')
            for x in e:
                sys.stderr.write('api error: %d %s\n' % (x, x))
        exit('api error: %d %s' % (e, e))
    ...
```

### Exception handling

Here's code using the CLI command `cli4` of the responses passed back in exceptions.

First a simple get with a clean (non-error) response.

```bash
$ cli4 /zones/:example.com/dns_records | jq -c '.[]|{"name":.name,"type":.type,"content":.content}'
{"name":"example.com","type":"MX","content":"something.example.com"}
{"name":"something.example.com","type":"A","content":"10.10.10.10"}
$
```

Next a simple/single error response.
This is simulated by providing incorrect authentication information.

```bash
$ CLOUDFLARE_EMAIL='someone@example.com' cli4 /zones/
cli4: /zones - 9103 Unknown X-Auth-Key or X-Auth-Email
$
```

More than one call can be done on the same command line. In this mode, the connection is preserved between calls.

```bash
$ cli4 /user/organizations /user/invites
...
$
```
Note that the output is presently two JSON structures one after the other - so less useful that you may think.

Finally, a command that provides more than one error response.
This is simulated by passing an invalid IPv4 address to a DNS record creation.

```bash
$ cli4 --post name='foo' type=A content="NOT-A-VALID-IP-ADDRESS" /zones/:example.com/dns_records
cli4: /zones/:example.com/dns_records - 9005 Content for A record is invalid. Must be a valid IPv4 address
cli4: /zones/:example.com/dns_records - 1004 DNS Validation Error
$
```

## Included example code

The [examples](https://github.com/cloudflare/python-cloudflare/tree/master/examples) folder contains many examples in both simple and verbose formats.

You can see the installed path of these files directly via `cli4 -e` (or `cli4 --examples`) command.

```bash
$ cli4 -e
Python .py files:
	...
	/opt/homebrew/lib/python3.11/site-packages/examples/example_always_use_https.py
	...
Bash .sh files:
	...
	/opt/homebrew/lib/python3.11/site-packages/examples/example_paging_thru_zones.sh
	...
$
```

The exact path will vary depending on your system.
The above example is MacOS and Python 3.9 hence the `/opt/homebrew/lib/python3.11/site-packages/` path.
One Linux, the Python pip command may install the code is a system location like `/usr/lib/python3/dist-packages` or `~/.local/lib/python3.9/site-packages/` or something different.
The `cli4 -e` command will try to decode the location and display the example files.

If you are running release before Python 3.9 then you will be asked to install the following:

```bash
$ pip install importlib_resources
...
$
```

It will show up if you are running on an older system. For example, this is the results from running on Win7:

```bash
U:\Users\Bobby>cli4 -e
Module "importlib_resources" missing - please "pip install importlib_resources" as your Python version is lower than 3.9

U:\Users\Bobby>python -V
Python 3.8.3

U:\Users\Bobby>
```

Upgrading from an older version of Python is always recommended. Upgrading from Win7 is by-default even more important!

## A DNS zone code example

```python
#!/usr/bin/env python

import sys
import CloudFlare

def main():
    zone_name = sys.argv[1]
    cf = CloudFlare.CloudFlare()
    zone_info = cf.zones.post(data={'jump_start':False, 'name': zone_name})
    zone_id = zone_info['id']

    dns_records = [
        {'name':'foo', 'type':'AAAA', 'content':'2001:d8b::1'},
        {'name':'foo', 'type':'A', 'content':'192.168.0.1'},
        {'name':'duh', 'type':'A', 'content':'10.0.0.1', 'ttl':120},
        {'name':'bar', 'type':'CNAME', 'content':'foo'},
        {'name':'shakespeare', 'type':'TXT', 'content':"What's in a name? That which we call a rose by any other name ..."}
    ]

    for dns_record in dns_records:
        r = cf.zones.dns_records.post(zone_id, data=dns_record)
    exit(0)

if __name__ == '__main__':
    main()
```

## A DNS zone delete code example (be careful)

```python
#!/usr/bin/env python

import sys
import CloudFlare

def main():
    zone_name = sys.argv[1]
    cf = CloudFlare.CloudFlare()
    zone_info = cf.zones.get(params={'name': zone_name})
    zone_id = zone_info['id']

    dns_name = sys.argv[2]
    dns_records = cf.zones.dns_records.get(zone_id, params={'name':dns_name + '.' + zone_name})
    for dns_record in dns_records:
        dns_record_id = dns_record['id']
        r = cf.zones.dns_records.delete(zone_id, dns_record_id)
    exit(0)

if __name__ == '__main__':
    main()
```

## CLI

All API calls can be called from the command line via the `cli4` command.
Additionally, the `cli4` command will convert domain name or account name prefixed with a colon (`:`) into the correct identifier.
e.g. to view `example.com` you can use `cli4 /zones/:example.com`.
You can pass the zone identifier (or account identifier or any identifier) with a colon followed by the identifier as a hex number 32 characters long.

```bash
$ cli4 [-V|--version] [-h|--help] [-v|--verbose] \
    [-e|--examples] \
    [-q|--quiet] \
    [-j|--json] [-y|--yaml] [-n|--ndjson] [-i|--image] \
    [-r|--raw] \
    [-d|--dump] \
    [-A|--openapi url] \
    [-b|--binary] \
    [-p|--profile profile-name] \
    [--get|--patch|--post|--put|--delete] \
    [item=value|item=@filename|@filename ...] /command ...
```

### CLI parameters for POST/PUT/PATCH

For API calls that need to pass data or parameters there is various formats to use.

The simplest form is `item=value`. This passes the value as a string within the APIs JSON data.

If you need a numeric value passed then `==` can be used to force the value to be treated as a numeric value within the APIs JSON data.
For example: `item==value`.

if you need to pass a list of items; then `[]` can be used. For example:

```bash
pool_id1="11111111111111111111111111111111"
pool_id2="22222222222222222222222222222222"
pool_id3="33333333333333333333333333333333"
cli4 --post global_pools="[ ${pool_id1}, ${pool_id2}, ${pool_id3} ]" region_pools="[ ]" /user/load_balancers/maps
```

Data or parameters can be either named or unnamed.
It can not be both.
Named is the majority format; as described above.
Unnamed parameters simply don't have anything before the `=` sign, as in `=value`.
This format is presently only used by the Cloudflare Load Balancer API calls.
For example:

```bash
cli4 --put ="00000000000000000000000000000000" /user/load_balancers/maps/:00000000000000000000000000000000/region/:WNAM
```

Data can also be uploaded from file contents. Using the `item=@filename` format will open the file and the contents uploaded in the POST.

### CLI output

The default output from the CLI command is in JSON.
It can also output YAML format (i.e. human readable).
This is controled by the `--yaml` or `--json` flags (JSON is the default).
There is also a `--ndjson` flag for use with line based JSON data - this is mainly used for log data.

Additonally the output can be plain text or binary image format depending on the results from the API call (some calls results in non JSON results).
The `--image` flag will return the data in the same format as the API's results.

### Simple CLI calls

 * `cli4 /user/billing/profile`
 * `cli4 /user/invites`

 * `cli4 /zones/:example.com`
 * `cli4 /zones/:example.com/dnssec`
 * `cli4 /zones/:example.com/settings/ipv6`
 * `cli4 --put /zones/:example.com/activation_check`
 * `cli4 /zones/:example.com/keyless_certificates`

 * `cli4 /zones/:example.com/analytics/dashboard`

### More complex CLI calls

Here is the creation of a DNS entry, followed by a listing of that entry and then the deletion of that entry.

```bash
$ $ cli4 --post name="test" type="A" content="10.0.0.1" /zones/:example.com/dns_records
{
    "id": "00000000000000000000000000000000",
    "name": "test.example.com",
    "type": "A",
    "content": "10.0.0.1",
    ...
}
$

$ cli4 /zones/:example.com/dns_records/:test.example.com | jq '{"id":.id,"name":.name,"type":.type,"content":.content}'
{
  "id": "00000000000000000000000000000000",
  "name": "test.example.com",
  "type": "A",
  "content": "10.0.0.1"
}

$ cli4 --delete /zones/:example.com/dns_records/:test.example.com | jq -c .
{"id":"00000000000000000000000000000000"}
$
```

There's the ability to handle dns entries with multiple values.
This produces more than one API call within the command.

```bash
$ cli4 /zones/:example.com/dns_records/:test.example.com | jq -c '.[]|{"id":.id,"name":.name,"type":.type,"content":.content}'
{"id":"00000000000000000000000000000000","name":"test.example.com","type":"A","content":"192.168.0.1"}
{"id":"00000000000000000000000000000000","name":"test.example.com","type":"AAAA","content":"2001:d8b::1"}
$
```

Here are the cache purging commands.

```bash
$ cli4 --delete purge_everything=true /zones/:example.com/purge_cache | jq -c .
{"id":"00000000000000000000000000000000"}
$

$ cli4 --delete files='[http://example.com/css/styles.css]' /zones/:example.com/purge_cache | jq -c .
{"id":"00000000000000000000000000000000"}
$

$ cli4 --delete files='[http://example.com/css/styles.css,http://example.com/js/script.js]' /zones/:example.com/purge_cache | jq -c .
{"id":"00000000000000000000000000000000"}
$

$ cli4 --delete tags='[tag1,tag2,tag3]' /zones/:example.com/purge_cache | jq -c .
cli4: /zones/:example.com/purge_cache - 1107 Only enterprise zones can purge by tag.
$
```

A somewhat useful listing of available plans for a specific zone.

```bash
$ cli4 /zones/:example.com/available_plans | jq -c '.[]|{"id":.id,"name":.name}'
{"id":"00000000000000000000000000000000","name":"Pro Website"}
{"id":"00000000000000000000000000000000","name":"Business Website"}
{"id":"00000000000000000000000000000000","name":"Enterprise Website"}
{"id":"0feeeeeeeeeeeeeeeeeeeeeeeeeeeeee","name":"Free Website"}
$
```

### Cloudflare CA CLI calls

Here's some Cloudflare CA calls. Note the need of the `zone_id=` parameter with the basic `/certificates` call.

```bash
$ cli4 /zones/:example.com | jq -c '.|{"id":.id,"name":.name}'
{"id":"12345678901234567890123456789012","name":"example.com"}
$

$ cli4 zone_id=12345678901234567890123456789012 /certificates | jq -c '.[]|{"id":.id,"expires_on":.expires_on,"hostnames":.hostnames,"certificate":.certificate}'
{"id":"123456789012345678901234567890123456789012345678","expires_on":"2032-01-29 22:36:00 +0000 UTC","hostnames":["*.example.com","example.com"],"certificate":"-----BEGIN CERTIFICATE-----\n ... "}
{"id":"123456789012345678901234567890123456789012345678","expires_on":"2032-01-28 23:23:00 +0000 UTC","hostnames":["*.example.com","example.com"],"certificate":"-----BEGIN CERTIFICATE-----\n ... "}
{"id":"123456789012345678901234567890123456789012345678","expires_on":"2032-01-28 23:20:00 +0000 UTC","hostnames":["*.example.com","example.com"],"certificate":"-----BEGIN CERTIFICATE-----\n ... "}
$
```

A certificate can be viewed via a simple GET request.

```bash
$ cli4 /certificates/:123456789012345678901234567890123456789012345678
{
    "certificate": "-----BEGIN CERTIFICATE-----\n ... ",
    "expires_on": "2032-01-29 22:36:00 +0000 UTC",
    "hostnames": [
        "*.example.com",
        "example.com"
    ],
    "id": "123456789012345678901234567890123456789012345678",
    "request_type": "origin-rsa"
}
$
```

Creating a certificate. This is done with a `POST` request. Note the use of `==` in order to pass a decimal number (vs. string) in JSON. The CSR is not shown for simplicity sake.

```bash
$ CSR=`cat example.com.csr`
$ cli4 --post hostnames='["example.com","*.example.com"]' requested_validity==365 request_type="origin-ecc" csr="$CSR" /certificates
{
    "certificate": "-----BEGIN CERTIFICATE-----\n ... ",
    "csr": "-----BEGIN CERTIFICATE REQUEST-----\n ... ",
    "expires_on": "2018-09-27 21:47:00 +0000 UTC",
    "hostnames": [
        "*.example.com",
        "example.com"
    ],
    "id": "123456789012345678901234567890123456789012345678",
    "request_type": "origin-ecc",
    "requested_validity": 365
}
$
```

Deleting a certificate can be done with a `DELETE` call.

```bash
$ cli4 --delete /certificates/:123456789012345678901234567890123456789012345678
{
    "id": "123456789012345678901234567890123456789012345678",
    "revoked_at": "0000-00-00T00:00:00Z"
}
$
```

### Paging CLI calls

The `--raw` command provides access to the paging returned values.
See the API documentation for all the info.
Here's an example of how to page thru a list of zones (it's included in the examples folder as `example_paging_thru_zones.sh`).
Note the use of `==` to pass a number vs a string as paramater.

```bash
:
tmp=/tmp/$$_
trap "rm ${tmp}; exit 0" 0 1 2 15
PAGE=0
while true
do
        cli4 --raw per_page==5 page==${PAGE} /zones > ${tmp}
        domains=`jq -c '.|.result|.[]|.name' < ${tmp} | tr -d '"'`
        result_info=`jq -c '.|.result_info' < ${tmp}`
        COUNT=`      echo "${result_info}" | jq .count`
        PAGE=`       echo "${result_info}" | jq .page`
        PER_PAGE=`   echo "${result_info}" | jq .per_page`
        TOTAL_COUNT=`echo "${result_info}" | jq .total_count`
        TOTAL_PAGES=`echo "${result_info}" | jq .total_pages`
        echo COUNT=${COUNT} PAGE=${PAGE} PER_PAGE=${PER_PAGE} TOTAL_COUNT=${TOTAL_COUNT} TOTAL_PAGES=${TOTAL_PAGES} -- ${domains}
        if [ "${PAGE}" == "${TOTAL_PAGES}" ]
        then
                ## last section
                break
        fi
        # grab the next page
        PAGE=`expr ${PAGE} + 1`
done
```

It produces the following results.

```bash
COUNT=5 PAGE=1 PER_PAGE=5 TOTAL_COUNT=31 TOTAL_PAGES=7 -- accumsan.example auctor.example consectetur.example dapibus.example elementum.example
COUNT=5 PAGE=2 PER_PAGE=5 TOTAL_COUNT=31 TOTAL_PAGES=7 -- felis.example iaculis.example ipsum.example justo.example lacus.example
COUNT=5 PAGE=3 PER_PAGE=5 TOTAL_COUNT=31 TOTAL_PAGES=7 -- lectus.example lobortis.example maximus.example morbi.example pharetra.example
COUNT=5 PAGE=4 PER_PAGE=5 TOTAL_COUNT=31 TOTAL_PAGES=7 -- porttitor.example potenti.example pretium.example purus.example quisque.example
COUNT=5 PAGE=5 PER_PAGE=5 TOTAL_COUNT=31 TOTAL_PAGES=7 -- sagittis.example semper.example sollicitudin.example suspendisse.example tortor.example
COUNT=1 PAGE=7 PER_PAGE=5 TOTAL_COUNT=31 TOTAL_PAGES=7 -- varius.example vehicula.example velit.example velit.example vitae.example
COUNT=5 PAGE=6 PER_PAGE=5 TOTAL_COUNT=31 TOTAL_PAGES=7 -- vivamus.example
```

### Paging thru lists (using cursors)

Some API calls use cursors to read beyond the initally returned values. See the API page in order to see which API calls do this.

```bash
$ ACCOUNT_ID="00000000000000000000000000000000"
$ LIST_ID="00000000000000000000000000000000"
$
$ cli4 --raw /accounts/::${ACCOUNT_ID}/rules/lists/::${LIST_ID}/items > /tmp/page1.json
$ after=`jq -r '.result_info.cursors.after' < /tmp/page1.json`
$ echo "after=$after"
after=Mxm4GVmKjYbFjy2VxMPipnJigm1M_s6lCS9ABR9wx-RM2A
$
```

Once we have the `after` value, we can pass it along in order to read the next hunk of values. We finish when `after` returns as null (or isn't present).

```bash
$ cli4 --raw cursor="$after" /accounts/::${ACCOUNT_ID}/rules/lists/::${LIST_ID}/items > /tmp/page2.json
$ after=`jq -r '.result_info.cursors.after' < /tmp/page2.json`
$ echo "after=$after"
after=null
$
```

We can see the results now in two files.

```bash
$ jq -c '.result[]' < /tmp/page1.json | wc -l
      25
$

$ jq -c '.result[]' < /tmp/page2.json | wc -l
       5
$

$ for f in /tmp/page?.json ; do jq -r '.result[]|.id,.ip,.comment' < $f | paste - - - ; done | column -s'   ' -t
0fe44928258549feb47126a966fbf4a0  0.0.0.0           all zero
2e1e02120f5e466f8c0e26375e4cf4c8  1.0.0.1           Cloudflare DNS a
9ca5fd0ac6f54fdbb9dedd3fb72ce2da  1.1.1.1           Cloudflare DNS b
b3654987446743738c782f36ebe074f5  10.0.0.0/8        RFC1918 space
90bec8ce37d242faa2e27d1e78c1d8e2  103.21.244.0/22   Cloudflare IP
970a3c810cda41af9bef2c36a1892f7e  103.22.200.0/22   Cloudflare IP
3ec8516158bf4f3cac18210f611ee541  103.31.4.0/22     Cloudflare IP
ee9d268367204e6bb8e5e4c907f22de8  104.16.0.0/12     Cloudflare IP
93ae02eda9774c45840af367a02fe529  108.162.192.0/18  Cloudflare IP
62891ebf6db44aa494d79a6401af185e  131.0.72.0/22     Cloudflare IP
cac40cd940cc470582b8c912a8a12bea  141.101.64.0/18   Cloudflare IP
f6d5eacd81a2407f8e0d81caee21e7f8  162.158.0.0/15    Cloudflare IP
3d538dfc38ab471d9d3fe78332acfa4e  172.16.0.0/12     RFC1918 space
f353cb8f98424837ad35382a22b9debe  172.64.0.0/13     Cloudflare IP
78f3e1a0bafc41f88d4d40ad49a642e0  173.245.48.0/20   Cloudflare IP
c23a545475c54c32a7681c6b508d3e80  188.114.96.0/20   Cloudflare IP
f693237c9e294fe481221cbc2d7c20ef  190.93.240.0/20   Cloudflare IP
6d465ab3a0994c07827ebdcf8f34d977  192.168.0.0/16    RFC1918 space
1ad1e634b3664bac939086185c62faf7  197.234.240.0/22  Cloudflare IP
5d2968e7b3114d8e869a379d71c8ba86  198.41.128.0/17   Cloudflare IP
6a69de60b31448fa864f0a3ac5abe8d0  224.0.0.0/24      Multicast
30749cce89af4ab3a80e308294f46a46  240.0.0.0/4       Class E
2b32c67ea4d044628abe39f28662d8f0  255.255.255.255   all ones
cc7cd828b2fb4bcfb9391c2d3ef8d068  2400:cb00::/32    Cloudflare IP
b30d4cbd7dcd48729e8ebeda552e48a8  2405:8100::/32    Cloudflare IP
49db60758c8344959c338a74afc9748a  2405:b500::/32    Cloudflare IP
96e9eca1923c40d5a84865145f5a5d6a  2606:4700::/32    Cloudflare IP
21bc52a26e10405d89b7180ddcf49302  2803:f800::/32    Cloudflare IP
ff78f842188e4b869eb5389ae9ab8f41  2a06:98c0::/29    Cloudflare IP
0880cdfc40b14f6fa0639522a728859d  2c0f:f248::/32    Cloudflare IP
$
```

The `result_info.cursors` area also contains a `before` value for reverse scrolling.

As with `per_page` scrolling, raw mode is used.

### DNSSEC CLI calls

```bash
$ cli4 /zones/:example.com/dnssec | jq -c '{"status":.status}'
{"status":"disabled"}
$

$ cli4 --patch status=active /zones/:example.com/dnssec | jq -c '{"status":.status}'
{"status":"pending"}
$

$ cli4 /zones/:example.com/dnssec
{
    "algorithm": "13",
    "digest": "41600621c65065b09230ebc9556ced937eb7fd86e31635d0025326ccf09a7194",
    "digest_algorithm": "SHA256",
    "digest_type": "2",
    "ds": "example.com. 3600 IN DS 2371 13 2 41600621c65065b09230ebc9556ced937eb7fd86e31635d0025326ccf09a7194",
    "flags": 257,
    "key_tag": 2371,
    "key_type": "ECDSAP256SHA256",
    "modified_on": "2016-05-01T22:42:15.591158Z",
    "public_key": "mdsswUyr3DPW132mOi8V9xESWE8jTo0dxCjjnopKl+GqJxpVXckHAeF+KkxLbxILfDLUT0rAK9iUzy1L53eKGQ==",
    "status": "pending"
}
$
```

### Zone file upload (i.e. import) CLI calls (uses BIND format files)

Refer to [Import DNS records](https://api.cloudflare.com/#dns-records-for-a-zone-import-dns-records) on API documentation for this feature.

```bash
$ cat zone.txt
example.com.            IN      SOA     somewhere.example.com. someone.example.com. (
                                2017010101
                                3H
                                15
                                1w
                                3h
                        )

record1.example.com.    IN      A       10.0.0.1
record2.example.com.    IN      AAAA    2001:d8b::2
record3.example.com.    IN      CNAME   record1.example.com.
record4.example.com.    IN      TXT     "some text"
$

$ cli4 --post file=@zone.txt /zones/:example.com/dns_records/import
{
    "recs_added": 4,
    "total_records_parsed": 4
}
$
```

### Zone file upload (i.e. import) Python calls (uses BIND format files)

Because `import` is a keyword (or reserved word) in Python we append a `_` (underscore) to the verb in order to use.
The `cli4` command does not need this edit.

```python
    #
    # "import" is a reserved word and hence we add '_' to the end of verb.
    #
    r = cf.zones.dns_records.import_.post(zone_id, files={'file':fd})
```

See [examples/example_dns_import.py](https://github.com/cloudflare/python-cloudflare/tree/master/examples/example_dns_import.py) for working code.

### Zone file download (i.e. export) CLI calls (uses BIND format files)

The following is documented within the **Advanced** option of the DNS page within the Cloudflare portal.

```bash
$ cli4 /zones/:example.com/dns_records/export | egrep -v '^;;|^$'
$ORIGIN .
@       3600    IN      SOA     example.com.    root.example.com.       (
                2025552311      ; serial
                7200            ; refresh
                3600            ; retry
                86400           ; expire
                3600)           ; minimum
example.com.    300     IN      NS      REPLACE&ME$WITH^YOUR@NAMESERVER.
record4.example.com.    300     IN      TXT     "some text"
record3.example.com.    300     IN      CNAME   record1.example.com.
record1.example.com.    300     IN      A       10.0.0.1
record2.example.com.    300     IN      AAAA    2001:d8b::2
$
```

The egrep is used for documentation brevity.

This can also be done via Python code with the following example.

```python
#!/usr/bin/env python
import sys
import CloudFlare

def main():
    zone_name = sys.argv[1]
    cf = CloudFlare.CloudFlare()

    zones = cf.zones.get(params={'name': zone_name})
    zone_id = zones[0]['id']

    dns_records = cf.zones.dns_records.export.get(zone_id)
    for l in dns_records.splitlines():
        if len(l) == 0 or l[0] == ';':
            continue
        print(l)
    exit(0)

if __name__ == '__main__':
    main()
```

### Cloudflare Workers

Cloudflare Workers are described on the Cloudflare blog at
[here](https://blog.cloudflare.com/introducing-cloudflare-workers/) and
[here](https://blog.cloudflare.com/code-everywhere-cloudflare-workers/), with the beta release announced
[here](https://blog.cloudflare.com/cloudflare-workers-is-now-on-open-beta/).

The Python libraries now support the Cloudflare Workers API calls. The following javascript is lifted from [https://cloudflareworkers.com/](https://cloudflareworkers.com/) and slightly modified.

```bash
$ cat modify-body.js
addEventListener("fetch", event => {
  event.respondWith(fetchAndModify(event.request));
});

async function fetchAndModify(request) {
  console.log("got a request:", request);

  // Send the request on to the origin server.
  const response = await fetch(request);

  // Read response body.
  const text = await response.text();

  // Modify it.
  const modified = text.replace(
  "<body>",
  "<body style=\"background: #ff0;\">");

  // Return modified response.
  return new Response(modified, {
    status: response.status,
    statusText: response.statusText,
    headers: response.headers
  });
}
$
```

Here's the website with it's simple `<body>` statement

```bash
$ curl -sS https://example.com/ | fgrep '<body'
  <body>
$
```

Now lets add the script. Looking above, you will see that it's simple action is to modify the `<body>` statement and make the background yellow.

```bash
$ cli4 --put @- /zones/:example.com/workers/script < modify-body.js
{
    "etag": "1234567890123456789012345678901234567890123456789012345678901234",
    "id": "example-com",
    "modified_on": "2018-02-15T00:00:00.000000Z",
    "script": "addEventListener(\"fetch\", event => {\n  event.respondWith(fetchAndModify(event.request));\n});\n\nasync function fetchAndModify(request) {\n  console.log(\"got a request:\", request);\n\n  // Send the request on to the origin server.\n  const response = await fetch(request);\n\n  // Read response body.\n  const text = await response.text();\n\n  // Modify it.\n  const modified = text.replace(\n  \"<body>\",\n  \"<body style=\\\"background: #ff0;\\\">\");\n\n  // Return modified response.\n  return new Response(modified, {\n    status: response.status,\n    statusText: response.statusText,\n    headers: response.headers\n  });\n}\n",
    "size": 603
}
$
```

The following call checks that the script is associated with the zone. In this case, it's the only script added by this user.

```bash
$ cli4 /user/workers/scripts
[
    {
        "created_on": "2018-02-15T00:00:00.000000Z",
        "etag": "1234567890123456789012345678901234567890123456789012345678901234",
        "id": "example-com",
        "modified_on": "2018-02-15T00:00:00.000000Z"
    }
]
$
```

Next step is to make sure a route is added for that script on that zone.

```bash
$ cli4 --post pattern="example.com/*" script="example-com" /zones/:example.com/workers/routes
{
    "id": "12345678901234567890123456789012"
}
$

$ cli4 /zones/:example.com/workers/routes
[
    {
        "id": "12345678901234567890123456789012",
        "pattern": "example.com/*",
        "script": "example-com"
    }
]
$
```

With that script added to the zone and the route added, we can now see the website has been modified because of the Cloudflare Worker.

```bash
$ curl -sS https://example.com/ | fgrep '<body'
  <body style="background: #ff0;">
$
```

All this can be removed; hence bringing the website back to its initial state.

```bash
$ cli4 --delete /zones/:example.com/workers/script
12345678901234567890123456789012
$ cli4 --delete /zones/:example.com/workers/routes/:12345678901234567890123456789012
true
$

$ curl -sS https://example.com/ | fgrep '<body'
  <body>
$
```

Refer to the Cloudflare Workers API documentation for more information.

## Cloudflare Instant Logs

Please see https://developers.cloudflare.com/logs/instant-logs for all the information on how to use this feature.
The `cli4` command along with the Python libaries can be used to control the instant logs; however, the websocket reading is outside the scope of this library.

To query the states of the instant logs:

```bash
$ cli4 /zones/:example.com/logpush/edge/jobs | jq .
[]
$
```

To add monitoring:

```bash
$ cli4 --post \
        ='{
                "fields": "ClientIP,ClientRequestHost,ClientRequestMethod,ClientRequestURI,EdgeEndTimestamp,EdgeResponseBytes,EdgeResponseStatus,EdgeStartTimestamp,RayID",
                "sample": 1,
                "filter": "",
                "kind": "instant-logs"
        }' \
        /zones/:example.com/logpush/edge/jobs | jq .
{
  "destination_conf": "wss://logs.cloudflare.com/instant-logs/ws/sessions/00000000000000000000000000000000",
  "fields": "ClientIP,ClientRequestHost,ClientRequestMethod,ClientRequestURI,EdgeEndTimestamp,EdgeResponseBytes,EdgeResponseStatus,EdgeStartTimestamp,RayID",
  "filter": "",
  "kind": "instant-logs",
  "sample": 1,
  "session_id": "00000000000000000000000000000000"
}
$
```

To see the results:

```bash
$ cli4 /zones/:example.com/logpush/edge/jobs | jq .
[
  {
    "fields": "ClientIP,ClientRequestHost,ClientRequestMethod,ClientRequestURI,EdgeEndTimestamp,EdgeResponseBytes,EdgeResponseStatus,EdgeStartTimestamp,RayID",
    "filter": "",
    "kind": "instant-logs",
    "sample": 1,
    "session_id": "00000000000000000000000000000000"
  }
]
$
```

## Cloudflare GraphQL

The GraphQL interface can be accessed via the command line or via Python.

```
    query="""
      query {
        viewer {
            zones(filter: {zoneTag: "%s"} ) {
            httpRequests1dGroups(limit:40, filter:{date_lt: "%s", date_gt: "%s"}) {
              sum { countryMap { bytes, requests, clientCountryName } }
              dimensions { date }
            }
          }
        }
      }
    """ % (zone_id, date_before[0:10], date_after[0:10])

    r = cf.graphql.post(data={'query':query})

    httpRequests1dGroups = zone_info = r['data']['viewer']['zones'][0]['httpRequests1dGroups']
```

See the [examples/example_graphql.sh](examples/example_graphql.sh) and [examples/example_graphql.py](https://github.com/cloudflare/python-cloudflare/tree/master/examples/example_graphql.py) files for working examples.
Here is the working example of the shell version:

```bash
$ examples/example_graphql.sh example.com
2020-07-14T02:00:00Z    34880
2020-07-14T03:00:00Z    18953
2020-07-14T04:00:00Z    28700
2020-07-14T05:00:00Z    2358
2020-07-14T06:00:00Z    34905
2020-07-14T07:00:00Z    779
2020-07-14T08:00:00Z    35450
2020-07-14T10:00:00Z    17803
2020-07-14T11:00:00Z    32678
2020-07-14T12:00:00Z    19947
2020-07-14T13:00:00Z    4956
2020-07-14T14:00:00Z    34585
2020-07-14T15:00:00Z    3022
2020-07-14T16:00:00Z    5224
2020-07-14T18:00:00Z    79482
2020-07-14T21:00:00Z    10609
2020-07-14T22:00:00Z    5740
2020-07-14T23:00:00Z    2545
2020-07-15T01:00:00Z    10777
$
```

For more information on how to use GraphQL at Cloudflare, refer to the [Cloudflare GraphQL Analytics API](https://developers.cloudflare.com/analytics/graphql-api).
It contains a full overview of Cloudflare's GraphQL features and keywords.

## Cloudflare AI

See https://blog.cloudflare.com/workers-ai-update-stable-diffusion-code-llama-workers-ai-in-100-cities/ for the introduction,
along with https://developers.cloudflare.com/workers-ai/models/ for the nitty gritty details.

There are three AI calls included within the example folder.

### Image creation.

```bash
$ python examples/example_ai_images.py A happy llama running through an orange cloud > /tmp/image.png
$
$ file /tmp/image.png
/tmp/image.png: PNG image data, 1024 x 1024, 8-bit/color RGB, non-interlaced
$
```

### Translation.

```bash
$ python examples/example_ai_translate.py I\'ll have an order of the moule frites
Je vais avoir une commande des frites de moule
$
```

### Speech Recognition with the openai/whisper model.

The following downloads a speech as an mp3 file and passes it to the AI API.
It does a very good job transcribing; however, there's a good chance these mp3 files were use for training.
That said, the example code is here to show how the API works vs testing the AI/ML quality.

```bash
$ python examples/example_ai_speechrecognition.py
mp3 received: length=700367
My fellow Americans, Michelle and I have been so touched by all the well wishes that we've received over the past few weeks. But tonight, tonight it's my turn to say thanks.
$
```

This is presently work-in-progress because of the non-Python calling method. The syntax could change in the future.

They can also be called via `cli4`.

```bash
$ cli4 --image --post text="I'll have an order of the moule frites" source_lang=english target_lang=french /accounts/:AccountID/ai/run/@cf/meta/m2m100-1.2b
{'translated_text': 'Je vais avoir une commande des frites de moule'}
$
```

Presently you will need the following in your `cloudflare.cfg` file.

```bash
$ cat ~/.cloudflare/cloudflare.cfg
[CloudFlare]
global_request_timeout = 120
max_request_retries = 1
extras =
    /accounts/:id/ai/run/@cf/meta/llama-2-7b-chat-fp16
    /accounts/:id/ai/run/@cf/meta/llama-2-7b-chat-int8
    /accounts/:id/ai/run/@cf/mistral/mistral-7b-instruct-v0.1
    /accounts/:id/ai/run/@cf/openai/whisper
    /accounts/:id/ai/run/@cf/meta/m2m100-1.2b
    /accounts/:id/ai/run/@cf/huggingface/distilbert-sst-2-int8
    /accounts/:id/ai/run/@cf/microsoft/resnet-50
    /accounts/:id/ai/run/@cf/stabilityai/stable-diffusion-xl-base-1.0
    /accounts/:id/ai/run/@cf/baai/bge-base-en-v1.5
    /accounts/:id/ai/run/@cf/baai/bge-large-en-v1.5
    /accounts/:id/ai/run/@cf/baai/bge-small-en-v1.5

$
```

As the `@` (at) symbol and the `.` (dot) symbol aren't allowed in python variable names; you'll have the replace `@cf` with `at_cf` and `.` with `_`.
There's already notes above that state that `-` (dash) is replaced with `_` in the code.
That will be needed with some model names.

The `cli4` command does not need this edit. It is done on the fly!

For example, the following code is valid:
```python
    r = cf.accounts.ai.run.at_cf.openai.whisper.post(account_id, data=audio_data)
    r = cf.accounts.ai.run.at_cf.meta.m2m100_1_2b.post(account_id, data=translate_data)
    r = cf.accounts.ai.run.at_cf.stabilityai.stable_diffusion_xl_base_1_0.post(account_id, data=image_create_data)
```

Or you can use the `find()` call can will do this conversion for you.
```python
    translate_data = {'text':"I'll have an order of the moule frites", 'source_lang':'english', 'target_lang':'french'}

    m = cf.find('/accounts/:id/ai/run/@cf/meta/m2m100-1.2b')
    r = m.post(account_id, data=translate_data)
    print(r['translated_text'])
```

You will also have to run with a version of the library above `2.18.2`.

## Implemented API calls

The `--dump` argument to cli4 will produce a list of all the call implemented within the library.

```bash
$ cli4 --dump
/certificates
/ips
/organizations
...
/zones/ssl/analyze
/zones/ssl/certificate_packs
/zones/ssl/verification
$
```

### Table of commands

An automatically generated table of commands is provided [here](TABLE-OF-COMMANDS.md).

## Adding extra API calls manually

Extra API calls can be added via the configuration file

```bash
$ cat ~/.cloudflare/cloudflare.cfg
[Cloudflare]
extras =
    /client/v4/command
    /client/v4/command/:command_identifier
    /client/v4/command/:command_identifier/settings
$
```

While it's easy to call anything within Cloudflare's API, it's not very useful to add items in here as they will simply return API URL errors.
Technically, this is only useful for internal testing within Cloudflare.

## Issues

The following error can be caused by an out of date SSL/TLS library and/or out of date Python.

```bash
/usr/local/lib/python2.7/dist-packages/requests/packages/urllib3/util/ssl_.py:318: SNIMissingWarning: An HTTPS request has been made, but the SNI (Subject Name Indication) extension to TLS is not available on this platform. This may cause the server to present an incorrect TLS certificate, which can cause validation failures. You can upgrade to a newer version of Python to solve this. For more information, see https://urllib3.readthedocs.org/en/latest/security.html#snimissingwarning.
  SNIMissingWarning
/usr/local/lib/python2.7/dist-packages/requests/packages/urllib3/util/ssl_.py:122: InsecurePlatformWarning: A true SSLContext object is not available. This prevents urllib3 from configuring SSL appropriately and may cause certain SSL connections to fail. You can upgrade to a newer version of Python to solve this. For more information, see https://urllib3.readthedocs.org/en/latest/security.html#insecureplatformwarning.
  InsecurePlatformWarning
```

The solution can be found [here](https://urllib3.readthedocs.org/en/latest/security.html#insecureplatformwarning) and/or [here](http://stackoverflow.com/questions/35144550/how-to-install-cryptography-on-ubuntu).

## Python 2.x vs 3.x support

As of May/June 2016 the code is now tested against pylint.
This was required in order to move the codebase into Python 3.x.
The motivation for this came from [Danielle Madeley (danni)](https://github.com/danni).

~~While the codebase has been edited to run on Python 3.x, there's not been enough Python 3.x testing performed.~~
~~If you can help in this regard; please contact the maintainers.~~

As of January 2020 the code is Python3 clean.

As of January 2020 the code is shipped up to pypi with Python2 support removed.

As of January 2020 the code is Python3.8 clean. The new `SyntaxWarning` messages (i.e. `SyntaxWarning: "is" with a literal. Did you mean "=="?`) meant minor edits were needed.

As of late 2023 the code is Python3.11 clean.

## pypi and GitHub signed releases

As of October/2022, the code is signed by the maintainers personal email address: `mahtin@mahtin.com` `7EA1 39C4 0C1C 842F 9D41 AAF9 4A34 925D 0517 2859`

## Credit

This is based on work by [Felix Wong (gnowxilef)](https://github.com/gnowxilef) found [here](https://github.com/cloudflare-api/python-cloudflare-v4).
It has been seriously expanded upon.

## Changelog

An automatically generated CHANGELOG is provided [here](CHANGELOG.md).

## Copyright

Portions copyright [Felix Wong (gnowxilef)](https://github.com/gnowxilef) 2015 and Cloudflare 2016 thru 2023.
