# Rules

## Rule levels
A rule returns feedback on a specific level: Error, Warning or Informational.

### Errors
Errors will start with the letter `E`. Errors will (or can) result in a hard failure for the template being validated.

### Warnings
Warnings start with the letter `W`. Warnings alert you when the template doesn't follow best practices but should still function.  *Example: If you use a parameter for a RDS master password you should have the parameter property NoEcho set to true.*

### Informational
Informational results start with the letter `I`. Informational alert you when the template doesn't follow best practices, just like the Warnings, but in a non-blocking way. The Informational level is disabled by default and has to be enabled explicitly using `-c I` or `--include-checks I`.

## Categories

| Rule Numbers          | Category |
| --------------------- | ------------- |
| (E&#124;W&#124;I)0XXX | Basic Template Errors. Examples: Not parseable, main sections (Outputs, Resources, etc.) |
| (E&#124;W&#124;I)1XXX | Functions (Ref, GetAtt, etc.) |
| (E&#124;W&#124;I)2XXX | Parameters |
| (E&#124;W&#124;I)3XXX | Resources |
| (E&#124;W&#124;I)4XXX | Metadata |
| (E&#124;W&#124;I)6xxx | Outputs |
| (E&#124;W&#124;I)7xxx | Mappings |
| (E&#124;W&#124;I)8xxx | Conditions |
| (E&#124;W&#124;I)9xxx | Reserved for users rules |

*Warning*
Rule `E3012` is used to check the types for value of a resource property. A number is a number, string is a string, etc.  There are occasions where this could be just a warning and other times it could be an error. cfn-lint doesn't have an exception process so all instances of this issue are considered errors. You can disable this rule using `--ignore-checks` if it is not required for your internal best practices. Conversely, strict typing is **NOT** enforced by default for this rule, so if strict adherence to resource value types is necessary for your use case, you can use `--configure-rule E3012:strict=true` to enforce the rule.

## Experimental rules
Sometimes there are (new) rules that might be complex, that doesn't have enough solid test templates and examples and/or might have unexpected results. We support adding in these rules so they can be tested, tweaked and improved before they become generally available.

A rule can be marked as experimental by setting the `experimental` flag in the Rule's MetaData:

```python
class MyExperimentalRule(CloudFormationLintRule):
    id = 'E1234'
    ...
    experimental = True
```

To include these rules, use the `-e/include-experimental` argument when running cfn-lint.

## Rules
(_This documentation is generated by running `cfn-lint --update-documentation`, do not alter this manually_)

The following **237** rules are applied by this linter:

| Rule ID  | Title | Description | Config<br />(Name:Type:Default) | Source | Tags |
| -------- | ----- | ----------- | ---------- | ------ | ---- |
| [E0000<a name="E0000"></a>](../src/cfnlint/rules/errors/parse.py) | Parsing error found when parsing the template | Checks for JSON/YAML formatting errors in your template |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `base` |
| [E0001<a name="E0001"></a>](../src/cfnlint/rules/errors/transform.py) | Error found when transforming the template | Errors found when performing transformation on the template |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `base`,`transform` |
| [E0002<a name="E0002"></a>](../src/cfnlint/rules/errors/rule.py) | Error processing rule on the template | Errors found when processing a rule on the template |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `base`,`rule` |
| [E1001<a name="E1001"></a>](../src/cfnlint/rules/jsonschema/JsonSchema.py) | Basic CloudFormation Template Configuration | Making sure the basic CloudFormation template components are properly configured | sections:string: | [Source](https://github.com/aws-cloudformation/cfn-lint) | `base` |
| [E1002<a name="E1002"></a>](../src/cfnlint/rules/templates/LimitSize.py) | Validate if a template size is too large | Check the size of the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `limits` |
| [E1003<a name="E1003"></a>](../src/cfnlint/rules/templates/LimitDescription.py) | Validate the max size of a description | Check if the size of the template description is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `description`,`limits` |
| [E1004<a name="E1004"></a>](../src/cfnlint/rules/templates/Description.py) | Template description can only be a string | Template description can only be a string |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-description-structure.html) | `description` |
| [E1005<a name="E1005"></a>](../src/cfnlint/rules/transforms/Configuration.py) | Validate Transform configuration | Validate that the transforms section of a template is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html) | `transform` |
| [E1010<a name="E1010"></a>](../src/cfnlint/rules/functions/GetAtt.py) | GetAtt validation of parameters | Validates that GetAtt parameters are to valid resources and properties of those resources |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-getatt.html) | `functions`,`getatt` |
| [E1011<a name="E1011"></a>](../src/cfnlint/rules/functions/FindInMap.py) | FindInMap validation of configuration | Making sure the function is a list of appropriate config |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-findinmap.html) | `functions`,`findinmap` |
| [E1015<a name="E1015"></a>](../src/cfnlint/rules/functions/GetAz.py) | GetAz validation of parameters | Making sure the GetAz function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-getavailabilityzones.html) | `functions`,`getaz` |
| [E1016<a name="E1016"></a>](../src/cfnlint/rules/functions/ImportValue.py) | ImportValue validation of parameters | Making sure the ImportValue function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-importvalue.html) | `functions`,`importvalue` |
| [E1017<a name="E1017"></a>](../src/cfnlint/rules/functions/Select.py) | Select validation of parameters | Making sure the Select function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-select.html) | `functions`,`select` |
| [E1018<a name="E1018"></a>](../src/cfnlint/rules/functions/Split.py) | Split validation of parameters | Making sure the split function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-split.html) | `functions`,`split` |
| [E1019<a name="E1019"></a>](../src/cfnlint/rules/functions/Sub.py) | Sub validation of parameters | Making sure the sub function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub` |
| [E1020<a name="E1020"></a>](../src/cfnlint/rules/functions/Ref.py) | Ref validation of value | Making sure the Ref has a String value (no other functions are supported) |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-ref.html) | `functions`,`ref` |
| [E1021<a name="E1021"></a>](../src/cfnlint/rules/functions/Base64.py) | Base64 validation of parameters | Making sure the Base64 function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-base64.html) | `functions`,`base64` |
| [E1022<a name="E1022"></a>](../src/cfnlint/rules/functions/Join.py) | Join validation of parameters | Making sure the join function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-join.html) | `functions`,`join` |
| [E1024<a name="E1024"></a>](../src/cfnlint/rules/functions/Cidr.py) | Cidr validation of parameters | Making sure the function CIDR is a list with valid values |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-cidr.html) | `functions`,`cidr` |
| [E1027<a name="E1027"></a>](../src/cfnlint/rules/functions/DynamicReferenceSecureString.py) | Check dynamic references secure strings are in supported locations | Dynamic References Secure Strings are only supported for a small set of resource properties.  Validate that they are being used in the correct location when checking values and Fn::Sub in resource properties. Currently doesn't check outputs, maps, conditions, parameters, and descriptions. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html) | `functions`,`dynamic reference` |
| [E1028<a name="E1028"></a>](../src/cfnlint/rules/functions/If.py) | Check Fn::If structure for validity | Check Fn::If to make sure its valid.  Condition has to be a string. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-if) | `functions`,`if` |
| [E1029<a name="E1029"></a>](../src/cfnlint/rules/functions/SubNeeded.py) | Sub is required if a variable is used in a string | If a substitution variable exists in a string but isn't wrapped with the Fn::Sub function the deployment will fail. | custom_excludes:string: | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub` |
| [E1030<a name="E1030"></a>](../src/cfnlint/rules/functions/Length.py) | Length validation of parameters | Making sure Fn::Length is configured correctly |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-length.html) | `functions`,`length` |
| [E1031<a name="E1031"></a>](../src/cfnlint/rules/functions/ToJsonString.py) | ToJsonString validation of parameters | Making sure Fn::ToJsonString is configured correctly |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference.html) | `functions`,`toJsonString` |
| [E1032<a name="E1032"></a>](../src/cfnlint/rules/functions/ForEach.py) | Validates ForEach functions | Validates that ForEach parameters have a valid configuration |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-getatt.html) | `functions`,`foreach` |
| [E1040<a name="E1040"></a>](../src/cfnlint/rules/functions/GetAttFormat.py) | Check if GetAtt matches destination format | Validate that if source and destination format exists that they match |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/best-practices.html#parmtypes) | `parameters`,`ec2`,`imageid` |
| [E1050<a name="E1050"></a>](../src/cfnlint/rules/functions/DynamicReference.py) | Validate the structure of a dynamic reference | Make sure dynamic reference strings have the correct syntax |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html) | `functions`,`dynamic reference` |
| [E1051<a name="E1051"></a>](../src/cfnlint/rules/functions/DynamicReferenceSecretsManagerPath.py) | Validate dynamic references to secrets manager are only in resource properties | Dynamic references from secrets manager can only be used in resource properties |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html#dynamic-references-secretsmanager) | `functions`,`dynamic reference` |
| [E1052<a name="E1052"></a>](../src/cfnlint/rules/functions/DynamicReferenceSsmPath.py) | Validate dynamic references to SSM are in a valid location | Dynamic references to SSM parameters are only supported in certain locations |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html#dynamic-references-ssm) | `functions`,`dynamic reference` |
| [E1101<a name="E1101"></a>](../src/cfnlint/rules/jsonschema/CfnLint.py) | Validate an item against additional checks | Use supplemental logic to validate properties against |  | [Source]() |  |
| [E1103<a name="E1103"></a>](../src/cfnlint/rules/formats/Format.py) | Validate the format of a value | Parent rule for validating the format keyword in schemas |  | [Source]() |  |
| [E1150<a name="E1150"></a>](../src/cfnlint/rules/formats/SecurityGroupId.py) | Validate security group format | Security groups have to ref/gettatt to a security group or has the valid pattern |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/format_keyword.md#AWS::EC2::SecurityGroup.GroupId) |  |
| [E1151<a name="E1151"></a>](../src/cfnlint/rules/formats/VpcId.py) | Validate VPC id format | Check that a VPC id matches a pattern |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/format_keyword.md#AWS::EC2::VPC.Id) |  |
| [E1152<a name="E1152"></a>](../src/cfnlint/rules/formats/ImageId.py) | Validate AMI id format | Check that a AMI id matches a pattern |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/format_keyword.md#AWS::EC2::Image.Id) |  |
| [E1153<a name="E1153"></a>](../src/cfnlint/rules/formats/SecurityGroupName.py) | Validate security group name | Security group names have to valid pattern |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/format_keyword.md#AWS::EC2::SecurityGroup.GroupName) |  |
| [E1154<a name="E1154"></a>](../src/cfnlint/rules/formats/SubnetId.py) | Validate VPC subnet id format | Check that a VPC subnet id matches a pattern |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/format_keyword.md#AWS::EC2::Subnet.Id) |  |
| [E1700<a name="E1700"></a>](../src/cfnlint/rules/rules/Configuration.py) | Rules have the appropriate configuration | Making sure the Rules section is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/rules-section-structure.html) | `rules` |
| [E1701<a name="E1701"></a>](../src/cfnlint/rules/rules/Assert.py) | Validate the configuration of Assertions | Make sure the Assert value in a Rule is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/rules-section-structure.html) | `rules` |
| [E1702<a name="E1702"></a>](../src/cfnlint/rules/rules/RuleCondition.py) | Validate the configuration of Rules RuleCondition | Make sure the RuleCondition in a Rule is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/rules-section-structure.html) | `rules` |
| [E2001<a name="E2001"></a>](../src/cfnlint/rules/parameters/Configuration.py) | Parameters have appropriate properties | Making sure the parameters are properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html) | `parameters` |
| [E2002<a name="E2002"></a>](../src/cfnlint/rules/parameters/Types.py) | Parameters have appropriate type | Making sure the parameters have a correct type |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/best-practices.html#parmtypes) | `parameters` |
| [E2003<a name="E2003"></a>](../src/cfnlint/rules/parameters/Pattern.py) | Parameters have appropriate names | Check if Parameters are properly named (A-Za-z0-9) |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`name` |
| [E2010<a name="E2010"></a>](../src/cfnlint/rules/parameters/MaxProperties.py) | Parameter limit not exceeded | Check the number of Parameters in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`limits` |
| [E2011<a name="E2011"></a>](../src/cfnlint/rules/parameters/PropertyNames.py) | Validate the name for a parameter | Validate the name of a parameter with special handling of the max length length |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`limits` |
| [E2015<a name="E2015"></a>](../src/cfnlint/rules/parameters/Default.py) | Default value is within parameter constraints | Making sure the parameters have a default value inside AllowedValues, MinValue, MaxValue, AllowedPattern |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html) | `parameters` |
| [E2529<a name="E2529"></a>](../src/cfnlint/rules/resources/lmbd/EventsLogGroupName.py) | Check for SubscriptionFilters have beyond 2 attachments to a CloudWatch Log Group | The current limit for a CloudWatch Log Group is they can have 2 subscription filters. We will look for duplicate LogGroupNames inside Subscription Filters and make sure they are within 2. This doesn't account for any other subscription filters getting set. |  | [Source](https://github.com/awslabs/serverless-application-model/blob/master/versions/2016-10-31.md#user-content-cloudwatchlogs) | `resources`,`lambda` |
| [E2530<a name="E2530"></a>](../src/cfnlint/rules/resources/lmbd/SnapStartSupported.py) | SnapStart supports the configured runtime | To properly leverage SnapStart, you must have a runtime of Java11 or greater |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html) | `resources`,`lambda` |
| [E2531<a name="E2531"></a>](../src/cfnlint/rules/resources/lmbd/DeprecatedRuntimeCreate.py) | Validate if lambda runtime is deprecated | Check the lambda runtime has reached the end of life |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html) | `resources`,`lambda`,`runtime` |
| [E2533<a name="E2533"></a>](../src/cfnlint/rules/resources/lmbd/DeprecatedRuntimeUpdate.py) | Check if Lambda Function Runtimes are updatable | Check if an EOL Lambda Runtime is specified and you cannot update the function |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/runtime-support-policy.html) | `resources`,`lambda`,`runtime` |
| [E2540<a name="E2540"></a>](../src/cfnlint/rules/resources/codepipeline/CodepipelineStages.py) | CodePipeline Stages | See if CodePipeline stages are set correctly |  | [Source](https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements) | `properties`,`codepipeline` |
| [E2541<a name="E2541"></a>](../src/cfnlint/rules/resources/codepipeline/CodepipelineStageActions.py) | CodePipeline Stage Actions | See if CodePipeline stage actions are set correctly |  | [Source](https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements) | `resources`,`codepipeline` |
| [E3001<a name="E3001"></a>](../src/cfnlint/rules/resources/Configuration.py) | Basic CloudFormation Resource Check | Making sure the basic CloudFormation resources are properly configured |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `resources` |
| [E3002<a name="E3002"></a>](../src/cfnlint/rules/resources/properties/Properties.py) | Resource properties are invalid | Making sure that resources properties are properly configured |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#properties) | `resources` |
| [E3003<a name="E3003"></a>](../src/cfnlint/rules/resources/properties/Required.py) | Required Resource properties are missing | Make sure that Resources properties that are required exist |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#required) | `resources`,`properties`,`required` |
| [E3004<a name="E3004"></a>](../src/cfnlint/rules/resources/CircularDependency.py) | Resource dependencies are not circular | Check that Resources are not circularly dependent by DependsOn, Ref, Sub, or GetAtt |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `resources`,`circularly`,`dependson`,`ref`,`sub`,`getatt` |
| [E3005<a name="E3005"></a>](../src/cfnlint/rules/resources/DependsOn.py) | Check DependsOn values for Resources | Check that the DependsOn values are valid |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html) | `resources`,`dependson` |
| [E3006<a name="E3006"></a>](../src/cfnlint/rules/resources/ResourceType.py) | Validate the CloudFormation resource type | Resource types are validated against the spec accounting for regions |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `resources` |
| [E3007<a name="E3007"></a>](../src/cfnlint/rules/resources/UniqueNames.py) | Unique resource and parameter names | All resources and parameters must have unique names |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html) | `parameters`,`resources` |
| [E3008<a name="E3008"></a>](../src/cfnlint/rules/resources/properties/PrefixItems.py) | Validate an array in order | Will validate arrays in order for schema validation |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#prefixitems) | `resources`,`properties`,`array`,`prefixItems` |
| [E3009<a name="E3009"></a>](../src/cfnlint/rules/resources/CfnInit.py) | Check CloudFormation init configuration | Validate that the items in a CloudFormation init adhere to standards |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html#aws-resource-cloudformation-init-syntax) | `resources`,`cloudformation init` |
| [E3010<a name="E3010"></a>](../src/cfnlint/rules/resources/MaxProperties.py) | Resource limit not exceeded | Check the number of Resources in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `resources`,`limits` |
| [E3011<a name="E3011"></a>](../src/cfnlint/rules/resources/PropertyNames.py) | Check property names in Resources | Validate property names are property configured in Resources |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `resources`,`limits` |
| [E3012<a name="E3012"></a>](../src/cfnlint/rules/resources/properties/Type.py) | Check resource properties values | Checks resource property values with Primitive Types for values that match those types. | strict:boolean:False | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#type) | `resources` |
| [E3013<a name="E3013"></a>](../src/cfnlint/rules/resources/cloudfront/Aliases.py) | CloudFront Aliases | CloudFront aliases should contain valid domain names |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-aliases) | `properties`,`cloudfront` |
| [E3014<a name="E3014"></a>](../src/cfnlint/rules/resources/properties/RequiredXor.py) | Validate only one of a set of required properties are specified | Make sure that Resources properties that are required exist. Along with other properties not being specified |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#requiredxor) | `resources` |
| [E3015<a name="E3015"></a>](../src/cfnlint/rules/resources/Condition.py) | Validate the resource condition is valid | Check the condition of a resource to make sure it exists inside the template |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html) | `resources`,`conditions` |
| [E3016<a name="E3016"></a>](../src/cfnlint/rules/resources/updatepolicy/Configuration.py) | Check the configuration of a resources UpdatePolicy | Make sure a resources UpdatePolicy is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatepolicy.html) | `resources`,`updatepolicy` |
| [E3017<a name="E3017"></a>](../src/cfnlint/rules/resources/properties/AnyOf.py) | Check Properties that need at least one of a list of properties | Making sure CloudFormation properties that require at least one property from a list. More than one can be included. |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `resources` |
| [E3018<a name="E3018"></a>](../src/cfnlint/rules/resources/properties/OneOf.py) | Check Properties that need only one of a list of properties | Making sure CloudFormation properties that require only one property from a list. One has to be specified. |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `resources` |
| [E3019<a name="E3019"></a>](../src/cfnlint/rules/resources/PrimaryIdentifiers.py) | Validate that all resources have unique primary identifiers | Use the primary identifiers in a resource schema to validate that resources inside the template are unique |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html) | `parameters`,`resources` |
| [E3020<a name="E3020"></a>](../src/cfnlint/rules/resources/properties/DependentExcluded.py) | Validate that when a property is specified another property should be excluded | When certain properties are specified other properties should not be included |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#dependentexcluded) | `resources` |
| [E3021<a name="E3021"></a>](../src/cfnlint/rules/resources/properties/DependentRequired.py) | Validate that when a property is specified that other properties should be included | When certain properties are specified it results in other properties to be required |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#dependentrequired) | `resources` |
| [E3022<a name="E3022"></a>](../src/cfnlint/rules/resources/ectwo/RouteTableAssociation.py) | Resource SubnetRouteTableAssociation Properties | Validate there is only one SubnetRouteTableAssociation per subnet |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-subnet-route-table-assoc.html) | `resources`,`ec2`,`subnet`,`route table` |
| [E3023<a name="E3023"></a>](../src/cfnlint/rules/resources/route53/RecordSet.py) | Validate Route53 RecordSets | Check if all RecordSets are correctly configured |  | [Source](https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html) | `resources`,`route53`,`record_set` |
| [E3024<a name="E3024"></a>](../src/cfnlint/rules/resources/properties/Tagging.py) | Validate tag configuration | Validates tag values to make sure they have unique keys and they follow pattern requirements |  | [Source](https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html) | `parameters`,`resources`,`tags` |
| [E3025<a name="E3025"></a>](../src/cfnlint/rules/resources/rds/DbInstanceDbInstanceClassEnum.py) | Validates RDS DB Instance Class | Validates the RDS DB instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3026<a name="E3026"></a>](../src/cfnlint/rules/resources/elasticache/CacheClusterFailover.py) | Check Elastic Cache Redis Cluster settings | Evaluate Redis Cluster groups to make sure automatic failover is enabled when cluster mode is enabled |  | [Source](https://github.com/awslabs/cfn-lint) | `resources`,`elasticcache` |
| [E3027<a name="E3027"></a>](../src/cfnlint/rules/resources/events/RuleScheduleExpression.py) | Validate AWS Event ScheduleExpression format | Validate the formation of the AWS::Event ScheduleExpression |  | [Source](https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/ScheduledEvents.html) | `resources`,`events` |
| [E3028<a name="E3028"></a>](../src/cfnlint/rules/resources/Metadata.py) | Validate the metadata section of a resource | The metadata section can be unstructured but we do validate the items we can |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `resources` |
| [E3029<a name="E3029"></a>](../src/cfnlint/rules/resources/route53/RecordSetAlias.py) | Validate Route53 record set aliases | When using alias records you can't specify TTL or certain types are allowed |  | [Source](https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html) | `resources`,`route53`,`record_set` |
| [E3030<a name="E3030"></a>](../src/cfnlint/rules/resources/properties/Enum.py) | Check if properties have a valid value | Check if properties have a valid value in case of an enumator |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#enum) | `resources`,`property`,`allowed value` |
| [E3031<a name="E3031"></a>](../src/cfnlint/rules/resources/properties/Pattern.py) | Check if property values adhere to a specific pattern | Check if properties have a valid value in case of a pattern (Regular Expression) | exceptions:list:[] | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#pattern) | `resources`,`property`,`allowed pattern`,`regex` |
| [E3032<a name="E3032"></a>](../src/cfnlint/rules/resources/properties/ArrayLength.py) | Check if a array has between min and max number of values specified | Check array for the number of items in the list to validate they are between the minimum and maximum |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#arraylength) | `resources`,`property`,`array`,`length` |
| [E3033<a name="E3033"></a>](../src/cfnlint/rules/resources/properties/StringLength.py) | Check if a string has between min and max number of values specified | Check strings for its length between the minimum and maximum |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#length) | `resources`,`property`,`string`,`size` |
| [E3034<a name="E3034"></a>](../src/cfnlint/rules/resources/properties/NumberRange.py) | Check if a number is between min and max | Check numbers (integers and floats) for its value being between the minimum and maximum |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#number-range) | `resources`,`property`,`number`,`size` |
| [E3035<a name="E3035"></a>](../src/cfnlint/rules/resources/DeletionPolicy.py) | Check DeletionPolicy values for Resources | Check that the DeletionPolicy values are valid |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html) | `resources`,`deletionpolicy` |
| [E3036<a name="E3036"></a>](../src/cfnlint/rules/resources/UpdateReplacePolicy.py) | Check UpdateReplacePolicy values for Resources | Check that the UpdateReplacePolicy values are valid |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatereplacepolicy.html) | `resources`,`updatereplacepolicy` |
| [E3037<a name="E3037"></a>](../src/cfnlint/rules/resources/properties/UniqueItems.py) | Check if a list has duplicate values | Certain lists don't support duplicate items. Check when duplicates are provided but not supported. |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#uniqueitems) | `resources`,`property`,`list` |
| [E3038<a name="E3038"></a>](../src/cfnlint/rules/resources/ServerlessTransform.py) | Check if Serverless Resources have Serverless Transform | Check that a template with Serverless Resources also includes the Serverless Transform |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-aws-serverless.html) | `resources`,`transform` |
| [E3039<a name="E3039"></a>](../src/cfnlint/rules/resources/dynamodb/AttributeMismatch.py) | AttributeDefinitions / KeySchemas mismatch | Verify the set of Attributes in AttributeDefinitions and KeySchemas match |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-dynamodb-table.html) | `resources`,`dynamodb` |
| [E3040<a name="E3040"></a>](../src/cfnlint/rules/resources/properties/ReadOnly.py) | Validate we aren't configuring read only properties | Read only properties can be configured in a CloudFormation template but they aren't sent to the resource provider code and can cause drift. |  | [Source](https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html#schema-properties-readonlyproperties) | `resources`,`properties` |
| [E3041<a name="E3041"></a>](../src/cfnlint/rules/resources/route53/RecordSetName.py) | RecordSet HostedZoneName is a superdomain of or equal to Name | In a RecordSet, the HostedZoneName must be a superdomain of or equal to the Name being validated |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-route53-recordset.html#cfn-route53-recordset-name) | `resource`,`properties`,`route53` |
| [E3042<a name="E3042"></a>](../src/cfnlint/rules/resources/ecs/TaskDefinitionEssentialContainer.py) | Validate at least one essential container is specified | Check that every TaskDefinition specifies at least one essential container |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-essential) | `properties`,`ecs`,`task`,`container`,`fargate` |
| [E3043<a name="E3043"></a>](../src/cfnlint/rules/resources/cloudformation/NestedStackParameters.py) | Validate parameters for in a nested stack | Evalute if parameters for a nested stack are specified and if parameters are specified for a nested stack that aren't required. |  | [Source](https://github.com/awslabs/cfn-lint) | `resources`,`cloudformation` |
| [E3044<a name="E3044"></a>](../src/cfnlint/rules/resources/ecs/FargateDeploymentSchedulingStrategy.py) | ECS service using FARGATE or EXTERNAL can only use SchedulingStrategy of REPLICA | When using a LaunchType of Fargate the SchedulingStrategy has to be Replica |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-schedulingstrategy) | `properties`,`ecs`,`service`,`container`,`fargate` |
| [E3045<a name="E3045"></a>](../src/cfnlint/rules/resources/s3/AccessControlOwnership.py) | Validate AccessControl are set with OwnershipControls | When using AccessControl other than private you must also configure OwnershipControls. The default is bucket owner enforced which disables ACLs. |  | [Source](https://docs.aws.amazon.com/AmazonS3/latest/userguide/about-object-ownership.html) | `resources`,`s3` |
| [E3046<a name="E3046"></a>](../src/cfnlint/rules/resources/ecs/LogConfiguration.py) | Validate ECS task logging configuration for awslogs | When 'awslogs' the options 'awslogs-group' and 'awslogs-region' are required |  | [Source]() | `resources` |
| [E3047<a name="E3047"></a>](../src/cfnlint/rules/resources/ecs/FargateCpuMemory.py) | Validate ECS Fargate tasks have the right combination of CPU and memory | When using a ECS Fargate task there is a specfic combination of memory and cpu that can be used |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-memory) | `properties`,`ecs`,`service`,`container`,`fargate` |
| [E3048<a name="E3048"></a>](../src/cfnlint/rules/resources/ecs/TaskFargateProperties.py) | Validate ECS Fargate tasks have required properties and values | When using a ECS Fargate task there is a specfic combination of required properties and values |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-memory) | `properties`,`ecs`,`service`,`container`,`fargate` |
| [E3049<a name="E3049"></a>](../src/cfnlint/rules/resources/ecs/ServiceDynamicPorts.py) | Validate ECS tasks with dynamic host port have traffic-port ELB target groups | When using an ECS task definition of host port 0 and associating that container to an ELB the target group has to have a 'HealthCheckPort' of 'traffic-port' |  | [Source]() | `resources` |
| [E3050<a name="E3050"></a>](../src/cfnlint/rules/resources/iam/RefWithPath.py) | Check if REFing to a IAM resource with path set | Some resources don't support looking up the IAM resource by name. This check validates when a REF is being used and the Path is not '/' |  | [Source](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements.html) | `properties`,`iam` |
| [E3051<a name="E3051"></a>](../src/cfnlint/rules/resources/ssm/Document.py) | Validate the structure of a SSM document | SSM documents are nested JSON/YAML in CloudFormation this rule adds validation to those documents |  | [Source](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements.html) | `properties`,`ssm`,`document` |
| [E3052<a name="E3052"></a>](../src/cfnlint/rules/resources/ecs/ServiceNetworkConfiguration.py) | Validate ECS service requires NetworkConfiguration | When using an ECS task definition has NetworkMode set to 'awsvpc' then 'NetworkConfiguration' is required |  | [Source]() | `resources`,`ecs` |
| [E3053<a name="E3053"></a>](../src/cfnlint/rules/resources/ecs/TaskDefinitionAwsVpc.py) | Validate ECS task definition is has correct values for 'HostPort' | The 'HostPort' must either be undefined or equal to the 'ContainerPort' value |  | [Source]() | `resources`,`ecs` |
| [E3054<a name="E3054"></a>](../src/cfnlint/rules/resources/ecs/ServiceFargate.py) | Validate ECS service using Fargate uses TaskDefinition that allows Fargate | When using an ECS service with 'LaunchType' of 'FARGATE' the associated task definition must have 'RequiresCompatibilities' specified with 'FARGATE' listed |  | [Source]() | `resources`,`ecs` |
| [E3055<a name="E3055"></a>](../src/cfnlint/rules/resources/CreationPolicy.py) | Check CreationPolicy values for Resources | Check that the CreationPolicy values are valid |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html) | `resources`,`creationPolicy` |
| [E3056<a name="E3056"></a>](../src/cfnlint/rules/resources/ecs/ServiceHealthCheckGracePeriodSeconds.py) | ECS service using HealthCheckGracePeriodSeconds must also have LoadBalancers specified | When using a HealthCheckGracePeriodSeconds on an ECS service, the service must also have a LoadBalancers specified with at least one LoadBalancer in the array. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-healthcheckgraceperiodseconds) | `properties`,`ecs`,`service`,`container` |
| [E3057<a name="E3057"></a>](../src/cfnlint/rules/resources/cloudfront/DistributionTargetOriginId.py) | Validate that CloudFront TargetOriginId is a specified Origin | CloudFront TargetOriginId has to map to an Origin Id that is in the same DistributionConfig |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-targetoriginid) | `properties`,`cloudfront` |
| [E3058<a name="E3058"></a>](../src/cfnlint/rules/resources/properties/RequiredOr.py) | Validate at least one of the properties are required | Make sure at least one of the resource properties are included |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#requiredor) | `resources` |
| [E3503<a name="E3503"></a>](../src/cfnlint/rules/resources/certificatemanager/DomainValidationOptions.py) | ValidationDomain is superdomain of DomainName | In ValidationDomainOptions, the ValidationDomain must be a superdomain of the DomainName being validated |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-certificatemanager-certificate-domainvalidationoption.html#cfn-certificatemanager-certificate-domainvalidationoption-validationdomain) | `certificate`,`certificatemanager`,`domainvalidationoptions`,`validationdomain` |
| [E3504<a name="E3504"></a>](../src/cfnlint/rules/resources/backup/BackupPlanLifecycleRule.py) | Check minimum 90 period is met between BackupPlan cold and delete | Check that Backup plans with lifecycle rules have >= 90 days between cold and delete |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-backup-backupplan-lifecycleresourcetype.html) | `properties`,`backup`,`plan`,`lifecycle` |
| [E3510<a name="E3510"></a>](../src/cfnlint/rules/resources/iam/IdentityPolicy.py) | Validate identity based IAM polices | IAM identity polices are embedded JSON in CloudFormation. This rule validates those embedded policies. |  | [Source](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_identity-vs-resource.html) | `resources`,`iam` |
| [E3511<a name="E3511"></a>](../src/cfnlint/rules/resources/iam/RoleArnPattern.py) | Validate IAM role arn pattern | Validate an IAM role arn pattern matches |  | [Source](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html) | `parameters`,`iam` |
| [E3512<a name="E3512"></a>](../src/cfnlint/rules/resources/iam/ResourcePolicy.py) | Validate resource based IAM polices | IAM resources polices are embedded JSON in CloudFormation. This rule validates those embedded policies. |  | [Source](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_identity-vs-resource.html) | `resources`,`iam` |
| [E3513<a name="E3513"></a>](../src/cfnlint/rules/resources/iam/ResourceEcrPolicy.py) | Validate ECR repository policy | Private ECR repositories have a policy. This rule validates those policies. |  | [Source](https://docs.aws.amazon.com/AmazonECR/latest/userguide/repository-policies.html) | `resources`,`iam`,`ecr` |
| [E3601<a name="E3601"></a>](../src/cfnlint/rules/resources/stepfunctions/StateMachineDefinition.py) | Validate the structure of a StateMachine definition | Validate the Definition or DefinitionString inside a AWS::StepFunctions::StateMachine resource |  | [Source](https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-state-machine-structure.html) | `resources`,`statemachine` |
| [E3615<a name="E3615"></a>](../src/cfnlint/rules/resources/cloudwatch/AlarmPeriod.py) | Validate the period is a valid value | Valid values are 10, 30, 60, and any multiple of 60. |  | [Source]() | `resources`,`cloudwatch` |
| [E3617<a name="E3617"></a>](../src/cfnlint/rules/resources/managedblockchain/NodeNodeConfigurationInstanceTypeEnum.py) | Validate ManagedBlockchain instance type | Validates the ManagedBlockchain instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3620<a name="E3620"></a>](../src/cfnlint/rules/resources/docdb/DBInstanceClassEnum.py) | Validate a DocDB DB Instance class | Validates the DocDB instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3621<a name="E3621"></a>](../src/cfnlint/rules/resources/appstream/FleetInstanceTypeEnum.py) | Validate the instance types for AppStream Fleet | Validates the AppStream Fleet instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3628<a name="E3628"></a>](../src/cfnlint/rules/resources/ectwo/InstanceInstanceTypeEnum.py) | Validate EC2 instance types based on region | Validates the EC2 instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3633<a name="E3633"></a>](../src/cfnlint/rules/resources/lmbd/EventSourceMappingEventSourceArnStreamInclusive.py) | Validate Lambda event source mapping StartingPosition is used correctly | When 'EventSourceArn' is associate to Kinesis, Kafka, or DynamoDB you must specify 'StartingPosition |  | [Source]() | `resources` |
| [E3634<a name="E3634"></a>](../src/cfnlint/rules/resources/lmbd/EventSourceMappingEventSourceArnSqsExclusive.py) | Validate Lambda event source mapping starting position is used with SQS | When 'EventSourceArn' is associated to SQS don't specify 'StartingPosition' |  | [Source]() | `resources` |
| [E3635<a name="E3635"></a>](../src/cfnlint/rules/resources/neptune/DBInstanceDBInstanceClassEnum.py) | Validate Neptune DB instance class | Validates the instance types for Neptune DB based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3638<a name="E3638"></a>](../src/cfnlint/rules/resources/dynamodb/TableBillingModeExclusive.py) | Validate DynamoDB BillingMode pay per request configuration | When 'BillingMode' is 'PAY_PER_REQUEST' don't specify 'ProvisionedThroughput' |  | [Source]() | `resources` |
| [E3639<a name="E3639"></a>](../src/cfnlint/rules/resources/dynamodb/TableBillingModeProvisioned.py) | When BillingMode is Provisioned you must specify ProvisionedThroughput | When 'BillingMode' is 'Provisioned' 'ProvisionedThroughput' is required |  | [Source]() | `resources` |
| [E3641<a name="E3641"></a>](../src/cfnlint/rules/resources/gamelift/FleetEc2InstanceTypeEnum.py) | Validate GameLift Fleet EC2 instance type | Validates the GameLift Fleet EC2 instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3647<a name="E3647"></a>](../src/cfnlint/rules/resources/elasticache/CacheClusterCacheNodeTypeEnum.py) | Validate ElastiCache cluster cache node type | Validates the ElastiCache instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3652<a name="E3652"></a>](../src/cfnlint/rules/resources/elasticsearch/DomainClusterConfigInstanceTypeEnum.py) | Validate Elasticsearch domain cluster instance | Validates the Elasticsearch instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3660<a name="E3660"></a>](../src/cfnlint/rules/resources/apigateway/RestApiOpenApi.py) | RestApi requires a name when not using an OpenAPI specification | When using AWS::ApiGateway::RestApi you have to provide 'Name' if you don't provide 'Body' or 'BodyS3Location' |  | [Source]() | `resources`,`apigateway` |
| [E3661<a name="E3661"></a>](../src/cfnlint/rules/resources/route53/HealthCheckHealthCheckConfigTypeInclusive.py) | Validate Route53 health check has AlarmIdentifier when using CloudWatch | When 'Type' is 'CLOUDWATCH_METRIC' you must specify 'AlarmIdentifier' |  | [Source]() | `resources` |
| [E3663<a name="E3663"></a>](../src/cfnlint/rules/resources/lmbd/FunctionEnvironmentKeys.py) | Validate Lambda environment variable names aren't reserved | Lambda reserves a set of environment variable names for its use. This rule validates that the provided environment variable names don't use the reserved variable names |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html) | `resources`,`lambda`,`runtime` |
| [E3667<a name="E3667"></a>](../src/cfnlint/rules/resources/redshift/ClusterNodeTypeEnum.py) | Validate RedShift cluster node type | Validates the RedShift instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3670<a name="E3670"></a>](../src/cfnlint/rules/resources/amazonmq/BrokerInstanceTypeEnum.py) | Validate the instance types for an AmazonMQ Broker | Validates the instance types for AmazonMQ broker based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3671<a name="E3671"></a>](../src/cfnlint/rules/resources/ectwo/Ebs.py) | Validate block device mapping configuration | Certain volume types require Iops to be specified |  | [Source]() | `resources`,`ec2` |
| [E3672<a name="E3672"></a>](../src/cfnlint/rules/resources/dax/ClusterNodeTypeEnum.py) | Validate the cluster node type for a DAX Cluster | Validates the DAX cluster instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3673<a name="E3673"></a>](../src/cfnlint/rules/resources/ectwo/InstanceImageId.py) | Validate if an ImageId is required | Validate if an ImageID is required. It can be required if the associated LaunchTemplate doesn't specify an ImageID |  | [Source]() | `resources`,`ec2` |
| [E3674<a name="E3674"></a>](../src/cfnlint/rules/resources/ectwo/PrivateIpWithNetworkInterface.py) | Primary cannoy be True when PrivateIpAddress is specified | Only specify the private IP address for an instance in one spot |  | [Source]() | `resources`,`ec2` |
| [E3675<a name="E3675"></a>](../src/cfnlint/rules/resources/emr/ClusterInstanceTypeConfigInstanceTypeEnum.py) | Validate EMR cluster instance type | Validates the EMR cluster instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E3676<a name="E3676"></a>](../src/cfnlint/rules/resources/elasticloadbalancingv2/ListenerCertificate.py) | Validate ELBv2 protocols that require certificates have a certificate specified | When using HTTP or TLS you must specify a certificate |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-elasticloadbalancingv2-listener.html#cfn-elasticloadbalancingv2-listener-certificates) | `resources` |
| [E3677<a name="E3677"></a>](../src/cfnlint/rules/resources/lmbd/FunctionZipfileRuntimeEnum.py) | Validate Lambda using ZipFile requires an allowable runtime | Using the ZipFile attribute requires a javascript or python runtime to be specified |  | [Source]() | `resources` |
| [E3678<a name="E3678"></a>](../src/cfnlint/rules/resources/lmbd/FunctionZipfileRuntimeExists.py) | Using the ZipFile attribute requires a runtime to be specified | Using the ZipFile attribute requires a runtime to be specified |  | [Source]() | `resources` |
| [E3679<a name="E3679"></a>](../src/cfnlint/rules/resources/elb/Certificate.py) | Validate ELB protocols that require certificates have a certificate specified | When using HTTPS or SSL you must provide a certificate |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-elasticloadbalancing-loadbalancer-listeners.html#cfn-elasticloadbalancing-loadbalancer-listeners-sslcertificateid) | `resources` |
| [E3680<a name="E3680"></a>](../src/cfnlint/rules/resources/elasticloadbalancingv2/LoadBalancerApplicationSubnets.py) | Application load balancers require at least 2 subnets |  |  | [Source]() | `resources` |
| [E3682<a name="E3682"></a>](../src/cfnlint/rules/resources/rds/DbInstanceAuroraExclusive.py) | Validate when using Aurora certain properies aren't required | When creating an aurora DBInstance don't specify 'AllocatedStorage', 'BackupRetentionPeriod', 'CopyTagsToSnapshot', 'DeletionProtection', 'EnableIAMDatabaseAuthentication', 'MasterUserPassword', or 'StorageEncrypted' |  | [Source]() | `resources` |
| [E3686<a name="E3686"></a>](../src/cfnlint/rules/resources/rds/DbClusterServerlessExclusive.py) | Validate allowed properties when using a serverless RDS DB cluster | Validate that when EngineMode is 'serverless' or 'provisioned' that the appropriate allowed properties are provided. If 'EngineMode' is not provided make sure serverless properties don't exist at all. |  | [Source]() | `resources` |
| [E3687<a name="E3687"></a>](../src/cfnlint/rules/resources/ectwo/SecurityGroupProtocolsAndPortsInclusive.py) | Validate to and from ports based on the protocol | When using  icmp, icmpv6, tcp, or udp you have to specify the to and from port ranges |  | [Source]() | `resources` |
| [E3688<a name="E3688"></a>](../src/cfnlint/rules/resources/ectwo/SecurityGroupAllToAndFromPorts.py) | Validate that to and from ports are both -1 | When ToPort or FromPort are -1 the other one must also be -1 |  | [Source]() | `resources` |
| [E3689<a name="E3689"></a>](../src/cfnlint/rules/resources/rds/DbClusterMonitoring.py) | Validate MonitoringInterval and MonitoringRoleArn are used together | When MonitoringInterval is greater than 0 you need to specify MonitoringRoleArn. If MonitoringRoleArn is specified MonitoringInterval has to be greather than 0. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-monitoringinterval) | `resources` |
| [E3690<a name="E3690"></a>](../src/cfnlint/rules/resources/rds/DbClusterEngineVersion.py) | Validate DB Cluster Engine and Engine Version | Validate the DB Cluster engine along with the engine version |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engineversion) | `resources` |
| [E3691<a name="E3691"></a>](../src/cfnlint/rules/resources/rds/DbInstanceEngineVersion.py) | Validate DB Instance Engine and Engine Version | Validate the DB Instance engine along with the engine version |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-engine) | `resources` |
| [E3692<a name="E3692"></a>](../src/cfnlint/rules/resources/rds/DbClusterMultiAz.py) | Validate Multi-AZ DB cluster configuration | When creating a Multi-AZ DB Cluster there are required fields and the allowed values are different |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engineversion) | `resources` |
| [E3693<a name="E3693"></a>](../src/cfnlint/rules/resources/rds/DbClusterAurora.py) | Validate Aurora DB cluster configuration | When creating an Aurora DB Cluster there are required fields and the allowed values are different |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engineversion) | `resources` |
| [E3694<a name="E3694"></a>](../src/cfnlint/rules/resources/rds/DbClusterInstanceClassEnum.py) | Validates RDS DB Cluster instance class | Validates the RDS DB Cluster instance types based on region and data gathered from the pricing APIs |  | [Source]() | `resources` |
| [E4001<a name="E4001"></a>](../src/cfnlint/rules/metadata/InterfaceConfiguration.py) | Metadata Interface have appropriate properties | Metadata Interface properties are properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-interface.html) | `metadata` |
| [E4002<a name="E4002"></a>](../src/cfnlint/rules/metadata/Configuration.py) | Validate the configuration of the Metadata section | Validates that Metadata section is an object and has no null values |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/metadata-section-structure.html) | `metadata` |
| [E5001<a name="E5001"></a>](../src/cfnlint/rules/resources/Modules.py) | Check that Modules resources are valid | Check that Modules resources are valid |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/) | `resources`,`modules` |
| [E6001<a name="E6001"></a>](../src/cfnlint/rules/outputs/Configuration.py) | Check the properties of Outputs | Validate the property structure for outputs |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E6002<a name="E6002"></a>](../src/cfnlint/rules/outputs/Required.py) | Outputs have required properties | Making sure the outputs have required properties |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E6003<a name="E6003"></a>](../src/cfnlint/rules/outputs/Type.py) | Check the type of Outputs | Validate the type of properties in the Outputs section |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E6004<a name="E6004"></a>](../src/cfnlint/rules/outputs/Pattern.py) | Outputs have appropriate names | Check if Outputs are properly named (A-Za-z0-9) |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E6005<a name="E6005"></a>](../src/cfnlint/rules/outputs/Condition.py) | Validate the Output condition is valid | Check the condition of an output to make sure it exists inside the template |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs`,`conditions` |
| [E6010<a name="E6010"></a>](../src/cfnlint/rules/outputs/MaxProperties.py) | Output limit not exceeded | Check the number of Outputs in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `outputs`,`limits` |
| [E6011<a name="E6011"></a>](../src/cfnlint/rules/outputs/MaxLength.py) | Check property names in Outputs | Validate property names are property configured in Outputs |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs`,`limits` |
| [E6101<a name="E6101"></a>](../src/cfnlint/rules/outputs/Value.py) | Validate that outputs values are a string | Make sure that output values have a type of string |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E6102<a name="E6102"></a>](../src/cfnlint/rules/outputs/Export.py) | Validate that output exports have values of strings | Make sure output exports have a value of type string |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E7001<a name="E7001"></a>](../src/cfnlint/rules/mappings/Configuration.py) | Mappings are appropriately configured | Check if Mappings are properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html) | `mappings` |
| [E7002<a name="E7002"></a>](../src/cfnlint/rules/mappings/MaxLength.py) | Check property names in Mappings | Validate property names are property configured in Mappings |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html) | `mappings`,`limits` |
| [E7010<a name="E7010"></a>](../src/cfnlint/rules/mappings/MaxProperties.py) | Max number of properties for Mappings | Check the number of Mappings in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `mappings`,`limits` |
| [E8001<a name="E8001"></a>](../src/cfnlint/rules/conditions/Configuration.py) | Conditions have appropriate properties | Check if Conditions are properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/conditions-section-structure.html) | `conditions` |
| [E8002<a name="E8002"></a>](../src/cfnlint/rules/conditions/Exists.py) | Check if the referenced Conditions are defined | Making sure the used conditions are actually defined in the Conditions section |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/conditions-section-structure.html) | `conditions` |
| [E8003<a name="E8003"></a>](../src/cfnlint/rules/conditions/Equals.py) | Check Fn::Equals structure for validity | Check Fn::Equals is a list of two elements |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-equals) | `functions`,`equals` |
| [E8004<a name="E8004"></a>](../src/cfnlint/rules/conditions/And.py) | Check Fn::And structure for validity | Check Fn::And is a list of two elements |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-and) | `functions`,`and` |
| [E8005<a name="E8005"></a>](../src/cfnlint/rules/conditions/Not.py) | Check Fn::Not structure for validity | Check Fn::Not is a list of one element |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-not) | `functions`,`not` |
| [E8006<a name="E8006"></a>](../src/cfnlint/rules/conditions/Or.py) | Check Fn::Or structure for validity | Check Fn::Or is a list of two elements |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-or) | `functions`,`or` |
| [E8007<a name="E8007"></a>](../src/cfnlint/rules/conditions/Condition.py) | Check Condition structure for validity | Check Condition has a value of another condition |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-and) | `functions`,`and` |
| [I1002<a name="I1002"></a>](../src/cfnlint/rules/templates/ApproachingLimitSize.py) | Validate approaching the template size limit | Check the size of the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `limits` |
| [I1003<a name="I1003"></a>](../src/cfnlint/rules/templates/ApproachingLimitDescription.py) | Validate if we are approaching the max size of a description | Check if the size of the template description is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `description`,`limits` |
| [I1022<a name="I1022"></a>](../src/cfnlint/rules/functions/SubNotJoin.py) | Use Sub instead of Join | Prefer a sub instead of Join when using a join delimiter that is empty |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub`,`join` |
| [I2010<a name="I2010"></a>](../src/cfnlint/rules/parameters/ApproachingMaxProperties.py) | Parameter limit | Check the number of Parameters in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`limits` |
| [I2011<a name="I2011"></a>](../src/cfnlint/rules/parameters/ApproachingMaxLength.py) | Parameter name limit | Check the size of Parameter names in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`limits` |
| [I2530<a name="I2530"></a>](../src/cfnlint/rules/resources/lmbd/SnapStartEnabled.py) | Validate that SnapStart is configured for >= Java11 runtimes | SnapStart is a no-cost feature that can increase performance up to 10x. Enable SnapStart for Java 11 and greater runtimes |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html) | `resources`,`lambda` |
| [I3010<a name="I3010"></a>](../src/cfnlint/rules/resources/ApproachingMaxProperties.py) | Resource limit | Check the number of Resources in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `resources`,`limits` |
| [I3011<a name="I3011"></a>](../src/cfnlint/rules/resources/UpdateReplacePolicyDeletionPolicyOnStatefulResourceTypes.py) | Check stateful resources have a set UpdateReplacePolicy/DeletionPolicy | The default action when replacing/removing a resource is to delete it. This check requires you to explicitly set policies |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html) | `resources`,`updatereplacepolicy`,`deletionpolicy` |
| [I3012<a name="I3012"></a>](../src/cfnlint/rules/resources/ApproachingMaxLength.py) | Resource name limit | Check the size of Resource names in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `resources`,`limits` |
| [I3013<a name="I3013"></a>](../src/cfnlint/rules/resources/RetentionPeriodOnResourceTypesWithAutoExpiringContent.py) | Check resources with auto expiring content have explicit retention period | The behaviour for data retention is different across AWS Services.If no retention period is specified the default for some services is to delete the data after a period of time.This check requires you to explicitly set the retention period for those resources to avoid unexpected data losses |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `resources`,`retentionperiod` |
| [I3037<a name="I3037"></a>](../src/cfnlint/rules/resources/properties/UniqueItemsAllowed.py) | Check if a list that allows duplicates has any duplicates | Certain lists support duplicate items.Provide an alert when list of strings or numbers have repeats. |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/rules.md#rules-1) | `resources`,`property`,`list` |
| [I3042<a name="I3042"></a>](../src/cfnlint/rules/resources/HardCodedArnProperties.py) | ARNs should use correctly placed Pseudo Parameters | Checks Resources if ARNs use correctly placed Pseudo Parameters instead of hardcoded Partition, Region, and Account Number | partition:boolean:True<br />region:boolean:False<br />accountId:boolean:False | [Source]() | `resources` |
| [I3100<a name="I3100"></a>](../src/cfnlint/rules/resources/PreviousGenerationInstanceType.py) | Checks for legacy instance type generations | New instance type generations increase performance and decrease cost |  | [Source](https://aws.amazon.com/ec2/previous-generation/) | `resources`,`ec2`,`rds`,`elasticcache`,`elasticsearch` |
| [I6010<a name="I6010"></a>](../src/cfnlint/rules/outputs/ApproachingMaxProperties.py) | Output limit | Check the number of Outputs in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `outputs`,`limits` |
| [I6011<a name="I6011"></a>](../src/cfnlint/rules/outputs/ApproachingMaxLength.py) | Output name limit | Check the size of Output names in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `outputs`,`limits` |
| [I7002<a name="I7002"></a>](../src/cfnlint/rules/mappings/ApproachingMaxLength.py) | Mapping name limit | Check the size of Mapping names in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `mappings`,`limits` |
| [I7010<a name="I7010"></a>](../src/cfnlint/rules/mappings/ApproachingMaxProperties.py) | Mapping limit | Check the number of Mappings in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `mappings`,`limits` |
| [W1001<a name="W1001"></a>](../src/cfnlint/rules/functions/RelationshipConditions.py) | Ref/GetAtt to resource that is available when conditions are applied | Check the Conditions that affect a Ref/GetAtt to make sure the resource being related to is available when there is a resource condition. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-ref.html) | `conditions`,`resources`,`relationships`,`ref`,`getatt`,`sub` |
| [W1011<a name="W1011"></a>](../src/cfnlint/rules/parameters/DynamicReferenceSecret.py) | Instead of REFing a parameter for a secret use a dynamic reference | Instead of REFing a parameter for a secret use a dynamic reference. Solutions like SSM parameter store and secrets manager provide better security of sercrets |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/security-best-practices.html#creds) | `functions`,`dynamic reference`,`ref` |
| [W1019<a name="W1019"></a>](../src/cfnlint/rules/functions/SubParametersUsed.py) | Validate that parameters to a Fn::Sub are used | Validate that Fn::Sub Parameters are used |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub` |
| [W1020<a name="W1020"></a>](../src/cfnlint/rules/functions/SubUnneeded.py) | Sub isn't needed if it doesn't have a variable defined | Checks sub strings to see if a variable is defined. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub` |
| [W1028<a name="W1028"></a>](../src/cfnlint/rules/functions/IfUnsatisfiable.py) | Check Fn::If has a path that cannot be reached | Check Fn::If path can be reached |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-if) | `functions`,`if` |
| [W1030<a name="W1030"></a>](../src/cfnlint/rules/functions/RefResolved.py) | Validate the values that come from a Ref function | Resolve the Ref and then validate the values against the schema |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-ref.html) | `functions`,`ref` |
| [W1031<a name="W1031"></a>](../src/cfnlint/rules/functions/SubResolved.py) | Validate the values that come from a Fn::Sub function | Resolve the Fn::Sub and then validate the values against the schema |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub` |
| [W1032<a name="W1032"></a>](../src/cfnlint/rules/functions/JoinResolved.py) | Validate the values that come from a Fn::Join function | Resolve the Fn::Join and then validate the values against the schema |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-join.html) | `functions`,`join` |
| [W1033<a name="W1033"></a>](../src/cfnlint/rules/functions/SplitResolved.py) | Validate the values that come from a Fn::Split function | Resolve the Fn::Split and then validate the values against the schema |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-split.html) | `functions`,`split` |
| [W1034<a name="W1034"></a>](../src/cfnlint/rules/functions/FindInMapResolved.py) | Validate the values that come from a Fn::FindInMap function | Resolve the Fn::FindInMap and then validate the values against the schema |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-findinmap.html) | `functions`,`findinmap` |
| [W1035<a name="W1035"></a>](../src/cfnlint/rules/functions/SelectResolved.py) | Validate the values that come from a Fn::Select function | Resolve the Fn::Select and then validate the values against the schema |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-select.html) | `functions`,`select` |
| [W1036<a name="W1036"></a>](../src/cfnlint/rules/functions/GetAzResolved.py) | Validate the values that come from a Fn::GetAZs function | Resolve the Fn::GetAZs and then validate the values against the schema |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-getavailabilityzones.html) | `functions`,`getazs` |
| [W1040<a name="W1040"></a>](../src/cfnlint/rules/functions/ToJsonStringResolved.py) | Validate the values that come from a Fn::ToJsonString function | Resolve the Fn::ToJsonString and then validate the values against the schema |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-ToJsonString.html) | `functions`,`tojsonstring` |
| [W2001<a name="W2001"></a>](../src/cfnlint/rules/parameters/Used.py) | Check if Parameters are Used | Making sure the parameters defined are used |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `parameters` |
| [W2010<a name="W2010"></a>](../src/cfnlint/rules/parameters/NoEcho.py) | NoEcho parameters are not masked when used in Metadata and Outputs | Using the NoEcho attribute does not mask any information stored in the following: Metadata, Outputs, Resource Metadata |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html) | `functions`,`dynamic reference`,`ref` |
| [W2030<a name="W2030"></a>](../src/cfnlint/rules/parameters/Enum.py) | Check if parameters have a valid value | Check if parameters have a valid value in case of an enumator. The Parameter's allowed values is based on the usages in property (Ref) |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#enum) | `parameters`,`resources`,`property`,`allowed value` |
| [W2031<a name="W2031"></a>](../src/cfnlint/rules/parameters/ValuePattern.py) | Check if parameters have a valid value based on an allowed pattern | Check if parameters have a valid value in a pattern. The Parameter's allowed pattern is based on the usages in property (Ref) |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#pattern) | `parameters`,`resources`,`property`,`pattern` |
| [W2501<a name="W2501"></a>](../src/cfnlint/rules/resources/properties/Password.py) | Check if Password Properties are correctly configured | Password properties should not be strings and if parameter using NoEcho |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/best-practices.html#creds) | `parameters`,`passwords`,`security`,`dynamic reference` |
| [W2506<a name="W2506"></a>](../src/cfnlint/rules/resources/properties/ImageId.py) | Check if ImageId Parameters have the correct type | See if there are any refs for ImageId to a parameter of inappropriate type. Appropriate Types are [AWS::EC2::Image::Id, AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>] |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/best-practices.html#parmtypes) | `parameters`,`ec2`,`imageid` |
| [W2511<a name="W2511"></a>](../src/cfnlint/rules/resources/iam/PolicyVersion.py) | Check IAM Resource Policies syntax | See if the elements inside an IAM Resource policy are configured correctly. |  | [Source](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements.html) | `properties`,`iam` |
| [W2530<a name="W2530"></a>](../src/cfnlint/rules/resources/lmbd/SnapStart.py) | Validate that SnapStart is properly configured | To properly leverage SnapStart, you must configure both the lambda function and attach a Lambda version resource |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html) | `resources`,`lambda` |
| [W2531<a name="W2531"></a>](../src/cfnlint/rules/resources/lmbd/DeprecatedRuntimeEol.py) | Check if EOL Lambda Function Runtimes are used | Check if an EOL Lambda Runtime is specified and give a warning if used.  |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/runtime-support-policy.html) | `resources`,`lambda`,`runtime` |
| [W2533<a name="W2533"></a>](../src/cfnlint/rules/resources/lmbd/ZipPackageRequiredProperties.py) | Check required properties for Lambda if the deployment package is a .zip file | When the package type is Zip, you must also specify the `handler` and `runtime` properties. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html) | `resources`,`lambda` |
| [W3002<a name="W3002"></a>](../src/cfnlint/rules/resources/properties/PropertiesTemplated.py) | Warn when properties are configured to only work with the package command | Some properties can be configured to only work with the CloudFormationpackage command. Warn when this is the case so user is aware. |  | [Source](https://docs.aws.amazon.com/cli/latest/reference/cloudformation/package.html) | `resources` |
| [W3005<a name="W3005"></a>](../src/cfnlint/rules/resources/DependsOnObsolete.py) | Check obsolete DependsOn configuration for Resources | Check if DependsOn is specified if not needed. A Ref or a Fn::GetAtt already is an implicit dependency. |  | [Source](https://aws.amazon.com/blogs/devops/optimize-aws-cloudformation-templates/) | `resources`,`dependson`,`ref`,`getatt` |
| [W3010<a name="W3010"></a>](../src/cfnlint/rules/resources/properties/AvailabilityZone.py) | Availability zone properties should not be hardcoded | Check if an Availability Zone property is hardcoded. |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `parameters`,`availabilityzone` |
| [W3011<a name="W3011"></a>](../src/cfnlint/rules/resources/BothUpdateReplacePolicyDeletionPolicyNeeded.py) | Check resources with UpdateReplacePolicy/DeletionPolicy have both | Both UpdateReplacePolicy and DeletionPolicy are needed to protect resources from deletion |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html) | `resources`,`updatereplacepolicy`,`deletionpolicy` |
| [W3034<a name="W3034"></a>](../src/cfnlint/rules/parameters/NumberRange.py) | Check if parameter values are between min and max | Check if parameter values value being between the minimum and maximum |  | [Source](https://github.com/aws-cloudformation/cfn-lint/blob/main/docs/cfn-schema-specification.md#number-size) | `resources`,`property`,`number`,`size` |
| [W3037*<a name="W3037*"></a>](../src/cfnlint/rules/resources/iam/Permissions.py) | Check IAM Permission configuration | Check for valid IAM Permissions |  | [Source](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) | `properties`,`iam`,`permissions` |
| [W3045<a name="W3045"></a>](../src/cfnlint/rules/resources/s3/AccessControlObsolete.py) | Controlling access to an S3 bucket should be done with bucket policies | Nearly all access control configurations can be more successfully achieved with bucket policies. Consider using bucket policies instead of access control. |  | [Source](https://docs.aws.amazon.com/AmazonS3/latest/userguide/about-object-ownership.html) | `resources`,`s3` |
| [W3663<a name="W3663"></a>](../src/cfnlint/rules/resources/lmbd/PermissionSourceAccount.py) | Validate SourceAccount is required property | When configuration a Lambda permission with a SourceArn that doesn't have an AccountId you should also specify the SourceAccount |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-sourceaccount) | `resources`,`lambda`,`permission` |
| [W3687<a name="W3687"></a>](../src/cfnlint/rules/resources/ectwo/SecurityGroupProtocolsAndPortsExclusive.py) | Validate that ports aren't specified for certain protocols | When using a protocol other than icmp, icmpv6, tcp, or udp the port ranges properties are ignored |  | [Source]() | `resources` |
| [W3688<a name="W3688"></a>](../src/cfnlint/rules/resources/rds/DbClusterSnapshotIdentifier.py) | When restoring DBCluster certain properties are ignored | When creating a DBCluster extra properties are ignored and could result in drift |  | [Source]() | `resources`,`rds` |
| [W3689<a name="W3689"></a>](../src/cfnlint/rules/resources/rds/DbClusterSourceDBClusterIdentifier.py) | When using a source DB certain properties are ignored | When creating a DBCluster from a source certain properties are ignored and could result in drift |  | [Source]() | `resources`,`rds` |
| [W3693<a name="W3693"></a>](../src/cfnlint/rules/resources/rds/DbClusterAuroraWarning.py) | Validate Aurora DB cluster configuration for ignored properties | When creating an Aurora DB Cluster there are fields that will allow for successful deployment but are ignored |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engineversion) | `resources` |
| [W4001<a name="W4001"></a>](../src/cfnlint/rules/metadata/InterfaceParameterExists.py) | Metadata Interface parameters exist | Metadata Interface parameters actually exist |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-interface.html) | `metadata` |
| [W4005<a name="W4005"></a>](../src/cfnlint/rules/metadata/CfnLint.py) | Validate cfnlint configuration in the Metadata | Metadata cfn-lint configuration has many values and we want to validate that |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-interface.html) | `metadata` |
| [W6001<a name="W6001"></a>](../src/cfnlint/rules/outputs/ImportValue.py) | Check Outputs using ImportValue | Check if the Output value is set using ImportValue, so creating an Output of an Output |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `outputs`,`importvalue` |
| [W7001<a name="W7001"></a>](../src/cfnlint/rules/mappings/Used.py) | Check if Mappings are Used | Making sure the mappings defined are used |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `mappings` |
| [W8001<a name="W8001"></a>](../src/cfnlint/rules/conditions/Used.py) | Check if Conditions are Used | Making sure the conditions defined are used |  | [Source](https://github.com/aws-cloudformation/cfn-lint) | `conditions` |
| [W8003<a name="W8003"></a>](../src/cfnlint/rules/conditions/EqualsIsUseful.py) | Fn::Equals will always return true or false | Validate Fn::Equals to see if its comparing two strings or two equal items. While this works it may not be intended. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-equals) | `functions`,`equals` |

\* experimental rules
