# COBYLA Python interface
# @(#) $Jeannot: cobyla.py,v 1.7 2004/04/18 15:07:09 js Exp $

# Copyright (c) 2004, Jean-Sebastien Roy (js@jeannot.org)

# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:

# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
# CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
# TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

"""
COBYLA: A python interface to the COBYLA non-linear optimizer

COBYLA, a contrained optimization by linear approximation package developed by
Michael J. D. Powell, is a non-linear derivative-free optimizer. To use it, you
must provide a function to minimize. The function must take the list of
coordinates where to evaluate the function; and it must return either a tuple,
whose first element is the value of the function, and whose second argument is
the values of the constraints (as a list of values); or None, to abort the
minimization.

COBYLA will try to make all the values of the constraints positive.
So if you want to input a constraint j such as x[i] <= MAX, set::

  con[j] = MAX - x[i]
"""

import moduleCobyla

MSG_NONE = 0 # No messages
MSG_EXIT = 1 # Exit reasons
MSG_ITER = 2 # Rho and Sigma changes
MSG_INFO = 3 # Informational messages

MSGS = {
        MSG_NONE : "No messages",
        MSG_EXIT : "Exit reasons",
        MSG_ITER : "Rho and Sigma changes",
        MSG_INFO : "Informational messages",
}

HUGE_VAL=1e200*1e200 # No standard representation of Infinity in Python 2.3.3

NORMAL    = 0 # Normal return
MAXFUN    = 1 # Max. number of function evaluations reach
ROUNDING  = 2 # Rounding errors are becoming damaging
USERABORT = 3 # User requested end of minimization

RCSTRINGS = {
        NORMAL    : "Normal return",
        MAXFUN    : "Max. number of function evaluations reach",
        ROUNDING  : "Rounding errors are becoming damaging",
        USERABORT : "User requested end of minimization"
}

class BoundConstrained:
        """Private auxillary class for bound constrained problems"""
        def __init__(self, function, low = None, up = None):
                self.function = function
                self.low = low
                self.up = up

        def __call__(self, x):
                """Call the function and calculate the value of the constraints induced
                by the bounds"""
                res = self.function(x)
                if res == None: return None
                if isinstance(res,tuple):
                        f, con = res
                else:
                        f = res
                        con = []
                if self.low != None:
                        for b,v in zip(self.low,x):
                                if b != None and b != -HUGE_VAL: con.append(v - b)
                if self.up != None:
                        for b,v in zip(self.up,x):
                                if b != None and b != HUGE_VAL: con.append(b - v)
                return f, con

        def addConstraints(self):
                "Count the number of constraints generated by the bounds"
                n = 0
                if self.low != None:
                        for b in self.low:
                                if b != None: n += 1
                if self.up != None:
                        for b in self.up:
                                if b != None: n += 1
                return n

def minimize(function, x, low = None, up = None, m = None,
        rhobeg = 0.5, rhoend = 1.0e-6, messages = MSG_NONE, maxfun = None):
        """Minimize a function with variables subject to non-linear constraints

        returns (rc, nfeval, x).

        Parameters
    ----------
        function
        The function to minimize. Signature must be::

            f, con = function(x)

        where *f* is the value of the function and *con* is
            the value of the constraints (a list of m floats).
            If the function returns None, the minimization is aborted.
        x
        initial estimate (a list of floats)
        low : list of floats
        The lower bounds. Set low[i] to None to remove the lower bound.
        If None (the default), the lower bounds are removed.
    up : list of floats
        The upper bounds. Set up[i] to None to remove the upper bound.
            If None (the default), the upper bounds are removed.
        m
        The number of constraints. If None, the correct value will be selected
        by calling the function once.
        rhobeg
        A reasonable initial change to the variables. Defaults to 0.5.
        rhoend
        The required accuracy for the variables. Defaults to 1.0e-6.
        messages
        Value to select messages display during minimization. Values are
        defined in the MSGS dict. Defaults to MGS_NONE.
        maxfun
        Maximum number of function evaluations. If None (the default),
        maxnfeval is set to max(1000, 100*len(x0)).

        Returns
    -------
        rc
        Return code as defined in the RCSTRINGS dict
        nfeval
        The number of function evaluations
        x
        the solution (a list of floats)
    """

        if m == None:
                res = function(x)
                if isinstance(res, tuple):
                        m = len(res[1])
                elif res is None:
                        return USERABORT, 0, x
                else:
                        m = 0

        if maxfun == None:
                maxfun = max(1000, 100*len(x))
        if low != None and len(low) != len(x): raise ValueError, "cobyla: invalid bounds."
        if up != None and len(up) != len(x): raise ValueError, "cobyla: invalid bounds."
        if low != None or up != None:
                function = BoundConstrained(function, low, up)
                m += function.addConstraints()

        return moduleCobyla.minimize(function, x, m, rhobeg, rhoend, messages, maxfun)

if __name__ == '__main__':
        # Examples for COBYLA

        def example():
                print "Example"
                # A function to minimize
                def function(x):
                        return pow(x[0],2.0)+pow(abs(x[1]),3.0)

                # Optimizer call
                rc, nf, x = minimize(function, [-7, 3], [-10, 1], [10, 10])

                print "After", nf, "function evaluations, COBYLA returned:", RCSTRINGS[rc]
                print "x =", x
                print "exact value = [0, 1]"
                print

        example()

        def example2():
                print "Example 2"
                # A function to minimize
                def function(x):
                        f = pow(x[0],2.0)+pow(abs(x[1]),3.0)
                        con = [0,0,0,0]
                        con[0] = x[0] - -10 # x[0] >= -10
                        con[1] = x[1] - 1 # x[1] >= 1
                        con[2] = 10 - x[0] # x[0] <= 10
                        con[3] = 10 - x[1] # x[1] <= 10
                        return f, con

                # Optimizer call
                rc, nf, x = minimize(function, [-7, 3])

                print "After", nf, "function evaluations, COBYLA returned:", RCSTRINGS[rc]
                print "x =", x
                print "exact value = [0, 1]"
                print

        example2()

        # Tests
        # These tests are taken from Prof. K. Schittkowski test examples for
        # constrained nonlinear programming.
        # http://www.uni-bayreuth.de/departments/math/~kschittkowski/home.htm
        tests = []

        def test2f(x):
                f = 100.0*pow((x[1]-pow(x[0],2)),2)+pow(1.0-x[0],2)
                return f
        tests.append((test2f, [-2,1], [None, 1.5], None, [-1.2210262419616387,1.5]))

        def test3f(x):
                f = x[1]+pow(x[1]-x[0],2)*1.0e-5
                return f
        tests.append((test3f, [10,1], [None, 0.0], None, [0,0]))

        def test4f(x):
                f = pow(x[0]+1.0,3)/3.0+x[1]
                return f
        tests.append((test4f, [1.125,0.125], [1, 0], None, [1,0]))

        from math import *

        def test5f(x):
                f = sin(x[0]+x[1])+pow(x[0]-x[1],2)-1.5*x[0]+2.5*x[1]+1.0
                return f
        tests.append((test5f, [0,0], [-1.5, -3], [4,3], [-0.54719755119659763, -1.5471975511965976]))

        def test45f(x):
                f = 2.0-x[0]*x[1]*x[2]*x[3]*x[4]/120.0
                return f
        tests.append((test45f, [2]*5, [0]*5, [1,2,3,4,5], [1,2,3,4,5]))

        def test(f, x, low, up, xopt):
                print "** Test", f.__name__
                rc, nf, x = minimize(f, x, low, up, maxfun = 200)
                print "After", nf, "function evaluations, COBYLA returned:", RCSTRINGS[rc]
                print "x =", x
                print "exact value =", xopt
                enorm = 0.0
                norm = 1.0
                for y,yo in zip(x, xopt):
                        enorm += (y-yo)*(y-yo)
                        norm += yo*yo
                e = pow(enorm/norm, 0.5)
                print "Error =", e
                if e > 1e-6:
                        raise "Test "+f.__name__+" failed"

        for f, x, low, up, xopt in tests:
                test(f, x, low, up, xopt)

        print
        print "** All COBYLA tests passed."
