/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2006-2010, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.beans;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyEditorSupport;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

public class BooleanEditor extends PropertyEditorSupport {

	private static boolean allowNullByDefault = true;
	private static String nameOfNull = "null";
	private static String nameOfCreation = "inherited";
	private Box panel;
	JButton nullButton;
	private JButton button;
	private boolean nullAllowed = true;
	private JCheckBox checkbox;
	
	public BooleanEditor() {
		this(allowNullByDefault);
	}
	
	public BooleanEditor(boolean withNullOption) {
		nullAllowed = withNullOption;
		customize();
		panel = new Box(BoxLayout.X_AXIS);

		checkbox = new JCheckBox();
		checkbox.setMaximumSize(new Dimension(123,20));
		checkbox.addChangeListener(new ChangeListener() {			
			public void stateChanged(ChangeEvent e) {
					setValue(checkbox.isSelected());
			}
		});
		
		button = new JButton(nameOfCreation);
		button.setMaximumSize(new Dimension(32000,25));
		button.setMargin(new Insets(2,2,2,2));
		button.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setValue(checkbox.isSelected());
			}
		});
		if (nullAllowed) {
			nullButton = new JButton(nameOfNull);
			nullButton.setMargin(new Insets(2,2,2,2));
			nullButton.setMaximumSize(new Dimension(1234,25));
			nullButton.setForeground(Color.red);
			nullButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent arg0) {
					setValue(null);
				}
			});
			panel.add(button);
		} else {
			panel.add(checkbox);
		}
	}

	public static String getNameOfNull() {
		return nameOfNull;
	}

	public static void setNameOfNull(String nameOfNull) {
		BooleanEditor.nameOfNull = nameOfNull;
	}

	public boolean supportsCustomEditor() {
		return true;
	}

	public Component getCustomEditor() {
		return panel;
	}

	protected void customize() {};

	public void setValue(Object value) {
		if (nullAllowed) {
			if (value != null) {
				boolean val = (Boolean) value;
				checkbox.setSelected(val);
				if (getValue() == null) {
					panel.removeAll();
					panel.add(checkbox);
					panel.add(nullButton);
					panel.revalidate();
					panel.doLayout();
					panel.repaint();
				}
			}
			if (value == null && getValue() != null) {
				panel.removeAll();
				panel.add(button);
				panel.revalidate();
				panel.doLayout();
				panel.repaint();
			}
		} else {
			checkbox.setSelected(value != null && (Boolean)value);
		}
		super.setValue(value);
	}

	public String getAsText() {
		if (getValue() == null) return "";
		else return ((Integer)getValue()).toString();
	}

	public void setAsText(String text) {
		if (text == null) setValue(null);
		else {
			try {
				Integer val = new Integer(Integer.parseInt(text));
				setValue(val);
			} catch (NumberFormatException nfe) {
				System.out.println("nfe "+text);
				setValue(null);
			}
		}
	}

	public static String getNameOfCreation() {
		return nameOfCreation;
	}

	public static void setNameOfCreation(String nameOfCreation) {
		BooleanEditor.nameOfCreation = nameOfCreation;
	}

	public static boolean isAllowNullByDefault() {
		return allowNullByDefault;
	}

	public static void setAllowNullByDefault(boolean allowNullByDefault) {
		BooleanEditor.allowNullByDefault = allowNullByDefault;
	}
}
