/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2006-2010, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.beans;
import java.awt.Component;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.RectangularShape;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;


public class SymmetricShapeChooser extends AbstractDialog {

	private static final long serialVersionUID = 3389775807616984946L;
	private static SymmetricShapeChooser sharedInstance;
	
	private JComboBox box = new JComboBox();
	private boolean updating = false;
	
	private SpinnerNumberModel radiusModel = new SpinnerNumberModel(
			new Double(1),
			new Double(0),
			null,
			new Double(1)
	);
	
	public SymmetricShapeChooser() {
		Box column = new Box(BoxLayout.Y_AXIS);
		box.addItem("circle");
		box.addItem("square");
		box.addItemListener(
				new ItemListener() {
					public void itemStateChanged(ItemEvent arg0) {
						if (!updating) {
							fireStateChanged();
						}
					}
				});
		column.add(box);
		radiusModel.addChangeListener(
				new ChangeListener() {
					public void stateChanged(ChangeEvent arg0) {
						if (!updating) {
							fireStateChanged();
						}
					}
				});
		Box row = new Box(BoxLayout.X_AXIS);
		JLabel label = new JLabel("radius");
		label.setBorder(new EmptyBorder(0,0,0,5));
		row.add(label);
		row.add(new JSpinner(radiusModel));
		row.setBorder(new EmptyBorder(5,0,5,0));
		column.add(row);
		setMainComponent(column);
		pack();
	}
	
	public Object getValue() {
		double r = radiusModel.getNumber().doubleValue();
		if (box.getSelectedItem().equals("circle")) {
			return new Ellipse2D.Double(-r, -r, 2*r, 2*r);
		} else {
			return new Rectangle2D.Double(-r, -r, 2*r, 2*r);
		}
	}
	
	public void setValue(Object s) {
		updating = true;
		if (s instanceof RectangularShape && isSymmetric((RectangularShape) s)) {
			double r = ((RectangularShape)s).getWidth()/2;
			radiusModel.setValue(new Double(r));
			if (s instanceof Rectangle2D.Double) {
				box.setSelectedItem("square");
			} else {
				box.setSelectedItem("circle");
			}
		}
		updating = false;
	}
	
	public static SymmetricShapeChooser sharedInstance() {
		if (sharedInstance == null) {
			sharedInstance = new SymmetricShapeChooser();
		}
		return sharedInstance;
	}
	
	public boolean accept(Component c, Object s) {
		boolean accepted = false;
		if (s!= null) {
			if (s instanceof Ellipse2D.Double || s instanceof Rectangle2D.Double) {
				RectangularShape r = (RectangularShape) s;
				accepted = isSymmetric(r);
			}
			if (accepted == false) {
				int decision = JOptionPane.showConfirmDialog(
						c,
						"Proceeding will replace the present shape of type "
						+s.getClass().getName()+
						" with a circle or a sqare. Do you really want this?",
						"Warning",
						JOptionPane.OK_CANCEL_OPTION
				);
				accepted = decision == JOptionPane.OK_OPTION;
			}
		} else {
			accepted = true;
		}
		return accepted;
	}
	
	private static boolean isSymmetric(RectangularShape s) {
		return s.getCenterX()==0 && s.getCenterY() == 0
			&& s.getHeight() == s.getWidth();
	}
}
