<?php
/**
 * File_MogileTest 
 * 
 * PHP Version 5.2.0
 * 
 * @uses PHPUnit_Framework_TestCase
 * @category  File
 * @package   File_Mogile
 * @author    Bill Shupp <hostmaster@shupp.org>
 * @copyright 2009 Digg.com, Inc.
 * @license   http://www.opensource.org/licenses/bsd-license.php BSD
 * @version   CVS: $Id:$
 * @link      http://pear.php.net/package/File_Mogile
 */

require_once 'PHPUnit/Framework.php';
require_once 'File/Mogile.php';

/**
 * Tests for File_Mogile.  No mogile server needed.
 * 
 * Generated by PHPUnit on 2009-01-15 at 20:32:56.
 * 
 * @uses      PHPUnit_Framework_TestCase
 * @category  File
 * @package   File_Mogile
 * @author    Bill Shupp <hostmaster@shupp.org>
 * @copyright 2007 Digg.com, Inc.
 * @license   http://www.opensource.org/licenses/bsd-license.php BSD
 * @link      http://pear.php.net/package/File_Mogile
 */
class File_MogileTest extends PHPUnit_Framework_TestCase
{
    
    /**
     * Dummy hosts for constructor
     * 
     * @var string
     */
    protected $hosts = array('http://example.com');

    /**
     * Methods to mock
     * 
     * @var array
     */
    protected $methods = array('connect',
                               'socketConnect',
                               'socketRead',
                               'socketWrite',
                               'socketClose',
                               'sendReproxyHeader',
                               'curlExec');

    /**
     * Test file data contents
     * 
     * @var string
     */
    protected $fileData = "Test Text\n";

    /**
     * @var File_Mogile
     */
    protected $object;

    /**
     * Sets up the fixture, for example, opens a network connection.
     * This method is called before a test is executed.
     *
     * @return void
     */
    protected function setUp()
    {
        $this->object = $this->getMock('File_Mogile',
                                       $this->methods,
                                       array($this->hosts, 'production'));
        $this->object->expects($this->any())
                     ->method('socketConnect')
                     ->will($this->returnValue(true));
    }

    /**
     * Tears down the fixture, for example, closes a network connection.
     * This method is called after a test is executed.
     *
     * @return void
     */
    protected function tearDown()
    {
        unset($this->object);
    }

    /**
     * Test that the constructor sets options correcty
     * 
     * @return void
     */
    public function testConstructor()
    {
        $socketTimeout             = 1.1;
        $streamTimeoutSeconds      = 1;
        $streamTimeoutMicroSeconds = 200;
        $commandTimeout            = 4;

        $options = array('socketTimeout'             => $socketTimeout,
                         'streamTimeoutSeconds'      => $streamTimeoutSeconds,
                         'streamTimeoutMicroSeconds' => $streamTimeoutMicroSeconds,
                         'commandTimeout'            => $commandTimeout);

        $object = $this->getMock('File_Mogile',
                                 $this->methods,
                                 array($this->hosts, 'production', $options));

        $this->assertSame($socketTimeout, File_Mogile::$socketTimeout);
        $this->assertSame($streamTimeoutSeconds, File_Mogile::$streamTimeoutSeconds);
        $this->assertSame($streamTimeoutMicroSeconds, File_Mogile::$streamTimeoutMicroSeconds);
        $this->assertSame($commandTimeout, File_Mogile::$commandTimeout);
    }

    /**
     * Test __construct failure
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testConstructorFail()
    {
        $options = array('foo' => 'bar');
        $object  = $this->getMock('File_Mogile',
                                  $this->methods,
                                  array($this->hosts, 'production', $options));
    }

    /**
     * Test GET_DOMAINS
     *
     * @return void
     */
    public function testGetDomains()
    {
        $response = "OK domains=1&classes=1&domain1=staging&domain1classes=1&domain1class1name=testclass1&domain1class1mindevcount=50";
        $expected = array('staging' => array('testclass1' => '50'));

        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->assertSame($expected, $this->object->getDomains());
    }

    /**
     * Test false result from socketWrite()
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testRequestFailSocketWrite()
    {
        $this->object->expects($this->any())
                     ->method('socketWrite')
                     ->will($this->returnValue(false));
        $this->object->getDomains();
    }

    /**
     * Test false result from socketRead()
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testRequestFailSocketRead()
    {
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue(false));
        $this->object->getDomains();
    }

    /**
     * Test ERR response from mogile
     *
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testRequestFailFromMogile()
    {
        $response = "ERR fobar";

        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->object->getDomains();
    }

    /**
     * Test GET_PATHS
     *
     * @return void
     */
    public function testGetPaths()
    {
        $response = "OK path2=http://127.0.0.1:7500/dev1/0/000/435/0000435316.fid&path1=http://127.0.0.1:7500/dev2/0/000/435/0000435316.fid&paths=2";
        $expected = array(
            'path2' => 'http://127.0.0.1:7500/dev1/0/000/435/0000435316.fid',
            'path1' => 'http://127.0.0.1:7500/dev2/0/000/435/0000435316.fid'
        );
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->assertSame($expected, $this->object->getPaths('somekey'));
    }

    /**
     * Test DELETE
     * 
     * @return void
     */
    public function testDelete()
    {
        $response = 'OK ';
        $expected = null;
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->assertSame($expected, $this->object->delete('somekey'));
    }

    /**
     * Test RENAME
     *
     * @return void
     */
    public function testRename()
    {
        $response = 'OK ';
        $expected = null;
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->assertSame($expected, $this->object->rename('from', 'to'));
    }

    /**
     * Test LIST_KEYS
     * 
     * @return void
     */
    public function testListKeys()
    {
        $nextAfter = '/thumb/2/o';
        $limit     = 2;
        $response  = "OK key_1=/thumb/3/o&key_2=/thumb/4/o&key_count=2&next_after=/thumb/4/o";
        $expected  = array('/thumb/4/o', array('/thumb/3/o', '/thumb/4/o'));
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->assertSame($expected, $this->object->listKeys('',
                                                             $nextAfter,
                                                             $limit));
    }

    /**
     * Test LIST_KEYS Failure, missign key_count
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testListKeysFailMissingKeyCount()
    {
        $nextAfter = '/thumb/2/o';
        $limit     = 2;
        $response  = "OK key_1=/thumb/3/o&key_2=/thumb/4/o&next_after=/thumb/4/o";
        $expected  = array('/thumb/4/o', array('/thumb/3/o', '/thumb/4/o'));
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->assertSame($expected, $this->object->listKeys('',
                                                             $nextAfter,
                                                             $limit));
    }

    /**
     * Test LIST_KEYS Failure, missign next_after
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testListKeysFailMissingNextAfter()
    {
        $nextAfter = '/thumb/2/o';
        $limit     = 2;
        $response  = "OK key_1=/thumb/3/o&key_2=/thumb/4/o&key_count=2";
        $expected  = array('/thumb/4/o', array('/thumb/3/o', '/thumb/4/o'));
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->assertSame($expected, $this->object->listKeys('',
                                                             $nextAfter,
                                                             $limit));
    }
 
    /**
     * Test LIST_KEYS Failure, incorrect key names in response
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testListKeysFailIncorrectKeyNames()
    {
        $nextAfter = '/thumb/2/o';
        $limit     = 2;
        $response  = "OK key_1=/thumb/3/o&key_3=/thumb/4/o&key_count=2&next_after=/thumb/4/o";
        $expected  = array('/thumb/4/o', array('/thumb/3/o', '/thumb/4/o'));
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->assertSame($expected, $this->object->listKeys('',
                                                             $nextAfter,
                                                             $limit));
    }

    /**
     * Test reproxy() success
     * 
     * @return void
     */
    public function testReproxy()
    {
        $response = "OK path2=http://127.0.0.1:7500/dev1/0/000/435/0000435316.fid&path1=http://127.0.0.1:7500/dev2/0/000/435/0000435316.fid&paths=2";
        $paths    = array('path1' => 'http://example.com/foo',
                          'path2' => 'http://example.com/bar');
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->assertNull($this->object->reproxy('destination'));
        $this->assertNull($this->object->reproxy($paths));
    }

    /**
     * Test invalid destination argument to reproxy()
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testReproxyFail()
    {
        $this->assertNull($this->object->reproxy(5));
    }

    /**
     * Test getFileData() success
     * 
     * @return void
     */
    public function testGetFileData()
    {
        $filename2 = $this->getTestDataFile();
        $filename1 = $filename2 . '1';
        $response  = "OK path1=$filename1&path2=$filename2&paths=2";

        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));

        // Suppress a warning for the invalid file
        $result = @$this->object->getFileData('key');
        $this->assertSame($this->fileData, $result);
    }

    /**
     * Test no valid paths
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testGetFileDataFail()
    {
        $filename = $this->getTestDataFile() . '1';
        $response = "OK path1=$filename&paths=1";

        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));

        // Suppress a warning for the invalid file
        $result = @$this->object->getFileData('key');
    }

    /**
     * Test passthru()
     * 
     * @return void
     */
    public function testPassthru()
    {
        $filename = $this->getTestDataFile();
        $response = "OK path1=$filename&paths=1";
        $paths    = array($filename);

        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));

        foreach (array('key', $paths) as $arg) {
            ob_start();
            $this->object->passthru($arg);
            $result = ob_get_contents();
            ob_end_clean();
            $this->assertSame($this->fileData, $result);
        }
    }

    /**
     * Tests several passthru() failures
     * 
     * @return void
     */
    public function testPassthruFail()
    {
        $filename = $this->getTestDataFile() . '1';
        $response = "OK path1=$filename&paths=1";

        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));

        // Test failure for fopen()
        try {
            ob_start();
            @$this->object->passthru($filename);
            $result = ob_get_contents();
            // Should not get here
            ob_end_clean();
            $this->assertFalse(true);
        } catch (File_Mogile_Exception $e) {
        }

        // Test invalid destination
        try {
            $this->object->passthru(1);
            // Should not get here
            $this->assertFalse(true);
        } catch (File_Mogile_Exception $e) {
        }
    }
    /**
     * Helper function for determining test data file
     * 
     * @return string
     */
    protected function getTestDataFile()
    {
        return dirname(__FILE__) . DIRECTORY_SEPARATOR . 'testdata.txt';
    }

    /**
     * Tests passthruFileData()
     *
     * @return void
     */
    public function testPassthruFileData()
    {
        $filename = $this->getTestDataFile();
        $response = "OK path1=$filename&paths=1";

        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));

        ob_start();
        $this->object->passthruFileData($filename);
        $result = ob_get_contents();
        ob_end_clean();
        $this->assertSame($this->fileData, $result);
    }

    /**
     * Test invalid argument to passthruFileData()
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testPassthruFileDataFail()
    {
        $this->object->passthruFileData(1);
    }

    /**
     * Test storing a file
     *
     * @return void
     */
    public function testStoreFile()
    {
        $response = 'OK devid=2&fid=943794&path=http://127.0.0.1:7500/dev2/0/000/943/0000943794.fid';
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->object->expects($this->any())
                     ->method('curlExec')
                     ->will($this->returnValue(true));

        $result = $this->object->storeFile('foobarkey',
                                           'myclass',
                                           $this->getTestDataFile());
        $this->assertNull($result);
    }

    /**
     * testStoreFileFailValidation 
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testStoreFileFailValidation()
    {
        $response = 'OK devid=2&fid=943794&path=http:/127.0.0.1:7500/dev2/0/000/943/0000943794.fid';
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));

        $result = $this->object->storeFile('foobarkey',
                                           'myclass',
                                           $this->getTestDataFile());
    }

    /**
     * Test missing file
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testStoreFileFailfopen()
    {
        $response = 'OK devid=2&fid=943794&path=http://127.0.0.1:7500/dev2/0/000/943/0000943794.fid';
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->object->expects($this->any())
                     ->method('curlExec')
                     ->will($this->returnValue(true));

        $result = @$this->object->storeFile('foobarkey',
                                            'myclass',
                                            $this->getTestDataFile() . '1');
    }

    /**
     * Test curl error
     * 
     * @expectedException File_Mogile_Exception
     * @return void
     */
    public function testStoreFileFailCurl()
    {
        $response = 'OK devid=2&fid=943794&path=http://127.0.0.1:7500/dev2/0/000/943/0000943794.fid';
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->object->expects($this->any())
                     ->method('curlExec')
                     ->will($this->returnValue(false));

        $result = $this->object->storeFile('foobarkey',
                                            'myclass',
                                            $this->getTestDataFile());
    }

    /**
     * Tests storeData()
     *
     * @return void
     */
    public function testStoreData()
    {
        $response = 'OK devid=2&fid=943794&path=http://127.0.0.1:7500/dev2/0/000/943/0000943794.fid';
        $this->object->expects($this->any())
                     ->method('socketRead')
                     ->will($this->returnValue($response));
        $this->object->expects($this->any())
                     ->method('curlExec')
                     ->will($this->returnValue(true));

        // Suppress warning from fopen()
        $result = $this->object->storeData('foobarkey',
                                           'myclass',
                                           $this->fileData);
        $this->assertNull($result);
    }
}
?>
