/*
 * Decompiled with CFR 0.152.
 */
package org.openjdk.jmc.flightrecorder.util;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Iterator;
import java.util.NoSuchElementException;
import org.openjdk.jmc.common.io.IOToolkit;
import org.openjdk.jmc.flightrecorder.internal.util.DataInputToolkit;

public final class ChunkReader {
    private static final byte[] JFR_MAGIC_BYTES = new byte[]{70, 76, 82, 0};
    private static final int[] JFR_MAGIC = new int[]{70, 76, 82, 0};
    private static final int[] ZIP_MAGIC = new int[]{31, 139};
    private static final int[] GZ_MAGIC = new int[]{31, 139};
    private static final int HEADER_SIZE = 16;

    private ChunkReader() {
        throw new UnsupportedOperationException("Not to be instantiated");
    }

    public static Iterator<byte[]> readChunks(File jfrFile) throws IOException {
        if (IOToolkit.isCompressedFile((File)jfrFile)) {
            return new StreamChunkIterator(IOToolkit.openUncompressedStream((File)jfrFile));
        }
        return new ChunkIterator(jfrFile);
    }

    public static Iterator<byte[]> readChunks(InputStream jfrStream) throws IOException {
        return new StreamChunkIterator(IOToolkit.openUncompressedStream((InputStream)jfrStream));
    }

    public static void main(String[] args) throws IOException {
        File file;
        long nanoStart = System.nanoTime();
        int chunkCount = 0;
        int byteCount = 0;
        if (args.length != 1) {
            System.out.println("Usage: ChunkReader <file>");
            System.exit(2);
        }
        if (!(file = new File(args[0])).exists()) {
            System.out.println("The file " + file.getAbsolutePath() + " does not exist. Exiting...");
            System.exit(3);
        }
        Iterator<byte[]> iter = ChunkReader.readChunks(file);
        while (iter.hasNext()) {
            byte[] bytes = iter.next();
            byteCount += bytes.length;
            System.out.println("Chunk #" + ++chunkCount + " size: " + bytes.length);
        }
        double duration = (double)(System.nanoTime() - nanoStart) / 1000000.0;
        System.out.println("Chunks: " + chunkCount + " Byte count: " + byteCount + " Time taken: " + duration + " ms");
    }

    private static class StreamChunkIterator
    implements Iterator<byte[]> {
        private final DataInputStream inputStream;
        private StreamState streamState = StreamState.NEXT_CHUNK;
        private Throwable lastError = null;

        public StreamChunkIterator(InputStream inputStream) {
            this.inputStream = this.getDataStream(inputStream);
        }

        private DataInputStream getDataStream(InputStream is) {
            if (is.markSupported()) {
                return new DataInputStream(is);
            }
            return new DataInputStream(new BufferedInputStream(is));
        }

        @Override
        public boolean hasNext() {
            boolean hasNext = false;
            if (this.streamState == StreamState.NEXT_CHUNK) {
                hasNext = this.validateJFRMagic();
            } else if (this.streamState == StreamState.JFR_CHECKED) {
                hasNext = true;
            }
            if (!hasNext) {
                IOToolkit.closeSilently((Closeable)this.inputStream);
            }
            return hasNext;
        }

        private boolean validateJFRMagic() {
            try {
                if (IOToolkit.hasMagic((InputStream)this.inputStream, (int[])JFR_MAGIC)) {
                    this.streamState = StreamState.JFR_CHECKED;
                    return true;
                }
                this.streamState = StreamState.ERROR;
                this.lastError = new Exception("Next chunk has no JFR magic. It is either no JFR file at all or corrupt.");
                return false;
            }
            catch (IOException e) {
                this.streamState = StreamState.ERROR;
                this.lastError = e;
                return false;
            }
        }

        @Override
        public byte[] next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            switch (this.streamState) {
                case ERROR: {
                    throw new IllegalArgumentException(this.lastError);
                }
                case NEXT_CHUNK: {
                    if (!this.validateJFRMagic()) {
                        throw new IllegalArgumentException(this.lastError);
                    }
                }
                case JFR_CHECKED: {
                    try {
                        return this.retrieveNextChunk();
                    }
                    catch (IOException e) {
                        this.lastError = e;
                        throw new IllegalArgumentException(e);
                    }
                }
            }
            throw new IllegalArgumentException("Unknown stream state");
        }

        private byte[] retrieveNextChunk() throws IOException {
            byte[] chunkHeader = new byte[16];
            System.arraycopy(JFR_MAGIC_BYTES, 0, chunkHeader, 0, JFR_MAGIC_BYTES.length);
            this.readBytesFromStream(chunkHeader, JFR_MAGIC_BYTES.length, 16 - JFR_MAGIC_BYTES.length);
            short majorVersion = DataInputToolkit.readShort(chunkHeader, JFR_MAGIC_BYTES.length);
            byte[] chunkTotal = null;
            if (majorVersion >= 1) {
                long fullSize = DataInputToolkit.readLong(chunkHeader, 8);
                int readSize = (int)fullSize - 16;
                chunkTotal = new byte[(int)fullSize];
                System.arraycopy(chunkHeader, 0, chunkTotal, 0, chunkHeader.length);
                this.readBytesFromStream(chunkTotal, 16, readSize);
            } else {
                long metadataIndex = DataInputToolkit.readLong(chunkHeader, 8);
                int eventReadSize = (int)(metadataIndex - 16L + 4L);
                byte[] chunkEvents = new byte[eventReadSize];
                this.readBytesFromStream(chunkEvents, 0, chunkEvents.length);
                int metadataEventSize = DataInputToolkit.readInt(chunkEvents, eventReadSize - 4) - 4;
                byte[] chunkMetadata = new byte[metadataEventSize];
                this.readBytesFromStream(chunkMetadata, 0, chunkMetadata.length);
                chunkTotal = new byte[chunkHeader.length + chunkEvents.length + chunkMetadata.length];
                System.arraycopy(chunkHeader, 0, chunkTotal, 0, chunkHeader.length);
                System.arraycopy(chunkEvents, 0, chunkTotal, chunkHeader.length, chunkEvents.length);
                System.arraycopy(chunkMetadata, 0, chunkTotal, chunkHeader.length + chunkEvents.length, chunkMetadata.length);
            }
            this.streamState = StreamState.NEXT_CHUNK;
            return chunkTotal;
        }

        private void readBytesFromStream(byte[] bytes, int offset, int count) throws IOException {
            int read;
            for (int totalRead = 0; totalRead < count; totalRead += read) {
                read = this.inputStream.read(bytes, offset + totalRead, count - totalRead);
                if (read != -1) continue;
                throw new IOException("Unexpected end of data.");
            }
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("Cannot remove chunks");
        }
    }

    private static enum StreamState {
        NEXT_CHUNK,
        JFR_CHECKED,
        ERROR;

    }

    private static class ChunkIterator
    implements Iterator<byte[]> {
        int lastChunkOffset;
        private RandomAccessFile file;
        private final FileChannel channel;
        private final MappedByteBuffer buffer;

        private ChunkIterator(File jfrFile) throws IOException {
            try {
                this.file = new RandomAccessFile(jfrFile, "r");
                this.channel = this.file.getChannel();
                this.buffer = this.channel.map(FileChannel.MapMode.READ_ONLY, 0L, this.channel.size());
                if (!this.bufferHasMagic(JFR_MAGIC)) {
                    if (this.bufferHasMagic(GZ_MAGIC) || this.bufferHasMagic(ZIP_MAGIC)) {
                        throw new IOException("Cannot use the ChunkIterators with gzipped JMC files. Please use unzipped recordings.");
                    }
                    throw new IOException("The provided file (" + String.valueOf(jfrFile) + ") is not a JFR file!");
                }
            }
            catch (Exception e) {
                if (this.file != null) {
                    this.file.close();
                }
                throw e;
            }
        }

        @Override
        public boolean hasNext() {
            boolean hasNext = this.checkHasMore();
            if (!hasNext) {
                try {
                    this.channel.close();
                    this.file.close();
                }
                catch (IOException e) {
                    e.printStackTrace();
                }
            }
            return hasNext;
        }

        private boolean checkHasMore() {
            return this.lastChunkOffset < this.buffer.limit();
        }

        @Override
        public byte[] next() {
            if (!this.checkHasMore()) {
                throw new NoSuchElementException();
            }
            if (!this.bufferHasMagic(JFR_MAGIC)) {
                this.lastChunkOffset = this.buffer.limit() + 1;
                throw new IllegalArgumentException("Corrupted chunk encountered! Aborting!");
            }
            int index = this.lastChunkOffset + JFR_MAGIC.length;
            short versionMSB = this.buffer.getShort(index);
            index += 4;
            int size = 0;
            if (versionMSB >= 1) {
                size = (int)this.buffer.getLong(index);
                index = this.lastChunkOffset + size;
            } else {
                index = this.lastChunkOffset + (int)this.buffer.getLong(index);
                int lastEventSize = this.buffer.getInt(index);
                size = (index += lastEventSize) - this.lastChunkOffset;
            }
            byte[] result = new byte[size];
            this.buffer.position(this.lastChunkOffset);
            this.buffer.get(result, 0, result.length);
            this.lastChunkOffset = index;
            return result;
        }

        private boolean bufferHasMagic(int[] magicBytes) {
            for (int i = 0; i < magicBytes.length; ++i) {
                if (this.buffer.get(this.lastChunkOffset + i) == magicBytes[i]) continue;
                return false;
            }
            return true;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("Cannot remove chunks");
        }
    }
}

