
/*  $Id: file.c,v 1.9 2005/03/29 19:25:40 alien-science Exp $ */

#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>

#include "common.h"
#include "read_buf.h"
#include "file.h"
#include "crypt.h"
#include "header.h"
#include "prog_config.h"
#include "pipe.h"
#include "checksum.h"

/* The size of the read buffer to use */
#define BUFFER_SIZE (1024 * 1024)

/* The size of the window used in calculating checksums */
#define CHECKSUM_WINDOW 200

/*
 * Convert a 32 bit value into a byte array for dumping
 */
void serial_uint32(unsigned char *buf, unsigned val)
{
   buf[0]   = (unsigned char)  (val & 0x000000ff);
   buf[1]   = (unsigned char) ((val & 0x0000ff00) >> 8);
   buf[2]   = (unsigned char) ((val & 0x00ff0000) >> 16);
   buf[3]   = (unsigned char) ((val & 0xff000000) >> 24);
}

/*
 * Convert byte array into 32 bit value
 */
unsigned unserial_uint32(unsigned char *buf)
{
   return (unsigned) buf[0] + ((unsigned)buf[1] <<  8) +
                              ((unsigned)buf[2] << 16) +
                              ((unsigned)buf[3] << 24);
}

/*
 * Open a file or return a default file handle
 */
int
open_file(char *name, int flags, mode_t mode, int default_fh)
{

   int fh;
   char     *err_mess;

   /* Open the file */
   if (0 == name || 0 == strcmp("-", name)) {
      fh = default_fh;
      }
   else if (-1 == (fh = open(name, flags, mode))) {
      err_mess = XMALLOC(char, strlen(name) + 40);
      sprintf(err_mess, "Cannot open %s", name);
      perror(err_mess);
      XFREE(err_mess);
      fh = -1;
      }

   return fh;
}


/*
 * Reads an encrypted block from a file
 */
size_t read_block( int fh, char *buf, size_t max_len, size_t *block_len) 
{
   size_t   read_in;
   unsigned char lengthbuf[4];

   /* Read the length of the block */
   ASSERT(read_in = read(fh, lengthbuf, 4));

   /* Check for EOF */
   if (0 == read_in) {
      *block_len = 0;
      OK;
      }

   /* Get the length as a size_t */
   *block_len = unserial_uint32(lengthbuf);

   if (*block_len > max_len) {
      ERROR("Block found that is bigger than the maximum allowed");
      }

   /* Read in the block */
   ASSERT(read_in = read(fh, buf, *block_len));

   /* Check all in */
   if (read_in != *block_len) {
      ERROR("Partial block read in");
      }

   OK;
}

/*
 * Helper function that writes out data compatible with read_block
 */
int
write_block(int fh, void *buf, size_t len)
{
   char lengthbuf[4];
   size_t written;

   serial_uint32(lengthbuf, len);
   written  = write(fh, lengthbuf, 4);
   written += write(fh, buf, len);

   if (written != len + 4) {
      ERROR("Cannot write block to file");
      }

   OK;
}

/*
 * Encrypt a read buffer to the given file handle
 */
int
buffer_encrypt(struct config *conf, struct rbuf *inbuf, int out_fh)
{

   struct pipe pipe_s;
   struct pipe *pipe;
   struct digest digest;
   size_t available, enc_len, block_len, buf_size;
   char   *buf1;
   char   *actual_iv;
   unsigned char dump_crc[4];
   unsigned crc = 0;
   int      ret = 0;

   /* Allocate a buffer to hold the encrypted block */
   buf_size = conf->max_block_size * 2;
   buf1     = XMALLOC(char, buf_size);

   /* Allocate a buffer to hold the changing iv */
   actual_iv = XMALLOC(char, conf->cipher->ivlen);

   pipe = &pipe_s;
   pipe_alloc(pipe, buf_size, conf);

   while (!rbuf_empty(inbuf)) {

      /* Set up a window on the data so that checksums can be calculated */ 
      LASSERT( rbuf_alloc(inbuf, conf->max_block_size, &available));

      /* Get a block of the read buffer that matches a checksum */
      LASSERT( ck_findblock(rbuf_p(inbuf), 
                            MIN(conf->max_block_size, available),
                            conf,
                            CHECKSUM_WINDOW,
                            &block_len));
      
      /* Take the crc of the block */
      crc = crc_calc(crc, rbuf_p(inbuf), block_len);

      /* Take the digest of the block which will be used as an iv */
      LASSERT( crypt_digest(conf, rbuf_p(inbuf), block_len, &digest));

      /* Initialise the encryption for this block */
      enc_len = 0;
      LASSERT( pipe_encrypt_init(pipe, digest.value, digest.len, 
                                       buf1, &enc_len));

      /* Process this block */
      LASSERT( pipe_encrypt(pipe, rbuf_p(inbuf), block_len, 
                                  buf1 + enc_len, buf_size - enc_len, 
                                  &enc_len));

      /* We have finished with the current bit of buffer */
      LASSERT( rbuf_free(inbuf, block_len));

      /* Is this the end of the data coming in? */
      if (rbuf_empty(inbuf)) {
         /* Dump the CRC */
         serial_uint32(dump_crc, crc);
         LASSERT( pipe_encrypt(pipe, 
                               dump_crc, 4, 
                               buf1 + enc_len, buf_size - enc_len, 
                               &enc_len));
         }

      /* No more data for this block -- flush the pipe */
      LASSERT( pipe_enc_end(pipe, 
                            buf1 + enc_len, buf_size - enc_len, 
                            &enc_len));

      VERBOSE_2("block @ %#08x, encrypt size %#08x", block_len, enc_len); 

      /* Write out the encrypted block */
      LASSERT(write_block(out_fh, buf1, enc_len));
      }

   pipe_free(pipe);
   XFREE(actual_iv);
   XFREE(buf1);

   return ret;
}


/*
 * Decrypt the given file handle
 */
int
fh_decrypt(struct config *conf, int in_fh, int out_fh)
{
   size_t   buf_size, read_in, read_ahead, written, to_file;
   char     *buf1, *buf2, *header_buf;
   unsigned crc, file_crc;
   struct pipe pipe_s;
   struct pipe *pipe;

   size_t   block       = 0;
   int      ret         = 0;

   pipe     = &pipe_s;
   crc      = 0;

   /* Read in the header */
   read_in    = 0;
   header_buf = XMALLOC(char, MAX_HEADER_SIZE);
   read_block(in_fh, header_buf, MAX_HEADER_SIZE, &read_in);
   ASSERT(header_read(conf, header_buf, read_in));
   XFREE(header_buf);

   /* Allocate temporary buffers */
   buf_size =  conf->max_block_size * 2;
   buf1     = XMALLOC(char, buf_size);
   buf2     = XMALLOC(char, buf_size);

   /* Initialise a pipe to decrypt/uncompress the data */
   pipe_alloc(pipe, buf_size, conf);

   /* Process the file */
   read_in = 0;
   read_block(in_fh, buf1, buf_size, &read_in);

   while (read_in > 0) {

      /* Process the current block */
      written = 0;
      LASSERT( pipe_decrypt(pipe, buf1, read_in, buf2, buf_size, &written));

      /* Read in the next block */
      LASSERT( read_block(in_fh, buf1, buf_size, &read_ahead));

      /* Is it EOF? */
      if (0 == read_ahead) {

         /* The last 4 bytes of the unencrypted data are the CRC */
         block    = written - 4;
         crc      = crc_calc(crc, buf2, block); 
         file_crc = unserial_uint32(buf2 + block);
         if (file_crc != crc){
            ret = -1;
            fprintf(stderr, "File crc : %08x, Found crc : %08x\n", 
                    file_crc, crc);
            break;
            }
         }
      else {
         /* CRC the whole block */
         block = written;
         crc   = crc_calc(crc, buf2, block); 
         }

      VERBOSE_2("block @ %#08x, encrypt size %#08x", block, read_in);

      /* Write out the block */
      to_file = write(out_fh, buf2, block);

      if (to_file != block) {
         LERROR("Error writing out to file");
         }
     
      /* Prepare for next iteration */
      read_in = read_ahead;
      }

   pipe_free(pipe);
   XFREE(buf1);
   XFREE(buf2);

   return ret;
}


/*
 * Encrypt a file
 */
int 
file_encrypt(char *inname, char *outname, struct config *conf)
{
   struct   rbuf    *rbuf;
   int      out_fh;  
   int      ret = 0;

   /* See if the output file was specified */
   if (inname && outname == inname) {
      outname = XMALLOC(char, strlen(inname) + 5);
      strcpy(outname, inname);
      strcat(outname, ".zm");
      }

   /* Create a read buffer on the input file */
   rbuf = XMALLOC(struct rbuf, 1);

   do {

      LASSERT( rbuf_new(rbuf, BUFFER_SIZE, inname)); 

      /* Open the output file defaulting to stdout */
      LASSERT( out_fh = open_file(outname, (O_WRONLY | O_CREAT | O_TRUNC), 
                                  00644, STDOUT_FILENO));

      /* Write the header */
      LASSERT( header_write(conf, out_fh)); 

      /* Encrypt the input file via its buffer */
      LASSERT( buffer_encrypt(conf, rbuf, out_fh));

      } while (0);

   /* Close the buffer, file and free memory */
   rbuf_close(rbuf);
   close(out_fh);
   XFREE(rbuf);
   XFREE(outname);

   return ret;
}


/*
 * Decrypt a file
 */
int 
file_decrypt(char *inname, char *outname, struct config *conf)
{
   int      in_fh, 
            out_fh;  
   int      ret = 0;
   char     *pext;

   /* See if the output file was specified */
   if (inname && outname == inname) {
      if (strstr(inname, ".zm")) {
         outname = XMALLOC(char, strlen(inname) + 1);
         strcpy(outname, inname);
         pext = strstr(outname, ".zm");
         *pext = '\0';
         }
      else {   
         ERROR("Input file doesn't seem to have a known extension,"
               " please specify an output file");
         }
      }

   do {

      /* Open the input file defaulting to stdin */
      LASSERT( in_fh = open_file(inname, (O_RDONLY), 00644, STDIN_FILENO));

      /* Open the output file defaulting to stdout */
      LASSERT( out_fh = open_file(outname, (O_WRONLY | O_CREAT | O_TRUNC), 
                                  00644, STDOUT_FILENO));

      /* Decrypt the input file */
      LASSERT( fh_decrypt(conf, in_fh, out_fh));

      } while (0);

   /* Close the files */
   close(in_fh);
   close(out_fh);
   XFREE(outname);

   return ret;
}


