/***************************** LICENSE START ***********************************

 Copyright 2018 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvQStyleHelp.h"

#include <QtGlobal>
#include <QApplication>
#include <QComboBox>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPainter>
#include <QSortFilterProxyModel>
#include <QToolButton>
#include <QTreeView>
#include <QVBoxLayout>

#include "MvQStyleDb.h"
#include "MvQStyleLine.h"

#include "MvMiscelaneous.h"

#include "HelpFactory.h"

//=====================================================
//
//  MvQStyleModel
//
//=====================================================

MvQStyleModel::MvQStyleModel(QObject *parent) :
    QAbstractItemModel(parent)
{
}

MvQStyleModel::~MvQStyleModel()
{
}

int MvQStyleModel::columnCount( const QModelIndex& /*parent */ ) const
{
     return 3;
}

int MvQStyleModel::rowCount( const QModelIndex& parent) const
{
    //Parent is the root:
    if(!parent.isValid())
    {
        return MvQStyleDb::instance()->items().count();
    }

    return 0;
}

QVariant MvQStyleModel::data( const QModelIndex& index, int role ) const
{
    int row=index.row();
    if(row < 0 || row >= MvQStyleDb::instance()->items().count())
        return QVariant();

    if(role == Qt::DisplayRole)
    {
        if(index.column() == 0)
            return row;
        else if(index.column() == 1)
            return MvQStyleDb::instance()->items()[row]->name();
        else if(index.column() == 2)
            return MvQStyleDb::instance()->items()[row]->description();
    }
    else if(role == Qt::UserRole)
    {
        return (isFiltered(MvQStyleDb::instance()->items()[row]) == true)?"1":"0";
    }
    else if(role == SortRole)
    {
        if(index.column() == 0)
            return row;
        else if(index.column() == 1)
            return MvQStyleDb::instance()->items()[row]->name();
    }

    return QVariant();
}

QVariant MvQStyleModel::headerData( const int section, const Qt::Orientation orient , const int role ) const
{
    if ( orient != Qt::Horizontal || (role != Qt::DisplayRole &&  role != Qt::ToolTipRole))
              return QAbstractItemModel::headerData( section, orient, role );

    if(role == Qt::DisplayRole)
    {
        switch ( section )
        {
        case 0: return "Preview";
        case 1: return "Name";
        case 2: return "Description";
        default: return QVariant();
        }
    }

    return QVariant();
}

QModelIndex MvQStyleModel::index( int row, int column, const QModelIndex & parent ) const
{
    if(row < 0 || column < 0)
    {
        return QModelIndex();
    }

    //When parent is the root this index refers to a node or server
    if(!parent.isValid())
    {
        return createIndex(row,column);
    }

    return QModelIndex();

}

QModelIndex MvQStyleModel::parent(const QModelIndex &child) const
{
    return QModelIndex();
}

void MvQStyleModel::setKeywordFilter(QString s)
{
    keywordFilter_=s;
}

void MvQStyleModel::setColourFilter(QString s)
{
    colourFilter_=s;
}

void MvQStyleModel::setLayerFilter(QString s)
{
    layerFilter_=s;
}

void MvQStyleModel::setNameFilter(QString s)
{
    nameFilter_=s;
}

void MvQStyleModel::clearFilter()
{
    nameFilter_.clear();
    keywordFilter_.clear();
    colourFilter_.clear();
    layerFilter_.clear();
}

bool MvQStyleModel::isFiltered(MvQStyleDbItem *item) const
{
    if(!nameFilter_.isEmpty())
    {
        if(!item->name().contains(nameFilter_,Qt::CaseInsensitive))
            return false;
    }

    if(!keywordFilter_.isEmpty())
    {
        if(!item->keywords().contains(keywordFilter_))
            return false;
    }

    if(!colourFilter_.isEmpty())
    {
        if(!item->colours().contains(colourFilter_))
            return false;
    }

    if(!layerFilter_.isEmpty())
    {
        if(!item->layers().contains(layerFilter_))
            return false;
    }

    return true;
}

//==============================
//
// MvQStyleDelegate
//
//==============================

MvQStyleDelegate::MvQStyleDelegate(QWidget *parent) :
    QStyledItemDelegate(parent),
    borderCol_(QColor(210,210,210))
{
}

void MvQStyleDelegate::paint(QPainter *painter,const QStyleOptionViewItem &option,
                   const QModelIndex& index) const
{
    painter->save();

    if(index.column() == 0)
    {
        //Background
        QStyleOptionViewItem vopt(option);
        initStyleOption(&vopt, index);

        const QStyle *style = vopt.widget ? vopt.widget->style() : QApplication::style();
        const QWidget* widget = vopt.widget;

        //painter->fillRect(option.rect,QColor(238,238,238));

        //We render everything with the default method
        style->drawControl(QStyle::CE_ItemViewItem, &vopt, painter, widget);

        int row=index.data(Qt::DisplayRole).toInt();
        if(row < 0)
            return;

        QRect rect=option.rect.adjusted(2,2,-4,-2);

        MvQStyleDbItem* item=MvQStyleDb::instance()->items()[row];
        Q_ASSERT(item);
        if(item)
            item->paint(painter,rect);
    }
    else
    {
        QStyledItemDelegate::paint(painter,option,index);
    }

    //Render the horizontal border for rows. We only render the top border line.
    //With this technique we miss the bottom border line of the last row!!!
    //QRect fullRect=QRect(0,option.rect.y(),painter->device()->width(),option.rect.height());
    QRect bgRect=option.rect;
    painter->setPen(borderCol_);
    painter->drawLine(bgRect.topLeft(),bgRect.topRight());

    painter->restore();
}

QSize MvQStyleDelegate::sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const
{
    //s.height()+2
    QSize s=QStyledItemDelegate::sizeHint(option,index);
    if(index.column() == 0)
        return QSize(180,50);

    return QSize(s.width(),50);
}



//=====================================================
//
// MvQStyleSelectionWidget
//
//=====================================================

MvQStyleSelectionWidget::MvQStyleSelectionWidget(QWidget* parent) :
    QWidget(parent),
    ignoreFilterChanged_(false)
{
    QVBoxLayout* vb=new QVBoxLayout(this);

    resetTb_=new QToolButton(this);
    resetTb_->setToolButtonStyle(Qt::ToolButtonTextBesideIcon);
    resetTb_->setText(tr("Clear all filters"));
    resetTb_->setIcon(QPixmap(":/desktop/undo.svg"));
    vb->addWidget(resetTb_);

    QGridLayout* grid=new QGridLayout();
    vb->addLayout(grid);

    int cbWidth=42; //number of characters!

    //name
    int row=0;
    grid->addWidget(new QLabel(tr("Name:"),this),row,0);
    grid->setContentsMargins(1,1,1,1);
    grid->setVerticalSpacing(2);

    nameLe_=new QLineEdit(this);
    nameLe_->setPlaceholderText("ANY");
//#if QT_VERSION >= QT_VERSION_CHECK(5, 2, 0)
//    nameLe_->setClearButtonEnabled(true);
//#endif
    grid->addWidget(nameLe_,row,1);

    nameResetTb_=new QToolButton(this);
    nameResetTb_->setIcon(QPixmap(":/desktop/undo.svg"));
    nameResetTb_->setAutoRaise(true);
    nameResetTb_->setToolTip(tr("Clear filter"));
    grid->addWidget(nameResetTb_,row,2);
    row++;

    //keyword
    grid->addWidget(new QLabel(tr("Keyword:"),this),row,0);
    grid->setContentsMargins(1,1,1,1);
    grid->setVerticalSpacing(2);

    keywordCb_=new QComboBox(this);
    keywordCb_->setSizeAdjustPolicy(QComboBox::AdjustToMinimumContentsLengthWithIcon);
    keywordCb_->setMinimumContentsLength(cbWidth);
    keywordCb_->addItem("ANY");
    keywordCb_->addItems(MvQStyleDb::instance()->keywords());
    grid->addWidget(keywordCb_,row,1);

    keywordResetTb_=new QToolButton(this);
    keywordResetTb_->setIcon(QPixmap(":/desktop/undo.svg"));
    keywordResetTb_->setAutoRaise(true);
    keywordResetTb_->setToolTip(tr("Clear filter"));
    grid->addWidget(keywordResetTb_,row,2);
    row++;

    //colour
    grid->addWidget(new QLabel(tr("Colour:"),this),row,0);

    colourCb_=new QComboBox(this);
    colourCb_->setSizeAdjustPolicy(QComboBox::AdjustToMinimumContentsLengthWithIcon);
    colourCb_->setMinimumContentsLength(cbWidth);
    colourCb_->addItem("ANY");
    colourCb_->addItems(MvQStyleDb::instance()->colours());
    grid->addWidget(colourCb_,row,1);

    colourResetTb_=new QToolButton(this);
    colourResetTb_->setIcon(QPixmap(":/desktop/undo.svg"));
    colourResetTb_->setAutoRaise(true);
    colourResetTb_->setToolTip(tr("Clear filter"));
    grid->addWidget(colourResetTb_,row,2);
    row++;

    //layers
    grid->addWidget(new QLabel(tr("Style:"),this),row,0);

    layerCb_=new QComboBox(this);
    layerCb_->setSizeAdjustPolicy(QComboBox::AdjustToMinimumContentsLengthWithIcon);
    layerCb_->setMinimumContentsLength(cbWidth);
    layerCb_->addItem("ANY");
    layerCb_->addItems(MvQStyleDb::instance()->layers());
    grid->addWidget(layerCb_,row,1);

    layerResetTb_=new QToolButton(this);
    layerResetTb_->setIcon(QPixmap(":/desktop/undo.svg"));
    layerResetTb_->setAutoRaise(true);
    layerResetTb_->setToolTip(tr("Clear filter"));
    grid->addWidget(layerResetTb_,row,2);

    row++;

    grid->setColumnStretch(1,1);

    //tree view and model

    tree_=new QTreeView(this);
    tree_->setRootIsDecorated(false);
    tree_->setUniformRowHeights(true);
    tree_->setMinimumHeight(200);
    vb->addWidget(tree_,1);

    model_=new MvQStyleModel(this);
    tree_->setItemDelegate(new MvQStyleDelegate(this));

    sortModel_=new QSortFilterProxyModel(this);
    sortModel_->setFilterRegExp("1");
    sortModel_->setFilterRole(Qt::UserRole);
    sortModel_->setSortRole(MvQStyleModel::SortRole);
    sortModel_->setSourceModel(model_);

    tree_->setSortingEnabled(true);
    tree_->sortByColumn(1,Qt::AscendingOrder);
    tree_->setModel(sortModel_);

    connect(resetTb_,SIGNAL(clicked()),
            this,SLOT(slotClearFilter()));

    connect(nameLe_,SIGNAL(textChanged(QString)),
            this,SLOT(slotNameFilter(QString)));

    connect(nameResetTb_,SIGNAL(clicked()),
            nameLe_,SLOT(clear()));

    connect(keywordCb_,SIGNAL(currentIndexChanged(int)),
            this,SLOT(slotKeywordFilter(int)));

    connect(keywordResetTb_,SIGNAL(clicked()),
            this,SLOT(slotResetKeywordFilter()));

    connect(colourCb_,SIGNAL(currentIndexChanged(int)),
            this,SLOT(slotColourFilter(int)));

    connect(colourResetTb_,SIGNAL(clicked()),
            this,SLOT(slotResetColourFilter()));

    connect(layerCb_,SIGNAL(currentIndexChanged(int)),
            this,SLOT(slotLayerFilter(int)));

    connect(layerResetTb_,SIGNAL(clicked()),
            this,SLOT(slotResetLayerFilter()));

    connect(tree_,SIGNAL(clicked(QModelIndex)),
            this,SLOT(slotItemSelected(QModelIndex)));

    //Init the button states!!
    checkButtonState();
}

void MvQStyleSelectionWidget::slotKeywordFilter(int)
{
    if(ignoreFilterChanged_)
        return;

    QString s=keywordCb_->currentText();
    if(s ==  "ANY") s.clear();
    model_->setKeywordFilter(s);
    sortModel_->invalidate();
    checkButtonState();
}

void MvQStyleSelectionWidget::slotColourFilter(int)
{
    if(ignoreFilterChanged_)
        return;

    QString s=colourCb_->currentText();
    if(s ==  "ANY") s.clear();
    model_->setColourFilter(s);
    sortModel_->invalidate();
    checkButtonState();
}

void MvQStyleSelectionWidget::slotLayerFilter(int)
{
    if(ignoreFilterChanged_)
        return;

    QString s=layerCb_->currentText();
    if(s ==  "ANY") s.clear();
    model_->setLayerFilter(s);
    sortModel_->invalidate();
    checkButtonState();
}

void MvQStyleSelectionWidget::slotNameFilter(QString s)
{
    if(ignoreFilterChanged_)
        return;

    if(s == "ANY") s.clear();
    model_->setNameFilter(s);
    sortModel_->invalidate();
    checkButtonState();
}

void MvQStyleSelectionWidget::slotResetKeywordFilter()
{
    keywordCb_->setCurrentIndex(0);
}

void MvQStyleSelectionWidget::slotResetColourFilter()
{
    colourCb_->setCurrentIndex(0);
}

void MvQStyleSelectionWidget::slotResetLayerFilter()
{
    layerCb_->setCurrentIndex(0);
}

void MvQStyleSelectionWidget::slotItemSelected(const QModelIndex& idx)
{
    QModelIndex idxSrc=sortModel_->mapToSource(idx);
    if(idxSrc.isValid())
    {
        emit itemSelected(idxSrc.row());
    }
}

void MvQStyleSelectionWidget::slotClearFilter()
{
    ignoreFilterChanged_=true;
    nameLe_->clear();
    keywordCb_->setCurrentIndex(0);
    colourCb_->setCurrentIndex(0);
    layerCb_->setCurrentIndex(0);
    ignoreFilterChanged_=false;

    model_->clearFilter();
    sortModel_->invalidate();
    checkButtonState();
}

void MvQStyleSelectionWidget::checkButtonState()
{
    nameResetTb_->setEnabled(!nameLe_->text().isEmpty());
    keywordResetTb_->setEnabled(keywordCb_->currentText() != "ANY");
    colourResetTb_->setEnabled(colourCb_->currentText() != "ANY");
    layerResetTb_->setEnabled(layerCb_->currentText() != "ANY");

    resetTb_->setEnabled(keywordResetTb_->isEnabled() ||
                         nameResetTb_->isEnabled() || colourResetTb_->isEnabled() ||
                         layerResetTb_->isEnabled());
}

void MvQStyleSelectionWidget::setCurrent(const std::string& name)
{
    QModelIndex idxSrc=sortModel_->mapToSource(tree_->currentIndex());
    if(idxSrc.isValid())
    {
        int row=idxSrc.row();
        if(row >=0 && row < MvQStyleDb::instance()->items().size() &&
           MvQStyleDb::instance()->items()[row]->name().toStdString() == name)
           return;
    }

    slotClearFilter();
    int row=MvQStyleDb::instance()->indexOf(name);
    if(row >= 0)
    {
        QModelIndex idxSrc=model_->index(row,0);
        if(idxSrc.isValid())
        {
            tree_->setCurrentIndex(sortModel_->mapFromSource(idxSrc));
        }
    }
}

//==================================================
//
//  MvQStyleHelp
//
//==================================================

MvQStyleHelp::MvQStyleHelp(RequestPanel& owner,const Parameter& param) :
           MvQRequestPanelHelp(owner,param)
{
    selector_=new MvQStyleSelectionWidget(parentWidget_);

    connect(selector_,SIGNAL(itemSelected(int)),
            this,SLOT(slotSelected(int)));

}

void MvQStyleHelp::slotSelected(int idx)
{
    if(idx >=0  && idx < MvQStyleDb::instance()->items().count())
    {
        if(MvQStyleDb::instance()->items()[idx]->name() != oriName_)
        {
            std::vector<string> vs;
            vs.push_back(metview::toString<int>(idx));
            emit edited(vs);
            oriName_.clear();
        }
    }
}

void MvQStyleHelp::refresh(const vector<string>& values)
{
    if(values.size() == 0)
        return;

    oriName_=QString::fromStdString(values[0]);
    selector_->setCurrent(values[0]);
}

static HelpMaker<MvQStyleHelp> maker2("help_style");
