/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
* prf_shl.c                                                                    *
* Copyright (c) 2003-2005 Troy Hanson                                          *
*******************************************************************************/
#include "libut/ut_internal.h"


#define PCT_MAXLEN 5

extern UT_prf_global_type UT_prf_global;
extern char *UT_prf_scale_strs[];


static char *te_non_1s_15min_hdr[] = { "name                 ", "count  ", 
        "<1s", "1-10s", "10-60s", "1-2M", "2-5M", "5-10M", "10-15M", ">15M", NULL};
static char *te_log_1u_10s_hdr[] = { "name                 ", "count  ", 
        "<1u", "1-10u", "10-100u", "0.1-1m", "1-10m", "10-100m", "0.1-1s", 
        "1-10s", "10-100s", ">100s", NULL };
static char *te_log_1m_10s_hdr[] = { "name                 ", 
        "count  ", "<1m", "1-10m", "10-100m", "0.1-1s", "1-10s", ">10s", NULL };
static char *te_100u_1m_hdr[] = { "name                 ", "count  ", 
        "<100u", "100-200u", "200-300u", "300-400u", "400-500u", "500-600u", 
        "600-700u", "700-800u", "800-900u", "900-1000u", ">1m", NULL };
static char *te_10u_100u_hdr[] = { "name                 ", "count  ", 
        "<10u", "10-20u", "20-30u", "30-40u", "40-50u", "50-60u", 
        "60-70u", "70-80u", "80-90u", "90-100u", ">100u", NULL };
static char *te_minute_hdr[] = { "name                 ", "count  ", 
        "<1s", "1-9s", "10-19s", "20-29s", "30-39s", "40-49s", 
        "50-59s", ">60s", NULL };
static char *te_hour_hdr[] = { "name                 ", "count  ", 
        "<1m", "1-9m", "10-19m", "20-29m", "30-39m", "40-49m", 
        "50-59m", ">60m", NULL };

#define EL(x,y) {x,x ## _hdr},
struct hdrmap {
      UT_prf_scale type;
      char **hdr;
} hdrtbl[] = { PRF_SCALES {-1,NULL} };
#undef EL

/******************************************************************************
 * UT_prf_shlcmd()                                                            *
 * Impleementation of the shl "prf" command                                   *
 *****************************************************************************/
int UT_prf_shlcmd( int argc, char *argv[] ) {
    int i;

    switch (argc) {
        case 1:
            UT_prf_show_all();
            break;
        case 2:
            if (!strcmp(argv[1],"scale")) {
                UT_shlf("Supported scales are: \n");
                for (i=0; UT_prf_scale_strs[i]; i++ ) 
                    UT_shlf("(%d) %s\n", i, UT_prf_scale_strs[i] );
            } else UT_prf_show_prf( argv[1] );
            break;
        case 4:
            /* support the "-rescale" option */
            if (!strcmp("scale", argv[1])) {
                    UT_prf_rescale_shlcmd( argv[2], argv[3]);
                    break;
            } 
        default:
            UT_prf_show_usage(0,NULL);
            break;
    }
    return SHL_OK;
}

/******************************************************************************
 * UT_prf_show_usage()                                                        *
 * print usage help text                                                      *
 *****************************************************************************/
int UT_prf_show_usage(int argc,char*argv[]) {
        UT_shlf("Usage:\n");
        UT_shlf("prf                       -- view all prfs\n");
        UT_shlf("prf <prf>                 -- view <prf>\n");
        UT_shlf("prf scale                 -- list scales\n");
        UT_shlf("prf scale <prf> <scale>   -- rescale <prf>\n");
}


/******************************************************************************
 * UT_prf_show_all()                                                          *
 * Print all the prfs out                                                     *
 *****************************************************************************/
int UT_prf_show_all() {
    UT_prf *prf;
    char *hdr= 
        "\n=================================================================\n";

    prf = UT_prf_global.prfs;

    while (prf) {
        UT_shlf("%s%16s: %s%s", hdr, prf->name, prf->desc, hdr );
        UT_prf_show_prf( prf->name );
        prf = prf->next;
    }
    return 0;
}

/******************************************************************************
 * UT_prf_show_prf()                                                          *
 * Display the prf header and the prf itself                                  *
 *****************************************************************************/
int UT_prf_show_prf( char *prfname ) {
    UT_prf *prf;
    UT_prf_row *row;
    char **cols;
    struct hdrmap *map;

    LL_FIND( UT_prf_global.prfs, prf, prfname )
    if (!prf) {
        UT_shlf("prf %s not found", prfname );
        return 0;
    }

    /* Figure out the right header to use for this scale */
    map = hdrtbl;
    while (map->type != -1 && map->type != prf->scale_type) map++;
    if (map->type == -1) {
            UT_LOG(Error, "Unknown prf scale %d", prf->scale_type);
            return;
    }
    cols = map->hdr;
    
    /* print the prf header */
    UT_shl_hdr( cols );

    /* Loop over the prf rows and print the columns. */
    row = prf->rows;
    while (row) {
        UT_prf_fmt_prfrow( prf, row, cols );
        row = row->next;
    }
    return 0;
}

/******************************************************************************
 * UT_prf_fmt_prfrow()                                                        *
 * Format a prf row using the given NULL-terminated array of headers          *
 *****************************************************************************/
int UT_prf_fmt_prfrow( UT_prf *prf, UT_prf_row *row, char *hdrs[] ) {
    unsigned b,wid,sum=0;
    double pct;
    char str[PCT_MAXLEN]; /* longest value is "100%" + null terminator */

    /* count the total hits across buckets in this row */
    for ( b = 0; b < prf->num_buckets_per_row; b++) sum += row->bucket_hits[b];

    /* assume the first two columns are name and count and treat them special.*/
    UT_shlf("%-*s ", (*hdrs ? strlen(*(hdrs++)) : 1), row->name );
    UT_shlf("%-*u ", (*hdrs ? strlen(*(hdrs++)) : 1), sum );

    /* Print the x% in each bucket; but leave empty buckets blank */
    if (sum > 0) {
        for ( b = 0; b < prf->num_buckets_per_row; b++) {
            wid = (*hdrs ? strlen(*(hdrs++)) : 1);
            pct = row->bucket_hits[b] * 100.0 / sum;
            
            if (row->bucket_hits[b] == 0) str[0] = '\0';
            else snprintf( str, PCT_MAXLEN, "%.0f%%", pct);

            UT_shlf( "%*s " , wid, str);  
        }
    }

    UT_shlf("\n");
}

/******************************************************************************
 * UT_prf_rescale_shlcmd()                                                    *
 * Rescales a prf; new data added to the prf accumulates using the new buckets*
 *****************************************************************************/
int UT_prf_rescale_shlcmd( char *prf_name, char *new_scale) {
    UT_prf *prf;
    int i,idx,isnum;
    char *c;

    /* Make sure the prf exists */
    LL_FIND(UT_prf_global.prfs, prf, prf_name)
    if (!prf) {
        UT_shlf("prf %s not found", prf_name);
        return SHL_OK;
    }

    /* Make sure the requested scale is a supported one. */
    /* did they pass a numeric or string prf scale name ? */
    for (c=new_scale,isnum=1;*c != '\0';c++) if (*c < '0' || *c > '9') isnum=0;
    if (isnum) {
        for (i=0; UT_prf_scale_strs[i]; i++) {}  /* i = num strings */
        sscanf(new_scale,"%d",&idx);
        if (idx >= i) idx = -1;
    } else {
        idx = UT_stridx(new_scale,UT_prf_scale_strs);
    }

    if (idx != -1) {
       UT_prf_rescale( prf_name, idx );
       UT_shlf("scaled prf %s to %s", prf_name, new_scale );
    } else {
        UT_shlf("Supported scales are: \n");
        for (i=0; UT_prf_scale_strs[i]; i++ ) 
            UT_shlf("(%d) %s\n", i, UT_prf_scale_strs[i] );
    }
    
    return SHL_OK;
}


/******************************************************************************
 * UT_prf_init_shl()                                                          *
 * Register the prf shl command                                               *
 *****************************************************************************/
int UT_prf_init_shl( void ) {
    UT_shl_cmd_create( "prf", "Performance/profiling stats",
            (UT_shlcmd *)UT_prf_shlcmd, (UT_helpcb*)UT_prf_show_usage);
    return 0;
}

