/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <sys/time.h>
#include <string.h>
#include <errno.h>
#include "libut/ut_internal.h"


extern int UT_prf_init_shl(void);

/* A trick to define strings matching the prf scale type enum */
#define EL(x,y) #x,
char *UT_prf_scale_strs[] = { PRF_SCALES NULL };
#undef EL
#define EL(x,y) y,
int UT_prf_scale_num_buckets[] = { PRF_SCALES };
#undef EL


/*******************************************************************************
* prf.c                                                                        *
* Copyright (c) 2003-2005 Troy Hanson                                          *
*******************************************************************************/
UT_prf_global_type UT_prf_global  = { NULL };

void UT_prf_init() {
    UT_mem_pool_create( UTPRFS, sizeof(UT_prf), 10 );
    UT_mem_pool_create( UTPRF_ROWS, sizeof(UT_prf_row), 10 );
    UT_mem_pool_create( UTPRF_BUCKETS, sizeof(unsigned), 100 );
    UT_mem_pool_create( UTPRF_BUCKET_TOPS, sizeof(struct timeval), 10 );
    UT_prf_init_shl();
}

UT_API void UT_prf_create( char *prf_name, char *desc, UT_prf_scale scale_type) {
    UT_prf *prf,*tmp;
    int i, bucket_top;

    /* Make the prf structure */
    prf = (UT_prf*)UT_mem_alloc(UTPRFS, 1);
    UT_strncpy( prf->name, prf_name, UT_PRF_NAME_MAXLEN);
    UT_strncpy( prf->desc, desc, UT_PRF_DESC_MAXLEN);
    prf->scale_type = scale_type;
    prf->num_buckets_per_row = UT_prf_scale_num_buckets[scale_type];  
    prf->bucket_top = (struct timeval*)UT_mem_alloc(UTPRF_BUCKET_TOPS, 
                        prf->num_buckets_per_row );
    memset(prf->bucket_top, 0, prf->num_buckets_per_row * sizeof(struct timeval));
    prf->rows = NULL;
    prf->next = NULL;

    /* Initialize the bucket_top array appropriately for this scale type */
    switch (scale_type) {
        case te_non_1s_15min:
            prf->bucket_top[0].tv_usec = 0; prf->bucket_top[0].tv_sec = 1; 
            prf->bucket_top[1].tv_usec = 0; prf->bucket_top[1].tv_sec = 10; 
            prf->bucket_top[2].tv_usec = 0; prf->bucket_top[2].tv_sec = 60; 
            prf->bucket_top[3].tv_usec = 0; prf->bucket_top[3].tv_sec = 120; 
            prf->bucket_top[4].tv_usec = 0; prf->bucket_top[4].tv_sec = 5*60; 
            prf->bucket_top[5].tv_usec = 0; prf->bucket_top[5].tv_sec = 10*60; 
            prf->bucket_top[6].tv_usec = 0; prf->bucket_top[6].tv_sec = 15*60; 
            prf->bucket_top[7].tv_usec = 0; prf->bucket_top[7].tv_sec = 20*60; /* arb */
            break;
        case te_log_1u_10s:
            for(i=0; i< prf->num_buckets_per_row; i++) {
                prf->bucket_top[i].tv_usec = (i == 0) ? 1 : (prf->bucket_top[i-1].tv_usec * 10);
                prf->bucket_top[i].tv_sec = (i == 0) ? 0 : (prf->bucket_top[i-1].tv_sec);
                while (prf->bucket_top[i].tv_usec >= 1000000) {
                    prf->bucket_top[i].tv_sec++;
                    prf->bucket_top[i].tv_usec -= 1000000;
                }
            }
            break;
        case te_log_1m_10s:
            for(i=0; i< prf->num_buckets_per_row; i++) {
                prf->bucket_top[i].tv_usec = (i == 0) ? 1000 : (prf->bucket_top[i-1].tv_usec * 10);
                prf->bucket_top[i].tv_sec = (i == 0) ? 0 : (prf->bucket_top[i-1].tv_sec);
                while (prf->bucket_top[i].tv_usec >= 1000000) {
                    prf->bucket_top[i].tv_sec++;
                    prf->bucket_top[i].tv_usec -= 1000000;
                }
            }
            break;
        case te_100u_1m:
            for(i=0; i< prf->num_buckets_per_row; i++) {
                prf->bucket_top[i].tv_usec = (i == 0) ? 100 : (prf->bucket_top[i-1].tv_usec + 100);
                prf->bucket_top[i].tv_sec = (i == 0) ? 0 : (prf->bucket_top[i-1].tv_sec);
                while (prf->bucket_top[i].tv_usec >= 1000000) {
                    prf->bucket_top[i].tv_sec++;
                    prf->bucket_top[i].tv_usec -= 1000000;
                }
            }
            break;
        case te_10u_100u:
            for(i=0; i< prf->num_buckets_per_row; i++) {
                prf->bucket_top[i].tv_usec = (i == 0) ? 10 : (prf->bucket_top[i-1].tv_usec + 10);
                prf->bucket_top[i].tv_sec = (i == 0) ? 0 : (prf->bucket_top[i-1].tv_sec);
                while (prf->bucket_top[i].tv_usec >= 1000000) {
                    prf->bucket_top[i].tv_sec++;
                    prf->bucket_top[i].tv_usec -= 1000000;
                }
            }
            break;
        case te_minute:
            for(i=0; i< prf->num_buckets_per_row; i++) {
                prf->bucket_top[i].tv_usec = 0;
                prf->bucket_top[i].tv_sec = (i == 0) ? 1 : (prf->bucket_top[i-1].tv_sec + 8);
            }
            break;
        case te_hour:
            for(i=0; i< prf->num_buckets_per_row; i++) {
                prf->bucket_top[i].tv_usec = 0;
                prf->bucket_top[i].tv_sec = (i == 0) ? 60 : (prf->bucket_top[i-1].tv_sec + 8*60);
            }
            break;
        default:
            UT_LOG( Error, "Unsupported scale %d for prf %s", scale_type, prf_name );
            UT_mem_free( UTPRF_BUCKET_TOPS, prf->bucket_top, prf->num_buckets_per_row );
            UT_mem_free( UTPRFS, prf, 1 );
            return;
    }   

    /* add the structure to the global list of prfs */
    HASH_ADD_STR( UT_prf_global.prfs, tmp, name, prf );
}

/* This function is used internally to create a new, empty row within a prf. */
UT_prf_row *UT_create_prf_row( UT_prf *prf, char *row_name ) {
    UT_prf_row *row, *rowtmp;
    int i;

    row = (UT_prf_row*)UT_mem_alloc( UTPRF_ROWS, 1);
    UT_strncpy( row->name, row_name, UT_PRF_NAME_MAXLEN);
    row->bucket_hits =(unsigned*)UT_mem_alloc(UTPRF_BUCKETS,prf->num_buckets_per_row);
    for( i=0; i < prf->num_buckets_per_row; i++) row->bucket_hits[i] = 0; 
    row->tv_start.tv_sec = 0;
    row->tv_start.tv_usec = 0;
    row->next = NULL;
    HASH_ADD_STR( prf->rows, rowtmp, name, row );

    return row;
}

/**********************************************************************
 * UT_prf_bgn  --   Mark beginning of a timing interval that'll become*
 * a prf statistic                                                    *
 *********************************************************************/
UT_API void UT_prf_bgn( char *prf_name, char *row_name) {
    UT_prf *prf;
    UT_prf_row *row;

    HASH_FIND_STR( UT_prf_global.prfs, prf, name, prf_name);
    if (!prf) {
        UT_LOG ( Error, "Can't add data to non-existent prf %s", prf_name);
        return;
    }

    /* Find the intended prf row, creating it if necessary. */
    HASH_FIND_STR( prf->rows, row, name, row_name);
    if (!row) row = UT_create_prf_row( prf, row_name );

    /* Store the current time the prf row as the start time for this measurement */
    if ( gettimeofday( &row->tv_start, NULL) != 0 ) {
        UT_LOG( Error, "gettimeofday error: %s", strerror(errno) );
        return;
    }
}

/********************************************************************************
 * UT_prf_end  --   Mark end of a timing interval and add it into the prf statistics
 *******************************************************************************/
UT_API void UT_prf_end( char *prf_name, char *row_name) {
    UT_prf *prf;
    UT_prf_row *row;
    int b=0;
    struct timeval tv, tv_elapsed;

    HASH_FIND_STR( UT_prf_global.prfs, prf, name, prf_name);
    if (!prf) {
        UT_LOG ( Error, "Can't add data to non-existent prf %s", prf_name);
        return;
    }

    /* Find the intended prf row. */
    HASH_FIND_STR( prf->rows, row, name, row_name);
    if (!row) {
        UT_LOG( Error, "Can't end a prf interval that hasn't begun");
        return;
    }

    /* Grab the current time, and figure out time elapsed since prf bgn. */
    if ( gettimeofday( &tv, NULL) != 0 ) {
        UT_LOG( Error, "gettimeofday error: %s", strerror(errno) );
        return;
    }

    timersub(&tv, &row->tv_start, &tv_elapsed);

    /* Figure out which bucket this elapsed time falls into, and incrment  */
    while ((b < prf->num_buckets_per_row - 1) && 
          (tv_elapsed.tv_sec > prf->bucket_top[b].tv_sec || 
          (tv_elapsed.tv_sec == prf->bucket_top[b].tv_sec && tv_elapsed.tv_usec > prf->bucket_top[b].tv_usec))) b++;
    row->bucket_hits[b]++;
}

/******************************************************************
 * UT_prf_rescale()                                               *
 * Change the scale of a prf, by deleting it and recreating it.   *
 *****************************************************************/
void UT_prf_rescale( char *prf_name, UT_prf_scale new_scale ) {
    UT_prf *prf;
    char name[UT_PRF_NAME_MAXLEN];
    char desc[UT_PRF_DESC_MAXLEN];
    UT_prf_scale scale_type;

    HASH_FIND_STR( UT_prf_global.prfs, prf, name, prf_name);
    if (!prf) {
        UT_LOG(Error, "Cannot delete prf %s - not found");
        return;
    }

    /* Grab the definition of the prf so we can re-create */
    UT_strncpy(name, prf->name, UT_PRF_NAME_MAXLEN );
    UT_strncpy(desc, prf->desc, UT_PRF_DESC_MAXLEN );
    scale_type = prf->scale_type;

    /* Out with the old, in with the new. */
    UT_prf_del( prf_name );
    UT_prf_create( prf_name, desc, new_scale); 
}

/******************************************************************
 * UT_prf_del()  Deletes a prf and its rows.                      *
 *****************************************************************/
void UT_prf_del( char *prf_name ) {
    UT_prf *prf,*tmp;
    UT_prf_row *row;

    HASH_FIND_STR( UT_prf_global.prfs, prf, name, prf_name);
    if (!prf) {
        UT_LOG(Error, "Cannot delete prf %s - not found");
        return;
    }

    row = prf->rows;
    while (row) {
        UT_mem_free( UTPRF_BUCKETS, row->bucket_hits, prf->num_buckets_per_row);
        UT_mem_free( UTPRF_ROWS, row, 1 );
        row = row->next;
    }
    UT_mem_free( UTPRF_BUCKET_TOPS, prf->bucket_top, prf->num_buckets_per_row);
    UT_mem_free( UTPRFS, prf, 1 );

    HASH_DEL( UT_prf_global.prfs, tmp, prf );

    UT_LOG( Debug, "Deleted prf %s", prf_name);
}
