/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <sys/time.h>

/**********************************************************
 *  UT prf subs     copyright (c) 2003 Troy Hanson        *
 *                                                        *
 * Performance/profiling information graphs               * 
 *                                                        * 
 * A histogram stores the # of data points in each bucket *
 **********************************************************/

/* These are the names of memory pools. */
#define UTPRFS "ut_prf"
#define UTPRF_ROWS "ut_prf_row"
#define UTPRF_BUCKETS "ut_prf_bkt_cnt"
#define UTPRF_BUCKET_TOPS "ut_prf_bkt_top"

#define UT_PRF_NAME_MAXLEN 40
#define UT_PRF_DESC_MAXLEN 80

/********************************************************************
 * te_non_1s_15min                      
 *      
 *      bucket range          top of bucket
 *      -----------------     -------------
 *      0s -     1s              
 *      1s -    10s             
 *      10s -   60s            
 *      60s -  2min              
 *     2min -  5min             
 *     5min - 10min             
 *    10min - 15min             
 *    15min - inf
 *******************************************************************/
/********************************************************************
 * te_log_1m_10s                  
 *      
 *      bucket range          top of bucket
 *      -----------------     -------------
 *      0 - 1 msec            10^3 usec           
 *      1 - 10 msec           10^4 usec           
 *      10 - 100 msec         10^5 usec             
 *      1/10 s - 1 s          10^6 usec            
 *      1s - 10s              10^7 usec        
 *      10s - inf             infinity                  
 *******************************************************************/

/********************************************************************
 * te_log_1u_10s:
 *      
 *      bucket range          top of bucket
 *      -----------------     -------------
 *      0 - 1 usec            10^0 usec          
 *      1 - 10 usec           10^1 usec           
 *      10 - 100 usec         10^2 usec             
 *      100 - 1000 usec       10^3 usec               
 *      1 - 10 msec           10^4 usec           
 *      10 - 100 msec         10^5 usec             
 *      1/10 s - 1 s          10^6 usec            
 *      1s - 10s              10^7 usec        
 *      10s - 100s            10^8 usec
 *      100s - inf            infinity                  
 *******************************************************************/

/********************************************************************
 * te_10u_1u:
 *      
 *      bucket range            top of bucket
 *      -----------------       -------------
 *      0.00 - 0.01 msec        10 usec
 *      0.01 - 0.02 msec        20 usec
 *      0.02 - 0.03 msec        30 usec
 *      0.03 - 0.04 msec        40 usec
 *      0.04 - 0.05 msec        50 usec
 *      0.05 - 0.06 msec        60 usec
 *      0.06 - 0.07 msec        70 usec
 *      0.07 - 0.08 msec        80 usec
 *      0.08 - 0.09 msec        90 usec
 *      0.09 - 0.10 msec        100 usec
 *      1msec - inf             infinity                  
 *******************************************************************/

/********************************************************************
 * te_100u_1m:
 *      
 *      bucket range          top of bucket
 *      -----------------     -------------
 *      0.0 - 0.1 msec          100 usec
 *      0.1 - 0.2 msec          200 usec
 *      0.2 - 0.3 msec          300 usec
 *      0.3 - 0.4 msec          400 usec
 *      0.4 - 0.5 msec          500 usec
 *      0.5 - 0.6 msec          600 usec
 *      0.6 - 0.7 msec          700 usec
 *      0.7 - 0.8 msec          800 usec
 *      0.8 - 0.9 msec          900 usec
 *      0.9 - 1 msec            1000 usec
 *      1msec - inf             infinity                  
 *******************************************************************/

/******************************************************************
 * te_minute
 *      bucket range          top of bucket
 *      -----------------     -------------
 *      0 - 1 sec             1 sec
 *      1 - 9 sec             9 sec
 *      10 -19 sec            19 sec
 *      20 -29 sec            29 sec
 *      30 -39 sec            39 sec
 *      40 -49 sec            49 sec
 *      50 -59 sec            59 sec
 *      60sec - inf           infinity
 *****************************************************************/

/******************************************************************
 * te_hour
 *      bucket range          top of bucket
 *      -----------------     -------------
 *      0 - 1 min             1 min
 *      1 - 9 min             9 min
 *      10 -19 min            19 min
 *      20 -29 min            29 min
 *      30 -39 min            39 min
 *      40 -49 min            49 min
 *      50 -59 min            59 min
 *      60min - inf           infinity
 *****************************************************************/

typedef struct UT_prf_row {
        char name[UT_PRF_NAME_MAXLEN];
        unsigned *bucket_hits;   /* ptr to alloc'd array */
        struct timeval tv_start; /* start time of current measurement */
        UT_hash_handle hh;
        struct UT_prf_row *next;
} UT_prf_row;

typedef struct UT_prf {
        char name[UT_PRF_NAME_MAXLEN];
        char desc[UT_PRF_DESC_MAXLEN];
        UT_prf_scale scale_type;
        unsigned num_buckets_per_row;
        struct timeval *bucket_top;  /* ptr to alloc'd array */
        UT_prf_row *rows;
        UT_hash_handle hh;
        struct UT_prf *next;
} UT_prf;

/* There is one global of the following struct to consolidate prf data */
typedef struct UT_prf_global_type {
        UT_prf *prfs;
} UT_prf_global_type;

void UT_prf_del(char *prf_name);
void UT_prf_rescale(char *prf_name, UT_prf_scale type);
