/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <string.h>
#include "libut/ut_internal.h"


extern UT_var_global_type UT_var_global;

/*******************************************************************************
* UT_var_get_restriction_data()                                                *
* Convenience function to lookup whether UT_var_restrict data is set on a var. *
*******************************************************************************/
void *UT_var_get_restriction_data(UT_var *var) {
    UT_callback *cbdata;
    for(cbdata=var->cb;cbdata;cbdata=cbdata->next) {
        if (cbdata->cb.varcb == UT_var_restrict_cb) return cbdata->data;
    }
    return NULL;
}

/*******************************************************************************
* UT_var_shlcmd()                                                              *
*******************************************************************************/
int UT_var_shlcmd(int argc, char *argv[] ) {
    UT_var *var;
    void *data;
    double *bounds_f,f_val;
    int *bounds_i,i_val;
    char **s;
    char *cols[] = {" name                ", 
                    "description                   ", 
                    "value               ", NULL };

    switch (argc) {
        case 1:
            UT_shl_hdr(cols);
            for(var = UT_var_global.vars; var; var=var->next) {
                UT_shlf("%c%-20.20s %-30.30s ", 
                        (var->flags & VAR_READONLY ?  ' ' : '*'),
                        var->name, 
                        var->desc);
                switch (var->type) {
                    case UT_var_double:
                        UT_shlf("%f\n", var->val.f);
                        break;
                    case UT_var_int:
                        UT_shlf("%d\n", var->val.i);
                        break;
                    case UT_var_string:
                        UT_shlf("%s\n", var->val.s);
                        break;
                }
            }
            UT_shlf("\nVariables prefixed with '*' can be changed.\n");
            break;
        case 2:
            HASH_FIND_STR(UT_var_global.vars, var, name, argv[1]);
            if (!var) {
                UT_shle("No such variable %s", argv[1]);
                return SHL_ERROR;
            }
            switch (var->type) {
                case UT_var_double:
                    UT_shlf("%f\n", var->val.f);
                    if ( (data = UT_var_get_restriction_data(var)) != NULL ) {
                        bounds_f = (double*)data;
                        UT_shlf("Min: %f\nMax: %f\n", bounds_f[0],bounds_f[1]);
                    }
                    break;
                case UT_var_int:
                    UT_shlf("%d\n", var->val.i);
                    if ( (data = UT_var_get_restriction_data(var)) != NULL ) {
                        bounds_i = (int*)data;
                        UT_shlf("Min: %d\nMax: %d\n", bounds_i[0],bounds_i[1]);
                    }
                    break;
                case UT_var_string:
                    UT_shlf("%s\n", var->val.s);
                    if ( (data = UT_var_get_restriction_data(var)) != NULL ) {
                        UT_shlf("\nEnumerated allowed values:\n");
                        for(s=(char**)data;*s != NULL; s++) 
                            UT_shlf(" %s\n",*s);
                    }
                    break;
                default:
                    UT_LOG(Fatal,"unknown type of var %s", argv[1]);
                    break;
            }
            break;
        case 4:
            if (strcmp(argv[2],"=") != 0) {
                UT_shle("usage error\n");
                return SHL_ERROR;
            }
            /* no break; fall into case 3 */
        case 3:
            /* var name [=] value */
            HASH_FIND_STR(UT_var_global.vars, var, name, argv[1]);
            if (!var) {
                UT_shle("No such variable %s", argv[1]);
                return SHL_ERROR;
            }
            switch (var->type) {
                case UT_var_double:
                    if (sscanf(argv[argc-1], "%lf", &f_val) != 1) {
                        UT_shle("non-numeric value %s\n", argv[argc-1]);
                        return SHL_ERROR;
                    }
                    if ( (data = UT_var_get_restriction_data(var)) != NULL ) {
                        bounds_f = (double*)data;
                        if (f_val < bounds_f[0] || f_val > bounds_f[1]) {
                            UT_shle("out of bounds value %f\n", f_val);
                            return SHL_ERROR;
                        }
                    }
                    UT_var_set_shl(var,f_val);
                    break;
                case UT_var_int:
                    if (sscanf(argv[argc-1], "%d", &i_val) != 1) {
                        UT_shle("non-numeric value %s\n", argv[argc-1]);
                        return SHL_ERROR;
                    }
                    if ( (data = UT_var_get_restriction_data(var)) != NULL ) {
                        bounds_i = (int*)data;
                        if (i_val < bounds_i[0] || i_val > bounds_i[1]) {
                            UT_shle("out of bounds value %d\n", i_val);
                            return SHL_ERROR;
                        }
                    }
                    UT_var_set_shl(var,i_val);
                    break;
                case UT_var_string:
                    if ( (data = UT_var_get_restriction_data(var)) != NULL ) {
                        if (UT_stridx(argv[argc-1],
                                    (char**)data) < 0) {
                            UT_shle("enumeration forbids %s\n", 
                                    argv[argc-1]);
                            return SHL_ERROR;
                        }
                    }
                    UT_var_set_shl(var,argv[argc-1]);
                    break;
                default:
                    UT_LOG(Fatal,"unknown type of var %s", argv[1]);
                    break;
            }
            break;
    }
    return SHL_OK;
}

int UT_var_usage(int argc,char*argv[]) {
    UT_shlf("Usage:\n");
    UT_shlf(" var                    -- show all vars\n");
    UT_shlf(" var varname            -- show named var\n");
    UT_shlf(" var varname [=] newval -- set var to new value\n");
}

void UT_var_shl_init() {
    UT_shl_cmd_create( "var", "Display or set config variables",
            (UT_shlcmd *)UT_var_shlcmd, (UT_helpcb*)UT_var_usage);
}

