/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/********************************************************************************
 *  Routines to support shl configuration variables.                            *
 *  copyright (c) 2003-2005 Troy Hanson                                         *
 *******************************************************************************/

#include "libut/ut_internal.h"
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <limits.h>



UT_var_global_type UT_var_global = { 
    .vars  = NULL
};


int UT_var_init(void) {
    UT_mem_pool_create( UTSHL_VARS, sizeof(UT_var), 10);
    UT_mem_pool_create( CBDATA, sizeof(UT_callback), 10 );
    UT_var_shl_init();
}

UT_API int UT_var_create(char *name, char *desc, UT_var_type type, ...) {
    UT_var *var, *tmp;
    va_list ap;

    var = (UT_var *)UT_mem_alloc( UTSHL_VARS, 1);
    UT_strncpy( var->name, name, UT_VAR_NAME_MAXLEN );
    UT_strncpy( var->desc, desc, UT_VAR_DESC_MAXLEN );
    var->type = type;
    memset(&var->val, 0, sizeof(var->val));
    var->bound_cvar = NULL;
    var->cb = NULL;
    var->flags = 0;
    var->next = NULL;

    va_start(ap,type);
    UT_var_set2(var,ap);
    va_end(ap);

    HASH_ADD_STR(UT_var_global.vars, tmp, name, var);
    return 0;
}

UT_API int UT_var_set(char *name, ... ) {
    va_list ap;
    UT_var *var;
    int rc;

    HASH_FIND_STR(UT_var_global.vars, var, name, name);
    if (!var) UT_LOG(Fatal, "can't set non-existent variable %s", name);

    va_start(ap,name);
    rc = UT_var_set2(var,ap);
    va_end(ap);

    return rc;
}

int UT_var_set2(UT_var *var, va_list ap) {
    char *c;
    void *oldv;
    UT_callback *cbdata;
    UT_var_upd_cb *failcb;
    int rc=0;

    oldv = (void*)malloc(sizeof(var->val));
    memcpy(oldv, &var->val, sizeof(var->val));
    switch (var->type) {
        case UT_var_double:
            var->val.f = va_arg(ap,double);
            break;
        case UT_var_int:
            var->val.i = va_arg(ap,int);
            break;
        case UT_var_string:
            c = va_arg(ap,char*);
            if (c == NULL) var->val.s = c;
            else {
                var->val.s = (char*)malloc( strlen(c) + 1 );
                if (var->val.s == NULL) UT_LOG(Fatal, "malloc failure");
                else strcpy(var->val.s, c);
            }
            break;
        default:
            UT_LOG(Fatal, "Unknown var type %d", var->type);
    }
    if (var->bound_cvar) UT_var_set_cvar(var);
    /* invoke chain of callbacks, however if there is more than one, and
     * one (not the first one) averts the update, the earlier one(s) will
     * need to be retroactively called with the restored value, below. */
    for(cbdata=var->cb; cbdata && rc >= 0; cbdata=cbdata->next) {
        if ( (rc = (cbdata->cb.varcb)(var->name,cbdata->data)) < 0) {
            UT_LOG(Warning, "%s change averted by cb", var->name);
            if (var->type == UT_var_string && var->val.s) free(var->val.s);
            memcpy(&var->val,oldv,sizeof(var->val));
            if (var->bound_cvar) UT_var_set_cvar(var);
            failcb = cbdata->cb.varcb;
        }
    }
    /* Re-run the cb's prior to the one that failed, w/the restored value. * 
     * This time, they cannot revert the change however.                   */
    if (rc < 0) {
        for(cbdata=var->cb; cbdata ; cbdata=cbdata->next) {
            if (cbdata->cb.varcb == failcb) break;
            if ( (rc = (cbdata->cb.varcb)(var->name,cbdata->data)) < 0) {
                UT_LOG(Warning, "Var %s change averted AND reversion averted", 
                        var->name);
            }
        }
    } else if (var->type == UT_var_string && (c = *(char**)oldv)) free(c);
    free(oldv);
    return rc;
}

int UT_var_set_shl(UT_var *var, ...) {
    va_list ap;
    int rc;

    var->flags |= VAR_IN_SHL_CONTEXT;

    va_start(ap,var);
    rc = UT_var_set2(var, ap);
    va_end(ap);

    var->flags &= ~VAR_IN_SHL_CONTEXT;
    return rc;
}

int UT_var_set_cvar(UT_var *var) {
    switch (var->type) {
        case UT_var_double:
            if (var->bound_cvar) *((double*)var->bound_cvar)= var->val.f;
            break;
        case UT_var_int:
            if (var->bound_cvar) *((int*)var->bound_cvar) = var->val.i;
            break;
        case UT_var_string:
            if (var->bound_cvar) *((char**)var->bound_cvar) = var->val.s;
            break;
        default:
            UT_LOG(Fatal, "Unknown var type %d", var->type);
            break;
    }
}

UT_API int UT_var_get(char *name, void *val) {
    UT_var *var;

    HASH_FIND_STR(UT_var_global.vars, var, name, name);
    if (!var) UT_LOG(Fatal, "can't get non-existent variable %s", name);

    switch (var->type) {
        case UT_var_double:
            *(double*)val = var->val.f;
            break;
        case UT_var_int:
            *(int*)val = var->val.i;
            break;
        case UT_var_string:
            *(char**)val = var->val.s;
            break;
        default:
            UT_LOG(Fatal, "Unknown var type %d", var->type);
            break;
    }
    return 0;
}

/*******************************************************************************
*  UT_var_reg_cb()                                                             *
*  Specifies a callback to invoke when the given var is written to.            *
*******************************************************************************/
UT_API int UT_var_reg_cb( char *name, UT_var_upd_cb *cb, void *data ) {
    UT_var *var;
    UT_callback *cbdata,*tmp;

    HASH_FIND_STR(UT_var_global.vars, var, name, name);
    if (!var) UT_LOG(Fatal, "can't set cb on non-existent variable %s", name);

    cbdata = (UT_callback*)UT_mem_alloc(CBDATA, 1);
    cbdata->cb.varcb = cb;
    cbdata->data = data;
    cbdata->next = NULL;
    LL_ADD(var->cb, tmp, cbdata);
    return 0;
}

/*******************************************************************************
* UT_var_bindc()                                                               *
* Designate a C variable to be updated when a var is written to.               *
* The type of cvar must be int*, double*, or char** appropriate to the var.    *
* The C variable is meant for "read only" use. To write to it use UT_var_set.  *
*******************************************************************************/
UT_API int UT_var_bindc( char *name, void *cvar ) {
    UT_var *var;

    HASH_FIND_STR(UT_var_global.vars, var, name, name);
    if (!var) UT_LOG(Fatal, "can't bind non-existent variable %s", name);
    UT_LOG(Debugk, "%s var %s binding", (var->bound_cvar ? "changed" : "set"), 
            name);
    var->bound_cvar = cvar;
    return 0;
}

/*******************************************************************************
*  UT_var_restrict()                                                           *
*  Restrict the value a given var can have based on given args.                *
*******************************************************************************/
UT_API int UT_var_restrict(char *name, UT_var_restrict_type type, ...) {
    UT_var *var;
    double *bounds_f;
    int *bounds_i;
    va_list ap;
    void *data;

    HASH_FIND_STR(UT_var_global.vars, var, name, name);
    if (!var) UT_LOG(Fatal, "can't restrict non-existent variable %s", name);

    va_start(ap,type);
    switch (type) {
        case range:
            switch (var->type) {
                case UT_var_double:
                    bounds_f = (double*)malloc(2 * sizeof(double));
                    if (!bounds_f) UT_LOG(Fatal, "malloc failure");
                    bounds_f[0] = va_arg(ap,double);
                    bounds_f[1] = va_arg(ap,double);
                    data = (void*)bounds_f;
                    break;
                case UT_var_int:
                    bounds_i = (int*)malloc(2 * sizeof(int));
                    if (!bounds_i) UT_LOG(Fatal, "malloc failure");
                    bounds_i[0] = va_arg(ap,int);
                    bounds_i[1] = va_arg(ap,int);
                    data = (void*)bounds_i;
                    break;
                default:
                    UT_LOG(Fatal, "can't restrict type of var %s", name);
                    break;
            }
            break;
        case strenum:
            if (var->type != UT_var_string) 
                UT_LOG(Fatal,"enumeration unsupported on type of var %s", name);
            data = va_arg(ap,void*);
            break;
        case readonly:
            var->flags |= VAR_READONLY;
            break;
        default:
            UT_LOG(Fatal,"Invalid restriction type %d: var %s", type, name);
            break;
    }
    va_end(ap);
    UT_var_reg_cb(name, UT_var_restrict_cb, data);
    return 0;
}

/*******************************************************************************
* UT_var_restrict_cb()                                                         *
* Used internally to enforce that the value assigned to a variable falls       *
* within a range (for numeric types) or is one of an enumeration of strings.   *
*******************************************************************************/
int UT_var_restrict_cb(char *name, void*data) {
    UT_var *var;
    double *bounds_f;
    int *bounds_i;

    HASH_FIND_STR(UT_var_global.vars, var, name, name);
    if (!var) UT_LOG(Fatal, "non-existent variable %s", name);

    if (var->flags & VAR_READONLY && var->flags & VAR_IN_SHL_CONTEXT) return -1;

    /* For now, cheat and "know" the type of restriction maps to the var type */
    switch (var->type) {
        case UT_var_double:
            bounds_f = (double*)data;
            if (var->val.f < bounds_f[0] || var->val.f > bounds_f[1]) {
                UT_LOG(Warning, "var %s forbids value %f", name, var->val.f);
                return -1;
            }
            break;
        case UT_var_int:
            bounds_i = (int*)data;
            if (var->val.i < bounds_i[0] || var->val.i > bounds_i[1]) {
                UT_LOG(Warning, "var %s forbids value %d", name, var->val.i);
                return -1;
            }
            break;
        case UT_var_string:
            if (UT_stridx(var->val.s,(char**)data) < 0) {
                UT_LOG(Warning, "var %s forbids value %s", name, var->val.s);
                return -1;
            }
            break;
        default:
            UT_LOG(Fatal,"unsupported var type");
            break;
    }
    return 0;
}
