/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/******************************************************************************
 * loop.c                                                                     *
 * Copyright (c) 2003-2005 Troy Hanson                                        *
 * Central loop of libut                                                      *
 *****************************************************************************/
static const char id[]="$Id: loop.c,v 1.11 2005/10/23 21:12:36 thanson Exp $";

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <netinet/in.h>
#include <sys/uio.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/time.h>
#include <time.h>
#include "libut/ut_internal.h"


extern int UT_tmr_update(void);

UT_loop_global_type UT_loop_global = { 
    .tmrs = NULL,
    .fds = NULL,
    .per_loop_cbs = NULL,
    .tmr_first = NULL,
    .TOD = {.tv_sec =0, .tv_usec =0},
    .sig_max = 0,
    .sig_flag = 0,
    .sig_cb = 0
};

/******************************************************************************
 * UT_figure_select_timeout()                                                 *
 * Determine how long we should sleep waiting for i/o in select. This is      *
 * either "until a timer expires" (may be zero) or until the given max wait.  *
 *****************************************************************************/
struct timeval UT_figure_select_timeout( unsigned max_wait_msec ) {
    struct timeval tv, wait_wall, micropause;

    /* Skip the calculation if caller requested zero wait */
    if (max_wait_msec == 0) {
        timerclear(&tv);
        return tv;
    }

    if (gettimeofday(&UT_loop_global.TOD, NULL) != 0) 
        UT_LOG(Fatal, "gettimeofday failure: %s", strerror(errno));

    /* Convert max_wait_msec to a walltime comparable to gettimeofday */
    wait_wall = UT_msec_to_walltime( max_wait_msec );

    /* Determine the max wait time in select() if there's no I/O: 
     *   (a) don't wait at all because a timer has already expired 
     *   (b) wait til the first timer expires- its sooner than (c)
     *   (c) wait til max_wait_msec has elapsed         */
    if ( UT_loop_global.tmr_first ) {
      if (timercmp( UT_loop_global.tmr_first, &UT_loop_global.TOD, <)) {
        timerclear(&tv);
        return tv;
      } else if (timercmp( UT_loop_global.tmr_first, &wait_wall, <)) {
        micropause.tv_sec = 0;                 /* nudge factor so we wakeup */
        micropause.tv_usec = UT_MICROPAUSE;    /* just *after* timer expire */
        timersub( UT_loop_global.tmr_first, &UT_loop_global.TOD, &tv); 
        timeradd( &tv, &micropause, &tv); 
        return tv;
      } /* else fall through */
    }

    tv.tv_sec =  max_wait_msec / 1000;         /* convert msec to sec  */
    tv.tv_usec = max_wait_msec % 1000 * 1000;  /* convert msec to usec */
    return tv;
}

/******************************************************************************
 * UT_event_once()                                                       *
 * This function is one of the most important in libut. It executes one "loop"*
 * which consists of checking for read/writable file descriptors and invoking *
 * their callbacks, and checking for expired timers and invoking those cb's.  *
 *****************************************************************************/
UT_API void UT_event_once( unsigned max_wait_msec ) {
    int selrc, i, j, maxfd, fds_left, r, w;
    struct timeval tv;
    fd_set rset, wset;
    UT_fd *fdx, *fdx_copy, *fdc;

    if (UT_loop_global.per_loop_cbs != NULL) UT_invoke_per_loop_cbs();


    /* Construct the list of fd's awaiting readability, writability */
    FD_ZERO( &rset);
    FD_ZERO( &wset);
    for (fdx = UT_loop_global.fds,maxfd=0; fdx; fdx = fdx->next) {
        if (fdx->flags & UTFD_R) { 
            FD_SET( fdx->fd, &rset ); 
        }
        if (fdx->flags & (UTFD_W | UTFD_WRITE_PENDING)) { 
            FD_SET( fdx->fd, &wset ); 
        }
        if (fdx->fd > maxfd) maxfd = fdx->fd;
    }

    tv = UT_figure_select_timeout( max_wait_msec );

    /* Here's the heart of libut. Check for read/writable descriptors. */
    selrc = select( maxfd + 1, &rset, &wset, NULL, &tv);

    /* Update the time of day (TOD) since select may have slept awhile */
    if (gettimeofday(&UT_loop_global.TOD, NULL) != 0) 
        UT_LOG(Fatal, "gettimeofday failure: %s", strerror(errno));

    /* If any fd's were read/writable, make list of cb's to invoke */
    if (selrc > 0) {
        fdx_copy = (UT_fd*)UT_mem_alloc(FD_POOL, selrc);
        for (i=0, j=0, fds_left=selrc; i<=maxfd && fds_left; i++) {
            if ( (r = FD_ISSET( i, &rset )) || (w = FD_ISSET( i, &wset)) ) {
                HASH_FIND_INT(UT_loop_global.fds, fdx, fd, &i);
                if ( fdx ) {
                    fdx->cb_count++;
                    fdx_copy[j] = *fdx;
                    fdc = &fdx_copy[j];
                    if (r) fdc->flags |= UTFD_IS_READABLE; 
                    if (w) fdc->flags |= UTFD_IS_WRITABLE;
                    j++;
                } else UT_LOG(Fatal, "no UT_fd for fd %d", i);
                fds_left--;
            }
        }

        /* Invoke each of the fd's that are ready for i/o. */
        UT_prf_bgn( "libut", "i/o callbacks");  
        for (i = 0; i < j; i++ ) {
            fdx = &fdx_copy[i];

            /* This block implements delayed writes to fd's awaiting writability */
            if (fdx->flags & UTFD_WRITE_PENDING && fdx->flags & UTFD_IS_WRITABLE) {
                UT_fd_write_pending(fdx);
                fdx->flags &= ~UTFD_IS_WRITABLE;  /* may no longer be writable */
                /* done if cb closed fd, or fd was only writable (not readable) */
                if ((fdx->flags & UTFD_CLOSED_IN_CB) ||
                   (!(fdx->flags & UTFD_R && fdx->flags & UTFD_IS_READABLE)))
                    continue;
            }

            UT_LOG(Debugk, "Invoking io callback for fd %d", fdx->fd);
            (fdx->cb)(fdx->fd, fdx->name, fdx->flags, fdx->data);
        }
        UT_prf_end( "libut", "i/o callbacks");
        UT_mem_free(FD_POOL, fdx_copy, selrc);
    }

    /* invoke expired timers */
    UT_tmr_update();    

    /* handle any signals received */
    UT_signal_update();
}

/*******************************************************************************
* UT_invoke_per_loop_cbs()                                                     *
* Executed from UT_event_once() if there are per-loop callbacks to run.   *
*******************************************************************************/
int UT_invoke_per_loop_cbs() {
    UT_per_loop_cb_t *pscb,*pscb_tmp;
    int i,j;

    /* copy the per-loop cb's to a temporary list so we can traverse safely */
    for(pscb = UT_loop_global.per_loop_cbs,i=0; pscb; pscb=pscb->next) i++; 
    pscb_tmp = (UT_per_loop_cb_t*)UT_mem_alloc(PER_LOOP_CB_POOL, i);
    for(pscb = UT_loop_global.per_loop_cbs,i=0; pscb; pscb=pscb->next) {
        pscb_tmp[i] = *pscb;
        pscb_tmp[i].next = pscb; /* misuse 'next'; point to original struct */
        i++;
    }

    /* Invoke each callback, checking its return value: <= 0 tells us to remove
     * the per-loop cb. We've kept a pointer to its original struct in 'next'.*/
    for(j=0; j < i; j++) {
        if ( pscb_tmp[j].cb() <= 0 ) {
            LL_DEL(UT_loop_global.per_loop_cbs, pscb, pscb_tmp[j].next);
            UT_mem_free(PER_LOOP_CB_POOL, pscb_tmp[j].next, 1);
        }
    }

    /* Free the copies we've made.*/
    UT_mem_free(PER_LOOP_CB_POOL, pscb_tmp, i);
}

int UT_loop_init() {
    UT_mem_pool_create(PER_LOOP_CB_POOL, sizeof(UT_per_loop_cb_t), 2);
}

/******************************************************************************
 * UT_event_loop()                                                            *
 * This function never returns. It eternally loops the UT event loop.         *
 * Applications built using UT must either call this at some point after UT   *
 * init, or they must call UT_event_once periodically in their own loop. *
 *****************************************************************************/
UT_API void UT_event_loop() {
    UT_LOG(Info, "Entered UT_event_loop().");
    for (;;) UT_event_once( UT_LOOP_SELECT_MSEC );
    /* not reached */
}

