/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/******************************************************************************
 * mem.h                                                                      *
 *****************************************************************************/

/* An application may have many memory pools, one for each heavily- *
 * used data structure. We keep a master list of the pools. Each    *
 * pool consists of rows of buffers.                                */

/* The API for memory subs consists of:
 * UT_mem_pool_create - define a memory pool of bufs you can allocate  *
 * UT_mem_free_pool - delete a memory pool and free all its bufs    *
 * UT_mem_alloc - get a buf from a memory pool                      *
 * UT_mem_free - return a buf to a memory pool                      *
 *                                                                  *
 * If an allocation fails, the program exits.                       */


#define UT_MEM_ROW_NUM_BKTS 16
#define POOL_NAME_MAX_LEN 100

#define MALLOC_FAILED 1
#define BAD_POOL_NAME 2
#define BUF_SCAN_FAILED 3

#define BUF_USED(u,i) (u[(i>>3)] & (1 << (i & 7)))
#define BUF_SET(u,i) (u[(i>>3)] |= (1 << (i & 7)))
#define BUF_CLR(u,i) (u[(i>>3)] &= ~(1 << (i & 7)))

/* Find the mem row bucket corresponding to the given num_bufs      *
 * Please use 'unsigned' num_bufs to avoid bit-shifting signed's.   */
#define GET_BKT(bkt,num_bufs) \
    bkt=0; \
    while(num_bufs >> bkt != 0) bkt++; \
    if (bkt >= UT_MEM_ROW_NUM_BKTS) bkt = UT_MEM_ROW_NUM_BKTS-1;

typedef struct UT_mem_pool_row {
        char *usage_bitmap;        /* which bufs are free (1 bit per buf) */
        unsigned num_bufs;
        unsigned max_extent;       /* largest contig span of unused buffers*/
        unsigned max_extent_idx;   /* index of first element of max extent */
        void *data;
        struct UT_mem_pool_row *next;
} UT_mem_pool_row;

typedef struct UT_mem_pool {
        char name[POOL_NAME_MAX_LEN];
        unsigned buf_size;
        unsigned bufs_per_row;  /* default; can be overriden in individual rows */
        unsigned freed_buf_stats;
        unsigned allocd_buf_stats;
        UT_mem_pool_row *rows[UT_MEM_ROW_NUM_BKTS];
        struct UT_hash_handle hh;
        struct UT_mem_pool *next;
} UT_mem_pool;


typedef struct UT_mem_global_type {
        UT_mem_pool *UT_mem_pools_all;
} UT_mem_global_type;

void UT_mem_init(void);
void UT_malloc_failure( int error);
void **UT_mem_get_buf_map( char *poolname);
int UT_hash_rescale(UT_hash_table *);
void UT_mem_pool_free(char *poolname);
