/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/******************************************************************************
 * connect.c                                                                  *
 * Copyright (c) 2003-2005  Troy Hanson                                       *
 *****************************************************************************/
static const char id[]="$Id: connect.c,v 1.14 2005/11/07 01:45:58 thanson Exp $";

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <netinet/in.h>
#include <sys/uio.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdlib.h>
#include "libut/ut_internal.h"



/******************************************************************************
 * UT_nb_connect_cb()                                                         *
 * Used internally, when a non-blocking connect() completes. It either failed *
 * or succeeded. The logic for determining success or failure is from Stevens *
 * UNPv1 p409-411. On successful connect install the application I/O callback.*
 *****************************************************************************/
int UT_nb_connect_cb(int fd, char *name, int flags, UT_callback *cbdata) {
    int error, len, rc;
    UT_fd_aux_type aux;

    UT_fd_cntl(fd, UTFD_GET_AUX, &aux);

    /* Solaris indicates connect() failure with rc==-1 and sets errno.     *
     * Other systems indiciate connect() failure with rc==0 and set error. */
    len = sizeof( error );
    rc = getsockopt(fd, SOL_SOCKET, SO_ERROR, &error, &len);
    if (rc == 0 && error) {
        errno = error;      /* Convert non-Solaris failure indication  */
        rc = -1;            /* to the same rc/errno that Solaris sets. */
    }

    if (rc < 0) {
        /* Connect failed. Invoke callback to notify application. */
        UT_LOG(Error,"Can't connect from %s to %s: %s", aux.socket.local_ipport,
                aux.socket.remote_ipport, strerror(errno));
        (cbdata->cb.fdcb)(fd, name, flags|UTFD_IS_CONFAILURE, cbdata->data );
        UT_fd_unreg(fd); 
        close(fd);   
    } else {
        /* Connect succeeded. Install app cb for I/O. Invoke cb to notify app.*/
        UT_LOG(Debug,"Connected from %s to %s", aux.socket.local_ipport,
                aux.socket.remote_ipport);
        /* changing flags, cb, data and aux could be done with UT_fd_cntl */
        flags &= ~UTFD_SOCKET_CONNECTING;
        flags &= ~UTFD_W;
        flags |= UTFD_SOCKET_CONNECTED;
        UT_fd_reg(fd, name, cbdata->cb.fdcb, cbdata->data, flags, 
                aux.socket.local_ipport, aux.socket.remote_ipport);
        (cbdata->cb.fdcb)(fd, name, flags|UTFD_IS_CONSUCCESS, cbdata->data );
    }

    UT_mem_free( CBDATA, cbdata, 1);
}

/*******************************************************************************
* UT_net_connect2()                                                            *
*                                                                              *
*  Non-blocking connects                                                       *
*  Things to keep in mind. (See Stevens UNPv1 p409.):                          *
*  A non-blocking connect may return (with an established                      *
*  connection) immediately, such as when connecting to a local process;        *
*  or it can return an "error" (EINPROGRESS) which is normal and means         *
*  connect is in progress.  In the latter case, when it finishes (fd           *
*  becomes r/w) we still need to check its completion status to see if         *
*  it succeeded. That is done using getsockopt() in UT_nb_conect_cb.           *
*                                                                              *
*                                     TODO:                                    *
*  DNS Rotors - gethostbyname() will alternate the order in which IP addrs     *
*      are listed each time it is called, for hostnames having multi IP's.     *
*      Therefore if connect() fails for one IP address of a multi-IP host,     *
*      would should ideally re-invoke UT_net_connect() for each remaining IP.  *
*  DNS Blocking - gethostbyname() blocks until the DNS response is received.   *
*      Aspire to use a non-blocking DNS query but that                         *
*      would require some restructuring (perhaps a wrapper function).          *
*******************************************************************************/
int UT_net_connect2(char *name, UT_fd_cb *cb, void *data, int flags, va_list ap)
{
    struct sockaddr_in servaddr,locladdr;
    struct hostent *host_ent;
    struct in_addr *first_ip;
    in_addr_t local_ip,remote_ip;
    int fd, fdflags, n, local_port, remote_port, flags2;
    char *local_ipport,*remote_ipport;
    char s[INET_ADDRSTRLEN], r[UT_IPPORT_MAXLEN], *l;  /* scratch formatting */
    UT_callback *cbdata;

    /* Sanity check the flags */
    if ( !(flags & UT_CONNECT_TO_IPPORT)) {
        UT_LOG(Error, "No host/ip/port passed to UT_net_connect (%s)", name);
        return -1;
    }

    local_ipport = (flags & UT_CONNECT_FROM_IPPORT) ? (va_arg(ap,char*)) : NULL;
    remote_ipport = (flags & UT_CONNECT_TO_IPPORT) ? (va_arg(ap,char*)) : NULL;

    if (UT_ipport_parse(remote_ipport, &remote_ip, &remote_port) < 0)  {
        UT_LOG(Error,"invalid ip:port specification %s", remote_ipport);
        (cb)(fd, name, UTFD_IS_CONFAILURE, data);
        return -1;
    }

    /* Now remote_ip (in_addr_t) and remote_port (int) are ready */

    /* Create the socket. Let the application know if it fails. */
    if ( (fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        UT_LOG(Error, "socket failed: %s", strerror(errno));
        (cb)(fd, name, UTFD_IS_CONFAILURE, data);
        return -1;
    }

    /* bind is optional. bind a local IP:port if given, else its ephemeral. */
    if (local_ipport) {
        memset( &locladdr, 0, sizeof(servaddr) );
        locladdr.sin_family = AF_INET;
        if (UT_ipport_parse(local_ipport,&local_ip,&local_port) < 0) {
            UT_LOG(Error, "connect: parse error in IP/port %s", local_ipport);
            (cb)(fd, name, UTFD_IS_CONFAILURE, data);
            close(fd);
            return -1;
        }
        locladdr.sin_addr.s_addr = local_ip;
        locladdr.sin_port =  htons( local_port  );
        if (bind( fd, (struct sockaddr *)&locladdr, sizeof(servaddr) ) == -1) {
            UT_LOG(Error, "bind socket to %s failed: %s", local_ipport,
                    strerror(errno)); (cb)(fd, name, UTFD_IS_CONFAILURE, data);
            close(fd);
            return -1;
        }
    }

    /* Create the destination address structure */
    memset(&servaddr, 0, sizeof(servaddr));
    servaddr.sin_family = AF_INET;
    servaddr.sin_port = htons(remote_port);
    servaddr.sin_addr.s_addr = remote_ip; 

    /* Make the socket non-blocking for the connect and subsequent I/O. */
    fdflags = fcntl( fd, F_GETFL, 0 );
    fcntl( fd, F_SETFL, fdflags | O_NONBLOCK );
     
    /* Set up (l)ocal and (r)emote IP:port presentation variables */
    inet_ntop(AF_INET, &remote_ip, s, INET_ADDRSTRLEN);
    snprintf(r, UT_IPPORT_MAXLEN, "%s:%d", s, remote_port);
    l = local_ipport ? local_ipport : "*:0";

    /* Initiate 3-way handshake. If connect is immediate, notify application. */
    if ( (n = connect(fd,(struct sockaddr*)&servaddr, sizeof(servaddr))) == 0) {
        flags2 = UTFD_R | UTFD_SOCKET | UTFD_SOCKET_CONNECTED |
            UTFD_SOCKET_LOCALADDR | UTFD_SOCKET_REMOTEADDR;
        UT_fd_reg(fd, name, cb, data, flags2, l, r);
        (cb)(fd, name, UTFD_IS_CONSUCCESS, data);
        return 1;
    }

    /* If the connect retured an error other than EINPROGRESS, it failed. */
    if ( n < 0 && errno != EINPROGRESS ) {
        UT_LOG(Error,"Can't connect from %s to %s: %s",l,r,strerror(errno));
        (cb)(fd, name, UTFD_IS_CONFAILURE, data);
        close( fd );   
        return -1;
    }

    /* The connect is in progress in the background. Record the application
     * callback and data pointer in a structure, to be passed back to us when
     * the connection completes, and arrange for a callback when that occurs. */
    UT_LOG( Debugk, "Non-blocking connect in progress");
    cbdata = (UT_callback*)UT_mem_alloc( CBDATA, 1 );
    cbdata->cb.fdcb = cb;
    cbdata->data = data;
    flags2 = UTFD_R |UTFD_W | UTFD_SOCKET | UTFD_SOCKET_CONNECTING |
        UTFD_SOCKET_LOCALADDR | UTFD_SOCKET_REMOTEADDR;
    UT_fd_reg( fd, name, (UT_fd_cb*)UT_nb_connect_cb, cbdata, flags2, l, r);
    return 0;
}

/*******************************************************************************
* UT_net_resolve()                                                             *
* Perform a DNS query to resolve a hostname to an IP address.                  *
* The returned string is NULL on error, or a pointer to a static string.       *
* Permits a ":<port>" suffix which is unused in the DNS lookup but appended    *
* back to the output string.                                                   *
*******************************************************************************/
UT_API char *UT_net_resolve(char *hostname) {
    char *c, *hostcopy;
    struct hostent *host_ent;
    in_addr_t ip;
    int hostlen;
    char s[INET_ADDRSTRLEN]; 
    static char r[UT_IPPORT_MAXLEN];

    c = hostname;
    while (*c != '\0' && *c != ':') c++;
    hostlen = c - hostname;
    if (hostlen == 0) return NULL;
    if ( (hostcopy = malloc( hostlen+1 )) == NULL) UT_LOG(Fatal,"malloc error");
    memcpy( hostcopy, hostname, hostlen );
    hostcopy[hostlen] = '\0';

    if ( (host_ent = gethostbyname( hostcopy )) == NULL ) {
        UT_LOG(Error,"DNS lookup error [%s]: %s", hostcopy, hstrerror(h_errno));
        free(hostcopy);
        return NULL;
    }
    free(hostcopy);

    /* Use the host's first IP (index 0); incr. idx til NULL for all IPs */
    ip = ((struct in_addr*)(host_ent->h_addr_list[0]))->s_addr;

    /* convert IP address numeric form to presentation form */
    if (inet_ntop(AF_INET, &ip, s, INET_ADDRSTRLEN) == NULL) {
        UT_LOG(Error, "inet_ntop error: %s", strerror(errno));
        return NULL;
    }

    if (snprintf(r, UT_IPPORT_MAXLEN, "%s%s", s, c) >= UT_IPPORT_MAXLEN) {
        UT_LOG(Error,"UT_net_resolve: result exceeds buffer [%s%s]",s,c);
        return NULL;
    }

    return r;
}

/*******************************************************************************
* UT_net_connect()                                                             *
* Open a TCP/IP connection to the given host/port and set up I/O callback.     *
* The source IP/port may be specified, by setting UT_CONNECT_FROM_IPPORT       *
* in the "flags" argument and passing a string like "44.2.4.1:3333" as the     *
* first vararg after flags. (Pass 0 for the port to let kernel choose it).     *
*                                                                              *
* The destination host (or IP) and port must be specified. You must set        *
* UT_CONNECT_TO_IPPORT in the flags argument. The next vararg is a             *
* IP/port string like the one shown above.                                     *
*******************************************************************************/
UT_API int UT_net_connect(char *name, UT_fd_cb *cb, void *data, int flags, ...)
{
    va_list ap;
    int rc;

    va_start(ap, flags);
    rc = UT_net_connect2( name, cb, data, flags, ap);
    va_end(ap);

    return rc;
}

