##   tageditor.py -- additional functions for libtagedit python plugin
##   Copyright (C) 2002-2004 Torsten Marek <shlomme@gmx.net>
##  
##   This program is free software; you can redistribute it and/or modify
##   it under the terms of the GNU General Public License as published by
##   the Free Software Foundation; either version 2 of the License, or
##   (at your option) any later version.
##  
##   This program is distributed in the hope that it will be useful,
##   but WITHOUT ANY WARRANTY; without even the implied warranty of
##   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##   GNU General Public License for more details.
##   You should have received a copy of the GNU General Public License
##   along with this program; if not, write to the Free Software
##   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
import os.path
import re
import sys
from _tageditor import Artist, Title, Album, Genre, Year, Track, Comment, \
     _libsupportedExtensions, TagEditor, createTagEditor, LIBTAGEDIT_VERSION
def tagEditorExts():
    """tagEditorExts() -> seq

    Return the extensions of all music file formats 
    supported by libtagedit."""
    s = _libsupportedExtensions()[:-1]
    return map(str, s.replace("*", "").split(";"))


def createM3U(files, playlist):
    """createM3U(files, filename)

    Create an m3u playlist of all songs in files 
    and write it to filename."""
    if not destfile.endswith(".m3u"):
        destfile += ".m3u"
    m3u = file(destfile, "w")
    files.sort()
    m3u.write("#EXTM3U\n")
    for l in files:
        t = createTagEditor(l)
        if t and t.isOK():
            m3u.write("#EXTINF:%i,%s - %s\n" % (t.length(),
                                                t["Artist"],
                                                t["Title"]))
            m3u.write(t.filename()+  "\n")


def buildFilename(tag, format):
    """buildFilename(tag, format) -> string

    Create a new name for the song in tag according to <format>
    and return it.
    
    The format description is in FilenameBuilder"""
    t = FilenameBuilder(format)
    return t.fromTag(tag)


class FilenameBuilder(object):
    """FilenameBuilder(string) -> new FilenameBuilder
    Create new filename by tag information in music files.
    
    Format string examples:
    "%Track - %Artist - %Title"
    "%Artist - %Album - %Year/%Track - %Title"
    
    For all available field names, look at TagEditor.fieldnames()
    Please note that you have to create new directories manually
    and there is no check on whether a processed format string is
    a valid file name or not."""
    _fieldmatcher = re.compile("\%([A-Z][a-z]+)")
    _fields = TagEditor.fieldnames()
    def __init__(self, format):
        self.parsed_format = self._parseFormat(format)
        
    def _parseFormat(self, format):
        def replacer(z):
            f = z.group(0)[1:]
            if f in self._fields:
                return "%%(%s)s" % f
            else:
                raise ValueError, "Wrong specifier %s" % f
        return unicode(self._fieldmatcher.sub(replacer, format) + ".%(Ext)s")

    def fromTag(self, tag):
        """fromTag(TagEditor) -> string
        
        Create a new name for the song in tag and return it."""
        tdict = tag.asDict()
        if tdict["Track"] != "":
            tdict["Track"] = "%02i" % int(re.sub("\/[0-9]*$", "",
                                                 tdict["Track"]))
        tdict["Ext"] = tag.extension()
        dirname = unicode(os.path.dirname(tag.filename()), sys.getfilesystemencoding())
        return os.path.join(dirname,
                            self.parsed_format % tdict)

    
def formatNameScanner(expression, keywords=(), withext=True):
    """formatNameScanner(format, [keywords=(), [withext=True]]) -> regular expression
    
    Create a regular expression from a format string to
    scan music file names.

    Format is the same as in FilenameBuilder.
    You can include additional field names as a sequence by the keywords
    argument.

    Example: "%Album/%Artist - %Track - %Title"
    """
    #example  creates
    #"^.+/(?P<Album>.+)/(?P<Artist>.+) - (?P<Track>\d+) - (?P<Title>.+)\..+$"
    fields = TagEditor.fieldnames() + keywords
    def replacer(field):
        f = field.group(0)[1:]
        if f in fields:
            if f != "Track":
                return "(?P<%s>.+)" % f
            else:
                return "(?P<%s>\d+)" % f
        else:
            raise ValueError, "%s is not a valid specifier" % f
    expression = FilenameBuilder._fieldmatcher.sub(replacer, expression)
    if expression[0] == "/" or expression.startswith("(?P"):
        expression = os.path.join("^.*", expression)
    else:
        expression = "^.*/.*" + expression
    if expression[-1] != "$":
        if withext:
            expression += ".*\..+$"
        else:
            expression += ".*$"
    return expression
