/*********************************************************************
 * 
 *   Copyright (C) 2001-2004 Torsten Marek
 *	 Parts of the code (especially for writing the ogg tag) are taken
 *   from the vorbiscomment program from Michael Smith <msmith@labyrinth.net.au>
 * 
 * Filename:      oggtag.cpp
 * Description:   implementation of the oggTag class
 * Author:        Torsten Marek <shlomme@gmx.net>
 *                
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 ********************************************************************/
#include "oggtag.hpp"
#include <cctype>
#include <iostream>
#include <vorbis/vorbisfile.h>

using namespace std;

oggTag::oggTag(const char* n) : vorbisComment(n)
{
    prevW = 0;
    readOggTag();
}

oggTag::~oggTag()
{
}

void oggTag::clearInternals()
{
    if(vc) {
        vorbis_comment_clear(vc);
        delete(vc);
        vc=NULL;
    }
    if(os) {
        ogg_stream_clear(os);
        delete(os);
        os=NULL;
    }
    if(oy) {
        ogg_sync_clear(oy);
        delete(oy);
        oy=NULL;
    }
}

int oggTag::readOggTag()
{
    in = fopen(fname.c_str(), "rb");
    if(in == 0) {
        tag_insane = 1;
        return -1;
    }
    OggVorbis_File vf;
    if(ov_open(in, &vf, NULL, 0) == 0) {
		vc = ov_comment(&vf, -1);
        file_bitrate = ov_bitrate(&vf, 0) / 1000;
        file_length = (unsigned int)ov_time_total(&vf, -1);
	} else {
        return -1;
    }
	for(int i = 0; i < vc->comments; ++i) {
		string c = vc->user_comments[i];
		int e = c.find('=');
		string tag = c.substr(0, e);
		string::iterator it = tag.begin();
		for(; it != tag.end(); ++it)
			*it = toupper(*it);
		c = c.substr(e+1);
        // tag information is always encoded as UTF-8 (says xiph.org)
        comments[tag] = c;
	}
	ov_clear(&vf);
	return 0;
}


int oggTag::_blocksize(ogg_packet *p)
{
	int that = vorbis_packet_blocksize(&vi, p);	
	int ret = (that + prevW)/4;	
	if(!prevW) {
		prevW = that;
		return 0;
	}
	prevW = that;	
	return ret;	
}

int oggTag::_fetch_next_packet(ogg_packet *p)
{	
	int result;	
	ogg_page og;	
	char *buffer;
	int bytes;

	result = ogg_stream_packetout(os, p);
	if(result > 0)
		return 1;
	else {
		while(ogg_sync_pageout(oy, &og) <= 0) {
			buffer = ogg_sync_buffer(oy, CHUNKSIZE);
			bytes = fread(buffer, 1, CHUNKSIZE, in);
			ogg_sync_wrote(oy, bytes);
			if(bytes == 0) 
				return 0;
		}
		ogg_stream_pagein(os, &og);
		return _fetch_next_packet(p);
	}
}

// writes ogg tag to file
// return values
// 0: everything ok
// -1: file could not be opened for reading/writing
// -2: no tempfile could be created
// -3: old file could not be removed
int oggTag::writeTag() 
{
	try {
		in = fopen(fname.c_str(), "rb");
		if(in == 0)
			throw -1;
		out = fopen((fname + ".tmp").c_str(), "wb");
		if(out == 0)
			throw -2;
		writeOggTag();
		fclose(in);
		fclose(out);
		if(rename((fname + ".tmp").c_str(), fname.c_str()) == -1)
			throw -3;
		tag_altered = 0;
		return 0;
	}
	catch(int f){
        if(in != 0)
            fclose(in);
        if(out != 0)
            fclose(out);
        if(f == -3) {
            vorbis_info_clear(&vi);
            ogg_stream_clear(&streamout);
            clearInternals();
        }
        return f;
	}
}

int oggTag::writeOggTag()
{
	char *buffer;
	ogg_packet	header_main;
	ogg_packet  header_comments;
	ogg_packet	header_codebooks;
	ogg_page    og, ogout;
	ogg_packet op;
	ogg_int64_t granpos = 0;
	int result,	eosin = 0, needflush = 0, needout = 0, bytes, i = 0;	
	long serial;

	vc = new vorbis_comment;
	vorbis_comment_init(vc);
    map<string, string>::iterator it = comments.begin();
	for(; it != comments.end(); ++it) {
        string c = it->second;
		vorbis_comment_add_tag(vc, (char*)it->first.c_str(), (char*)c.c_str());
	}
	vorbis_commentheader_out(vc, &header_comments);
	oy = new ogg_sync_state;
	ogg_sync_init(oy);
	buffer = ogg_sync_buffer(oy, CHUNKSIZE);
	bytes = fread((void*)buffer, 1, CHUNKSIZE, in);
	ogg_sync_wrote(oy, bytes);
	if(ogg_sync_pageout(oy, &og) != 1) {
		if(bytes<CHUNKSIZE)
			throw -4;//"Input truncated or empty.";
		else
			throw -4;//"Input is not an Ogg bitstream.";
	}
	serial = ogg_page_serialno(&og);
	ogg_stream_init(&streamout, serial);
	os = new ogg_stream_state;
	ogg_stream_init(os, serial);
	vorbis_info_init(&vi);
	if(ogg_stream_pagein(os, &og) < 0)
		throw -4;//"Error reading first page of Ogg bitstream.";
	if(ogg_stream_packetout(os, &header_main) != 1)
		throw -4;//"Error reading initial header packet.";
	if(vorbis_synthesis_headerin(&vi, vc, &header_main) < 0)
		throw -4;//"Ogg bitstream does not contain vorbis data.";
	ogg_stream_packetin(&streamout, &header_main);

	while(i < 2) {
		while(i < 2) {
			int result = ogg_sync_pageout(oy, &og);
			if(result == 0)
				break; // Too little data so far 
			else if(result == 1) {
				ogg_stream_pagein(os, &og);
				while(i < 2) {
					result = ogg_stream_packetout(os, &header_codebooks);
					if(result == 0)
						break;
					if(result == -1) 
						throw -4;//"Corrupt secondary header.";
					vorbis_synthesis_headerin(&vi, vc, &header_codebooks);
					if(i == 1) {
						ogg_stream_packetin(&streamout, &header_comments);
						ogg_stream_packetin(&streamout, &header_codebooks);
					}
					i++;
				}
			}
		}
		buffer = ogg_sync_buffer(oy, CHUNKSIZE);
		bytes = fread(buffer, 1, CHUNKSIZE, in);
		if(bytes == 0 && i < 2)
			throw -4;//"EOF before end of vorbis headers.";
		ogg_sync_wrote(oy, bytes);
	}
	while((result = ogg_stream_flush(&streamout, &ogout))) {
		if(fwrite(ogout.header, 1, ogout.header_len, out) != (size_t) ogout.header_len)
			return -1;
		if(fwrite(ogout.body, 1 ,ogout.body_len, out) != (size_t) ogout.body_len)
			return -1;
	}
	while(_fetch_next_packet(&op)) {
		int size;
		size = _blocksize(&op);
		granpos += size;
		if(needflush) {
			if(ogg_stream_flush(&streamout, &ogout)) {
				if(fwrite(ogout.header,1,ogout.header_len, out) != (size_t) ogout.header_len)
					return -1;
				if(fwrite(ogout.body,1,ogout.body_len, out) != (size_t) ogout.body_len)
					return -1;
			}
		} else if(needout) {
			if(ogg_stream_pageout(&streamout, &ogout)) {
				if(fwrite(ogout.header,1,ogout.header_len, out) != (size_t) ogout.header_len)
					return -1;
				if(fwrite(ogout.body,1,ogout.body_len, out) != (size_t) ogout.body_len)
					return -1;
			}
		}
		needflush = needout = 0;
		if(op.granulepos == -1) {
			op.granulepos = granpos;
			ogg_stream_packetin(&streamout, &op);
		} else {
			if(granpos > op.granulepos) {
				granpos = op.granulepos;
				ogg_stream_packetin(&streamout, &op);
				needflush=1;
			} else {
				ogg_stream_packetin(&streamout, &op);
				needout=1;
			}
		}		
	}
	streamout.e_o_s = 1;
	while(ogg_stream_flush(&streamout, &ogout)) {
		if(fwrite(ogout.header,1,ogout.header_len, out) != (size_t) ogout.header_len)
			return -1;
		if(fwrite(ogout.body,1,ogout.body_len, out) != (size_t) ogout.body_len)
			return -1;
	}
	vorbis_info_clear(&vi);
	ogg_stream_clear(&streamout);
	ogg_packet_clear(&header_comments);
	clearInternals();
	return 0;
}
