/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif
#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#if HAVE_TIME_H
# include <time.h>
#endif

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) // x
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#include "pftputil.h" 
#include "str.h"
#include "pftp_getconf.h"
#include "setconf.h"
#include "gethosts.h"
#include "sethosts.h"
#include "util.h"

#include "hosts_settings.h"

typedef struct ftpserver_s *ftpserver_t;

struct ftpserver_s {
    pftp_settings_t settings;
    pftp_server_t ctrl;
    int disconnected;
    uint16_t free_port;
    char *save_path;
};

static ftpserver_t ftp_server = NULL;
static size_t ftp_servers = 0;
static pftp_conf_t global = NULL;
static uint16_t global_free_port = 0;
static pftp_comm_centralFunc_t cli_comm_central = NULL;

int util_comm_central(pftp_msg_t msg, pftp_server_t ftp, void *userdata, 
		      pftp_param_t param1, pftp_param_t param2);

void pftp_initUtil(pftp_comm_centralFunc_t comm_central)
{
    int tmp_hosts = 0;
    pftp_settings_t *tmp_host = NULL;

    cli_comm_central = comm_central;

    pftp_initlib();

    global = pftp_get_conf();

    if (!global->ftp->port_start)
	global->ftp->port_start = 32000;
    
    global_free_port = global->ftp->port_start;

    tmp_host = get_hosts(&tmp_hosts);

    if (tmp_hosts) {
	size_t i;
	ftp_servers = tmp_hosts;
	ftp_server = malloc(ftp_servers * sizeof(struct ftpserver_s));
	memset(ftp_server, 0, sizeof(struct ftpserver_s) * ftp_servers);
	
	for (i = 0; i < ftp_servers; i++) {
	    ftp_server[i].settings = tmp_host[i];
	    ftp_server[i].settings->global = global->ftp;
	    ftp_server[i].free_port = ftp_server[i].settings->port_start;
	}

	free(tmp_host);
    }
}

void pftp_freeUtil(void)
{
    size_t f;

    for (f = 0; f < ftp_servers; f++) {
	pftp_logout(&ftp_server[f].ctrl);
	freeFTPSettings(&ftp_server[f].settings);
	if (ftp_server[f].save_path) 
	    free(ftp_server[f].save_path);
    }

    if (ftp_server) {
	free(ftp_server);
	ftp_server = NULL;
    }

    ftp_servers = 0;

    pftp_free_conf(global);
    pftp_close_conf();

    pftp_donelib();
}

pftp_settings_t pftp_getFTPSettings(const char *ftpname)
{
    size_t f;

    if (!ftpname) 
	return NULL;

    for (f = 0; f < ftp_servers; f++) 
	if (strcmp(ftp_server[f].settings->name, ftpname) == 0) {
	    return ftp_server[f].settings;	    
	}

    return NULL;
}

static pftp_settings_t getFTPSettings2(pftp_server_t ftp)
{
    size_t f;

    for (f = 0; f < ftp_servers; f++) {
	if (ftp_server[f].ctrl == ftp)
	    return ftp_server[f].settings;
    }

    return NULL;
}

pftp_server_t pftp_getFTPCtrl(const char *ftpname)
{
    size_t f;

    if (!ftpname) 
	return NULL;

    for (f = 0; f < ftp_servers; f++) 
	if (strcmp(ftp_server[f].settings->name, ftpname) == 0) {
	    if (ftp_server[f].disconnected && ftp_server[f].ctrl) {
		pftp_logout(&ftp_server[f].ctrl);
		ftp_server[f].disconnected = 0;
	    }
	    
	    if (!ftp_server[f].ctrl) {
		ftp_server[f].ctrl = pftp_login(ftp_server[f].settings,
						util_comm_central);

		if (ftp_server[f].ctrl && ftp_server[f].save_path) {
		    pftp_cd(ftp_server[f].ctrl, ftp_server[f].save_path, NULL);
		    free(ftp_server[f].save_path);
		    ftp_server[f].save_path = NULL;
		}
	    }
	    
	    return ftp_server[f].ctrl;	    
	}
    
    return NULL;
}

void pftp_closeFTPCtrl(const char *ftpname)
{
    size_t f;

    if (!ftpname) 
	return;

    for (f = 0; f < ftp_servers; f++) 
	if (strcmp(ftp_server[f].settings->name, ftpname) == 0) {
	    pftp_logout(&ftp_server[f].ctrl);
	    return;
	}   
}

int pftp_getFTPConnected(const char *ftpname)
{
    size_t f;

    if (!ftpname) 
	return 0;

    for (f = 0; f < ftp_servers; f++) 
	if (strcmp(ftp_server[f].settings->name, ftpname) == 0)
	    return (ftp_server[f].ctrl != NULL);	

    return 0;
}

pftp_default_t pftp_getDefaultSettings(void)
{
    return global->ftp;
}

pftp_conf_t pftp_getConfig(void)
{
    return global;
}

static int makelocalPath(const char *path, mode_t mode)
{
    size_t len = strlen(path);
    int ret = 0;
    char *cpy = malloc(len + 1);
    memcpy(cpy, path, len + 1);
    if (len && cpy[len - 1] == '/')
	cpy[--len] = '\0';

    if (len == 0) {
	/* "/" will always be there for you */
	free(cpy);
	return 0;
    }
    
    if (mkdir(cpy, mode)) {
	if (errno == EEXIST) {
	    ret = 0;
	} else if (errno == ENOENT) {
	    char *s = strrchr(cpy, '/');
	    if (s) {
		s[0] = '\0';
		if (makelocalPath(cpy, mode)) {
		    ret = -1;
		} else {
		    ret = makelocalPath(path, mode);
		}
	    } else {
		ret = -1;
	    }
	} else {
	    ret = -1;
	}
    } else {
	ret = 0;
    }

    free(cpy);
    return ret;
}

static int makeremotePath(pftp_server_t ftp, const char *path)
{
    pftp_settings_t ftp_settings;
    size_t len = strlen(path);
    int ret = 0;
    char *cpy = malloc(len + 1);
    memcpy(cpy, path, len + 1);
    if (len && cpy[len - 1] == '/')
	cpy[--len] = '\0';

    if (len == 0) {
	/* "/" will always be there for you */
	free(cpy);
	return 0;
    }
    if (!ftp) {
	free(cpy);
	return -1;
    }

    ftp_settings = getFTPSettings2(ftp);

    cli_comm_central(PFTP_NEXTSTATUS_IS_NOT_FATAL, ftp, 
		     ftp_settings ? ftp_settings->userdata : NULL,
		     NULL, NULL);

    if (pftp_cd(ftp, cpy, NULL)) {	
	if (pftp_mkdir(ftp, cpy)) {
	    char *s = strrchr(cpy, '/');
	    if (s) {
		s[0] = '\0';
		if (makeremotePath(ftp, cpy)) {
		    ret = -1;
		} else {
		    ret = makeremotePath(ftp, path);
		}
	    } else {
		ret = -1;
	    }
	} else {
	    ret = 0;
	}
    } else {
	ret = 0;
    }

    free(cpy);
    return ret;
}

int pftp_makePath(const char *ftpname, const char *path)
{
    if (ftpname == NULL) {
	return makelocalPath(path, 0755);
    } else {
	return makeremotePath(pftp_getFTPCtrl(ftpname), path);
    }
}

void pftp_createFTPSettings(const char *ftpname, const char *username, 
			    const char *passwd, const char *hostname, 
			    unsigned short port, int ssl, int sftp)
{
    ftp_server = realloc(ftp_server,
			 (ftp_servers + 1) * sizeof(struct ftpserver_s));
    memset(ftp_server + ftp_servers, 0, sizeof(struct ftpserver_s));

    ftp_server[ftp_servers].settings = newFTPSettings(ftpname, global->ftp);
    ftp_server[ftp_servers].settings->username = alloc_strcpy(username);
    ftp_server[ftp_servers].settings->password = alloc_strcpy(passwd);

    ftp_server[ftp_servers].settings->hosts = 1;
    ftp_server[ftp_servers].settings->hostname = malloc(sizeof(char *));
    ftp_server[ftp_servers].settings->port = malloc(sizeof(uint16_t));
    ftp_server[ftp_servers].settings->hostname[0] = alloc_strcpy(hostname);
    ftp_server[ftp_servers].settings->port[0] = port;
    ftp_server[ftp_servers].settings->implicid_ssl = ssl;
    ftp_server[ftp_servers].settings->sftp = sftp;
    ftp_servers++;
}

size_t pftp_getFreeFTPID(void)
{
    size_t id = 0, f;

    for (f = 0; f < ftp_servers; f++)
	if (ftp_server[f].settings->name[0] == '\n')
	    id = strtoul(ftp_server[f].settings->name + 1, NULL, 10) + 1;

    return id;
}

size_t pftp_getNrOfSettings(void)
{
    size_t real_servers = 0, f;

    for (f = 0; f < ftp_servers; f++) 
	if (ftp_server[f].settings->name[0] != '\n')
	    real_servers++;

    return real_servers;
}

pftp_settings_t pftp_getFTPSettings2(size_t id)
{
    size_t f, cur_id = 0;

    for (f = 0; f < ftp_servers; f++) {
	if (ftp_server[f].settings->name[0] != '\n') {
	    if (cur_id == id)
		return ftp_server[f].settings;
	    cur_id++;	    
	}
    }

    return NULL;
}

int util_comm_central(pftp_msg_t msg, pftp_server_t ftp, void *userdata,
		      pftp_param_t param1, pftp_param_t param2)
{
    if (msg == PFTP_NEEDPORT) {
	size_t f;
	uint16_t *port = (uint16_t *)param1;
	const pftp_settings_t settings = (const pftp_settings_t)param2;

	(*port) = 0;
	
	if (settings->port_start == 0) {
	    (*port) = global_free_port++;
	    if (global_free_port > global->ftp->port_stop)
		global_free_port = global->ftp->port_start;		
	} else {
	    for (f = 0; f < ftp_servers; f++) {
		if (ftp_server[f].settings == settings) {
		    if (ftp_server[f].free_port < settings->port_start)
			ftp_server[f].free_port = settings->port_start;
		    (*port) = ftp_server[f].free_port++;
		    if (ftp_server[f].free_port > settings->port_stop)
			ftp_server[f].free_port = settings->port_start;
		    break;
		}
	    }
	    
	    if (!(*port)) {
		/* Fallback to global */
		(*port) = global_free_port++;
		if (global_free_port > global->ftp->port_stop)
		    global_free_port = global->ftp->port_start;	
	    }
	}
	
	return 0;
    } else if (msg == PFTP_DISCONNECTED) {
	size_t f;
	for (f = 0; f < ftp_servers; f++) {
	    if (ftp_server[f].ctrl == ftp) {
		if (!ftp_server[f].save_path)
		    pftp_curdir(ftp, &ftp_server[f].save_path, 0);
		ftp_server[f].disconnected = 1;
		break;
	    }
	}
	return 0;
    }

    return cli_comm_central(msg, ftp, userdata, param1, param2);
}

int pftp_saveFTPSettings(const char *orgname, const char *targetname,
			 int save_password, int clear_password)
{
    if (orgname && targetname) {
	/* Save site settings */
	pftp_settings_t conf = NULL;
	int ret;
	size_t cur = 0;
	
	assert((!save_password && !clear_password) || 
	       (save_password && !clear_password) ||
	       (!save_password && clear_password));

	for (cur = 0; cur < ftp_servers; cur++) 
	    if (strcmp(ftp_server[cur].settings->name, orgname) == 0) {
		conf = ftp_server[cur].settings;
		break;
	    }

	if (!conf) {
	    util_error("ERR: Unknown origional name `%s'", orgname);
	    return -1;
	}

	ret = sethost(targetname, save_password, clear_password, conf);
	if (ret || strcmp(orgname, targetname) == 0)
	    return ret;

	if (orgname[0] == '\n') {
	    /* Just change name */
	    ftp_server[cur].settings->name = 
		realloc_strcpy(ftp_server[cur].settings->name, targetname);
	} else {
	    /* Add a new copy */
	    ftp_server = realloc(ftp_server,
				 (ftp_servers+1) * sizeof(struct ftpserver_s));
	    memset(ftp_server + ftp_servers, 0, sizeof(struct ftpserver_s));
	    ftp_server[ftp_servers].settings = newFTPSettings("", global->ftp);
	    copyFTPSettings(ftp_server[ftp_servers].settings, conf);
	    ftp_server[ftp_servers].settings->name = 
		realloc_strcpy(ftp_server[ftp_servers].settings->name,
			       targetname);
	}

	return ret;
    } else {
	/* Save default settings */
	return setconf(global);
    }
}
