/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif
#if HAVE_CTYPE_H
# include <ctype.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#if !HAVE_GETLINE
# include "getline.h"
#endif
#include "str.h"
#include "parse_utils.h"
#include "gethosts.h"
#include "hosts_settings.h"
#include "util.h"
#include "b64.h"

/* Set default configuration settings and return.
 * Secure connections will be used if possible as
 * default.
 */
static pftp_settings_t *host_default;

/* Init default settings. */
static void init_default2(void);
static pftp_settings_t *parse_hosts_file(FILE *file, char *conf, 
					 int *nr_settings);

/* Get hosts for the client. First look at ~/.pftphosts then /etc/pftphosts and
 * as last resort configuration defaults.
 */
pftp_settings_t *get_hosts(int *nr_settings)
{
#ifdef WIN32
    const char *homedrive = getenv("HOMEDRIVE");
    const char *homepath = getenv("HOMEPATH");
#else
    const char *home = getenv("HOME");
#endif
    char *conf = NULL;
    pftp_settings_t *ret = NULL;
    FILE *file = NULL;
    (*nr_settings) = 0;
    init_default2();
#ifdef WIN32
    if (homepath != NULL && homedrive != NULL) {
        conf = alloc_strcpy(homedrive);
        conf = realloc_strcat(conf, homepath);
	conf = realloc_strcat(conf, "\\.pftphosts");
#else
    if (home != NULL) {
        conf = alloc_strcpy(home);
    	conf = realloc_strcat(conf, "/.pftphosts");
#endif
	file = fopen(conf, "rb");
    }
#ifndef WIN32
    if (file == NULL) {
	util_error("Couldn't open file %s: %s, checking /etc/pftphosts.\n", 
		   conf, strerror(errno));
	free(conf);
	conf = alloc_strcpy("/etc/pftphosts");
	file = fopen(conf, "rb");
    }
#endif
    if (file == NULL) {
	util_error("Couldn't open file %s: %s, using defaults.\n"
		   "Use setdef to set default values and savedef to save.\n"
		   "Use set to set ftp attributes and save to save.\n",
		   conf,
		   strerror(errno));
	free(conf);
	return host_default;
    }
    ret = parse_hosts_file(file, conf, nr_settings);
    fclose(file);
    free(conf);
    return ret;
}

void init_default2(void)
{
    host_default = NULL;
}

static int parse_host_row(char *line, pftp_settings_t conf, int line_count,
			  const char *file_name)
{
    int valid = 0;
    char *name, *value;

    if (line[0] == '[')	return 0;

    if (split_conf_row(line, &name, &value)) {
	util_error("%s:%d: Invalid line %s", file_name, line_count, line);
	return 0;
    }

    if (!strcmp(name, "PASV")) {
	if (!(set_yes_or_no(value, &(conf->pasv))))
	    return 0;
    } else if (!strcmp(name, "PORT_START")) {
	long port_start	= parse_port(value, &valid);
	if (valid) {
	    conf->port_start = (uint16_t) port_start;
	    return 0;
	}
    } else if (!strcmp(name, "PORT_STOP")) {
	long port_stop = parse_port(value, &valid);
	if (valid) {
	    conf->port_stop = (uint16_t) port_stop;
	    return 0;
	}
    } else if (!strcmp(name, "IF")) {	
	if (conf->bind_to) free(conf->bind_to);

	if (check_interface(value)) {
	    conf->bind_to = NULL;
	    util_error("%s:%d: Invalid interface `%s'", file_name, line_count, 
		       value);   
	} else {
	    conf->bind_to = strdup(value);
	}
	return 0;
    } else if (!strcmp(name, "OUTWARD_IP")) {
	if (conf->myaddr) free(conf->myaddr);

	if (check_ip(value)) {
	    conf->myaddr = NULL;
	    util_error("%s:%d: Invalid outward ip `%s'", file_name, line_count,
		       value);   
	} else {
	    conf->myaddr = strdup(value);
	}
	return 0;
    } else if (!strcmp(name, "PRIO_LIST")) {
	free_split(&conf->prio_list, &conf->prio_list_len);
	conf->prio_list_len = split(value, " ", &conf->prio_list, NULL);
	if (check_list((const char **)conf->prio_list, conf->prio_list_len)) {
	    free_split(&conf->prio_list, &conf->prio_list_len);
	    util_error("%s:%d: Invalid priority list `%s'", file_name, 
		       line_count, value);
	}
	return 0;
    } else if (!strcmp(name, "IGNORE_LIST")) {
	free_split(&conf->ignore_list, &conf->ignore_list_len);
	conf->ignore_list_len = split(value, " ", &conf->ignore_list,NULL);
	if (check_list((const char **)conf->ignore_list, 
		       conf->ignore_list_len)) {
	    free_split(&conf->ignore_list, &conf->ignore_list_len);
	    util_error("%s:%d: Invalid ignore list `%s'", file_name, 
		       line_count, value);
	}
	return 0;
    } else if (!strcmp(name, "USE_SECURE")) {
	if (!(set_yes_or_no(value, &(conf->use_secure))))
	    return 0;
    } else if (!strcmp(name, "SECURE_LIST")) {
	if (!(set_yes_or_no(value, &(conf->secure_list))))
	    return 0;
    } else if (!strcmp(name, "SECURE_DATA")) {
	if (!(set_yes_or_no(value, &(conf->secure_data))))
	    return 0;
    } else if (!strcmp(name, "IMPLICIT_SSL")) {
	if (!(set_yes_or_no(value, &(conf->implicid_ssl))))
	    return 0;
    } else if (!strcmp(name, "USERNAME")) {
	conf->username = realloc_strcpy(conf->username, value);
	return 0;
    } else if (!strcmp(name, "PASSWORD")) {
	char *enc_pass;
	size_t new_len, old_len;
	enc_pass = strdup(value);
	old_len = strlen(enc_pass);
	if (old_len > 0) {
	    b64_decode(enc_pass, old_len, &conf->password, &new_len);
	}
	free(enc_pass);
	return 0;
    } else if (!strcmp(name, "ACCOUNT")) {
	conf->account = realloc_strcpy(conf->account, value);
	return 0;
    } else if (!strcmp(name, "INCLUDE_GLOBAL_PRIO")) {
	if (!(set_yes_or_no(value, &(conf->include_global_prio))))
	    return 0;
    } else if (!strcmp(name, "INCLUDE_GLOBAL_IGNORE")) {
	if (!(set_yes_or_no(value, &(conf->include_global_ignore))))
	    return 0;
    } else if (!strcmp(name, "SERVERS")) {
	if (!set_servers(value, &(conf->hostname), &(conf->port),
			 &(conf->hosts))) {
	    return 0;
	} else {
	    util_error("%s:%d: No servers for host, skipping...", file_name,
		       line_count);
	    return -1;
	}
    } else if (!strcmp(name, "PORT")) {
	util_error("%s:%d: Error parsing %s, deprecated.",
		   file_name,
		   line_count,
		   line);
	return 0;
    } else if (line[0] == '\0') {
	return 0;
    }
    util_error("%s:%d: Error parsing %s", file_name, line_count, line);
    return 0;
}

static void jump_comments(FILE *file, int *line_count)
{
    char* check = NULL;
    size_t buf = 0;
    ssize_t line_len = 0, read_len = 0;
    do {
	line_len = read_len = getline(&check, &buf, file);
	if (line_len == -1) break;
	trim(check);
	line_len = (int) strlen(check);
	(*line_count)++;
    } while (line_len == 0 || (line_len > 0 && check[0] == '#'));
    if (check)
	free(check);
    (*line_count)--;
    if (line_len == -1)
	return;
    fseek(file, -read_len, SEEK_CUR);
}

static void set_host_defaults(pftp_settings_t host)
{
    memset(host, 0, sizeof (struct pftp_settings_s));
    host->username = alloc_strcpy("anonymous");
    host->pasv = -1;
    host->include_global_prio = 1;
    host->include_global_ignore = 1;
    host->use_secure = 1;
    host->secure_list = 1;
    host->secure_data = 1;
}

static void set_host_name(pftp_settings_t host, char *name, int name_len)
{
    host->name = malloc(name_len + 1);
    strcpy(host->name, name + 1);
    trim_right(host->name);
    host->name[strlen(host->name) - 1] = '\0';
}

static int parse_host(pftp_settings_t host, char *conf, int *line_count, 
		      FILE *file)
{
    char *line = NULL;
    size_t buf = 0;
    int line_len = 0;
    do {
	jump_comments(file, line_count);
	line_len = getline(&line, &buf, file);
	if (line_len == -1)
	    break;
	(*line_count)++;
	trim(line);
	if (parse_host_row(line, host, (*line_count), conf) == -1) {
	    free(line);
	    return -1;
	}
    } while (line_len == 0 || (line_len > 0 && !(line[0] == '[')));
    if (line_len > 0) {
	fseek(file, -line_len, SEEK_CUR);
	(*line_count)--;
    }
    free(line);
    return 0;
}

static int add_host(pftp_settings_t **hosts, int *nr_settings, char *conf,
		    int *line_count, FILE *file)
{
    char *line = NULL;
    size_t buf = 0;
    int line_len = 0;
    pftp_settings_t new_host = malloc(sizeof(struct pftp_settings_s));
    do {
	line_len = getline(&line, &buf, file);
	(*line_count)++;
    } while (line_len == 0 || (line_len > 0 && line[0] != '['));
    set_host_defaults(new_host);
    set_host_name(new_host, line, line_len);
    if (line_len != -1) {
	if (!parse_host(new_host, conf, line_count, file)) {
	    (*hosts) = realloc((*hosts), 
			       ((*nr_settings) + 1) * sizeof(pftp_settings_t));
	    (*hosts)[(*nr_settings)++] = new_host;
	    free(line);
	    return 0;
	}
    }
    freeFTPSettings(&new_host);
    free(line);
    return -1;
}

static pftp_settings_t *parse_hosts_file(FILE *file, char *conf, 
					 int *nr_settings)
{
    char *line = NULL;
    size_t buf = 0;
    int line_len = 0;
    pftp_settings_t *ret = NULL;
    int line_count = 0;
    jump_comments(file, &line_count);
    line_len = getline(&line, &buf, file);
    trim_right(line);
    line_count++;
    if (strcmp(line, "[pftp-hosts]") != 0) {
	util_error("%s: Not a valid hosts file. \"[pftp-hosts]\" must be first (not `%s').", conf, line);
	free(line);
	return ret;
    }
    while (!feof(file)) {
	jump_comments(file, &line_count);
	add_host(&ret, nr_settings, conf, &line_count, file);
    }
    free(line);
    return ret;
}
