/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <inttypes.h>
#include <string.h>

#include <pftp_settings.h>
#include <pftp_default.h>
#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif

#include "parse_utils.h"
#include "set_utils.h"
#include "pftputil_settings.h"
#include "str.h"

#define getValue(__f, __v) \
  (__f->__v == -1 ? __f->global->__v : __f->__v)
#define getStrValue(__f, __v) \
  (__f->__v == NULL ? __f->global->__v:__f->__v)

static void printVal(char **out, const char *var, const char *value);

static int _getSiteValue(pftp_settings_t conf, const char *var, char **out)
{
    if (strcmp(var, "servers") == 0) {
	char *tmpstr = alloc_strcpy(""), *tmpstr2 = NULL;
	size_t h, len;
	for (h = 0; h < conf->hosts; h++) {
	    if (!conf->hostname[h])
		continue;
	    len = strlen(conf->hostname[h]);
	    tmpstr2 = realloc(tmpstr2, len + 25);
	    snprintf(tmpstr2, len + 25, "%s:%u ", conf->hostname[h],
		     (conf->port[h] == 0) ? 21 : conf->port[h]);
	    tmpstr = realloc_strcat(tmpstr, tmpstr2);
	}
	printVal(out, var, tmpstr);
	free(tmpstr);
	if (tmpstr2) free(tmpstr2);
	return 0;
    } else if (strcmp(var, "username") == 0) {
	printVal(out, var, conf->username);
	return 0;
    } else if (strcmp(var, "pasv") == 0) {
	printVal(out, var, yesno(getValue(conf, pasv)));
	return 0;
    } else if (strcmp(var, "port_start") == 0) {
	char *tmpstr = NULL;
	longval(&tmpstr, conf->port_start ? conf->port_start : 
		     conf->global->port_start);
	printVal(out, var, tmpstr);
	free(tmpstr);
	return 0;
    } else if (strcmp(var, "port_stop") == 0) {
	char *tmpstr = NULL;
	longval(&tmpstr, conf->port_start ? conf->port_stop :
		     conf->global->port_stop);
	printVal(out, var, tmpstr);
	free(tmpstr);
	return 0;
    } else if (strcmp(var, "if") == 0) {
	printVal(out, var, getStrValue(conf, bind_to));
	return 0;
    } else if (strcmp(var, "outward_ip") == 0) {
	printVal(out, var, getStrValue(conf, myaddr));
	return 0;
    } else if (strcmp(var, "use_secure") == 0) {
	printVal(out, var, yesno(getValue(conf, use_secure)));
	return 0;
    } else if (strcmp(var, "secure_list") == 0) {
	printVal(out, var, yesno(getValue(conf, secure_list)));
	return 0;
    } else if (strcmp(var, "secure_data") == 0) {
	printVal(out, var, yesno(getValue(conf, secure_data)));
	return 0;
    } else if (strcmp(var, "implicit_ssl") == 0) {
	printVal(out, var, yesno(getValue(conf, implicid_ssl)));
	return 0;
    } else if (strcmp(var, "account") == 0) {
	printVal(out, var, conf->account);
	return 0;
    } else if (strcmp(var, "include_global_prio") == 0) {
	printVal(out, var, yesno(conf->include_global_prio));
	return 0;
    } else if (strcmp(var, "include_global_ignore") == 0) {
	printVal(out, var, yesno(conf->include_global_ignore));
	return 0;
    } else if (strcmp(var, "prio_list") == 0) {
	char *tmpstr = unsplit((const char **)conf->prio_list, 
			       conf->prio_list_len, " ");
	printVal(out, var, tmpstr);
	free(tmpstr);
	return 0;
    } else if (strcmp(var, "ignore_list") == 0) {
	char *tmpstr = unsplit((const char **)conf->ignore_list, 
			       conf->ignore_list_len, " ");
	printVal(out, var, tmpstr);
	free(tmpstr);
	return 0;
    }

    return -1;
}

int getSiteValue(pftp_settings_t conf, const char *var, char **out)
{
    if (var) {
	(*out) = realloc_strcpy((*out), "");
	return _getSiteValue(conf, var, out);
    } else {
	(*out) = realloc_strcpy((*out), "");
	if (_getSiteValue(conf, "servers", out))
	    return -1;
	if (_getSiteValue(conf, "username", out))
	    return -1;
	if (_getSiteValue(conf, "pasv", out))
	    return -1;
	if (_getSiteValue(conf, "port_start", out))
	    return -1;
	if (_getSiteValue(conf, "port_stop", out))
	    return -1;
	if (_getSiteValue(conf, "if", out))
	    return -1;
	if (_getSiteValue(conf, "outward_ip", out))
	    return -1;
	if (_getSiteValue(conf, "use_secure", out))
	    return -1;
	if (_getSiteValue(conf, "secure_list", out))
	    return -1;
	if (_getSiteValue(conf, "secure_data", out))
	    return -1;
	if (_getSiteValue(conf, "implicit_ssl", out))
	    return -1;
	if (_getSiteValue(conf, "account", out))
	    return -1;
	if (_getSiteValue(conf, "include_global_prio", out))
	    return -1;
	if (_getSiteValue(conf, "include_global_ignore", out))
	    return -1;
	if (_getSiteValue(conf, "prio_list", out))
	    return -1;
	if (_getSiteValue(conf, "ignore_list", out))
	    return -1;

	return 0;
    }
}

static int _getDefValue(pftp_default_t conf, const char *var, char **out)
{
    if (strcmp(var, "pasv") == 0) {
	printVal(out, var, yesno(conf->pasv));
	return 0;
    } else if (strcmp(var, "port_start") == 0) {
	char *tmpstr = NULL;
	longval(&tmpstr, conf->port_start);
	printVal(out, var, tmpstr);
	free(tmpstr);
	return 0;
    } else if (strcmp(var, "port_stop") == 0) {
	char *tmpstr = NULL;
	longval(&tmpstr, conf->port_stop);
	printVal(out, var, tmpstr);
	free(tmpstr);
	return 0;
    } else if (strcmp(var, "if") == 0) {
	printVal(out, var, conf->bind_to);
	return 0;
    } else if (strcmp(var, "outward_ip") == 0) {
	printVal(out, var, conf->myaddr);
	return 0;
    } else if (strcmp(var, "use_secure") == 0) {
	printVal(out, var, yesno(conf->use_secure));
	return 0;
    } else if (strcmp(var, "secure_list") == 0) {
	printVal(out, var, yesno(conf->secure_list));
	return 0;
    } else if (strcmp(var, "secure_data") == 0) {
	printVal(out, var, yesno(conf->secure_data));
	return 0;
    } else if (strcmp(var, "implicit_ssl") == 0) {
	printVal(out, var, yesno(conf->implicid_ssl));
	return 0;
    } else if (strcmp(var, "prio_list") == 0) {
	char *tmpstr = unsplit((const char **)conf->prio_list, 
			       conf->prio_list_len, " ");
	printVal(out, var, tmpstr);
	free(tmpstr);
	return 0;
    } else if (strcmp(var, "ignore_list") == 0) {
	char *tmpstr = unsplit((const char **)conf->ignore_list, 
			       conf->ignore_list_len, " ");
	printVal(out, var, tmpstr);
	free(tmpstr);
	return 0;
    }

    return -1;
}

int getDefValue(pftp_default_t conf, const char *var, char **out)
{
    if (var) {
	(*out) = realloc_strcpy((*out), "");
	return _getDefValue(conf, var, out);
    } else {
	(*out) = realloc_strcpy((*out), "");
	if (_getDefValue(conf, "pasv", out))
	    return -1;
	if (_getDefValue(conf, "port_start", out))
	    return -1;
	if (_getDefValue(conf, "port_stop", out))
	    return -1;
	if (_getDefValue(conf, "if", out))
	    return -1;
	if (_getDefValue(conf, "outward_ip", out))
	    return -1;
	if (_getDefValue(conf, "use_secure", out))
	    return -1;
	if (_getDefValue(conf, "secure_list", out))
	    return -1;
	if (_getDefValue(conf, "secure_data", out))
	    return -1;
	if (_getDefValue(conf, "implicit_ssl", out))
	    return -1;
	if (_getDefValue(conf, "prio_list", out))
	    return -1;
	if (_getDefValue(conf, "ignore_list", out))
	    return -1;

	return 0;
    }
}


static int setYesNoValue(int *var, const char *in);
static int setPortValue(uint16_t *port, const char *in);
static int setStrValue(char **str, const char *in, 
		       int (* check_func)(const char *val));
static int setListValue(char ***list, size_t *listlen, const char *in);

int setSiteValue(pftp_settings_t conf, const char *var, const char *in)
{
    if (strcmp(var, "servers") == 0) {
	char **tmp = NULL;
	uint16_t *port;
	size_t len = 0, c;
	if (set_servers(in, &tmp, &port, &len)) {
	    for (c = 0; c < len; c++)
		free(tmp[c]);
	    if (tmp) free(tmp);
	    if (port) free(port);
	    return -1;
	}
	for (c = 0; c < conf->hosts; c++)
	    free(conf->hostname[c]);
	if (conf->hostname) free(conf->hostname);
	if (conf->port) free(conf->port);

	conf->hostname = tmp;
	conf->port = port;
	conf->hosts = len;

	return 0;
    } else if (strcmp(var, "username") == 0) {
	conf->username = realloc_strcpy(conf->username, in);
	return 0;
    } else if (strcmp(var, "pasv") == 0) {
	return setYesNoValue(&conf->pasv, in);
    } else if (strcmp(var, "port_start") == 0) {
	return setPortValue(&conf->port_start, in);
    } else if (strcmp(var, "port_stop") == 0) {
	return setPortValue(&conf->port_stop, in);
    } else if (strcmp(var, "if") == 0) {
	return setStrValue(&conf->bind_to, in, check_interface);
    } else if (strcmp(var, "outward_ip") == 0) {
	return setStrValue(&conf->myaddr, in, check_ip);
    } else if (strcmp(var, "use_secure") == 0) {
	return setYesNoValue(&conf->use_secure, in);
    } else if (strcmp(var, "secure_list") == 0) {
	return setYesNoValue(&conf->secure_list, in);
    } else if (strcmp(var, "secure_data") == 0) {
	return setYesNoValue(&conf->secure_data, in);
    } else if (strcmp(var, "implicit_ssl") == 0) {
	return setYesNoValue(&conf->implicid_ssl, in);
    } else if (strcmp(var, "account") == 0) {
	conf->account = realloc_strcpy(conf->account, in);
	return 0;
    } else if (strcmp(var, "include_global_prio") == 0) {
	return setYesNoValue(&conf->include_global_prio, in);
    } else if (strcmp(var, "include_global_ignore") == 0) {
	return setYesNoValue(&conf->include_global_ignore, in);
    } else if (strcmp(var, "prio_list") == 0) {
	return setListValue(&conf->prio_list, &conf->prio_list_len, in);
    } else if (strcmp(var, "ignore_list") == 0) {
	return setListValue(&conf->ignore_list, &conf->ignore_list_len, in);
    }

    return 1;
}

int setDefValue(pftp_default_t conf, const char *var, const char *in)
{
    if (strcmp(var, "pasv") == 0) {
	return setYesNoValue(&conf->pasv, in);
    } else if (strcmp(var, "port_start") == 0) {
	return setPortValue(&conf->port_start, in);
    } else if (strcmp(var, "port_stop") == 0) {
	return setPortValue(&conf->port_stop, in);
    } else if (strcmp(var, "if") == 0) {
	return setStrValue(&conf->bind_to, in, check_interface);
    } else if (strcmp(var, "outward_ip") == 0) {
	return setStrValue(&conf->myaddr, in, check_ip);
    } else if (strcmp(var, "use_secure") == 0) {
	return setYesNoValue(&conf->use_secure, in);
    } else if (strcmp(var, "secure_list") == 0) {
	return setYesNoValue(&conf->secure_list, in);
    } else if (strcmp(var, "secure_data") == 0) {
	return setYesNoValue(&conf->secure_data, in);
    } else if (strcmp(var, "implicit_ssl") == 0) {
	return setYesNoValue(&conf->implicid_ssl, in);
    } else if (strcmp(var, "prio_list") == 0) {
	return setListValue(&conf->prio_list, &conf->prio_list_len, in);
    } else if (strcmp(var, "ignore_list") == 0) {
	return setListValue(&conf->ignore_list, &conf->ignore_list_len, in);
    }

    return 1;
}

void printVal(char **out, const char *var, const char *value)
{
    (*out) = realloc_strcat((*out), var);
    (*out) = realloc_strcat((*out), "=");
    (*out) = realloc_strcat((*out), value);
    (*out) = realloc_strcat((*out), "\n");
}

int setYesNoValue(int *var, const char *in)
{
    int ret = yes_or_no(in);
    if (ret == -1 && strlen(in) > 0)
	return -1;
    
    (*var) = ret;

    return 0;
}

int setPortValue(uint16_t *port, const char *in)
{
    int valid;
    long tmp = parse_port(in, &valid);

    if (!valid && strlen(in) > 0)
	return -1;

    if (!valid)
	(*port) = 0;
    else
	(*port) = (uint16_t)tmp;

    return 0;
}

int setStrValue(char **str, const char *in, 
		int (* check_func)(const char *val))
{
    int ret = check_func(in);

    if (ret == 0) {
	(*str) = realloc_strcpy((*str), in);
	return 0;
    } 

    return -1;
}

int setListValue(char ***list, size_t *listlen, const char *in)
{
    char **tmp = NULL;
    size_t len = split(in, " ", &tmp);

    if (check_list((const char **)tmp, len)) {
	free_split(&tmp, &len);
	return -1;
    }

    free_split(list, listlen);
    (*list) = tmp;
    (*listlen) = len;

    return 0;
}

