/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <unistd.h>
#include <inttypes.h>

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#include <pftp_settings.h>

#if !HAVE_GETLINE
# include "getline.h"
#endif
#include "str.h"
#include "parse_utils.h"
#include "pftp_getconf.h"
#include "util.h"
#include "hosts_settings.h"

/* Set default configuration settings and return.
 * Secure connections will be used if possible as
 * default.
 */
static pftp_conf_t conf_defaults = NULL;

/* Init default settings. */
static void init_default(void);
static void free_default(void);

static pftp_conf_t parse_conf_file(FILE *file, const char *file_name);

/* Get configuration for the client. First look at ~/.pftprc
 * then /etc/pftprc and as last way out resort to configuration
 * defaults.
 */
pftp_conf_t pftp_get_conf(void)
{
    const char *home = getenv("HOME");
    char *conf = NULL;
    FILE *file = NULL;
    pftp_conf_t ret;
    init_default();
    if (home != NULL) {
        conf = alloc_strcpy(home);
	conf = realloc_strcat(conf, "/.pftprc");
	file = fopen(conf, "r");
    }
    if (file == NULL) {
	util_error("Couldn't open file %s: %s, checking /etc/pftprc.",
		   conf, strerror(errno));
	free(conf);
	conf = alloc_strcpy("/etc/pftprc");
	file = fopen(conf, "r");
    }
    if (file == NULL) {
	util_error("Couldn't open file %s: %s, using defaults.",
		   conf, strerror(errno));
	free(conf);
	return conf_defaults;
    }
    ret = parse_conf_file(file, conf);
    if (!strcmp(conf, "/etc/pftprc")) {
	free(conf);
	return ret;
    } else if (ret == conf_defaults) {
	free(conf);
	conf = alloc_strcpy("/etc/pftprc");
	file = fopen(conf, "r");
	if (file != NULL) {
	    free(conf);
	    return parse_conf_file(file, conf);
	} else {
	    util_error("Couldn't open file %s: %s, using defaults.",
		       conf, strerror(errno));
	    free(conf);
	    return conf_defaults;
	}
    } else {
	free(conf);
	return ret;
    }
    free(conf);
    return conf_defaults;
}

void init_default()
{
    conf_defaults = malloc(sizeof(struct pftp_conf_s));
    memset(conf_defaults, 0, sizeof(struct pftp_conf_s));
    conf_defaults->ftp = newDefault();
    conf_defaults->ftp->secure_list = 1;
    conf_defaults->ftp->secure_data = 1;
    conf_defaults->ftp->use_secure = 1;
}

static void parse_conf_row(char *line, pftp_conf_t conf, int line_count,
			   const char *file_name)
{
    int valid = 0;
    int len = strlen(line);

    if (len > 0 && line[len - 1] == '\n')
	line[len - 1] = '\0';

    if (!strncmp(line, "PASV=", 5)) {	
	if (!(set_yes_or_no(line + 5, &(conf->ftp->pasv)))) {
	    return;
	}
    } else if (!strncmp(line, "PORT_START=", 11)) {
	long port_start	= parse_port(line + 11, &valid);
	if (valid) {
	    conf->ftp->port_start = port_start;
	    return;
	}
    } else if (!strncmp(line, "PORT_STOP=", 10)) {
	long port_stop = parse_port(line + 10, &valid);
	if (valid) {
	    conf->ftp->port_stop = port_stop;
	    return;
	}
    } else if (!strncmp(line, "IF=", 3)) {
	rm_head_ws(line + 3, &conf->ftp->bind_to);
	if (check_interface(conf->ftp->bind_to)) {
	    free(conf->ftp->bind_to);
	    conf->ftp->bind_to = NULL;
	    util_error("%s:%d: Invalid interface `%s'", file_name, line_count, 
		       line + 3);   
	}
	return;
    } else if (!strncmp(line, "OUTWARD_IP=", 11)) {
	rm_head_ws(line + 11, &conf->ftp->myaddr);
	if (check_ip(conf->ftp->myaddr)) {
	    free(conf->ftp->myaddr);
	    conf->ftp->myaddr = NULL;
	    util_error("%s:%d: Invalid outward ip `%s'", file_name, line_count,
		       line + 11);   
	}
	return;
    } else if (!strncmp(line, "PRIO_LIST=", 10)) {
	conf->ftp->prio_list_len = split(line + 10, " ", 
					     &conf->ftp->prio_list, NULL);
	if (check_list((const char **)conf->ftp->prio_list, 
		       conf->ftp->prio_list_len)) {
	    free_split(&conf->ftp->prio_list, 
		       &conf->ftp->prio_list_len);
	    util_error("%s:%d: Invalid priority list `%s'", file_name, 
		       line_count, line + 10);
	}
	return;
    } else if (!strncmp(line, "IGNORE_LIST=", 12)) {
	conf->ftp->ignore_list_len = split(line + 12, " ", 
					       &conf->ftp->ignore_list,
					       NULL);
	if (check_list((const char **)conf->ftp->ignore_list, 
		       conf->ftp->ignore_list_len)) {
	    free_split(&conf->ftp->ignore_list, 
		       &conf->ftp->ignore_list_len);
	    util_error("%s:%d: Invalid ignore list `%s'", file_name, 
		       line_count, line + 10);
	}
	return;
    } else if (!strncmp(line, "USE_SECURE=", 11)) {
	if (!(set_yes_or_no(line + 11, &(conf->ftp->use_secure))))
	    return;
    } else if (!strncmp(line, "SECURE_LIST=", 12)) {
	if (!(set_yes_or_no(line + 12, &(conf->ftp->secure_list))))
	    return;
    } else if (!strncmp(line, "SECURE_DATA=", 12)) {
	if (!(set_yes_or_no(line + 12, &(conf->ftp->secure_data))))
	    return;
    } else if (!strncmp(line, "IMPLICIT_SSL=", 13)) {
	if (!(set_yes_or_no(line + 13, &(conf->ftp->implicid_ssl))))
	    return;
    } else if (!strncmp(line, "GUI_WIDTH=", 10)) {
	unsigned long tmp = parse_ulong(line + 10, &valid);
	if (valid) {
	    conf->gui_width = tmp;
	    return;
	}
    } else if (!strncmp(line, "GUI_HEIGHT=", 11)) {
	unsigned long tmp = parse_ulong(line + 11, &valid);
	if (valid) {
	    conf->gui_height = tmp;
	    return;
	}
    } else if (!strncmp(line, "HORIZONTAL_PANEL1_POS=", 22)) {
	unsigned long tmp = parse_ulong(line + 22, &valid);
	if (valid) {
	    conf->horzpan1 = tmp;
	    return;
	}
    } else if (!strncmp(line, "HORIZONTAL_PANEL2_POS=", 22)) {    
	unsigned long tmp = parse_ulong(line + 22, &valid);
	if (valid) {
	    conf->horzpan2 = tmp;
	    return;
	}
    } else if (line[0] == '\0') {
	return;
    }
    util_error("%s:%d: Error parsing %s", file_name, line_count, line);
}

static void conf_cpy(pftp_conf_t DEST, pftp_conf_t SOURCE)
{
    memcpy(DEST, SOURCE, sizeof(struct pftp_conf_s));
    DEST->ftp = newDefault();
    cpyDefault(DEST->ftp, SOURCE->ftp);
}

void pftp_free_conf(pftp_conf_t conf)
{
    if (conf) {
	if (conf->ftp)
	    freeDefault(&conf_defaults->ftp);
	free(conf);
	if (conf == conf_defaults)
	    conf_defaults = NULL;	
    }
}

void free_default(void) 
{
    pftp_free_conf(conf_defaults);
}

pftp_conf_t parse_conf_file(FILE *file, const char *file_name)
{
    char *line = NULL, *clean_line = NULL;
    int line_count = 0;
    size_t N = 0;
    int first_line_ok = -1;
    pftp_conf_t ret =  malloc(sizeof(struct pftp_conf_s));
    conf_cpy(ret, conf_defaults);

    if (!feof(file)) {
	while (first_line_ok == -1) {
	    line_count++;
	    if (getline(&line, &N, file) == -1) {
		util_error("Error reading config file.");
		if (line) free(line);
		if (clean_line) free(clean_line);
		pftp_free_conf(ret);
		ret = conf_defaults;
		return ret;
	    }
	    rm_head_ws(line, &clean_line);
	    if (clean_line[0] == '#') {
		continue;
	    } else if (!strncmp(clean_line, "[pftp]", 6)) {
		break;
	    } else {
		util_error("Invalid config file.");
		if (line) free(line);
		if (clean_line) free(clean_line);
		pftp_free_conf(ret);
		ret = conf_defaults;
		return ret;
	    }
	}
    }
    while (!feof(file)) {
	line_count++;
	getline(&line, &N, file);
	rm_head_ws(line, &clean_line);
	if (clean_line[0] == '#') {
	    continue;
	} else {
	    parse_conf_row(clean_line, ret, line_count, file_name);
	}
    }

    if (line) free(line);
    if (clean_line) free(clean_line);
    return ret;
}

void pftp_close_conf(void)
{
    free_default();
}
