/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <inttypes.h>
#include <limits.h>
#include <errno.h>
#include <fnmatch.h>
#include <stdarg.h>
#if HAVE_STRING_H
# include <string.h>
#else
# if HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include <sys/stat.h>
#if HAVE_DIRENT_H
# include <dirent.h>
#else
# if HAVE_NDIR_H
#  include <ndir.h>
# else
#  if HAVE_SYS_DIR_H
#   include <sys/dir.h>
#  else
#   if HAVE_SYS_NDIR_H
#    include <sys/ndir.h>
#   endif
#  endif
# endif
#endif
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <pwd.h>
#include <grp.h>
#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif

#include <pftp.h>

#include "pftp_item.h"
#include "pftputil.h"
#include "str.h"
#include "pftp_client.h"
#include "util.h"

#ifndef LLONG_MAX
# define LLONG_MAX LONG_LONG_MAX
#endif

/* Local type used by begin_trans */
typedef struct {
    size_t count;
    char **path;
    pftp_server_t *ftp;
} clear_cache_t;

/* Function from libutil.c */
int util_comm_central(pftp_msg_t msg, pftp_server_t ftp, void *userdata, 
		      pftp_param_t param1, pftp_param_t param2);

/* Local functions */
static void get_username(uid_t uid, char **name);
static void get_groupname(gid_t gid, char **name);
static pftp_server_t _getFTP(const char *name);

static int trans_cd(const char *ftp_name, const char *path) 
{
    if (!ftp_name) {
	return chdir(path);
    } else { 
	pftp_server_t ftp = _getFTP(ftp_name);
	
	return (ftp ? pftp_cd(ftp, path, NULL) : -1);
    }
}

static pftp_directory_t *inter_ls(pftp_server_t ftp, const char *filemask, 
				  int hidden)
{
    pftp_directory_t *Cdir = malloc(sizeof(pftp_directory_t));
    memset(Cdir, 0, sizeof(pftp_directory_t));    
    if (pftp_ls(ftp, Cdir, 0, NULL) == 0) {
	pftp_apply_mask(Cdir, filemask, hidden);
	return Cdir;
    }

    pftp_free_fdt((*Cdir));
    free(Cdir);
    return NULL;
}

int pftp_deleteDirectoryFTP(pftp_server_t ftp, const char *path, 
			    const char *dirname)
{
    int ret = 0;
    char *fullpath = alloc_strcpy(path);
    if (fullpath[strlen(fullpath) - 1] != '/')
	fullpath = realloc_strcat(fullpath, "/");
    fullpath = realloc_strcat(fullpath, dirname);

    if (!pftp_cd(ftp, path, NULL) && !pftp_cd(ftp, dirname, NULL)) {
	pftp_directory_t *ls;
	if ((ls = inter_ls(ftp, "*", 1))) {
	    size_t f;
	    for (f = 0; f < ls->length; f++) {
		if (ls->files[f].type == pft_directory) {
		    if (strcmp(ls->files[f].name, ".") &&
			strcmp(ls->files[f].name, "..")) {
			if (pftp_deleteDirectoryFTP(ftp, fullpath, 
						    ls->files[f].name)) {
			    ret = -1;
			}
		    }
		} else {
	            if (pftp_rm(ftp, ls->files[f].name)) {
			pftp_clear_dir_cache(ftp, ".");
			ret = -1;		    
		    }
		}
	    }
	    pftp_free_fdt((*ls));
	    free(ls);
	}
	
	if (!pftp_cd(ftp, path, NULL)) {	    
	    if (pftp_rmdir(ftp, dirname)) {
		pftp_clear_dir_cache(ftp, dirname);
		ret = -1;	    
	    }
	} else {
	    ret = -1;
	}
    } else {
	ret = -1;
    }

    pftp_clear_dir_cache(ftp, path);
    free(fullpath);
    
    return ret;
}

static pftp_file_type_t parse_file_type(mode_t M, int *link)
{
    if (link) {
	if(S_ISLNK(M)) {
	    (*link) = 1;
	    return pft_unknown;
	}
	(*link) = 0;
    }
    if(S_ISDIR(M))
	return pft_directory;
    if(S_ISCHR(M))
	return pft_character;
    if(S_ISBLK(M))
	return pft_block;
    if(S_ISREG(M))
	return pft_file;
    if(S_ISFIFO(M))
	return pft_fifo;
    if(S_ISSOCK(M))
	return pft_s_socket;
    
    return pft_unknown;      
}

static char *readlink_malloc(const char *filename)
{
    size_t size = 100;
    int ret;
    char *buffer = NULL;
          
    for (;;) {
	buffer = realloc(buffer, size);
	ret = readlink(filename, buffer, size);
	if (ret < 0) {
	    free(buffer);
	    return NULL;
	}
	if (ret < size) {
	    buffer[ret] = '\0';
	    return buffer;
	}
            
	size *= 2;
    }
}

static pftp_directory_t *local_dir_parse(char *dir, char **curpath)
{
    pftp_file_t file;
    struct dirent *d;
    struct stat64 buf;
    time_t now = time(NULL);
    struct tm *tm, *today = gmtime(&now);
    DIR *dstream = opendir(dir);
    size_t base_len = strlen(dir), tmp_len;
    char *path = malloc(base_len + 2), *tmp = NULL;
    pftp_directory_t *res = malloc(sizeof(pftp_directory_t));
    memset(res, 0, sizeof(pftp_directory_t));

    memcpy(path, dir, base_len + 1);
    if (path[0] && path[base_len-1] != '/') {
	path[base_len++] = '/';
	path[base_len] = '\0';
    }

    if (curpath) {
	if (!pftp_get_path_dir(NULL, curpath)) {
	    size_t len;
	    len = strlen((*curpath));
	    if (len > 1 && (*curpath)[len - 1] != '/')
		(*curpath) = realloc_strcat((*curpath), "/");
	    if (path[0])
		(*curpath) = realloc_strcat((*curpath), path);

	    pftp_compactPath(curpath);			    
	}
    }
    
    if (dstream) {
	while ((d = readdir(dstream))) {
	    tmp_len = base_len + strlen(d->d_name) + 1;
	    tmp = realloc(tmp, tmp_len);
	    memcpy(tmp, path, base_len);
	    memcpy(tmp + base_len, d->d_name, tmp_len - base_len);
	    lstat64(tmp, &buf);
	    file.name = alloc_strcpy(d->d_name);
	    file.type = parse_file_type(buf.st_mode, &file.link);
	    if (file.link) {
		struct stat64 buf2;
		char *target = NULL;
		
		if (!stat64(tmp, &buf2)) {
		    file.type = parse_file_type(buf2.st_mode, NULL);
		}
		
		if ((target = readlink_malloc(tmp))) {
		    file.name = realloc_strcat(file.name, " -> ");
		    file.name = realloc_strcat(file.name, target);
		    free(target);
		}
	    }
	    file.perm = buf.st_mode & 0777;
	    file.links = buf.st_nlink;
	    file.user = file.group = NULL;
	    get_username(buf.st_uid, &file.user);
	    get_groupname(buf.st_gid, &file.group);
	    file.size = buf.st_size;
	    tm = localtime(&buf.st_mtime);
	    
	    if (tm->tm_year == today->tm_year) {
		file.changed = malloc(13);
		strftime(file.changed, 13, "%b %e %H:%M", tm);
	    } else {
		file.changed = malloc(13);
		strftime(file.changed, 13, "%b %e  %Y", tm);
	    }
	    res->length++;
	    res->files = realloc(res->files, sizeof(pftp_file_t) * res->length);
	    res->files[res->length - 1] = file;
	}
	closedir (dstream);
    } else {
	util_error("Error: Couldn't open the directory `%s'. %s", dir, 
		   strerror(errno));
    }

    if (path) free(path);
    if (tmp) free(tmp);

    return res;
}

pftp_directory_t *pftp_trans_ls(const char *ftp_name, const char *pattern,
				char **path, int hidden, int link_names)
{
    if (ftp_name) {
	pftp_server_t ftp = _getFTP(ftp_name);

	if (ftp)
	    return pftp_trans_ls2(ftp, pattern, path, hidden, link_names);
    } else {
	return pftp_trans_ls2(NULL, pattern, path, hidden, link_names);
    }

    return NULL;
}

static void remove_link_names(pftp_directory_t *dir)
{
    size_t f = 0;

    while (f < dir->length) {
	char *pos = strstr(dir->files[f].name, " -> ");
	if (pos)
	    pos[0] = '\0';
	f++;
    }
}

pftp_directory_t *pftp_trans_ls2(pftp_server_t ftp, const char *pattern, 
				 char **path, int hidden, int link_names)
{
    pftp_directory_t *ret = NULL;
    char *_path = NULL, *pos = strrchr(pattern, '/');
    const char *real_pattern = pattern;
    
    if (pos) {
	real_pattern = ++pos;
	_path = malloc(pos - pattern);
	memcpy(_path, pattern, pos - pattern);	
	_path[pos - pattern] = '\0';
    }

    if (!ftp) {
	ret = local_dir_parse(_path ? _path : ".", path);
	if (ret) pftp_apply_mask(ret, 
				 strlen(real_pattern) ? real_pattern : "*", 
				 hidden);
    } else { 
	if (_path) {
	    char *old = NULL;
	    pftp_curdir(ftp, &old, 0);
	    if (old) {
		if (!pftp_cd(ftp, _path, NULL)) {
		    if (path)
			pftp_curdir(ftp, path, 0);
		    ret = inter_ls(ftp, 
				   strlen(real_pattern) ? real_pattern : "*", 
				   hidden);
		    pftp_cd(ftp, old, NULL);
		}
		free(old);
	    }
	} else {
	    if (path)
		pftp_curdir(ftp, path, 0);
	    ret = inter_ls(ftp, strlen(real_pattern) ? real_pattern : "*", 
			   hidden);
	}
    }

    if (ret && !link_names)
	remove_link_names(ret);

    return ret;
}

int pftp_get_path_dir(const char *ftpserver, char **path)
{
    if (ftpserver) {
	pftp_server_t ftp = _getFTP(ftpserver);

	if (!ftp) {
	    util_error("Error: Unable to connect to source. `%s'.", ftpserver);
	    return -1;
	}

	pftp_curdir(ftp, path, 0);
   
	if (!path) {
	    util_error("Error: Unable to get remote path.");
	    return -1;
	}
    } else {
	size_t size = 100;
	
	for (;;) {
	    (*path) = realloc((*path), size);
	    errno = 0;
	    if (getcwd((*path), size) == (*path))
		break;
	    if (errno != ERANGE) {
		util_error("Error: Unable to get local path. %s", 
			   strerror(errno));
		free((*path));
		return -1;
	    }

	    size *= 2;
	}
    }
    
    return 0;
}

void pftp_apply_mask(pftp_directory_t *filelist, const char *mask, 
		     int hidden)
{
    size_t i;

/*    if (strcmp(mask, "*"))
      hidden = 0;*/

    for (i = 1; i <= filelist->length; i++) {
	if ((strcmp(filelist->files[i - 1].name, ".") == 0 ||
	     strcmp(filelist->files[i - 1].name, "..") == 0) ||
	    (!hidden && filelist->files[i - 1].name[0] == '.') ||
	    fnmatch(mask, filelist->files[i - 1].name, 0)) {
	    pftp_free_fft(filelist->files[i - 1]);

	    if (i < filelist->length)
		memmove(filelist->files + (i - 1), filelist->files + i, 
			(filelist->length - i) * sizeof(pftp_file_t));
	    filelist->length--;
	    i--;
	}
    }    
}

static void sort_files(pftp_file_t *files, size_t len)
{
    size_t i, j;
    pftp_file_t tmpf;
    for (i = 0; i < len; i++) {
	for (j = i + 1; j < len; j++) {
	    if (strcmp(files[i].name, files[j].name) == 1) {
		memcpy(&tmpf, files + i, sizeof(pftp_file_t));
		memmove(files + i, files + j, sizeof(pftp_file_t));
		memcpy(files + j, &tmpf, sizeof(pftp_file_t));
	    }   
	}
    }
}

int pftp_get_files_ordered(const char *from, const char *to, 
			   const char *srcpath, pftp_directory_t *FileList, 
			   pftp_que_t fileque, int last)
{
    size_t I;
    char *destpath = NULL;
    pftp_directory_t dir;
    pftp_file_t tmpf;
    int ret = 0;
    size_t nrDirs = 0;

    dir.files = malloc(FileList->length * sizeof(pftp_file_t));
    dir.length = FileList->length;
    memcpy(dir.files, FileList->files, FileList->length * sizeof(pftp_file_t));
    
    /* Sort directories last and remove . and .. if found */
    if (dir.length > 1) {
	for (I = 1; I <= dir.length - nrDirs; I++) {
	    if (dir.files[I-1].type == pft_directory) {
		if (strcmp(dir.files[I-1].name, ".") == 0 ||
		    strcmp(dir.files[I-1].name, "..") == 0) {
		    /* no need to free here as pointers are really from
		       FileList
		       free_fft(dir.files[I-1]);
		    */
		    if (I < dir.length)
			memmove(dir.files + (I-1), dir.files + I, 
				(dir.length - I) * sizeof(pftp_file_t));
		    dir.length--;
		    I--;
		} else {
		    memcpy(&tmpf, dir.files + (I-1), sizeof(pftp_file_t));
		    memmove(dir.files + (I-1), 
			    dir.files + (dir.length - (1 + nrDirs)),
			    sizeof(pftp_file_t));
		    memcpy(dir.files + (dir.length - (1 + nrDirs)), &tmpf, 
			   sizeof(pftp_file_t));
		    nrDirs++;
		}
	    }
	}
    } else {
	for (I = 1; I <= dir.length; I++) {
	    if (strcmp(dir.files[I-1].name, ".") == 0 ||
		strcmp(dir.files[I-1].name, "..") == 0) {
		/* no need to free here as pointers are really from
		   FileList
		   free_fft(dir.files[I-1]);
		*/
		if (I < dir.length)
		    memmove(dir.files + (I-1), dir.files + I, 
			    (dir.length - I) * sizeof(pftp_file_t));
		dir.length--;
		I--;
	    }	    
	}
    }

    /* Sort files */
    sort_files(dir.files, dir.length - nrDirs);
    /* Sort directories */
    sort_files(dir.files + (dir.length - nrDirs), nrDirs);
    
    if (pftp_get_path_dir(to, &destpath)) {
	free(dir.files);
	return -1;
    }

    if ((ret = pftp_que_add_dir(from, to, srcpath, destpath, &dir, fileque, 
				last)))
	util_error("Error: Unable to add files to que.");
    
    free(destpath);
    free(dir.files);

    return ret;
}

static int trans_mask(const char *from, const char *to, const char *mask, 
		      pftp_que_t fileque)
{
    char *srcpath = NULL;
    pftp_directory_t *dir = pftp_trans_ls(from, mask, &srcpath, 1, 0);
    int ret = -1;
   
    if (dir) {
	ret = pftp_get_files_ordered(from, to, srcpath, dir, fileque, 0);
	pftp_free_fdt((*dir));
	free(dir);
    }

    if (srcpath)
	free(srcpath);

    return ret;
}

static int does_exist(pftp_item_t item, pftp_ExistingFilefunc_t dialog, 
		      uint64_t *filesize)
{
    pftp_directory_t *filelist = NULL;
    int response = 0;
    const char *target = item->target ? item->target : item->filename;
    filelist = pftp_trans_ls(item->ftp_to, target, NULL, 1, 0);

    if (filelist) {
	if (filelist->length) {
	    response = dialog(filelist->files[0], item);
	    *filesize = filelist->files[0].size;
	} else {
	    response = 0; // doesn't exist
	}
	pftp_free_fdt(*filelist);
	free(filelist);
    } else { 
	response = 0; // doesn't exist    
    }

    return response;
}

static int change_directories(const char *ftp_to, const char *destpath,
			      const char *ftp_from, const char *srcpath)
{
    if (trans_cd(ftp_to, destpath) != 0){
	util_error("Error: Unable to change to destination directory");
	return 1;
    } else if (trans_cd(ftp_from, srcpath) != 0) {
	util_error("Error: Unable to change to source directory");
	return 1;
    }
    return 0;
}

static int directory_trans(pftp_item_t item, pftp_que_t fileque)
{
    const char *target = item->target ? item->target : item->filename;

    if (trans_cd(item->ftp_from, item->filename)) {
	util_error("Error: Unable to change source directory");
	return 1;
    } else if (pftp_makePath(item->ftp_to, target)) {
	util_error("Error: Unable to create destination directory");
	return 1;
    } else if (trans_cd(item->ftp_to, item->destpath)) {
	return 1;
    } else if (trans_cd(item->ftp_to, target)) {
	util_error("Error: Unable to enter destination directory");
	return 1;
    } else if (trans_mask(item->ftp_from, item->ftp_to, "*", fileque)) {
	util_error("Error: Unable to walk recursively");
	return 1;
    }
    return 0;
}

static int checking_trans(pftp_item_t *item, pftp_ExistingFilefunc_t dialog,
			  uint64_t *filesize, int *ow, pftp_que_t fileque,
			  pftp_que_t errorque)
{
    (*ow) = does_exist((*item), dialog, filesize);
    if ((*ow) & PFTP_DLG_ABORT) {
	while (((*item) = pftp_get_next(fileque))) {
	    util_error("purging que");
	    pftp_que_add_last((*item), errorque);
	}
    }
    return ((*ow) & PFTP_DLG_ALL);
}

static void clear_and_free_cache(clear_cache_t *clear_cache)
{
    size_t i;

    for (i = 0; i < clear_cache->count; i++) {
	pftp_clear_dir_cache(clear_cache->ftp[i], clear_cache->path[i]);
	free(clear_cache->path[i]);
    }

    if (clear_cache->ftp)
	free(clear_cache->ftp);
    if (clear_cache->path)
	free(clear_cache->path);

    memset(clear_cache, 0, sizeof(clear_cache_t));
}

static void add_curdir_to_clear_cache(clear_cache_t *clear_cache, 
				      pftp_server_t ftp)
{
    char *path = NULL;
    size_t i;
    
    if (!ftp) {
	/* Local, isn't cached. */
	return;
    }
    
    pftp_curdir(ftp, &path, 0);
    if (!path) {
	/* Error, not really usefull to fix here */
	return; 
    }
    
    for (i = 0; i < clear_cache->count; i++) {
	if (clear_cache->ftp[i] == ftp &&
	    strcmp(clear_cache->path[i], path) == 0) {
	    /* Already in list */
	    free(path);
	    return;
	}	    
    }

    clear_cache->ftp = realloc(clear_cache->ftp, (clear_cache->count + 1) *
			       sizeof(pftp_server_t));
    clear_cache->path = realloc(clear_cache->path, (clear_cache->count + 1) *
				sizeof(char *));
    clear_cache->ftp[clear_cache->count] = ftp;
    clear_cache->path[clear_cache->count] = path;
    clear_cache->count++;
}

static void non_checking_trans(int *err, clear_cache_t *clear_cache, 
			       pftp_item_t item, int ow, uint64_t filesize)
{
    FILE *fh = NULL;
    const char *target = item->target ? item->target : item->filename;

    if (item->ftp_from == NULL) {
	if (!(fh = fopen64(item->filename, "rb"))) {
	    util_error("Error: Unable to open file. %s", strerror(errno));
	    (*err) = 1;
	} else {
	    pftp_server_t ftp = _getFTP(item->ftp_to);
	    if (ow & PFTP_DLG_OVERWRITE) {
		//If overwrite
		if (pftp_put(ftp, target, 'B', fh, 0, item->size)) {
		    /* If failed, try to remove target first */
		    if (pftp_rm(ftp, target)) {
			util_error("Error: Unable to upload file.");
			(*err) = 1;
		    } else if (pftp_put(ftp, target, 'B', fh, 0, item->size)) {
			util_error("Error: Unable to upload file.");
			(*err) = 1;
		    }
		}
	    } else if (ow & PFTP_DLG_RESUME) { //Resume
		uint64_t tmp = filesize;
		if (tmp > LLONG_MAX) {
		    fseeko64(fh, LLONG_MAX, SEEK_SET);
		    tmp -= LLONG_MAX;
		} 
		fseeko64(fh, tmp, SEEK_SET);

		if (pftp_put(ftp, target, 'B', fh, filesize, item->size)) {    
		    util_error("Error: Unable to upload file.");
		    (*err) = 1;
		}
	    } else if (!(ow & PFTP_DLG_SKIP)) {
		if (pftp_put(ftp, target, 'B', fh, 0, item->size)) {
		    util_error("Error: Unable to upload file.");
		    (*err) = 1;
		}
	    }

	    add_curdir_to_clear_cache(clear_cache, ftp);
	}
    } else if (item->ftp_to == NULL) {
	if (ow & PFTP_DLG_RESUME) {
	    if (!(fh = fopen64(target, "ab"))) {
		util_error("Error: Unable to create file. %s", 
			   strerror(errno));
		(*err) = 1;
	    } else {
		uint64_t tmp = filesize;
		if (tmp > LLONG_MAX) {
		    fseeko64(fh, LLONG_MAX, SEEK_SET);
		    tmp -= LLONG_MAX;
		} 
		fseeko64(fh, tmp, SEEK_SET);
	    } 
	} else if ((ow & PFTP_DLG_OVERWRITE) || ow == 0) {
	    if (!(fh = fopen64(target, "wb"))) {
		util_error("Error: Unable to create file. %s", 
			   strerror(errno));
		(*err) = 1;
	    } else {
		filesize = 0;
	    }
	}
	if (!(ow & PFTP_DLG_SKIP)) {
	    if (pftp_get(_getFTP(item->ftp_from), item->filename, 'B', fh, 
			 filesize, item->size, NULL)) {
		util_error("Error: Unable to download file.");
		(*err) = 1;
	    }
	}
    } else {
	if (ow & PFTP_DLG_RESUME) {
	    /* don\'t need to do anything */
	} else if ((ow & PFTP_DLG_OVERWRITE) || (ow == 0)) {
	    filesize = 0;
	}
	if (!(ow & PFTP_DLG_SKIP)) {
	    pftp_server_t ftp = _getFTP(item->ftp_to);
	    if (pftp_fxp(_getFTP(item->ftp_from), item->filename, ftp, target, 
			 'B', filesize, filesize, item->size)) {
		util_error("Error: Unable to FXP.");
		(*err) = 1;	    
	    } else {
		add_curdir_to_clear_cache(clear_cache, ftp);
	    }
	}
    }

    if (fh) {
	fclose(fh);
    }
}

static void add_to_origpath(const char *ftp, char ***name, char ***path,
			    size_t *paths)
{
    size_t o;
    if (ftp) {
	for (o = 0; o < (*paths); o++) {
	    if ((*name)[o] && strcmp(ftp, (*name)[o]) == 0) {
		/* Already saved */
		return;
	    }
	}
    } else {
	for (o = 0; o < (*paths); o++) {
	    if (!(*name)[o]) {
		/* Already saved */
		return;
	    }
	}
    }

    (*name) = realloc((*name), ((*paths) + 1) * sizeof(char *));
    (*path) = realloc((*path), ((*paths) + 1) * sizeof(char *));
    (*name)[(*paths)] = ftp ? alloc_strcpy(ftp) : NULL;
    (*path)[(*paths)] = NULL;
    pftp_get_path_dir(ftp, &(*path)[(*paths)]);    
    (*paths)++;
}

int pftp_begin_trans(char *args, pftp_ExistingFilefunc_t dialog, 
		     pftp_que_t fileque, pftp_que_t errorque)
{
    pftp_item_t item = NULL;
    char **origpath = NULL, **origpathname = NULL;
    size_t origpaths = 0, que_len = 0, que_pos = 0, que_err = 0;
    int err, ow = 0, all = 0, _error = 0;
    uint64_t filesize;
    clear_cache_t clear_cache;

    memset(&clear_cache, 0, sizeof(clear_cache_t));

    que_len = pftp_queLength(fileque);
    util_comm_central(PFTPUTIL_INITQUE, NULL, NULL, (pftp_param_t)&que_len, 
		      NULL);
    
    while ((item = pftp_get_next(fileque))) {
	if (util_comm_central(PFTPUTIL_QUE, NULL, NULL, (pftp_param_t)&que_pos,
			      (pftp_param_t)&que_err) == 0) {
	    pftp_que_add_first(item, fileque);
	    _error = 1;
	    break;
	}
	que_pos++;

	add_to_origpath(item->ftp_to, &origpathname, &origpath, &origpaths);
	add_to_origpath(item->ftp_from, &origpathname, &origpath, &origpaths);

	/* New for each file */
	filesize = 0;
	err = change_directories(item->ftp_to, item->destpath,
				 item->ftp_from, item->srcpath);

	if (!err) {
	    if (item->dir) {
		err = directory_trans(item, fileque);
		que_len = pftp_queLength(fileque);
		util_comm_central(PFTPUTIL_UPDATEQUE, NULL, NULL,
				  (pftp_param_t)&que_len, NULL);
	    } else {
		/* Check file transfer if no all option has been
		   selected */
		if (!all) {
		    all = checking_trans(&item, dialog, &filesize, &ow,
					 fileque, errorque);
		}
		if (!(ow & PFTP_DLG_ABORT)) {
		    non_checking_trans(&err, &clear_cache, item, ow, filesize);
		}
	    }
	}	   
	
	if (err) {
	    if (item->target) {
		util_error("%s -> %s failed.", item->filename, item->target);
	    } else {
		util_error("%s failed.", item->filename);
	    }
	    pftp_que_add_last(item, errorque);
	    que_err++;
	} else {
	    pftp_free_queitem(&item); 
	}
    }

    util_comm_central(PFTPUTIL_DONEQUE, NULL, NULL, 
		      (pftp_param_t)_error, NULL);

    //Change dirs to original directories
    if (origpaths) {
	size_t o;

	for (o = 0; o < origpaths; o++) {
	    trans_cd(origpathname[o], origpath[o]);
	    if (origpathname[o]) free(origpathname[o]);
	    if (origpath[o]) free(origpath[o]);
	}

	free(origpathname);
	free(origpath);
    }

    clear_and_free_cache(&clear_cache);
	
    return 0;
}

static void get_username(uid_t uid, char **name)
{
    struct passwd *data = getpwuid(uid);
    
    if (data) {
	(*name) = realloc_strcpy((*name), data->pw_name);
    } else {
	(*name) = realloc_strcpy((*name), "unknown");
    }
}

static void get_groupname(gid_t gid, char **name)
{
    struct group *data = getgrgid(gid);
    
    if (data) {
	(*name) = realloc_strcpy((*name), data->gr_name);
    } else {
	(*name) = realloc_strcpy((*name), "unknown");
    }
}

static pftp_server_t _getFTP(const char *name)
{
    pftp_server_t ret = NULL;
    if (util_comm_central(PFTPUTIL_NEEDCURFTP, NULL, NULL, (pftp_param_t)name, 
		     (pftp_param_t)&ret))
	return NULL;
    return ret;
}

void util_error(const char *format, ...)
{
    char *buf = malloc(1024);
    va_list list;

    va_start(list, format);

    vsnprintf(buf, 1024, format, list);

    util_comm_central(PFTPUTIL_ERROR, NULL, NULL, (pftp_param_t)buf, NULL);
    free(buf);

    va_end(list);
}
