/*
 * Copyright (C) 1995-2011 University of Karlsruhe.  All right reserved.
 *
 * This file is part of libFirm.
 *
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file LICENSE.GPL included in the
 * packaging of this file.
 *
 * Licensees holding valid libFirm Professional Edition licenses may use
 * this file in accordance with the libFirm Commercial License.
 * Agreement provided with the Software.
 *
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */

/**
 * @file
 * @brief    Memory disambiguator
 * @author   Michael Beck
 * @date     27.12.2006
 */
#include "config.h"

#include <stdlib.h>
#include <stdbool.h>

#include "adt/pmap.h"
#include "irnode_t.h"
#include "irgraph_t.h"
#include "irprog_t.h"
#include "irmemory_t.h"
#include "irmemory.h"
#include "irflag.h"
#include "hashptr.h"
#include "irflag.h"
#include "irouts.h"
#include "irgwalk.h"
#include "irprintf.h"
#include "debug.h"
#include "error.h"
#include "typerep.h"
#include "irpass.h"

/** The debug handle. */
DEBUG_ONLY(static firm_dbg_module_t *dbg = NULL;)
DEBUG_ONLY(static firm_dbg_module_t *dbgcall = NULL;)

/** The source language specific language disambiguator function. */
static DISAMBIGUATOR_FUNC language_disambuigator = NULL;

/** The global memory disambiguator options. */
static unsigned global_mem_disamgig_opt = aa_opt_no_opt;

const char *get_ir_alias_relation_name(ir_alias_relation rel)
{
#define X(a) case a: return #a
	switch (rel) {
	X(ir_no_alias);
	X(ir_may_alias);
	X(ir_sure_alias);
	default:
		panic("UNKNOWN alias relation");
	}
#undef X
}

unsigned get_irg_memory_disambiguator_options(const ir_graph *irg)
{
	unsigned opt = irg->mem_disambig_opt;
	if (opt & aa_opt_inherited)
		return global_mem_disamgig_opt;
	return opt;
}

void set_irg_memory_disambiguator_options(ir_graph *irg, unsigned options)
{
	irg->mem_disambig_opt = options & ~aa_opt_inherited;
}

void set_irp_memory_disambiguator_options(unsigned options)
{
	global_mem_disamgig_opt = options;
}

ir_storage_class_class_t get_base_sc(ir_storage_class_class_t x)
{
	return x & ~ir_sc_modifiers;
}

/**
 * Find the base address and entity of an Sel node.
 *
 * @param sel  the node
 * @param pEnt after return points to the base entity.
 *
 * @return the base address.
 */
static ir_node *find_base_adr(const ir_node *sel, ir_entity **pEnt)
{
	ir_node *ptr = get_Sel_ptr(sel);

	while (is_Sel(ptr)) {
		sel = ptr;
		ptr = get_Sel_ptr(sel);
	}
	*pEnt = get_Sel_entity(sel);
	return ptr;
}

/**
 * Check if a given Const node is greater or equal a given size.
 *
 * @param cns   a Const node
 * @param size  a integer size
 *
 * @return ir_no_alias if the Const is greater, ir_may_alias else
 */
static ir_alias_relation check_const(const ir_node *cns, int size)
{
	ir_tarval *tv = get_Const_tarval(cns);
	ir_tarval *tv_size;

	if (size == 0)
		return tarval_is_null(tv) ? ir_may_alias : ir_no_alias;
	tv_size = new_tarval_from_long(size, get_tarval_mode(tv));
	return tarval_cmp(tv_size, tv) & (ir_relation_less_equal) ? ir_no_alias : ir_may_alias;
}

/**
 * Treat idx1 and idx2 as integer indexes and check if they differ always more than size.
 *
 * @param idx1  a node representing the first index
 * @param idx2  a node representing the second index
 * @param size  an integer size
 *
 * @return ir_sure_alias iff idx1 == idx2
 *         ir_no_alias iff they ALWAYS differ more than size
 *         ir_may_alias else
 */
static ir_alias_relation different_index(const ir_node *idx1, const ir_node *idx2, int size)
{
	if (idx1 == idx2)
		return ir_sure_alias;
	if (is_Const(idx1) && is_Const(idx2)) {
		/* both are const, we can compare them */
		ir_tarval *tv1 = get_Const_tarval(idx1);
		ir_tarval *tv2 = get_Const_tarval(idx2);
		ir_tarval *tv, *tv_size;
		ir_mode *m1, *m2;

		if (size == 0)
			return tv1 == tv2 ? ir_sure_alias : ir_no_alias;

		/* arg, modes may be different */
		m1 = get_tarval_mode(tv1);
		m2 = get_tarval_mode(tv2);
		if (m1 != m2) {
			int size = get_mode_size_bits(m1) - get_mode_size_bits(m2);

			if (size < 0) {
				/* m1 is a small mode, cast up */
				m1 = mode_is_signed(m1) ? find_signed_mode(m2) : find_unsigned_mode(m2);
				if (m1 == NULL) {
					/* should NOT happen, but if it does we give up here */
					return ir_may_alias;
				}
				tv1 = tarval_convert_to(tv1, m1);
			} else if (size > 0) {
				/* m2 is a small mode, cast up */
				m2 = mode_is_signed(m2) ? find_signed_mode(m1) : find_unsigned_mode(m1);
				if (m2 == NULL) {
					/* should NOT happen, but if it does we give up here */
					return ir_may_alias;
				}
				tv2 = tarval_convert_to(tv2, m2);
			}
			/* here the size should be identical, check for signed */
			if (get_mode_sign(m1) != get_mode_sign(m2)) {
				/* find the signed */
				if (mode_is_signed(m2)) {
					ir_tarval *t = tv1;
					ir_mode *tm = m1;
					tv1 = tv2; m1 = m2;
					tv2 = t;   m2 = tm;
				}

				/* m1 is now the signed one */
				if (!tarval_is_negative(tv1)) {
					/* tv1 is signed, but >= 0, simply cast into unsigned */
					tv1 = tarval_convert_to(tv1, m2);
				} else {
					tv_size = new_tarval_from_long(size, m2);

					if (tarval_cmp(tv2, tv_size) & (ir_relation_greater_equal)) {
						/* tv1 is negative and tv2 >= tv_size, so the difference is bigger than size */
						return ir_no_alias;
					}
					/* tv_size > tv2, so we can subtract without overflow */
					tv2 = tarval_sub(tv_size, tv2, NULL);

					/* tv1 is < 0, so we can negate it */
					tv1 = tarval_neg(tv1);

					/* cast it into unsigned. for two-complement it does the right thing for MIN_INT */
					tv1 = tarval_convert_to(tv1, m2);

					/* now we can compare without overflow */
					return tarval_cmp(tv1, tv2) & (ir_relation_greater_equal) ? ir_no_alias : ir_may_alias;
				}
			}
		}
		if (tarval_cmp(tv1, tv2) == ir_relation_greater) {
			ir_tarval *t = tv1;
			tv1 = tv2;
			tv2 = t;
		}
		/* tv1 is now the "smaller" one */
		tv      = tarval_sub(tv2, tv1, NULL);
		tv_size = new_tarval_from_long(size, get_tarval_mode(tv));
		return tarval_cmp(tv_size, tv) & (ir_relation_less_equal) ? ir_no_alias : ir_may_alias;
	}

	/* Note: we rely here on the fact that normalization puts constants on the RIGHT side */
	if (is_Add(idx1)) {
		ir_node *l1 = get_Add_left(idx1);
		ir_node *r1 = get_Add_right(idx1);

		if (l1 == idx2) {
			/* x + c == y */
			if (is_Const(r1))
				return check_const(r1, size);
		}
		if (is_Add(idx2)) {
			/* both are Adds, check if they are of x + a == x + b kind */
			ir_node *l2 = get_Add_left(idx2);
			ir_node *r2 = get_Add_right(idx2);

			if (l1 == l2)
				return different_index(r1, r2, size);
			else if (l1 == r2)
				return different_index(r1, l2, size);
			else if (r1 == r2)
				return different_index(l1, l2, size);
			else if (r1 == l2)
				return different_index(l1, r2, size);
		}
	}
	if (is_Add(idx2)) {
		ir_node *l2 = get_Add_left(idx2);
		ir_node *r2 = get_Add_right(idx2);

		if (l2 == idx1) {
			/* x + c == y */
			if (is_Const(r2))
				return check_const(r2, size);
		}
	}

	if (is_Sub(idx1)) {
		ir_node *l1 = get_Sub_left(idx1);
		ir_node *r1 = get_Sub_right(idx1);

		if (l1 == idx2) {
			/* x - c == y */
			if (is_Const(r1))
				return check_const(r1, size);
		}

		if (is_Sub(idx2)) {
			/* both are Subs, check if they are of x - a == x - b kind */
			ir_node *l2 = get_Sub_left(idx2);

			if (l1 == l2) {
				ir_node *r2 = get_Sub_right(idx2);
				return different_index(r1, r2, size);
			}
		}
	}
	if (is_Sub(idx2)) {
		ir_node *l2 = get_Sub_left(idx2);
		ir_node *r2 = get_Sub_right(idx2);

		if (l2 == idx1) {
			/* x - c == y */
			if (is_Const(r2))
				return check_const(r2, size);
		}

	}
	return ir_may_alias;
}

/**
 * Two Sel addresses have the same base address, check if their offsets are
 * different.
 *
 * @param adr1  The first address.
 * @param adr2  The second address.
 */
static ir_alias_relation different_sel_offsets(const ir_node *sel1, const ir_node *sel2)
{
	/* seems to be broken */
	(void) sel1;
	(void) sel2;
#if 0
	ir_entity *ent1 = get_Sel_entity(sel1);
	ir_entity *ent2 = get_Sel_entity(sel2);
	int i, check_arr = 0;

	if (ent1 == ent2)
		check_arr = 1;
	else {
		ir_type *tp1 = get_entity_type(ent1);
		ir_type *tp2 = get_entity_type(ent2);

		if (tp1 == tp2)
			check_arr = 1;
		else if (get_type_state(tp1) == layout_fixed && get_type_state(tp2) == layout_fixed &&
		         get_type_size_bits(tp1) == get_type_size_bits(tp2))
			check_arr = 1;
	}
	if (check_arr) {
		/* we select an entity of same size, check for indexes */
		int n = get_Sel_n_indexs(sel1);
		int have_no = 0;

		if (n > 0 && n == get_Sel_n_indexs(sel2)) {
			/* same non-zero number of indexes, an array access, check */
			for (i = 0; i < n; ++i) {
				ir_node *idx1 = get_Sel_index(sel1, i);
				ir_node *idx2 = get_Sel_index(sel2, i);
				ir_alias_relation res = different_index(idx1, idx2, 0); /* we can safely IGNORE the size here if it's at least >0 */

				if (res == may_alias)
					return may_alias;
				else if (res == no_alias)
					have_no = 1;
			}
			/* if we have at least one no_alias, there is no alias relation, else we have sure */
			return have_no > 0 ? no_alias : sure_alias;
		}
	}
#else
	(void) different_index;
#endif
	return ir_may_alias;
}

/**
 * Determine the alias relation by checking if adr1 and adr2 are pointer
 * to different type.
 *
 * @param adr1    The first address.
 * @param adr2    The second address.
 */
static ir_alias_relation different_types(const ir_node *adr1, const ir_node *adr2)
{
	ir_entity *ent1 = NULL, *ent2 = NULL;

	if (is_SymConst_addr_ent(adr1))
		ent1 = get_SymConst_entity(adr1);
	else if (is_Sel(adr1))
		ent1 = get_Sel_entity(adr1);

	if (is_SymConst_addr_ent(adr2))
		ent2 = get_SymConst_entity(adr2);
	else if (is_Sel(adr2))
		ent2 = get_Sel_entity(adr2);

	if (ent1 != NULL && ent2 != NULL) {
		ir_type *tp1 = get_entity_type(ent1);
		ir_type *tp2 = get_entity_type(ent2);

		if (tp1 != tp2) {
			/* do deref until no pointer types are found */
			while (is_Pointer_type(tp1) && is_Pointer_type(tp2)) {
				tp1 = get_pointer_points_to_type(tp1);
				tp2 = get_pointer_points_to_type(tp2);
			}

			if (get_type_tpop(tp1) != get_type_tpop(tp2)) {
				/* different type structure */
				return ir_no_alias;
			}
			if (is_Class_type(tp1)) {
				/* check class hierarchy */
				if (! is_SubClass_of(tp1, tp2) &&
					! is_SubClass_of(tp2, tp1))
					return ir_no_alias;
			} else {
				/* different types */
				return ir_no_alias;
			}
		}
	}
	return ir_may_alias;
}

/**
 * Returns non-zero if a node is a result on a malloc-like routine.
 *
 * @param node  the Proj node to test
 */
static int is_malloc_Result(const ir_node *node)
{
	node = get_Proj_pred(node);
	if (! is_Proj(node))
		return 0;
	node = get_Proj_pred(node);
	if (! is_Call(node))
		return 0;
	node = get_Call_ptr(node);
	if (is_SymConst_addr_ent(node)) {
		ir_entity *ent = get_SymConst_entity(node);

		if (get_entity_additional_properties(ent) & mtp_property_malloc)
			return 1;
		return 0;
	}
	return 0;
}

ir_storage_class_class_t classify_pointer(const ir_node *irn,
                                          const ir_entity *ent)
{
	ir_graph *irg = get_irn_irg(irn);
	ir_storage_class_class_t res = ir_sc_pointer;
	if (is_SymConst_addr_ent(irn)) {
		ir_entity *entity = get_SymConst_entity(irn);
		ir_type   *owner  = get_entity_owner(entity);
		res = owner == get_tls_type() ? ir_sc_tls : ir_sc_globalvar;
		if (! (get_entity_usage(entity) & ir_usage_address_taken))
			res |= ir_sc_modifier_nottaken;
	} else if (irn == get_irg_frame(irg)) {
		res = ir_sc_localvar;
		if (ent != NULL && !(get_entity_usage(ent) & ir_usage_address_taken))
			res |= ir_sc_modifier_nottaken;
	} else if (is_Proj(irn) && is_malloc_Result(irn)) {
		return ir_sc_malloced;
	} else if (is_Const(irn)) {
		return ir_sc_globaladdr;
	} else if (is_arg_Proj(irn)) {
		res |= ir_sc_modifier_argument;
	}

	return res;
}

/**
 * If adr represents a Bitfield Sel, skip it
 */
static const ir_node *skip_Bitfield_Sels(const ir_node *adr)
{
	if (is_Sel(adr)) {
		ir_entity *ent     = get_Sel_entity(adr);
		ir_type   *bf_type = get_entity_type(ent);

		/* is it a bitfield type? */
		if (is_Primitive_type(bf_type) && get_primitive_base_type(bf_type) != NULL)
			adr = get_Sel_ptr(adr);
	}
	return adr;
}

/**
 * Determine the alias relation between two addresses.
 *
 * @param addr1  pointer address of the first memory operation
 * @param mode1  the mode of the accessed data through addr1
 * @param addr2  pointer address of the second memory operation
 * @param mode2  the mode of the accessed data through addr2
 *
 * @return found memory relation
 */
static ir_alias_relation _get_alias_relation(
	const ir_node *adr1, const ir_mode *mode1,
	const ir_node *adr2, const ir_mode *mode2)
{
	ir_entity             *ent1, *ent2;
	unsigned              options;
	long                  offset1 = 0;
	long                  offset2 = 0;
	const ir_node         *base1;
	const ir_node         *base2;
	const ir_node         *orig_adr1 = adr1;
	const ir_node         *orig_adr2 = adr2;
	ir_graph              *irg;
	unsigned              mode_size;
	ir_storage_class_class_t class1, class2, mod1, mod2;
	int                   have_const_offsets;

	if (! get_opt_alias_analysis())
		return ir_may_alias;

	if (adr1 == adr2)
		return ir_sure_alias;

	irg = get_irn_irg(adr1);
	options = get_irg_memory_disambiguator_options(irg);

	/* The Armageddon switch */
	if (options & aa_opt_no_alias)
		return ir_no_alias;

	/* do the addresses have constants offsets?
	 *  Note: nodes are normalized to have constants at right inputs,
	 *        sub X, C is normalized to add X, -C
	 */
	have_const_offsets = 1;
	while (is_Add(adr1)) {
		ir_node *add_right = get_Add_right(adr1);
		if (is_Const(add_right) && !mode_is_reference(get_irn_mode(add_right))) {
			ir_tarval *tv  = get_Const_tarval(add_right);
			offset1    += get_tarval_long(tv);
			adr1        = get_Add_left(adr1);
		} else if (mode_is_reference(get_irn_mode(add_right))) {
			adr1 = add_right;
			have_const_offsets = 0;
		} else {
			adr1 = get_Add_left(adr1);
			have_const_offsets = 0;
		}
	}
	while (is_Add(adr2)) {
		ir_node *add_right = get_Add_right(adr2);
		if (is_Const(add_right) && !mode_is_reference(get_irn_mode(add_right))) {
			ir_tarval *tv  = get_Const_tarval(add_right);
			offset2    += get_tarval_long(tv);
			adr2        = get_Add_left(adr2);
		} else if (mode_is_reference(get_irn_mode(add_right))) {
			adr2 = add_right;
			have_const_offsets = 0;
		} else {
			adr2 = get_Add_left(adr2);
			have_const_offsets = 0;
		}
	}

	mode_size = get_mode_size_bytes(mode1);
	if (get_mode_size_bytes(mode2) > mode_size) {
		mode_size = get_mode_size_bytes(mode2);
	}

	/* same base address -> compare offsets if possible.
	 * FIXME: type long is not sufficient for this task ...
	 */
	if (adr1 == adr2 && have_const_offsets) {
		if ((unsigned long)labs(offset2 - offset1) >= mode_size)
			return ir_no_alias;
		else
			return ir_sure_alias;
	}

	/*
	 * Bitfields can be constructed as Sels from its base address.
	 * As they have different entities, the disambiguator would find that they are
	 * alias free. While this is true for its values, it is false for the addresses
	 * (strictly speaking, the Sel's are NOT the addresses of the bitfields).
	 * So, skip those bitfield selecting Sel's.
	 */
	adr1 = skip_Bitfield_Sels(adr1);
	adr2 = skip_Bitfield_Sels(adr2);

	/* skip Sels */
	base1 = adr1;
	base2 = adr2;
	ent1  = NULL;
	ent2  = NULL;
	if (is_Sel(adr1)) {
		base1 = find_base_adr(adr1, &ent1);
	}
	if (is_Sel(adr2)) {
		base2 = find_base_adr(adr2, &ent2);
	}

	/* same base address -> compare Sel entities */
	if (base1 == base2 && ent1 != NULL && ent2 != NULL) {
		if (ent1 != ent2)
			return ir_no_alias;
		else if (have_const_offsets)
			return different_sel_offsets(adr1, adr2);
	}

	mod1 = classify_pointer(base1, ent1);
	mod2 = classify_pointer(base2, ent2);

	class1 = get_base_sc(mod1);
	class2 = get_base_sc(mod2);

	/* struct-access cannot alias with variables */
	if (ent1 == NULL && ent2 != NULL && is_compound_type(get_entity_owner(ent2))
		&& (class1 == ir_sc_globalvar || class1 == ir_sc_localvar || class1 == ir_sc_tls || class1 == ir_sc_globaladdr)) {
		return ir_no_alias;
	}
	if (ent2 == NULL && ent1 != NULL && is_compound_type(get_entity_owner(ent1))
		&& (class2 == ir_sc_globalvar || class2 == ir_sc_localvar || class2 == ir_sc_tls || class2 == ir_sc_globaladdr)) {
		return ir_no_alias;
	}

	if (class1 == ir_sc_pointer || class2 == ir_sc_pointer) {
		/* swap pointer class to class1 */
		if (class2 == ir_sc_pointer) {
			ir_storage_class_class_t temp = mod1;
			mod1 = mod2;
			mod2 = temp;
			class1 = get_base_sc(mod1);
			class2 = get_base_sc(mod2);
		}
		/* a pointer and an object whose address was never taken */
		if (mod2 & ir_sc_modifier_nottaken) {
			return ir_no_alias;
		}
		if (mod1 & ir_sc_modifier_argument) {
			if ( (options & aa_opt_no_alias_args)
					&& (mod2 & ir_sc_modifier_argument))
				return ir_no_alias;
			if ( (options & aa_opt_no_alias_args_global)
					&& (class2 == ir_sc_globalvar
						|| class2 == ir_sc_tls
						|| class2 == ir_sc_globaladdr))
				return ir_no_alias;
		}
	} else if (class1 != class2) {
		/* two objects from different memory spaces */
		return ir_no_alias;
	} else {
		/* both classes are equal */
		if (class1 == ir_sc_globalvar) {
			ir_entity *entity1 = get_SymConst_entity(base1);
			ir_entity *entity2 = get_SymConst_entity(base2);
			if (entity1 != entity2)
				return ir_no_alias;

			/* for some reason CSE didn't happen yet for the 2 SymConsts... */
			return ir_may_alias;
		} else if (class1 == ir_sc_globaladdr) {
			ir_tarval *tv = get_Const_tarval(base1);
			offset1      += get_tarval_long(tv);
			tv            = get_Const_tarval(base2);
			offset2      += get_tarval_long(tv);

			if ((unsigned long)labs(offset2 - offset1) >= mode_size)
				return ir_no_alias;
			else
				return ir_sure_alias;
		}
	}

	/* Type based alias analysis */
	if (options & aa_opt_type_based) {
		ir_alias_relation rel;

		if (options & aa_opt_byte_type_may_alias) {
			if (get_mode_size_bits(mode1) == 8 || get_mode_size_bits(mode2) == 8) {
				/* One of the modes address a byte. Assume a ir_may_alias and leave
				   the type based check. */
				goto leave_type_based_alias;
			}
		}
		/* cheap check: If the mode sizes did not match, the types MUST be different */
		if (get_mode_size_bits(mode1) != get_mode_size_bits(mode2))
			return ir_no_alias;

		/* cheap test: if only one is a reference mode, no alias */
		if (mode_is_reference(mode1) != mode_is_reference(mode2))
			return ir_no_alias;

		/* cheap test: if arithmetic is different, no alias */
		if (get_mode_arithmetic(mode1) != get_mode_arithmetic(mode2))
			return ir_no_alias;

		/* try rule R5 */
		rel = different_types(orig_adr1, orig_adr2);
		if (rel != ir_may_alias)
			return rel;
leave_type_based_alias:;
	}

	/* do we have a language specific memory disambiguator? */
	if (language_disambuigator != NULL) {
		ir_alias_relation rel = language_disambuigator(orig_adr1, mode1, orig_adr2, mode2);
		if (rel != ir_may_alias)
			return rel;
	}

	/* access points-to information here */
	return ir_may_alias;
}

ir_alias_relation get_alias_relation(
	const ir_node *adr1, const ir_mode *mode1,
	const ir_node *adr2, const ir_mode *mode2)
{
	ir_alias_relation rel = _get_alias_relation(adr1, mode1, adr2, mode2);
	DB((dbg, LEVEL_1, "alias(%+F, %+F) = %s\n", adr1, adr2, get_ir_alias_relation_name(rel)));
	return rel;
}

void set_language_memory_disambiguator(DISAMBIGUATOR_FUNC func)
{
	language_disambuigator = func;
}

/** The result cache for the memory disambiguator. */
static set *result_cache = NULL;

/** An entry in the relation cache. */
typedef struct mem_disambig_entry {
	const ir_node     *adr1;    /**< The first address. */
	const ir_mode     *mode1;   /**< The first address mode. */
	const ir_node     *adr2;    /**< The second address. */
	const ir_mode     *mode2;   /**< The second address mode. */
	ir_alias_relation result;   /**< The alias relation result. */
} mem_disambig_entry;

#define HASH_ENTRY(adr1, adr2)  (hash_ptr(adr1) ^ hash_ptr(adr2))

/**
 * Compare two relation cache entries.
 */
static int cmp_mem_disambig_entry(const void *elt, const void *key, size_t size)
{
	const mem_disambig_entry *p1 = (const mem_disambig_entry*) elt;
	const mem_disambig_entry *p2 = (const mem_disambig_entry*) key;
	(void) size;

	return p1->adr1 == p2->adr1 && p1->adr2 == p2->adr2 &&
	       p1->mode1 == p2->mode1 && p1->mode2 == p2->mode2;
}

void mem_disambig_init(void)
{
	result_cache = new_set(cmp_mem_disambig_entry, 8);
}

ir_alias_relation get_alias_relation_ex(
	const ir_node *adr1, const ir_mode *mode1,
	const ir_node *adr2, const ir_mode *mode2)
{
	mem_disambig_entry key, *entry;

	ir_fprintf(stderr, "%+F <-> %+F\n", adr1, adr2);

	if (! get_opt_alias_analysis())
		return ir_may_alias;

	if (get_irn_opcode(adr1) > get_irn_opcode(adr2)) {
		const ir_node *t = adr1;
		adr1 = adr2;
		adr2 = t;
	}

	key.adr1  = adr1;
	key.adr2  = adr2;
	key.mode1 = mode1;
	key.mode2 = mode2;
	entry = set_find(mem_disambig_entry, result_cache, &key, sizeof(key), HASH_ENTRY(adr1, adr2));
	if (entry != NULL)
		return entry->result;

	key.result = get_alias_relation(adr1, mode1, adr2, mode2);

	(void)set_insert(mem_disambig_entry, result_cache, &key, sizeof(key), HASH_ENTRY(adr1, adr2));
	return key.result;
}

void mem_disambig_term(void)
{
	if (result_cache != NULL) {
		del_set(result_cache);
		result_cache = NULL;
	}
}

/**
 * Check the mode of a Load/Store with the mode of the entity
 * that is accessed.
 * If the mode of the entity and the Load/Store mode do not match, we
 * have the bad reinterpret case:
 *
 * int i;
 * char b = *(char *)&i;
 *
 * We do NOT count this as one value and return address_taken
 * in that case.
 * However, we support an often used case. If the mode is two-complement
 * we allow casts between signed/unsigned.
 *
 * @param mode     the mode of the Load/Store
 * @param ent_mode the mode of the accessed entity
 *
 * @return non-zero if the Load/Store is a hidden cast, zero else
 */
static int is_hidden_cast(const ir_mode *mode, const ir_mode *ent_mode)
{
	if (ent_mode == NULL)
		return false;

	if (ent_mode != mode) {
		if (ent_mode == NULL ||
			get_mode_size_bits(ent_mode) != get_mode_size_bits(mode) ||
			get_mode_arithmetic(ent_mode) != irma_twos_complement ||
			get_mode_arithmetic(mode) != irma_twos_complement)
			return true;
	}
	return false;
}

/**
 * Determine the usage state of a node (or its successor Sels).
 *
 * @param irn  the node
 */
static ir_entity_usage determine_entity_usage(const ir_node *irn, ir_entity *entity)
{
	int       i;
	ir_mode   *emode, *mode;
	ir_node   *value;
	ir_type   *tp;
	unsigned   res = 0;

	for (i = get_irn_n_outs(irn) - 1; i >= 0; --i) {
		ir_node *succ = get_irn_out(irn, i);

		switch (get_irn_opcode(succ)) {
		case iro_Load:
			/* beware: irn might be a Id node here, so irn might be not
			   equal to get_Load_ptr(succ) */
			res |= ir_usage_read;

			/* check if this load is not a hidden conversion */
			mode  = get_Load_mode(succ);
			emode = get_type_mode(get_entity_type(entity));
			if (is_hidden_cast(mode, emode))
				res |= ir_usage_reinterpret_cast;
			break;

		case iro_Store:
			/* check that the node is not the Store's value */
			if (irn == get_Store_value(succ)) {
				res |= ir_usage_unknown;
			}
			if (irn == get_Store_ptr(succ)) {
				res |= ir_usage_write;

				/* check if this Store is not a hidden conversion */
				value = get_Store_value(succ);
				mode  = get_irn_mode(value);
				emode = get_type_mode(get_entity_type(entity));
				if (is_hidden_cast(mode, emode))
					res |= ir_usage_reinterpret_cast;
			}
			assert(irn != get_Store_mem(succ));
			break;

		case iro_CopyB:
			/* CopyB are like Loads/Stores */
			tp  = get_entity_type(entity);
			if (tp != get_CopyB_type(succ)) {
				/* bad, different types, might be a hidden conversion */
				res |= ir_usage_reinterpret_cast;
			}
			if (irn == get_CopyB_dst(succ)) {
				res |= ir_usage_write;
			} else {
				assert(irn == get_CopyB_src(succ));
				res |= ir_usage_read;
			}
			break;

		case iro_Add:
		case iro_Sub:
			/* Check the successor of irn. */
			res |= determine_entity_usage(succ, entity);
			break;
		case iro_Sel: {
			ir_entity *sel_entity = get_Sel_entity(succ);
			/* this analysis can't handle unions correctly */
			if (is_Union_type(get_entity_owner(sel_entity))) {
				res |= ir_usage_unknown;
				break;
			}
			/* Check the successor of irn. */
			res |= determine_entity_usage(succ, sel_entity);
			break;
		}

		case iro_Call:
			if (irn == get_Call_ptr(succ)) {
				/* TODO: we could check for reinterpret casts here...
				 * But I doubt anyone is interested in that bit for
				 * function entities and I'm too lazy to write the code now.
				 */
				res |= ir_usage_read;
			} else {
				assert(irn != get_Call_mem(succ));
				res |= ir_usage_unknown;
			}
			break;

		/* skip identities */
		case iro_Id:
			res |= determine_entity_usage(succ, entity);
			break;

		/* skip tuples */
		case iro_Tuple: {
			int input_nr;
			for (input_nr = get_Tuple_n_preds(succ) - 1; input_nr >= 0;
					--input_nr) {
				ir_node *pred = get_Tuple_pred(succ, input_nr);
				if (pred == irn) {
					int k;
					/* we found one input */
					for (k = get_irn_n_outs(succ) - 1; k >= 0; --k) {
						ir_node *proj = get_irn_out(succ, k);

						if (is_Proj(proj) && get_Proj_proj(proj) == input_nr) {
							res |= determine_entity_usage(proj, entity);
							break;
						}
					}
				}
			}
			break;
		}

		default:
			/* another op, we don't know anything (we could do more advanced
			 * things like a dataflow analysis here) */
			res |= ir_usage_unknown;
			break;
		}
	}

	return (ir_entity_usage) res;
}

/**
 * Update the usage flags of all frame entities.
 */
static void analyse_irg_entity_usage(ir_graph *irg)
{
	ir_type *ft = get_irg_frame_type(irg);
	ir_node *irg_frame;
	size_t i, n;
	int j, k, static_link_arg;

	assure_irg_properties(irg, IR_GRAPH_PROPERTY_CONSISTENT_OUTS);

	/* set initial state to not_taken, as this is the "smallest" state */
	for (i = 0, n = get_class_n_members(ft); i < n; ++i) {
		ir_entity *ent = get_class_member(ft, i);

		/* methods can only be analyzed globally */
		if (! is_method_entity(ent)) {
			ir_entity_usage flags = ir_usage_none;
			if (get_entity_linkage(ent) & IR_LINKAGE_HIDDEN_USER)
				flags = ir_usage_unknown;
			set_entity_usage(ent, flags);
		}
	}

	irg_frame = get_irg_frame(irg);

	for (j = get_irn_n_outs(irg_frame) - 1; j >= 0; --j) {
		ir_node        *succ = get_irn_out(irg_frame, j);
		ir_entity      *entity;
		unsigned        flags;

		if (!is_Sel(succ))
			continue;

		entity = get_Sel_entity(succ);
		flags  = get_entity_usage(entity);
		flags |= determine_entity_usage(succ, entity);
		set_entity_usage(entity, (ir_entity_usage) flags);
	}

	/* check inner functions accessing outer frame */
	static_link_arg = 0;
	for (i = 0, n = get_class_n_members(ft); i < n; ++i) {
		ir_entity *ent = get_class_member(ft, i);
		ir_graph  *inner_irg;
		ir_node   *args;

		if (! is_method_entity(ent))
			continue;

		inner_irg = get_entity_irg(ent);
		if (inner_irg == NULL)
			continue;

		assure_irg_outs(inner_irg);
		args = get_irg_args(inner_irg);
		for (j = get_irn_n_outs(args) - 1; j >= 0; --j) {
			ir_node *arg = get_irn_out(args, j);

			if (get_Proj_proj(arg) == static_link_arg) {
				for (k = get_irn_n_outs(arg) - 1; k >= 0; --k) {
					ir_node *succ = get_irn_out(arg, k);

					if (is_Sel(succ)) {
						ir_entity *entity = get_Sel_entity(succ);

						if (get_entity_owner(entity) == ft) {
							/* found an access to the outer frame */
							unsigned flags;

							flags  = get_entity_usage(entity);
							flags |= determine_entity_usage(succ, entity);
							set_entity_usage(entity, (ir_entity_usage) flags);
						}
					}
				}
			}
		}
	}

	/* now computed */
	add_irg_properties(irg, IR_GRAPH_PROPERTY_CONSISTENT_ENTITY_USAGE);
}

void assure_irg_entity_usage_computed(ir_graph *irg)
{
	if (irg_has_properties(irg, IR_GRAPH_PROPERTY_CONSISTENT_ENTITY_USAGE))
		return;

	analyse_irg_entity_usage(irg);
}


/**
 * Initialize the entity_usage flag for a global type like type.
 */
static void init_entity_usage(ir_type *tp)
{
	size_t i, n;

	/* We have to be conservative: All external visible entities are unknown */
	for (i = 0, n = get_compound_n_members(tp); i < n; ++i) {
		ir_entity       *ent   = get_compound_member(tp, i);
		unsigned         flags = ir_usage_none;

		if (entity_is_externally_visible(ent)) {
			flags |= ir_usage_unknown;
		}
		set_entity_usage(ent, (ir_entity_usage) flags);
	}
}

/**
 * Mark all entities used in the initializer as unknown usage.
 *
 * @param initializer  the initializer to check
 */
static void check_initializer_nodes(ir_initializer_t *initializer)
{
	size_t  i;
	ir_node *n;

	switch (initializer->kind) {
	case IR_INITIALIZER_CONST:
		/* let's check if it's an address */
		n = initializer->consti.value;
		if (is_SymConst_addr_ent(n)) {
			ir_entity *ent = get_SymConst_entity(n);
			set_entity_usage(ent, ir_usage_unknown);
		}
		return;
	case IR_INITIALIZER_TARVAL:
	case IR_INITIALIZER_NULL:
		return;
	case IR_INITIALIZER_COMPOUND:
		for (i = 0; i < initializer->compound.n_initializers; ++i) {
			ir_initializer_t *sub_initializer
				= initializer->compound.initializers[i];
			check_initializer_nodes(sub_initializer);
		}
		return;
	}
	panic("invalid initializer found");
}

/**
 * Mark all entities used in the initializer for the given entity as unknown
 * usage.
 *
 * @param ent  the entity
 */
static void check_initializer(ir_entity *ent)
{
	/* Beware: Methods are always initialized with "themself". This does not
	 * count as a taken address.
	 * TODO: this initialisation with "themself" is wrong and should be removed
	 */
	if (is_Method_type(get_entity_type(ent)))
		return;

	if (ent->initializer != NULL) {
		check_initializer_nodes(ent->initializer);
	}
}


/**
 * Mark all entities used in initializers as unknown usage.
 *
 * @param tp  a compound type
 */
static void check_initializers(ir_type *tp)
{
	size_t i, n;

	for (i = 0, n = get_compound_n_members(tp); i < n; ++i) {
		ir_entity *ent = get_compound_member(tp, i);

		check_initializer(ent);
	}
}

#ifdef DEBUG_libfirm
/**
 * Print the entity usage flags of all entities of a given type for debugging.
 *
 * @param tp  a compound type
 */
static void print_entity_usage_flags(const ir_type *tp)
{
	size_t i, n;
	for (i = 0, n = get_compound_n_members(tp); i < n; ++i) {
		ir_entity *ent = get_compound_member(tp, i);
		ir_entity_usage flags = get_entity_usage(ent);

		if (flags == 0)
			continue;
		ir_printf("%+F:", ent);
		if (flags & ir_usage_address_taken)
			printf(" address_taken");
		if (flags & ir_usage_read)
			printf(" read");
		if (flags & ir_usage_write)
			printf(" write");
		if (flags & ir_usage_reinterpret_cast)
			printf(" reinterp_cast");
		printf("\n");
	}
}
#endif /* DEBUG_libfirm */

/**
 * Post-walker: check for global entity address
 */
static void check_global_address(ir_node *irn, void *data)
{
	ir_entity *ent;
	unsigned flags;
	(void) data;

	if (is_SymConst_addr_ent(irn)) {
		/* A global. */
		ent = get_SymConst_entity(irn);
	} else
		return;

	flags = get_entity_usage(ent);
	flags |= determine_entity_usage(irn, ent);
	set_entity_usage(ent, (ir_entity_usage) flags);
}

/**
 * Update the entity usage flags of all global entities.
 */
static void analyse_irp_globals_entity_usage(void)
{
	size_t i, n;
	ir_segment_t s;

	for (s = IR_SEGMENT_FIRST; s <= IR_SEGMENT_LAST; ++s) {
		ir_type *type = get_segment_type(s);
		init_entity_usage(type);
	}

	for (s = IR_SEGMENT_FIRST; s <= IR_SEGMENT_LAST; ++s) {
		ir_type *type = get_segment_type(s);
		check_initializers(type);
	}

	for (i = 0, n = get_irp_n_irgs(); i < n; ++i) {
		ir_graph *irg = get_irp_irg(i);

		assure_irg_outs(irg);
		irg_walk_graph(irg, NULL, check_global_address, NULL);
	}

#ifdef DEBUG_libfirm
	if (firm_dbg_get_mask(dbg) & LEVEL_1) {
		for (s = IR_SEGMENT_FIRST; s <= IR_SEGMENT_LAST; ++s) {
			print_entity_usage_flags(get_segment_type(s));
		}
	}
#endif /* DEBUG_libfirm */

	/* now computed */
	irp->globals_entity_usage_state = ir_entity_usage_computed;
}

ir_entity_usage_computed_state get_irp_globals_entity_usage_state(void)
{
	return irp->globals_entity_usage_state;
}

void set_irp_globals_entity_usage_state(ir_entity_usage_computed_state state)
{
	irp->globals_entity_usage_state = state;
}

void assure_irp_globals_entity_usage_computed(void)
{
	if (irp->globals_entity_usage_state != ir_entity_usage_not_computed)
		return;

	analyse_irp_globals_entity_usage();
}

void firm_init_memory_disambiguator(void)
{
	FIRM_DBG_REGISTER(dbg, "firm.ana.irmemory");
	FIRM_DBG_REGISTER(dbgcall, "firm.opt.cc");
}


/** Maps method types to cloned method types. */
static pmap *mtp_map;

/**
 * Clone a method type if not already cloned.
 *
 * @param tp  the type to clone
 */
static ir_type *clone_type_and_cache(ir_type *tp)
{
	ir_type *res = pmap_get(ir_type, mtp_map, tp);

	if (res == NULL) {
		res = clone_type_method(tp);
		pmap_insert(mtp_map, tp, res);
	}

	return res;
}

/**
 * Walker: clone all call types of Calls to methods having the
 * mtp_property_private property set.
 */
static void update_calls_to_private(ir_node *call, void *env)
{
	(void) env;
	if (is_Call(call)) {
		ir_node *ptr = get_Call_ptr(call);

		if (is_SymConst(ptr)) {
			ir_entity *ent = get_SymConst_entity(ptr);
			ir_type *ctp = get_Call_type(call);

			if (get_entity_additional_properties(ent) & mtp_property_private) {
				if ((get_method_additional_properties(ctp) & mtp_property_private) == 0) {
					ctp = clone_type_and_cache(ctp);
					add_method_additional_properties(ctp, mtp_property_private);
					set_Call_type(call, ctp);
					DB((dbgcall, LEVEL_1, "changed call to private method %+F using cloned type %+F\n", ent, ctp));
				}
			}
		}
	}
}

void mark_private_methods(void)
{
	size_t i, n;
	int changed = 0;

	assure_irp_globals_entity_usage_computed();

	mtp_map = pmap_create();

	/* first step: change the calling conventions of the local non-escaped entities */
	for (i = 0, n = get_irp_n_irgs(); i < n; ++i) {
		ir_graph        *irg   = get_irp_irg(i);
		ir_entity       *ent   = get_irg_entity(irg);
		ir_entity_usage  flags = get_entity_usage(ent);

		if (!(flags & ir_usage_address_taken) && !entity_is_externally_visible(ent)) {
			ir_type *mtp = get_entity_type(ent);

			add_entity_additional_properties(ent, mtp_property_private);
			DB((dbgcall, LEVEL_1, "found private method %+F\n", ent));
			if ((get_method_additional_properties(mtp) & mtp_property_private) == 0) {
				/* need a new type */
				mtp = clone_type_and_cache(mtp);
				add_method_additional_properties(mtp, mtp_property_private);
				set_entity_type(ent, mtp);
				DB((dbgcall, LEVEL_2, "changed entity type of %+F to %+F\n", ent, mtp));
				changed = 1;
			}
		}
	}

	if (changed)
		all_irg_walk(NULL, update_calls_to_private, NULL);

	pmap_destroy(mtp_map);
}

ir_prog_pass_t *mark_private_methods_pass(const char *name)
{
	return def_prog_pass(name ? name : "mark_private_methods", mark_private_methods);
}
