/* LIBDS
 * =====
 * This software is Copyright (c) 2002-03 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */
#include "DSHashTable.h"

#include <stdio.h>

DSHashTable::DSHashTable(unsigned int newBuckets, unsigned int newPrimeA, unsigned int newPrimeB)
{
	init(newBuckets,newPrimeA,newPrimeB);
}

void DSHashTable::init(unsigned int newBuckets, unsigned int newPrimeA, unsigned int newPrimeB)
{
	unsigned int i;
	buckets=newBuckets;
	primeA=newPrimeA;
	primeB=newPrimeB;
	bucket=(DSList **)malloc(sizeof(DSList*)*buckets);
	for (i=0;i<buckets;i++) {
		bucket[i]=new DSList();
	}
}

DSHashTable::~DSHashTable()
{
	unsigned int i;
	close();
	for (i=0;i<getBuckets();i++) {
		delete bucket[i];
	}
	free(bucket);
}


void DSHashTable::close()
{
	unsigned int i;
	for (i=0;i<getBuckets();i++) {
		bucket[i]->close();
	}
}

unsigned int DSHashTable::getBuckets()
{
	return buckets;
}

DSList* DSHashTable::getList()
{
	DSList * retval;
	unsigned int i;
	DSList* l;
	DSListElement* le;
	DSListElement* lenew;
	retval=new DSList();

	for (i=0;i<getBuckets();i++) {
		l=getList(i);
		for (le=l->getHead();le!=NULL;le=le->getNext()) {
			lenew=new DSListElement(le);
			lenew->setCleanup(0);
			retval->append(lenew);
		}
	}
	return retval;

}

DSList* DSHashTable::getList(unsigned int i)
{
	return (bucket[i]);
}

void DSHashTable::delValueInt(unsigned int Key)
{
	delItem(Key);
}

void DSHashTable::delValueString(const char* Key)
{
	delItem(Key);
}

void DSHashTable::delItem(unsigned int Key)
{
	unsigned int bucket_num;
	bucket_num=hash(Key);
	bucket[bucket_num]->delItem(Key);
}

void DSHashTable::delItem(const char* Key)
{
	unsigned int bucket_num;
	bucket_num=hash((unsigned char *)Key);
	bucket[bucket_num]->delItem(Key);
}

DSListElement* DSHashTable::getElementInt(unsigned int Key)
{
	return getElement(Key);
}

DSListElement* DSHashTable::getElementString(const char* Key)
{
	return getElement(Key);
}

DSListElement* DSHashTable::getElement(unsigned int Key)
{
	unsigned int bucket_num;
	bucket_num=hash(Key);
	return(bucket[bucket_num]->getElement(Key));
}

DSListElement* DSHashTable::getElement(const char* Key)
{
	unsigned int bucket_num;
	bucket_num=hash((unsigned char *)Key);
	return(bucket[bucket_num]->getElement(Key));
}


void* DSHashTable::getValueString(const char* Key)
{
	return getPtrValue(Key);
}

void* DSHashTable::getValueInt(unsigned int Key)
{
	return getPtrValue(Key);
}

void* DSHashTable::getPtrValue(const char* Key)
{
	unsigned int bucket_num;
	bucket_num=hash((unsigned char *)Key);
	return(bucket[bucket_num]->getPtrValue(Key));
}

void* DSHashTable::getPtrValue(unsigned int Key)
{
	unsigned int bucket_num;
	bucket_num=hash(Key);
	return(bucket[bucket_num]->getPtrValue(Key));
}

unsigned int DSHashTable::getIntString(const char* Key)
{
	return getNumericValue(Key);
}

unsigned int DSHashTable::getIntInt(unsigned int Key)
{
	return getNumericValue(Key);
}

unsigned int DSHashTable::getNumericValue(const char* Key)
{
	unsigned int bucket_num;
	bucket_num=hash((unsigned char *)Key);
	return(bucket[bucket_num]->getNumericValue(Key));
}

unsigned int DSHashTable::getNumericValue(unsigned int Key)
{
	unsigned int bucket_num;
	bucket_num=hash(Key);
	return(bucket[bucket_num]->getNumericValue(Key));
}

unsigned int DSHashTable::hash(unsigned char * input_value)
{
	unsigned int temp;
	temp = primeA;
	while (input_value[0]!='\0') {
		temp ^= (*input_value + (temp<<5) + (temp>>2));
		input_value++;
	}
	return((temp%primeB)%getBuckets());
}

unsigned int DSHashTable::hash(unsigned int input_value)
{
	return(input_value%getBuckets());
}

void DSHashTable::insert(char * Key,void * value,int cleanup)
{
	unsigned int bucket_num;
	bucket_num=hash((unsigned char *)Key);
	bucket[bucket_num]->insert(Key,value,cleanup);
}

void DSHashTable::insert(unsigned int Key,void * value,int cleanup)
{
	unsigned int bucket_num;
	bucket_num=hash(Key);
	bucket[bucket_num]->insert(Key,value,cleanup);
}

void DSHashTable::insert(char * Key,unsigned int value,int cleanup)
{
	unsigned int bucket_num;
	bucket_num=hash((unsigned char *)Key);
	bucket[bucket_num]->insert(Key,value,cleanup);
}

void DSHashTable::insert(unsigned int Key,unsigned int value,int cleanup)
{
	unsigned int bucket_num;
	bucket_num=hash(Key);
	bucket[bucket_num]->insert(Key,value,cleanup);
}

unsigned int DSHashTable::occurenceCount(const char * key)
{
	unsigned int bucket_num;
	bucket_num=hash((unsigned char *)key);
	return(bucket[bucket_num]->occurenceCount(key));
}


void DSHashTable::print()
{
	unsigned int i;
	DSListElement* current;
	for (i=0;i<getBuckets();i++) {
		printf("Bucket %i\n",i);
		current=bucket[i]->getHead();
		while (current) {
			printf("  %s=%s\n",current->getKeyString(),(char *)current->getDataPtr());
			current=current->getNext();
		}
	}
	fflush(NULL);
}

/*
 * Callback driver.  Call this method if you want a callback function
 * executed for every entry in the specified section of the DSHashTable.
 * Takes in the section, callback routine, and custom data passed unchanged
 * into your callback routines.
 */
void DSHashTable::forEach(void (*callback)(char *,char *,void *), void * extra)
{
	unsigned int i;
	DSList* l;
	DSListElement* le;
	for (i=0;i<getBuckets();i++) {
		l=getList(i);
		for (le=l->getHead();le!=NULL;le=le->getNext())
			callback(le->getKeyString(),(char *)le->getDataPtr(),
					extra);
	}
}

BOOL DSHashTable::setValue(const char * Key, void * data)
{
	unsigned int bucket_num;
	bucket_num=hash((unsigned char *)Key);
	return(bucket[bucket_num]->setValue(Key,data));
}

BOOL DSHashTable::setValue(unsigned int Key, void * data)
{
	unsigned int bucket_num;
	bucket_num=hash(Key);
	return(bucket[bucket_num]->setValue(Key,data));
}

BOOL DSHashTable::setValue(const char * Key, unsigned int data)
{
	unsigned int bucket_num;
	bucket_num=hash((unsigned char *)Key);
	return(bucket[bucket_num]->setValue(Key,data));
}

BOOL DSHashTable::setValue(unsigned int Key, unsigned int data)
{
	unsigned int bucket_num;
	bucket_num=hash(Key);
	return(bucket[bucket_num]->setValue(Key,data));
}

