/*
 * Copyright (c) 1998-2011 Apple Inc. All rights reserved.
 *
 * @APPLE_OSREFERENCE_LICENSE_HEADER_START@
 * 
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. The rights granted to you under the License
 * may not be used to create, or enable the creation or redistribution of,
 * unlawful or unlicensed copies of an Apple operating system, or to
 * circumvent, violate, or enable the circumvention or violation of, any
 * terms of an Apple operating system software license agreement.
 * 
 * Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 * 
 * @APPLE_OSREFERENCE_LICENSE_HEADER_END@
 */

#include <libkern/c++/OSUnserialize.h>
#include <libkern/c++/OSKext.h>
#include <libkern/version.h>
#include <IOKit/IORegistryEntry.h>
#include <IOKit/IODeviceTreeSupport.h>
#include <IOKit/IOCatalogue.h>
#include <IOKit/IOUserClient.h>
#include <IOKit/IOMemoryDescriptor.h>
#include <IOKit/IOPlatformExpert.h>
#include <IOKit/IOLib.h>
#include <IOKit/IOKitKeys.h>
#include <IOKit/IOKitDebug.h>
#include <IOKit/pwr_mgt/IOPMinformeeList.h>

#include <IOKit/assert.h>

#include "IOKitKernelInternal.h"

extern "C" {

extern void OSlibkernInit (void);

void iokit_post_constructor_init(void) __attribute__((section("__TEXT, initcode")));

#include <kern/clock.h>
#include <sys/time.h>

void IOKitInitializeTime( void )
{
	mach_timespec_t		t;

	t.tv_sec = 30;
	t.tv_nsec = 0;
	IOService::waitForService(
		IOService::resourceMatching("IORTC"), &t );
#ifdef ppc
	IOService::waitForService(
		IOService::resourceMatching("IONVRAM"), &t );
#endif

    clock_initialize_calendar();
}

void IOKitResetTime( void )
{
    clock_sec_t		secs;
	clock_usec_t	microsecs;

    clock_initialize_calendar();

    clock_get_calendar_microtime(&secs, &microsecs);
    gIOLastWakeTime.tv_sec  = secs;
    gIOLastWakeTime.tv_usec = microsecs;
}

void iokit_post_constructor_init(void)
{
    IORegistryEntry *		root;
    OSObject *			obj;

    root = IORegistryEntry::initialize();
    assert( root );
    IOService::initialize();
    IOCatalogue::initialize();
    IOUserClient::initialize();
    IOMemoryDescriptor::initialize();

    // Initializes IOPMinformeeList class-wide shared lock
    IOPMinformeeList::getSharedRecursiveLock();

    obj = OSString::withCString( version );
    assert( obj );
    if( obj ) {
        root->setProperty( kIOKitBuildVersionKey, obj );
	obj->release();
    }
    obj = IOKitDiagnostics::diagnostics();
    if( obj ) {
        root->setProperty( kIOKitDiagnosticsKey, obj );
	obj->release();
    }

}

// From <osfmk/kern/debug.c>
extern int debug_mode;

/*****
 * Pointer into bootstrap KLD segment for functions never used past startup.
 */
void (*record_startup_extensions_function)(void) = 0;

void StartIOKit( void * p1, void * p2, void * p3, void * p4 )
{
    IOPlatformExpertDevice *	rootNub;
    int				debugFlags;
	uint32_t		intThreshold;

    if( PE_parse_boot_argn( "io", &debugFlags, sizeof (debugFlags) ))
	gIOKitDebug = debugFlags;

    if( PE_parse_boot_argn( "iotrace", &debugFlags, sizeof (debugFlags) ))
		gIOKitTrace = debugFlags;
	
	// Compat for boot-args
	gIOKitTrace |= (gIOKitDebug & kIOTraceCompatBootArgs);

    if( PE_parse_boot_argn( "iointthreshold", &intThreshold, sizeof (intThreshold) ))
		gIOInterruptThresholdNS = intThreshold * 1000;
	
    // Check for the log synchronous bit set in io
    if (gIOKitDebug & kIOLogSynchronous)
        debug_mode = true;
	
    //
    // Have to start IOKit environment before we attempt to start
    // the C++ runtime environment.  At some stage we have to clean up
    // the initialisation path so that OS C++ can initialise independantly
    // of iokit basic service initialisation, or better we have IOLib stuff
    // initialise as basic OS services.
    //
    IOLibInit(); 
    OSlibkernInit();

    rootNub = new IOPlatformExpertDevice;

    if( rootNub && rootNub->initWithArgs( p1, p2, p3, p4)) {
        rootNub->attach( 0 );

       /* If the bootstrap segment set up a function to record startup
        * extensions, call it now.
        */
        if (record_startup_extensions_function) {
            record_startup_extensions_function();
        }

        rootNub->registerService();

#if !NO_KEXTD
       /* Add a busy count to keep the registry busy until kextd has
        * completely finished launching. This is decremented when kextd
        * messages the kernel after the in-kernel linker has been
        * removed and personalities have been sent.
        */
        IOService::getServiceRoot()->adjustBusy(1);
#endif
    }
}

void
IORegistrySetOSBuildVersion(char * build_version)
{
    IORegistryEntry * root = IORegistryEntry::getRegistryRoot();

    if (root) {
        if (build_version) {
            root->setProperty(kOSBuildVersionKey, build_version);
        } else {
            root->removeProperty(kOSBuildVersionKey);
        }
    }

    return;
}

}; /* extern "C" */
