/*
 * Copyright (c) 2007 Apple Inc. All rights reserved.
 *
 * @APPLE_OSREFERENCE_LICENSE_HEADER_START@
 * 
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. The rights granted to you under the License
 * may not be used to create, or enable the creation or redistribution of,
 * unlawful or unlicensed copies of an Apple operating system, or to
 * circumvent, violate, or enable the circumvention or violation of, any
 * terms of an Apple operating system software license agreement.
 * 
 * Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 * 
 * @APPLE_OSREFERENCE_LICENSE_HEADER_END@
 */

/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */
/*
 * Copyright 2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

/*
 * #pragma ident	"@(#)dtrace_subr.c	1.12	05/06/08 SMI"
 */

#define MACH__POSIX_C_SOURCE_PRIVATE 1 /* pulls in suitable savearea from mach/ppc/thread_status.h */
#include <sys/dtrace.h>
#include <sys/dtrace_glue.h>
#include <sys/dtrace_impl.h>
#include <sys/fasttrap.h>
#include <sys/vm.h>
#include <sys/user.h>
#include <sys/kauth.h>
#include <kern/debug.h>

int (*dtrace_pid_probe_ptr)(ppc_saved_state_t *);
int (*dtrace_return_probe_ptr)(ppc_saved_state_t *);
kern_return_t dtrace_user_probe(ppc_saved_state_t *sv);

kern_return_t
dtrace_user_probe(ppc_saved_state_t *sv)
{

	lck_rw_t *rwp;
	struct proc *p = current_proc();

	uthread_t uthread = (uthread_t)get_bsdthread_info(current_thread());
	/*
	 * DTrace accesses t_cred in probe context.	 t_cred
	 * must always be either NULL, or point to a valid,
	 * allocated cred structure.
	 */
	kauth_cred_uthread_update(uthread, p);

	if (sv->save_exception == T_DTRACE_RET) {

/*
 *		T_DTRACE_RET is generated by the kernel when an emulation sequence 
 *		ends.  Unlike the x86 implementation, this can not be caused by
 *		a user state trap instruction.  It is a system error if it occurs
 *		when not stepping and is, therefore, a panickable offence.
 */

		if(uthread->t_dtrace_step == 0) {	/* Are we supposed to be tracing? */
			panic("dtrace_user_probe: T_DTRACE_RET when not stepping\n");
		}

		if (uthread->t_dtrace_ast) {
			printf("dtrace_user_probe() should be calling aston()\n");
			// aston(uthread);
			// uthread->t_sig_check = 1;
		}

		/*
		 * Clear all user tracing flags.
		 */
		uthread->t_dtrace_ft = 0;

		/*
		 * We need to wait until after we've called the
		 * dtrace_return_probe_ptr function pointer to step the pc.
		 */
		rwp = &CPU->cpu_ft_lock;
		lck_rw_lock_shared(rwp);

		if (dtrace_return_probe_ptr != NULL) (void)(*dtrace_return_probe_ptr)(sv);
		lck_rw_unlock_shared(rwp);

		sv->save_srr0 = sv->save_srr0 + 4;	/* Step to next instruction */
		if(!(sv->save_srr1 & 0x8000000000000000ULL)) sv->save_srr0 &= 0x00000000FFFFFFFF;	/* Trim if in 32-bit mode */

		return KERN_SUCCESS;
		
	} else {

/*
 *	We have taken our normal trap to get here.  Make sure we expect it
 */
		uint32_t instr;
		rwp = &CPU->cpu_ft_lock;

		/*
		 * The DTrace fasttrap provider uses a trap, "twi 31,r31,0xDDDD".
		 * We will only be here if dtrace (or someone pretending to be us)
		 * sets the trap.
		 * We let DTrace take the first crack at handling
		 * this trap; if it's not a probe that DTrace knowns about,
		 * we call into the trap() routine to handle it like a
		 * breakpoint placed by a conventional debugger.
		 */

		/*
		 * APPLE NOTE: I believe the purpose of the reader/writers lock
		 * is thus: There are times which dtrace needs to prevent calling
		 * dtrace_pid_probe_ptr(). Sun's original impl grabbed a plain
		 * mutex here. However, that serialized all probe calls, and
		 * destroyed MP behavior. So now they use a RW lock, with probes
		 * as readers, and the top level synchronization as a writer.
		 */
		lck_rw_lock_shared(rwp);
		if (dtrace_pid_probe_ptr != NULL && 
			(*dtrace_pid_probe_ptr)(sv) == 0) {
			lck_rw_unlock_shared(rwp);
			return KERN_SUCCESS;
		}
		lck_rw_unlock_shared(rwp);

		/*
		 * If the instruction that caused the breakpoint trap doesn't
		 * look like our trap anymore, it may be that this tracepoint
		 * was removed just after the user thread executed it. In
		 * that case, return to user land to retry the instuction.
		 *
		 * Note that the PC is correct because we do not advance it until after emulation.
		 */
		if (fuword32(sv->save_srr0, &instr) == 0 && instr != FASTTRAP_INSTR) {
			return KERN_SUCCESS;
		}

	}

/*
 *	If we get here, we go back to throw an exception
 */

	return KERN_FAILURE;
}

void
dtrace_safe_synchronous_signal(void)
{
// This is commented out of the x86 code and is never called.
}

int
dtrace_safe_defer_signal(void)
{
// This is commented out of the x86 code and is never called.
	return 0;
}
