#! /usr/bin/python3

# Script to generate unicode data for LDMud
# You'll need the Unicode char data base in XML format.
# (See https://www.unicode.org/Public/UCD/latest/ucdxml/)

import xml.etree.ElementTree as ET
import argparse

parser = argparse.ArgumentParser(description='Generate unicode data for LDMud')
parser.add_argument('xmlfile', help='The unicode XML file', type=argparse.FileType('rt'))

args = parser.parse_args()

xmltree = ET.parse(args.xmlfile)
xmlroot = xmltree.getroot()
current = None          # Current XML node
current_iter = None     # Iterator through the XML nodes

def get_chardata(ch: int):
    """Gets the XML node for a single character."""
    global current, current_iter

    first = True
    while True:
        # First we look at the current XML node and starting from there
        # at all the following nodes.
        while current is not None:
            if 'cp' in current.attrib:
                if ch == int(current.get('cp'),16):
                    data = current.attrib
                    current = next(current_iter, None)
                    return data
            elif 'first-cp' in current.attrib and 'last-cp' in current.attrib:
                if ch >= int(current.get('first-cp'), 16) and ch <= int(current.get('last-cp'), 16):
                    return current.attrib
            current = next(current_iter, None)

        # We haven't found it, start from the beginning again.
        if not first:
            return {}
        first = False
        current_iter = iter(xmlroot.iter())
        current = next(current_iter, None)

# Translation of the Grapheme Cluster Break property into our enum.
GCB = {
    "CN":  'GRAPHEME_CONTROL',
    "CR":  'GRAPHEME_CARRIAGE_RETURN',
    "LF":  'GRAPHEME_LINE_FEED',
    "SM":  'GRAPHEME_SPACING_MARK',
    "EX":  'GRAPHEME_EXTEND',
    "PP":  'GRAPHEME_PREPEND',
    "L":   'GRAPHEME_HANGUL_SYLLABLE_L',
    "LV":  'GRAPHEME_HANGUL_SYLLABLE_LV',
    "LVT": 'GRAPHEME_HANGUL_SYLLABLE_LVT',
    "T":   'GRAPHEME_HANGUL_SYLLABLE_T',
    "V"  : 'GRAPHEME_HANGUL_SYLLABLE_V',
    "RI":  'GRAPHEME_REGIONAL_INDICATOR',
    "ZWJ": 'GRAPHEME_ZERO_WIDTH_JOINER',
    "XX":  'GRAPHEME_OTHER',
}

# Translation of the East-Asian Width property into our enum.
EAW = {
    "A":  'WIDTH_NARROW',
    "F":  'WIDTH_WIDE',
    "H":  'WIDTH_NARROW',
    "N":  'WIDTH_NARROW',
    "Na": 'WIDTH_NARROW',
    "W":  'WIDTH_WIDE',
}

# Special character widths
CHW = {
    0x0009:   'WIDTH_NARROW',   # TAB: It's a control, but at least 1 cell wide
    0xfe0f:   'WIDTH_WIDE',     # Variation Selector 16: Makes Emojis big and colourful.
    0x1f1e6:  'WIDTH_WIDE',     # Regional Indicators are mostly displayed with double width.
    0x1f1e7:  'WIDTH_WIDE',
    0x1f1e8:  'WIDTH_WIDE',
    0x1f1e9:  'WIDTH_WIDE',
    0x1f1ea:  'WIDTH_WIDE',
    0x1f1eb:  'WIDTH_WIDE',
    0x1f1ec:  'WIDTH_WIDE',
    0x1f1ed:  'WIDTH_WIDE',
    0x1f1ee:  'WIDTH_WIDE',
    0x1f1ef:  'WIDTH_WIDE',
    0x1f1f0:  'WIDTH_WIDE',
    0x1f1f1:  'WIDTH_WIDE',
    0x1f1f2:  'WIDTH_WIDE',
    0x1f1f3:  'WIDTH_WIDE',
    0x1f1f4:  'WIDTH_WIDE',
    0x1f1f5:  'WIDTH_WIDE',
    0x1f1f6:  'WIDTH_WIDE',
    0x1f1f7:  'WIDTH_WIDE',
    0x1f1f8:  'WIDTH_WIDE',
    0x1f1f9:  'WIDTH_WIDE',
    0x1f1fa:  'WIDTH_WIDE',
    0x1f1fb:  'WIDTH_WIDE',
    0x1f1fc:  'WIDTH_WIDE',
    0x1f1fd:  'WIDTH_WIDE',
    0x1f1fe:  'WIDTH_WIDE',
    0x1f1ff:  'WIDTH_WIDE',
}

# Special categories
GCW = {
    "Cc": 'WIDTH_NEUTRAL',     # Control characters have no width.
}


# We generate two tables, the index into the first will be character >> SHIFT.
SHIFT=8
MASK=(1<<SHIFT)-1

entry_list = []
entries = {}

blocks = {}

table1 = []
table2 = []
current_table1_block = []

def get_type(num: int):
    """Returns the apropriate C type for a given maximum value."""
    if num < 256:
        return "unsigned char"
    elif num < 65536:
        return "unsigned short"
    else:
        return "unsigned int"

xmldesc = xmlroot.find('ucd:description', {'ucd':'http://www.unicode.org/ns/2003/ucd/1.0'})

print("""/* Do not edit this file.
 *
 * This file was generated by unigenerate.py.
 *
 * Standard Version: %s
 */

#include \"unidata.h\"

struct unicode_char_data_s unicode_char_data[] = {""" % (xmldesc is not None and xmldesc.text or 'Unknown',))

for ch in range(0x110000):
    chardata = get_chardata(ch)
    entry = "{ %s, %s }" % (
        GCB.get(chardata.get("GCB", ""), 'GRAPHEME_OTHER'),
        CHW.get(ch) or GCW.get(chardata.get("gc", "")) or EAW.get(chardata.get("ea", "")) or 'WIDTH_NEUTRAL',
    )
    if entry in entries:
        entry_id = entries[entry]
    else:
        entries[entry] = entry_id = len(entries)
        print("    %s," % (entry,))

    current_table1_block.append(entry_id)
    if (ch & MASK) == MASK:
        block = tuple(current_table1_block)
        if block in blocks:
            blockid = blocks[block]
        else:
            for i in range(len(current_table1_block)-1, -1, -1):
                if i == 0:
                    blocks[block] = blockid = len(table2)
                    table2.extend(current_table1_block)
                elif table2[-i:] == current_table1_block[:i]:
                    table2.extend(current_table1_block[i:])
                    blocks[block] = blockid = len(table2) - len(current_table1_block)
                    break

        table1.append(blockid)

        current_table1_block = []

print("""};

%s unicode_table1[] = {""" % (get_type(len(table2)),))

idx = 0
for entry in table1:
    if idx == 0:
        print("   ", end="")

    print(" %d," % (entry,), end="")
    idx += 1
    if idx == 16:
        print("")
        idx = 0


print("""};

%s unicode_table2[] = {""" % (get_type(len(entries)),))

idx = 0
for entry in table2:
    if idx == 0:
        print("   ", end="")

    print(" %d," % (entry,), end="")
    idx += 1
    if idx == 16:
        print("")
        idx = 0


print("};")
