/*---------------------------------------------------------------------------
 * LPC-Compiler: Preprocessor and Lexical Analysis.
 *
 *---------------------------------------------------------------------------
 * The lexer is initialised by a call to init_lexer(). This function sets
 * up the internal tables and also reads the permanent defines from
 * the lpc_predefs list the caller set up before the call.
 *
 * A new compilation is set up by a call to start_new_file(), passing
 * the filedescriptor of the file to compile as argument. Control is
 * then passed to the parser in prolang, which calls yylex() here to
 * get the next token. After the compilation is finished, end_new_file()
 * performs the cleanup.
 *
 * The lexer also holds the table of instructions (instrs[]) and the
 * driver's own ctype tables. Both are included from the file efun_defs.c
 * which is generated by the program make_func during the building
 * process.
 *
 * For an explanation of the datastructures look at the places of
 * definition of the structures - it's too much to put here, too.
 *---------------------------------------------------------------------------
 */

#include "driver.h"
#include "typedefs.h"

#include "my-alloca.h"
#include <stdio.h>
#include <assert.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdarg.h>
#include <stddef.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <wctype.h>

#include "lex.h"
#include "prolang.h"

#include "array.h"
#include "backend.h"
#include "closure.h"
#include "comm.h"
#include "exec.h"
#include "filestat.h"
#include "gcollect.h"
#include "hash.h"
#include "iconv_opt.h"
#include "instrs.h"
#include "interpret.h"
#include "lang.h"
#include "main.h"
#include "mempools.h"
#include "mstrings.h"
#include "object.h"
#include "patchlevel.h"
#include "simulate.h"
#include "simul_efun.h"
#include "stdstrings.h"
#include "strfuns.h"
#include "svalue.h"
#include "wiz_list.h" /* wizlist_name[] */
#include "xalloc.h"

#include "pkg-python.h"
#include "i-eval_cost.h"

#include "../mudlib/sys/driver_hook.h"

/* TODO: Implement the # and ## operators. With this, #define X(a) (a + "a")
 * TODO:: can be made sane (X(b) -> 'b + "a"' instead of 'b + "b"').
 * TODO: New predefs' __BASENAME__.
 * TODO: #define macro(a,b,...) -> ... is assigned to __VA_ARGS__ (see oncoming
 * TODO:: C standard).
 * TODO: Does Standard-C allow recursive macro expansion? If not, we
 * TODO:: should disallow it, too.
 */

/* We can't use the EOF character directly, as in its (char) representation
 * clashes with ISO-8859 character 0xFF. Instead we use ascii SOH (0x01),
 * which in turn is not allowed as input character.
 */

#define CHAR_EOF ((char)0x01)

/*-------------------------------------------------------------------------*/

#define MLEN   4096
  /* Maximum length of a macro text (definition)
   */

#define NSIZE  256
  /* Maximum length of a macro (argument) name.
   */

#define NARGS 25
  /* Max number of macro arguments
   */

#define EXPANDMAX 25000
  /* Maximum number of expansions per line.
   */

#define MAXLINE 2048
  /* Maximum length of an input line.
   */

#define MAX_ANSI_CONCAT 4096
  /* Maximum length of an ANSI-style string literal concatenation.
   */

#define INC_OPEN_BUFSIZE 1024
  /* Maximum length of an include filename.
   */

#ifndef DEFMAX
#    define DEFMAX 12000
#endif
  /* Maximum length of an expanded macro.
   */

#define MAX_TOTAL_BUF 400000
  /* Maximum length of macro expansion buffer
   */

#define DEFBUF_1STLEN (DEFMAX+MAXLINE+1)
  /* Initial length of macro expansion buffer, enough
   * to allow DEFMAX + an input line + '\0'
   */

#define CONVBUFSIZE 2*MAXLINE
  /* Size of the conversion buffer.
   */

/*-------------------------------------------------------------------------*/

source_loc_t current_loc;
  /* The current compilation location.
   */

int total_lines;
  /* Total number of lines compiled so far (used to compute the average
   * compiled lines/s)
   */

static const char *object_file;
  /* Name of the file for which the lexer was originally called.
   */

Bool pragma_warn_missing_return;
  /* True: generate a runtime warning if a value-returning function
   * does end with an explicit return statement.
   */

Bool pragma_check_overloads;
  /* TRUE if function redefinitions have to match the originals in their
   * types. This pragma is meant mainly to ease the adaption of old
   * mudlibs.
   */

pragma_cttype_checks_e pragma_strict_types;
  /* Type enforcing mode: PRAGMA_WEAK_TYPES, PRAGMA_STRONG_TYPES
   * and PRAGMA_STRICT_TYPES.
   */

Bool pragma_save_types;
  /* True: save argument types after compilation.
   */

Bool pragma_no_clone;
  /* True: prevent the object from being clone.
   */

Bool pragma_no_inherit;
  /* True: prevent the program from being inherited.
   */

Bool pragma_no_shadow;
  /* True: prevent the program from being shadowed.
   */

Bool pragma_pedantic;
  /* True: treat a number of sloppy language constructs as errors.
   */

Bool pragma_warn_deprecated;
  /* True: warn if deprecated efuns are used.
   */

Bool pragma_range_check;
  /* True: warn (at runtime) if array ranges are invalid.
   */

Bool pragma_share_variables;
  /* TRUE: Share the blueprint's variables with its clones.
   */

Bool pragma_warn_empty_casts;
  /* True: warn if a type is casted to itself.
   */

Bool pragma_rtt_checks;
  /* True: enable runtime type checks for this program
   */

Bool pragma_warn_rtt_checks;
  /* True: emit warnings when doing runtime type checks for this program
   */

bool pragma_no_bytes_type;
  /* True: 'bytes' will be deactivated as a keyword and 'string'
   *       means <string|bytes>.
   */

bool pragma_warn_unused_variables;
  /* True: Warn about declared but never used variables.
   */

string_t *last_lex_string;
  /* When lexing string literals, this is the (shared) string lexed
   * so far. It is used to pass string values to lang.c and may be
   * freed there. It is also used for byte sequence literals.
   */

struct lpc_predef_s *lpc_predefs = NULL;
  /* List of macros predefined by other parts of the driver, especially from
   * main.c for the '-D' commandline option.
   */

static source_file_t * src_file_list = NULL;
  /* List of source_file structures during a compile.
   */

static Mempool lexpool = NULL;
  /* Fifopool to hold the allocations for the include and lpc_ifstate_t stacks.
   */

/*-------------------------------------------------------------------------*/
/* The lexer can take data from either a file or a string buffer.
 * The handling is unified using the struct source_s structure.
 * TODO: Use this source similar to auto-include to expand macros in the
 * TODO:: the compile. This would make it easier to find errors caused
 * TODO:: by macro replacements.
 */

typedef struct source_s
{
    int        fd;           /* Filedescriptor or -1 */
    iconv_t    cd;           /* Conversion descriptor, or -1 (if fd is valid, then cd is valid also) */
    char*      convbuf;      /* Conversion buffer of size CONVBUFSIZE, or NULL */
    char*      convstart;    /* Start position of available bytes in the conversion buffer.*/
    size_t     convleft;     /* Number of available bytes in the conversion buffer. */
    char       convbytes[4]; /* Bytes that didn't fit into the destination buffer. */
    string_t * str;          /* The source string (referenced), or NULL */
    size_t     current;      /* Current position in .str */
} source_t;

static source_t yyin;
  /* Current input source.
   */

/*-------------------------------------------------------------------------*/
/* The lexer uses a combined file-input/macro-expansion buffer
 * called defbuf[] of length <defbuf_len>. Within this buffer, the last
 * MAXLINE bytes are reserved as (initial) file-input buffer, its beginning
 * and end marked with the pointers linebufstart and linebufend. In this
 * space, pointer outp points to the next character to process.
 *
 * The file-input buffer may contain several textlines, all terminated
 * with a '\n'. After the last (complete) textline, a '\0' is set as
 * sentinel. Usually this will overwrite the first character of the
 * incomplete line right at the end of the input buffer, therefore this
 * character is stored in the variable saved_char.
 *
 * When all lines in the buffer have been processed (ie. outp points to
 * the '\0' sentinel), the remaining fragment of the yet incomplete line
 * is copied _before_ linebufstart (and outp set accordingly), then
 * the next MAXLINE bytes are read into the buffer starting at
 * linebufstart.
 *
 * If there are less than MAXLINE bytes left to read, the end of the file
 * is marked in the buffer with the CHAR_EOF character (a \0 sentinel is not
 * necessary as compilation and thus lexing will end with the CHAR_EOF
 * character).
 *
 * When including files, a new area of MAXLINE bytes is reserved in defbuf,
 * which ends exactly at the current outp. The position of the current
 * area is recorded with the current position of linebufstart relative to
 * the end of defbuf. Obviously this can be repeated until the max size
 * of defbuf (MAX_TOTAL_BUF) is reached.
 *
 * Macro expansions are done such that the replacement text for a macro
 * copied right before outp (which at that time points at the character
 * after the macro use), then outp is set back to point at the beginning
 * of the added text, lexing the just expanded text next.
 *
 */

static char *defbuf = NULL;
  /* The combined input/expansion buffer.
   */

static unsigned long defbuf_len = 0;
  /* Current length of defbuf.
   */

static char *outp;
  /* Pointer to the next character in defbuf[] to be processed.
   */

static char *linebufstart;
  /* Begin of the input line buffer within defbuf[].
   */

static char *linebufend;
  /* End of the input line buffer within defbuf[].
   */

static char saved_char;
  /* The first character of the incomplete line after the last complete
   * one in the input buffer. Saved here because in the buffer itself
   * it is overwritten with '\0'.
   */

/*-------------------------------------------------------------------------*/

static Bool lex_fatal;
  /* True: lexer encountered fatal error.
   */

static int lex_error_pos = -1;
  /* For encoding errors, the byte position of the error.
   */

static svalue_t *inc_list;
  /* An array of pathnames to search for <>-include files.
   * This is a pointer to the vector.item[] held in the driver_hook[]
   * array.
   */

static size_t inc_list_size;
  /* The number of names in <inc_list>.
   */

static mp_int inc_list_maxlen;
  /* The lenght of the longest name in <inc_list>.
   */

static int nexpands;
  /* Number of macro expansions on this line so far.
   */

static char yytext[MAXLINE];
  /* Temporary buffer used to collect data.
   */


/*-------------------------------------------------------------------------*/
/* Enforce an appropriate range for ITABLE_SIZE
 * The hash used in ident_s is of type hash16_t. Therefore the hash table must
 * not contain more hash chains than SHRT_MAX.
 */
#if ITABLE_SIZE < 256 || ITABLE_SIZE > MAX_HASH16
#error "ITABLE_SIZE must in the range of 256 and MAX_HASH16 (usually 2^16)."
This is the end...
#endif

static ident_t *ident_table[ITABLE_SIZE];
  /* The lexer stores all encountered identifiers in a hashtable of struct
   * idents. The table follows the usual structure: the index (hash value)
   *  is computed from the name of the identifier, the indexed table element
   * the points to the head of a chain of different identifier values with
   * identical hash. The most recently used identifier is always moved to
   * the head of the chain.
   *
   * The table is used to store all identifiers and their value: starting
   * from efun names and reserved words (like 'while') over preprocessor
   * macros to 'normal' lfun and variable names. The identifiers are
   * distinguished by the .type field in the ident structure. Should one
   * identifier used with several different types at the same time, one entry
   * is generated for each type, and they are all linked together by their
   * .inferior pointers into a list ordered by falling type value. The entry
   * with the highest type value is the one put into the hashtable's chain.
   */
static INLINE hash16_t identhash(const char* s)
  /* Hash an identifier name (c-string) into a table index.
   */
{
#if !( (ITABLE_SIZE) & (ITABLE_SIZE)-1 )
    // use faster masking if ITABLE_SIZE is a power of 2.
    return hash_string(s, strlen(s)) & (ITABLE_SIZE-1);
#else
    return hash_string(s, strlen(s)) % ITABLE_SIZE;
#endif
}

static INLINE hash16_t identhash_n(const char* s, size_t len)
  /* Hash an identifier name (<s> with length <len>) into a table index.
   */
{
#if !( (ITABLE_SIZE) & (ITABLE_SIZE)-1 )
    // use faster masking if ITABLE_SIZE is a power of 2.
    return hash_string(s, len) & (ITABLE_SIZE-1);
#else
    return hash_string(s, len) % ITABLE_SIZE;
#endif
}

  /* In addition to this, the lexer keeps two lists for all efuns and
   * preprocessor defines: all_efuns and all_defines. These are linked
   * together with the .next_all field in the ident structure.
   */

ident_t *all_efuns = NULL;
  /* The list of efuns. */

static ident_t *all_defines = NULL;
  /* The list of all non-permanent macro defines.
   * Entries with a NULL .name are undefined macros.
   */

static ident_t *permanent_defines = NULL;
  /* The list of all permanent macro defines. */


static ident_t *undefined_permanent_defines = NULL;
  /* 'Parking list' for permanent defines which have been #undef'ined.
   * After the compilation is complete, they will be put back into
   * the ident_table.
   */

/*-------------------------------------------------------------------------*/

/* The stack to handle nested #if...#else...#endif constructs.
 */

typedef struct lpc_ifstate_s
{
    struct lpc_ifstate_s *next;
    int                   state;  /* which token to expect */
} lpc_ifstate_t;

/* lpc_ifstate_t.state values: */

#define EXPECT_ELSE  1
#define EXPECT_ENDIF 2

static lpc_ifstate_t *iftop = NULL;

/*-------------------------------------------------------------------------*/

/* The stack to save important state information when handling
 * nested includes.
 */

static struct incstate
{
    struct incstate * next;

    source_t     yyin;           /* The current input source */
    source_loc_t loc;            /* Current source location */
    ptrdiff_t    linebufoffset;  /* Position of linebufstart */
    mp_uint      inc_offset;     /* Handle returned by store_include_info() */
    char         saved_char;
} *inctop = NULL;

/*-------------------------------------------------------------------------*/

/* Translation table of reserved words into the lexcodes assigned by yacc
 * in lang.h.
 */

struct s_reswords
{
    char *name;  /* The reserved word */
    int   code;  /* The assigned code */
};

static struct s_reswords reswords[]
 = { { "break",          L_BREAK         }
   , { "bytes",          L_BYTES_DECL    }
   , { "case",           L_CASE          }
   , { "catch",          L_CATCH         }
   , { "closure",        L_CLOSURE_DECL  }
   , { "continue",       L_CONTINUE      }
   , { "default",        L_DEFAULT       }
   , { "do",             L_DO            }
   , { "else",           L_ELSE          }
   , { "float",          L_FLOAT_DECL    }
   , { "for",            L_FOR           }
   , { "foreach",        L_FOREACH       }
   , { "function",       L_FUNC          }
   , { "if",             L_IF            }
#ifdef L_IN
   , { "in",             L_IN            }
#endif
   , { "inherit",        L_INHERIT       }
   , { "int",            L_INT           }
   , { "mapping",        L_MAPPING       }
   , { "mixed",          L_MIXED         }
   , { "nomask",         L_NO_MASK       }
   , { "nosave",         L_NOSAVE        }
   , { "deprecated",     L_DEPRECATED    }
   , { "object",         L_OBJECT        }
   , { "private",        L_PRIVATE       }
   , { "protected",      L_PROTECTED     }
   , { "public",         L_PUBLIC        }
   , { "return",         L_RETURN        }
   , { "static",         L_STATIC        }
   , { "status",         L_STATUS        }
   , { "struct",         L_STRUCT        }
   , { "string",         L_STRING_DECL   }
   , { "switch",         L_SWITCH        }
   , { "symbol",         L_SYMBOL_DECL   }
   , { "varargs",        L_VARARGS       }
   , { "virtual",        L_VIRTUAL       }
   , { "visible",        L_VISIBLE       }
   , { "void",           L_VOID          }
   , { "while",          L_WHILE         }
   };

/*-------------------------------------------------------------------------*/

/* The definitions and tables for the preprocessor expression evaluator.
 */


#define BNOT   1  /* Unary operator opcodes*/
#define LNOT   2
#define UMINUS 3
#define UPLUS  4

#define MULT   1  /* Binary operator opcodes */
#define DIV    2
#define MOD    3
#define BPLUS  4
#define BMINUS 5
#define LSHIFT 6
#define RSHIFT 7
#define LESS   8
#define LEQ    9
#define GREAT 10
#define GEQ   11
#define EQ    12
#define NEQ   13
#define BAND  14
#define XOR   15
#define BOR   16
#define LAND  17
#define LOR   18
#define QMARK 19

  /* lookup table for initial operator characters.
   * The table covers the characters [' '..'~'].
   * 0 for no operator, else index into optab2.
   */
static char _optab[]
 = {0,6,0,0,0,46,50,0,0,0,2,18,0,14,0,10,0,0,0,0,0,0,0,0,0,0,0,0,22,42,32,68,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,57,0,1
 };

  /* Lookup table for the complete operator data in a serial format.
   *
   * optab2[index-1] : operation code for unary operator, 0 for none.
   * optab2[index+0 .. +3 .. +6 ...] :
   * two character binary operators: second character, operation code, prio
   * one character binary operator & end: 0,           operation code, prio
   * end: 0, 0
   *
   * Note that some entries overlap.
   */
static char optab2[]
  = { BNOT, 0                                 /*  1: ~         */
    ,         0,   MULT,   11                 /*  2: *         */
    , LNOT,   '=', NEQ,     7                 /*  6: !, !=     */
    , 0,      0,   DIV,    11                 /* 10: /         */
    , UMINUS, 0,   BMINUS, 10                 /* 14: -x, x-y   */
    , UPLUS,  0,   BPLUS,  10                 /* 18: +x, x+y   */
    , 0,      '<', LSHIFT,  9, '=', LEQ,  8, 0, LESS,  8
                                              /* 22: <<, <=, < */
    , 0,      '>', RSHIFT,  9, '=', GEQ,  8, 0, GREAT, 8
                                              /* 32: >>, >=, > */
    , 0,      '=', EQ,      7                 /* 42: ==        */
    , 0,      0,   MOD,    11                 /* 46: %         */
    , 0,      '&', LAND,    3, 0,   BAND, 6   /* 50: &&, &     */
    , 0,      '|', LOR,     2, 0,   BOR,  4   /* 57: ||, |     */
    , 0,      0,   XOR,     5                 /* 64: ^         */
    , 0,      0,   QMARK,   1                 /* 68: ?         */
};

#define optab1(c) (isascii(c) ? _optab[(c)-' '] : 0)
  /* Use optab1 to index _optab with raw characters.
   */

/*-------------------------------------------------------------------------*/

  /* A handy macro to statically determine the number of
   * elements in an array.
   */
#define NELEM(a) (sizeof (a) / sizeof (a)[0])

  /* Save the character in variable 'c' in the yytext buffer, if
   * there is enough space left.
   */
#define SAVEC \
    if (yyp < yytext+MAXLINE-5)\
       *yyp++ = (char)c;\
    else {\
       lexerror("Line too long");\
       break;\
    }

  /* The magic character used for function macros to mark the places
   * in the replacement text where the macro arguments are to be
   * inserted.
   * The marking sequence for argument n is { '@', '@'+n+1 }, and
   * the character '@' itself is stored as { '@', '@' }.
   */
#define MARKS '@'

/*-------------------------------------------------------------------------*/
/* Forward declarations */

static INLINE int number(long);
static INLINE int string(char *, size_t);
static void handle_define(char *, Bool);
static void add_define(char *, short, char *, source_loc_t);
static void add_permanent_define_str(char *name, short nargs, char *text);
static void add_permanent_define_fun(char *name, short nargs, defn_fun fun);
static Bool expand_define(void);
static Bool _expand_define(struct defn*, ident_t*);
static INLINE void myungetc(char);
static p_int cond_get_exp(int, svalue_t *);
static p_int exgetc (char** chpos);
static char *get_current_file(char **);
static char *get_current_line_buf(char **);
static char *get_current_function(char **);
static char *get_version(char **);
static char *get_hostname(char **);
static char *get_domainname(char **);
static char *get_current_dir(char **);
static char *get_sub_path(char **);
static char *efun_defined(char **);
static char *get_reset_time_buf(char **);
static char *get_cleanup_time_buf(char **);
static char *get_memory_limit_buf(char **);
static void lexerrorf VARPROT((char *, ...), printf, 1, 2);
static void lexerror(char *);
static ident_t *lookup_define(char *);
static void skip_comment(void);

/*-------------------------------------------------------------------------*/

#include "efun_defs.c"

/* struct instr instrs[] = { ... };
 *
 * Information about all instructions and efuns, generated by make_func.
 * Also included are the table for our own ctype functions.
 *
 * The numbers of arguments are used by the compiler.
 * If min == max, then no information has to be coded about the
 * actual number of arguments. Otherwise, the actual number of arguments
 * will be stored in the byte after the instruction.
 * A maximum value of -1 means unlimited maximum value.
 *
 * If an argument has type 0 (T_INVALID) specified, then no checks will
 * be done at run time.
 *
 * The argument types are checked by the compiler if type checking is enabled,
 * and always at runtime.
 */

/*-------------------------------------------------------------------------*/
void
init_lexer(void)

/* Initialize the various lexer tables, including the predefined macros
 * from the commandline given in lpc_predefs.
 * The lpc_predefs list is deallocated by this call.
 */

{
    size_t i, n;
    char mtext[MLEN];

    /* Allocate enough memory for 20 nested includes/ifs */
    lexpool = new_lifopool(size_lifopool( sizeof(lpc_ifstate_t)
                                         +sizeof(struct incstate)));
    if (!lexpool)
        fatal("Out of memory.\n");

    current_loc.file = NULL;
    current_loc.line = 0;


    /* Clear the table of identifiers */
    for (i = 0; i < ITABLE_SIZE; i++)
        ident_table[i] = NULL;

    /* For every efun, create a global entry in ident_table[] */
    for (n = 0; n < NELEM(instrs); n++)
    {
        ident_t *p;

        if (instrs[n].Default == -1)
            continue;

        /* In !compat mode, skip certain efuns */
        if (!compat_mode
         && (   !strcmp(instrs[n].name, "creator")
#ifdef USE_DEPRECATED
             ||  n == F_TRANSFER
#endif /* USE_DEPRECATED */
            )
           )
            continue;

        p = make_shared_identifier(instrs[n].name, I_TYPE_GLOBAL, 0);
        if (!p)
            fatal("Out of memory\n");
        if (p->type != I_TYPE_UNKNOWN)
        {
            fatal("Duplicate efun '%s'.\n", instrs[n].name);
            /* NOTREACHED */
            continue;
        }
        init_global_identifier(p, /* bVariable: */ MY_FALSE);
        p->u.global.efun = (short)n;
        p->next_all = all_efuns;
        all_efuns = p;
    }

    /* For every reserved word, create a global entry in ident_table[] */
    for (i = 0; i < NELEM(reswords); i++)
    {
        ident_t *p;

        p = make_shared_identifier(reswords[i].name, I_TYPE_RESWORD, 0);
        if (!p)
            fatal("Out of memory\n");
        p->type = I_TYPE_RESWORD;
        p->u.code = reswords[i].code;
    }


    /* Add the standard permanent macro definitions */
    /* TODO: Make the strings tabled */

    add_permanent_define_str("LPC3", -1, "");
    add_permanent_define_str("__LDMUD__", -1, "");
    if (compat_mode)
    {
        add_permanent_define_str("COMPAT_FLAG", -1, "");
        add_permanent_define_str("__COMPAT_MODE__", -1, "");
    }
    add_permanent_define_str("__EUIDS__", -1, "");

    if (allow_filename_spaces)
        add_permanent_define_str("__FILENAME_SPACES__", -1, "");
    if (strict_euids)
        add_permanent_define_str("__STRICT_EUIDS__", -1, "");

    if (compat_mode)
    {
        mtext[0] = '"';
        strcpy(mtext+1, master_name);
        strcat(mtext+1, "\"");
    }
    else
    {
        mtext[0] = '"';
        mtext[1] = '/';
        strcpy(mtext+2, master_name);
        strcat(mtext+2, "\"");
    }

    add_permanent_define_str("__MASTER_OBJECT__", -1, mtext);
    add_permanent_define_fun("__FILE__", -1, get_current_file);
    add_permanent_define_fun("__FUNCTION__", -1, get_current_function);
    add_permanent_define_fun("__DIR__", -1, get_current_dir);
    add_permanent_define_fun("__PATH__", 1, get_sub_path);
    add_permanent_define_fun("__LINE__", -1, get_current_line_buf);
    add_permanent_define_fun("__VERSION__", -1, get_version);
    add_permanent_define_str("__VERSION_MAJOR__", -1, VERSION_MAJOR);
    add_permanent_define_str("__VERSION_MINOR__", -1, VERSION_MINOR);
    add_permanent_define_str("__VERSION_MICRO__", -1, VERSION_MICRO);
    add_permanent_define_str("__VERSION_PATCH__", -1, VERSION_PATCH);
    add_permanent_define_str("__VERSION_COMMITID__", -1, "\"" COMMIT_ID "\"");
    add_permanent_define_str("__VERSION_LOCAL__", -1, LOCAL_LEVEL);

    add_permanent_define_fun("__HOST_NAME__", -1, get_hostname);
    add_permanent_define_fun("__DOMAIN_NAME__", -1, get_domainname);
    add_permanent_define_fun("__HOST_IP_NUMBER__", -1, (defn_fun) get_host_ip_number);
    sprintf(mtext, "%d", MAX_USER_TRACE);
    add_permanent_define_str("__MAX_RECURSION__", -1, mtext);
    add_permanent_define_fun("__EFUN_DEFINED__", 1, efun_defined);
#ifdef ERQ_DEMON
    sprintf(mtext, "%d", ERQ_MAX_SEND);
    add_permanent_define_str("__ERQ_MAX_SEND__", -1, mtext);
    sprintf(mtext, "%d", ERQ_MAX_REPLY);
    add_permanent_define_str("__ERQ_MAX_REPLY__", -1, mtext);
#endif
    add_permanent_define_fun("__MAX_MALLOC__", -1, get_memory_limit_buf);
    sprintf(mtext, "%"PRId32, def_eval_cost);
    add_permanent_define_str("__MAX_EVAL_COST__", -1, mtext);
    sprintf(mtext, "%ld", (long)CATCH_RESERVED_COST);
    add_permanent_define_str("__CATCH_EVAL_COST__", -1, mtext);
    sprintf(mtext, "%ld", (long)MASTER_RESERVED_COST);
    add_permanent_define_fun("__RESET_TIME__", -1, get_reset_time_buf);
    add_permanent_define_fun("__CLEANUP_TIME__", -1, get_cleanup_time_buf);
    sprintf(mtext, "%ld", alarm_time);
    add_permanent_define_str("__ALARM_TIME__", -1, mtext);
    sprintf(mtext, "%ld", heart_beat_interval);
    add_permanent_define_str("__HEART_BEAT_INTERVAL__", -1, mtext);
    if (synch_heart_beats)
        add_permanent_define_str("__SYNCHRONOUS_HEART_BEAT__", -1, "1");
    sprintf(mtext, "%zu", (size_t)MAX_COMMAND_LENGTH - 1);
    add_permanent_define_str("__MAX_COMMAND_LENGTH__", -1, mtext);
#ifdef EVAL_COST_TRACE
    add_permanent_define_str("__EVAL_COST_TRACE__", -1, "1");
#endif
#ifdef HAS_IDN
    add_permanent_define_str("__IDNA__", -1, "1");
#endif
#ifdef USE_IPV6
    add_permanent_define_str("__IPV6__", -1, "1");
#endif
#ifdef USE_MCCP
    add_permanent_define_str("__MCCP__", -1, "1");
#endif
#ifdef USE_MYSQL
    add_permanent_define_str("__MYSQL__", -1, "1");
#endif
#ifdef USE_PGSQL
    add_permanent_define_str("__PGSQL__", -1, "1");
#endif
#ifdef USE_SQLITE
    add_permanent_define_str("__SQLITE__", -1, "1");
#endif
#ifdef USE_XML
    add_permanent_define_str("__XML_DOM__", -1, "1");
#endif
#ifdef USE_JSON
    add_permanent_define_str("__JSON__", -1, "1");
#endif
#ifdef HAS_PCRE
    add_permanent_define_str("__PCRE__", -1, "1");
#endif
    add_permanent_define_str("__LPC_NOSAVE__", -1, "1");
#ifdef USE_DEPRECATED
    add_permanent_define_str("__DEPRECATED__", -1, "1");
#endif
    add_permanent_define_str("__LPC_STRUCTS__", -1, "1");
    add_permanent_define_str("__LPC_INLINE_CLOSURES__", -1, "1");
    add_permanent_define_str("__LPC_ARRAY_CALLS__", -1, "1");
#ifdef USE_TLS
    add_permanent_define_str("__TLS__", -1, "1");
#ifdef HAS_GNUTLS
    add_permanent_define_str("__GNUTLS__", -1, "1");
#endif
#ifdef HAS_OPENSSL
    add_permanent_define_str("__OPENSSL__", -1, "1");
#endif
#endif
#ifdef USE_GCRYPT
    add_permanent_define_str("__GCRYPT__", -1, "1");
#endif
#ifdef USE_PYTHON
    add_permanent_define_str("__PYTHON__", -1, "1");
#endif
    if (wizlist_name[0] != '\0')
    {
        if (compat_mode)
        {
            mtext[0] = '"';
            strcpy(mtext+1, wizlist_name);
            strcat(mtext+1, "\"");
        }
        else
        {
            mtext[0] = '"';
            mtext[1] = '/';
            strcpy(mtext+2, wizlist_name);
            strcat(mtext+2, "\"");
        }
        add_permanent_define_str("__WIZLIST__", -1, mtext);
    }

    sprintf(mtext, "(%"PRIdPINT")", PINT_MAX);
    add_permanent_define_str("__INT_MAX__", -1, mtext);
    sprintf(mtext, "(%"PRIdPINT")", PINT_MIN);
    add_permanent_define_str("__INT_MIN__", -1, mtext);
    sprintf(mtext, "(%.17g)", DBL_MAX);
    add_permanent_define_str("__FLOAT_MAX__", -1, mtext);
    sprintf(mtext, "(%.17g)", DBL_MIN);
    add_permanent_define_str("__FLOAT_MIN__", -1, mtext);
    sprintf(mtext, "%"PRIdMPINT, get_current_time());
    add_permanent_define_str("__BOOT_TIME__", -1, mtext);

    /* Add the permanent macro definitions given on the commandline */

    while (NULL != lpc_predefs)
    {
        char namebuf[NSIZE];
        struct lpc_predef_s *tmpf;

        tmpf = lpc_predefs;
        lpc_predefs = lpc_predefs->next;

        *mtext = '\0';
        sscanf(tmpf->flag, "%[^=]=%[ -~=]", namebuf, mtext);
        if ( strlen(namebuf) >= NSIZE )
            fatal("-D%s: macroname too long (>%d)\n", tmpf->flag, NSIZE);
        if ( strlen(mtext) >= MLEN )
            fatal("-D%s: macrotext too long (>%d)\n", tmpf->flag, MLEN);
        add_permanent_define_str(namebuf, -1, mtext);

        xfree(tmpf->flag);
        xfree(tmpf);
    }
} /* init_lexer() */

/*-------------------------------------------------------------------------*/
static inline bool
unicode_iswhite (p_int ch)

/* Returns true, if the given character is a whitespace (excluding line breaks).
 */

{
    return ch < 128 ? (_my_ctype[(unsigned char)ch]&_MCTs) : iswspace((wint_t) ch);
} /* unicode_iswhite() */

/*-------------------------------------------------------------------------*/
static inline bool
unicode_isalunum (p_int ch)

/* Returns true, if the given character is a alphanumeric or an underscore.
 */

{
    return ch < 128 ? (_my_ctype[(unsigned char)ch]&_MCTa) : iswalnum((wint_t) ch);
} /* unicode_isalunum() */

static inline bool
unicode_isdigit (p_int ch)

/* Returns true, if the given character is an ASCII digit.
 */

{
    return ch >= '0' && ch <= '9';
} /* unicode_isalunum() */

/*-------------------------------------------------------------------------*/
static inline size_t
utf8_iswhite (const char* str)

/* Returns the number of bytes of the first UTF8 character in <str>,
 * if this character is a whitespace (excluding line breaks).
 * Returns 0 if it isn't.
 * The text is assumed to be zero-terminated.
 */

{
    p_int ch;
    size_t len = utf8_to_unicode(str, 4, &ch);
    if (!len)
        return 0;

    return unicode_iswhite(ch) ? len : 0;
} /* utf8_iswhite() */

/*-------------------------------------------------------------------------*/
static inline size_t
utf8_isalunum (const char* str)

/* Returns the number of bytes of the first UTF8 character in <str>,
 * if this character is a alphanumeric or an underscore.
 * Returns 0 if it isn't.
 * The text is assumed to be zero-terminated.
 */

{
    p_int ch;
    size_t len = utf8_to_unicode(str, 4, &ch);
    if (!len)
        return 0;

    return unicode_isalunum(ch) ? len : 0;
} /* utf8_iswhite() */

/*-------------------------------------------------------------------------*/
static inline char*
skip_line (char* str)

/* Returns the position of the character after the next newline in <str>.
 * It is assumed that there is a next newline.
 */

{
    while (*str++ != '\n');

    return str;
} /* skip_line() */

/*-------------------------------------------------------------------------*/
static inline char*
skip_white (char* str)

/* Returns the position of the next non-white character in <str>.
 * The text is assumed to be zero-terminated.
 */

{
    size_t len;

    while ((len = utf8_iswhite(str)) != 0)
        str += len;

    return str;
} /* skip_white() */

/*-------------------------------------------------------------------------*/
static inline char*
skip_alunum (char* str)

/* Returns the position of the next character in <str>,
 * that is not alphanumeric or an underscore.
 * The text is assumed to be zero-terminated.
 */

{
    size_t len;

    while ((len = utf8_isalunum(str)) != 0)
        str += len;

    return str;
} /* skip_alunum() */

/*-------------------------------------------------------------------------*/
static inline char*
skip_nonspace (char* str)

/* Returns the position of the next whitespace character in <str>.
 * It is assumed that there is a whitespace character in the text.
 */

{
    while (true)
    {
        p_int ch;
        size_t len = utf8_to_unicode(str, 4, &ch);

        if (!len)
            return str;

        if (!ch || iswspace((wint_t) ch))
            return str;

        str += len;
    }
} /* skip_nonspace() */

/*-------------------------------------------------------------------------*/
int
symbol_operator (const char *symbol, const char **endp)

/* Analyse the text starting at <symbol> (which points to the first character
 * after the assumed "#'") if it describes a closure symbol. If yes, return
 * the operator code and set *<endp> to the first character after the
 * recognized operator.
 * If no operator can be recognized, return -1 and set *<endp> to <symbol>.
 *
 * The function is called from ed.c and from symbol_efun().
 *
 * Recognized are the following operators:
 *
 *   #'+=     -> F_ADD_EQ
 *   #'++     -> F_POST_INC
 *   #'+      -> F_ADD
 *   #'-=     -> F_SUB_EQ
 *   #'--     -> F_POST_DEC
 *   #'-      -> F_SUBTRACT
 *   #'*=     -> F_MULT_EQ
 *   #'*      -> F_MULTIPLY
 *   #'/=     -> F_DIV_EQ
 *   #'/      -> F_DIVIDE
 *   #'%=     -> F_MOD_EQ
 *   #'%      -> F_MOD
 *   #',      -> F_POP_VALUE
 *   #'^=     -> F_XOR_EQ
 *   #'^      -> F_XOR
 *   #'||     -> F_LOR
 *   #'||=    -> F_LOR_EQ
 *   #'|=     -> F_OR_EQ
 *   #'|      -> F_OR
 *   #'&&     -> F_LAND
 *   #'&&=    -> F_LAND_EQ
 *   #'&=     -> F_AND_EQ
 *   #'&      -> F_AND
 *   #'~      -> F_COMPL
 *   #'<=     -> F_LE
 *   #'<<=    -> F_LSH_EQ
 *   #'<<     -> F_LSH
 *   #'<      -> F_LT
 *   #'>=     -> F_GE
 *   #'>>=    -> F_RSH_EQ
 *   #'>>>=   -> F_RSHL_EQ
 *   #'>>>    -> F_RSHL
 *   #'>>     -> F_RSH
 *   #'>      -> F_GT
 *   #'==     -> F_EQ
 *   #'=      -> F_ASSIGN
 *   #'!=     -> F_NE
 *   #'!      -> F_NOT
 *   #'?!     -> F_BRANCH_WHEN_NON_ZERO
 *   #'?      -> F_BRANCH_WHEN_ZERO
 *   #'[..]   -> F_RANGE
 *   #'[..<]  -> F_NR_RANGE
 *   #'[<..]  -> F_RN_RANGE
 *   #'[<..<] -> F_RR_RANGE
 *   #'[..>]  -> F_NA_RANGE
 *   #'[>..]  -> F_AN_RANGE
 *   #'[<..>] -> F_RA_RANGE
 *   #'[>..<] -> F_AR_RANGE
 *   #'[>..>] -> F_AA_RANGE
 *   #'[..    -> F_NX_RANGE
 *   #'[<..   -> F_RX_RANGE
 *   #'[>..   -> F_AX_RANGE
 *   #'[,]    -> F_MAP_INDEX
 *   #'[      -> F_INDEX
 *   #'[<     -> F_RINDEX
 *   #'[>     -> F_AINDEX
 *   #'({     -> F_AGGREGATE
 *   #'([     -> F_M_CAGGREGATE
 *   #'.      -> F_S_INDEX
 *   #'->     -> F_SX_INDEX
 *   #'(<     -> F_S_AGGREGATE
 *
 * Note that all operators must have a instrs[].Default value of '0'.
 * If necessary, update the lex::init_lexer()::binary_operators[] to
 * include the operator values.
 */

{
    char c;
    int ret;

    switch(*symbol)
    {
    case '+':
        c = symbol[1];
        if (c == '=')
        {
            symbol++;
            ret = F_ADD_EQ;
            break;
        }
        else if (c == '+')
        {
            symbol++;
            ret = F_POST_INC;
            break;
        }
        ret = F_ADD;
        break;

    case '-':
        c = symbol[1];
        if (c == '=')
        {
            symbol++;
            ret = F_SUB_EQ;
            break;
        }
        else if (c == '-')
        {
            symbol++;
            ret = F_POST_DEC;
            break;
        }
        else if (c == '>')
        {
            symbol++;
            ret = F_SX_INDEX;
            break;
        }

        ret = F_SUBTRACT;
        break;

    case '*':
        if (symbol[1] == '=')
        {
            symbol++;
            ret = F_MULT_EQ;
            break;
        }
        ret = F_MULTIPLY;
        break;

    case '/':
        if (symbol[1] == '=')
        {
            symbol++;
            ret = F_DIV_EQ;
            break;
        }
        ret = F_DIVIDE;
        break;

    case '%':
        if (symbol[1] == '=')
        {
            symbol++;
            ret = F_MOD_EQ;
            break;
        }
        ret = F_MOD;
        break;

    case '.':
        ret = F_S_INDEX;
        break;

    case ',':
        ret = F_POP_VALUE;
        break;

    case '^':
        if (symbol[1] == '=')
        {
            symbol++;
            ret = F_XOR_EQ;
            break;
        }
        ret = F_XOR;
        break;

    case '|':
        c = *++symbol;
        if (c == '|')
        {
            if (symbol[1] == '=')
            {
                symbol++;
                ret = F_LOR_EQ;
            }
            else
                ret = F_LOR;
            break;
        }
        else if (c == '=')
        {
            ret = F_OR_EQ;
            break;
        }
        symbol--;
        ret = F_OR;
        break;

    case '&':
        c = *++symbol;
        if (c == '&')
        {
            if (symbol[1] == '=')
            {
                symbol++;
                ret = F_LAND_EQ;
            }
            else
                ret = F_LAND;
            break;
        }
        else if (c == '=')
        {
            ret = F_AND_EQ;
            break;
        }
        symbol--;
        ret = F_AND;
        break;

    case '~':
        ret = F_COMPL;
        break;

    case '<':
        c = *++symbol;
        if (c == '=')
        {
            ret = F_LE;
            break;
        }
        else if (c == '<')
        {
            if (symbol[1] == '=')
            {
                symbol++;
                ret = F_LSH_EQ;
                break;
            }
            ret = F_LSH;
            break;
        }
        symbol--;
        ret = F_LT;
        break;

    case '>':
        c = *++symbol;
        if (c == '=')
        {
            ret = F_GE;
            break;
        }
        else if (c == '>')
        {
            if (symbol[1] == '=')
            {
                symbol++;
                ret = F_RSH_EQ;
                break;
            }
            if (symbol[1] == '>')
            {
                symbol++;
                if (symbol[1] == '=')
                {
                    symbol++;
                    ret = F_RSHL_EQ;
                    break;
                }
                ret = F_RSHL;
                break;
            }
            ret = F_RSH;
            break;
        }
        symbol--;
        ret = F_GT;
        break;

    case '=':
        if (symbol[1] == '=')
        {
            symbol++;
            ret = F_EQ;
            break;
        }
        ret = F_ASSIGN;
        break;

    case '!':
        if (symbol[1] == '=')
        {
            symbol++;
            ret = F_NE;
            break;
        }
        ret = F_NOT;
        break;

    case '?':
        if (symbol[1] == '!')
        {
            symbol++;
            ret = F_BRANCH_WHEN_NON_ZERO;
            break;
        }
        ret = F_BRANCH_WHEN_ZERO;
        break;

    case '[':
        c = *++symbol;
        if (c == '<')
        {
            if (symbol[1] == '.' && symbol[2] == '.')
            {
                c = *(symbol+=3);
                if (c == ']')
                {
                    ret = F_RN_RANGE;
                    break;
                }
                else if (c == '>' && symbol[1] == ']')
                {
                    symbol++;
                    ret = F_RA_RANGE;
                    break;
                }
                else if (c == '<' && symbol[1] == ']')
                {
                    symbol++;
                    ret = F_RR_RANGE;
                    break;
                }
                symbol--;
                ret = F_RX_RANGE;
                break;
            }
            ret = F_RINDEX;
            break;
        }
        else if (c == '>')
        {
            if (symbol[1] == '.' && symbol[2] == '.')
            {
                c = *(symbol+=3);
                if (c == ']')
                {
                    ret = F_AN_RANGE;
                    break;
                }
                else if (c == '>' && symbol[1] == ']')
                {
                    symbol++;
                    ret = F_AA_RANGE;
                    break;
                }
                else if (c == '<' && symbol[1] == ']')
                {
                    symbol++;
                    ret = F_AR_RANGE;
                    break;
                }
                symbol--;
                ret = F_AX_RANGE;
                break;
            }
            ret = F_AINDEX;
            break;
        }
        else if (c == '.' && symbol[1] == '.')
        {
            c = *(symbol+=2);
            if (c == ']') {
                ret = F_RANGE;
                break;
            } else if (c == '>' && symbol[1] == ']') {
                symbol++;
                ret = F_NA_RANGE;
                break;
            } else if (c == '<' && symbol[1] == ']') {
                symbol++;
                ret = F_NR_RANGE;
                break;
            }
            symbol--;
            ret = F_NX_RANGE;
            break;
        }
        else if (c == ',' && symbol[1] == ']')
        {
            symbol++;
            ret = F_MAP_INDEX;
            break;
        }
        symbol--;
        ret = F_INDEX;
        break;

    case '(':
        c = *++symbol;
        if (c == '{')
        {
            ret = F_AGGREGATE;
            break;
        }
        else if (c == '[')
        {
            ret = F_M_CAGGREGATE;
            break;
        }
        else if (c == '<')
        {
            ret = F_S_AGGREGATE;
            break;
        }

        symbol--;
        /* FALL THROUGH */
    default:
        ret = -1;
        symbol--;
        break;
    }

    /* Symbol is not an operator */
    *endp = symbol+1;
    return ret;
} /* symbol_operator() */

/*-------------------------------------------------------------------------*/
static INLINE int
symbol_resword (ident_t *p)

/* This function implements the resword lookup for closures.
 *
 * If ident <p> is a reserved word with a closure representation, return
 * the corresponding instruction code:
 *
 *   #'if          -> F_BRANCH_WHEN_ZERO     
 *   #'do          -> F_BBRANCH_WHEN_NON_ZERO
 *   #'while       -> F_BBRANCH_WHEN_ZERO    
 *   #'foreach     -> F_FOREACH              
 *   #'continue    -> F_BRANCH               
 *   #'default     -> F_CSTRING0             
 *   #'switch      -> F_SWITCH               
 *   #'break       -> F_BREAK                
 *   #'return      -> F_RETURN               
 *   #'catch       -> F_CATCH                
 *
 * If ident <p> is not a reserved word, or a word without closure
 * representation, return 0.
 */

{
    int code = 0;

    if (p->type != I_TYPE_RESWORD)
        return 0;

    switch(p->u.code)
    {
    default:
        /* Unimplemented reserved word */
        code = 0;
        break;

    case L_IF:
        code = F_BRANCH_WHEN_ZERO;
        break;

    case L_DO:
        code = F_BBRANCH_WHEN_NON_ZERO;
        break;

    case L_WHILE:
        /* the politically correct code was already taken, see above. */
        code = F_BBRANCH_WHEN_ZERO;
        break;

    case L_FOREACH:
        code = F_FOREACH;
        break;

    case L_CONTINUE:
        code = F_BRANCH;
        break;

    case L_DEFAULT:
        code = F_CSTRING0;
        break;

    case L_SWITCH:
        code = F_SWITCH;
        break;
    case L_BREAK:
        code = F_BREAK;
        break;
    case L_RETURN:
        code = F_RETURN;
        break;
    case L_CATCH:
        code = F_CATCH;
        break;
    }

    return code;
} /* symbol_resword() */

/*-------------------------------------------------------------------------*/
void
symbol_efun_str (const char * str, size_t len, svalue_t *sp, efun_override_t is_efun, bool privileged)

/* This function implements the efun/operator/sefun part of efun
 * symbol_function().
 *
 * It is also called by parse_command to lookup the (simul)efuns find_living()
 * and find_player() at runtime, and by restore_svalue().
 *
 * The function takes the string <str> of length <len> and looks up the named
 * efun, sefun or operator. If the efun/operator is found, the value <sp> is
 * turned into the proper closure value, otherwise it is set to the numeric
 * value 0.  If <is_efun> is OVERRIDE_SEFUN, <str> is resolved as an efun even
 * if it doesn't contain the 'efun::' prefix, similar with OVERRIDE_SEFUN.
 *
 * inter_sp must be set properly before the call.
 *
 * if <privileged> is true, then no check for nomask simul-efuns is done.
 *
 * Accepted symbols are:
 *
 *   #'<operator>: see lex::symbol_operator()
 *
 *   #'if          -> F_BRANCH_WHEN_ZERO       +CLOSURE_OPERATOR
 *   #'do          -> F_BBRANCH_WHEN_NON_ZERO  +CLOSURE_OPERATOR
 *   #'while       -> F_BBRANCH_WHEN_ZERO      +CLOSURE_OPERATOR
 *   #'foreach     -> F_FOREACH                +CLOSURE_OPERATOR
 *   #'continue    -> F_BRANCH                 +CLOSURE_OPERATOR
 *   #'default     -> F_CSTRING0               +CLOSURE_OPERATOR
 *   #'switch      -> F_SWITCH                 +CLOSURE_OPERATOR
 *   #'break       -> F_BREAK                  +CLOSURE_OPERATOR
 *   #'return      -> F_RETURN                 +CLOSURE_OPERATOR
 *   #'catch       -> F_CATCH                  +CLOSURE_OPERATOR
 *
 *   #'<efun>      -> F_<efun>                 +CLOSURE_EFUN
 *   #'<sefun>     -> <function-index>         +CLOSURE_SIMUL_EFUN
 */

{
    efun_override_t efun_override = is_efun;

    /* If the first character is alphanumeric, the string names a function,
     * otherwise an operator.
     */
    if (utf8_isalunum(str))
    {
        /* It is a function or keyword.
         */

        ident_t *p;

        /* Take care of an leading efun override */

        if ( len >= 6 && !strncmp(str, "efun::", 6) )
        {
            str += 6;
            len -= 6;
            efun_override = OVERRIDE_EFUN;
        }
        else if ( len >= 7 && !strncmp(str, "sefun::", 7) )
        {
            str += 7;
            len -= 7;
            efun_override = OVERRIDE_SEFUN;
        }

        /* Lookup the identifier in the string in the global table
         * of identifiers.
         */
        if ( !(p = make_shared_identifier_n(str, len, I_TYPE_GLOBAL, 0)) )
        {
            outofmem(len, "identifier");
        }

        /* Loop through the possible multiple definitions.
         */
        while (p->type > I_TYPE_GLOBAL)
        {
            /* Is it a reserved word? */
            if (p->type == I_TYPE_RESWORD
             && efun_override != OVERRIDE_SEFUN)
            {
                int code = symbol_resword(p);

                if (!code)
                {
                    /* Unimplemented reserved word */
                    if ( NULL != (p = p->inferior) )
                        continue;
                    goto undefined_function;
                }

                /* Got the reserved word: return the closure value */

                sp->type = T_CLOSURE;
                sp->x.closure_type = (short)(code + CLOSURE_OPERATOR);
                sp->u.ob = ref_object(current_object, "symbol_efun");
                return;
            }
            if ( !(p = p->inferior) )
                break; /* Found a valid definition */
        }

        /* It is a real identifier */

        if (!p || p->type < I_TYPE_GLOBAL
         || (( efun_override == OVERRIDE_EFUN || p->u.global.sim_efun == I_GLOBAL_SEFUN_OTHER )
          && ( efun_override == OVERRIDE_SEFUN  || 
              ( p->u.global.efun == I_GLOBAL_EFUN_OTHER
#ifdef USE_PYTHON
             && !is_python_efun(p)
#endif
              )))
           )
        {
            /* But it's a (new) local identifier or a non-existing function */
            if (p && p->type == I_TYPE_UNKNOWN)
                free_shared_identifier(p);

undefined_function:
            put_number(sp, 0);
            return;
        }

        /* Attempting to override a 'nomask' simul efun?
         * Check it with a privilege violation.
         */
        if (!privileged && efun_override == OVERRIDE_EFUN && p->u.global.sim_efun != I_GLOBAL_SEFUN_OTHER
         && simul_efunp[p->u.global.sim_efun].flags & TYPE_MOD_NO_MASK)
        {
            svalue_t *res;

            push_ref_string(inter_sp, STR_NOMASK_SIMUL_EFUN);
            push_ref_valid_object(inter_sp, current_object, "nomask simul_efun");
            push_ref_string(inter_sp, p->name);
            res = apply_master(STR_PRIVILEGE, 3);

            if (!res || res->type != T_NUMBER || res->u.number < 0)
            {
                /* Override attempt is fatal */
                errorf(
                  "Privilege violation: nomask simul_efun %s\n",
                  get_txt(p->name)
                );
            }
            else if (!res->u.number)
            {
                /* Override attempt not fatal, but rejected nevertheless */
                efun_override = OVERRIDE_NONE;
            }
        }

        /* Symbol is ok - create the closure value */

        sp->type = T_CLOSURE;
        if (efun_override != OVERRIDE_EFUN && p->u.global.sim_efun != I_GLOBAL_SEFUN_OTHER)
        {
            /* Handle non-overridden simul efuns */

            sp->x.closure_type = (short)(p->u.global.sim_efun + CLOSURE_SIMUL_EFUN);
            sp->u.ob = ref_object(current_object, "symbol_efun");
        }
#ifdef USE_PYTHON
        else if (is_python_efun(p))
        {
            sp->x.closure_type = (short)(p->u.global.python_efun + CLOSURE_PYTHON_EFUN);
            sp->u.ob = ref_object(current_object, "symbol_efun");
        }
#endif
        else
        {
            /* Handle efuns (possibly aliased).
             * We know that p->u.global.efun != I_GLOBAL_EFUN_OTHER here.
             */
            sp->x.closure_type = (short)(p->u.global.efun + CLOSURE_EFUN);
            if (sp->x.closure_type > LAST_INSTRUCTION_CODE + CLOSURE_EFUN)
                sp->x.closure_type = (short)(CLOSURE_EFUN +
                  efun_aliases[
                    sp->x.closure_type - CLOSURE_EFUN - LAST_INSTRUCTION_CODE - 1]);
            sp->u.ob = ref_object(current_object, "symbol_efun");
        }
    }
    else
    {
        int i;
        const char *end;

        i = symbol_operator(str, &end);
        /* If there was a valid operator with trailing junk, *end, but i >= 0.
         * On the other hand, if we passed the empty string, i < 0, but !*end.
         * Thus, we have to test for (*end || i < 0) .
         */

        if (*end || i < 0)
        {
            put_number(sp, 0);
            return;
        }
        sp->type = T_CLOSURE;
        if (instrs[i].Default == -1) {
            sp->x.closure_type = (short)(i + CLOSURE_OPERATOR);
        } else {
            sp->x.closure_type = (short)(i + CLOSURE_EFUN);
        }
        sp->u.ob = ref_object(current_object, "symbol_efun");
    }
} /* symbol_efun_str() */

/*-------------------------------------------------------------------------*/
void
symbol_efun (string_t *name, svalue_t *sp)

/* This function is a wrapper around symbol_efun_str(), taking a regular
 * string <name> as argument.
 */

{
    symbol_efun_str(get_txt(name), mstrsize(name), sp, OVERRIDE_NONE, false);
} /* symbol_efun() */

/*-------------------------------------------------------------------------*/
source_file_t *
new_source_file (const char * name, source_loc_t * parent)

/* Create a new source_file structure for file <name>.
 *
 * If <name> is non-NULL, a new string is allocated and the content of <name>
 * is copied. If <name> is NULL, the caller has to set the filename in
 * the returned structure.
 *
 * If <parent> is non-NULL, it denotes the parent file location this source was
 * included from.
 *
 * Result is the new structure, or NULL if out of memory.
 *
 * Once allocated, the structure can be removed only through the general
 * cleanup routined cleanup_source_files().
 */

{
    source_file_t * rc;

    rc = xalloc(sizeof(*rc));
    if (!rc)
        return NULL;
    if (name)
    {
        rc->name = string_copy(name);
        if (!rc->name)
        {
            xfree(rc);
            return NULL;
        }
    }
    else
        rc->name = NULL;
    
    if (parent)
        rc->parent = *parent;
    else
    {
        rc->parent.file = NULL;
        rc->parent.line = 0;
    }

    rc->next = src_file_list;
    src_file_list = rc;

    return rc;
} /* new_source_file() */

/*-------------------------------------------------------------------------*/
static void
cleanup_source_files (void)

/* Deallocate all listed source_file structures.
 */

{
    source_file_t * this;

    while ((this = src_file_list) != NULL)
    {
        src_file_list = this->next;

        if (this->name)
            xfree(this->name);
        xfree(this);
    }

    current_loc.file = NULL;
    current_loc.line = 0;
} /* cleanup_source_files() */

/*-------------------------------------------------------------------------*/
void
init_global_identifier (ident_t * ident, Bool bVariable)

/* The (newly created or to be reused) identifier <ident> is set up
 * to be a global identifier, with all the .global.* fields set to
 * a suitable default. The caller has to fill in the information specifying
 * what kind of global this is.
 *
 * <bVariable> is to be TRUE if the caller intends to use the identifier
 * for a (local or global) variable or lfun; and FALSE if it is for a
 * efun/sefun.
 *
 * The function is rather small, but having it here makes it easier to
 * guarantee that all fields are set to a proper default.
 */

{
    ident->type = I_TYPE_GLOBAL;
    ident->u.global.function  = I_GLOBAL_FUNCTION_OTHER;
    if (bVariable)
        ident->u.global.variable  = I_GLOBAL_VARIABLE_OTHER;
    else
        ident->u.global.variable = I_GLOBAL_VARIABLE_FUN;
    ident->u.global.efun     = I_GLOBAL_EFUN_OTHER;
    ident->u.global.sim_efun = I_GLOBAL_SEFUN_OTHER;
    ident->u.global.struct_id = I_GLOBAL_STRUCT_NONE;
#ifdef USE_PYTHON
    ident->u.global.python_efun = I_GLOBAL_PYTHON_EFUN_OTHER;
#endif

} /* init_global_identifier() */

/*-------------------------------------------------------------------------*/
ident_t *
lookfor_shared_identifier (const char *s, size_t len, int n, int depth, bool bCreate, bool bExactDepth)

/* Aliases: make_shared_identifier():   bCreate passed as true, bExactDepth as false
 *          find_shared_identifier():   bCreate and bExactDepth passed as false
 *          insert_shared_identifier(): bCreate and bExactDepth passed true
 *
 * Find and/or add identifier <s> with size <len> of type <n> to the
 * ident_table, and return a pointer to the found/generated struct ident.
 * Local identifiers (<n> == I_TYPE_LOCAL) are additionally distinguished
 * by their definition <depth>.
 *
 * If bCreate is FALSE, the function just checks if the given identfier
 * exists in the table. The identifier is considered found, if there
 * is an entry in the table for this very name, and with a type equal
 * or greater than <n>. If <n> is LOCAL and the found identifier is LOCAL
 * as well, the identifier is considered found if <depth> is equal or smaller
 * than the depth of the found identifier. If <bExactDepth> is true, then
 * the depth must match exactly. The result is the pointer to the found
 * identifier, or NULL if not found.
 *
 * If bCreate is TRUE, the identifier is created if not found. If an
 * identifier with the same name but a lower type exists in the table,
 * it is shifted down: a new entry for this name created and put into the
 * table, the original entry is referenced by the .inferior pointer in the
 * new entry. The same happens when a new LOCAL of greater depth is
 * added to an existing LOCAL of smaller depth. If bExactDepth is true,
 * the depth must match exactly, otherwise an identifier will be inserted
 * into the hierarchy. Newly generated entries have their type set to
 * I_TYPE_UNKNOWN regardless of <n>. The result is the pointer to the
 * found/new entry, or NULL when out of memory.
 */

{
    ident_t  *curr, *prev;
    hash16_t h;
    string_t *str;

#if defined(LEXDEBUG)
    printf("%s lookfor_shared_identifier called: %.*s\n", time_stamp(), len, s);
#endif

    h = identhash_n(s, len);  /* the identifiers hash code */

    /* look for the identifier in the table */

    curr = ident_table[h];
    prev = NULL;
    while (curr)
    {
#if defined(LEXDEBUG)
        printf("%s checking %s.\n", time_stamp(), get_txt(curr->name));
#endif
        if (mstrsize(curr->name) == len
         && !strncmp(get_txt(curr->name), s, len)) /* found it */
        {
#if defined(LEXDEBUG)
            printf("%s  -> found.\n", time_stamp());
#endif
            /* Move the found entry to the head of the chain */
            if (prev) /* not at head of chain */
            {
                prev->next = curr->next;
                curr->next = ident_table[h];
                ident_table[h] = curr;
            }

            /* If the found entry is of inferior type, shift it down */
            if (n > curr->type
             || (   I_TYPE_LOCAL == curr->type && I_TYPE_LOCAL == n
                 && depth > curr->u.local.depth)
               )
            {
                if (bCreate)
                {
                    ident_t *inferior = curr;

#if defined(LEXDEBUG)
                    printf("%s     shifting down inferior.\n", time_stamp());
#endif
                    curr = xalloc(sizeof *curr);
                    if ( NULL != curr )
                    {
                        curr->name = ref_mstring(inferior->name);
                        curr->next = inferior->next;
                        curr->type = I_TYPE_UNKNOWN;
                        curr->inferior = inferior;
                        curr->hash = h;
                        ident_table[h] = curr;
                    }
                }
                else
                    curr = NULL;
            }

            if (bExactDepth
             && I_TYPE_LOCAL == curr->type && I_TYPE_LOCAL == n
             && depth != curr->u.local.depth)
            {
                /* We have an identifier with a greater depth
                 * than was requested. Look for an inferior identifier.
                 */
                do
                {
                    prev = curr;
                    curr = curr->inferior;
                }
                while (curr->type == I_TYPE_LOCAL && curr->u.local.depth > depth);

                if (curr->type != I_TYPE_LOCAL || curr->u.local.depth != depth)
                {
                    /* We haven't found the requested identifier. */
                    if (bCreate)
                    {
                        curr = xalloc(sizeof *curr);
                        if ( NULL != curr )
                        {
                            curr->name = ref_mstring(prev->name);
                            curr->type = I_TYPE_UNKNOWN;
                            curr->inferior = prev->inferior;
                            curr->hash = h;
                            prev->inferior = curr;
                        }
                    }
                    else
                        curr = NULL;
                }
            }

            /* Return the found (or generated) entry */
            return curr;
        }

        prev = curr;
        curr = curr->next;
    }

    if (bCreate)
    {
        /* Identifier is not in table, so create a new entry */

        str = new_n_unicode_tabled(s, len);
        if (!str)
            return NULL;
        curr = xalloc(sizeof *curr);
        if (!curr)
        {
            free_mstring(str);
            return NULL;
        }

        curr->name = str;
        curr->next = ident_table[h];
        curr->type = I_TYPE_UNKNOWN;
        curr->inferior = NULL;
        curr->hash = h;
        ident_table[h] = curr;
    }
    /* else curr is NULL */

    return curr;
} /* lookfor_shared_identifier() */

/*-------------------------------------------------------------------------*/
ident_t *
make_global_identifier (char *s, int n)

/* Create an identifier <s> on level I_TYPE_GLOBAL, after searching for it
 * using type <n>.
 *
 * The difference to make_shared_identifier() is that if an identifier for
 * this name already exists and is of higher level than I_TYPE_GLOBAL (e.g.
 * somebody created a #define for this name), the function will insert
 * an appropriate I_TYPE_GLOBAL entry into the inferior list.
 *
 * Result is the pointer to the identifier, or NULL when out of memory
 * (yyerror() is called in that situation, too).
 */

{
    ident_t *ip, *q;

    ip = make_shared_identifier(s, n, 0);
    if (!ip)
    {
        yyerrorf("Out of memory: identifier '%s'", s);
        return NULL;
    }

    if (ip->type > I_TYPE_GLOBAL)
    {
        /* Somebody created a #define with this name.
         * Back-insert an ident-table entry.
         */
        do {
            q = ip;
            ip = ip->inferior;
        } while (ip && ip->type > I_TYPE_GLOBAL);

        if (!ip)
        {
            ip = xalloc(sizeof(ident_t));
            if (!ip) {
                yyerrorf("Out of memory: identifier (%zu bytes)",
                         sizeof(ident_t));
                return NULL;
            }
            ip->name = ref_mstring(q->name);
            ip->type = I_TYPE_UNKNOWN;
            ip->inferior = NULL;
            ip->hash = q->hash;
            q->inferior = ip;
        }
    }

    return ip;
} /* make_global_identifier() */

/*-------------------------------------------------------------------------*/
static INLINE void
free_identifier (ident_t *p)

/* Deallocate the identifier <p> which must not be in any list or table
 * anymore.
 * It is a fatal error if it can't be found.
 */

{
    free_mstring(p->name);
    xfree(p);
} /* free_identifier() */

/*-------------------------------------------------------------------------*/
static INLINE void
unlink_shared_identifier (ident_t *p)

/* Unlink the identifier <p> (which may be an inferior entry ) from the
 * identifier table.
 * It is a fatal error if it can't be found.
 */

{
    ident_t *curr, **q;
    hash16_t  h;
    string_t *s;

    h = p->hash;

    q = &ident_table[h];
    curr = *q;
    s = p->name;

#if defined(LEXDEBUG)
    printf("%s unlinking '%s'\n", time_stamp(), get_txt(s));
    fflush(stdout);
#endif

    /* Look for the hashed entry with the same name */

    while (curr)
    {
        if (curr->name == s
#ifdef DEBUG
         || mstreq(curr->name, s)
#endif

           ) /* found matching name */
        {
            ident_t *first = curr;

            /* Search the list of inferiors for entry <p> */

            while (curr)
            {
                if (curr == p) /* this is the right one */
                {
                    /* Remove the entry from the inferior list */

                    if (first == curr)
                    {
                        if (curr->inferior)
                        {
                            curr->inferior->next = curr->next;
                            *q = curr->inferior;
                            return; /* success */
                        }
                        *q = curr->next;
                        return;
                    }

                    *q = curr->inferior;
                    return; /* success */
                }
                q = &curr->inferior;
                curr = *q;
            }
            fatal("free_shared_identifier: entry '%s' not found!\n"
                 , get_txt(p->name));
            /* NOTREACHED */
        }

        q = &curr->next;
        curr = *q;
    } /* not found */

    fatal("free_shared_identifier: name '%s' not found!\n", get_txt(p->name));
    /* NOTREACHED */
} /* unlink_shared_identifier() */

/*-------------------------------------------------------------------------*/
void
free_shared_identifier (ident_t *p)

/* Remove the identifier <p> (which may be an inferior entry ) from the
 * identifier table.
 * It is a fatal error if it can't be found.
 */

{
#if defined(LEXDEBUG)
    printf("%s freeing '%s'\n", time_stamp(), get_txt(p->name));
    fflush(stdout);
#endif

    unlink_shared_identifier(p);
    free_identifier(p);
} /* free_shared_identifier() */

/*-------------------------------------------------------------------------*/
static void
realloc_defbuf (void)

/* Increase the size of defbuf[] (unless it would exceed MAX_TOTAL_BUF).
 * The old content of defbuf[] is copied to the end of the new buffer.
 * outp is corrected to the new position, other pointers into defbuf
 * become invalid.
 */

{
    char * old_defbuf = defbuf;
    size_t old_defbuf_len = defbuf_len;
    char * old_outp = outp;
    ptrdiff_t outp_off;

    if (MAX_TOTAL_BUF <= defbuf_len)
      return;

    outp_off = &defbuf[defbuf_len] - outp;

    /* Double the current size of defbuf, but top off at MAX_TOTAL_BUF. */
    if (defbuf_len > (MAX_TOTAL_BUF >> 1) )
    {
        defbuf_len = MAX_TOTAL_BUF;
    } else {
        defbuf_len <<= 1;
    }
    if (comp_flag)
        fprintf(stderr, "%s (reallocating defbuf from %zu (%td left) to %lu) "
               , time_stamp(), old_defbuf_len, (ptrdiff_t)(old_outp-defbuf)
               , defbuf_len);
    defbuf = xalloc(defbuf_len);
    memcpy(defbuf+defbuf_len-old_defbuf_len, old_defbuf, old_defbuf_len);
    xfree(old_defbuf);
    outp = &defbuf[defbuf_len] - outp_off;
} /* realloc_defbuf() */

/*-------------------------------------------------------------------------*/
static void
set_input_source (int fd, const char* fname, string_t * str)

/* Set the current input source to <fd>/<str>.
 * If <str> is given, it will be referenced.
 */

{
    yyin.convbuf = NULL;
    yyin.convbytes[0] = 0;

    yyin.fd = fd;
    if (fd != -1)
    {
        /* Initialize the converter. */
        string_t *encoding = NULL;

        if (driver_hook[H_FILE_ENCODING].type == T_STRING)
        {
            encoding = driver_hook[H_FILE_ENCODING].u.str;
        }
        else if (driver_hook[H_FILE_ENCODING].type == T_CLOSURE)
        {
            svalue_t *svp;

            /* Setup and call the closure */
            push_c_string(inter_sp, fname);
            svp = secure_apply_lambda(driver_hook+H_FILE_ENCODING, 1);

            if (svp && svp->type == T_STRING)
                encoding = svp->u.str;
        }

        yyin.cd = iconv_open("utf-8", encoding == NULL ? "ascii" : get_txt(encoding));
        if (!iconv_valid(yyin.cd))
        {
            if (errno == EINVAL)
                lexerrorf("Unsupported encoding '%s'.", get_txt(encoding));
            else
                lexerror(strerror(errno));
        }
        else
        {
            yyin.convbuf = xalloc(CONVBUFSIZE);

            if (yyin.convbuf == NULL)
            {
                iconv_close(yyin.cd);
                yyin.cd = iconv_init();
                lexerror("Out of memory while allocating file buffer.");
            }
            else
            {
                yyin.convstart = yyin.convbuf + CONVBUFSIZE;
                yyin.convleft = 0;
            }
        }
    }
    else
        yyin.cd = iconv_init();

    yyin.str = str ? ref_mstring(str) : NULL;
    yyin.current = 0;
} /* set_input_source() */

/*-------------------------------------------------------------------------*/
static void
close_input_source (bool dontclosefd)

/* Close the current input source: a file is closed, a string is deallocated
 * If <dontclosefd> is true, the file descriptor shall stay open.
 */

{
    if (yyin.fd != -1)
    {
        if (!dontclosefd)
            close(yyin.fd);
        yyin.fd = -1;
    }
    if (iconv_valid(yyin.cd))
    {
        iconv_close(yyin.cd);
        yyin.cd = iconv_init();
    }
    if (yyin.convbuf != NULL)
    {
        xfree(yyin.convbuf);
        yyin.convbuf = NULL;
    }
    if (yyin.str != NULL)
    {
        free_mstring(yyin.str);
        yyin.str = NULL;
    }
    yyin.current = 0;
} /* close_input_source() */

/*-------------------------------------------------------------------------*/
static void
lexencodingerror (char* pos, char* msg)

/* An error during decoding the file.
 * We temporary set the current_loc to the position where
 * the error occurred.
 */

{
    int forward_lines = 0;
    char *p = linebufstart;
    char *linestart = p;

    for (; p < pos; p++)
        if (*p == '\n')
        {
            forward_lines++;
            linestart = p;
        }

    current_loc.line += forward_lines;
    lex_error_pos = pos - linestart;

    lexerror(msg);

    current_loc.line -= forward_lines;
    lex_error_pos = -1;
} /* lexencodingerror() */

/*-------------------------------------------------------------------------*/
static /* NO inline */ char *
_myfilbuf (void)

/* Read the next MAXLINE bytes from the input source <yyin> and store
 * them in the input-buffer. If there were the beginning of an incomplete
 * line left in the buffer, they are copied right before linebufstart.
 * The end of the last complete line in the buffer is marked with a '\0'
 * sentinel, or, if the file is exhausted, the end of data is marked
 * with the CHAR_EOF char.
 *
 * outp is set to point to the new data (which may be the copied remnants
 * from the incomplete line) and also returned as result.
 *
 * The function must not be called unless all lines in the buffer have
 * been processed. This macro */

#define myfilbuf() (*outp?0:_myfilbuf())

 /* takes care of that.
  */

{
    int i = 0;
    char *p;

    /* Restore the data clobbered by the old sentinel */
    *outp = saved_char;

    /* Copy any remnants of an incomplete line before the buffer begin
     * and reset outp.
     */
    if (linebufend < outp)
        fatal("(lex.c) linebufend %p < outp %p\n", linebufend, outp);
    if (linebufend - outp)
        memcpy(outp-MAXLINE, outp, (size_t)(linebufend - outp));
    outp -= MAXLINE;

    *(outp-1) = '\n'; /* so an ungetc() gives a sensible result */

    /* Read the next block of data */
    p = linebufstart; /* == linebufend - MAXLINE */
    if (yyin.fd != -1)
    {
        if (iconv_valid(yyin.cd))
        {
            size_t outleft = MAXLINE;
            char*  outptr  = p;
            bool   ineof   = false;

            assert(yyin.convbuf != NULL);

            /* First look at any leftovers. */
            for (int num = 0; num < 4 && yyin.convbytes[num]; num++, outptr++, outleft--)
                *outptr = yyin.convbytes[num];
            yyin.convbytes[0] = 0;

            while (true)
            {
                size_t rc = (size_t)-1;
                bool fillbuf = false;

                if (yyin.convleft != 0)
                    rc = iconv(yyin.cd, &yyin.convstart, &yyin.convleft, &outptr, &outleft);
                else if (ineof)
                    rc = iconv(yyin.cd, NULL, NULL, &outptr, &outleft);
                else
                    fillbuf = true;

                if (rc == (size_t)-1)
                {
                    i = outptr - p;

                    /* Incomplete sequence, need to fill the input buffer. */
                    if (fillbuf || errno == EINVAL)
                    {
                        if (yyin.convstart + yyin.convleft == yyin.convbuf + CONVBUFSIZE)
                        {
                            size_t j;

                            if (yyin.convleft > 0)
                                memmove(yyin.convbuf, yyin.convstart, yyin.convleft);

                            j = read(yyin.fd, yyin.convbuf + yyin.convleft, CONVBUFSIZE - yyin.convleft);

                            yyin.convstart = yyin.convbuf;
                            yyin.convleft += j;

                            if (j != 0)
                                continue;
                        }

                        if (fillbuf)
                        {
                            ineof = true;
                            continue;
                        }
                        else
                        {
                            lexencodingerror(outptr, "Unexpected end of file");
                            break;
                        }
                    }

                    /* Output buffer to small? Then we are finished for now. */
                    if (errno == E2BIG)
                    {
                        /* i needs to be MAXLINE, so we try to convert one
                         * additional codepoint und put the remaining bytes
                         * into yyin.convbytes.
                         */
                        if (i < MAXLINE)
                        {
                            size_t j = 0, k = 0;
                            outleft = 4;
                            outptr  = yyin.convbytes;

                            iconv(yyin.cd, &yyin.convstart, &yyin.convleft, &outptr, &outleft);

                            for (; j < 4-outleft && i < MAXLINE; i++, j++)
                                p[i] = yyin.convbytes[j];
                            for (k = 0; j < 4-outleft; j++, k++)
                                yyin.convbytes[k] = yyin.convbytes[j];
                            yyin.convbytes[k] = 0;
                        }
                        break;
                    }

                    /* Invalid sequence, abort. */
                    if (errno == EILSEQ)
                    {
                        lexencodingerror(outptr, "Invalid character sequence");
                        break;
                    }

                    /* Other error, abort also. */
                    lexencodingerror(outptr, strerror(errno));
                    break;
                }

                if (ineof)
                    break;
            }
        }
        else
            i = read(yyin.fd, p, MAXLINE);
    }
    else
    {
        i = mstrsize(yyin.str) - yyin.current;

        if (i > MAXLINE)
            i = MAXLINE;

        memcpy(p, get_txt(yyin.str)+yyin.current, i);
        yyin.current += i;
    }

    if (i < MAXLINE)
    {
        /* End of file or error: put in the final EOF marker */

        if (i < 0)
        {
            i = 0;
        }

        p += i;
        if ((p - outp) ? (p[-1] != '\n') : (current_loc.line == 1))
            *p++ = '\n';
        *p++ = CHAR_EOF;
        return outp;
    }

    /* Buffer filled: mark the last line with the '\0' sentinel */
    p += i;
    while (*--p != '\n') NOOP; /* find last newline */
    if (p < linebufstart)
    {
        lexerror("line too long");
        *(p = linebufend-1) = '\n';
    }
    p++;
    saved_char = *p;
    *p = '\0';

    return outp;
} /* _myfilbuf() */

/*-------------------------------------------------------------------------*/
static void
add_input (char *p)

/* Copy the text <p> into defbuf[] right before the current position of
 * outp and set outp back to point at the beginning of the new text.
 *
 * Main use is by the macro expansion routines.
 */
{
    size_t l = strlen(p);

#if defined(LEXDEBUG)
    if (l > 0)
        fprintf(stderr, "%s add '%s'\n", time_stamp(), p);
#endif
    if ((ptrdiff_t)l > outp - &defbuf[10])
    {
        lexerror("Macro expansion buffer overflow");
        return;
    }

    outp -= l;
    memcpy(outp, p, l);
}

/*-------------------------------------------------------------------------*/
static INLINE char
mygetc (void)

/* Retrieve the next character from the file input buffer.
 */

{
#if 0
    fprintf(stderr, "c='%c' %x, ", *outp, *outp);
#endif
#if defined(LEXDEBUG)
    putc(*outp, stderr);
    fflush(stderr);
#endif
    return *outp++;
}

/*-------------------------------------------------------------------------*/
static INLINE void
myungetc (char c)

/* Store character <c> in the file input buffer so the next mygetc()
 * can retrieve it.
 */

{
    *--outp = c;
}

/*-------------------------------------------------------------------------*/
static INLINE Bool
gobble (char c)

/* Skip the next character in the input buffer if it is <c> and return true.
 * If the next character is not <c>, don't advance in the buffer and
 * return false.
 */

{
    if (c == *outp)
    {
        outp++;
        return MY_TRUE;
    }

    return MY_FALSE;
}

/*-------------------------------------------------------------------------*/
static void
lexerrorf (char *format, ...)

/* Generate an lexerror() using printf()-style arguments.
 */

{
    va_list va;
    char buff[5120];
    char fixed_fmt[1000];

    format = limit_error_format(fixed_fmt, sizeof(fixed_fmt), format);
    va_start(va, format);
    vsprintf(buff, format, va);
    va_end(va);
    lexerror(buff);
} /* lexerrorf() */

/*-------------------------------------------------------------------------*/
static void
lexerror (char *s)

/* The lexer encountered fatal error <s>. Print the error via yyerror()
 * and set lex_fatal.
 */

{
    yyerror(s);
    lex_fatal = MY_TRUE;
}

/*-------------------------------------------------------------------------*/
static void
warn_trailing_chars (const char* stmt)

/* Give a warning (or error when pedantic) about trailing characters
 * in a preprocessor statement.
 */

{
    if (pragma_pedantic)
        yyerrorf("Unrecognized #%s (trailing characters)", stmt);
    else
        yywarnf("Unrecognized #%s (trailing characters)", stmt);
} /* warn_trailing_chars() */

/*-------------------------------------------------------------------------*/
static int
wordcmp (const char* word, const char* against, size_t wordlen)

/* Compares the word <wordlen> bytes of <word> against <against>.
 * <against> is a null-terminated string and must be of size <wordlen>,
 * otherwise the comparison fails.
 * Returns 0 for a match and values smaller or greater than zero,
 * if <word> is smaller or greater than <against>.
 */

{
    int result = strncmp(word, against, wordlen);
    if (result)
        return result;
    return wordlen - strlen(against);
} /* wordcmp() */

/*-------------------------------------------------------------------------*/
static Bool
skip_to (char *token, char *atoken)

/* Skip the file linewise until one of the following preprocessor statements
 * is encountered:
 *   #<token> : returns true, outp is set to the following line.
 *   #<atoken>: returns false, outp is set to the following line.
 *   #elif    : returns false, the statement is rewritten to #if and
 *                outp is set to point to the '#' in the new statement.
 * If an end of file occurs, an error is generated and the function returns
 * true after setting outp to the character before the CHAR_EOF.
 *
 * Nested #if ... #endif blocks are skipped altogether.
 *
 * <atoken> may be the NULL pointer and is ignored in that case.
 */

{
    char *p;  /* Local copy of outp */
    char *q;  /* The start of the preprocessor statement */
    char c;
    char nl = '\n';
    size_t len;
    int nest; /* Current nesting depth of #if...#endif blocks */

    p = outp;

    for (nest = 0; ; ) {
        current_loc.line++;
        total_lines++;
        c = *p++;

        if (c == '#')
        {
            /* Parse the preprocessor statement */

            /* Set q to the first non-blank character of the keyword */
            q = p = skip_white(p);

            /* Mark the end of the preprocessor keyword with \0 */
            p = skip_alunum(p);
            len = p - q; /* characters to compare. */

            /* Set p to the first character of the next line */
            p = skip_line(p);

            /* Evaluate the token at <q> */

            if ((wordcmp(q, "if", len) == 0)
             || (wordcmp(q, "ifdef", len) == 0)
             || (wordcmp(q, "ifndef", len) == 0))
            {
                nest++;
            }
            else if (nest > 0)
            {
                if (wordcmp(q, "endif", len) == 0)
                    nest--;
            }
            else
            {
                if (wordcmp(q, token, len) == 0)
                {
                    char *end = skip_white(q+len);
                    while (end + 1 < p)
                    {
                        if (end[0] == '/' && end[1] == '/')
                            break;
                        if (end[0] == '/' && end[1] == '*')
                        {
                            outp = end + 2;
                            skip_comment();
                            end = skip_white(outp);
                            continue;
                        }

                        warn_trailing_chars(token);
                        break;
                    }

                    outp = p;
                    if (!*p)
                        _myfilbuf();

                    return MY_TRUE;
                }
                else if (atoken)
                {
                    if (wordcmp(q, atoken, len) == 0)
                    {
                        char *end = skip_white(q+len);
                        while (end + 1 < p)
                        {
                            if (end[0] == '/' && end[1] == '/')
                                break;
                            if (end[0] == '/' && end[1] == '*')
                            {
                                outp = end + 2;
                                skip_comment();
                                end = skip_white(outp);
                                continue;
                            }

                            warn_trailing_chars(atoken);
                            break;
                        }

                        outp = p;
                        if (!*p)
                            _myfilbuf();

                        return MY_FALSE;
                    }
                    else if (wordcmp(q, "elif", len) == 0)
                    {
                        /* Morph the 'elif' into '#if' and reparse it */
                        current_loc.line--;
                        total_lines--;
                        q[0] = nl;
                        q[1] = '#';
                        outp = q+1;
                        return MY_FALSE;
                    }
                }
            }
        }
        else /* not a preprocessor statement */
        {
            if (c == CHAR_EOF)
            {
                outp = p - 2;
                current_loc.line--;
                total_lines--;
                lexerror("Unexpected end of file while skipping");
                return MY_TRUE;
            }

            /* Skip the rest of the line */
            if (c != nl)
                p = skip_line(p);
        }

        /* Read new data from the file if necessary */
        if (!*p)
        {
            outp = p;
            p = _myfilbuf();
        }
    } /* for () */

    /* NOTREACHED */
} /* skip_to() */

/*-------------------------------------------------------------------------*/
static void
handle_cond (Bool c)

/* Evaluate the boolean condition <c> of a preprocessor #if statement.
 * If necessary, skip to the condition branch to read next, and/or
 * push a new state onto the ifstate-stack.
 */
{
    lpc_ifstate_t *p;

    if (c || skip_to("else", "endif")) {
        p = mempool_alloc(lexpool, sizeof(lpc_ifstate_t));
        p->next = iftop;
        iftop = p;
        p->state = c ? EXPECT_ELSE : EXPECT_ENDIF;
    }
} /* handle_cond() */

/*-------------------------------------------------------------------------*/
static Bool
start_new_include (int fd, string_t * str
                  , char * name, char * name_ext, char delim)

/* The lexer is about to read data from an included source (either file
 * <fd> or string <str> which will be referenced) - handle setting up the
 * include information. <name> is the name of the file to be read, <name_ext>
 * is NULL or a string to add to <name> as " (<name_ext>)", <delim> is the
 * delimiter ('"', '>' or ')') of the include filename.
 *
 * Return TRUE on success, FALSE if something failed.
 */

{
    struct incstate *is, *ip;
    source_file_t * src_file;
    size_t namelen;
    int inc_depth;
    ptrdiff_t linebufoffset;

    /* Prepare defbuf for a (nested) include */
    linebufoffset = linebufstart - &defbuf[defbuf_len];
    if (outp - defbuf < 3*MAXLINE)
    {
        realloc_defbuf();
        /* linebufstart is invalid now */
        if (outp - defbuf < 2*MAXLINE)
        {
            lexerror("Maximum total buffer size exceeded");
            return MY_FALSE;
        }
    }

    /* Copy the current state, but don't put it on the stack
     * yet in case we run into an error further down.
     */
    is = mempool_alloc(lexpool, sizeof(struct incstate));
    if (!is) {
        lexerror("Out of memory");
        return MY_FALSE;
    }

    src_file = new_source_file(NULL, &current_loc);
    if (!src_file)
    {
        mempool_free(lexpool, is);
        lexerror("Out of memory");
        return MY_FALSE;
    }

    is->yyin = yyin;
    is->loc = current_loc;
    is->linebufoffset = linebufoffset;
    is->saved_char = saved_char;
    is->next = inctop;


    /* Copy the new filename into src_file */

    namelen = strlen(name);
    if (name_ext != NULL)
        namelen += 3 + strlen(name_ext);

    src_file->name = xalloc(namelen+1);
    if (!src_file->name)
    {
        mempool_free(lexpool, is);
        lexerror("Out of memory");
        return MY_FALSE;
    }
    strcpy(src_file->name, name);
    if (name_ext)
    {
        strcat(src_file->name, " (");
        strcat(src_file->name, name_ext);
        strcat(src_file->name, ")");
    }

    /* Now it is save to put the saved state onto the stack*/
    inctop = is;

    /* Compute the include depth and store the include information */
    for (inc_depth = 0, ip = inctop; ip; ip = ip->next)
        inc_depth++;

    if (name_ext)
        inctop->inc_offset = store_include_info(name_ext, src_file->name, delim, inc_depth);
    else
        inctop->inc_offset = store_include_info(name, src_file->name, delim, inc_depth);

    /* Initialise the rest of the lexer state */
    current_loc.file = src_file;
    current_loc.line = 0;
    linebufend   = outp - 1; /* allow trailing zero */
    linebufstart = linebufend - MAXLINE;
    *(outp = linebufend) = '\0';
    set_input_source(fd, name, str);
    _myfilbuf();

    return MY_TRUE;
} /* start_new_include() */

/*-------------------------------------------------------------------------*/
static void
add_auto_include (const char * obj_file, const char *cur_file, Bool sys_include)

/* A new file <cur_file> was opened while compiling object <object_file>.
 * Add the auto-include information if available.
 *
 * If <cur_file> is NULL, then the <object_file> itself has just been
 * opened, otherwise <cur_file> is an included file. In the latter case,
 * flag <sys_include> purveys if it was a <>-type include.
 *
 * The global <current_loc.line> must be valid and will be modified.
 */

{
    string_t * auto_include_string = NULL;

    if (driver_hook[H_AUTO_INCLUDE].type == T_STRING
     && cur_file == NULL
       )
    {
        auto_include_string = driver_hook[H_AUTO_INCLUDE].u.str;
    }
    else if (driver_hook[H_AUTO_INCLUDE].type == T_CLOSURE)
    {
        svalue_t *svp;

        /* Setup and call the closure */
        push_c_string(inter_sp, obj_file);
        if (cur_file != NULL)
        {
            push_c_string(inter_sp, (char *)cur_file);
            push_number(inter_sp, sys_include ? 1 : 0);
        }
        else
        {
            push_number(inter_sp, 0);
            push_number(inter_sp, 0);
        }
        svp = secure_apply_lambda(driver_hook+H_AUTO_INCLUDE, 3);
        if (svp && svp->type == T_STRING)
        {
            auto_include_string = svp->u.str;
        }
    }

    if (auto_include_string != NULL)
    {
        /* The auto include string is handled like a normal include */
        if (cur_file != NULL)   /* Otherwise we already are at line 1 */
            current_loc.line++; /* Make sure to restore to line 1 */
        (void)start_new_include(-1, auto_include_string
                               , current_loc.file->name, "auto include", ')');
        if (cur_file == NULL)   /* Otherwise #include will increment it */
            current_loc.line++; /* Make sure to start at line 1 */
    }
} /* add_auto_include() */

/*-------------------------------------------------------------------------*/
static void
merge (char *name, mp_int namelen, char *deststart)

/* Take the given include file <name> of length <namelen>, make it
 * a proper absolute pathname and store it into the buffer <deststart>.
 * This buffer must be at least INC_OPEN_BUFSIZE bytes big.
 * On a failure, return the empty string in *deststart.
 *
 * If <name> is a relative pathname, it is interpreted to the location
 * of <currentfile>. './' and '../' sequences in the name are properly
 * resolved (includes from above the mudlib are caught).
 */

{
    char *from;  /* Next character in <name> to process */
    char *dest;  /* Destination pointer into <deststart> */

    from = name;

    /* If <name> is an absolute pathname, skip any leading '/'.
     * Else extract the pathpart from <currentfile>, put
     * it into the destination buffer and set dest to point after it.
     */
    if (*from == '/')
    {
        /* absolute path */

        dest = deststart;
        do from++; while (*from == '/');
    }
    else
    {
        /* relative path */

        char *cp, *dp;

        dest = (dp = deststart) - 1;
        for (cp = current_loc.file->name; *cp; *dp++ = *cp++)
        {
            if (*cp == '/')
                dest = dp;
        }
        dest++;
    }

    /* Make sure the bufferlimits are not overrun. */
    if ((dest - deststart) + namelen >= INC_OPEN_BUFSIZE)
    {
        *deststart = '\0';
        return;
    }

    /* Append the <name> to the buffer starting at <dest>,
     * collapsing './' and '../' sequences while doing it.
     */
    for (;;)
    {
        /* <from> now points to the character after the last '/'.
         */

        if (*from == '.')
        {
            if (from[1] == '.' && from[2] == '/')
            {
                /* '../': remove the pathpart copied last */

                if (dest == deststart)
                {
                    /* including from above mudlib is NOT allowed */
                    *deststart = '\0';
                    return;
                }

                for (--dest;;)
                {
                    if (*--dest == '/')
                    {
                        dest++;
                        break;
                    }
                    if (dest == deststart)
                        break;
                }
                from += 3;
                continue;

            } else if (from[1] == '/')
            {
                /* just skip these two characters */

                from += 2;
                continue;
            }
        }

        /* Copy all characters up to and including the next '/'
         * from <name> into the destination buffer.
         * Return when at the end of the name.
         */

        {
            char c;

            do {
                c = *from++;
                *dest++ = c;
                if (!c)
                  return;
            } while (c != '/');
            while (*from == '/')
                from++;
        }
    } /* for (;;) */

    /* NOTREACHED */
} /* merge() */

/*-------------------------------------------------------------------------*/
static int
open_include_file (char *buf, char *name, mp_int namelen, char delim)

/* Open the include file <name> (length <namelen>) and return the file
 * descriptor. On failure, generate an error message and return -1.
 *
 * <buf> is a buffer of size INC_OPEN_BUFSIZE, where the real
 * filename will be written to - <name> is just the name given
 * in the #include statement.
 *
 * <delim> is '"' for #include ""-type includes, and '>' else.
 * Relative "-includes are searched relative to the current file.
 * <-includes are searched in the path(s) defined by the H_INCLUDE_DIRS
 * driver hook.
 */

{
    int fd;
    int i;
    struct stat aStat;

    /* First, try to call master->include_file().
     * Since simulate::load_object() makes sure that the master has been
     * loaded, this test can only fail when the master is compiled.
     */
    if (master_ob && !(master_ob->flags & O_DESTRUCTED)
     && (!EVALUATION_TOO_LONG())
       )
    {
        svalue_t *res;

        push_c_n_string(inter_sp, name, namelen);

        if (!compat_mode)
        {
            char * filename;
            filename = alloca(strlen(current_loc.file->name)+2);
            *filename = '/';
            strcpy(filename+1, current_loc.file->name);
            push_c_string(inter_sp, filename);
        }
        else
            push_c_string(inter_sp, current_loc.file->name);

        push_number(inter_sp, (delim == '"') ? 0 : 1);
        res = apply_master(STR_INCLUDE_FILE, 3);

        if (res && !(res->type == T_NUMBER && !res->u.number))
        {
            /* We got a result - either a new name or a "reject it"
             * value.
             */

            char * cp;

            if (res->type != T_STRING)
            {
                yyerrorf("Illegal to include file '%s'.", name);
                return -1;
            }

            if (mstrsize(res->u.str) >= INC_OPEN_BUFSIZE)
            {
                yyerrorf("Include name '%s' too long.", get_txt(res->u.str));
                return -1;
            }

            for (cp = get_txt(res->u.str); *cp == '/'; cp++) NOOP;

            if (!legal_path(cp))
            {
                yyerrorf("Illegal path '%s'.", get_txt(res->u.str));
                return -1;
            }

            if (!convert_path_to_native_buf(cp, mstrsize(res->u.str) + get_txt(res->u.str) - cp, buf, INC_OPEN_BUFSIZE))
            {
                yyerrorf("Could not encode path '%s'.", get_txt(res->u.str));
                return -1;
            }

            if (!stat(buf, &aStat)
             && S_ISREG(aStat.st_mode)
             && (fd = ixopen(buf, O_RDONLY|O_BINARY)) >= 0 )
            {
                strcpy(buf, cp); /* Put the UTF-8 encoded name into <buf>. */
                FCOUNT_INCL(buf);
                return fd;
            }
            if (errno == EMFILE) lexerror("File descriptors exhausted");
#if ENFILE
            if (errno == ENFILE) lexerror("File table overflow");
#endif

            /* If we come here, we fail: file not found */
            return -1;
        }
    }
    else if (EVALUATION_TOO_LONG())
    {
        yyerrorf("Can't call master::%s for '%s': eval cost too big"
                , get_txt(STR_INCLUDE_FILE), name);
    }

    /* The master apply didn't succeed, try the manual handling */

    if (delim == '"') /* It's a "-include */
    {
        char *native;

        /* Merge the <name> with the current filename. */
        merge(name, namelen, buf);

        native = convert_path_to_native(buf, strlen(buf));
        if (native == NULL)
        {
            yyerrorf("Could not encode path '%s'.", buf);
            return -1;
        }

        /* Test the file and open it */
        if (!stat(native, &aStat)
         && S_ISREG(aStat.st_mode)
         && (fd = ixopen(native, O_RDONLY|O_BINARY)) >= 0)
        {
            FCOUNT_INCL(buf);
            return fd;
        }

        if (errno == EMFILE)
            lexerror("File descriptors exhausted");
#ifdef ENFILE
        if (errno == ENFILE)
            lexerror("File table overflow");
#endif
        /* Include not found - fall back onto <> search pattern */
    }

    /* Handle a '<'-include. */

    if (driver_hook[H_INCLUDE_DIRS].type == T_POINTER)
    {
        char * cp;

        /* H_INCLUDE_DIRS is a vector of include directories.
         */

        if (namelen + inc_list_maxlen >= INC_OPEN_BUFSIZE)
        {
            yyerror("Include name too long.");
            return -1;
        }

        for (cp = name; *cp == '/'; cp++) NOOP;

        /* The filename must not specifiy parent directories */
        if (!check_no_parentdirs(cp))
            return -1;

        /* Search all include dirs specified.
         */
        for (i = 0; (size_t)i < inc_list_size; i++)
        {
            char * iname, * native;

            sprintf(buf, "%s%s", get_txt(inc_list[i].u.str), name);
            for (iname = buf; *iname == '/'; iname++) NOOP;

            native = convert_path_to_native(iname, strlen(iname));
            if (native == NULL)
            {
                yyerrorf("Could not encode path '%s'.", buf);
                return -1;
            }

            if (!stat(native, &aStat)
             && S_ISREG(aStat.st_mode)
             && (fd = ixopen(native, O_RDONLY|O_BINARY)) >= 0 )
            {
                FCOUNT_INCL(iname);
                return fd;
            }
            if (errno == EMFILE) lexerror("File descriptors exhausted");
#if ENFILE
            if (errno == ENFILE) lexerror("File table overflow");
#endif
        }

        /* If we come here, the include file was not found */
    }
    else if (driver_hook[H_INCLUDE_DIRS].type == T_CLOSURE)
    {
        /* H_INCLUDE_DIRS is a function generating the full
         * include file name.
         */

        svalue_t *svp;

        /* Setup and call the closure */
        push_c_string(inter_sp, name);
        push_c_string(inter_sp, current_loc.file->name);
        if (driver_hook[H_INCLUDE_DIRS].x.closure_type == CLOSURE_LAMBDA)
        {
            free_object(driver_hook[H_INCLUDE_DIRS].u.lambda->ob, "open_include_file");
            driver_hook[H_INCLUDE_DIRS].u.lambda->ob = ref_object(current_object, "open_include_file");
        }
        svp = secure_apply_lambda(&driver_hook[H_INCLUDE_DIRS], 2);

        /* The result must be legal relative pathname */
        if (!svp || svp->type != T_STRING)
            return -1;

        if (mstrsize(svp->u.str) < INC_OPEN_BUFSIZE)
        {
            char * cp;

            for (cp = get_txt(svp->u.str); *cp == '/'; cp++) NOOP;

            if (!legal_path(cp))
                return -1;

            if (!convert_path_to_native_buf(cp, mstrsize(svp->u.str) + get_txt(svp->u.str) - cp, buf, INC_OPEN_BUFSIZE))
            {
                yyerrorf("Could not encode path '%s'.", get_txt(svp->u.str));
                return -1;
            }

            if (!stat(buf, &aStat)
             && S_ISREG(aStat.st_mode)
             && (fd = ixopen(buf, O_RDONLY|O_BINARY)) >= 0 )
            {
                strcpy(buf, cp);
                FCOUNT_INCL(buf);
                return fd;
            }
            if (errno == EMFILE) lexerror("File descriptors exhausted");
#if ENFILE
            if (errno == ENFILE) lexerror("File table overflow");
#endif
        }

        /* If we come here, the include file was not found */
    }

    /* File not found */
    return -1;
} /* open_include_file() */

/*-------------------------------------------------------------------------*/
void *
get_include_handle (void)

/* Helper function for inline closures: return the current inctop
 * setting so that the compiler can check if a closures spans files.
 */

{
    return (void*)inctop;
} /* get_include_handle() */

/*-------------------------------------------------------------------------*/
static INLINE void
handle_include (char *name)

/* Handle an #include statement, <name> points to the first non-blank
 * character after the '#include'.
 * If the include succeeds, a new incstate is created and pushed
 * onto the include stack. Else an error message is generated.
 */

{
    char *p;
    int   fd;        /* fd of new include file */
    char  delim;     /* Filename end-delimiter ('"' or '>'). */
    char *old_outp;  /* Save the original outp */
    Bool  in_buffer = MY_FALSE; /* True if macro was expanded */
    char  buf[INC_OPEN_BUFSIZE];

#if 0
    if (nbuf) {
        lexerror("Internal preprocessor error");
        return;
    }
#endif
    old_outp = outp;

    /* If <name> doesn't start with '"' or '<', assume that it
     * is a macro. Attempt to expand these macros until <name>
     * starts with a proper delimiter.
     */
    while (*name != '"' && *name != '<')
    {
        char c;
        ident_t *d;

        /* Locate the end of the macro and look it up */
        p = skip_alunum(name);
        c = *p;
        *p = '\0';
        d = lookup_define(name);
        *p = c;

        /* Prepare to expand the macro */
        if (in_buffer)
        {
            outp = p;
        }
        else
        {
            myungetc('\n');
            add_input(p);
            in_buffer = MY_TRUE;
        }

        /* Expand the macro */
        if (!d || !_expand_define(&d->u.define, d) ) {
            yyerror("Missing leading \" or < in #include");
            return;
        }

        /* Set name to the first non-blank of the expansion */
        name = skip_white(outp);
    }

    /* Store the delimiter and set p to the closing delimiter */
    delim = (char)((*name++ == '"') ? '"' : '>');
    for(p = name; *p && *p != delim; p++) NOOP;

    if (!*p) {
        yyerror("Missing trailing \" or > in #include");
        outp = old_outp;
        return;
    }
    *p = '\0';


    /* For "-includes, look for following macros or "<path>"
     * fragments on the same line and append these to the <name>.
     * The new name is created in the yytext[] buffer (if at all).
     */

    if (delim == '"')
    {
        char *q;

        q = p + 1;
        for (;;)
        {
            /* Find the first non-blank character after p */
            q = skip_white(q);
            if (!*q || *q == '\n')
                break;

            /* First, try to expand a macros */
            while (*q != delim)
            {
                char *r, c;
                ident_t *d;

                /* Set r to the first blank after the macro name */
                r = skip_alunum(q);

                /* Lookup the macro */
                c = *r;
                *r = '\0';
                d = lookup_define(q);
                *r = c;

                /* Prepare to expand the macro */
                if (in_buffer)
                {
                    outp = r;
                    if (name != yytext)
                    {
                        if ( (p - name) >= MAXLINE - 1)
                        {
                            yyerror("Include name too long.");
                            outp = old_outp;
                            return;
                        }
                        *p = '\0';
                        strcpy(yytext, name);
                        p += yytext - name;
                        name = yytext;
                    }
                }
                else
                {
                    myungetc('\n');
                    add_input(r);
                    in_buffer = MY_TRUE;
                }

                /* Expand the macro */
                if (!d || !_expand_define(&d->u.define, d) ) {
                    yyerror("Missing leading \" in #include");
                    outp = old_outp;
                    return;
                }

                /* Skip the blanks until the next macro/filename */
                q = skip_white(outp);
            }

            /* Second, try to parse a string literal */
            while (*++q && *q != delim)
            {
                if ( (p - name) >= MAXLINE - 1)
                {
                    yyerror("Include name too long.");
                    outp = old_outp;
                    return;
                }
                *p++ = *q;
            }
            if (!*q++) {
                yyerror("Missing trailing \" in #include");
                outp = old_outp;
                return;
            }
        } /* for (;;) */
    } /* if (delim == '"') */
    else
    {
        /* Check that only whitespaces come after the <>. Comments
         * have already been removed by handle_preprocessor_statement().
         */
        char *q = skip_white(p+1);

        if (*q && *q != '\n')
            warn_trailing_chars("include");
    }

    /* p now points to the character after the parsed filename */

    outp = old_outp;  /* restore outp */
    *p = '\0';        /* mark the end of the filename */

    /* Open the include file, put the current lexer state onto
     * the incstack, and set up for the new file.
     */
    if ((fd = open_include_file(buf, name, p - name, delim)) >= 0)
    {
        if (!start_new_include(fd, NULL, buf, NULL, delim))
            return;
        add_auto_include(object_file, current_loc.file->name, delim != '"');
    }
    else
    {
        yyerrorf("Cannot #include '%s'", name);
    }
} /* handle_include() */

/*-------------------------------------------------------------------------*/
static void
skip_comment (void)

/* Skip a block comment (/ * ... * /). The function is entered with outp
 * pointing to the first character after the comment introducer, and left
 * with outp pointing to the first character after the comment end delimiter.
 */

{
    register char c, *p;

    p = outp;
    for(;;)
    {
        /* Skip everything until the next '*' */
        while((c =  *p++) != '*')
        {
            if (c == '\n') {
                store_line_number_info();
                nexpands = 0;
                if ((c = *p) == CHAR_EOF) {
                    outp = p - 1;
                    lexerror("End of file (or 0x01 character) in a comment");
                    return;
                }
                current_loc.line++;
                if (!c)
                {
                    outp = p;
                    p = _myfilbuf();
                }
            }
        } /* while (c == '*') */

        /* Skip all '*' until we find '/' or something else */
        do
        {
            if ((c = *p++) == '/')
            {
                outp = p;
                return;
            }

            if (c == '\n') {
                store_line_number_info();
                nexpands = 0;
                if ((c = *p) == CHAR_EOF)
                {
                    outp = p - 1;
                    lexerror("End of file (or 0x01 character) in a comment");
                    return;
                }
                current_loc.line++;
                if (!c)
                {
                    outp = p;
                    p = _myfilbuf();
                }
                c = '\0'; /* Make sure to terminate the '*' loop */
            }
        } while(c == '*');
    } /* for() */

    /* NOTREACHED */
} /* skip_comment() */

/*-------------------------------------------------------------------------*/
static char *
skip_pp_comment (char *p)

/* Skip a '//' line comment. <p> points to the first character after
 * the comment introducer, the function returns a pointer to the first
 * character after the terminating newline. If the comment is ended
 * prematurely by the end of file, the returned pointer will point at the
 * EOF character.
 * Note that a '\<newline>' lineend does not terminate the comment.
 */

{
    char c;

    for (;;)
    {
        c = *p++;
        if (CHAR_EOF == c)
        {
            return p-1;
        }
        if (c == '\n')
        {
            store_line_number_info();
            current_loc.line++;
            if (p[-2] == '\\')
            {
                if (!*p)
                {
                    outp = p;
                    p = _myfilbuf();
                }
                continue;
            }
            nexpands = 0;
            if (!*p)
            {
                outp = p;
                p = _myfilbuf();
            }
            return p;
        }
    }

    /* NOTREACHED */
} /* skip_pp_comment() */

/*-------------------------------------------------------------------------*/
static void
checktrail (char *sp, const char* stmt)

/* Check that only one word is contained in <sp>. Remove any whitespaces
 * after it by setting the first whitespace to '\0'. If there are any
 * non-whitespace characters give a warning or (when pedantic) error.
 */

{
    if (!*sp)
    {
        lexerror("Illegal # command");
    }
    else
    {
        char *p = skip_nonspace(sp);
        char *q = skip_white(p);
        *p = '\0';

        if (*q && *q != '\n')
            warn_trailing_chars(stmt);
    }
} /* checktrail() */

/*-------------------------------------------------------------------------*/
static void
handle_pragma (char *str)

/* Handle the pragma <str>. Unknown pragmas are ignored.
 * One pragma string can contain multiple actual pragmas, separated
 * with comma (and additional spaces).
 */

{
    char * base, * next;

#if defined(LEXDEBUG)
    printf("%s handle pragma:'%s'\n", time_stamp(), str);
#endif

    /* Loop over the pragma(s).
     * If valid, base points to the first character of the pragma name,
     * or to spaces before it.
     */
    for ( base = str, next = NULL
        ; base != NULL && *base != '\0' && *base != '\r'
        ; base = next
        )
    {
        size_t namelen;
        Bool validPragma;

        /* Skip spaces */
        base = base + strspn(base, " \t\r");
        if ('\0' == *base || '\r' == *base)
            break;

        /* Find next delimiter, if any, and determine the
         * length of the pragma name.
         */
        next = strpbrk(base, " \t,\r");
        if (NULL == next)
            namelen = strlen(base);
        else
            namelen = next - base;

        /* Evaluate the found pragma name */
        validPragma = MY_FALSE;

        if (namelen == 0)
        {
            if (master_ob)
            {
                yywarnf("Empty #pragma");
            }
            else
            {
                debug_message("Warning: Empty #pragma"
                              ": file %s, line %d\n"
                             , current_loc.file->name, current_loc.line);
            }
            validPragma = MY_TRUE; /* Since we already issued a warning */
        }
        else if (wordcmp(base, "strict_types", namelen) == 0)
        {
            pragma_strict_types = PRAGMA_STRICT_TYPES;
            instrs[F_CALL_OTHER].ret_type = lpctype_unknown;
            instrs[F_CALL_DIRECT].ret_type = lpctype_unknown;
            instrs[F_CALL_STRICT].ret_type = lpctype_unknown;
            instrs[F_CALL_DIRECT_STRICT].ret_type = lpctype_unknown;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "strong_types", namelen) == 0)
        {
            pragma_strict_types = PRAGMA_STRONG_TYPES;
            instrs[F_CALL_OTHER].ret_type = lpctype_mixed;
            instrs[F_CALL_DIRECT].ret_type = lpctype_mixed;
            instrs[F_CALL_STRICT].ret_type = lpctype_mixed;
            instrs[F_CALL_DIRECT_STRICT].ret_type = lpctype_mixed;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "weak_types", namelen) == 0)
        {
            pragma_strict_types = PRAGMA_WEAK_TYPES;
            instrs[F_CALL_OTHER].ret_type = lpctype_mixed;
            instrs[F_CALL_DIRECT].ret_type = lpctype_mixed;
            instrs[F_CALL_STRICT].ret_type = lpctype_mixed;
            instrs[F_CALL_DIRECT_STRICT].ret_type = lpctype_mixed;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "save_types", namelen) == 0)
        {
            pragma_save_types = MY_TRUE;
            validPragma = MY_TRUE;
        }
        // the following two pragmas are ignored.
        else if (wordcmp(base, "combine_strings", namelen) == 0)
        {
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_combine_strings", namelen) == 0)
        {
            validPragma = MY_TRUE;
        }
        // verbose_error is ignored, its behaviour is always enabled.
        else if (wordcmp(base, "verbose_errors", namelen) == 0)
        {
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_clone", namelen) == 0)
        {
            pragma_no_clone = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_inherit", namelen) == 0)
        {
            pragma_no_inherit = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_shadow", namelen) == 0)
        {
            pragma_no_shadow = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "pedantic", namelen) == 0)
        {
            pragma_pedantic = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "sloppy", namelen) == 0)
        {
            pragma_pedantic = MY_FALSE;
            validPragma = MY_TRUE;
        }
        // These two pragmas are ignored.
        else if (wordcmp(base, "no_local_scopes", namelen) == 0)
        {
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "local_scopes", namelen) == 0)
        {
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "warn_missing_return", namelen) == 0)
        {
            pragma_warn_missing_return = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_warn_missing_return", namelen) == 0)
        {
            pragma_warn_missing_return = MY_FALSE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "warn_function_inconsistent", namelen) == 0)
        {
            pragma_check_overloads = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_warn_function_inconsistent", namelen) == 0)
        {
            pragma_check_overloads = MY_FALSE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "warn_deprecated", namelen) == 0)
        {
            pragma_warn_deprecated = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_warn_deprecated", namelen) == 0)
        {
            pragma_warn_deprecated = MY_FALSE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "range_check", namelen) == 0)
        {
            pragma_range_check = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_range_check", namelen) == 0)
        {
            pragma_range_check = MY_FALSE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "warn_empty_casts", namelen) == 0)
        {
            pragma_warn_empty_casts = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_warn_empty_casts", namelen) == 0)
        {
            pragma_warn_empty_casts = MY_FALSE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "rtt_checks", namelen) == 0)
        {
            pragma_rtt_checks = MY_TRUE;
            pragma_save_types = MY_TRUE;
            pragma_warn_rtt_checks = MY_FALSE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "warn_rtt_checks", namelen) == 0)
        {
            pragma_rtt_checks = MY_TRUE;
            pragma_save_types = MY_TRUE;
            pragma_warn_rtt_checks = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_rtt_checks", namelen) == 0)
        {
            pragma_rtt_checks = MY_FALSE;
            pragma_warn_rtt_checks = MY_FALSE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "warn_unused_variables", namelen) == 0)
        {
            pragma_warn_unused_variables = true;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_warn_unused_variables", namelen) == 0)
        {
            pragma_warn_unused_variables = false;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "share_variables", namelen) == 0)
        {
            if (variables_defined)
            {
                yywarnf("Can't use #pragma share_variables after defining "
                        "variables");
            }
            else
                pragma_share_variables = MY_TRUE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "init_variables", namelen) == 0)
        {
            if (variables_defined)
            {
                yywarnf("Can't use #pragma init_variables after defining "
                        "variables");
            }
            else
                pragma_share_variables = MY_FALSE;
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "no_bytes_type", namelen) == 0)
        {
            if (!pragma_no_bytes_type)
            {
                /* Remove the bytes keyword. */
                for (ident_t *p = find_shared_identifier_n("bytes", 5, I_TYPE_RESWORD, 0);
                     p != NULL && p->type >= I_TYPE_RESWORD;
                     p = p->inferior)
                {
                    if (p->type == I_TYPE_RESWORD)
                    {
                        free_shared_identifier(p);
                        break;
                    }
                }

                pragma_no_bytes_type = true;
            }

            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "bytes_type", namelen) == 0)
        {
            if (pragma_no_bytes_type)
            {
                /* Restore the bytes keyword. */
                ident_t *p = make_shared_identifier("bytes", I_TYPE_RESWORD, 0);
                if (!p)
                    fatal("Out of memory\n");
                if (p->type == I_TYPE_UNKNOWN)
                {
                    p->type = I_TYPE_RESWORD;
                    p->u.code = L_BYTES_DECL;
                }
                else if (p->type > I_TYPE_RESWORD)
                {
                    /* We found a define... We have to insert a new entry below that. */
                    ident_t *r;

                    /* Remove the define. */
                    unlink_shared_identifier(p);

                    r = make_shared_identifier("bytes", I_TYPE_RESWORD, 0);
                    r->type = I_TYPE_RESWORD;
                    r->u.code = L_BYTES_DECL;

                    /* And reinsert the define. */
                    assert(ident_table[p->hash] == r);
                    p->next = r->next;
                    p->inferior = r;
                    ident_table[p->hash] = p;
                }

                pragma_no_bytes_type = false;
            }

            validPragma = MY_TRUE;
        }
#if defined( DEBUG ) && defined ( TRACE_CODE )
        else if (wordcmp(base, "set_code_window", namelen) == 0)
        {
            set_code_window();
            validPragma = MY_TRUE;
        }
        else if (wordcmp(base, "show_code_window", namelen) == 0)
        {
            show_code_window();
            validPragma = MY_TRUE;
        }
#endif

        /* Advance next to the next scanning position so that the
         * for loop increment works.
         */
        if (NULL != next)
        {
            /* Skip spaces */
            next = next + strspn(next, " \t\r");

            if (',' == *next)
            {
                /* Skip the one allowed comma.
                 * We allow the comma to be followed by lineend
                 */
                next++;
            }
            else if ('\0' != *next && '\r' != *next)
            {
                if (master_ob)
                {
                    yywarnf("Missing comma between #pragma options");
                }
                else
                {
                    debug_message("Warning: Missing comma between #pragma options"
                                  ": file %s, line %d\n"
                                 , current_loc.file->name, current_loc.line);
                }
            }

            if ('\0' == *next || '\r' == *next)
            {
                /* End of string */
                next = NULL;
            }

            /* If next now points to something else but space or a pragma
             * name, the next loop iteration will complain about an illegal
             * pragma.
             */
        }

        /* Finally check if the pragma was valid */
        if (!validPragma)
        {
            if (master_ob)
            {
                /* Calling yywarnf() without a master can cause the game
                 * to shut down, because yywarnf() eventually tries to call
                 * a master lfun.
                 */
                yywarnf("Unknown #pragma '%.*s'", (int)namelen, base);
            }
            else
            {
                debug_message("Warning: Unknown #pragma '%.*s': file %s, line %d\n"
                             , (int)namelen, base, current_loc.file->name, current_loc.line);
            }
        }

    } /* for (base) */
} /* handle_pragma() */

/*-------------------------------------------------------------------------*/
static INLINE int
number (long i)

/* Return a number to yacc: set yylval.number to <i> and return L_NUMBER.
 */

{
#ifdef LEXDEBUG
    printf("%s returning number %d.\n", time_stamp(), i);
#endif
    yylval.number = i;
    return L_NUMBER;
} /* number() */

/*-------------------------------------------------------------------------*/
static INLINE char *
parse_numeric_escape (char * cp, p_int * p_char)

/* Parse a character constant in one of the following formats:
 *   <decimal>      (max 3 digits)
 *   0o<octal>      (max 3 digits)
 *   0x<sedecimal>  (max 2 digits)
 *   x<sedecimal>   (max 2 digits)
 *   0b<binary>     (max 8 digits)
 *   u<sedecimal>   (max 4 digits)
 *   U<sedecimal>   (max 8 digits)
 *
 * with <cp> pointing to the first character. The function parses
 * until the first illegal character, but at max the given number of
 * digits.
 *
 * The parsed number is stored in *<p_num>, the function returns the pointer
 * to the first character after the number.
 * If no valid character constant could be found, NULL is returned.
 */

{
    char c;
    int num_digits = 3;
    bool check_digits = false;
    unsigned long l;
    unsigned long base = 10;

    c = *cp++;

    switch (c)
    {
        case '0':
            /* '0' introduces decimal, octal, binary and sedecimal numbers, or it
             * can be a float.
             *
             * Sedecimals are handled in a following if-clause to allow the
             * two possible prefixes.
             */

            c = *cp++;

            switch (c)
            {
                case 'X': case 'x':
                    num_digits = 2;
                    break;

                case 'b': case 'B':
                {
                    c = *cp++;
                    num_digits = 8;
                    base = 2;
                    break;
                }

                case 'o': case 'O':
                    c = *cp++;
                    base = 8;
                    num_digits = 3;
                    break;

                default:
                    c = '0';
                    cp--;
                    break;
            } /* switch(c) */
            break;

        case 'X': case 'x':
            num_digits = 2;
            break;

        case 'u':
            num_digits = 4;
            check_digits = true;
            break;

        case 'U':
            num_digits = 8;
            check_digits = true;
            break;
    } /* switch(c) */

    if ( c == 'X' || c == 'x' || c == 'U' || c == 'u' )
    {
        if (!leXdigit(*cp))
        {
            yywarn("Character constant used with no valid digits");
            return NULL;
        }

        /* strtol() gets the sign bit wrong,
         * strtoul() isn't portable enough.
         * TODO: strtoul should be portable enough today... Re-check if we
         * TODO::require C99.
         */
        l = 0;
        while(leXdigit(c = *cp++) && num_digits-- > 0)
        {
            if (c > '9')
                c = (char)((c & 0xf) + ( '9' + 1 - ('a' & 0xf) ));
            l <<= 4;
            l += c - '0';
        }

        if (num_digits > 0 && check_digits)
        {
            yywarn("Missing digits in Unicode character constant.");
        }

        if (l >= 0x110000)
        {
            yywarn("Character constant out of range (> 0x10FFFF)");
            l &= 0xff;
        }
    }
    else
    {
        /* Parse a normal number from here */

        l = c - '0';
        /* l is unsigned. So any c smaller than '0' will be wrapped into 
         * positive values and be larger then base as well. Therefore an 
         * additional comparison of l < 0 is not explicitly needed here. */
        if  (l > base)
        {
            yywarn("Character constant used with no valid digits");
            return NULL;
        }
        while (lexdigit(c = *cp++) && c < (char)('0'+base) && --num_digits > 0)
              l = l * base + (c - '0');

        if (l >= 256)
        {
            yywarn("Character constant out of range (> 255)");
            l &= 0xff;
        }
    }

    *p_char = l;
    return cp-1;

} /* parse_numeric_escape() */

/*-------------------------------------------------------------------------*/
static INLINE char *
parse_number (char * cp, unsigned long * p_num, Bool * p_overflow)

/* Parse a positive integer number in one of the following formats:
 *   <decimal>
 *   0o<octal>
 *   0x<sedecimal>
 *   x<sedecimal>
 *   0b<binary>
 *
 * with <cp> pointing to the first character.
 *
 * The parsed number is stored in *<p_num>, the function returns the pointer
 * to the first character after the number. If the parsed number exceeded
 * the numerical limits, *<p_overflow> is set to TRUE, otherwise to FALSE.
 *
 * The function is also available to the other parts of the driver.
 */

{
    char c;
    unsigned long l;
    unsigned long base = 10;
    unsigned long max_shiftable = ULONG_MAX / base;

    *p_overflow = MY_FALSE;
    c = *cp++;

    if ('0' == c)
    {
        /* '0' introduces decimal, octal, binary and sedecimal numbers, or it
         * can be a float.
         *
         * Sedecimals are handled in a following if-clause to allow the
         * two possible prefixes.
         */

        c = *cp++;

        switch (c)
        {
        case 'X': case 'x':
            /* Sedecimal number are handled below - here just fall
             * through.
             */
            NOOP;
            break;

        case 'b': case 'B':
          {
            l = 0;
            max_shiftable = ULONG_MAX / 2;
            --cp;
            while('0' == (c = *++cp) || '1' == c)
            {
                *p_overflow = *p_overflow || (l > max_shiftable);
                l <<= 1;
                l += c - '0';
            }

            *p_num = *p_overflow ? LONG_MAX : l;
            return cp;
          }

        case 'o': case 'O':
            c = '0';
            base = 8;
            max_shiftable = ULONG_MAX / base;
            break;

        default:
            /* If some non-digit follows, it's just the number 0.
             */
            if (!lexdigit(c))
            {
                *p_num = 0;
                return cp-1;
            }
            break;
        } /* switch(c) */
    } /* if ('0' == c) */

    if ( c == 'X' || c == 'x' )
    {

        /* strtol() gets the sign bit wrong,
         * strtoul() isn't portable enough.
         */
        max_shiftable = ULONG_MAX / 16;
        l = 0;
        --cp;
        while(leXdigit(c = *++cp))
        {
            *p_overflow = *p_overflow || (l > max_shiftable);
            if (c > '9')
                c = (char)((c & 0xf) + ( '9' + 1 - ('a' & 0xf) ));
            l <<= 4;
            l += c - '0';
        }
        *p_num = *p_overflow ? LONG_MAX : l;
        return cp;
    }

    /* Parse a normal number from here */

    max_shiftable = ULONG_MAX / base;
    l = c - '0';
    while (lexdigit(c = *cp++) && c < (char)('0'+base))
    {
        *p_overflow = *p_overflow || (l > max_shiftable);
        c -= '0';
        l = l * base + c;
        *p_overflow = *p_overflow || (l < (unsigned long)c);
    }

    *p_num = *p_overflow ? LONG_MAX : l;
    return cp-1;

} /* parse_number() */

/*-------------------------------------------------------------------------*/
char *
lex_parse_number (char * cp, unsigned long * p_num, Bool * p_overflow)

/* Parse a positive integer number in one of the following formats:
 *   <decimal>
 *   0o<octal>
 *   0x<sedecimal>
 *   0b<binary>
 *
 * with <cp> pointing to the first character.
 *
 * The parsed number is stored in *<p_num>, the function returns the pointer
 * to the first character after the number. If the parsed number exceeded
 * the numerical limits, *<p_overflow> is set to TRUE, otherwise to FALSE.
 *
 * If the string is not a number, p_num will be unchanged, and cp will
 * be returned.
 */

{
    char c = *cp;

    *p_overflow = MY_FALSE;

    if (isdigit(c))
    {
        cp = parse_number(cp, p_num, p_overflow);
    }
    return cp;
} /* lex_parse_number() */

/*-------------------------------------------------------------------------*/
static INLINE char *
parse_escaped_char (char * cp, bool is_bytes, p_int * p_char)

/* Parse the sequence for an escaped character:
 *
 *   \a : Bell (0x07)
 *   \b : Backspace (0x08)
 *   \e : Escape (0x1b)
 *   \f : Formfeed (0x0c)
 *   \n : Newline (0x0a)
 *   \r : Carriage-Return (0x0d)
 *   \t : Tab (0x09)
 *   \<decimal>, \0o<octal>, \x<sedecimal>, \0x<sedecimal>, \0b<binary>:
 *        the character with the given code.
 *   \<other printable character> : the printable character
 *
 * with <cp> pointing to the character after the '\'.
 *
 * The parsed character is stored in *<p_char>, the function returns the
 * pointer to the first character after the sequence.
 *
 * If the sequence is not one of the recognized sequences, NULL is returned.
 */

{
    char c;

    switch (c = *cp++)
    {
        case '\n':
        case CHAR_EOF:
            return NULL; break;

        case 'a': *p_char = '\007'; break;
        case 'b': *p_char = '\b';   break;
        case 'e': *p_char = '\033'; break;
        case 'f': *p_char = '\014'; break;
        case 'n': *p_char = '\n';   break;
        case 'r': *p_char = '\r';   break;
        case 't': *p_char = '\t';   break;
        case '0': case '1': case '2': case '3': case '4':
        case '5': case '6': case '7': case '8': case '9':

        case 'u': case 'U':
            /* No unicode characters for byte sequences. */
            if (is_bytes)
                return NULL;
            /* FALLTHROUGH */

        case 'x': case 'X':
        {
            char * cp2;

            /* If no valid escaped character is found, treat the sequence
             * as a normal escaped character.
             */
            cp2 = parse_numeric_escape(cp-1, p_char);
            if (cp2 != NULL)
                cp = cp2;
            else
                *p_char = c;
            break;
        }

        default:
        {
            size_t len = utf8_to_unicode(cp-1, 4, p_char);
            if (!len)
                return NULL;
            if (is_bytes && *p_char > 127)
                return NULL;

            cp += len - 1;
            break;
        }
    } /* switch() */

    return cp;
} /* parse_escaped_char() */

/*-------------------------------------------------------------------------*/
static void
add_lex_string (char *str, size_t slen)

/* Add <str> with length <slen> to the global <last_lex_string> in order
 * to implement Standard-C style string concatenation.
 */

{
    size_t len1;
    string_t *new;

    len1 = mstrsize(last_lex_string);
    if (len1+slen > MAX_ANSI_CONCAT)
    {
        /* Without this test, compilation would still terminate eventually,
         * thus it would still be 'correct', but it could take several hours.
         */
        lexerror("Too long ansi style string concatenation");
        /* leave the old string, ignore the new addition */
        return;
    }
    new = mstr_add_txt(last_lex_string, str, slen);
    if (!new)
    {
        lexerrorf("Out of memory for string concatenation (%zu bytes)",
                  len1+slen);
    }
    free_mstring(last_lex_string);
    if (new->info.unicode == STRING_ASCII && !is_ascii(str, slen))
        new->info.unicode = STRING_UTF8;
    last_lex_string = make_tabled(new);
} /* add_lex_string() */

/*-------------------------------------------------------------------------*/
static INLINE int
string (char *str, size_t slen)

/* Return a string to yacc: set last_lex_string to <str> of length <slen>
 * and return L_STRING.
 * If there is a string in last_lex_string already, <str> is appended
 * and yylex() is called recursively to allow ANSI string concatenation.
 */

{
    if (last_lex_string)
    {
        if (last_lex_string->info.unicode == STRING_BYTES)
        {
            lexerror("Can't concatenate string to byte sequence.");
            return L_STRING;
        }

        add_lex_string(str,  slen);
        return yylex();
    }
    else
    {
        last_lex_string = new_n_unicode_tabled(str, slen);
        if (!last_lex_string)
        {
            lexerrorf("Out of memory for string literal (%zu bytes)",
                      slen);
        }
    }
    return L_STRING;
} /* string() */

/*-------------------------------------------------------------------------*/
static INLINE int
bytes (char *str, size_t slen)

/* Return a byte sequence to yacc: set last_lex_string to <str>
 * of length <slen> and return L_BYTES.
 * If there is a string in last_lex_string already, <str> is appended
 * and yylex() is called recursively to allow ANSI string concatenation.
 */

{
    if (last_lex_string)
    {
        if (last_lex_string->info.unicode != STRING_BYTES)
        {
            lexerror("Can't concatenate byte sequence to string.");
            return L_BYTES;
        }

        add_lex_string(str,  slen);
        return yylex();
    }
    else
    {
        last_lex_string = new_n_tabled(str, slen, STRING_BYTES);
        if (!last_lex_string)
        {
            lexerrorf("Out of memory for bytes literal (%zu bytes)",
                      slen);
        }
    }
    return L_BYTES;
} /* bytes() */

/*-------------------------------------------------------------------------*/
static INLINE int
closure (char *in_yyp)

/* The lexer has found a closure token (#'...), with <in_yyp> pointing
 * to the quote.  Parse the token into yylval and return the proper
 * token code.
 */
{
    register char * yyp = in_yyp;

    ident_t *p;
    char *wordstart = ++yyp;
    char *super_name = NULL;
    efun_override_t efun_override;
    /* Set if 'efun::' or 'sefun::' is specified. */

    /* Set yyp to the last character of the functionname
     * after the #'.
     */
    yyp = skip_alunum(yyp);

    /* Just one character? It must be an operator */
    if (yyp == wordstart && *yyp != ':')
    {
        int i;

        if ((i = symbol_operator(yyp, (const char **)&outp)) < 0)
        {
            yyerror("Missing function name after #'");
            yylval.closure.number = CLOSURE_EFUN_OFFS;
        }
        else
            yylval.closure.number = i + CLOSURE_EFUN_OFFS;
        yylval.closure.inhIndex = 0;
        return L_CLOSURE;
    }

    /* Test for an inherited function name specification.
     * If found, set super_name to the inherit name, and
     * reset wordstart/yyp to point to the name after the '::'.
     */
    if (':' == *yyp && ':' == *(yyp+1))
    {
        super_name = wordstart;
        wordstart = yyp += 2;
        yyp = skip_alunum(yyp);
    }

    /* Test for the 'efun::' override.
     */
    efun_override = OVERRIDE_NONE;
    if (super_name != NULL && !strncmp(super_name, "efun::", 6))
    {
        efun_override = OVERRIDE_EFUN;
        super_name = NULL;
    }
    else if (super_name != NULL && !disable_sefuns && !strncmp(super_name, "sefun::", 7))
    {
        efun_override = OVERRIDE_SEFUN;
        super_name = NULL;
    }
    else if (super_name != NULL && !strncmp(super_name, "lfun::", 6))
    {
        efun_override = OVERRIDE_LFUN;
        super_name = NULL;
    }
    else if (super_name != NULL && !strncmp(super_name, "var::", 5))
    {
        efun_override = OVERRIDE_VAR;
        super_name = NULL;
    }

    outp = yyp;

    /* Lookup the name parsed from the text */

    if (super_name != NULL)
    {
        unsigned short ix;
        unsigned short inhIndex;
        funflag_t flags;
        char c = *yyp;

        *yyp = '\0';
        *(wordstart-2) = '\0';
        ix = find_inherited_function(super_name, wordstart, &inhIndex, &flags);
        inhIndex++;
        if (ix == USHRT_MAX)
        {
            yyerrorf("Undefined function: %.50s::%.50s"
                    , super_name, wordstart);
            ix = CLOSURE_EFUN_OFFS;
        }
        *yyp = c;
        *(wordstart-2) = ':';

        yylval.closure.number = ix;
        yylval.closure.inhIndex = inhIndex;
        // check for deprecated functions
        // this is done here, because here we directly have the flags of the inherited function.
        if (flags & TYPE_MOD_DEPRECATED)
        {
            yywarnf("Creating lfun closure to deprecated function %.50s::%.50s",
                    super_name, wordstart);
        }
        
        return L_CLOSURE;
    }

    p = make_shared_identifier_n(wordstart, yyp-wordstart, I_TYPE_GLOBAL, 0);
    if (!p) {
        lexerror("Out of memory");
        return 0;
    }

    /* #' can be used only on identifiers with global visibility
     * or better. Look along the .inferior chain for such an
     * identifier. If the identifier happens to be a reserved
     * word, the better for us.
     */
    while (p->type > I_TYPE_GLOBAL)
    {
        if (p->type == I_TYPE_RESWORD
         && (efun_override == OVERRIDE_NONE || efun_override == OVERRIDE_EFUN))
        {
            int code = symbol_resword(p);

            if (!code)
            {
                /* There aren't efuns with reswords as names, and
                 * it is impossible to define local / global vars
                 * or functions with such a name.
                 * Thus, !p->inferior .
                 */
                yyerrorf(
                  "No closure associated with reserved word '%s'",
                  get_txt(p->name)
                );
            }

            yylval.closure.number = code + CLOSURE_EFUN_OFFS;
            yylval.closure.inhIndex = 0;
            return L_CLOSURE;
        }
        if ( !(p = p->inferior) )
            break;
    } /* while (p->type > I_TYPE_GLOBAL */

    /* Did we find a suitable identifier? */
    if (!p || p->type < I_TYPE_GLOBAL)
    {
        char c = *yyp;

        if (p && p->type == I_TYPE_UNKNOWN)
            free_shared_identifier(p);
        *yyp = '\0';
        yyerrorf(efun_override == OVERRIDE_VAR ? "Undefined variable: %.50s" : "Undefined function: %.50s", wordstart);
        *yyp = c;
        yylval.closure.number = CLOSURE_EFUN_OFFS;
        yylval.closure.inhIndex = 0;
        return L_CLOSURE;
    }

    /* An attempt to override a nomask simul-efun causes
     * a privilege violation. If the master decides to allow
     * this attempt, the efun-override will still be deactivated
     * (iow: a nomask simul-efun overrules an efun override).
     */
    if (efun_override == OVERRIDE_EFUN
     && p->u.global.sim_efun != I_GLOBAL_SEFUN_OTHER
     && simul_efunp[p->u.global.sim_efun].flags & TYPE_MOD_NO_MASK
     && (p->u.global.efun != I_GLOBAL_EFUN_OTHER
#ifdef USE_PYTHON
      || is_python_efun(p)
#endif
        )
     && master_ob && !disable_sefuns
     && (!EVALUATION_TOO_LONG())
       )
    {
        svalue_t *res;

        push_ref_string(inter_sp, STR_NOMASK_SIMUL_EFUN);
        push_c_string(inter_sp, current_loc.file->name);
        push_ref_string(inter_sp, p->name);
        res = apply_master(STR_PRIVILEGE, 3);
        if (!res || res->type != T_NUMBER || res->u.number < 0)
        {
            yyerrorf(
              "Privilege violation: nomask simul_efun %s",
              get_txt(p->name)
            );
            efun_override = MY_FALSE;
        }
        else if (!res->u.number)
        {
            efun_override = MY_FALSE;
        }
    }
    else if (EVALUATION_TOO_LONG())
    {
        yyerrorf("Can't call master::%s for "
                 "'nomask simul_efun %s': eval cost too big"
                , get_txt(STR_PRIVILEGE), get_txt(p->name));
        efun_override = MY_FALSE;
    }

    /* The code will be L_CLOSURE, now determine the right
     * closure number to put into yylval.closure.number.
     * The number is usually the index in the appropriate
     * table, plus an offset indicating the type of the closure.
     *
     * The switch() serves just as a simple try... environment.
     */
    yylval.closure.inhIndex = 0;
    switch(0) { default:

        /* lfun? */
        if ((efun_override == OVERRIDE_NONE || efun_override == OVERRIDE_LFUN)
         && p->u.global.function != I_GLOBAL_FUNCTION_OTHER)
        {
            int i;

            i = p->u.global.function;
            yylval.closure.number = i;
            if (i >= CLOSURE_IDENTIFIER_OFFS)
                yyerrorf(
                  "Too high function index of %s for #'",
                  get_txt(p->name)
                );
            break;
        }

        /* simul-efun? */
        if ((efun_override == OVERRIDE_NONE || efun_override == OVERRIDE_SEFUN)
         && p->u.global.sim_efun != I_GLOBAL_SEFUN_OTHER && !disable_sefuns)
        {
            yylval.closure.number =
              p->u.global.sim_efun + CLOSURE_SIMUL_EFUN_OFFS;
            break;
        }

#ifdef USE_PYTHON
        /* python-defined efun? */
        if ((efun_override == OVERRIDE_NONE || efun_override == OVERRIDE_EFUN)
         && is_python_efun(p))
        {
            yylval.closure.number = p->u.global.python_efun + CLOSURE_PYTHON_EFUN_OFFS;
            break;
        }
#endif

        /* efun? */
        if ((efun_override == OVERRIDE_NONE || efun_override == OVERRIDE_EFUN)
         && p->u.global.efun != I_GLOBAL_EFUN_OTHER)
        {
            yylval.closure.number =
              p->u.global.efun + CLOSURE_EFUN_OFFS;
            if (yylval.closure.number >
                LAST_INSTRUCTION_CODE + CLOSURE_EFUN_OFFS)
            {
                yylval.closure.number =
                  efun_aliases[
                    yylval.closure.number - CLOSURE_EFUN_OFFS
                      - LAST_INSTRUCTION_CODE - 1
                  ] + CLOSURE_EFUN_OFFS;
            }
            break;
        }

        /* object variable? */
        if ((efun_override == OVERRIDE_NONE || efun_override == OVERRIDE_VAR)
         && p->u.global.variable != I_GLOBAL_VARIABLE_OTHER
         && p->u.global.variable != I_GLOBAL_VARIABLE_FUN)
        {
            if (p->u.global.variable & VIRTUAL_VAR_TAG) {
                /* Handling this would require an extra coding of
                 * this closure type, and special treatment in
                 * replace_program_lambda_adjust(). Also deprecated-check in the
                 * L_CLOSURE rule in prolang.y must be adjusted.
                 */
                yyerrorf("closure of virtual variable");
                yylval.closure.number = CLOSURE_IDENTIFIER_OFFS;
                break;
            }
            yylval.closure.number =
              p->u.global.variable + num_virtual_variables +
              CLOSURE_IDENTIFIER_OFFS;
            break;
        }

        /* None of these all */
        {
            char c = *yyp;
            *yyp = 0;
            yyerrorf(efun_override == OVERRIDE_VAR ? "Undefined variable: %.50s" : "Undefined function: %.50s", wordstart);
            *yyp = c;
            yylval.closure.number = CLOSURE_EFUN_OFFS;

            break;
        }
    }
    return L_CLOSURE;
} /* closure() */

/*-------------------------------------------------------------------------*/
static char *
handle_preprocessor_statement (char * in_yyp)

/* The lexer has found a preprocessor statement (<newline>#), an <in_yyp>
 * is pointing to the character after the '#'. Parse the statement and return
 * the new character pointer.
 */

{
    register char *yyp;
    char *sp = NULL; /* Begin of second word */
    Bool quote; /* In "" string? */
    size_t wlen;  /* Length of the preproc keyword */
    char last;
      /* Character last read, used to implement \-sequences */

    /* Copy the first/only line of the preprocessor statement
     * from the input buffer into yytext[] while stripping
     * comments.
     */

    /* Skip initial blanks */
    outp = skip_white(in_yyp);
    yyp = yytext;
    wlen = 0;

    for (quote = MY_FALSE, last = '\0';;)
    {
        register char c = mygetc();
        p_int uc;
        size_t uclen;

        /* Skip comments */
        while (!quote && c == '/')
        {
            char c2;

            if ( (c2 = mygetc()) == '*')
            {
                skip_comment();
                c = mygetc();
            }
            else if (c2 == '/')
            {
                outp = skip_pp_comment(outp);
                current_loc.line--;
                c = '\n';
            }
            else
            {
                --outp;
                break;
            }
        }

        /* If the last character was '\', take this one as
         * as it is, else interpret this character.
         */
        if (last == '\\')
            last = '\0';
        else if (c == '"')
            quote = !quote;
        else
            last = c;

        if (c == '\n')
        {
            /* Remember end of the first word in the line */
            if (!sp)
            {
                sp = yyp;
                wlen = yyp - yytext;
            }
            break;
        }

        /* Need to read the whole UTF8 code. */
        outp--;
        uclen = utf8_to_unicode(outp, 4, &uc);
        if (uclen)
        {
            if (!sp && !unicode_isalunum(uc))
            {
                sp = yyp;
                wlen = yyp - yytext;
            }

            if (yyp >= yytext + MAXLINE - 4 - uclen)
            {
                lexerror("Line too long");
                break;
            }

            do
            {
                *yyp++ = *outp++;
            } while (--uclen);
        }
    }

    /* Terminate the line copied to yytext[] */
    *yyp = '\0';

    /* Remember the end of the first word.
     * Let sp point to the next word then.
     */
    if (sp)
    {
        sp = skip_white(sp);
    }
    else
    {
        /* No end found in the copy loop - the next 'word'
         * will be the terminating '\0'.
         */
        sp = yyp;
        wlen = yyp - yytext;
    }

    /* Evaluate the preprocessor statement */
    if (wordcmp(yytext, "include", wlen) == 0)
    {
        /* Calling myfilbuf() before handle_include() is a waste
         * of time and memory. However, since the include
         * attempt might fail, we have to call it afterwards
         * to make sure that the lex can continue.
         */
        handle_include(sp);

        current_loc.line++;
        myfilbuf();
        current_loc.line--;
    }
    else
    {
       /* Make sure there is enough data in the buffer. */
       current_loc.line++;
       myfilbuf();
       current_loc.line--;

    if (wordcmp(yytext, "define", wlen) == 0)
    {
        if (*sp == '\0')
            yyerror("Missing definition in #define");
        else
            handle_define(sp, quote);
    }
    else if (wordcmp(yytext, "if", wlen) == 0)
    {
        int cond;
        svalue_t sv;

        myungetc('\n');
        add_input(sp);
        cond = cond_get_exp(0, &sv);
        free_svalue(&sv);
        if (mygetc() != '\n')
        {
            yyerror("Condition too complex in #if");
            while (mygetc() != '\n') NOOP;
        }
        else
        {
            current_loc.line++;
            myfilbuf();
            current_loc.line--;
            handle_cond(cond);
        }
    }
    else if (wordcmp(yytext, "ifdef", wlen) == 0)
    {
        checktrail(sp, "ifdef");
        handle_cond(lookup_define(sp) != 0);
    }
    else if (wordcmp(yytext, "ifndef", wlen) == 0)
    {
        checktrail(sp, "ifndef");
        handle_cond(lookup_define(sp) == 0);
    }
    else if (wordcmp(yytext, "else", wlen) == 0)
    {
        if (*sp != '\0')
            warn_trailing_chars("else");

        if (iftop && iftop->state == EXPECT_ELSE)
        {
            lpc_ifstate_t *p = iftop;

            iftop = p->next;
            mempool_free(lexpool, p);
            skip_to("endif", NULL);
        }
        else
        {
            yyerror("Unexpected #else");
        }
    }
    else if (wordcmp(yytext, "elif", wlen) == 0)
    {
        if (iftop && iftop->state == EXPECT_ELSE)
        {
            lpc_ifstate_t *p = iftop;

            iftop = p->next;
            mempool_free(lexpool, p);
            skip_to("endif", NULL);
        }
        else
        {
            yyerror("Unexpected #elif");
        }
    }
    else if (wordcmp(yytext, "endif", wlen) == 0)
    {
        if (*sp != '\0')
            warn_trailing_chars("endif");

        if (iftop
         && (   iftop->state == EXPECT_ENDIF
             || iftop->state == EXPECT_ELSE))
        {
            lpc_ifstate_t *p = iftop;

            iftop = p->next;
            mempool_free(lexpool, p);
        }
        else
        {
            yyerror("Unexpected #endif");
        }
    }
    else if (wordcmp(yytext, "undef", wlen) == 0)
    {
        ident_t *p, **q;
        int h;

        checktrail(sp, "undef");

        /* Lookup identifier <sp> in the ident_table and
         * remove it there if it is a #define'd identifier.
         * If it is a permanent define, park the ident
         * structure in the undefined_permanent_defines list.
         */
        h = identhash(sp);
        for (q = &ident_table[h]; NULL != ( p= *q); q=&p->next)
        {
            if (strcmp(sp, get_txt(p->name)))
                continue;

            if (p->type != I_TYPE_DEFINE) /* failure */
                break;

            if (!p->u.define.permanent)
            {
#if defined(LEXDEBUG)
                fprintf(stderr, "%s #undef define '%s' %d '%s'\n"
                       , time_stamp(), get_txt(p->name)
                       , p->u.define.nargs
                       , p->u.define.exps.str);
                fflush(stderr);
#endif
                if (p->inferior)
                {
                    p->inferior->next = p->next;
                    *q = p->inferior;
                }
                else
                {
                    *q = p->next;
                }
                xfree(p->u.define.exps.str);
                free_mstring(p->name);
                p->name = NULL;
                    /* mark for later freeing by all_defines */
                /* success */
                break;
           }
           else
           {
                if (p->inferior)
                {
                    p->inferior->next = p->next;
                    *q = p->inferior;
                }
                else
                {
                    *q = p->next;
                }
                p->next = undefined_permanent_defines;
                undefined_permanent_defines = p;
                /* success */
                break;
            }
        }
    }
    else if (wordcmp(yytext, "echo", wlen) == 0)
    {
        fprintf(stderr, "%s %s\n", time_stamp(), sp);
    }
    else if (wordcmp(yytext, "pragma", wlen) == 0)
    {
        handle_pragma(sp);
    }
    else if (wordcmp(yytext, "line", wlen) == 0)
    {
        char * end;
        long new_line;

        checktrail(sp, "line");
        new_line = strtol(sp, &end, 0);
        if (end == sp || *end != '\0')
            yyerror("Unrecognised #line directive");
        if (new_line < current_loc.line)
            store_line_number_backward(current_loc.line - new_line);
        current_loc.line = new_line - 1;
    }
    else
    {
        yyerror("Unrecognised # directive");
    }} /* if() { else if () {} } */

    store_line_number_info();
    nexpands = 0;
    current_loc.line++;
    total_lines++;

    return outp;
} /* handle_preprocessor_statement() */

/*-------------------------------------------------------------------------*/
static INLINE int
yylex1 (void)

/* Lex the next lexical element starting from outp and return its code.
 * For single characters, this is the character code itself. Multi-character
 * elements return the associated code define in lang.h.
 * Illegal characters are returned as spaces.
 * If the lexer runs into a fatal error or the end of file, -1 is returned.
 *
 * <depth> is the current nesting depth for local scopes, needed for
 * correct lookup of local identifiers.
 *
 * Some elements return additional information:
 *   L_ASSIGN:  yylval.number is the type of assignment operation
 *              e.g. F_ADD_EQ for '+='.
 *              '=' itself is returned as F_ASSIGN.
 *   L_NUMBER:  yylval.number is the parsed whole number or char constant.
 *   L_FLOAT:   yylval.float_number is the parsed float number.
 *   L_STRING:  last_lex_string is the (tabled) parsed string literal.
 *   L_BYTES:   last_lex_string is the (tabled) parsed string literal.
 *   L_CLOSURE: yylval.closure.number/.inhIndex identifies the closure. See
 *              the source for which value means what (it's a bit longish).
 *   L_QUOTED_AGGREGATE: yylval.number is the number of quotes
 *   L_SYMBOL:  yylval.symbol.name is the (shared) name of the symbol,
 *              yylval.symbol.quotes the number of quotes.
 */

{
    register char *yyp;
    bool is_byte_literal = false;

#define READ_CHAR       clen = utf8_to_unicode(yyp, 4, &c); if (!clen) goto badlex; yyp += clen;
#define RETURN(val)     { outp = yyp; return (val); }

    yyp = outp;

    while (true)
    {
        p_int c;
        size_t clen;

        READ_CHAR;

        switch(c)
        {

            /* --- End Of File --- */
            case CHAR_EOF:

                if (inctop)
                {
                    /* It's the end of an included file: return the previous
                     * file
                     */
                    struct incstate *p;
                    Bool was_string_source = (yyin.fd == -1);

                    p = inctop;

                    /* End the lexing of the included file */
                    close_input_source(false);
                    nexpands = 0;
                    store_include_end(p->inc_offset, p->loc.line);

                    /* Restore the previous state */
                    current_loc = p->loc;
                    if (!was_string_source)
                        current_loc.line++;

                    yyin = p->yyin;
                    saved_char = p->saved_char;
                    inctop = p->next;
                    *linebufend = '\n';
                    yyp = linebufend + 1;
                    linebufstart = &defbuf[defbuf_len] + p->linebufoffset;
                    linebufend   = linebufstart + MAXLINE;
                    mempool_free(lexpool, p);
                    if (!*yyp)
                    {
                        outp = yyp;
                        yyp = _myfilbuf();
                    }
                    break;
                }

                /* Here it's the end of the main file */

                if (iftop)
                {
                    /* Oops, pending #if!
                     * Note the error and clean up the if-stack.
                     */
                    lpc_ifstate_t *p = iftop;

                    yyerror(p->state == EXPECT_ENDIF ? "Missing #endif" : "Missing #else");
                    while(iftop)
                    {
                        p = iftop;
                        iftop = p->next;
                        mempool_free(lexpool, p);
                    }
                }

                /* Return the EOF condition */
                outp = yyp-1;
                return -1;


            /* --- Newline --- */
            case '\n':
            {
                store_line_number_info();
                nexpands = 0;
                current_loc.line++;
                total_lines++;
                if (!*yyp)
                {
                    outp = yyp;
                    yyp = _myfilbuf();
                }
                break;
            }


            /* --- Other line markers --- */
            case 0x1a: /* Used by some MSDOS editors as EOF */
            case '\r':
                *(yyp-1) = *(yyp-2);
                break;


            /* --- Byte Order Mark --- */
            case 0xfeff:
                break;

            /* --- Multi-Char Operators --- */
            case '+':
                READ_CHAR;

                switch(c)
                {
                    case '+':
                        RETURN(L_INC);

                    case '=':
                        yylval.number = F_ADD_EQ;
                        RETURN(L_ASSIGN);

                    default: 
                        yyp -= clen;
                        RETURN('+');
                }

            case '-':
                READ_CHAR;

                switch(c)
                {
                    case '>':
                        RETURN(L_ARROW);

                    case '-':
                        RETURN(L_DEC);

                    case '=':
                        yylval.number = F_SUB_EQ;
                        RETURN(L_ASSIGN);

                    default:
                        yyp -= clen;
                        RETURN('-');
                }

            case '&':
                READ_CHAR;

                switch(c)
                {
                    case '&':
                    {
                        READ_CHAR;

                        switch(c)
                        {
                            case '=':
                                yylval.number = F_LAND_EQ;
                                RETURN(L_ASSIGN);
                            default:
                                yyp -= clen;
                                RETURN(L_LAND);
                        }
                    }

                    case '=':
                        yylval.number = F_AND_EQ;
                        RETURN(L_ASSIGN);

                    default:
                        yyp -= clen;
                        RETURN('&');
                }

            case '|':
                READ_CHAR;

                switch(c)
                {
                    case '|':
                    {
                        READ_CHAR;

                        switch(c)
                        {
                            case '=':
                                yylval.number = F_LOR_EQ;
                                RETURN(L_ASSIGN);

                            default:
                                yyp -= clen;
                                RETURN(L_LOR);
                        }
                    }

                    case '=':
                        yylval.number = F_OR_EQ;
                        RETURN(L_ASSIGN);

                    default:
                        yyp -= clen;
                        RETURN('|');
                }

            case '^':
                if (*yyp == '=')
                {
                    yyp++;
                    yylval.number = F_XOR_EQ;
                    RETURN(L_ASSIGN);
                }
                else
                    RETURN('^');

            case '<':
                READ_CHAR;

                switch(c)
                {
                    case '<':
                        if (*yyp == '=')
                        {
                            yyp++;
                            yylval.number = F_LSH_EQ;
                            RETURN(L_ASSIGN);
                        }
                        else
                            RETURN(L_LSH);

                    case '=':
                        RETURN(L_LE);

                    default:
                        yyp -= clen;
                        RETURN('<');
                }

            case '>':
                READ_CHAR;

                switch(c)
                {
                    case '>':
                        switch(*yyp)
                        {
                            case '=':
                                yyp++;
                                yylval.number = F_RSH_EQ;
                                RETURN(L_ASSIGN);

                            case '>':
                                yyp++;
                                if (*yyp == '=')
                                {
                                    yyp++;
                                    yylval.number = F_RSHL_EQ;
                                    RETURN(L_ASSIGN);
                                }
                                else
                                    RETURN(L_RSHL);

                            default:
                                RETURN(L_RSH);
                        }

                    case '=':
                        RETURN(L_GE);

                    default:
                        yyp -= clen;
                        RETURN('>');
                }

            case '*':
                if (*yyp == '=')
                {
                    yyp++;
                    yylval.number = F_MULT_EQ;
                    RETURN(L_ASSIGN);
                }
                else
                    RETURN('*');

            case '%':
                if (*yyp == '=')
                {
                    yyp++;
                    yylval.number = F_MOD_EQ;
                    RETURN(L_ASSIGN);
                }
                else
                    RETURN('%');

            case '/':
                READ_CHAR;

                switch(c)
                {
                    case '*':
                    {
                        outp = yyp;
                        skip_comment();
                        yyp = outp;
                        if (lex_fatal)
                            return -1;

                        break;
                    }

                    case '/':
                        yyp = skip_pp_comment(yyp);
                        break;

                    case '=':
                        yylval.number = F_DIV_EQ;
                        RETURN(L_ASSIGN);

                    default:
                        yyp -= clen;
                        RETURN('/');
                }
                break;

            case '=':
                if (*yyp == '=')
                {
                    yyp++;
                    RETURN(L_EQ);
                }
                else
                {
                    yylval.number = F_ASSIGN;
                    RETURN(L_ASSIGN);
                }

            case '!':
                if (*yyp == '=')
                {
                    yyp++;
                    RETURN(L_NE);
                }
                else
                    RETURN(L_NOT);

            case '.':
                if (*yyp == '.')
                {
                    yyp++;
                    if (*yyp == '.')
                    {
                        yyp++;
                        RETURN(L_ELLIPSIS);
                    }
                    else
                        RETURN(L_RANGE);
                }
                else
                    RETURN('.');

            case ':':
                switch(*yyp)
                {
                    case ':':
                        yyp++;
                        RETURN(L_COLON_COLON);

                    case ')':
                        yyp++;
                        RETURN(L_END_INLINE);

                    default:
                        RETURN(':');
                }


            /* --- Inline Function --- */
            case '(':
                /* Check for '(:' but ignore '(::' which can occur e.g.
                 * in 'if (::remove())'. However, accept '(:::' e.g. from
                 * '(:::remove()', and '(::)'.
                 */

                if (*yyp == ':'
                 && (yyp[1] != ':' || yyp[2] == ':' || yyp[2] == ')'))
                {
                    yyp++;
                    RETURN(L_BEGIN_INLINE);
                }

                /* FALL THROUGH */


            /* --- Single-char Operators and Punctuation --- */
            /* case '(' is a fall through from above */
            case ';':
            case ')':
            case ',':
            case '{':
            case '}':
            case '~':
            case '[':
            case ']':
            case '?':
                RETURN(c);


            /* --- #: Preprocessor statement or symbol --- */
            case '#':
                if (*yyp == '\'')
                {
                    /* --- #': Closure Symbol --- */
                    return closure(yyp);
                } /* if (#') */

                if (*(yyp-2) == '\n' && !nexpands)
                {
                    /* --- <newline>#: Preprocessor statement --- */

                    yyp = handle_preprocessor_statement(yyp);
                    if (lex_fatal)
                        return -1;

                    if (!*yyp)
                    {
                        outp = yyp;
                        yyp = _myfilbuf();
                    }
                    break;
                }

                goto badlex;


            /* --- ': Character constant or lambda symbol --- */
            case '\'':
            {
                p_int c2;
                size_t c2len;

                READ_CHAR;

                if (c == '\\')
                {
                    /* Parse an escape sequence */

                    if ('\n' != *yyp && CHAR_EOF != *yyp)
                    {
                        char *cp = parse_escaped_char(yyp, false, &yylval.number);
                        if (!cp)
                            yyerror("Illegal character constant");
                        else
                            yyp = cp;
                    }

                    /* Test if it's terminated by a quote (this also
                     * catches the \<nl> and \<eof> case).
                     */
                    if (*yyp++ != '\'')
                    {
                        yyp--;
                        yyerror("Illegal character constant");
                    }

                    RETURN(L_NUMBER);
                }

                c2len = utf8_to_unicode(yyp, 4, &c2);
                if (!c2len)
                    goto badlex;
                yyp += c2len;

                if (c2 != '\''
                  || (   c == '\''
                      && (*yyp == '(' || utf8_isalunum(yyp) || *yyp == '\'')) )
                {
                    /* Parse the symbol or quoted aggregate.
                     *
                     * The test rejects all sequences of the form
                     *   'x'
                     * and
                     *   '''x, with x indicating that the ' character itself
                     *         is meant as the desired constant.
                     *
                     * It accepts all forms of quoted symbols, with one or
                     * more leading ' characters.
                     */

                    char *wordstart;
                    int quotes = 1;

                    /* Count the total number of ' characters, set wordstart
                     * on the first non-quote.
                     */
                    yyp -= clen + c2len;
                    while (*yyp == '\'')
                    {
                        quotes++;
                        yyp++;
                    }
                    wordstart = yyp;

                    /* If the first non-quote is not an alnum, it must
                     * be a quoted aggregrate or an error.
                     */
                    READ_CHAR;
                    if (!iswalpha(c) && c != '_')
                    {
                        if (c == '(' && *yyp == '{')
                        {
                            yyp++;
                            yylval.number = quotes;
                            RETURN(L_QUOTED_AGGREGATE);
                        }

                        yyerror("Illegal character constant");
                        yyp -= clen;
                        RETURN(L_NUMBER);
                    }

                    /* Find the end of the symbol and make it a shared string. */
                    yyp = skip_alunum(yyp);
                    yylval.symbol.name = new_n_unicode_tabled(wordstart, yyp-wordstart);
                    yylval.symbol.quotes = quotes;
                    RETURN(L_SYMBOL);
                }

                /* It's a normal character constant. */
                yylval.number = c;
                RETURN(L_NUMBER);
            }


            /* --- ": String or Bytes Literal --- */
            case '"':
            {
                char *p = yyp;
                bool was_bytes = is_byte_literal;

                is_byte_literal = false; /* for the next string. */

                /* Construct the string in yytext[], terminated with a \0.
                 * ANSI style string concatenation is done by a recursive
                 * call to yylex() after this literal is parsed completely.
                 * This way a mixture of macros and literals is easily
                 * handled.
                 */
                yyp = yytext;
                for(;;)
                {
                    c = *p++;

                    /* No unescaped newlines allowed */
                    if (c == '\n')
                    {
                        outp = p-1;
                        /* myfilbuf(); not needed */
                        if (was_bytes)
                        {
                            lexerror("Newline in byte sequence");
                            return bytes("", 0);
                        }
                        else
                        {
                            lexerror("Newline in string");
                            return string("", 0);
                        }
                    }

                    if (was_bytes && c < 0)
                        lexerror("Illegal character in byte sequence");

                    SAVEC;

                    /* Unescaped ": end of string */
                    if (c == '"')
                    {
                        *--yyp = '\0';
                        break;
                    }

                    /* Handle an escape sequence */
                    if (c == '\\')
                    {
                        yyp--; /* Undo the SAVEC */

                        switch(c = *p++)
                        {
                            case '\r':
                                /* \<cr><lf> defaults to \<lf>, but
                                 * \<cr> puts <cr> into the text.
                                 */
                                if (*p++ != '\n')
                                {
                                    p--;
                                    *yyp++ = c;
                                    break;
                                }
                                /* FALLTHROUGH*/

                            case '\n':
                                /* \<lf> and \<lf><cr> are ignored */
                                store_line_number_info();
                                current_loc.line++;
                                total_lines++;
                                if (*p == CHAR_EOF )
                                {
                                    outp = p;
                                    if (was_bytes)
                                    {
                                        lexerror("End of file (or 0x01 character) in byte sequence");
                                        return bytes("", 0);
                                    }
                                    else
                                    {
                                        lexerror("End of file (or 0x01 character) in string");
                                        return string("", 0);
                                    }
                                }
                                if (!*p)
                                {
                                    outp = p;
                                    p = _myfilbuf();
                                }
                                if (*p++ != '\r')
                                    p--;
                                break;

                            default:
                            {
                                char *cp;
                                p_int lc = 0;

                                cp = parse_escaped_char(p-1, was_bytes, &lc);
                                if (cp)
                                    p = cp;
                                else if (was_bytes)
                                    yyerror("Illegal escaped character in byte sequence.");
                                else
                                    yyerror("Illegal escaped character in string.");

                                if (yyp + 9 >= yytext + MAXLINE)
                                {
                                    lexerror("Line too long");
                                    return was_bytes ? bytes("", 0) : string("", 0);
                                }

                                if (was_bytes)
                                    *yyp++ = lc;
                                else
                                    yyp += unicode_to_utf8(lc, yyp);
                                break;
                            }
                        }
                    }
                } /* for() */

                outp = p;
                return was_bytes ? bytes(yytext, yyp-yytext) : string(yytext, yyp-yytext);
            }


            /* --- Numbers --- */
            case '0':case '1':case '2':case '3':case '4':
            case '5':case '6':case '7':case '8':case '9':
            {
                char *numstart = yyp-clen;
                unsigned long l;
                char ch;
                Bool overflow;

                /* Scan ahead to see if this is a float number */
                while (lexdigit(ch = *yyp++)) NOOP ;

                /* If it's a float (and not a range), simply use strtod()
                 * to convert the float and to update the text pointer.
                 */
                if ('.' == ch && '.' != *yyp)
                {
                    char * numend;  /* Because yyp is 'register' */
                    errno = 0; /* Because strtod() doesn't clear it on success */
                    yylval.float_number = strtod(numstart, &numend);
                    if (errno == ERANGE)
                    {
                        yywarn("Floating point number out of range.");
                    }
                    else if (errno == EINVAL)
                    {
                        yyerror("Floating point number can't be represented.");
                    }
                    outp = numend;
                    return L_FLOAT;
                }

                /* Nope, normal number */
                yyp = parse_number(numstart, &l, &overflow);
                if (overflow || (l > (unsigned long)LONG_MAX+1))
                {
                    /* Don't warn on __INT_MAX__+1 because there
                     * may be a minus preceeding this number.
                     */
                    yywarnf("Number exceeds numeric limits");
                }

                outp = yyp;
                return number((long)l);
            }

            case 'b':
                if (*yyp == '"')
                {
                    is_byte_literal = true;
                    break;
                }
                /* FALLTHROUGH */

            /* --- Character classes and everything else --- */
            default:
                /* --- Identifier --- */
                if (c == '$' || unicode_isalunum(c))
                {
                    ident_t *p;
                    char *wordstart = yyp-clen;

                    /* Find the end of the identifier */
                    yyp = skip_alunum(yyp);

                    /* Lookup/enter the identifier in the ident_table. */
                    p = make_shared_identifier_n(wordstart, yyp-wordstart, I_TYPE_UNKNOWN, 0);

                    if (!p)
                    {
                        lexerror("Out of memory");
                        return 0;
                    }

                    /* Handle the identifier according to its type */
                    switch(p->type)
                    {
                        case I_TYPE_DEFINE:
                            outp = yyp;
                            _expand_define(&p->u.define, p);
                            if (lex_fatal)
                                return -1;

                            yyp=outp;
                            continue;

                        case I_TYPE_RESWORD:
                            outp = yyp;
                            return p->u.code;

                        default:
                            /* _UNKNOWN identifiers get their type assigned by the
                             * parser.
                             */
                            yylval.ident = p;
                            RETURN(L_IDENTIFIER);
                    }
                }

                /* --- White space --- */
                if (iswspace(c))
                    break;

                /* --- All other --- */
                yyp -= clen;
                goto badlex;
        } /* switch (c) */

    } /* for() */

badlex:

    /* We come here after an unexpected character */

    if (lex_fatal)
        return -1;

    {
        char buff[100];
        p_int c;
        size_t clen = utf8_to_unicode(yyp, 4, &c);

        if (!clen)
        {
            /* Not an Unicode character, just print the hex code. */
            sprintf(buff, "Illegal character (hex %02x)", *yyp);
            outp = yyp + 1;
        }
        else
        {
            sprintf(buff, "Illegal character (hex %02" PRIxPINT") '%.*s'", c, (int)clen, yyp);
            outp = yyp + clen;
        }

        yyerror(buff);
        return ' ';
    }

#undef READ_CHAR
#undef RETURN

} /* yylex1() */

/*-------------------------------------------------------------------------*/
int
yylex (void)

/* The lex function called by the parser. The actual lexing is done
 * in yylex1(), this function just does any necessary pre- and post-
 * processing.
 * <depth> is the current nesting depth for local scopes, needed for
 * correct lookup of local identifiers.
 */

{
    int r;

#ifdef LEXDEBUG
    yytext[0] = '\0';
#endif
    r = yylex1();
#ifdef LEXDEBUG
    fprintf(stderr, "%s lex=%d(%s) ", time_stamp(), r, yytext);
#endif
    return r;
}

/*-------------------------------------------------------------------------*/
void
start_new_file (int fd, const char * fname)

/* Start the compilation/lexing of the lpc file opened on file <fd> with
 * name <fname>.
 * This must not be called for included files.
 */

{
    ident_t *p;

    object_file = fname;

    cleanup_source_files();
    free_defines();

    /* Restore the bytes keyword. */
    p = make_shared_identifier("bytes", I_TYPE_RESWORD, 0);
    if (!p)
        fatal("Out of memory\n");
    p->type = I_TYPE_RESWORD;
    p->u.code = L_BYTES_DECL;

    current_loc.file = new_source_file(fname, NULL);
    current_loc.line = 1; /* already used in first _myfilbuf() */

    set_input_source(fd, object_file, NULL);

    if (!defbuf_len)
    {
        defbuf = xalloc(DEFBUF_1STLEN);
        defbuf_len = DEFBUF_1STLEN;
    }

    *(outp = linebufend = (linebufstart = defbuf + DEFMAX) + MAXLINE) = '\0';

    _myfilbuf();

    lex_fatal = MY_FALSE;

    pragma_check_overloads = MY_TRUE;
    pragma_strict_types = PRAGMA_WEAK_TYPES;
    instrs[F_CALL_OTHER].ret_type = lpctype_mixed;
    instrs[F_CALL_DIRECT].ret_type = lpctype_mixed;
    instrs[F_CALL_STRICT].ret_type = lpctype_mixed;
    instrs[F_CALL_DIRECT_STRICT].ret_type = lpctype_mixed;
    pragma_save_types = MY_FALSE;
    pragma_no_clone = MY_FALSE;
    pragma_no_inherit = MY_FALSE;
    pragma_no_shadow = MY_FALSE;
    pragma_pedantic = MY_FALSE;
    pragma_warn_missing_return = MY_TRUE;
    pragma_warn_deprecated = MY_TRUE;
    pragma_range_check = MY_FALSE;
    pragma_warn_empty_casts = MY_TRUE;
    pragma_share_variables = share_variables;
    pragma_rtt_checks = MY_FALSE;
    pragma_warn_rtt_checks = MY_FALSE;
    pragma_no_bytes_type = false;
    pragma_warn_unused_variables = false;

    nexpands = 0;

    add_auto_include(object_file, NULL, MY_FALSE);
} /* start_new_file() */

/*-------------------------------------------------------------------------*/
void
end_new_file (void)

/* Clean up after a compilation terminated (successfully or not).
 */

{
    while (inctop)
    {
        struct incstate *p;
        p = inctop;
        close_input_source(false);
        yyin = p->yyin;
        inctop = p->next;
    }

    iftop = NULL;

    close_input_source(true);
    cleanup_source_files();

    mempool_reset(lexpool);
      /* Deallocates all incstates and ifstates at once */

    if (defbuf_len > DEFBUF_1STLEN)
    {
        xfree(defbuf);
        defbuf = NULL;
        defbuf_len = 0;
    }

    if (last_lex_string)
    {
        free_mstring(last_lex_string);
        last_lex_string = NULL;
    }

} /* end_new_file() */

/*-------------------------------------------------------------------------*/
void
lex_close (char *msg)

/* End the current lexing properly (ie. by calling end_new_file())
 * and throw the error message <msg>. If <msg> is NULL, a message
 * giving the current include depth.
 *
 * This function is used from two places: from within lang.c (at them
 * moment only for 'Out of memory') obviously, but also from the efun
 * write_file() if it is called from within a compile, e.g. to write
 * the error log.
 */

{
    if (!msg)
    {
        /* Count the include depth and make a nice message */

        int i;
        struct incstate *p;
        static char buf[] =
            "File descriptors exhausted, include nesting: 12345678";

        for (i = 0, p = inctop; p; p = p->next)
            i++;

        /* skip back terminating \0 and 8 digits */
        sprintf(buf + sizeof buf - 9, "%d", i);
        msg = buf;
    }

    end_new_file();
    outp = ("##")+1; /* TODO: Not really nice */

    lexerror(msg);
} /* lex_close() */

/*-------------------------------------------------------------------------*/
char *
get_f_name (int n)

/* Return the name of instruction <n>, it if has one.
 * The result is a pointer to a static buffer.
 */

{
    if (instrs[n].name)
        return instrs[n].name;
    else
    {
        static char buf[30];
        sprintf(buf, "<OTHER %d>", n);
        return buf;
    }
} /* get_f_name() */

/*-------------------------------------------------------------------------*/
static Bool
refill (Bool quote)

/* Read the next line from the input buffer into yytext[], skipping
 * comments, reading the final \n as space.
 * <quote> is true if at the time of call the text is supposed
 * to be within a string literal.
 * Result is the new value for <quote>: true if the next character to
 *   read is part of a string literal.
 */
{
    char *p;
    int c;
    char last = '\0';

    p = yytext;
    do
    {
        c = mygetc();

        if (c == '/' && !quote)
        {
            last = '\0';
            if (gobble('*'))
            {
                skip_comment();
                continue;
            }
            else if (gobble('/'))
            {
                outp = skip_pp_comment(outp);
                current_loc.line--;
                c = '\n';
            }
        }
        else if (last == '\\')
        {
            /* Take the current character as it is */
            last = '\0';
        }
        else if (c == '"')
            quote = !quote;
        else
            last = (char)c;

        if (p < yytext+MAXLINE-5)
            *p++ = (char)c;
        else
        {
            lexerror("Line too long");
            break;
        }
    } while (c != '\n' && c != CHAR_EOF);


    /* Refill the input buffer */
    current_loc.line+=2;        /* For correct erorr messages, because of yytext, current_loc is always one line behind. */
    myfilbuf();
    current_loc.line--;

    /* Replace the trailing \n by a space */
    if (p[-1] == '\n')
        p[-1] = ' ';
    *p = '\0';

    nexpands = 0;
    store_line_number_info();

    return quote;
} /* refill() */

/*-------------------------------------------------------------------------*/
static void
handle_define (char *yyt, Bool quote)

/* This function is called from yylex1() to handle '#define' statements.
 * The text of the line with the statement is in yytext[], <yyt> points
 * to the first word after '#define'. <quote> is true if at the end
 * of the line a string literal was still open.
 */

{
  /* Get the identfier (or punctuation) pointed to by <src> and copy it
   * as null-terminated string to <dest>, but at maximal <size>-1 characters.
   * Set <src> to the first character after that.
   */
#define GETALPHA(src, dest, size)                       \
    {                                                   \
        char * wordstart = src;                         \
        src = skip_alunum(src);                         \
                                                        \
        if (src - wordstart >= size)                    \
        {                                               \
            lexerror("Name too long");                  \
            return;                                     \
        }                                               \
                                                        \
        memcpy(dest, wordstart, src-wordstart);         \
        dest[src-wordstart] = 0;                        \
    }

    source_loc_t loc;         /* Location of the #define */
    char namebuf[NSIZE];      /* temp buffer for read identifiers */
    char args[NARGS][NSIZE];  /* parsed argument names of function macros */
    char mtext[MLEN];
      /* replacement text, with arguments replaced by the MARKS characters
       */
    char *p;                  /* current text pointer */

    loc = current_loc;

    p = yyt;
    strcat(yyt, " "); /* Make sure GETALPHA terminates */

    /* Get the defined name */
    GETALPHA(p, namebuf, NSIZE);

    if (*p == '(')
    {
        /* --- Function Macro --- */

        short arg;         /* Number of macro arguments */
        Bool inid;         /* true: parsing an identifier */
        char *ids = NULL;  /* Start of current identifier */
        char *q;           /* Destination for parsed text */

        p++;        /* skip '(' and following whitespace */
        p = skip_white(p);

        /* Parse the arguments (if any) */

        if (*p == ')')
        {
            /* no arguments */
            arg = 0;
        }
        else
        {
            /* Parse up to NARGS-1 arguments */

            for (arg = 0; arg < NARGS; )
            {
                /* Get the argname directly into args[][] */
                GETALPHA(p, args[arg], NSIZE);
                arg++;

                p = skip_white(p);

                /* ')' -> no further argument */

                if (*p == ')')
                    break;

                /* else a ',' is expected as separator */
                if (*p++ != ',')
                {
                    yyerror("Missing ',' in #define parameter list");
                    return;
                }

                p = skip_white(p);
            }

            if (arg == NARGS)
            {
                lexerrorf("Too many macro arguments");
                return;
            }
        }

        p++;  /* skip ')' */

        /* Parse the replacement text into mtext[], performing
         * macro argument marking as necessary.
         */

        for (inid = MY_FALSE, q = mtext; *p && *p != CHAR_EOF; )
        {
            p_int c;
            size_t len = utf8_to_unicode(p, 4, &c);
            if (!len)
            {
                // We'll take the character as it is.
                c = *(unsigned char*)p;
                len++;
            }

            /* Identifiers are parsed until complete, with the first
             * character pointed to by <ids>.
             */

            if (unicode_isalunum(c))
            {
                /* Identifier. If inid is false, it is a new one.
                 */

                if (!inid)
                {
                    inid = MY_TRUE;
                    ids = p;
                }
            }
            else
            {
                /* Not an identifier, or, if inid is true, the end
                 * of one.
                 */

                if (inid)
                {
                    int idlen = p - ids;
                    size_t l;
                    int n;

                    /* Check if the identifier matches one of the
                     * function arguments. If yes, replace it in mtext[]
                     * by the MARKS sequence.
                     */
                    for (n = 0; n < arg; n++)
                    {
                        l = strlen(args[n]);
                        if (l == (size_t)idlen && strncmp(args[n], ids, l) == 0)
                        {
                            q -= idlen;
                            *q++ = (char)MARKS;
                            *q++ = (char)(n+MARKS+1);
                            break;
                        }
                    }
                    inid = MY_FALSE;
                }
            }

            /* Whatever the character is, for now store it in mtext[].
             * Literal '@' are escaped.
             */
            if (q + len + 1 > mtext + MLEN)
            {
                lexerror("Macro text too long");
                return;
            }

            do
            {
                *q++ = *p++;
            } while (--len);

            if (p[-1] == MARKS)
                *q++ = MARKS;

            /* If we are at line's end and it is escaped with '\',
             * get the next line and continue.
             */
            if (!*p)
            {
                if (p > yytext + 1 && p[-2] == '\\')
                {
                    q -= 2;
                    quote = refill(quote);
                    p = yytext;
                }
                else if (p > yytext + 2 && p[-2] == '\r' && p[-3] == '\\' )
                {
                    q -= 3;
                    quote = refill(quote);
                    p = yytext;
                }
            }
        }

        /* If the defined was ended by EOF instead of lineend,
         * we have to pass on the EOF to the caller.
         */
        if (*p == CHAR_EOF)
        {
            myungetc(*p);
        }

        /* Terminate the text and add the macro */
        *--q = '\0';
        add_define(namebuf, arg, mtext, loc);
    }
    else
    {
        /* --- Normal Macro --- */

        char *q;           /* Destination for parsed text */

        /* Parse the replacement text into mtext[].
         */

        for (q = mtext; *p && *p != CHAR_EOF; )
        {
            *q = *p++;
            if (q < mtext+MLEN-2)
                q++;
            else
            {
                lexerror("Macro text too long");
                return;
            }

            /* If we are at line's end and it is escaped with '\',
             * get the next line and continue.
             */
            if (!*p)
            {
                if (p > yytext + 1 && p[-2] == '\\')
                {
                    q -= 2;
                    quote = refill(quote);
                    p = yytext;
                }
                else if (p > yytext + 2 && p[-2] == '\r' && p[-3] == '\\' )
                {
                    q -= 3;
                    quote = refill(quote);
                    p = yytext;
                }
            }
        }

        /* If the defined was ended by EOF instead of lineend,
         * we have to pass on the EOF to the caller.
         */
        if (*p == CHAR_EOF)
        {
            myungetc(*p);
        }

        /* Terminate the text and add the macro */
        *--q = '\0';
        add_define(namebuf, -1, mtext, loc);
    }

#undef GETALPHA
#undef SKIPWHITE

} /* handle_define() */

/*-------------------------------------------------------------------------*/
static void
add_define (char *name, short nargs, char *exps, source_loc_t loc)

/* Add a new macro definition for macro <name> with <nargs> arguments
 * and the replacement text <exps>. The positions where the arguments
 * are to be put into <exps> have to be marked with the MARKS character
 * as described elsewhere. The macro is defined at <loc> in the source.
 *
 * The new macro is stored in the ident_table[] and also put into
 * the list of all_defines.
 *
 * If the macro <name> is already defined, an error is generated.
 */

{
    ident_t *p;

    /* Lookup/create a new identifier entry */
    p = make_shared_identifier(name, I_TYPE_DEFINE, 0);
    if (!p)
    {
        lexerrorf("Out of memory for new macro '%s'", name);
        return;
    }

    /* If such a macro already exists with different meaning,
     * generate an error. If the meaning doesn't change, generate
     * a warning.
     */
    if (p->type != I_TYPE_UNKNOWN)
    {
        char buf[200+NSIZE+MAXPATHLEN];

        if (current_loc.line <= 0)
            sprintf(buf, "(in auto include text) #define %s already defined", name);
        else
            sprintf(buf, "#define %s already defined", name);

        if (p->u.define.loc.file != NULL)
        {
            char * add = &buf[strlen(buf)];

            sprintf(add, " (from %s line %d)"
                   , p->u.define.loc.file->name, p->u.define.loc.line);
        }

        if (nargs != p->u.define.nargs
         || p->u.define.special
         || strcmp(exps,p->u.define.exps.str) != 0)
        {
            yyerror(buf);
            return;
        }
        else
        {
            yywarn(buf);
        }
    }
    else
    {
        /* New macro: initialise the ident.u.define and
         * add it to the list of defines.
         */

        p->type = I_TYPE_DEFINE;
        p->u.define.nargs = nargs;
        p->u.define.permanent = MY_FALSE;
        p->u.define.special = MY_FALSE;
        if ( !(p->u.define.exps.str = xalloc(strlen(exps)+1)) )
        {
            free_shared_identifier(p);
            lexerrorf("Out of memory for new macro '%s'", name);
            return;
        }
        strcpy(p->u.define.exps.str, exps);
        p->u.define.loc = loc;

        p->next_all = all_defines;
        all_defines = p;
#if defined(LEXDEBUG)
        fprintf(stderr, "%s define '%s' %d '%s'\n"
               , time_stamp(), name, nargs, exps);
#endif
    }
} /* add_define() */

/*-------------------------------------------------------------------------*/
static ident_t*
add_permanent_define (char *name, short nargs)

/* Add a new permanent macro definition for macro <name>
 * with <nargs> arguments and returns the identifier for that define.
 * The replacement text have to be put into that by the caller.
 *
 * The new macro is stored in the ident_table[] and also put into
 * the list of permanent_defines.
 *
 * If the macro <name> is already defined, an error is generated.
 */

{
    ident_t *p;

    /* Lookup/create a new identifier entry */
    p = make_shared_identifier(name, I_TYPE_DEFINE, 0);
    if (!p)
    {
        errorf("Out of memory for permanent macro '%s'\n", name);
        return NULL;
    }

    /* If such a macro already exists, generate an error.
     */
    if (p->type != I_TYPE_UNKNOWN)
    {
        errorf("Permanent #define %s already defined\n", name);
        return NULL;
    }

    /* New macro: initialise the ident.u.define and
     * add it to the list of permanent defines.
     */

    p->type = I_TYPE_DEFINE;
    p->u.define.nargs = nargs;
    p->u.define.permanent = MY_TRUE;
    p->u.define.loc.file = NULL;
    p->u.define.loc.line = 0;
    p->next_all = permanent_defines;
    permanent_defines = p;

    return p;
} /* add_permanent_define() */

/*-------------------------------------------------------------------------*/
static void
add_permanent_define_str (char *name, short nargs, char *text)

/* Add a new permanent macro definition for macro <name>
 * with <nargs> arguments and the replacement text <text>.
 * The positions where the arguments are to be put into <text> have to be
 * marked with the MARKS character as described elsewhere.
 *
 * The text is assumed to be 7-bit-ASCII.
 *
 * The new macro is stored in the ident_table[] and also put into
 * the list of permanent_defines.
 *
 * If the macro <name> is already defined, an error is generated.
 */

{
    ident_t *p = add_permanent_define(name, nargs);
    if (!p)
        return;

    p->u.define.special = false;
    p->u.define.exps.str = string_copy(text);

} /* add_permanent_define_str() */

static void add_permanent_define_fun(char *name, short nargs, defn_fun fun)

/* Add a new permanent macro definition for macro <name>
 * with <nargs> arguments and a function <fun>.
 * Whenever the define is referenced the function is called
 * with the macro arguments. For plain defines (nargs == -1) the
 * function then shall return the replacement string, for macros
 * (nargs >= 0) the function shall add the result via add_input().
 *
 * The new macro is stored in the ident_table[] and also put into
 * the list of permanent_defines.
 *
 * If the macro <name> is already defined, an error is generated.
 */

{
    ident_t *p = add_permanent_define(name, nargs);
    if (!p)
        return;

    p->u.define.special = true;
    p->u.define.exps.fun = fun;
} /* add_permanent_define_fun() */
/*-------------------------------------------------------------------------*/
void
free_defines (void)

/* Free all non-permanent defines, and undo any undefine of a permanent
 * define.
 *
 * Also called from the garbage collector and simul_efun.c
 */

{
    ident_t *p, *q;

    /* Free all non-permanent defines */

    for (p = all_defines; p; p = q)
    {
        q = p->next_all;
        if (p->name)
        {
            if (!p->u.define.special)
                xfree(p->u.define.exps.str);
            free_shared_identifier(p);
        }
        else
        {
            /* has been undef'd. */
            xfree(p);
        }
    }
    all_defines = NULL;


    /* Reactivate undefined permanent defines */

    for (p = undefined_permanent_defines; p; p = q)
    {
        ident_t *curr, **prev;

        q = p->next;
        p->next = NULL;
        prev = &ident_table[p->hash];
        while ( NULL != (curr = *prev) )
        {
            if (curr->name == p->name) /* found it */
            {
                p->next = curr->next;
                break;
            }
            prev = &curr->next;
        } /* not found, create new one */
        p->inferior = curr;
        *prev = p;
    }
    undefined_permanent_defines = NULL;
    nexpands = 0;
} /* free_defines() */

/*-------------------------------------------------------------------------*/
static ident_t *
lookup_define (char *s)

/* Lookup the name <s> in the identtable and return a pointer to its
 * ident structure if it is a define. Return NULL else.
 */

{
    ident_t *curr, *prev;
    int h;

    h = identhash(s);

    curr = ident_table[h];
    prev = 0;
    while (curr)
    {
        if (!strcmp(get_txt(curr->name), s)) /* found it */
        {
            if (prev) /* not at head of list */
            {
                prev->next = curr->next;
                curr->next = ident_table[h];
                ident_table[h] = curr;
            }
            if (curr->type == I_TYPE_DEFINE)
                return curr;
            return NULL;
        }
        prev = curr;
        curr = curr->next;
    } /* not found */

    return NULL;
} /* lookup_define() */


/*-------------------------------------------------------------------------*/
static Bool
expand_define (void)

/* Check if yytext[] holds a macro and expand it if it is.
 * Return true if it was expanded, false if not.
 */

{
    ident_t *p;

    p = lookup_define(yytext);
    if (!p) {
        return MY_FALSE;
    }
    return _expand_define(&p->u.define, p);
} /* expand_define() */

/*-------------------------------------------------------------------------*/
static void
skip_outp_white_and_comments ()

/* Move <outp> to the next character that is not a whitespace
 * or part of a comment. Reads new lines as needed.
 */

{
    while (true)
    {
        outp = skip_white(outp);
        switch (*outp++)
        {
            case '/':
                if (gobble('*'))
                {
                    skip_comment();
                    break;
                }
                else if (gobble('/'))
                {
                    outp = skip_pp_comment(outp);
                    current_loc.line--;
                    // FALLTHROUGH
                }
                else
                {
                    outp--;
                    return;
                }

            // FALLTHROUGH from C++ comments.
            case '\n':
                myfilbuf();
                store_line_number_info();
                current_loc.line++;
                total_lines++;
                break;

            default:
                outp--;
                return;
        }
    }

} /* skip_outp_white_and_comments() */

/*-------------------------------------------------------------------------*/
static Bool
_expand_define (struct defn *p, ident_t * macro)

/* Expand the macro <p> and add_input() the expanded text.
 * For function macros, the function expects the next non-white character
 * in the input buffer to be the opening '(' of the argument list.
 * <macro> is the struct ident_s entry and is needed just for error
 * messages.
 *
 * Return true if the expansion was successfull, false if not.
 */

{
    static char *expbuf = NULL;
      /* The arguments of a function macro, separated by '\0' characters.
       */
    static char *buf = NULL;
      /* Construction buffer for the expanded macro text.
       */

      /* Both buffers are allocated on the first call to the
       * function and reused thereafter. Putting them on the
       * stack would make _expand_define() reentrant, but
       * very slow on systems without proper alloca().
       * Right now the only possibility for a recursive call
       * is an error during the expansion, with error handling requesting
       * another expansion. In this case, reentrancy is not an issue
       * because after returning from the error, the function itself
       * returns immediately.
       *
       * But should the need ever arise, the old fragments may be
       * changed to implement a stack of buffers. Using the stack-mempool
       * allocator, this could even be efficient.
       */

#if 0
    static int mutex = 0;
      /* TODO: The mutex may be used to implement a stack of buffers if needed.
       */

    /* TODO: This was a test for recursive calls. If a stack of buffers is
     * TODO:: needed, this code fragments allow an easy implementation,
     * TODO:: especially because the DEMUTEX macros are already where
     * TODO:: they have to be.
     */
    if (mutex++)
    {
        lexerror("Recursive call to _expand_define()");
        mutex--;
        return 0;
    }
#define DEMUTEX mutex--
#else
#define DEMUTEX NOOP
#endif

    /* Allocate the buffers if not done already */
    if (!expbuf)
        expbuf = pxalloc(DEFMAX);
    if (!buf)
        buf = pxalloc(DEFMAX);
    if (!expbuf || !buf) {
        lexerror("Stack overflow");
        DEMUTEX;
        return 0;
    }

    /* No more than EXPANDMAX expansions per line */
    if (nexpands++ > EXPANDMAX)
    {
        lexerror("Too many macro expansions");
        DEMUTEX;
        return MY_FALSE;
    }

    if (p->nargs == -1)
    {
        /* --- Normal Macro --- */

        if (!p->special)
        {
            add_input(p->exps.str);
        }
        else
        {
            char *e = (*p->exps.fun)(NULL);
            if (!e)
            {
                lexerror("Out of memory");
                DEMUTEX;
                return 0;
            }
            add_input(e);
            xfree(e);
        }

        /* That's it. Jump to the function's end now. */
    }
    else
    {
        /* --- Function Macro --- */

        char *args[NARGS];
          /* Pointers into expbuf[] to the beginning of the actual
           * macro arguments.
           */

        char *expptr;  /* Pointer to replacement text */
        char *bufptr;  /* Pointer into buf[] when expanding */

        int brakcnt = 0; /* Number of pending open '[' */
        int parcnt = 0;  /* Number of pending open' (' */
        Bool dquote = MY_FALSE; /* true: in "" */
        Bool squote = MY_FALSE; /* true: in '' */
        int argnum;           /* Number of parsed macro arguments */

        /* Look for the argument list */
        skip_outp_white_and_comments();
        if (*outp++ != '(')
        {
            yyerrorf("Macro '%s': Missing '(' in call", get_txt(macro->name));
            DEMUTEX;
            return MY_FALSE;
        }

        /* Parse the macro arguments and store them in args[].
         * This is a bit complex as we have to care for character
         * constants, string literals, parentheses, symbols and
         * comments.
         */

        skip_outp_white_and_comments();
        if (*outp == ')')
        {
            argnum = 0;  /* No args */
            outp++;
        }
        else
        {
            /* Setup */

            char *inptr = outp;         /* Next character to read from input buffer */
            char *argptr = expbuf;      /* Pointer into expbuf[] when parsing the args */
            char *whitestart = NULL;    /* Pointer into expbuf[] to the tailing white spaces. */
            args[0] = argptr;

            for (argnum = 0;;)
            {
                char c;
                /* expbuf holds DEFMAX bytes and we need space
                 * for at least one unicode character.
                 */
                if ((argptr-expbuf) >= DEFMAX - 5)
                {
                    lexerrorf("Macro '%s': argument overflow", get_txt(macro->name));
                    DEMUTEX;
                    return MY_FALSE;
                }

                switch(c = *inptr++)
                {
                    case '"' :
                        /* Begin of string literal, or '"' constant */
                        if (!squote)
                            dquote = !dquote;
                        *argptr++ = c;
                        whitestart = NULL;
                        continue;

                    case '#':
                        /* Outside of strings it must be a #'symbol.
                         */
                        *argptr++ = c;
                        whitestart = NULL;
                        if (!squote && !dquote && *inptr == '\'')
                        {
                            char *wordstart = ++inptr;
                            *argptr++ = '\'';

                            inptr = skip_alunum(inptr);

                            if (wordstart == inptr)
                            {
                                /* Not a word, it's an operator? */
                                const char *end;

                                if (symbol_operator(inptr, &end) < 0)
                                {
                                    yyerror("Missing function name after #'");
                                }
                                inptr = (char *)end;
                            }

                            /* Copy the whole symbol name. */
                            if (argptr - expbuf + inptr - wordstart >= DEFMAX - 1)
                            {
                                lexerrorf("Macro '%s': argument overflow", get_txt(macro->name));
                                DEMUTEX;
                                return MY_FALSE;
                            }

                            memcpy(argptr, wordstart, inptr - wordstart);
                            argptr += inptr - wordstart;
                        }
                        continue;

                    case '\'':
                        /* Begin of character constant or quoted symbol.
                         */
                        if (!dquote)
                        {
                            size_t clen = utf8_isalunum(inptr);
                            if ((!clen || inptr[clen] == '\'')
                             && (*inptr != '(' || inptr[1] != '{'))
                            {
                                squote = !squote;
                            }
                        }
                        *argptr++ = c;
                        whitestart = NULL;
                        continue;

                    case '[' :
                        /* Begin of array/mapping index.
                         */
                        if (!squote && !dquote)
                            brakcnt++;
                        *argptr++ = c;
                        whitestart = NULL;
                        continue;

                    case ']' :
                        /* End of array/mapping index.
                         */
                        if (!squote && !dquote && brakcnt > 0)
                        {
                            brakcnt--;
                        }
                        *argptr++ = c;
                        whitestart = NULL;
                        continue;

                    case '(' :
                        /* Begin of nested expression.
                         */
                        if (!squote && !dquote)
                            parcnt++;
                        *argptr++ = c;
                        whitestart = NULL;
                        continue;

                    case ')' :
                        /* End of nested expression.
                         */
                        if (!squote && !dquote)
                        {
                            parcnt--;
                            if (parcnt < 0)
                            {
                                /* We found the end of the argument list. Remove
                                 * trailing whitespace and terminate the arg. */
                                if (whitestart)
                                    argptr = whitestart;
                                *argptr++ = '\0'; // then terminate the arg.
                                argnum++;
                                break;
                            }
                        }
                        *argptr++ = c;
                        whitestart = NULL;
                        continue;

                    case '\\':
                        /* In strings, escaped sequence.
                         */
                        *argptr++ = c;
                        whitestart = NULL;
                        if (squote || dquote)
                        {
                            c = *inptr++;
                            if (c == '\r')
                                c = *inptr++;
                            if (c == '\n')  /* nope! This wracks consistency! */
                            {
                                store_line_number_info();
                                current_loc.line++;
                                total_lines++;
                                if (!*inptr)
                                {
                                    outp = inptr;
                                    inptr = _myfilbuf();
                                }
                                argptr--;        /* alas, long strings should work. */
                                continue;
                            }
                            if (c == CHAR_EOF) /* can't quote THAT */
                            {
                                inptr--;
                                continue;
                            }
                            *argptr++ = c;
                        }
                        continue;

                    case '\n':
                        /* Next line.
                         */
                        store_line_number_info();
                        current_loc.line++;
                        total_lines++;
                        if (!whitestart)
                            whitestart = argptr;
                        *argptr++ = ' ';
                        if (!*inptr)
                        {
                            outp = inptr;
                            inptr = _myfilbuf();
                        }
                        if (squote || dquote)
                        {
                            lexerror("Newline in string");
                            DEMUTEX;
                            return MY_FALSE;
                        }
                        continue;

                    case ',':
                        /* Argument separation
                         */
                        if (!parcnt && !dquote && !squote && !brakcnt)
                        {
                            /* Remove trailing whitespace and terminate the arg. */
                            if (whitestart)
                                argptr = whitestart;
                            *argptr++ = '\0'; // then terminate the arg.
                            // I don't skip the leading whitespace for the next
                            // argument because there may be things like
                            // linebreaks between two args which I don't want to
                            // deal with in this case. This will be done below.
                            args[++argnum] = argptr;
                            if (argnum == NARGS - 1)
                            {
                                lexerror("Maximum macro argument count exceeded");
                                DEMUTEX;
                                return MY_FALSE;
                            }
                            continue;
                        }
                        *argptr++ = c;
                        whitestart = NULL;
                        continue;

                    case CHAR_EOF:
                        lexerror("Unexpected end of file (or a spurious 0x01 character)");
                        DEMUTEX;
                        return MY_FALSE;

                    case '/':
                        /* Probable comment
                         */
                        if (!squote && !dquote)
                        {
                            if ( (c = *inptr++) == '*')
                            {
                                outp = inptr;
                                skip_comment();
                                inptr = outp;
                            }
                            else if ( c == '/')
                            {
                                inptr = skip_pp_comment(inptr);
                            }
                            else
                            {
                                --inptr;
                                *argptr++ = '/';
                                whitestart = NULL;
                            }
                            continue;
                        }

                    default:
                    {
                        p_int ch;
                        size_t clen = utf8_to_unicode(inptr-1, 4, &ch);

                        if (!clen)
                        {
                            *argptr++ = c;
                            whitestart = NULL;
                            continue;
                        }

                        if (!iswspace(ch))
                            whitestart = NULL;
                        else if(!whitestart)
                            whitestart = argptr;

                        inptr--;
                        do
                        {
                            *argptr++ = *inptr++;
                        } while(--clen);

                        continue;
                    }
                } /* end switch */

                /* The only way to come here is in the case ')' when the
                 * end of the argument list is detected. Hence, we can
                 * break the for().
                 */
                break;
            } /* for(argnum = 0..NARGS) */
            outp = inptr;
        } /* if (normal or function macro) */

        /* Proper number of arguments? */
        if (argnum != p->nargs)
        {
            yyerrorf("Macro '%s': Wrong number of arguments", get_txt(macro->name));
            DEMUTEX;
            return MY_FALSE;
        }

        /* (Don't) handle dynamic function macros */
        if (p->special)
        {
            (void)(*p->exps.fun)(args);
            DEMUTEX;
            return MY_TRUE;
        }

        /* Construct the expanded macro text in buf[] by simple
         * copy and replace.
         */

        bufptr = buf;
        expptr = p->exps.str;
        while (*expptr)
        {
            if (*expptr == MARKS)
            {
                if (*++expptr == MARKS)
                    *bufptr++ = *expptr++;
                else
                {
                    char *argptr = args[*expptr++ - MARKS - 1];
                    // the args may have leading whitespace (see above),
                    // we skip it here.
                    argptr = skip_white(argptr);

                    for ( ; *argptr ; )
                    {
                        *bufptr++ = *argptr++;
                        if (bufptr >= buf+DEFMAX)
                        {
                            lexerror("Macro expansion overflow");
                            DEMUTEX;
                            return MY_FALSE;
                        }
                    }
                }
            }
            else
            {
                *bufptr++ = *expptr++;
                if (bufptr >= buf+DEFMAX)
                {
                    lexerror("Macro expansion overflow");
                    DEMUTEX;
                    return MY_FALSE;
                }
            }
        }

        /* Terminate the expanded text and add it to the input */
        *bufptr++ = '\0';
        add_input(buf);
    }

    /* That's it. */

    DEMUTEX;
    return MY_TRUE;
}

/*-------------------------------------------------------------------------*/
static p_int
exgetc (char** chpos)

/* Get the first character  of the next element of a condition
 * and return it, leaving the input pointing to the rest of it.
 * Sets chpos to point to the first byte of the character.
 * Comments are skipped, identifiers not defined as macros are
 * replaced with ' 0 ', the predicate 'defined(<name>)' is
 * replaced with ' 0 ' or ' 1 ' depending on the result.
 */

{
    for (;;)
    {
        p_int c;
        size_t clen;

        clen = utf8_to_unicode(outp, 4, &c);
        if (!clen)
        {
            char buff[100];
            sprintf(buff, "Illegal character (hex %02x) '%c'", *outp, *outp);
            yyerror(buff);

            if (chpos != NULL)
                *chpos = outp;
            outp++;
            return ' ';
        }

        if ( iswalpha(c) || c=='_' )
        {
            /* It's an identifier, maybe a macro name, maybe it's
             * an 'defined()' predicate.
             */

            /* Get the full identifier in yytext[] */
            char *wordstart = outp;
            outp = skip_alunum(outp + clen);

            memcpy(yytext, wordstart, outp - wordstart);
            yytext[outp - wordstart] = 0;

            if (strcmp(yytext, "defined") == 0)
            {
                /* handle the 'defined' predicate */
                outp = skip_white(outp);
                if (*outp != '(')
                {
                    yyerror("Missing ( in defined");
                    continue;
                }

                wordstart = skip_white(outp + 1);
                outp = skip_alunum(wordstart);

                memcpy(yytext, wordstart, outp - wordstart);
                yytext[outp - wordstart] = 0;

                outp = skip_white(outp);
                if (*outp != ')')
                {
                    yyerror("Missing ) in defined");
                    continue;
                }

                outp = skip_white(outp + 1);

                if (lookup_define(yytext))
                    add_input(" 1 ");
                else
                    add_input(" 0 ");
            }
            else
            {
                /* Simple identifier */
                if (!expand_define())
                    add_input(" 0 ");
            }
        }
        else if (c == '\\' && (outp[clen] == '\n' || outp[clen] == '\r'))
        {
            /* Escaped new line: read the next line, strip
             * all comments, and then add the result again
             * for reparsing.
             */

            Bool quote;
            char *yyp;

            outp += clen + 1;
            if (outp[-1] == '\r' && *outp == '\n')
                outp++;

            current_loc.line++;
            total_lines++;
            myfilbuf();

            yyp = yytext;
            for(quote = MY_FALSE;;)
            {
                c = (unsigned char)mygetc();
                if (c == '"')
                    quote = !quote;
                while(!quote && c == '/')
                {
                    /* handle comments cpp-like */
                    char c2;

                    if ( (c2 = mygetc()) == '*')
                    {
                        skip_comment();
                        c=(unsigned char)mygetc();
                    }
                    else if (c2 == '/')
                    {
                        outp = skip_pp_comment(outp);
                        current_loc.line--;
                        c = '\n';
                    }
                    else
                    {
                        --outp;
                        break;
                    }
                }
                SAVEC;
                if (c == '\n')
                    break;
            }
            *yyp = '\0';
            add_input(yytext);
            nexpands = 0;
        }
        else
        {
            if (chpos != NULL)
                *chpos = outp;
            outp += clen;
            return c;
        }
    }

    /* NOTREACHED */

} /* exgetc() */

/*-------------------------------------------------------------------------*/
static p_int
cond_get_exp (int priority, svalue_t *svp)

/* Evaluate the expression in the input buffer at a priority of at least
 * <priority> and store the result in <svp> (which is assumed to be
 * invalid at the time of call).
 * Return the result if it is numeric, or a truthvalue for string
 * expressions.
 *
 * The function assumes to be called at the proper beginning of
 * an expression, i.e. if it encounters an operator even before a value,
 * it must be unary.
 */

{
    p_int c;
    p_int value = 0;
    char *opstart;      /* Will point to the first character of the last op. */

    svp->type = T_INVALID;
    do c = exgetc(NULL); while ( unicode_iswhite(c) );

    /* Evaluate the first value */

    if (c == '(')
    {
        /* It's a parenthesized subexpression */

        value = cond_get_exp(0, svp);

        do c = exgetc(NULL); while ( lexwhite(c) );
        if ( c != ')' )
        {
            yyerror("parentheses not paired in #if");
            if (c == '\n')
                myungetc('\n');
        }
    }
    else if ( iswpunct(c) )
    {
        /* It is a string or an unary operator */

        if (c == '"')
        {
            /* Get the string */

            char *p, *q;

            q = p = outp;
            for (;;)
            {
                c = *p++;
                if (c == '"')
                {
                    break;
                }
                if (c == '\n')
                {
                    yyerror("unexpected end of string in #if");
                    put_ref_string(svp, STR_EMPTY);
                    return 0;
                }
                if (c == '\\')
                {
                    c = *p++;
                    if (c == '\n')
                    {
                        current_loc.line++;
                        *--p = '"';
                        break;
                    }
                }
                *q++ = (char)c;
            }
            *q = '\0';
            put_c_string(svp, outp);
            outp = p;
        }
        else
        {
            /* Is it really an operator? */
            int x = optab1(c);
            if (!x)
            {
                yyerror("illegal character in #if");
                return 0;
            }

            /* Get the value for this unary operator */
            value = cond_get_exp(12, svp);

            /* Evaluate the operator */
            switch ( optab2[x-1] )
            {
                case BNOT  : value = ~value; break;
                case LNOT  : value = !value; break;
                case UMINUS: value = -value; break;
                case UPLUS : break; // no action needed
                default :
                    yyerror("illegal unary operator in #if");
                    free_svalue(svp);
                    svp->type = T_NUMBER;
                    return 0;
            }

            if (svp->type != T_NUMBER)
            {
                yyerror("illegal type to unary operator in #if");
                free_svalue(svp);
                svp->type = T_NUMBER;
                return 0;
            }
            svp->u.number = value;
        }
    }
    else if (c == '\n')
    {
        yyerror("missing expression in #if");
        myungetc(c);
        return 0;
    }
    else if (unicode_isdigit(c) )
    {
        /* It's a number. */

        int base, x;

        value = 0;

        /* Determine the base of the number */
        if (c != '0')
            base=10;
        else
        {
            c = mygetc();
            if (c == 'x' || c == 'X')
            {
                base = 16;
                c = mygetc();
            }
            else
                base = 8;
        }

        /* Now parse the number */
        for(;;)
        {
            if ( isdigit(c) )      x = -'0';
            else if ( isupper(c) ) x = -'A'+10;
            else if ( islower(c) ) x = -'a'+10;
            else break;
            x += c;
            if (x > base)
                break;
            value = value * base + x;
            c = mygetc();
        }
        myungetc((char)c);
        put_number(svp, value);
    }
    else
    {
        yyerror("illegal character in #if");
        return 0;
    }

    /* Now evaluate the following <binop> <expr> pairs (if any) */

    for (;;)
    {
        p_int value2;
        svalue_t sv2;
        int x;
        char c2;

        do c=exgetc(&opstart); while ( lexwhite(c) );

        /* An operator or string must come next */
        if ( !iswpunct(c) )
            break;

        /* If it's a string, make it a string addition */
        if (c == '"')
        {
            myungetc('"');
            c = '+';
        }

        /* Can it be an operator at all? */
        x = optab1(c);
        if (!x)
            break;

        /* See if the optab[] defines an operator for these characters
         */
        c2 = mygetc();
        for (;;x+=3)
        {
            if (!optab2[x])
            {
                myungetc(c2);
                if (!optab2[x+1])
                {
                    yyerror("illegal operator use in #if");
                    return 0;
                }
                break;
            }
            if (c2 == optab2[x])
                break;
        }

        /* If the priority of the operator is too low, we are done
         * with this (sub)expression.
         */
        if (priority >= optab2[x+2])
            break;

        /* Get the second operand */
        value2 = cond_get_exp(optab2[x+2], &sv2);

        /* Evaluate the operands:
         *   Full set of operations for numbers.
         *   Addition and lexicographic comparisons for strings.
         */
        if (svp->type == T_NUMBER && sv2.type == T_NUMBER)
        {
            switch (optab2[x+1])
            {
                case MULT   : value *= value2;                break;
                case DIV    : if (!value2) lexerror("Division by zero");
                              else value /= value2;         break;
                case MOD    : if (!value2) lexerror("Division by zero");
                              else value %= value2;         break;
                case BPLUS  : value += value2;                break;
                case BMINUS : value -= value2;                break;
                case LSHIFT : if ((uint)value2 > MAX_SHIFT) value = 0;
                              else value <<= value2; break;
                case RSHIFT : value >>= (uint)value2 > MAX_SHIFT ? (int)MAX_SHIFT : value2;
                              break;
                case LESS   : value = value <  value2;        break;
                case LEQ    : value = value <= value2;        break;
                case GREAT  : value = value >  value2;        break;
                case GEQ    : value = value >= value2;        break;
                case EQ     : value = value == value2;        break;
                case NEQ    : value = value != value2;        break;
                case BAND   : value &= value2;                break;
                case XOR    : value ^= value2;                break;
                case BOR    : value |= value2;                break;
                case LAND   : value = value && value2;        break;
                case LOR    : value = value || value2;        break;
                case QMARK  :
                    do c=exgetc(&opstart); while( lexwhite(c) );

                    if (c != ':')
                    {
                        yyerror("'?' without ':' in #if");
                        outp = opstart;
                        return 0;
                    }
                    if (value)
                    {
                        *svp = sv2;
                        cond_get_exp(1, &sv2);
                        free_svalue(&sv2);
                        value = value2;
                    }
                    else
                        value = cond_get_exp(1, svp);
                    break;
            } /* switch() */
        }
        else if (svp->type == T_STRING && sv2.type == T_STRING)
        {
            x = optab2[x+1];
            if (x == BPLUS)
            {
                svp->u.str = mstr_append(svp->u.str, sv2.u.str);
                free_string_svalue(&sv2);
            }
            else
            {
                value = mstrcmp(svp->u.str, sv2.u.str);
                free_string_svalue(svp);
                svp->type = T_NUMBER;
                free_string_svalue(&sv2);
                switch (x)
                {
                  case LESS   : value = value <  0; break;
                  case LEQ    : value = value <= 0; break;
                  case GREAT  : value = value >  0; break;
                  case GEQ    : value = value >= 0; break;
                  case EQ     : value = value == 0; break;
                  case NEQ    : value = value != 0; break;
                  default:
                    yyerror("illegal operator use in #if");
                    return 0;
                }
                put_number(svp, value);
            }
        }
        else
        {
            yyerror("operands in #if won't match");
            free_svalue(svp);
            svp->type = T_NUMBER;
            free_svalue(&sv2);
            return 0;
        }
    }

    outp = opstart;
    return value;
} /* cond_get_expr() */

/*-------------------------------------------------------------------------*/
void
set_inc_list (vector_t *v)

/* EFUN: set_driver_hook(H_INCLUDE_DIRS, ({ list }) )
 *
 * Set the list of pathnames to search for <>-include files to the
 * names in <v>.
 *
 * The function takes ownership of v->item[], but replaces all string
 * values by its own copies. Since the original v is held in
 * the driver_hook[] array, this is safe to do.
 */

{
    size_t i;
    char *p;
    svalue_t *svp;
    mp_int len, max;

    /* Count and test the passed pathnames */

    svp = v->item;
    for (i = 0, max = 0; i < (size_t)VEC_SIZE(v); i++, svp++)
    {
        string_t *new;
        if (svp->type != T_STRING)
        {
            errorf("H_INCLUDE_DIRS argument has a non-string array element\n");
        }

        /* Set p to the beginning of the pathname, skipping leading
         * '/' and './'.
         */
        p = get_txt(svp->u.str);
        for(;;) {
            if (*p == '/')
                p++;
            else if (*p == '.' && p[1] == '/')
                p += 2;
            else
                break;
        }

        /* Is the path legal? */
        if (!legal_path(p))
        {
            errorf("H_INCLUDE_DIRS path contains '..'\n");
        }
        if (*p == '.' && !p[1])
            errorf("H_INCLUDE_DIRS path is a single prefix dot\n");

        len = (mp_int)strlen(p);
        if (max < len)
            max = len;
        if (len >= 2 && p[len -1] == '.' && p[len - 2] == '/')
            errorf("H_INCLUDE_DIRS path ends in single prefix dot\n");

        /* Get and store our own copy of the pathname */
        new = unshare_mstring(svp->u.str);
        if (!new)
            errorf("Out of memory\n");

        put_string(svp, new); /* dup() already freed it */
    }

    inc_list = v->item;
    inc_list_size = VEC_SIZE(v);
    inc_list_maxlen = max;
} /* set_inc_list() */

/*-------------------------------------------------------------------------*/
static char *
get_current_file (char ** args UNUSED)

/* Dynamic macro __FILE__: return the name of the current file.
 * In compat mode, don't return a leading slash.
 */

{
#ifdef __MWERKS__
#    pragma unused(args)
#endif
    char *buf;

    buf = xalloc(strlen(current_loc.file->name)+4);
    if (!buf)
        return NULL;
    if (compat_mode)
        sprintf(buf, "\"%s\"", current_loc.file->name);
    else
        sprintf(buf, "\"/%s\"", current_loc.file->name);
    return buf;
} /* get_current_file() */

/*-------------------------------------------------------------------------*/
static char *
get_current_function (char ** args UNUSED)

/* Dynamic macro __FUNCTION__: expands to the name of the function
 * currently being defined.
 */
{
    const char *name = get_current_function_name();

    if (!name) {
        lexerror("__FUNCTION__ outside of function definition");
        return string_copy("");
    }

    char *buf = xalloc(strlen(name) + 4);
    if (!buf)
        return NULL;
    sprintf(buf, "\"%s\"", name);
    return buf;
}

/*-------------------------------------------------------------------------*/
static char *
get_current_dir (char ** args UNUSED)

/* Dynamic macro __DIR__: return the directory of the current file.
 * In compat mode, don't return a leading slash.
 */

{
#ifdef __MWERKS__
#    pragma unused(args)
#endif
    char *buf;
    int len;

    buf = current_loc.file->name + strlen(current_loc.file->name);
    while (buf > current_loc.file->name && buf[-1] != '/')
        buf--;
    len = (buf - current_loc.file->name);
    buf = xalloc(len + 4);
    if (!buf)
        return NULL;
    if (compat_mode)
        sprintf(buf, "\"%.*s\"", len, current_loc.file->name);
    else
        sprintf(buf, "\"/%.*s\"", len, current_loc.file->name);
    return buf;
} /* get_current_dir() */

/*-------------------------------------------------------------------------*/
static char *
get_sub_path (char ** args)

/* Dynamic macro __PATH__(n): return the directory of the current file,
 * where n is the number of directories to pop off from the right.
 * In compat mode, don't return a leading slash.
 */

{
    char *buf;
    int len, rm;

    rm = 0;
    sscanf(*args, "%d", &rm);
    if (rm < 0)
        rm = 0;
    buf = current_loc.file->name + strlen(current_loc.file->name);
    while (rm >= 0 && buf >= current_loc.file->name)
        if (*(--buf) == '/')
            rm--;
    len = (buf - current_loc.file->name) + 1;
    buf = alloca(len + 4);
    if (compat_mode)
        sprintf(buf, "\"%.*s\"", len, current_loc.file->name);
    else
        sprintf(buf, "\"/%.*s\"", len, current_loc.file->name);
    add_input(buf);
    return NULL;
} /* get_sub_path() */

#define DYNAMIC_MACRO_BUFFER_SIZE 21
/*-------------------------------------------------------------------------*/
static char *
get_current_line_buf (char ** args UNUSED)

/* Dynamic macro __LINE__: return the number of the current line.
 */

{
#ifdef __MWERKS__
#    pragma unused(args)
#endif
    char *buf;

    buf = xalloc(DYNAMIC_MACRO_BUFFER_SIZE);
    if (!buf)
        return NULL;
    snprintf(buf, DYNAMIC_MACRO_BUFFER_SIZE, "%d", current_loc.line);
    return buf;
} /* get_current_line_buf() */

/*-------------------------------------------------------------------------*/
static char *
get_reset_time_buf (char ** args UNUSED)

/* Dynamic macro __RESET_TIME__: return the current default reset interval.
 */

{
    char *buf = xalloc(DYNAMIC_MACRO_BUFFER_SIZE);
    if (!buf)
        return NULL;
    snprintf(buf, DYNAMIC_MACRO_BUFFER_SIZE, "%ld", time_to_reset);
    return buf;
} /* get_reset_time_buf() */

/*-------------------------------------------------------------------------*/
static char *
get_cleanup_time_buf (char ** args UNUSED)

/* Dynamic macro __CLEANUP_TIME__: return the current cleanup interval.
 */

{
    char *buf = xalloc(DYNAMIC_MACRO_BUFFER_SIZE);
    if (!buf)
        return NULL;
    snprintf(buf, DYNAMIC_MACRO_BUFFER_SIZE, "%ld", time_to_cleanup);
    return buf;
} /* get_cleanup_time_buf() */

/*-------------------------------------------------------------------------*/
static char *
get_memory_limit_buf (char ** args UNUSED)

/* Dynamic macro __MAX_MALLOC__: return the current hard memory limit.
 */

{
    char *buf = xalloc(DYNAMIC_MACRO_BUFFER_SIZE);
    if (!buf)
        return NULL;
    snprintf(buf, DYNAMIC_MACRO_BUFFER_SIZE, "%"PRIdMPINT, get_memory_limit(MALLOC_HARD_LIMIT));
    return buf;
} /* get_memory_limit_buf() */

/*-------------------------------------------------------------------------*/
static char *
get_version(char ** args UNUSED)

/* Dynamic macro __VERSION__: return the driver version.
 */

{
#ifdef __MWERKS__
#    pragma unused(args)
#endif
    char *buf;
    size_t len;

    len = strlen(DRIVER_VERSION LOCAL_LEVEL" ("COMMIT_ID")");
    buf = xalloc(3 + len);
    if (!buf) return 0;
    buf[0] = '"';
    strcpy(buf+1, DRIVER_VERSION LOCAL_LEVEL" ("COMMIT_ID")");
    buf[len+1] = '"';
    buf[len+2] = '\0';
    return buf;
} /* get_version() */

/*-------------------------------------------------------------------------*/
static char *
get_hostname (char ** args UNUSED)

/* Dynamic macro __HOSTNAME__: return the hostname.
 */

{
#ifdef __MWERKS__
#    pragma unused(args)
#endif
    char *tmp, *buf;

    tmp = query_host_name();
    buf = xalloc(strlen(tmp)+3);
    if (!buf) return 0;
    sprintf(buf, "\"%s\"", tmp);
    return buf;
} /* get_hostname() */

/*-------------------------------------------------------------------------*/
static char *
get_domainname (char ** args UNUSED)

/* Dynamic macro __DOMAINNAME__: return the domainname.
 */

{
#ifdef __MWERKS__
#    pragma unused(args)
#endif
    char *buf;

    buf = xalloc(strlen(domain_name)+3);
    if (!buf)
        return 0;
    sprintf(buf, "\"%s\"", domain_name);
    return buf;
} /* get_domainname() */

/*-------------------------------------------------------------------------*/
static char *
efun_defined (char **args)

/* Dynamic macro __EFUN_DEFINE__(name): test if the efun is defined
 * and add ' 0 ' or ' 1 ' depending on the result.
 */

{
    ident_t *p;

    p = make_shared_identifier(args[0], I_TYPE_GLOBAL, 0);
    if (!p)
    {
        lexerror("Out of memory");
        return NULL;
    }

    while (p->type > I_TYPE_GLOBAL)
    {
        if ( !(p = p->inferior) )
            break;
    }

    add_input(
      (p && p->type == I_TYPE_GLOBAL && (p->u.global.efun != I_GLOBAL_EFUN_OTHER
#ifdef USE_PYTHON
                                      || is_python_efun(p)
#endif
      )) ?
        " 1 " : " 0 "
    );

    if (p && p->type == I_TYPE_UNKNOWN)
        free_shared_identifier(p);

    return NULL;
} /* efun_defined() */

/*-------------------------------------------------------------------------*/
void
remove_unknown_identifier (void)

/* Remove all identifiers from the ident_table[] which are of
 * type I_TYPE_UNKNOWN.
 */

{
    int i;
    ident_t *id, *next;

    for (i = ITABLE_SIZE; --i >= 0; )
    {
        id = ident_table[i];
        for ( ; id; id = next)
        {
            next = id->next;
            if (id->type == I_TYPE_UNKNOWN)
                free_shared_identifier(id);
        }
    }
} /* remove_unknown_identifier() */

/*-------------------------------------------------------------------------*/
size_t
show_lexer_status (strbuf_t * sbuf, Bool verbose UNUSED)

/* Return the amount of memory used by the lexer.
 */

{
#if defined(__MWERKS__)
#    pragma unused(verbose)
#endif
    size_t sum;
    ident_t *p;
    int i;

    sum = 0;

    /* Count the space used by identifiers and defines */
    for (i = ITABLE_SIZE; --i >= 0; )
    {
        p = ident_table[i];
        for ( ; p; p = p->next) {
            sum += sizeof(*p);
            if (p->name && p->type == I_TYPE_DEFINE && !p->u.define.special)
                sum += strlen(p->u.define.exps.str)+1;
        }
    }

    sum += mempool_size(lexpool);
    sum += defbuf_len;
    sum += 2 * DEFMAX; /* for the buffers in _expand_define() */

    if (sbuf)
        strbuf_addf(sbuf, "Lexer structures\t\t\t %9zu\n", sum);
    return sum;
} /* show_lexer_status() */

/*-------------------------------------------------------------------------*/
#ifdef GC_SUPPORT

static INLINE void
count_ident_refs (ident_t *id)

/* GC support: count all references held by one identifier (ignoring
 * inferiors).
 */

{
    count_ref_from_string(id->name);
    note_malloced_block_ref(id);
} /* count_ident_refs() */

/*-------------------------------------------------------------------------*/
void
count_lex_refs (void)

/* GC support: count all references held by the lexer.
 */

{
    int i;
    ident_t *id;

    /* Identifier */
    for (i = ITABLE_SIZE; --i >= 0; )
    {
        id = ident_table[i];
        for ( ; id; id = id->next)
        {
            ident_t *id2;
            count_ident_refs(id);
            for (id2 = id->inferior; id2 != NULL; id2 = id2->next)
            {
                count_ident_refs(id2);
            }
        }
    }

    for (id = permanent_defines; id; id = id->next_all)
    {
        if (!id->u.define.special)
            note_malloced_block_ref(id->u.define.exps.str);
    }

    if (defbuf_len)
        note_malloced_block_ref(defbuf);

    if (lexpool)
        mempool_note_refs(lexpool);
}
#endif /* GC_SUPPORT */

/*-------------------------------------------------------------------------*/
char *
lex_error_context (void)

/* Create the current lexing context in a static buffer and return its
 * pointer.
 */

{
    // " before '" + 10 characters (max. 4 bytes each) + "'\0"
#define CONTEXT_LENGTH 10
    static char buf[11 + 4*CONTEXT_LENGTH];
    char *end;
    mp_int len;

    if (lex_error_pos >= 0)
    {
        /* An encoding error, we just print the byte position. */
        snprintf(buf, sizeof(buf), " at byte %d", lex_error_pos);
        return buf;
    }

    strcpy(buf, ((signed char)yychar == -1 || yychar == CHAR_EOF)
                ? (len = 6, " near ")
                : (len = 8, " before "));

    if (!yychar || !*outp)
    {
        strcpy(buf+len, "end of line");
    }
    else if ((signed char)*outp == -1 || *outp == CHAR_EOF)
    {
        strcpy(buf+len, "end of file");
    }
    else
    {
        ssize_t left;

        left = linebufend - outp;
        if (left > (ssize_t)sizeof(buf) - 3 - len)
            left = sizeof(buf) - 3 - len;
        if (left < 1)
            buf[0] = '\0';
        else
        {
            size_t num = char_to_byte_index(outp, left, CONTEXT_LENGTH, NULL);

            buf[len] = '\'';
            strncpy(buf + len + 1, outp, num);
            buf[len + num + 1] = '\'';
            buf[len + num + 2] = '\0';
            if ( NULL != (end = strchr(buf, '\n')) )
            {
                *end = '\'';
                *(end+1) = '\0';
                if (buf[len+1] == '\'')
                    strcpy(buf+len, "end of line");
            }
            if ( NULL != (end = strchr(buf, -1)) )
            {
                *end = '\'';
                *(end+1) = '\0';
                if (buf[len+1] == '\'')
                    strcpy(buf+len, "end of file");
            }
        }
    }
    return buf;
#undef CONTEXT_LENGTH
} /* lex_error_context() */

/*-------------------------------------------------------------------------*/
svalue_t *
f_expand_define (svalue_t *sp)

/* EFUN expand_define()
 *
 *   string expand_define (string name)
 *   string expand_define (string name, string arg, ...)
 *
 * Expands the macro <name> with the argument(s) <arg>... (default is
 * one empty string "").
 * Result is the expanded macro, or 0 if there is no macro with
 * that name.
 *
 * This efun is applicable only while an object is compiled,
 * therefore its usage is restricted to a few functions like the
 * H_INCLUDE_DIRS driver hook, or the masters runtime_error()
 * function.
 * TODO: Right now, only one arg is evaluated.
 */

{
    char *arg, *end;
    string_t *res;
    ident_t *d;

    /* Get the arguments from the stack */

    if (sp->type == T_STRING)
    {
        arg = get_txt(sp->u.str);
        /* TODO: Concatenate all strings on the stack */
    }
    else /* it's the number 0 */
        arg = "";

    res = NULL;

    /* If we are compiling, lookup the given name and store
     * the expansion in res.
     */
    if (current_loc.file  && current_loc.file->name
     && outp > defbuf && outp <= &defbuf[defbuf_len])
    {
        myungetc('\n');
        end = outp;
        add_input(arg);
        d = lookup_define(get_txt(sp[-1].u.str));
        if (d && _expand_define(&d->u.define, d) )
        {
            *end = '\0';
            res = new_unicode_mstring(outp);
            *end = '\n';  /* Restore the newline character */
        }
        outp = &end[1];
    }
    free_svalue(sp);
    free_svalue(--sp);

    /* Return the result */
    if (!res)
    {
        put_number(sp, 0);
    }
    else
    {
        put_string(sp, res);
    }

    return sp;
} /* f_expand_define() */

/***************************************************************************/

