/*
 * Copyright (C) 2012 David Keitel, Santiago Caride, Reinhard Prix
 * Copyright (C) 2008, 2013 Karl Wette
 * Copyright (C) 2007 Chris Messenger
 * Copyright (C) 2004, 2007, 2010 Reinhard Prix
 * Copyright (C) 2005, 2006 Reinhard Prix, Iraj Gholami
 * Copyright (C) 2002, 2003, 2004 M.A. Papa, X. Siemens, Y. Itoh
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with with program; see the file COPYING. If not, write to the
 *  Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 */

/*********************************************************************************/
/**
 * \author R. Prix, I. Gholami, Y. Ioth, Papa, X. Siemens, C. Messenger, K. Wette
 * \file
 * \ingroup lalapps_pulsar_Fstatistic
 * \brief
 * Calculate the F-statistic for a given parameter-space of pulsar GW signals.
 * Implements the so-called "F-statistic" as introduced in \cite JKS98 .
 *
 * This code is a descendant of an earlier implementation 'ComputeFStatistic.[ch]'
 * by Bruce Allen, Bernd Machenschalk, David Hammer, Jolien Creighton, Maria Alessandra Papa,
 * Reinhard Prix, Xavier Siemens, Scott Koranda, Yousuke Itoh
 *
 */
#include "config.h"

/* System includes */
#include <math.h>
#include <stdio.h>
#include <strings.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <gsl/gsl_math.h>

/* LAL-includes */
#include <lal/LALString.h>
#include <lal/AVFactories.h>
#include <lal/LALInitBarycenter.h>
#include <lal/UserInput.h>
#include <lal/TranslateAngles.h>
#include <lal/TranslateMJD.h>
#include <lal/SFTfileIO.h>
#include <lal/ExtrapolatePulsarSpins.h>
#include <lal/EstimateAmplitudeParams.h>

#include <lal/NormalizeSFTRngMed.h>
#include <lal/ComputeFstat.h>
#include <lal/LALHough.h>

#include <lal/LogPrintf.h>
#include <lal/DopplerFullScan.h>
#include <lal/BinaryPulsarTiming.h>

#include <lal/TransientCW_utils.h>
#include <lal/LineRobustStats.h>

#include <lalapps.h>

/* local includes */
#include "HeapToplist.h"

/*---------- DEFINES ----------*/

#define TRUE (1==1)
#define FALSE (1==0)

/*----- SWITCHES -----*/
/*----- Macros -----*/

/** convert GPS-time to REAL8 */
#define MYMAX(x,y) ( (x) > (y) ? (x) : (y) )
#define MYMIN(x,y) ( (x) < (y) ? (x) : (y) )

// NOTE: LAL's nan is more portable than either of libc or gsl !
#define LAL_NAN XLALREAL4FailNaN()
/*---------- internal types ----------*/

/** What info do we want to store in our toplist? */
typedef struct {
  PulsarDopplerParams doppler;		/**< Doppler params of this 'candidate' */
  REAL4 twoF;				/**< F-statistic value */
  UINT4 numDetectors;			/**< number of detectors = effective vector length. numDetectors=0 should make all code ignore the FX field. */
  REAL4 twoFX[PULSAR_MAX_DETECTORS];	/**< vector of single-detector F-statistic values (array of fixed size) */
  LIGOTimeGPS FaFb_refTime;		/**< reference time of Fa and Fb */
  COMPLEX16 Fa;				/**< complex amplitude Fa */
  COMPLEX16 Fb;				/**< complex amplitude Fb */
  AntennaPatternMatrix Mmunu;		/**< antenna-pattern matrix Mmunu = (h_mu|h_nu) */
  REAL4 log10BSGL;			/**< Line-robust statistic \f$\log_{10}B_{\mathrm{SGL}}\f$ */
} FstatCandidate;

/**
 * moving 'Scanline window' of candidates on the scan-line,
 * which is used to find local 1D maxima.
 */
typedef struct
{
  UINT4 length;
  FstatCandidate *window; 		/**< array holding candidates */
  FstatCandidate *center;		/**< pointer to middle candidate in window */
} scanlineWindow_t;

/**
 * Struct holding various timing measurements and relevant search parameters.
 * This is used to fit timing-models with measured times to predict search run-times
 */
typedef struct
{
  UINT4 NSFTs;			/**< total number of SFTs */
  UINT4 NFreq;			/**< number of frequency bins */
  REAL8 tauFstat;		/**< time to compute one Fstatistic over full data-duration (NSFT atoms) [in seconds]*/
  REAL8 tauTemplate;		/**< total loop time per template, includes candidate-handling (transient stats, toplist etc) */
  REAL8 tauF0;			/**< Demod timing constant = time per template per SFT */

  /* ----- transient-specific timings */
  UINT4 tauMin;			/**< shortest transient timescale [s] */
  UINT4 tauMax;			/**< longest transient timescale [s] */
  UINT4 NStart;			/**< number of transient start-time steps in FstatMap matrix */
  UINT4 NTau;			/**< number of transient timescale steps in FstatMap matrix */
  REAL8 tauTransFstatMap;	/**< time to compute transient-search Fstatistic-map over {t0, tau} [s]     */
  REAL8 tauTransMarg;		/**< time to marginalize the Fstat-map to compute transient-search Bayes [s] */

} timingInfo_t;

/**
 * Enum for which statistic is used to "rank" significance of candidates
 */
typedef enum {
  RANKBY_2F  = 0, 	/**< rank candidates by F-statistic */
  RANKBY_NC = 1,	/**< HierarchSearchGCT also has RANKBY_NC = 1, not applicable here */
  RANKBY_BSGL = 2  	/**< rank candidates by BSGListic */
} RankingStat_t;


/**
 * Configuration settings required for and defining a coherent pulsar search.
 * These are 'pre-processed' settings, which have been derived from the user-input.
 */
typedef struct {
  LIGOTimeGPS startTime;		    /**< starting timestamp of SFTs */
  REAL8 Tspan;				    /**< time spanned by all SFTs */
  REAL8 Alpha;                              /**< sky position alpha in radians */
  REAL8 Delta;                              /**< sky position delta in radians */
  REAL8 Tsft;                               /**< length of one SFT in seconds */
  DopplerRegion searchRegion;		    /**< parameter-space region to search over */
  DopplerFullScanState *scanState;          /**< current state of the Doppler-scan */
  PulsarDopplerParams stepSizes;	    /**< user-preferences on Doppler-param step-sizes */
  EphemerisData *ephemeris;		    /**< ephemeris data (from LALInitBarycenter()) */
  UINT4 NSFTs;				    /**< total number of all SFTs used */
  UINT4Vector *numSFTsPerDet;		    /**< number of SFTs per detector, for log strings, etc. */
  LALStringVector *detectorIDs;		    /**< detector ID names, for column headings string */
  FstatInput *Fstat_in;		    /**< Fstat input data struct */
  FstatQuantities Fstat_what;		    /**< Fstat quantities to compute */
  toplist_t* FstatToplist;		    /**< sorted 'toplist' of the NumCandidatesToKeep loudest candidates */
  scanlineWindow_t *scanlineWindow;         /**< moving window of candidates on scanline to find local maxima */
  CHAR *VCSInfoString;                      /**< LAL + LALapps Git version string */
  CHAR *logstring;                          /**< log containing max-info on the whole search setup */
  transientWindowRange_t transientWindowRange; /**< search range parameters for transient window */
  BSGLSetup *BSGLsetup;                    /**< pre-computed setup for line-robust statistic */
  RankingStat_t RankingStatistic;           /**< rank candidates according to F or BSGL */
  BOOLEAN useResamp;
  FstatMethodType FstatMethod;
  UINT4 numFreqBins_FBand;
  REAL8 dFreq;
} ConfigVariables;


/* ----- User-variables: can be set from config-file or command-line */
typedef struct {
  INT4 Dterms;			/**< number of terms in LALDemod Dirichlet kernel is Dterms+1 */
  CHAR *IFO;			/**< IFO name: only required if using v1 SFTs */
  BOOLEAN SignalOnly;		/**< FALSE: estimate noise-floor from data, TRUE: assume Sh=1 */
  BOOLEAN UseNoiseWeights;	/**< use SFT-specific noise-weights for each segment in Fstat-computation */

  REAL8 Freq;			/**< start-frequency of search */
  REAL8 FreqBand;		/**< Frequency-band to search over */
  REAL8 dFreq;			/**< user-specifyable Frequency stepsize */

  REAL8 Alpha;			/**< equatorial right-ascension in rad */
  CHAR *RA;
  REAL8 dAlpha;
  REAL8 AlphaBand;

  REAL8 Delta;			/**< equatorial declination in rad */
  CHAR *Dec;
  REAL8 dDelta;
  REAL8 DeltaBand;

  REAL8 f1dot;			/**< 1st spindown dFreq/dt */
  REAL8 df1dot;
  REAL8 f1dotBand;

  REAL8 f2dot;			/**< 2nd spindown d^2Freq/dt^2 */
  REAL8 df2dot;
  REAL8 f2dotBand;

  REAL8 f3dot;			/**< 3rd spindown d^3Freq/dt^3 */
  REAL8 df3dot;
  REAL8 f3dotBand;

  /* orbital parameters */
  REAL8 orbitPeriod;		/**< binary-system orbital period in s */
  REAL8 orbitasini;		/**< amplitude of radial motion */
  LIGOTimeGPS orbitTp;		/**< epoch of periapse passage */
  REAL8 orbitArgp;		/**< angle of periapse */
  REAL8 orbitEcc;		/**< orbital eccentricity */

  /* extra parameters for --gridType=GRID_SPINDOWN_AGEBRK parameter space */
  REAL8 spindownAge;            /**< spindown age of the object */
  REAL8 minBraking;             /**< minimum braking index */
  REAL8 maxBraking;             /**< maximum braking index */

  /* --- */
  REAL8 TwoFthreshold;		/**< output threshold on 2F */
  CHAR *ephemEarth;		/**< Earth ephemeris file to use */
  CHAR *ephemSun;		/**< Sun ephemeris file to use */

  INT4  gridType;		/**< type of template grid in parameter space */
  INT4  metricType;		/**< type of metric to use in metric template grids */
  BOOLEAN projectMetric;	/**< should we use frequency-projected metric? */
  REAL8 metricMismatch;		/**< maximal *nominal* metric mismatch *per* dimension */
  CHAR *skyRegion;		/**< list of skypositions defining a search-polygon */
  CHAR *DataFiles;		/**< glob-pattern for SFT data-files to use */

  BOOLEAN help;			/**< output help-string */
  CHAR *outputLogfile;		/**< write a log-file */
  CHAR *outputFstat;		/**< filename to output Fstatistic in */
  CHAR *outputLoudest;		/**< filename for loudest F-candidate plus parameter estimation */
  CHAR *outputLogPrintf;        /**< send output from LogPrintf statements to this file */

  CHAR *outputFstatHist;        /**< output discrete histogram of all Fstatistic values */
  REAL8 FstatHistBin;           /**< width of an Fstatistic histogram bin */

  BOOLEAN countTemplates;       /**< just count templates (if supported) instead of search */

  INT4 NumCandidatesToKeep;	/**< maximal number of toplist candidates to output */
  REAL8 FracCandidatesToKeep;	/**< fractional number of candidates to output in toplist */
  INT4 clusterOnScanline;	/**< number of points on "scanline" to use for 1-D local maxima finding */

  CHAR *gridFile;		/**< read template grid from this file */
  REAL8 dopplermax;		/**< maximal possible doppler-effect */

  INT4 RngMedWindow;		/**< running-median window for noise floor estimation */
  LIGOTimeGPS refTime;		/**< reference-time for definition of pulsar-parameters [GPS] */

  INT4 SSBprecision;		/**< full relativistic timing or Newtonian */

  LIGOTimeGPS minStartTime;	/**< Only use SFTs with timestamps starting from (including) this epoch (format 'xx.yy[GPS]' or 'xx.yyMJD') */
  LIGOTimeGPS maxStartTime;	/**< Only use SFTs with timestamps up to (excluding) this epoch (format 'xx.yy[GPS]' or 'xx.yyMJD') */
  CHAR *workingDir;		/**< directory to use for output files */
  REAL8 timerCount;		/**< output progress-meter every timerCount seconds */

  BOOLEAN version;		/**< output version information */

  CHAR *outputFstatAtoms;	/**< output per-SFT, per-IFO 'atoms', ie quantities required to compute F-stat */

  BOOLEAN outputSingleFstats;	/**< in multi-detector case, also output single-detector F-stats */
  CHAR *RankingStatistic;	/**< rank candidates according to F-stat or BSGL */

  BOOLEAN computeBSGL;		/**< get single-IFO F-stats and compute line-robust statistic */
  BOOLEAN BSGLlogcorr;		/**< BSGL: compute log-correction (slower) or not (faster) */
  REAL8   Fstar0;		/**< BSGL: transition-scale parameter 'Fstar0', see documentation for XLALCreateBSGLSetup() for details */
  LALStringVector *oLGX;	/**< BSGL: prior per-detector line-vs-Gauss odds 'oLGX', see XLALCreateBSGLSetup() for details */
  REAL8 BSGLthreshold;		/**< output threshold on BSGL */

  CHAR *outputTransientStats;	/**< output file for transient B-stat values */
  CHAR *transient_WindowType;	/**< name of transient window ('none', 'rect', 'exp',...) */
  REAL8 transient_t0Days;	/**< earliest GPS start-time for transient window search, as offset in days from dataStartGPS */
  REAL8 transient_t0DaysBand;	/**< Range of GPS start-times to search in transient search, in days */
  INT4  transient_dt0;		/**< Step-size for search/marginalization over transient-window start-time, in seconds */
  REAL8 transient_tauDays;	/**< smallest transient window length for marginalization, in days */
  REAL8 transient_tauDaysBand;	/**< Range of transient-window timescales to search, in days */
  INT4  transient_dtau;		/**< Step-size for search/marginalization over transient-window timescale, in seconds */
  BOOLEAN transient_useFReg;  	/**< FALSE: use 'standard' e^F for marginalization, TRUE: use e^FReg = (1/D)*e^F */

  CHAR *outputTiming;		/**< output timing measurements and parameters into this file [append!]*/

  CHAR *FstatMethod;		//!< select which method/algorithm to use to compute the F-statistic

  // ----- deprecated and obsolete variables, kept around for backwards-compatibility -----
  LIGOTimeGPS internalRefTime;   /**< [DEPRECATED] internal reference time. Has no effect, XLALComputeFstat() now always uses midtime anyway ... */

} UserInput_t;

/*---------- Global variables ----------*/
extern int vrbflg;		/**< defined in lalapps.c */

/* ---------- local prototypes ---------- */
int main(int argc,char *argv[]);
int initUserVars ( UserInput_t *uvar);
int InitFstat ( ConfigVariables *cfg, const UserInput_t *uvar );
void Freemem(ConfigVariables *cfg);

int checkUserInputConsistency (const UserInput_t *uvar);
int outputBeamTS( const CHAR *fname, const AMCoeffs *amcoe, const DetectorStateSeries *detStates );
MultiNoiseWeights *getUnitWeights ( const MultiSFTVector *multiSFTs );

int write_FstatCandidate_to_fp ( FILE *fp, const FstatCandidate *thisFCand );
int write_PulsarCandidate_to_fp ( FILE *fp,  const PulsarCandidate *pulsarParams, const FstatCandidate *Fcand );

int compareFstatCandidates ( const void *candA, const void *candB );
int compareFstatCandidates_BSGL ( const void *candA, const void *candB );

int WriteFstatLog ( const CHAR *log_fname, const CHAR *logstr );
CHAR *XLALGetLogString ( const ConfigVariables *cfg );

int write_TimingInfo ( const CHAR *timingFile, const timingInfo_t *ti, const ConfigVariables *cfg );

gsl_vector_int *resize_histogram(gsl_vector_int *old_hist, size_t size);

/* ---------- scanline window functions ---------- */
scanlineWindow_t *XLALCreateScanlineWindow ( UINT4 windowWings );
void XLALDestroyScanlineWindow ( scanlineWindow_t *scanlineWindow );
int XLALAdvanceScanlineWindow ( const FstatCandidate *nextCand, scanlineWindow_t *scanWindow );
BOOLEAN XLALCenterIsLocalMax ( const scanlineWindow_t *scanWindow, const UINT4 sortingStatistic );

/* ----- which timing function to use ----- */
#define GETTIME XLALGetCPUTime

/*----------------------------------------------------------------------*/
/* Function definitions start here */
/*----------------------------------------------------------------------*/

/**
 * MAIN function of ComputeFstatistic code.
 * Calculate the F-statistic over a given portion of the parameter-space
 * and write a list of 'candidates' into a file(default: 'Fstats').
 */
int main(int argc,char *argv[])
{
  FILE *fpFstat = NULL, *fpTransientStats = NULL;
  REAL8 numTemplates, templateCounter;
  time_t clock0;
  PulsarDopplerParams XLAL_INIT_DECL(dopplerpos);
  FstatCandidate XLAL_INIT_DECL(loudestFCand);
  FstatCandidate XLAL_INIT_DECL(thisFCand);
  FILE *fpLogPrintf = NULL;
  gsl_vector_int *Fstat_histogram = NULL;

  UserInput_t XLAL_INIT_DECL(uvar);
  ConfigVariables XLAL_INIT_DECL(GV);

  vrbflg = 1;	/* verbose error-messages */

  /* set LAL error-handler */
  lal_errhandler = LAL_ERR_EXIT;

  /* register all user-variable */
  XLAL_CHECK_MAIN ( initUserVars ( &uvar ) == XLAL_SUCCESS, XLAL_EFUNC );

  XLAL_CHECK_MAIN ( (GV.VCSInfoString = XLALGetVersionString(0)) != NULL, XLAL_EFUNC );

  /* do ALL cmdline and cfgfile handling */
  XLAL_CHECK_MAIN ( XLALUserVarReadAllInput(argc, argv) == XLAL_SUCCESS, XLAL_EFUNC );

  if (uvar.help) {	/* if help was requested, we're done here */
    exit (0);
  }

  if ( uvar.version )
    {
      printf ( "%s\n", GV.VCSInfoString );
      exit (0);
    }

  /* set log-level and open log-file */
  LogSetLevel ( lalDebugLevel );
  if ( XLALUserVarWasSet(&uvar.outputLogPrintf))
    {
      XLAL_CHECK_MAIN ((fpLogPrintf = fopen(uvar.outputLogPrintf, "wb")) != NULL, XLAL_ESYS, "\nError opening file '%s' for writing..\n\n", uvar.outputLogPrintf );
      LogSetFile(fpLogPrintf);
    }

  /* do some sanity checks on the user-input before we proceed */
  XLAL_CHECK_MAIN ( checkUserInputConsistency ( &uvar ) == XLAL_SUCCESS, XLAL_EFUNC );

  /* Initialization the common variables of the code, */
  /* like ephemeries data and template grids: */
  XLAL_CHECK_MAIN ( InitFstat( &GV, &uvar) == XLAL_SUCCESS, XLAL_EFUNC );

  /* ----- produce a log-string describing the specific run setup ----- */
  XLAL_CHECK_MAIN ( (GV.logstring = XLALGetLogString ( &GV )) != NULL, XLAL_EFUNC );
  LogPrintfVerbatim( LOG_DEBUG, "%s", GV.logstring );

  /* keep a log-file recording all relevant parameters of this search-run */
  if ( uvar.outputLogfile ) {
    XLAL_CHECK_MAIN ( WriteFstatLog ( uvar.outputLogfile, GV.logstring ) == XLAL_SUCCESS, XLAL_EFUNC );
  }

  /* if a complete output of the F-statistic file was requested,
   * we open and prepare the output-file here */
  if (uvar.outputFstat)
    {
      XLAL_CHECK_MAIN ( (fpFstat = fopen (uvar.outputFstat, "wb")) != NULL, XLAL_ESYS, "\nError opening file '%s' for writing..\n\n", uvar.outputFstat );
      fprintf (fpFstat, "%s", GV.logstring );

      /* assemble column headings string */
      char colum_headings_string_base[] = "freq alpha delta f1dot f2dot f3dot 2F";
      UINT4 column_headings_string_length = sizeof(colum_headings_string_base);
      char column_headings_string[column_headings_string_length];
      XLAL_INIT_MEM( column_headings_string );
      strcat ( column_headings_string, colum_headings_string_base );
      if ( uvar.computeBSGL )
        {
          const UINT4 numDetectors = GV.detectorIDs->length;
          column_headings_string_length += numDetectors*6; /* 6 per detector for " 2F_XY" */
          column_headings_string_length += 10; /* 3 for " log10BSGL"*/
          strcat ( column_headings_string, " log10BSGL" );
          for ( UINT4 X = 0; X < numDetectors ; X ++ )
            {
              char headingX[7];
              snprintf ( headingX, sizeof(headingX), " 2F_%s", GV.detectorIDs->data[X] );
              strcat ( column_headings_string, headingX );
            } /* for X < numDet */
        }
      fprintf (fpFstat, "%%%% columns:\n%%%% %s\n", column_headings_string );

    } /* if outputFstat */

  if ( uvar.outputTransientStats )
    {
      XLAL_CHECK_MAIN ( (fpTransientStats = fopen (uvar.outputTransientStats, "wb")) != NULL, XLAL_ESYS, "\nError opening file '%s' for writing..\n\n", uvar.outputTransientStats );
      fprintf (fpTransientStats, "%s", GV.logstring );			/* write search log comment */
      XLAL_CHECK_MAIN ( write_transientCandidate_to_fp ( fpTransientStats, NULL ) == XLAL_SUCCESS, XLAL_EFUNC );	/* write header-line comment */
    }

  /* start Fstatistic histogram with a single empty bin */
  if (uvar.outputFstatHist) {
    XLAL_CHECK_MAIN ((Fstat_histogram = gsl_vector_int_alloc(1)) != NULL, XLAL_ENOMEM );
    gsl_vector_int_set_zero(Fstat_histogram);
  }

  /* setup binary parameters */
  REAL8 orbit_asini = 0 /* isolated pulsar */;
  REAL8 orbit_period = 0;
  REAL8 orbit_ecc = 0;
  LIGOTimeGPS orbit_tp = LIGOTIMEGPSZERO;
  REAL8 orbit_argp = 0;
  if ( XLALUserVarWasSet ( &uvar.orbitasini ) && ( uvar.orbitasini > 0 ) )
    {
      orbit_tp = uvar.orbitTp;
      orbit_argp = uvar.orbitArgp;
      orbit_asini = uvar.orbitasini;
      orbit_ecc = uvar.orbitEcc;
      orbit_period = uvar.orbitPeriod;
    }

  /* count number of templates */
  numTemplates = XLALNumDopplerTemplates ( GV.scanState );
  if (uvar.countTemplates) {
    printf("%%%% Number of templates: %0.0f\n", numTemplates);
  }

  /*----------------------------------------------------------------------
   * main loop: demodulate data for each point in the sky-position grid
   * and for each value of the frequency-spindown
   */
  templateCounter = 0.0;
  clock0 = GETTIME();

  // ----- prepare timing info
  REAL8 tic0, tic, toc, timeOfLastProgressUpdate = 0;	// high-precision timing counters
  timingInfo_t XLAL_INIT_DECL(timing);			// timings of Fstatistic computation, transient Fstat-map, transient Bayes factor

  // pointer to Fstat results structure, will be allocated by XLALComputeFstat()
  FstatResults* Fstat_res = NULL;

  /* skip search if user supplied --countTemplates */
  while ( !uvar.countTemplates && (XLALNextDopplerPos( &dopplerpos, GV.scanState ) == 0) )
    {
      /* temporary solution until binary-gridding exists */
      dopplerpos.asini  = orbit_asini;
      dopplerpos.period = orbit_period;
      dopplerpos.ecc    = orbit_ecc;
      dopplerpos.tp     = orbit_tp;
      dopplerpos.argp   = orbit_argp;

      tic0 = tic = GETTIME();

      /* main function call: compute F-statistic for this template */
      XLAL_CHECK_MAIN ( XLALComputeFstat ( &Fstat_res, GV.Fstat_in, &dopplerpos, GV.numFreqBins_FBand, GV.Fstat_what) == XLAL_SUCCESS, XLAL_EFUNC );

      /* if single-only flag is given, add +4 to F-statistic */
      if ( uvar.SignalOnly ) {
        XLAL_CHECK_MAIN ( XLALAdd4ToFstatResults(Fstat_res) == XLAL_SUCCESS, XLAL_EFUNC );
      }

      toc = GETTIME();
      timing.tauFstat += (toc - tic);   // pure Fstat-calculation time

      /* Progress meter */
      templateCounter += 1.0;
      if ( lalDebugLevel && ( (toc - timeOfLastProgressUpdate) > uvar.timerCount) )
        {
          REAL8 diffSec = GETTIME() - clock0 ;  /* seconds since start of loop*/
          REAL8 taup = diffSec / templateCounter ;
          REAL8 timeLeft = (numTemplates - templateCounter) *  taup;
          LogPrintf (LOG_DEBUG, "Progress: %g/%g = %.2f %% done, Estimated time left: %.0f s\n",
                     templateCounter, numTemplates, templateCounter/numTemplates * 100.0, timeLeft);
          timeOfLastProgressUpdate = toc;
        }

      // here we use Santiago's trick to hack the resampling Fstat(f) into the single-F rest of the
      // main-loop: we simply loop the remaining body over all frequency-bins in the Fstat-vector,
      // this way nothing needs to be changed!  in the non-resampling case, this loop iterates only
      // once, so nothing is changed ...
      for ( UINT4 iFreq = 0; iFreq < GV.numFreqBins_FBand; iFreq ++ )
      {

        /* collect data on current 'Fstat-candidate' */
        thisFCand.doppler = dopplerpos;	// use 'original' dopplerpos @ refTime !
        thisFCand.doppler.fkdot[0] += iFreq * GV.dFreq; // this only does something for the resampling post-loop over frequency-bins, 0 otherwise ...
        thisFCand.twoF = Fstat_res->twoF[iFreq];
        if (GV.Fstat_what & FSTATQ_2F_PER_DET) {
          thisFCand.numDetectors = Fstat_res->numDetectors;
          for (UINT4 X = 0; X < thisFCand.numDetectors; ++X ) {
            thisFCand.twoFX[X] = Fstat_res->twoFPerDet[X][iFreq];
          }
        } else {
          thisFCand.numDetectors = 0;
        }
        if (GV.Fstat_what & FSTATQ_FAFB) {
          thisFCand.FaFb_refTime = Fstat_res->refTimePhase; // 'internal' reference time, used only for global phase estimate
          thisFCand.Fa = Fstat_res->Fa[iFreq];
          thisFCand.Fb = Fstat_res->Fb[iFreq];
        } else {
          thisFCand.Fa = thisFCand.Fb = crect(LAL_NAN,LAL_NAN);
        }
        thisFCand.Mmunu = Fstat_res->Mmunu;
        MultiFstatAtomVector* thisFAtoms = NULL;
        if (GV.Fstat_what & FSTATQ_ATOMS_PER_DET) {
          thisFAtoms = Fstat_res->multiFatoms[iFreq];
        }

        /* sanity check on the result */
        if ( !isfinite(thisFCand.twoF) )
	{
	  LogPrintf(LOG_CRITICAL, "non-finite 2F = %.16g, Fa=(%.16g,%.16g), Fb=(%.16g,%.16g)\n",
		    thisFCand.twoF, creal(thisFCand.Fa), cimag(thisFCand.Fa), creal(thisFCand.Fb), cimag(thisFCand.Fb) );
	  LogPrintf (LOG_CRITICAL, "[Alpha,Delta] = [%.16g,%.16g],\nfkdot=[%.16g,%.16g,%.16g,%16.g]\n",
		     thisFCand.doppler.Alpha, thisFCand.doppler.Delta,
		     thisFCand.doppler.fkdot[0], thisFCand.doppler.fkdot[1], thisFCand.doppler.fkdot[2], thisFCand.doppler.fkdot[3] );
	  if (thisFCand.doppler.asini > 0)
          {
            LogPrintf (LOG_CRITICAL, "tp = {%d s, %d ns}, argp = %f, asini = %f, ecc = %f, period = %f\n",
                       thisFCand.doppler.tp.gpsSeconds, thisFCand.doppler.tp.gpsNanoSeconds,
                       thisFCand.doppler.argp, thisFCand.doppler.asini,
                       thisFCand.doppler.ecc, thisFCand.doppler.period);
          }
	  return -1;
	}

      if ( uvar.computeBSGL )
        {
          thisFCand.log10BSGL = XLALComputeBSGL ( thisFCand.twoF, thisFCand.twoFX, GV.BSGLsetup );
          XLAL_CHECK_MAIN ( xlalErrno == 0, XLAL_EFUNC, "XLALComputeBSGL() failed with errno=%d\n", xlalErrno );
        }
      else
        {
          thisFCand.log10BSGL = LAL_NAN; /* in non-BSGL case, block field with NAN, needed for output checking in write_PulsarCandidate_to_fp() */
        }

      /* push new value onto scan-line buffer */
      XLALAdvanceScanlineWindow ( &thisFCand, GV.scanlineWindow );

      /* two types of threshold: fixed (TwoF- and/or BSGL-threshold) and dynamic (NumCandidatesToKeep) */
      BOOLEAN is1DlocalMax = FALSE;
      if ( XLALCenterIsLocalMax ( GV.scanlineWindow, GV.RankingStatistic ) ) /* must be 1D local maximum */
        is1DlocalMax = TRUE;
      BOOLEAN isOver2FThreshold = FALSE; /* will always be checked, so start at 'FALSE' */
      if ( GV.scanlineWindow->center->twoF >= uvar.TwoFthreshold ) /* fixed 2F threshold */
        isOver2FThreshold = TRUE;
      BOOLEAN isOverBSGLthreshold = TRUE;  /* will not be checked in non-BSGL case, so start at 'TRUE' */
      if ( uvar.computeBSGL && ( GV.scanlineWindow->center->log10BSGL < uvar.BSGLthreshold ) ) /* fixed threshold on log10BSGL */
        isOverBSGLthreshold = FALSE;
      if ( is1DlocalMax && isOver2FThreshold && isOverBSGLthreshold )
        {
	  FstatCandidate *writeCand = GV.scanlineWindow->center;

	  /* insert this into toplist if requested */
	  if ( GV.FstatToplist  )			/* dynamic threshold */
	    {
	      if ( insert_into_toplist(GV.FstatToplist, (void*)writeCand ) ) {
		LogPrintf ( LOG_DETAIL, "Added new candidate into toplist: 2F = %f", writeCand->twoF );
		if ( uvar.computeBSGL ) {
		  LogPrintfVerbatim ( LOG_DETAIL, ", 2F_H1 = %f, 2F_L1 = %f, log10BSGL = %f", writeCand->twoFX[0], writeCand->twoFX[1], writeCand->log10BSGL );
                }
	      }
	      else {
		LogPrintf ( LOG_DETAIL, "NOT added the candidate into toplist: 2F = %f", writeCand->twoF );
		if ( uvar.computeBSGL ) {
		  LogPrintfVerbatim ( LOG_DETAIL, ", 2F_H1 = %f, 2F_L1 = %f, log10BSGL = %f", writeCand->twoFX[0], writeCand->twoFX[1], writeCand->log10BSGL );
                }
	      }
	      LogPrintfVerbatim ( LOG_DETAIL, "\n" );
	    }
	  else if ( fpFstat ) 				/* no toplist :write out immediately */
	    {
	      if ( write_FstatCandidate_to_fp ( fpFstat, writeCand ) != 0 )
		{
		  LogPrintf (LOG_CRITICAL, "Failed to write candidate to file.\n");
		  return -1;
		}
	    } /* if outputFstat */

	} /* if candidate is local maximum and over threshold */

      /* separately keep track of loudest candidate (for --outputLoudest) */
      switch ( GV.RankingStatistic )
        {
        case RANKBY_2F:
          if ( thisFCand.twoF > loudestFCand.twoF ) {
            loudestFCand = thisFCand;
          }
          break;
        case RANKBY_BSGL:
          if ( thisFCand.log10BSGL > loudestFCand.log10BSGL ) {
            loudestFCand = thisFCand;
          }
          break;
        default:
          XLAL_ERROR_MAIN ( XLAL_EINVAL, "Invalid ranking statistic '%d', supported are 'F=0', and 'BSGL=2'\n", GV.RankingStatistic );
          break;
        }

      /* add Fstatistic to histogram if needed */
      if (uvar.outputFstatHist) {

	/* compute bin */
	const size_t bin = thisFCand.twoF / uvar.FstatHistBin;

	/* resize histogram vector if needed */
	if (!Fstat_histogram || bin >= Fstat_histogram->size) {
	  XLAL_CHECK_MAIN ( (Fstat_histogram = resize_histogram(Fstat_histogram, bin + 1)) != NULL, XLAL_EFUNC, "\nCouldn't (re)allocate 'Fstat_histogram'\n" );
        }

	/* add to bin */
	gsl_vector_int_set(Fstat_histogram, bin,
			   gsl_vector_int_get(Fstat_histogram, bin) + 1);

      }


      /* ----- output F-stat atoms into files: one file per Doppler-position ---------- */
      /* F-stat 'atoms' = per-SFT components {a,b,Fa,Fb}_alpha) */
      if (uvar.outputFstatAtoms)
	{
	  FILE *fpFstatAtoms = NULL;
	  CHAR *fnameAtoms = NULL;
	  CHAR *dopplerName;
	  UINT4 len;

	  XLAL_CHECK_MAIN ( (dopplerName = XLALPulsarDopplerParams2String ( &dopplerpos )) != NULL, XLAL_EFUNC );
	  len = strlen(uvar.outputFstatAtoms) + strlen(dopplerName) + 10;
	  XLAL_CHECK_MAIN ( (fnameAtoms = LALMalloc (len)) != NULL, XLAL_ENOMEM, "Failed to LALMalloc(%d)\n", len );
	  sprintf (fnameAtoms, "%s_%s.dat", uvar.outputFstatAtoms, dopplerName );

	  XLAL_CHECK_MAIN ( (fpFstatAtoms = fopen (fnameAtoms, "wb")) != NULL, XLAL_ESYS, "Error opening file '%s' for writing..\n\n", fnameAtoms );
	  XLALFree ( fnameAtoms );
	  XLALFree ( dopplerName );

	  fprintf (fpFstatAtoms, "%s", GV.logstring );

	  XLAL_CHECK_MAIN ( write_MultiFstatAtoms_to_fp ( fpFstatAtoms, thisFAtoms ) == XLAL_SUCCESS, XLAL_EFUNC );
	  fclose (fpFstatAtoms);

	} /* if outputFstatAtoms */

      /* ----- compute transient-CW statistics if their output was requested  ----- */
      if ( fpTransientStats )
        {
          transientCandidate_t XLAL_INIT_DECL(transientCand);

          /* compute Fstat map F_mn over {t0, tau} */
          tic = GETTIME();
          XLAL_CHECK_MAIN ( (transientCand.FstatMap = XLALComputeTransientFstatMap ( thisFAtoms, GV.transientWindowRange, uvar.transient_useFReg)) != NULL, XLAL_EFUNC );
          toc = GETTIME();
          timing.tauTransFstatMap += (toc - tic); // time to compute transient Fstat-map

          /* compute marginalized Bayes factor */
          tic = GETTIME();
          transientCand.logBstat = XLALComputeTransientBstat ( GV.transientWindowRange, transientCand.FstatMap );
          XLAL_CHECK_MAIN ( xlalErrno == 0, XLAL_EFUNC, "XLALComputeTransientBstat() failed with xlalErrno = %d\n", xlalErrno );
          toc = GETTIME();
          timing.tauTransMarg += (toc - tic);

          /* ----- compute parameter posteriors for {t0, tau} */
          pdf1D_t *pdf_t0  = NULL;
          pdf1D_t *pdf_tau = NULL;

          XLAL_CHECK_MAIN ( (pdf_t0 = XLALComputeTransientPosterior_t0 ( GV.transientWindowRange, transientCand.FstatMap )) != NULL, XLAL_EFUNC );
          XLAL_CHECK_MAIN ( (pdf_tau = XLALComputeTransientPosterior_tau ( GV.transientWindowRange, transientCand.FstatMap )) != NULL, XLAL_EFUNC );

          /* get maximum-posterior estimate (MP) from the modes of these pdfs */
          transientCand.t0_MP = XLALFindModeOfPDF1D ( pdf_t0 );
          XLAL_CHECK_MAIN ( xlalErrno == 0, XLAL_EFUNC, "mode-estimation failed for pdf_t0. xlalErrno = %d\n", xlalErrno );
          transientCand.tau_MP =  XLALFindModeOfPDF1D ( pdf_tau );
          XLAL_CHECK_MAIN ( xlalErrno == 0, XLAL_EFUNC, "mode-estimation failed for pdf_tau. xlalErrno = %d\n", xlalErrno );

          /* record timing-relevant transient search params */
          timing.tauMin  = GV.transientWindowRange.tau;
          timing.tauMax  = timing.tauMin + GV.transientWindowRange.tauBand;
          timing.NStart  = transientCand.FstatMap->F_mn->size1;
          timing.NTau    = transientCand.FstatMap->F_mn->size2;

          /* add meta-info on current transient-CW candidate */
          transientCand.doppler = dopplerpos;
          transientCand.windowRange = GV.transientWindowRange;

          /* correct for missing bias in signal-only case */
          if ( uvar.SignalOnly )
            transientCand.FstatMap->maxF += 2;

          /* output everything into stats-file (one line per candidate) */
          XLAL_CHECK_MAIN ( write_transientCandidate_to_fp ( fpTransientStats, &transientCand ) == XLAL_SUCCESS, XLAL_EFUNC );

          /* free dynamically allocated F-stat map */
          XLALDestroyTransientFstatMap ( transientCand.FstatMap );
          XLALDestroyPDF1D ( pdf_t0 );
          XLALDestroyPDF1D ( pdf_tau );

        } /* if fpTransientStats */

      } // for ( iFreq < numFreqBins_FBand )

      /* now measure total loop time per template */
      toc = GETTIME();
      timing.tauTemplate += (toc - tic0);

    } /* while more Doppler positions to scan */


  /* if requested: output timings into timing-file */
  if ( uvar.outputTiming )
    {
      REAL8 num_templates = numTemplates * GV.numFreqBins_FBand;	// 'templates' now refers to number of 'frequency-bands' in resampling case

      timing.NSFTs = GV.NSFTs;
      timing.NFreq = (UINT4) ( 1 + floor ( GV.searchRegion.fkdotBand[0] / GV.dFreq ) );

      // compute averages:
      timing.tauFstat    /= num_templates;
      timing.tauTemplate /= num_templates;
      timing.tauF0       =  timing.tauFstat / timing.NSFTs;

      XLAL_CHECK_MAIN ( write_TimingInfo ( uvar.outputTiming, &timing, &GV ) == XLAL_SUCCESS, XLAL_EFUNC );

    } /* if timing output requested */

  /* ----- if using toplist: sort and write it out to file now ----- */
  if ( fpFstat && GV.FstatToplist )
    {
      UINT4 el;

      /* sort toplist */
      LogPrintf ( LOG_DEBUG, "Sorting toplist ... ");
      if ( GV.RankingStatistic == RANKBY_2F )
        qsort_toplist ( GV.FstatToplist, compareFstatCandidates );
      else if ( GV.RankingStatistic == RANKBY_BSGL )
        qsort_toplist ( GV.FstatToplist, compareFstatCandidates_BSGL );
      else
        XLAL_ERROR ( XLAL_EINVAL, "Ranking statistic '%d' undefined here, allowed are 'F=0' and 'BSGL=2'\n", GV.RankingStatistic );
      LogPrintfVerbatim ( LOG_DEBUG, "done.\n");

      for ( el=0; el < GV.FstatToplist->elems; el ++ )
	{
	  const FstatCandidate *candi;
	  if ( ( candi = (const FstatCandidate *) toplist_elem ( GV.FstatToplist, el )) == NULL ) {
	    LogPrintf ( LOG_CRITICAL, "Internal consistency problems with toplist: contains fewer elements than expected!\n");
	    return -1;
	  }
	  if ( write_FstatCandidate_to_fp ( fpFstat, candi ) != 0 )
	    {
	      LogPrintf (LOG_CRITICAL, "Failed to write candidate to file.\n");
	      return -1;
	    }
	} /* for el < elems in toplist */

    } /* if fpFstat && toplist */

  if ( fpFstat )
    {
      fprintf (fpFstat, "%%DONE\n");
      fclose (fpFstat);
      fpFstat = NULL;
    }
  if ( fpTransientStats ) fclose (fpTransientStats);

  /* ----- estimate amplitude-parameters for the loudest canidate and output into separate file ----- */
  if ( uvar.outputLoudest )
    {
      FILE *fpLoudest;
      PulsarCandidate XLAL_INIT_DECL(pulsarParams);
      pulsarParams.Doppler = loudestFCand.doppler;

      XLAL_CHECK_MAIN ( XLALEstimatePulsarAmplitudeParams ( &pulsarParams, &loudestFCand.FaFb_refTime, loudestFCand.Fa, loudestFCand.Fb, &loudestFCand.Mmunu ) == XLAL_SUCCESS, XLAL_EFUNC );

      XLAL_CHECK_MAIN ( (fpLoudest = fopen (uvar.outputLoudest, "wb")) != NULL, XLAL_ESYS, "Error opening file '%s' for writing..\n\n", uvar.outputLoudest );

      /* write header with run-info */
      fprintf (fpLoudest, "%s", GV.logstring );

      /* write this 'candidate' to disc */
      XLAL_CHECK_MAIN ( write_PulsarCandidate_to_fp ( fpLoudest,  &pulsarParams, &loudestFCand) == XLAL_SUCCESS, XLAL_EFUNC );
      fclose (fpLoudest);

      gsl_matrix_free ( pulsarParams.AmpFisherMatrix );

    } /* write loudest candidate to file */

  LogPrintf (LOG_DEBUG, "Search finished.\n");

  /* write out the Fstatistic histogram */
  if (uvar.outputFstatHist) {

    size_t i = 0;
    FILE *fpFstatHist = fopen(uvar.outputFstatHist, "wb");

    XLAL_CHECK_MAIN (fpFstatHist != NULL, XLAL_ESYS, "\nError opening file '%s' for writing..\n\n", uvar.outputFstat );
    fprintf(fpFstatHist, "%s", GV.logstring);

    for (i = 0; i < Fstat_histogram->size; ++i)
      fprintf(fpFstatHist, "%0.3f %0.3f %i\n",
	      uvar.FstatHistBin * i,
	      uvar.FstatHistBin * (i + 1),
	      gsl_vector_int_get(Fstat_histogram, i));

    fprintf(fpFstatHist, "%%DONE\n");
    fclose(fpFstatHist);

  }

  /* Free memory */
  XLALDestroyDopplerFullScan ( GV.scanState);
  XLALDestroyFstatResults ( Fstat_res );

  Freemem ( &GV );

  if (Fstat_histogram) {
    gsl_vector_int_free(Fstat_histogram);
  }

  /* close log-file */
  if (fpLogPrintf) {
    fclose(fpLogPrintf);
    LogSetFile(fpLogPrintf = NULL);
  }

  /* did we forget anything ? */
  LALCheckMemoryLeaks();

  return 0;

} /* main() */


/**
 * Register all our "user-variables" that can be specified from cmd-line and/or config-file.
 * Here we set defaults for some user-variables and register them with the UserInput module.
 */
int
initUserVars ( UserInput_t *uvar )
{
  XLAL_CHECK ( uvar != NULL, XLAL_EINVAL );

  /* set a few defaults */
  uvar->FreqBand = 0.0;
  uvar->Alpha 	= 0.0;
  uvar->Delta 	= 0.0;
  uvar->RA       = NULL;
  uvar->Dec      = NULL;
  uvar->AlphaBand = 0;
  uvar->DeltaBand = 0;
  uvar->skyRegion = NULL;
  // Dterms-default used to be 16, but has to be 8 for SSE version
  uvar->Dterms 	= 8;

  uvar->ephemEarth = XLALStringDuplicate("earth00-19-DE405.dat.gz");
  uvar->ephemSun = XLALStringDuplicate("sun00-19-DE405.dat.gz");

  uvar->SignalOnly = FALSE;
  uvar->UseNoiseWeights = TRUE;

  /* default step-sizes for GRID_FLAT */
  uvar->dAlpha 	= 0.001;
  uvar->dDelta 	= 0.001;
  uvar->dFreq 	 = 0.0;		/* zero indicates 'not set by user==> i.e. use canonical default */
  uvar->df1dot    = 0.0;
  uvar->df2dot    = 0.0;
  uvar->df3dot    = 0.0;

  /* define default orbital semi-major axis */
  uvar->orbitasini = 0 /* isolated pulsar */;

  uvar->TwoFthreshold = 0.0;
  uvar->NumCandidatesToKeep = 0;
  uvar->FracCandidatesToKeep = 0.0;
  uvar->clusterOnScanline = 0;

  uvar->metricType =  LAL_PMETRIC_NONE;
  uvar->projectMetric = TRUE;
  uvar->gridType = GRID_FLAT;

  uvar->metricMismatch = 0.02;

  uvar->help = FALSE;
  uvar->version = FALSE;

  uvar->outputLogfile = NULL;
  uvar->outputFstat = NULL;
  uvar->outputLoudest = NULL;
  uvar->outputLogPrintf = NULL;

  uvar->outputFstatHist = NULL;
  uvar->FstatHistBin = 0.1;

  uvar->countTemplates = FALSE;

  uvar->gridFile = NULL;

  uvar->dopplermax =  1.05e-4;
  uvar->RngMedWindow = 50;	/* for running-median */

  uvar->SSBprecision = SSBPREC_RELATIVISTIC;

  uvar->minStartTime.gpsSeconds = 0;
  uvar->maxStartTime.gpsSeconds = LAL_INT4_MAX;

  uvar->workingDir = XLALStringDuplicate ( "." );

  uvar->timerCount = 10;	/* output a timer/progress count every N seconds */

  uvar->spindownAge = 0.0;
  uvar->minBraking = 0.0;
  uvar->maxBraking = 0.0;

  uvar->FstatMethod = XLALStringDuplicate("DemodBest");	// default to guessed 'best' demod hotloop variant

  uvar->outputSingleFstats = FALSE;
  uvar->RankingStatistic = XLALStringDuplicate ( "F" );

  uvar->computeBSGL = FALSE;
  uvar->BSGLlogcorr = TRUE;
  uvar->Fstar0 = 0.0;
  uvar->oLGX = NULL;       /* NULL is intepreted as oLGX[X] = 1.0/Ndet for all X */
  uvar->BSGLthreshold = - LAL_REAL8_MAX;

  uvar->transient_WindowType = XLALStringDuplicate ( "none" );
  uvar->transient_useFReg = 0;

  /* ---------- register all user-variables ---------- */
  XLALregBOOLUserStruct( 	help, 		'h', UVAR_HELP,     "Print this message");

  XLALregRAJUserStruct( 	Alpha, 		'a', UVAR_OPTIONAL, "Sky: equatorial J2000 right ascension (in radians or hours:minutes:seconds)");
  XLALregDECJUserStruct( 	Delta, 		'd', UVAR_OPTIONAL, "Sky: equatorial J2000 declination (in radians or degrees:minutes:seconds)");
  XLALregSTRINGUserStruct ( 	skyRegion,      'R', UVAR_OPTIONAL, "ALTERNATIVE: Sky-region polygon '(Alpha1,Delta1),(Alpha2,Delta2),...' or 'allsky'");

  XLALregREALUserStruct( 	Freq, 		'f', UVAR_OPTIONAL, "Starting search frequency Freq in Hz");
  XLALregREALUserStruct( 	f1dot, 		's', UVAR_OPTIONAL, "First spindown parameter  f1dot = dFreq/dt");
  XLALregREALUserStruct( 	f2dot, 		 0 , UVAR_OPTIONAL, "Second spindown parameter f2dot = d^2Freq/dt^2");
  XLALregREALUserStruct( 	f3dot, 		 0 , UVAR_OPTIONAL, "Third spindown parameter  f3dot = d^3Freq/dt^3");

  XLALregRAJUserStruct( 	AlphaBand, 	'z', UVAR_OPTIONAL, "Sky: search band from Alpha to Alpha+AlphaBand (in radians or h:m:s)");
  XLALregDECJUserStruct( 	DeltaBand, 	'c', UVAR_OPTIONAL, "Sky: search band from Delta to Delta+DeltaBand (in radians or d:m:s)");
  XLALregREALUserStruct( 	FreqBand, 	'b', UVAR_OPTIONAL, "Search band in frequency in Hz");
  XLALregREALUserStruct( 	f1dotBand, 	'm', UVAR_OPTIONAL, "Search band in f1dot in Hz/s");
  XLALregREALUserStruct( 	f2dotBand, 	 0 , UVAR_OPTIONAL, "Search band in f2dot in Hz/s^2");
  XLALregREALUserStruct( 	f3dotBand, 	 0 , UVAR_OPTIONAL, "Search band in f3dot in Hz/s^3");

  XLALregRAJUserStruct( 	dAlpha, 	'l', UVAR_OPTIONAL, "Sky: stepsize in Alpha (in radians or h:m:s)");
  XLALregDECJUserStruct( 	dDelta, 	'g', UVAR_OPTIONAL, "Sky: stepsize in Delta (in radians or d:m:s)");
  XLALregREALUserStruct(  	dFreq,          'r', UVAR_OPTIONAL, "Stepsize for frequency in Hz");
  XLALregREALUserStruct( 	df1dot, 	'e', UVAR_OPTIONAL, "Stepsize for f1dot in Hz/s");
  XLALregREALUserStruct( 	df2dot, 	 0 , UVAR_OPTIONAL, "Stepsize for f2dot in Hz/s^2");
  XLALregREALUserStruct( 	df3dot, 	 0 , UVAR_OPTIONAL, "Stepsize for f3dot in Hz/s^3");

  XLALregREALUserStruct( 	orbitasini, 	 0,  UVAR_OPTIONAL, "Binary Orbit: Projected semi-major axis in light-seconds [Default: 0.0]");
  XLALregREALUserStruct( 	orbitPeriod, 	 0,  UVAR_OPTIONAL, "Binary Orbit: Period in seconds");
  XLALregEPOCHUserStruct(	orbitTp, 	 0,  UVAR_OPTIONAL, "Binary Orbit: (true) epoch of periapsis: use 'xx.yy[GPS|MJD]' format.");
  XLALregREALUserStruct( 	orbitArgp, 	 0,  UVAR_OPTIONAL, "Binary Orbit: Orbital argument of periapse in radians");
  XLALregREALUserStruct( 	orbitEcc, 	 0,  UVAR_OPTIONAL, "Binary Orbit: Orbital eccentricity");

  XLALregSTRINGUserStruct(DataFiles, 	'D', UVAR_REQUIRED, "File-pattern specifying (also multi-IFO) input SFT-files");
  XLALregSTRINGUserStruct(IFO, 		'I', UVAR_OPTIONAL, "Detector: 'G1', 'L1', 'H1', 'H2' ...(useful for single-IFO v1-SFTs only!)");

  XLALregBOOLUserStruct( 	SignalOnly, 	'S', UVAR_OPTIONAL, "Signal only flag");

  XLALregREALUserStruct( 	TwoFthreshold,	'F', UVAR_OPTIONAL, "Set the threshold for selection of 2F");
  XLALregINTUserStruct( 	gridType,	 0 , UVAR_OPTIONAL, "Grid: 0=flat, 1=isotropic, 2=metric, 3=skygrid-file, 6=grid-file, 8=spin-square, 9=spin-age-brk");
  XLALregINTUserStruct( 	metricType,	'M', UVAR_OPTIONAL, "Metric: 0=none,1=Ptole-analytic,2=Ptole-numeric, 3=exact");
  XLALregREALUserStruct( 	metricMismatch,	'X', UVAR_OPTIONAL, "Maximal allowed mismatch for metric tiling");
  XLALregSTRINGUserStruct(outputLogfile,	 0,  UVAR_OPTIONAL, "Name of log-file identifying the code + search performed");
  XLALregSTRINGUserStruct(gridFile,	 0,  UVAR_OPTIONAL, "Load grid from this file: sky-grid or full-grid depending on --gridType.");
  XLALregEPOCHUserStruct(refTime,	 	 0,  UVAR_OPTIONAL, "Reference SSB epoch for pulsar-parameters: use 'xx.yy[GPS|MJD]' format [Default: startTime]");

  XLALregSTRINGUserStruct(outputFstat,	 0,  UVAR_OPTIONAL, "Output-file for F-statistic field over the parameter-space");
  XLALregSTRINGUserStruct(outputLoudest,	 0,  UVAR_OPTIONAL, "Loudest F-statistic candidate + estimated MLE amplitudes");

  XLALregSTRINGUserStruct(outputFstatHist, 0,  UVAR_OPTIONAL, "Output-file for a discrete histogram of all Fstatistic values");
  XLALregREALUserStruct(  FstatHistBin,    0,  UVAR_OPTIONAL, "Width of an Fstatistic histogram bin");

  XLALregINTUserStruct(  NumCandidatesToKeep,0, UVAR_OPTIONAL, "Number of Fstat 'candidates' to keep. (0 = All)");
  XLALregREALUserStruct(FracCandidatesToKeep,0, UVAR_OPTIONAL, "Fraction of Fstat 'candidates' to keep.");
  XLALregINTUserStruct(   clusterOnScanline, 0, UVAR_OPTIONAL, "Neighbors on each side for finding 1D local maxima on scanline");

  XLALregEPOCHUserStruct(minStartTime, 	 0,  UVAR_OPTIONAL, "Only use SFTs with timestamps starting from (including) this epoch (format 'xx.yy[GPS|MJD]') ");
  XLALregEPOCHUserStruct(maxStartTime, 	 0,  UVAR_OPTIONAL, "Only use SFTs with timestamps up to (excluding) this epoch (format 'xx.yy[GPS|MJD]')");

  XLALregSTRINGUserStruct(outputFstatAtoms,0,  UVAR_OPTIONAL, "Output filename *base* for F-statistic 'atoms' {a,b,Fa,Fb}_alpha. One file per doppler-point.");
  XLALregBOOLUserStruct(  outputSingleFstats,0,  UVAR_OPTIONAL, "In multi-detector case, also output single-detector F-stats?");
  XLALregSTRINGUserStruct(RankingStatistic,0,  UVAR_DEVELOPER, "Rank toplist candidates according to 'F' or 'BSGL' statistic");

  // ----- Line robust stats parameters ----------
  XLALregBOOLUserStruct(  computeBSGL,	0,  UVAR_OPTIONAL, "Compute and output line-robust statistic BSGL ");
  XLALregREALUserStruct(  Fstar0,		0,  UVAR_OPTIONAL, "BSGL: transition-scale parameter 'Fstar0'");
  XLALregLISTUserStruct(  oLGX,		0,  UVAR_OPTIONAL, "BSGL: prior per-detector line-vs-Gauss odds 'oLGX' (Defaults to oLGX=1/Ndet)");
  XLALregBOOLUserStruct(  BSGLlogcorr,	0,  UVAR_DEVELOPER,"BSGL: include log-correction terms (slower) or not (faster)");
  XLALregREALUserStruct( 	BSGLthreshold,	0,  UVAR_OPTIONAL, "BSGL threshold for candidate output");
  // --------------------------------------------

  XLALregSTRINGUserStruct(outputTransientStats,0,  UVAR_OPTIONAL, "TransientCW: Output filename for transient-CW statistics.");
  XLALregSTRINGUserStruct( transient_WindowType,0,UVAR_OPTIONAL,  "TransientCW: Type of transient signal window to use. ('none', 'rect', 'exp').");
  XLALregREALUserStruct ( transient_t0Days, 0,  UVAR_OPTIONAL,    "TransientCW: Earliest GPS start-time for transient window search, as offset in days from dataStartGPS");
  XLALregREALUserStruct ( transient_t0DaysBand,0,UVAR_OPTIONAL,   "TransientCW: Range of GPS start-times to search in transient search, in days");
  XLALregINTUserStruct ( transient_dt0,    0,  UVAR_OPTIONAL,     "TransientCW: Step-size in transient-CW start-time in seconds [Default:Tsft]");
  XLALregREALUserStruct( transient_tauDays,0,  UVAR_OPTIONAL,     "TransientCW: Minimal transient-CW duration timescale, in days");
  XLALregREALUserStruct( transient_tauDaysBand,0,  UVAR_OPTIONAL, "TransientCW: Range of transient-CW duration timescales to search, in days");
  XLALregINTUserStruct ( transient_dtau,   0,  UVAR_OPTIONAL,     "TransientCW: Step-size in transient-CW duration timescale, in seconds [Default:Tsft]");

  XLALregSTRINGUserStruct(FstatMethod,             0,  UVAR_OPTIONAL,  XLALFstatMethodHelpString() );

  XLALregBOOLUserStruct(  version,	'V', UVAR_SPECIAL,  "Output version information");

  /* ----- more experimental/expert options ----- */
  XLALregREALUserStruct( 	dopplermax, 	'q', UVAR_DEVELOPER, "Maximum doppler shift expected");
  XLALregBOOLUserStruct( 	UseNoiseWeights,'W', UVAR_DEVELOPER, "Use per-SFT noise weights");
  XLALregSTRINGUserStruct(ephemEarth, 	 0,  UVAR_DEVELOPER, "Earth ephemeris file to use");
  XLALregSTRINGUserStruct(ephemSun, 	 0,  UVAR_DEVELOPER, "Sun ephemeris file to use");

  XLALregINTUserStruct ( 	SSBprecision,	 0,  UVAR_DEVELOPER, "Precision to use for time-transformation to SSB: 0=Newtonian 1=relativistic");

  XLALregINTUserStruct( 	RngMedWindow,	'k', UVAR_DEVELOPER, "Running-Median window size");
  XLALregINTUserStruct(	Dterms,		't', UVAR_DEVELOPER, "Number of terms to keep in Dirichlet kernel sum");

  XLALregSTRINGUserStruct(workingDir,     'w', UVAR_DEVELOPER, "Directory to use as work directory.");
  XLALregREALUserStruct( 	timerCount, 	 0,  UVAR_DEVELOPER, "N: Output progress/timer info every N seconds");

  XLALregBOOLUserStruct( 	projectMetric, 	 0,  UVAR_DEVELOPER, "Use projected metric on Freq=const subspact");

  XLALregSTRINGUserStruct(outputLogPrintf, 0,  UVAR_DEVELOPER, "Send all output from LogPrintf statements to this file");

  XLALregBOOLUserStruct( 	countTemplates,  0,  UVAR_DEVELOPER, "Count number of templates (if supported) instead of search");

  XLALregREALUserStruct(  spindownAge,     0,  UVAR_DEVELOPER, "Spindown age for --gridType=9");
  XLALregREALUserStruct(  minBraking,      0,  UVAR_DEVELOPER, "Minimum braking index for --gridType=9");
  XLALregREALUserStruct(  maxBraking,      0,  UVAR_DEVELOPER, "Maximum braking index for --gridType=9");

  XLALregBOOLUserStruct(transient_useFReg,   	 0,  UVAR_DEVELOPER, "FALSE: use 'standard' e^F for marginalization, if TRUE: use e^FReg = (1/D)*e^F (BAD)");

  XLALregSTRINGUserStruct(outputTiming,         0,  UVAR_DEVELOPER, "Append timing measurements and parameters into this file");

  // ---------- deprecated but still-supported or tolerated options ----------
  XLALRegisterUvarMember ( RA,	STRING, 0, DEPRECATED, "Use --Alpha instead" );
  XLALRegisterUvarMember ( Dec, STRING, 0, DEPRECATED, "Use --Delta instead");

  XLALRegisterUvarMember ( internalRefTime, EPOCH, 0, DEPRECATED, "HAS NO EFFECT and should no longer be used: XLALComputeFstat() now always uses midtime internally ... ");
  // ---------- obsolete and unsupported options ----------

  return XLAL_SUCCESS;

} /* initUserVars() */

/** Initialized Fstat-code: handle user-input and set everything up.
 * NOTE: the logical *order* of things in here is very important, so be careful
 */
int
InitFstat ( ConfigVariables *cfg, const UserInput_t *uvar )
{
  XLAL_CHECK ( (cfg != NULL) && (uvar != NULL), XLAL_EINVAL );

  REAL8 fCoverMin, fCoverMax;	/* covering frequency-band to read from SFTs */
  SFTCatalog *catalog = NULL;
  SFTConstraints XLAL_INIT_DECL(constraints);
  LIGOTimeGPS endTime;
  size_t toplist_length = uvar->NumCandidatesToKeep;

  /* set the current working directory */
  XLAL_CHECK ( chdir ( uvar->workingDir ) == 0, XLAL_EINVAL, "Unable to change directory to workinDir '%s'\n", uvar->workingDir );

  /* ----- set computational parameters for F-statistic from User-input ----- */
  XLAL_CHECK ( XLALParseFstatMethodString ( &cfg->FstatMethod, uvar->FstatMethod ) == XLAL_SUCCESS, XLAL_EFUNC );

  cfg->useResamp = ( cfg->FstatMethod >= FMETHOD_RESAMP_GENERIC ); // use resampling;

  /* use IFO-contraint if one given by the user */
  if ( XLALUserVarWasSet ( &uvar->IFO ) ) {
    XLAL_CHECK ( (constraints.detector = XLALGetChannelPrefix ( uvar->IFO )) != NULL, XLAL_EFUNC );
  }
  LIGOTimeGPS minStartTime = uvar->minStartTime;
  LIGOTimeGPS maxStartTime = uvar->maxStartTime;
  constraints.minStartTime = &minStartTime;
  constraints.maxStartTime = &maxStartTime;

  /* get full SFT-catalog of all matching (multi-IFO) SFTs */
  LogPrintf (LOG_DEBUG, "Finding all SFTs to load ... ");
  XLAL_CHECK ( (catalog = XLALSFTdataFind ( uvar->DataFiles, &constraints )) != NULL, XLAL_EFUNC );
  LogPrintfVerbatim (LOG_DEBUG, "done. (found %d SFTs)\n", catalog->length);

  if ( constraints.detector ) {
    XLALFree ( constraints.detector );
  }

  XLAL_CHECK ( catalog->length > 0, XLAL_EINVAL, "\nSorry, didn't find any matching SFTs with pattern '%s'!\n\n", uvar->DataFiles );

  /* deduce start- and end-time of the observation spanned by the data */
  UINT4 numSFTfiles = catalog->length;
  cfg->Tsft = 1.0 / catalog->data[0].header.deltaF;
  cfg->startTime = catalog->data[0].header.epoch;
  endTime   = catalog->data[numSFTfiles - 1].header.epoch;
  XLALGPSAdd ( &endTime, cfg->Tsft );	/* add on Tsft to last SFT start-time */

  // time spanned by the SFTs
  cfg->Tspan = XLALGPSDiff ( &endTime, &cfg->startTime );

  /* ----- load ephemeris-data ----- */
  XLAL_CHECK ( (cfg->ephemeris = XLALInitBarycenter( uvar->ephemEarth, uvar->ephemSun )) != NULL, XLAL_EFUNC );

  /* ----- get reference-times (from user if given, use startTime otherwise): ----- */
  LIGOTimeGPS refTime;
  if ( XLALUserVarWasSet(&uvar->refTime) )
    {
      refTime = uvar->refTime;
    }
  else
    {
      refTime = cfg->startTime;
    }

  /* define sky position variables from user input */
  if (XLALUserVarWasSet(&uvar->RA))
    {
      XLALTranslateHMStoRAD ( &cfg->Alpha, uvar->RA );
    }
  else cfg->Alpha = uvar->Alpha;
  if (XLALUserVarWasSet(&uvar->Dec))
    {
      XLALTranslateDMStoRAD( &cfg->Delta, uvar->Dec );
    }
  else cfg->Delta = uvar->Delta;


  REAL8 fMin, fMax, f1dotMin, f1dotMax, f2dotMin, f2dotMax, f3dotMin, f3dotMax;
  { /* ----- prepare spin-range at refTime (in *canonical format*, ie all Bands >= 0) ----- */
    fMin = MYMIN ( uvar->Freq, uvar->Freq + uvar->FreqBand );
    fMax = MYMAX ( uvar->Freq, uvar->Freq + uvar->FreqBand );

    /* Specific to --gridType=GRID_SPINDOWN_AGEBRK parameter space */
    if (uvar->gridType == GRID_SPINDOWN_AGEBRK) {

      /* Set the first and second spindown ranges to the maximum that will be
       * encountered by the age-braking index parameter space. These will *ONLY*
       * be used by older code to, e.g. load the correct band of SFTs, and
       * will *NOT* be used by the tiling code itself.
       * The formulas used here are, with age=a, min braking=n, max braking=N:
       *
       * -f0/((n-1)*a) <= f1 <= -f0/((N-1)*a)
       * n*(f1^2)/f0 <= f2 <= N*(f1^2)/f0
       *
       * f0/f1 are taken as the maximum/minimum value appropriate for getting the
       * maximum/minimum values of f1/f2 (note that f1 is strictly negative here).
       */

      f1dotMin = -1.0 * fMax / ((uvar->minBraking - 1.0) * uvar->spindownAge);
      f1dotMax = -1.0 * fMin / ((uvar->maxBraking - 1.0) * uvar->spindownAge);

      f2dotMin = uvar->minBraking * (f1dotMax * f1dotMax) / fMax;
      f2dotMax = uvar->maxBraking * (f1dotMin * f1dotMin) / fMin;

    }
    else {     /* Used for all other --gridTypes */

      f1dotMin = MYMIN ( uvar->f1dot, uvar->f1dot + uvar->f1dotBand );
      f1dotMax = MYMAX ( uvar->f1dot, uvar->f1dot + uvar->f1dotBand );

      f2dotMin = MYMIN ( uvar->f2dot, uvar->f2dot + uvar->f2dotBand );
      f2dotMax = MYMAX ( uvar->f2dot, uvar->f2dot + uvar->f2dotBand );

    }

    f3dotMin = MYMIN ( uvar->f3dot, uvar->f3dot + uvar->f3dotBand );
    f3dotMax = MYMAX ( uvar->f3dot, uvar->f3dot + uvar->f3dotBand );

  } /* spin-range at refTime */


  { /* ----- set up Doppler region to scan ----- */
    BOOLEAN haveAlphaDelta = (XLALUserVarWasSet(&uvar->Alpha) && XLALUserVarWasSet(&uvar->Delta)) || (XLALUserVarWasSet(&uvar->RA) && XLALUserVarWasSet(&uvar->Dec));

    if (uvar->skyRegion)
      {
	XLAL_CHECK ( (cfg->searchRegion.skyRegionString = XLALCalloc(1, strlen(uvar->skyRegion)+1)) != NULL, XLAL_ENOMEM );
	strcpy (cfg->searchRegion.skyRegionString, uvar->skyRegion);
      }
    else if (haveAlphaDelta)    /* parse this into a sky-region */
      {
	XLAL_CHECK ( (cfg->searchRegion.skyRegionString = XLALSkySquare2String ( cfg->Alpha, cfg->Delta, uvar->AlphaBand, uvar->DeltaBand)) != NULL, XLAL_EFUNC );
      }
    else if (!uvar->gridFile)
      {
	XLAL_ERROR ( XLAL_EINVAL, "\nCould not setup searchRegion, have neither skyRegion nor (Alpha, Delta) nor a gridFile!\n\n" );
      }

    /* spin searchRegion defined by spin-range at reference-time */
    cfg->searchRegion.refTime = refTime;

    cfg->searchRegion.fkdot[0] = fMin;
    cfg->searchRegion.fkdot[1] = f1dotMin;
    cfg->searchRegion.fkdot[2] = f2dotMin;
    cfg->searchRegion.fkdot[3] = f3dotMin;

    cfg->searchRegion.fkdotBand[0] = fMax - fMin;
    cfg->searchRegion.fkdotBand[1] = f1dotMax - f1dotMin;
    cfg->searchRegion.fkdotBand[2] = f2dotMax - f2dotMin;
    cfg->searchRegion.fkdotBand[3] = f3dotMax - f3dotMin;

  } /* get DopplerRegion */

  /* ----- set fixed grid step-sizes from user-input: only used for GRID_FLAT ----- */
  cfg->stepSizes.Alpha = uvar->dAlpha;
  cfg->stepSizes.Delta = uvar->dDelta;
  cfg->stepSizes.fkdot[0] = uvar->dFreq;
  cfg->stepSizes.fkdot[1] = uvar->df1dot;
  cfg->stepSizes.fkdot[2] = uvar->df2dot;
  cfg->stepSizes.fkdot[3] = uvar->df3dot;


  REAL8 tmpFreqBandRef = cfg->searchRegion.fkdotBand[0];

  /* initialize full multi-dimensional Doppler-scanner */
  {
    DopplerFullScanInit scanInit;			/* init-structure for DopperScanner */

    scanInit.searchRegion = cfg->searchRegion;
    if ( cfg->useResamp ) {	// in the resampling-case, temporarily take out frequency-dimension of the Doppler template bank
      scanInit.searchRegion.fkdotBand[0] = 0;
    }

    scanInit.gridType = uvar->gridType;
    scanInit.gridFile = uvar->gridFile;
    scanInit.metricType = uvar->metricType;
    scanInit.projectMetric = uvar->projectMetric;
    scanInit.metricMismatch = uvar->metricMismatch;
    scanInit.stepSizes = cfg->stepSizes;
    scanInit.ephemeris = cfg->ephemeris;		/* used by Ephemeris-based metric */
    scanInit.startTime = cfg->startTime;
    scanInit.Tspan     = cfg->Tspan;

    // just use first SFTs' IFO for metric (should be irrelevant)
    LALDetector *detector;
    XLAL_CHECK ( (detector = XLALGetSiteInfo ( catalog->data[0].header.name ) ) != NULL, XLAL_EFUNC );
    scanInit.Detector  = detector;

    /* Specific to --gridType=GRID_SPINDOWN_AGEBRK parameter space */
    if (uvar->gridType == GRID_SPINDOWN_AGEBRK) {
      scanInit.extraArgs[0] = uvar->spindownAge;
      scanInit.extraArgs[1] = uvar->minBraking;
      scanInit.extraArgs[2] = uvar->maxBraking;
    }

    LogPrintf (LOG_DEBUG, "Setting up template grid ... ");
    XLAL_CHECK ( (cfg->scanState = XLALInitDopplerFullScan ( &scanInit)) != NULL, XLAL_EFUNC );
    LogPrintfVerbatim (LOG_DEBUG, "template grid ready.\n");
    XLALNumDopplerTemplates ( cfg->scanState );
    XLALFree ( detector );
  }


  { /* ----- What frequency-band do we need to read from the SFTs?
     * propagate spin-range from refTime to startTime and endTime of observation
     */
    PulsarSpinRange spinRangeRef, spinRangeStart, spinRangeEnd;	/* temporary only */
    REAL8 fmaxStart, fmaxEnd, fminStart, fminEnd;

    // extract spanned spin-range at reference-time from the template-bank
    XLAL_CHECK ( XLALGetDopplerSpinRange ( &spinRangeRef, cfg->scanState ) == XLAL_SUCCESS, XLAL_EFUNC );

    // in the resampling case, we need to restore the frequency-band info now, which we set to 0
    // before calling the DopplerInit template bank construction
    if ( cfg->useResamp ) {
      spinRangeRef.fkdotBand[0] = tmpFreqBandRef;
    }

    // write this search spin-range@refTime back into 'cfg' struct for users' reference
    cfg->searchRegion.refTime = spinRangeRef.refTime;
    memcpy ( cfg->searchRegion.fkdot, spinRangeRef.fkdot, sizeof(cfg->searchRegion.fkdot) );
    memcpy ( cfg->searchRegion.fkdotBand, spinRangeRef.fkdotBand, sizeof(cfg->searchRegion.fkdotBand) );

    /* compute spin-range at startTime of observation */
    REAL8 dtau = XLALGPSDiff ( &cfg->startTime, &spinRangeRef.refTime );
    XLAL_CHECK ( XLALExtrapolatePulsarSpinRange ( &spinRangeStart, &spinRangeRef, dtau ) == XLAL_SUCCESS, XLAL_EFUNC );
    /* compute spin-range at endTime of these SFTs */
    dtau = XLALGPSDiff ( &endTime, &spinRangeStart.refTime );
    XLAL_CHECK ( XLALExtrapolatePulsarSpinRange (&spinRangeEnd, &spinRangeStart, dtau) == XLAL_SUCCESS, XLAL_EFUNC );

    fminStart = spinRangeStart.fkdot[0];
    /* ranges are in canonical format! */
    fmaxStart = fminStart + spinRangeStart.fkdotBand[0];
    fminEnd   = spinRangeEnd.fkdot[0];
    fmaxEnd   = fminEnd + spinRangeEnd.fkdotBand[0];

    /*  get covering frequency-band  */
    fCoverMax = MYMAX ( fmaxStart, fmaxEnd );
    fCoverMin = MYMIN ( fminStart, fminEnd );

    /* correct for doppler-shift */
    fCoverMax *= 1.0 + uvar->dopplermax;
    fCoverMin *= 1.0 - uvar->dopplermax;

  } /* extrapolate spin-range */

  /* if single-only flag is given, assume a PSD with sqrt(S) = 1.0 */
  MultiNoiseFloor s_assumeSqrtSX, *assumeSqrtSX;
  if ( uvar->SignalOnly ) {
    s_assumeSqrtSX.length = XLALCountIFOsInCatalog(catalog);
    for (UINT4 X = 0; X < s_assumeSqrtSX.length; ++X) {
      s_assumeSqrtSX.sqrtSn[X] = 1.0;
    }
    assumeSqrtSX = &s_assumeSqrtSX;
  } else {
    assumeSqrtSX = NULL;
  }

  if ( XLALUserVarWasSet ( &uvar->dFreq) ) {
    cfg->dFreq = uvar->dFreq;
  } else {
    cfg->dFreq = 1.0/(2*cfg->Tspan);
  }
  if ( cfg->useResamp )	{ // handle special resampling case, where we deal with a vector of F-stat values instead of one
    cfg->numFreqBins_FBand = (UINT4) ( 1 + floor ( cfg->searchRegion.fkdotBand[0] / cfg->dFreq ) );
  } else {
    cfg->numFreqBins_FBand = 1;	// number of frequency-bins in the frequency-band used for resampling (1 if not using Resampling)
  }

  PulsarParamsVector *injectSources = NULL;
  MultiNoiseFloor *injectSqrtSX = NULL;
  FstatOptionalArgs optionalArgs = FstatOptionalArgsDefaults;
  optionalArgs.Dterms  = uvar->Dterms;
  optionalArgs.SSBprec = uvar->SSBprecision;
  optionalArgs.runningMedianWindow = uvar->RngMedWindow;
  optionalArgs.injectSources = injectSources;
  optionalArgs.injectSqrtSX = injectSqrtSX;
  optionalArgs.assumeSqrtSX = assumeSqrtSX;
  optionalArgs.FstatMethod = cfg->FstatMethod;

  XLAL_CHECK ( (cfg->Fstat_in = XLALCreateFstatInput( catalog, fCoverMin, fCoverMax, cfg->dFreq, cfg->ephemeris, &optionalArgs )) != NULL, XLAL_EFUNC );
  XLALDestroySFTCatalog(catalog);

  cfg->Fstat_what = FSTATQ_2F;   // always calculate multi-detector 2F
  if ( XLALUserVarWasSet( &uvar->outputLoudest ) ) {
    cfg->Fstat_what |= FSTATQ_FAFB;   // also calculate Fa,b parts for parameter estimation
  }

  /* get SFT detectors and timestamps */
  const MultiLALDetector *multiIFO;
  XLAL_CHECK ( (multiIFO = XLALGetFstatInputDetectors( cfg->Fstat_in )) != NULL, XLAL_EFUNC );
  const MultiLIGOTimeGPSVector *multiTS;
  XLAL_CHECK ( (multiTS = XLALGetFstatInputTimestamps( cfg->Fstat_in )) != NULL, XLAL_EFUNC );

  /* count total number of SFTs loaded */
  cfg->NSFTs = 0;
  for ( UINT4 X = 0; X < multiTS->length; X++ ) {
    cfg->NSFTs += multiTS->data[X]->length;
  }

  /* for column headings string, get number of detectors, detector name vector, and SFTs per detector vector */
  {
    const UINT4 numDetectors = multiIFO->length;
    XLAL_CHECK ( (cfg->numSFTsPerDet = XLALCreateUINT4Vector( numDetectors )) != NULL, XLAL_EFUNC );
    cfg->detectorIDs = NULL;
    for (UINT4 X = 0; X < numDetectors; X++) {
      cfg->numSFTsPerDet->data[X] = multiTS->data[X]->length;
      XLAL_CHECK ( (cfg->detectorIDs = XLALAppendString2Vector ( cfg->detectorIDs, multiIFO->sites[X].frDetector.prefix )) != NULL, XLAL_EFUNC );
    } /* for X < numDetectors */
  }

  /* ----- set up scanline-window if requested for 1D local-maximum clustering on scanline ----- */
  XLAL_CHECK ( (cfg->scanlineWindow = XLALCreateScanlineWindow ( uvar->clusterOnScanline )) != NULL, XLAL_EFUNC );

  /* set number of toplist candidates from fraction if asked to */
  if (0.0 < uvar->FracCandidatesToKeep && uvar->FracCandidatesToKeep <= 1.0) {
    if (XLALNumDopplerTemplates(cfg->scanState) <= 0.0) {
      XLAL_ERROR ( XLAL_EINVAL , "Cannot use FracCandidatesToKeep because number of templates was counted to be zero!\n");
    }
    toplist_length = ceil(XLALNumDopplerTemplates(cfg->scanState) * uvar->FracCandidatesToKeep);
  }

  /* ----- set up toplist if requested ----- */
  if ( toplist_length > 0 ) {
    if ( strcmp(uvar->RankingStatistic, "F") == 0 )
     cfg->RankingStatistic = RANKBY_2F;
    else if ( strcmp(uvar->RankingStatistic, "BSGL") == 0 )
      {
        if ( !uvar->computeBSGL ) {
          XLAL_ERROR ( XLAL_EINVAL, "\nERROR: Ranking by BSGL only possible if --computeBSGL given.\n\n");
        }
        cfg->RankingStatistic = RANKBY_BSGL;
      }
    else
      {
        XLAL_ERROR ( XLAL_EINVAL, "\nERROR: Invalid value specified for candidate ranking - supported are 'F' and 'BSGL'.\n\n");
      }

    if ( cfg->RankingStatistic == RANKBY_BSGL )
      {
        XLAL_CHECK ( create_toplist( &(cfg->FstatToplist), toplist_length, sizeof(FstatCandidate), compareFstatCandidates_BSGL) == 0, XLAL_EFUNC );
      }
    else // rank by F-stat
      {
        XLAL_CHECK ( create_toplist( &(cfg->FstatToplist), toplist_length, sizeof(FstatCandidate), compareFstatCandidates) == 0, XLAL_EFUNC );
      }
  } /* if toplist_length > 0 */


  /* ----- transient-window related parameters ----- */
  int twtype;
  XLAL_CHECK ( (twtype = XLALParseTransientWindowName ( uvar->transient_WindowType )) >= 0, XLAL_EFUNC );
  cfg->transientWindowRange.type = twtype;

  /* make sure user doesn't set window=none but sets window-parameters => indicates she didn't mean 'none' */
  if ( cfg->transientWindowRange.type == TRANSIENT_NONE )
    if ( XLALUserVarWasSet ( &uvar->transient_t0Days ) || XLALUserVarWasSet ( &uvar->transient_t0DaysBand ) || XLALUserVarWasSet ( &uvar->transient_dt0 ) ||
         XLALUserVarWasSet ( &uvar->transient_tauDays ) || XLALUserVarWasSet ( &uvar->transient_tauDaysBand ) || XLALUserVarWasSet ( &uvar->transient_dtau ) ) {
      XLAL_ERROR ( XLAL_EINVAL, "ERROR: transientWindow->type == NONE, but window-parameters were set! Use a different window-type!\n" );
    }

  if (   uvar->transient_t0DaysBand < 0 || uvar->transient_tauDaysBand < 0 ) {
    XLAL_ERROR (XLAL_EINVAL, "Only positive t0/tau bands allowed (%f, %f)\n", uvar->transient_t0DaysBand, uvar->transient_tauDaysBand );
  }

  cfg->transientWindowRange.t0      = cfg->startTime.gpsSeconds + uvar->transient_t0Days * DAY24;
  cfg->transientWindowRange.t0Band  = uvar->transient_t0DaysBand * DAY24;


  if ( XLALUserVarWasSet ( &uvar->transient_dt0 ) ) {
    cfg->transientWindowRange.dt0 = uvar->transient_dt0;
  }
  else {
    cfg->transientWindowRange.dt0 = cfg->Tsft;
  }

  cfg->transientWindowRange.tau     = (UINT4) ( uvar->transient_tauDays * DAY24 );
  cfg->transientWindowRange.tauBand = (UINT4) ( uvar->transient_tauDaysBand * DAY24 );

  if ( XLALUserVarWasSet ( &uvar->transient_dtau ) ) {
    cfg->transientWindowRange.dtau = uvar->transient_dtau;
  }
  else {
    cfg->transientWindowRange.dtau = cfg->Tsft;
  }

  /* get atoms back from Fstat-computing, either if atoms-output or transient-Bstat output was requested */
  if ( ( uvar->outputFstatAtoms != NULL ) || ( uvar->outputTransientStats != NULL ) ) {
    cfg->Fstat_what |= FSTATQ_ATOMS_PER_DET;
  }

  /* return single-IFO Fstat values for Line-robust statistic */
  if ( uvar->outputSingleFstats || uvar->computeBSGL ) {
    cfg->Fstat_what |= FSTATQ_2F_PER_DET;
  }

  /* ---------- prepare Line-robust statistics parameters ---------- */
  if ( uvar->computeBSGL )
    {
      UINT4 numDetectors = multiIFO->length;
      /* take BSGL user input and pre-compute the corresponding BSGLsetup */
      REAL4 *oLGX_p = NULL;
      REAL4 oLGX[PULSAR_MAX_DETECTORS];
      if ( uvar->oLGX != NULL )
        {
          if ( uvar->oLGX->length != numDetectors ) {
            XLAL_ERROR ( XLAL_EINVAL, "Invalid input: length(oLGX) = %d differs from number of detectors (%d)'\n", uvar->oLGX->length, numDetectors );
          }
          XLAL_CHECK ( XLALParseLinePriors ( &oLGX[0], uvar->oLGX ) == XLAL_SUCCESS, XLAL_EFUNC );
          oLGX_p = &oLGX[0];
        } // if uvar->oLGX != NULL

      XLAL_CHECK ( (cfg->BSGLsetup = XLALCreateBSGLSetup ( numDetectors, uvar->Fstar0, oLGX_p, uvar->BSGLlogcorr )) != NULL, XLAL_EFUNC );
    } // if uvar_computeBSGL

  return XLAL_SUCCESS;

} /* InitFstat() */

/**
 * Produce a log-string describing the present run-setup
 */
CHAR *
XLALGetLogString ( const ConfigVariables *cfg )
{
  XLAL_CHECK_NULL ( cfg != NULL, XLAL_EINVAL );

#define BUFLEN 1024
  CHAR buf[BUFLEN];

  CHAR *logstr = NULL;
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, "%% cmdline: " )) != NULL, XLAL_EFUNC );
  CHAR *cmdline;
  XLAL_CHECK_NULL ( (cmdline = XLALUserVarGetLog ( UVAR_LOGFMT_CMDLINE )) != NULL, XLAL_EFUNC );
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, cmdline )) != NULL, XLAL_EFUNC );
  XLALFree ( cmdline );
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, "\n" )) != NULL, XLAL_EFUNC );
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, cfg->VCSInfoString )) != NULL, XLAL_EFUNC );

  XLAL_CHECK_NULL ( snprintf ( buf, BUFLEN, "%%%% FstatMethod used: '%s'\n", XLALGetFstatInputMethodName ( cfg->Fstat_in ) ) < BUFLEN, XLAL_EBADLEN );
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, buf )) != NULL, XLAL_EFUNC );

  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, "%% Started search: " )) != NULL, XLAL_EFUNC );
  time_t tp = GETTIME();
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, asctime( gmtime( &tp ) ))) != NULL, XLAL_EFUNC );
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, "%% Loaded SFTs: [ " )) != NULL, XLAL_EFUNC );

  UINT4 numDet = cfg->detectorIDs->length;
  for ( UINT4 X=0; X < numDet; X ++ )
    {
      XLAL_CHECK_NULL ( snprintf ( buf, BUFLEN, "%s:%d%s",  cfg->detectorIDs->data[X], cfg->numSFTsPerDet->data[X], (X < numDet - 1) ? ", " : " ]\n" ) < BUFLEN, XLAL_EBADLEN );
      XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, buf )) != NULL, XLAL_EFUNC );
    }
  INT4 startTimeSeconds = cfg->startTime.gpsSeconds;
  struct tm startTimeUTC = *XLALGPSToUTC ( &startTimeUTC, startTimeSeconds );
  {
    CHAR *startTimeUTCString = XLALStringDuplicate ( asctime(&startTimeUTC) );
    startTimeUTCString[strlen(startTimeUTCString)-2] = 0;	// kill trailing newline
    XLAL_CHECK_NULL ( snprintf ( buf, BUFLEN, "%%%% GPS starttime         = %d (%s GMT)\n", startTimeSeconds, startTimeUTCString ) < BUFLEN, XLAL_EBADLEN );
    XLALFree ( startTimeUTCString );
  }
  char bufGPS[32];
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, buf )) != NULL, XLAL_EFUNC );
  XLAL_CHECK_NULL ( snprintf ( buf, BUFLEN, "%%%% Total time spanned    = %.0f s (%.2f hours)\n", cfg->Tspan, cfg->Tspan/3600.0 ) < BUFLEN, XLAL_EBADLEN );
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, buf )) != NULL, XLAL_EFUNC );
  XLAL_CHECK_NULL ( snprintf (buf, BUFLEN, "%%%% Pulsar-params refTime = %s\n", XLALGPSToStr ( bufGPS, &(cfg->searchRegion.refTime) )) < BUFLEN, XLAL_EBADLEN );
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, buf )) != NULL, XLAL_EFUNC );
  XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, "%% Spin-range at refTime: fkdot = [ " )) != NULL, XLAL_EFUNC );
  for (UINT4 k=0; k < PULSAR_MAX_SPINS; k ++ )
    {
      XLAL_CHECK_NULL ( snprintf(buf, BUFLEN, "%.16g:%.16g%s", cfg->searchRegion.fkdot[k], cfg->searchRegion.fkdot[k] + cfg->searchRegion.fkdotBand[k],
                                 (k < PULSAR_MAX_SPINS - 1)?", ":" ]\n") < BUFLEN, XLAL_EBADLEN );
      XLAL_CHECK_NULL ( (logstr = XLALStringAppend ( logstr, buf )) != NULL, XLAL_EFUNC );
    }

  /* return result */
  return logstr;

} // XLALGetLogString()



/***********************************************************************/
/**
 * Log the all relevant parameters of the present search-run to a log-file.
 * The name of the log-file is log_fname
 * <em>NOTE:</em> Currently this function only logs the user-input and code-versions.
 */
int
WriteFstatLog ( const CHAR *log_fname, const CHAR *log_string )
{
  XLAL_CHECK ( (log_fname != NULL) && (log_string != NULL), XLAL_EINVAL );

  /* prepare log-file for writing */
  FILE *fplog;
  XLAL_CHECK ( (fplog = fopen(log_fname, "wb" )) != NULL, XLAL_ESYS, "Failed to open log-file '%s' for writing.\n\n", log_fname );

  fprintf (fplog, "%%%% LOG-FILE for ComputeFstatistic run\n\n");
  fprintf (fplog, "%s", log_string);
  fclose (fplog);

  return XLAL_SUCCESS;

} /* WriteFstatLog() */


/** Free all globally allocated memory. */
void
Freemem( ConfigVariables *cfg )
{
  if ( !cfg ) {
    return;
  }
  XLALDestroyUINT4Vector ( cfg->numSFTsPerDet );
  XLALDestroyStringVector ( cfg->detectorIDs );

  XLALDestroyFstatInput ( cfg->Fstat_in );

  /* destroy FstatToplist if any */
  if ( cfg->FstatToplist ) {
    free_toplist( &(cfg->FstatToplist) );
  }

  if ( cfg->scanlineWindow ) {
    XLALDestroyScanlineWindow ( cfg->scanlineWindow );
  }

  /* Free config-Variables and userInput stuff */
  XLALDestroyUserVars();

  if ( cfg->searchRegion.skyRegionString ) {
    XLALFree ( cfg->searchRegion.skyRegionString );
  }

  /* Free ephemeris data */
  XLALDestroyEphemerisData ( cfg->ephemeris );

  if ( cfg->VCSInfoString ) {
    XLALFree ( cfg->VCSInfoString );
  }
  if ( cfg->logstring ) {
    XLALFree ( cfg->logstring );
  }

  XLALFree ( cfg->BSGLsetup );

  return;

} /* Freemem() */


/*----------------------------------------------------------------------*/
/**
 * Some general consistency-checks on user-input.
 * Throws an error plus prints error-message if problems are found.
 */
int
checkUserInputConsistency ( const UserInput_t *uvar )
{
  XLAL_CHECK ( uvar != NULL, XLAL_EINVAL );

  /* check that only alpha OR RA has been set */
  if ( XLALUserVarWasSet(&uvar->Alpha) && (XLALUserVarWasSet(&uvar->RA)) )
    {
      XLALPrintError ("\nInput either Alpha OR RA, not both!\n\n");
      XLAL_ERROR ( XLAL_EINVAL );
    }
  /* check that only delta OR Dec has been set */
  if ( XLALUserVarWasSet(&uvar->Delta) && (XLALUserVarWasSet(&uvar->Dec)) )
    {
      XLALPrintError ("\nInput either Delta OR Dec, not both!\n\n");
      XLAL_ERROR ( XLAL_EINVAL );
    }

  /* check for negative stepsizes in Freq, Alpha, Delta */
  if ( XLALUserVarWasSet(&uvar->dAlpha) && (uvar->dAlpha < 0) )
    {
      XLALPrintError ("\nNegative value of stepsize dAlpha not allowed!\n\n");
      XLAL_ERROR ( XLAL_EINVAL );
    }
  if ( XLALUserVarWasSet(&uvar->dDelta) && (uvar->dDelta < 0) )
    {
      XLALPrintError ("\nNegative value of stepsize dDelta not allowed!\n\n");
      XLAL_ERROR ( XLAL_EINVAL );
    }
  if ( XLALUserVarWasSet(&uvar->dFreq) && (uvar->dFreq < 0) )
    {
      XLALPrintError ("\nNegative value of stepsize dFreq not allowed!\n\n");
      XLAL_ERROR ( XLAL_EINVAL );
    }

  /* binary parameter checks */
  if ( XLALUserVarWasSet(&uvar->orbitPeriod) && (uvar->orbitPeriod <= 0) )
    {
      XLALPrintError ("\nNegative or zero value of orbital period not allowed!\n\n");
      XLAL_ERROR ( XLAL_EINVAL );
    }
  if ( XLALUserVarWasSet(&uvar->orbitasini) && (uvar->orbitasini < 0) )
    {
      XLALPrintError ("\nNegative value of projected orbital semi-major axis not allowed!\n\n");
      XLAL_ERROR ( XLAL_EINVAL );
    }
  if ( XLALUserVarWasSet(&uvar->orbitArgp) && ((uvar->orbitArgp < 0) || (uvar->orbitArgp >= LAL_TWOPI)) )
    {
      XLALPrintError ("\nOrbital argument of periapse must lie in range [0 2*PI)!\n\n");
      XLAL_ERROR ( XLAL_EINVAL );
    }
  if ( XLALUserVarWasSet(&uvar->orbitEcc) && (uvar->orbitEcc < 0) )
    {
      XLALPrintError ("\nNegative value of orbital eccentricity not allowed!\n\n");
      XLAL_ERROR ( XLAL_EINVAL );
    }

  /* grid-related checks */
  {
    BOOLEAN haveAlphaBand = XLALUserVarWasSet( &uvar->AlphaBand );
    BOOLEAN haveDeltaBand = XLALUserVarWasSet( &uvar->DeltaBand );
    BOOLEAN haveSkyRegion, haveAlphaDelta, haveGridFile;
    BOOLEAN useSkyGridFile, useFullGridFile, haveMetric, useMetric;

    haveSkyRegion  	= (uvar->skyRegion != NULL);
    haveAlphaDelta 	= (XLALUserVarWasSet(&uvar->Alpha) && XLALUserVarWasSet(&uvar->Delta) ) || (XLALUserVarWasSet(&uvar->RA) && XLALUserVarWasSet(&uvar->Dec) );
    haveGridFile      	= (uvar->gridFile != NULL);
    useSkyGridFile   	= (uvar->gridType == GRID_FILE_SKYGRID);
    useFullGridFile	= (uvar->gridType == GRID_FILE_FULLGRID);
    haveMetric     	= (uvar->metricType > LAL_PMETRIC_NONE);
    useMetric     	= (uvar->gridType == GRID_METRIC);

    if ( !useFullGridFile && !useSkyGridFile && haveGridFile )
      {
        XLALPrintError ("\nERROR: gridFile was specified but not needed for gridType=%d\n\n", uvar->gridType );
        XLAL_ERROR ( XLAL_EINVAL );
      }
    if ( useSkyGridFile && !haveGridFile )
      {
        XLALPrintError ("\nERROR: gridType=SKY-FILE, but no --gridFile specified!\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }
    if ( useFullGridFile && !haveGridFile )
      {
	XLALPrintError ("\nERROR: gridType=GRID-FILE, but no --gridFile specified!\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }

    if ( (haveAlphaBand && !haveDeltaBand) || (haveDeltaBand && !haveAlphaBand) )
      {
	XLALPrintError ("\nERROR: Need either BOTH (AlphaBand, DeltaBand) or NONE.\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }

    if ( haveSkyRegion && haveAlphaDelta )
      {
        XLALPrintError ("\nOverdetermined sky-region: only use EITHER (Alpha,Delta) OR skyRegion!\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }
    if ( !haveSkyRegion && !haveAlphaDelta && !useSkyGridFile && !useFullGridFile )
      {
        XLALPrintError ("\nUnderdetermined sky-region: use one of (Alpha,Delta), (RA,Dec), skyRegion or a gridFile!\n\n");
        XLAL_ERROR ( XLAL_EINVAL );

      }

    if ( !useMetric && haveMetric)
      {
        XLALPrintWarning ("\nWARNING: Metric was specified for non-metric grid... will be ignored!\n");
      }
    if ( useMetric && !haveMetric)
      {
        XLALPrintError ("\nERROR: metric grid-type selected, but no metricType selected\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }

    /* Specific checks for --gridType=GRID_SPINDOWN_{SQUARE,AGEBRK} parameter spaces */
    if (uvar->gridType == GRID_SPINDOWN_SQUARE || uvar->gridType == GRID_SPINDOWN_AGEBRK) {

      /* Check that no third spindown range were given */
      if (uvar->f3dot != 0.0 || uvar->f3dotBand != 0.0) {
        XLALPrintError ("\nERROR: f3dot and f3dotBand cannot be used with gridType={8,9}\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }

      /* Check that no grid spacings were given */
      if (uvar->df1dot != 0.0 || uvar->df2dot != 0.0 || uvar->df3dot != 0.0) {
        XLALPrintError ("\nERROR: df{1,2,3}dot cannot be used with gridType={8,9}\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }

    }

    /* Specific checks for --gridType=GRID_SPINDOWN_AGEBRK parameter space */
    if (uvar->gridType == GRID_SPINDOWN_AGEBRK) {

      /* Check age and braking indices */
      if (uvar->spindownAge <= 0.0) {
        XLALPrintError ("\nERROR: spindownAge must be strictly positive with gridType=9\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }
      if (uvar->minBraking <= 0.0) {
        XLALPrintError ("\nERROR: minBraking must be strictly positive with gridType=9\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }
      if (uvar->maxBraking <= 0.0) {
        XLALPrintError ("\nERROR: minBraking must be strictly positive with gridType=9\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }
      if (uvar->minBraking >= uvar->maxBraking) {
        XLALPrintError ("\nERROR: minBraking must be strictly less than maxBraking with gridType=9\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }

      /* Check that no first and second spindown ranges were given */
      if (uvar->f1dot != 0.0 || uvar->f1dotBand != 0.0) {
        XLALPrintError ("\nERROR: f1dot and f1dotBand cannot be used with gridType=9\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }
      if (uvar->f2dot != 0.0 || uvar->f2dotBand != 0.0) {
        XLALPrintError ("\nERROR: f2dot and f2dotBand cannot be used with gridType=9\n\n");
        XLAL_ERROR ( XLAL_EINVAL );
      }

    }

  } /* Grid-related checks */

  /* check NumCandidatesToKeep and FracCandidatesToKeep */
  if (XLALUserVarWasSet(&uvar->NumCandidatesToKeep) && XLALUserVarWasSet(&uvar->FracCandidatesToKeep)) {
    XLALPrintError ("\nERROR: NumCandidatesToKeep and FracCandidatesToKeep are mutually exclusive\n\n");
    XLAL_ERROR ( XLAL_EINVAL );
  }
  if (XLALUserVarWasSet(&uvar->FracCandidatesToKeep) && (uvar->FracCandidatesToKeep <= 0.0 || 1.0 < uvar->FracCandidatesToKeep)) {
    XLALPrintError ("\nERROR: FracCandidatesToKeep must be greater than 0.0 and less than or equal to 1.0\n\n");
    XLAL_ERROR ( XLAL_EINVAL );
  }

  return XLAL_SUCCESS;

} /* checkUserInputConsistency() */

/* debug-output a(t) and b(t) into given file.
 * return 0 = OK, -1 on error
 */
int
outputBeamTS( const CHAR *fname, const AMCoeffs *amcoe, const DetectorStateSeries *detStates )
{
  FILE *fp;
  UINT4 i, len;

  if ( !fname || !amcoe || !amcoe->a || !amcoe->b || !detStates)
    return -1;

  len = amcoe->a->length;
  if ( (len != amcoe->b->length) || ( len != detStates->length ) )
    return -1;

  if ( (fp = fopen(fname, "wb")) == NULL )
    return -1;

  for (i=0; i < len; i ++ )
    {
      INT4 ret;
      ret = fprintf (fp, "%9d %f %f %f \n",
		     detStates->data[i].tGPS.gpsSeconds, detStates->data[i].LMST, amcoe->a->data[i], amcoe->b->data[i] );
      if ( ret < 0 )
	{
	  fprintf (fp, "ERROR\n");
	  fclose(fp);
	  return -1;
	}
    }

  fclose(fp);
  return 0;
} /* outputBeamTS() */

/**
 * write full 'PulsarCandidate' (i.e. Doppler params + Amplitude params + error-bars + Fa,Fb, F, + A,B,C,D
 * RETURN 0 = OK, -1 = ERROR
 */
int
write_PulsarCandidate_to_fp ( FILE *fp,  const PulsarCandidate *pulsarParams, const FstatCandidate *Fcand )
{
  if ( !fp || !pulsarParams || !Fcand  )
    return -1;

  fprintf (fp, "\n");
  char bufGPS[32];
  fprintf (fp, "refTime  = %s;\n", XLALGPSToStr ( bufGPS, &pulsarParams->Doppler.refTime ) );
  fprintf (fp, "\n");

  /* Amplitude parameters with error-estimates */
  fprintf (fp, "h0       = % .6g;\n", pulsarParams->Amp.h0 );
  fprintf (fp, "dh0      = % .6g;\n", pulsarParams->dAmp.h0 );
  fprintf (fp, "cosi     = % .6g;\n", pulsarParams->Amp.cosi );
  fprintf (fp, "dcosi    = % .6g;\n", pulsarParams->dAmp.cosi );
  fprintf (fp, "phi0     = % .6g;\n", pulsarParams->Amp.phi0 );
  fprintf (fp, "dphi0    = % .6g;\n", pulsarParams->dAmp.phi0 );
  fprintf (fp, "psi      = % .6g;\n", pulsarParams->Amp.psi );
  fprintf (fp, "dpsi     = % .6g;\n", pulsarParams->dAmp.psi );

  fprintf (fp, "\n");

  /* Doppler parameters */
  fprintf (fp, "Alpha    = % .16g;\n", pulsarParams->Doppler.Alpha );
  fprintf (fp, "Delta    = % .16g;\n", pulsarParams->Doppler.Delta );
  fprintf (fp, "Freq     = % .16g;\n", pulsarParams->Doppler.fkdot[0] );
  fprintf (fp, "f1dot    = % .16g;\n", pulsarParams->Doppler.fkdot[1] );
  fprintf (fp, "f2dot    = % .16g;\n", pulsarParams->Doppler.fkdot[2] );
  fprintf (fp, "f3dot    = % .16g;\n", pulsarParams->Doppler.fkdot[3] );

  fprintf (fp, "\n");

  /* Binary parameters */
  if (pulsarParams->Doppler.asini > 0)
    {
      fprintf (fp, "orbitPeriod       = % .16g;\n", pulsarParams->Doppler.period );
      fprintf (fp, "orbitasini        = % .16g;\n", pulsarParams->Doppler.asini );
      fprintf (fp, "orbitTp           = %s;\n", XLALGPSToStr ( bufGPS, &(pulsarParams->Doppler.tp) ));
      fprintf (fp, "orbitArgp         = % .16g;\n", pulsarParams->Doppler.argp );
      fprintf (fp, "orbitEcc          = % .16g;\n", pulsarParams->Doppler.ecc );
    }

  /* Amplitude Modulation Coefficients */
  fprintf (fp, "Ad       = % .6g;\n", Fcand->Mmunu.Ad );
  fprintf (fp, "Bd       = % .6g;\n", Fcand->Mmunu.Bd );
  fprintf (fp, "Cd       = % .6g;\n", Fcand->Mmunu.Cd );
  fprintf (fp, "Ed       = % .6g;\n", Fcand->Mmunu.Ed );
  fprintf (fp, "Sinv_Tsft= % .6g;\n", Fcand->Mmunu.Sinv_Tsft );
  fprintf (fp, "\n");

  /* F-stat-values */
  fprintf (fp, "Fa       = % .6g  %+.6gi;\n", creal(Fcand->Fa), cimag(Fcand->Fa) );
  fprintf (fp, "Fb       = % .6g  %+.6gi;\n", creal(Fcand->Fb), cimag(Fcand->Fb) );
  fprintf (fp, "twoF     = % .6g;\n", Fcand->twoF );
  /* single-IFO F-stat values, if present */
  UINT4 X, numDet = Fcand->numDetectors;
  for ( X = 0; X < numDet ; X ++ )
    fprintf (fp, "twoF%d    = % .6g;\n", X, Fcand->twoFX[X] );
  /* BSGL */
  if ( !XLALIsREAL4FailNaN(Fcand->log10BSGL) ) /* if --computeBSGL=FALSE, the log10BSGL field was initialised to NAN - do not output it */
    fprintf (fp, "log10BSGL = % .6g;\n", Fcand->log10BSGL );

  fprintf (fp, "\nAmpFisher = \\\n" );
  XLALfprintfGSLmatrix ( fp, "%.9g",pulsarParams->AmpFisherMatrix );

  return 0;

} /* write_PulsarCandidate_to_fp() */

/** comparison function for our candidates toplist */
int
compareFstatCandidates ( const void *candA, const void *candB )
{
  REAL8 twoF1 = ((const FstatCandidate *)candA)->twoF;
  REAL8 twoF2 = ((const FstatCandidate *)candB)->twoF;
  if ( twoF1 < twoF2 )
    return 1;
  else if ( twoF1 > twoF2 )
    return -1;
  else
    return 0;

} /* compareFstatCandidates() */

/** comparison function for our candidates toplist with alternate sorting statistic BSGL=log10BSGL */
int
compareFstatCandidates_BSGL ( const void *candA, const void *candB )
{
  REAL8 BSGL1 = ((const FstatCandidate *)candA)->log10BSGL;
  REAL8 BSGL2 = ((const FstatCandidate *)candB)->log10BSGL;
  if ( BSGL1 < BSGL2 )
    return 1;
  else if ( BSGL1 > BSGL2 )
    return -1;
  else
    return 0;

} /* compareFstatCandidates_BSGL() */

/**
 * write one 'FstatCandidate' (i.e. only Doppler-params + Fstat) into file 'fp'.
 * Return: 0 = OK, -1 = ERROR
 */
int
write_FstatCandidate_to_fp ( FILE *fp, const FstatCandidate *thisFCand )
{

  if ( !fp || !thisFCand )
    return -1;

  /* add extra output-field containing per-detector FX if non-NULL */
  char extraStatsStr[256] = "";     /* defaults to empty */
  char buf0[256];
  /* BSGL */
  if ( !XLALIsREAL4FailNaN(thisFCand->log10BSGL) ) { /* if --computeBSGL=FALSE, the log10BSGL field was initialised to NAN - do not output it */
      snprintf ( extraStatsStr, sizeof(extraStatsStr), " %.9g", thisFCand->log10BSGL );
  }
  if ( thisFCand->numDetectors > 0 )
    {
      for ( UINT4 X = 0; X < thisFCand->numDetectors; X ++ )
        {
          snprintf ( buf0, sizeof(buf0), " %.9g", thisFCand->twoFX[X] );
          UINT4 len1 = strlen ( extraStatsStr ) + strlen ( buf0 ) + 1;
          if ( len1 > sizeof ( extraStatsStr ) ) {
            XLAL_ERROR ( XLAL_EINVAL, "assembled output string too long! (%d > %zu)\n", len1, sizeof(extraStatsStr ));
            break;      /* we can't really terminate with error in this function, but at least we avoid crashing */
          }
          strcat ( extraStatsStr, buf0 );
        } /* for X < numDet */
    } /* if FX */

  fprintf (fp, "%.16g %.16g %.16g %.16g %.16g %.16g %.9g%s\n",
	   thisFCand->doppler.fkdot[0], thisFCand->doppler.Alpha, thisFCand->doppler.Delta,
	   thisFCand->doppler.fkdot[1], thisFCand->doppler.fkdot[2], thisFCand->doppler.fkdot[3],
	   thisFCand->twoF, extraStatsStr );

  return 0;

} /* write_FstatCandidate_to_fp */

/* --------------------------------------------------------------------------------
 * Scanline window functions
 * FIXME: should go into a separate file once implementation is settled down ...
 *
 * --------------------------------------------------------------------------------*/

/**
 * Create a scanline window, with given windowWings >= 0.
 * Note: the actual window-size is 1 + 2 * windowWings
 */
scanlineWindow_t *
XLALCreateScanlineWindow ( UINT4 windowWings ) /**< number of neighbors on each side in scanlineWindow */
{
  scanlineWindow_t *ret = NULL;
  UINT4 windowLen = 1 + 2 * windowWings;

  XLAL_CHECK_NULL ( ( ret = LALCalloc ( 1, sizeof(*ret)) ) != NULL, XLAL_ENOMEM );
  ret->length = windowLen;

  XLAL_CHECK_NULL ( (ret->window = LALCalloc ( windowLen, sizeof( ret->window[0] ) )) != NULL, XLAL_ENOMEM );

  ret->center = &(ret->window[ windowWings ]);	/* points to central bin */

  return ret;

} /* XLALCreateScanlineWindow() */

void
XLALDestroyScanlineWindow ( scanlineWindow_t *scanlineWindow )
{
  if ( !scanlineWindow )
    return;

  if ( scanlineWindow->window )
    XLALFree ( scanlineWindow->window );

  XLALFree ( scanlineWindow );

  return;

} /* XLALDestroyScanlineWindow() */

/**
 * Advance by pushing a new candidate into the scanline-window
 */
int
XLALAdvanceScanlineWindow ( const FstatCandidate *nextCand, scanlineWindow_t *scanWindow )
{
  UINT4 i;

  if ( !nextCand || !scanWindow || !scanWindow->window ) {
    XLAL_ERROR ( XLAL_EINVAL );
  }

  for ( i=1; i < scanWindow->length; i ++ )
    scanWindow->window[i - 1] = scanWindow->window[i];

  scanWindow->window[ scanWindow->length - 1 ] = *nextCand;	/* struct-copy */

  return XLAL_SUCCESS;

} /* XLALAdvanceScanlineWindow() */

/**
 * check wether central candidate in Scanline-window is a local maximum
 */
BOOLEAN
XLALCenterIsLocalMax ( const scanlineWindow_t *scanWindow, const UINT4 rankingStatistic )
{

  if ( !scanWindow || !scanWindow->center )
    return FALSE;

  if ( rankingStatistic == RANKBY_2F ) /* F-statistic */
    {

      REAL8 twoF0 = scanWindow->center->twoF;

      for ( UINT4 i=0; i < scanWindow->length; i ++ )
        if ( scanWindow->window[i].twoF > twoF0 )
         return FALSE;

    }

  else if ( rankingStatistic == RANKBY_BSGL ) /* line-robust statistic log10BSGL */
    {

      REAL8 BSGL0 = scanWindow->center->log10BSGL;

      for ( UINT4 i=0; i < scanWindow->length; i ++ )
        if ( scanWindow->window[i].log10BSGL > BSGL0 )
         return FALSE;

    }
  else
    {
      XLALPrintError ("Unsupported ranking statistic '%d' ! Supported: 'F=0' and 'BSGL=2'.\n", rankingStatistic );
      return FALSE;
    }

  return TRUE;

} /* XLALCenterIsLocalMax() */

/**
 * Function to append one timing-info line to output file.
 *
 */
int
write_TimingInfo ( const CHAR *fname, const timingInfo_t *ti, const ConfigVariables *cfg )
{
  XLAL_CHECK ( (fname != NULL) && (ti != NULL), XLAL_EINVAL );

  FILE *fp;
  if ( (fp = fopen(fname,"rb" )) != NULL )
    {
      fclose(fp);
      XLAL_CHECK ( (fp = fopen( fname, "ab" ) ), XLAL_ESYS, "Failed to open existing timing-file '%s' for appending\n", fname );
    }
  else
    {
      XLAL_CHECK ( (fp = fopen( fname, "wb" ) ), XLAL_ESYS, "Failed to open new timing-file '%s' for writing\n", fname );
      fprintf ( fp, "%2s%6s %10s %10s %10s %10s %10s\n",
                "%%", "NSFTs", "NFreq", "tauF[s]", "tauFEff[s]", "tauF0[s]", "FstatMethod" );
    }

  fprintf ( fp, "%8d %10d %10.1e %10.1e %10.1e %10s\n",
            ti->NSFTs, ti->NFreq, ti->tauFstat, ti->tauTemplate, ti->tauF0, XLALGetFstatInputMethodName(cfg->Fstat_in) );

  fclose ( fp );
  return XLAL_SUCCESS;

} /* write_TimingInfo() */

/* Resize histogram */
gsl_vector_int *resize_histogram(gsl_vector_int *old_hist, size_t size) {
  gsl_vector_int *new_hist = gsl_vector_int_alloc(size);
  XLAL_CHECK_NULL(new_hist != NULL, XLAL_ENOMEM);
  gsl_vector_int_set_zero(new_hist);
  if (old_hist != NULL) {
    gsl_vector_int_view old = gsl_vector_int_subvector(old_hist, 0, GSL_MIN(old_hist->size, size));
    gsl_vector_int_view new = gsl_vector_int_subvector(new_hist, 0, GSL_MIN(old_hist->size, size));
    gsl_vector_int_memcpy(&new.vector, &old.vector);
    gsl_vector_int_free(old_hist);
  }
  return new_hist;
}
