# Copyright (C) 2011  Nickolas Fotopoulos, Stephen Privitera
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

from optparse import OptionParser
import sys

import numpy as np
from h5py import File as H5File

## get arguments
def parse_command_line():
    parser = OptionParser(description="Merge banksim .h5 files together, maximizing match for each injection over bank fragments as necessary. We assume that all banksims were given the identical HL-INJECTIONS file and disjoint bank fragments.")
    parser.add_option("-o", "--output", help="Write output to hdf5 output")

    opts, args = parser.parse_args()

    return opts, args

opts, args = parse_command_line()

if not args: sys.exit("Nothing to do")

# initialize output file
outfile = H5File(opts.output, "w")

# based on the first file...
with H5File(args[0], "r") as infile:
    # populate sims completely
    out_sims = outfile.create_dataset("/sim_inspiral", data=infile["sim_inspiral"], compression='gzip', compression_opts=1)
    outfile.flush()

    # copy process and process params table
    # FIXME: only takes metadata from first file!
    outfile.create_dataset("/process", data=infile["/process"])
    outfile.create_dataset("/process_params", data=infile["/process_params"])

    # but we'll have to build up sngls and the match map as we go
    out_map = np.zeros(shape=infile["/match_map"].shape,
                       dtype=infile["/match_map"].dtype)
    out_sngls_dtype = infile["/sngl_inspiral"].dtype
out_sngls = []

# build sngls and match map
for f in args[:]:
    with H5File(f, "r") as infile:
        # sanity check that we're probably using the same injection set
        assert len(infile["/sim_inspiral"]) == len(out_sims)

        # sanity check that the sim has one match_map entry per inj
        assert len(infile["/match_map"]) == len(out_sims)

        # tabulate best match per injection
        # NB: inj_ids have the same meaning across files, but best_tmplt_ids
        # require remapping to the output order.
        for inj_id2, (inj_id, inj_sigmasq, match, best_tmplt_id) in enumerate(infile["/match_map"]):
            assert inj_id == inj_id2  # we assume that the ids are in order
            if match > out_map[inj_id]["match"]:
                out_map[inj_id] = (inj_id, inj_sigmasq, match, best_tmplt_id + len(out_sngls))

        # and copy in the templates represented here
        out_sngls.extend(infile["/sngl_inspiral"])

# write to file
outfile.create_dataset("/sngl_inspiral", data=np.array(out_sngls, dtype=out_sngls_dtype), compression='gzip', compression_opts=1)
outfile.create_dataset("/match_map", data=out_map, compression='gzip', compression_opts=1)
outfile.close()
