/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#include <algorithm>
#include <iostream>
#include <sstream>

#include "xmldao.hpp"
#include "kitparser.hpp"

using namespace std;


/**
 * \brief Loads the data model from the previously set filename.
 *
 * The data model holds a rich graph of objects, representing the
 * entire list of categories and items.
 *
 * The filename must be set before calling this method by calling
 * XmlDao::set_filename(), otherwise an empty model is returned.
 *
 * \retval Returns a newly created data model.  The caller is
 * responsible for destroying the model.
 * \see KitModel
 */
KitModel* XmlDao::get_model() {
    KitModel* retval = new KitModel();
    try {
        if (m_filename.size() > 0) {
            KitParser parser(*retval);
            parser.set_validate();
            parser.set_substitute_entities();
            parser.parse_file(m_filename);
            // Clear the dirty, deleted and new flags etc.
        }
        ItemContainer* items = retval->get_all_items();
        ItemIter i = max_element(items->begin(), items->end(), ItemCompareId());
        if (i != items->end()) {
            Item* max_item = (*i);
            m_max_item_id = max_item->get_id();
        } else {
            m_max_item_id = -1;
        }
        delete items;
        CategoryContainer* categories = retval->get_categories();
        CategoryIter cat = std::max_element(categories->begin(), categories->end(), CategoryCompareId());
        if (cat != categories->end()) {
            Category* max_category = (*cat);
            m_max_category_id = max_category->get_id();
        } else {
            m_max_category_id = -1;
        }
        delete categories;
        retval->reset();
    } catch (const xmlpp::exception& ex) {
        std::cout << "libxml++ exception: " << ex.what() << std::endl;
    }
    return retval;
}


/**
 * \brief Adds the passed item to the current items' node.
 *
 * \see XmlDao::save_model(KitModel&)
 */
bool XmlDao::add_item_to_dom(ModelItem& item) {
    assert(m_items_node != NULL);
    if (!item.is_deleted()) {
        xmlpp::Element* itemNode = m_items_node->add_child("item");
        ostringstream os;
        os << item.get_id();
        itemNode->set_attribute("id", os.str());
        itemNode->set_attribute("checked", item.get_checked() ? "true" : "false");
        itemNode->set_child_text(item.get_description());
    }
    return false;
}


/**
 * \brief Adds the passed item to the current category's node.
 *
 * \see XmlDao::save_model(KitModel&)
 */
bool XmlDao::add_category_item_to_dom(Item& item) {
    assert(m_cat_items_node != NULL);
    if (!((ModelItem&) item).is_deleted()) {
        xmlpp::Element* itemNode = m_cat_items_node->add_child("category-item");
        ostringstream os;
        os << item.get_id();
        itemNode->set_attribute("id", os.str());
    }
    return false;
}


/**
 * \brief Adds the passed item to the current categories' node.
 *
 * \see XmlDao::save_model(KitModel&)
 */
bool XmlDao::add_category_to_dom(ModelCategory& category) {
    assert(m_categories_node != NULL);
    if (!category.is_deleted()) {
        xmlpp::Element* catNode = m_categories_node->add_child("category");
        ostringstream os;
        os << category.get_id();
        catNode->set_attribute("id", os.str());
        xmlpp::Element* catNameNode = catNode->add_child("category-name");
        catNameNode->set_child_text(category.get_name());
        m_cat_items_node = catNode->add_child("category-items");
        category.foreach_item( sigc::mem_fun(*this, &XmlDao::add_category_item_to_dom) );
    }
    return false;
}


/**
 * \brief Returns the next unused unique id for items.
 */
long XmlDao::get_next_item_id() {
    return ++m_max_item_id;
}


/**
 * Returns the next unused unique id for categories.
 */
long XmlDao::get_next_category_id() {
    return ++m_max_category_id;
}


/**
 * \brief Saves the model as an XML document.
 *
 * The filename must be set before calling this method by calling
 * XmlDao::set_filename();
 *
 * \param model The model to save.
 */
void XmlDao::save_model(KitModel* model) {
    xmlpp::Document document;
    xmlpp::Element* rootNode = document.create_root_node("kitlist");
    m_items_node = rootNode->add_child("items");
    model->foreach_item( sigc::mem_fun(*this, &XmlDao::add_item_to_dom) );
    m_categories_node = rootNode->add_child("categories");
    model->foreach_category( sigc::mem_fun(*this, &XmlDao::add_category_to_dom) );
    document.write_to_file_formatted(m_filename);
#ifdef XML_DAO
    // Remove deleted items from categories etc.
    model->purge();
    // Clear the dirty, deleted and new flags etc.
    model->reset();
#endif
}
