/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef KIT_LIST_PGSQL_DAO_H
#define KIT_LIST_PGSQL_DAO_H 1

#include <config.h>
#ifdef HAVE_LIBPQXX

#include <pqxx/pqxx>
#include "kitlistdao.hpp"
#include "kitmodel.hpp"

/**
 * \brief Implementation of a KitListDao for using a PostgreSQL
 * database as the persistence store.
 */
class KitListPgsqlDao : public KitListDao {
    pqxx::connection* m_db_connection;
    void connect();
    std::string generate_uuid();
protected:
    std::string m_dbname; ///< The name of the database.
    std::string m_user;  ///< The database user name to connect with.
    std::string m_port;  ///< The database port to connect to.
public :
    KitListPgsqlDao(int verbose = 0);

    /**
     * \brief Constructor specifying some Postgresql database
     * connection parameters.
     *
     * Note that the PostgreSQL libraries will use default values for
     * these parameters if they are not specified, including examining
     * environment variables.  See the PostgreSQL documentation for
     * full details.
     *
     * \param dbname The name of the database.
     * \param verbose Optional parameter indicating whether to include
     * verbose output to STDOUT.  Defaults to false.
     */
    KitListPgsqlDao(const std::string dbname, int verbose = 0);

    /**
     * \brief Constructor specifying Postgresql database connection
     * parameters.
     *
     * Note that the PostgreSQL libraries will use default values for
     * these parameters if they are not specified, including examining
     * environment variables.  See the PostgreSQL documentation for
     * full details.
     *
     * \param dbname The name of the database.
     * \param user The database user name to connect with.
     * \param port The database port to connect to.
     * \param verbose Optional parameter indicating whether to include
     * verbose output to STDOUT.  Defaults to false.
     */
    KitListPgsqlDao(const std::string dbname, const std::string user, const std::string port, int verbose = 0);

    ~KitListPgsqlDao();

    KitModel* get_model();

    void save_model(KitModel* model);

    Category* get_category(long cat_id, item_choice choice);

    ItemContainer* get_all_items(item_choice choice);

    long add_item(const std::string name);

    long add_item(const std::string name, long cat_id);

    void append_items_to_category(long to_cat_id, long from_cat_id, item_choice choice);

    void associate_item_with_category(long id, long cat_id);

    CategoryContainer get_categories();

    long new_category(const std::string name);

    void delete_item(long id);

    void update_item_checked_state(ItemContainer& items);

    void remove_item_from_category(long id, long cat_id);

    long get_next_item_id();

    long get_next_category_id();

    void delete_category(long id);

    void set_item_flag(long id);

    void unset_item_flag(long id);

    void set_category_flag(long id);

    void unset_category_flag(long id);

    void set_all_flags();

    void unset_all_flags();

};
#endif //HAVE_LIBPQXX
#endif //KIT_LIST_PGSQL_DAO_H
