/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef CATEGORY_H
#define CATEGORY_H 1
#include <iostream>
#include <vector>
#include "item.hpp"


class CategoryCompareName;

/**
 * \brief Represents a Category.
 *
 * Many categories may have one or more items.
 */
class Category {
 protected:
    long m_id; ///< Unique id
    std::string m_name; ///< The category name
    ItemContainer m_items; ///< List of associated items
 public:
    ~Category() {}
    void set_id(long id) { m_id = id; }
    long get_id() { return m_id; }
    void set_name (const std::string name) { m_name = name; }
    std::string get_name() { return m_name; }
    virtual void add_item(Item* item);
    virtual void remove_item(Item* item);
    /// Returns the number of items associated with this category.
    virtual size_t item_count() { return m_items.size(); }
    /// Returns true if there are any items associated with this category.
    virtual bool has_items() { return !m_items.empty(); }
    void foreach_item(const SlotForeachItem& slot);
    void execute(ItemFunctor& functor);
    friend class CategoryCompareName;
    friend class CategoryCompareId;
    friend class KitModel;
};


/**
 * \brief Comparator used for sorting Categories by name.
 * \see Category
 */
class CategoryCompareName {
 public:
    CategoryCompareName() {}
    int operator()(Category* c1, Category* c2) { return (c1->m_name < c2->m_name); }
};


/**
 * \brief Comparator used for comparing Categories by id.
 * \see Category
 */
class CategoryCompareId {
 public:
    CategoryCompareId() {}
    int operator()(Category* c1, Category* c2) { return (c1->m_id < c2->m_id); }
};


typedef std::vector<Category*> CategoryContainer;
typedef CategoryContainer::iterator CategoryIter;

#endif // CATEGORY_H
