open Syntax
open Metadata
open Util
open Errors


(* ------------------------------------------------------------
 * Errors
 * ------------------------------------------------------------ *)

let mfatal m s = fatal m.span s
let mwarning m s = warning m.span s
let pfatal m pretty = mfatal m (Prettyutil.pretty_to_string pretty)
let pwarning m pretty = mwarning m (Prettyutil.pretty_to_string pretty)
let pmsg pretty = print_endline (Prettyutil.pretty_to_string pretty)

let error_noprefix span prefix s =
	warning span ("variable name "^s^" should have prefix "^prefix)
let error_nosuffix span suffix s =
	warning span ("variable name "^s^" should have suffix "^suffix)
let fatal_expected_fundef span = 
	fatal span "expected function definition"
let fatal_simpledecl span = 
	fatal span "cannot declare multiple vars here" 


(* ------------------------------------------------------------
 * Ids
 * ------------------------------------------------------------ *)

let strid s = (nometa (),s)	
let strvar s = nometa (), Var (strid s)

let id_str = snd
let id_meta = fst
let id_span id = (id_meta id).span
let noid = (nometa (), "")

let id_add_suffix (m,id_str) suffix = (m,id_str ^ suffix)
let id_add_prefix (m,id_str as id) prefix = 
	if prefix = "" then id else m,prefix ^ "_" ^ id_str
let add_prefix prefix str =
	if prefix = "" then str else prefix ^ "_" ^ str
let id_strip_prefix prefix (m,s) =
	if not (is_prefix prefix s) then
  		(error_noprefix m.span prefix s; m,s)
	else
		m,strip_prefix prefix s
let id_strip_suffix suffix (m,s) =
	if not (is_suffix suffix s) then
		(error_nosuffix m.span suffix s; m,s)
	else
		m,strip_suffix suffix s
let id_is_prefix prefix (m,s) = is_prefix prefix s		
		

let tyiface_id = fst

let tywild id = [SCoreHere],TyWild id,[]
let tyvoid = [SCoreHere],TyVoid,[]
let tybool = [SInt;SCoreHere],TyBasic,[]
let tyint = [SInt;SCoreHere],TyBasic,[]

let id_equal x y = (id_str x = id_str y) 


(* ------------------------------------------------------------
 * Find the free vars in a type
 * ------------------------------------------------------------ *)

let rec coretyp_freevars coretyp = match coretyp with
	| TyWild id -> [id]
	| _ -> []
	
and declmod_freevars declmod = match declmod with
	| DFun (ArgsFull (_,_,args),va) -> concat_map decl_freevars args
	| DWithArgs args -> concat_map typ_freevars args
	| _ -> []	
	
and init_decl_freevars ((decltyp,id_opt),init) = 
	concat_map declmod_freevars decltyp
	
and decl_freevars (m,((specs,coretyp),inits)) = 
	let initfvs = concat_map init_decl_freevars inits in
	coretyp_freevars coretyp @ initfvs

and typ_freevars (specs,coretyp,declmods) =
	coretyp_freevars coretyp @ (concat_map declmod_freevars declmods)


(* ------------------------------------------------------------
 * Types
 * ------------------------------------------------------------ *)

let coretyp_name_nocheck coretyp = match coretyp with
	| TyStruct (kind,Some id,None) -> id
	| _ -> intfatal "Malformed dictionary"

let spec_name_nocheck spec = match spec with
	| SInt -> "int"
	| SChar -> "char"
	| SLong -> "long"
	| SFloat -> "float"
	| SDouble -> "double"
	| _ -> intfatal "Malformed dictionary"

let typ_name_nocheck typ = match typ with
	| [SCoreHere],coretyp,[] -> coretyp_name_nocheck coretyp
	| [SCoreHere],coretyp,[DWithArgs _] -> coretyp_name_nocheck coretyp
	| [spec],TyBasic,[] -> strid (spec_name_nocheck spec)
	| [SLong;SLong],TyBasic,[] -> strid "longlong"
	| _ -> intfatal "Malformed dictionary"

let typ_args_nocheck (specs,core,declmods) = match declmods with
	| DWithArgs args::_ -> args
	| _ -> []
	
let typ_mods (specs,core,declmods) = declmods
let typ_core (specs,core,declmods) = core

let is_tagged_typ typ = match typ with
	| [SCoreHere],TyStruct(SKTagged,_,_),[] -> true
	| [SCoreHere],TyStruct(SKTagged,_,_),[DWithArgs _] -> true
	| _ -> false
	
let is_closure_typ typ = match typ with
	| _, _, DFun (_,Closure)::_ -> true
	| _ -> false	
	
let is_function_typ typ = match typ with
	| _, _, DFun _::_ -> true
	| _ -> false

let is_wild_typ typ = match typ with
	| _, TyWild _, [] -> true
	| _ -> false
	
let is_basic_typ typ = match typ with
	| _, TyBasic, _ -> true
	| _ -> false

let coretyp_equal corety1 corety2 = match corety1,corety2 with
	| TyVoid, _ | _, TyVoid -> true
	| TyBasic, TyBasic -> true
	| TyWild id1, TyWild id2 -> id_equal id1 id2 
	| TyName id1, TyName id2 -> id_equal id1 id2 
	| TyStruct (kind1,id1,details1),TyStruct (kind2,id2,details2) ->
		kind1 = kind2 &&
		opt_equal id_equal id1 id2 &&
		opt_equal (fun x y -> false) details1 details2
	| TyEnum (id1,details1), TyEnum (id2,details2) ->
		opt_equal id_equal id1 id2 &&
		opt_equal (fun x y -> false) details1 details2
	| TyEnum _, TyBasic -> true
	| TyBasic, TyEnum _ -> true
	| _ -> false
	

(* ------------------------------------------------------------
 * Modify a type
 * ------------------------------------------------------------ *)

let as_dictmethod ifaceid tyname tyiface (m,fundecl) =
	let dictenv = if tyiface = [] then None else Some (ifaceid,tyname) in
	match fundecl with
	| basetyp, [(DFun(args,kind)::mods,id),init] ->
		m,(basetyp, [(DFun (args,DictMethod dictenv)::mods,id),init]) 
	| _ -> fatal_expected_fundef m.span 	

(* add extra type specifiers to a type *)
let rec insert_specs specs newspecs = match specs with
	| SCoreHere::xs -> newspecs @ xs
	| x::xs -> x :: insert_specs xs newspecs
	| [] -> []

(* ------------------------------------------------------------
 * Strip type arguments from a type, for more minimal C printing
 * ------------------------------------------------------------ *)

let strip_tyargs = List.filter (function DWithArgs _ -> false | _ -> true)
	
let convert_funptr mods = match mods with
	| DFun _::rest -> DPtr [] :: DParens :: mods
	| _ -> mods
	
let strip_typ_extras (core,base,mods) = 
	core,base,convert_funptr (strip_tyargs mods)
	

(* ------------------------------------------------------------
 * Declarations
 * ------------------------------------------------------------ *)

let split_fundecl (m,decl : declaration) = match decl with
	| (specs,coretyp),[((DFun (args,varargs)::declty,Some id),None)] ->
		id,(specs,coretyp,declty),args,varargs
	| _ -> intfatal "Function declaration is not a function"


let init_names ((declty,id_opt),init) = list_of_option id_opt
let decl_names (m,(basetyp,init_decs)) = concat_map init_names init_decs

let is_simpledecl (m,decl) = match decl with
	| _,[(_,Some _),_] -> true
	| _ -> false	

let simpledecl_typ (m,decl) = match decl with
	| (specs,coretyp),[(mods,_),_] ->
		(specs,coretyp,mods)
	| _ -> fatal_simpledecl m.span

let simpledecl_name (m,decl) = match decl with
	| _,[(_,Some id),_] -> id
	| _ -> fatal_simpledecl m.span	
	
let simpledecl_init (m,decl) = match decl with
	| _,[_,init] -> init
	| _ -> fatal_simpledecl m.span	

	
let rec declmods_strip_parens declmods = match declmods with
	| DParens :: xs -> declmods_strip_parens xs
	| x::xs -> x::(declmods_strip_parens xs)
	| [] -> []
	
let init_ds_strip_parens ((declmods,id_opt),init_opt) =
	(declmods_strip_parens declmods,id_opt),init_opt	
	
let decl_strip_parens (basetyp,init_ds) = 
	basetyp, List.map init_ds_strip_parens init_ds
	
(* ------------------------------------------------------------
 * Builtin operators 
 * ------------------------------------------------------------ *)

let fundef_decl (decl,krdecls,block : fundef) = decl
	
let numberiface = (strid "Number",[])
let tyvar = TyWild (strid "A")

let funsig_onearg op iface = (strid op,tyvar,[strid "A",[strid iface,[]]],
							[tyvar,strid "x"])
let funsig_twoarg op iface = (strid op,tyvar,[strid "A",[strid iface,[]]],
							[tyvar,strid "x";tyvar,strid "y"])
							
let builtin_twoint_ops = ["+";"-";"*";"/";"&";"|";"^";">>";"<<"]
let builtin_intcmp_ops = ["<";"<=";">";">=";"==";"!="]
let builtin_twobool_ops = ["&&";"||"]
let builtin_onebool_ops = ["!"]
let builtin_oneint_ops = ["++";"--";"~";"-"]
let builtin_ptrcmp_ops = ["==";"!="]
let builtin_prefix_ops = ["*";"&";"--";"++";"~";"-"]
let builtin_postfix_ops = ["--P";"++P"]

let builtin_ops = 	
			builtin_twoint_ops @ builtin_intcmp_ops @ 
			builtin_twobool_ops @ builtin_onebool_ops @ 
			builtin_oneint_ops @ builtin_postfix_ops

let trim_postfix_op opname = String.sub opname 0 (String.length opname - 1)


(* ------------------------------------------------------------
 * Equality
 * ------------------------------------------------------------ *)

let dictkind_equal kind1 kind2 = match kind1, kind2 with
	| DictArg,DictArg | DictEnv, DictEnv -> true
	| DictGlobal s1, DictGlobal s2 when s1 = s2 -> true
	| _ -> false
	
let impl_equal (iface1,id1,kind1) (iface2,id2,kind2) = 
	id_equal iface1 iface2 &&
	id_equal id1 id2 &&
	dictkind_equal kind1 kind2

let typaram_equal (tyvar1,iface1) (tyvar2,iface2) = 
	id_equal tyvar1 tyvar2 && id_equal iface1 iface2

let dictenv_entry_equal (typaram1,impl1) (typaram2,impl2) = 
	typaram_equal typaram1 typaram2 && impl_equal impl1 impl2

let dictenv_equal de1 de2 = 
	List.length de1 = List.length de2 &&
	List.for_all2 dictenv_entry_equal de1 de2	
