open Syntax
open Util
open Prettyutil
open Metadata
open Syntaxutil
open Names
open Cmdargs
open Ctojekyll


(* ------------------------------------------------------------
 * C/Jekyll differences
 * ------------------------------------------------------------ *)

let pprint_jlist f items = pprint_list_sep (fun () -> twin "," ";") f items

let rec pprint_ojlist f items = match items with
	| [] -> empty
	| [x] -> f x
	| x::xs ->  f x <+> onlyjkl "," <++> pprint_ojlist f xs


let pprint_seqlist f items = pprint_jlist f items <+> onlyc ";"
	

(* ------------------------------------------------------------
 * Simple util functions
 * ------------------------------------------------------------ *)

let braceblock p = str "{" <+> indent p <+> str "}"

let pprint_maybe f a = match a with
	| None -> empty
	| Some x -> f x

let maybe_c_sep l = if l <> [] then onlyc "," <+> space else empty

let pprint_pretty x = x

(* used for things we don't deal with yet *)
let unknown = str "???" 

type argstate = StartHidden | StartFresh | PrevOpen | PrevClosed

type initexp = (unit -> pretty) * bool


let pprint_id (m,s) = str s
let pp_id = pprint_id
  
let pp_pp x = x 


(* ------------------------------------------------------------
 * Names
 * ------------------------------------------------------------ *)

let pprint_iface_name id = twin (id_str id) (name_iface id)
let pprint_tagged_name id = twin (id_str id) (name_tagged id)
let pprint_dict_name iface id = twin (name_jkl_tyvar id) (name_dict id iface)

(* ------------------------------------------------------------
 * Type Parameters
 * ------------------------------------------------------------ *)

let pprint_tyvar id = twin (name_jkl_tyvar id) (name_c_tyvar id)

let pprint_thisvar id = onlyjkl_block (pprint_tyvar id)

let pprint_jekyll_typaram (id,ifaceid) =
	pprint_id ifaceid <++> pprint_tyvar id 

let pprint_typaram (id,ifaceid) =
	pprint_id ifaceid <+> pprint_tyvar id 

let pprint_typarams typarams = pprint_jlist pprint_typaram typarams 

let pprint_typaram_vars tyvars = 
	if tyvars = [] then empty else
	onlyjkl_block (str "<" <+> (pprint_jlist pprint_tyvar tyvars) <+> str ">")


(* ------------------------------------------------------------
 * Function Calls
 * ------------------------------------------------------------ *)

let lprint_c_dictname (ifaceid,id,kind) = match kind with
	| DictEnv -> [name_envvar;"->";name_dict id ifaceid]
	| DictArg -> [name_dict id ifaceid]
 	| DictGlobal _ | DictThis -> ["&";name_dict id ifaceid]

let lprint_c_dictnameind (ifaceid,id,kind) = match kind with
	| DictEnv | DictArg -> lprint_c_dictname (ifaceid,id,kind) @ ["->"]
	| DictGlobal _ | DictThis -> [name_dict id ifaceid;"."]
		
let pprint_c_dictenvref (ifaceid,id,kind) = break <+> match kind with
	| DictEnv -> str name_envvar <+> str "->" <+> str (name_dictenv id ifaceid)
	| DictArg -> str (name_dictenv id ifaceid)
	| DictGlobal None -> str "_DNULL"
	| DictGlobal (Some s) -> str "&" <+> str s
	| DictThis -> str name_envvar

let pprint_c_dictimpl dict = 
	pprint_seq str (lprint_c_dictname dict) <+> 
	onlyc "," <+> pprint_c_dictenvref dict

let pprint_c_dictimpl_marked dict = 
	pprint_seq str (lprint_c_dictname dict) <+> onlyc "," <+> 
	pprint_c_dictenvref dict
	

(* in C, print any dictionary and environment args *)
let pprint_hidden_args m dict_opt =
	let dictargs = get_dict_args m in
	let funenv = get_funenv m in
	if dictargs <> [] || dict_opt <> None then
		StartHidden,
		onlyc  "(" <+> 
		(match dict_opt with 
			| Some d -> onlyc_block 
					(pprint_c_dictenvref d <+> maybe_c_sep dictargs)
			| None -> empty) <+>
		onlyc_block (pprint_list pprint_c_dictimpl_marked dictargs)
	else if funenv <> None then
		StartHidden,
		onlyc "(" <+> onlyc (valof funenv)
	else
		StartFresh,
		empty

(* separate the current arg from the previous ones *)
let pprint_open_sep status = match status with
	| StartHidden | PrevOpen -> onlyc "," <+> space
	| StartFresh -> onlyc "("
	| PrevClosed -> twin ")" "," <+> space
let pprint_closed_sep status = match status with
	| StartHidden | PrevOpen -> twin "(" "," <+> space
	| StartFresh -> str "("
	| PrevClosed -> str "," 
let pprint_arg_end status = match status with
	| StartHidden | PrevClosed -> str ")"
	| StartFresh -> str "(" <+> str ")"
	| PrevOpen -> onlyc ")"

let rec pprint_generic_cast (m,exp) = match get_needscast m with
	| Some ty -> 
		onlyc_block ( parens (str "void" <+> str "*"))  <+>
		pprint_exp (m,exp)
	| None -> pprint_exp (m,exp)
	
and pprint_argexp exp = 
	pprint_generic_cast exp <+>
	match exp with
	| (m,Var id) when is_closure_typ (get_type m) ->
		onlyc_block (pprint_envvar_c id) 
	| (m,LocalFun _) ->
		let funinfo = get_localfun_info m in
		onlyc_block (str "," <++>  
			if funinfo.envvars <> [] then 
				str "&" <+> onlyc_var (get_localfun_info m).envname
			else str "_DNULL")
	| _ -> empty
	
and pprint_real_args status args = match args with
	| (m,LocalFun (args,block) as exp)::rest ->
		pprint_open_sep status <+>
		pprint_argexp exp <+>
		pprint_real_args PrevOpen rest
	| exp::rest ->
		pprint_closed_sep status <+>
		pprint_argexp exp <+>
		pprint_real_args PrevClosed rest
	| [] -> 
		pprint_arg_end status
		
and pprint_args m dict_opt args =
	let status,pp = pprint_hidden_args m dict_opt in
	pp <+> pprint_real_args status args
		
and pprint_nonop_call m func args =
	match get_call_dict m with
	| None -> pprint_exp func <+> pprint_args m None args
	| Some ((ifaceid,id,DictGlobal _ as dict),meth) ->
			twin meth (id_str id ^ "_" ^ meth) <+>
			pprint_args m (Some dict) args
	| Some (dict,meth) ->
			onlyc_prefix (lprint_c_dictnameind dict) 
				(str meth) <+>
			pprint_args m (Some dict) args
			 

and pprint_funcall m func args = match func,args with
	| (m,Var (_,op)),[x;y] when member op builtin_ops ->
		pprint_exp x <++> str op <++> pprint_exp y
	| (m,Var (_,op)),[x] when member op builtin_prefix_ops ->
		str op <+> pprint_exp x
	| (m,Var (_,op)),[x] when member op builtin_postfix_ops ->
		pprint_exp x <+> str (trim_postfix_op op)
	| (m,Var (m2,op)),[x;y] when is_operator_name op ->
		onlyc (get_opname m2) <+>
		onlyc "(" <+>
		pprint_exp x <+> twin op "," <+> pprint_exp y <+>
		onlyc ")"
	| _ -> pprint_nonop_call m func args


(* ------------------------------------------------------------
 * Constants
 * ------------------------------------------------------------ *)

and pprint_const const = match const with
	| ConstInt s -> str s
	| ConstFloat s -> str s
	| ConstString s -> str s
	| ConstSizeTy typ -> str "sizeof" <+> parens (pprint_abstyp typ)
	| ConstSizeExp exp -> str "sizeof" <+> pprint_exp exp
	| ConstAlignTy typ -> str "alignof" <+> parens (pprint_abstyp typ)
	| ConstAlignExp exp -> str "alignof" <+> pprint_exp exp


(* ------------------------------------------------------------
 * Initialisers
 * ------------------------------------------------------------ *)
(* C: a statement filling in fields, Jkl: an expression *)

and pprint_ppfield (printer,ispointer) = 
	printer () <+> if ispointer then str "->" else str "."
	
and pprint_ppexp (printer,ispointer) = 
	(if ispointer then str "*" else empty) <+> printer ()
	
and pp_makeptr (printer,ispointer) = 
	if ispointer then (fun () -> parens (str "*" <+> printer ())),ispointer 
		else (printer,true)	

and pprint_field_tinit (pp : initexp) (m,tinit) =
	pprint_tinit_exp ((fun () -> pprint_ppfield pp <+> str (get_fieldname m)),false) (m,tinit)

and pprint_tinit_exp (pp : initexp) (m,exp) = newline <+> match exp with
	| Init tinit -> pprint_tinit pp tinit
	| _ -> 
		onlyc_block (newline <+> pprint_ppexp pp <++> str "=") <+> 
		pprint_exp (m,exp) <+> onlyc ";"

and pprint_storage_kind kind = match kind with
	| Malloc -> twin "alloc" "jkl_malloc"
	| New -> twin "new" "GC_malloc"

and pprint_tinit pp (m,tinit) = newline <+> match tinit with
	| TConApp (tag,inner) ->
		onlyc_block (newline <+> pprint_ppfield pp <+> 
			str "_tag" <+> str "=") <+>
		pprint_id tag <+> onlyc ";" <+> 
		pprint_maybe (pprint_tinit_exp 
			((fun () -> pprint_ppfield pp <+> str "_body" <+> 
				str "." <+> pprint_id tag),false))
			inner 
	| TStruct fieldinits ->
		let flds = List.map snd fieldinits in
		onlyjkl "{" <+>
		pprint_ojlist (pprint_field_tinit pp) flds <+> 
		onlyjkl "}" 
	| TAlloc (kind,inner) ->
		let simpletyp = strip_typ_extras (get_type m) in
		onlyc_block (newline <+> pprint_ppexp pp <++> str "=") <+> 
		onlyc_block (parens (pprint_abstyp simpletyp <+> str "*")) <+>
		pprint_storage_kind kind <++>
		onlyc_block (parens 
			(str "sizeof" <+> parens (pprint_abstyp simpletyp))) <+>
		onlyc ";" <+>
		pprint_tinit_exp (pp_makeptr pp) inner
		

(* ------------------------------------------------------------
 * Expressions
 * ------------------------------------------------------------ *)

and pprint_localfun_envstruct_c funinfo = if funinfo.envvars = [] then empty else 
	str "struct" <++> onlyc_var funinfo.strname <++> 
	braceblock (pprint_seq 
					(fun (s,t) -> pprint_typ t (str "*" <+> str s) <+> str ";")
					funinfo.envvars
	) <+> str ";" <+> onlyc_block (newline)
	
and pprint_localfun_envarg_c funinfo = if funinfo.envvars = [] then
			str "void" <+> str "*" <++> str name_envvar 
		else
			str "struct" <++> onlyc_var funinfo.strname <+> 
			str "*" <++> str name_envvar

and pprint_localfun (m, args, block) =
	let funinfo = get_localfun_info m in
	onlyc_block (newline <+> pprint_localfun_envstruct_c funinfo) <+>
	onlyc_block (newline <+> pprint_typ (get_return_type m) 
						(onlyc_var funinfo.funname)) <+>
	twin "{" "(" <+>
	onlyc_block (pprint_localfun_envarg_c funinfo) <+>
	break <+> onlyc "," <+>
	pprint_decl_list args <+> 
	(if args <> [] then twin ":" ")" else onlyc ")") <+>
	onlyc "{" <+>
	indent (pprint_blockbody m block) <+>
	str "}"
	
and pprint_field_op deref = if deref then str "->" else str "."		

and pprint_envvar_c id = str "," <+> str (name_funenv id)

and with_shared m pretty =
	let sharedref = get_sharedpretty m in
	match !sharedref with
		| Some p -> p
		| None -> sharedref := Some pretty; pretty
		
and pprint_shared_tinit (m,tinit) = 
	let namevar = get_tempname m in
	with_shared m (
		pprint_tinit ((fun () -> onlyc_var namevar),false) (m,tinit)) 

and pprint_shared_lambda (m,args,body) =
	with_shared m (pprint_localfun (m,args,body))

and extract_c_tinit tinit = extract_c (pprint_shared_tinit tinit)
and extract_c_lambda lam = extract_c (pprint_shared_lambda lam)
	
and pprint_exp (m,exp) = match exp with
	| JklNonDet (key,opts,c) ->
		jkl_choice key (List.map pprint_exp opts) (pprint_exp c)		
	| LocalFun (args,block) -> 
		let funinfo = get_localfun_info m in
		extract_jkl (pprint_shared_lambda (m,args,block)) <++>
		break <+> onlyc "&" <+> onlyc_var funinfo.funname
	| FunCall (func,args) -> pprint_funcall m func args
	| Var id when get_is_envvar m -> 
		onlyc_prefix ["*";name_envvar;"->"] (pp_id id) 
	| Var id -> pprint_id id
	| Field (deref,lhs,field) -> 
		pprint_exp lhs <+> pprint_field_op deref <+> pprint_id field
	| Const const -> pprint_const const
	| Cast (typ,exp) -> parens (pprint_abstyp typ) <+> pprint_exp exp
	| Index (arr,idx) -> 
		pprint_exp arr <+> str "[" <+> pprint_exp idx <+> str "]"
	| Choice (cond,iftrue,iffalse) ->
		pprint_exp cond <+> str "?" <+> pprint_exp iftrue <+>
		str ":" <+> pprint_exp iffalse
	| Assign (lhs,op,rhs) ->
		pprint_exp lhs <+> str op <+> pprint_exp rhs
	| Parens exp -> parens (pprint_exp exp)
	| Unsafe exp -> str "unsafe" <++> pprint_exp exp
	| Init (m2,_ as tinit) -> extract_jkl (pprint_shared_tinit tinit) <++> 
			onlyc_var (get_tempname m2)
	| EBlock block -> pprint_block m block

and pprint_exp_debug (m,exp) = match exp with
	| Var id -> pprint_id id
	| Parens e -> parens (pprint_exp_debug e)
	| Field (deref,lhs,field) ->
		pprint_exp_debug lhs <+> pprint_field_op deref <+> pprint_id field
	| FunCall (func,args) ->
		pprint_exp_debug func <+> parens (pprint_list pprint_exp_debug args)
	| Init _ -> str "init"
	| _ -> str "???"
	

(* ------------------------------------------------------------
 * Statements
 * ------------------------------------------------------------ *)

and pprint_c_tagmatch tag =
	parens (pprint_exp (get_scrutinee (id_meta tag))) <+>
	str "." <+> str "_body" <+> 
	str "." <+> pprint_id tag

and pprint_case m pattern = newline <+> match pattern with
	| PTag (tag,None) -> 
			str "case" <++>
			pprint_id tag <+> str ":" 
	| PTag (tag,Some id) -> 
			str "case" <++>
			pprint_id tag <++> 
			onlyc ":" <++>
			pprint_id id <++>
			onlyc "=" <+>
			onlyc_block (pprint_c_tagmatch tag) <+>		
			twin ":" ";" 
	| PDefault -> str "default" <+> str ":"
	| PConst exp -> str "case" <++> pprint_exp exp <+> str ":"	

and pprint_jump details = match details with
	| JBreak -> str "break"
	| JContinue -> str "continue"
	| JReturn exp_opt -> str "return" <++> pprint_maybe pprint_exp exp_opt
	| JRet exp_opt -> str "ret" <++> pprint_maybe pprint_exp exp_opt
	| JGoto id -> str "goto" <++> pprint_id id

and pprint_switch (m,exp) body = 
	if is_tagged_typ (get_type m) then
		str "switch" <+> 
		parens (
		onlyc "(" <+> pprint_exp (m,exp) <+> 
				onlyc ")" <+> onlyc "." <+> onlyc "_tag") <+>
		pprint_stmt body
	else
		str "switch" <++> parens (pprint_exp (m,exp)) <+> pprint_stmt body

and pprint_lamenv funinfo = if funinfo.envvars = [] then empty else
	onlyc_block (newline <+> str "struct" <++> onlyc_var funinfo.strname <++> 
		onlyc_var funinfo.envname <++> str "=" <++> str "{" <+> 
		pprint_list (fun (s,t) -> str "&" <+> str s) funinfo.envvars <+>
		str "}" <+> str ";")  

and pprint_dictenv (name,tyname,ifaceid,dictenv) =
	newline <+>
	onlyc_block (str "struct" <++> str (name_dictenv tyname ifaceid) <++> 
		str name <++> str "=" <++> str "{" <+>
		pprint_list (fun (_,impl) -> pprint_c_dictimpl impl) dictenv <+> 
		str "}" <+> str ";")

and pprint_hiddendecl (name,typ) =
	let typ = strip_typ_extras typ in 
	newline <+> onlyc_block  (pprint_typ typ (onlyc_var name) <+> str ";")

and pprint_stmt (m,stmt) = 
	let ppstmt = pprint_real_stmt (m,stmt) in
	match get_maybehidden m with
	| Some k -> jkl_choice k [empty;ppstmt] ppstmt
	| None -> ppstmt

and pprint_real_stmt (m,stmt) =
	newline <+> 
	if has_tag m Hidden then empty else
	let temps = get_temps m in
	pprint_seq extract_c_tinit temps <+> 
	match stmt with
	| Label (id,stmt) -> pprint_id id <+> str ":" <+> pprint_stmt stmt
	| SExp exp -> pprint_exp exp <+> str ";"
	| Block block -> pprint_block m block
	| Switch (exp,body) -> pprint_switch exp body
	| Case (pat,stmt) -> pprint_case m pat <++> pprint_stmt stmt
	| If (cond,iftrue,iffalse_opt) ->
		str "if" <+> parens (pprint_exp cond) <+>
		pprint_stmt iftrue <+>
		pprint_maybe (fun s -> str "else" <+> pprint_stmt s) iffalse_opt
	| While (cond,body) ->
		str "while" <+> parens (pprint_exp cond) <+> pprint_stmt body
	| Do (body,cond) ->
		str "do" <+> pprint_stmt body <+> str "while" <+>
		parens (pprint_exp cond) <+> str ";"
	| For (init,cond,step,body) ->
		str "for" <+> parens (
			pprint_maybe pprint_exp init <+> str ";" <+>
			pprint_maybe pprint_exp cond <+> str ";" <+>
			pprint_maybe pprint_exp step
		) <+> pprint_stmt body
	| Jump details -> pprint_jump details <+> str ";"
	| Semicolon -> str ";"

and pprint_block m block = braceblock (pprint_blockbody m block)
	
and pprint_blockbody m (decls,stmts) =
	let fwddecls = get_fwddecls m in
	let tempdecls = get_tempdecls m in
	let lamenvs = get_lamenvs m in
	let dictenvs = get_dictenvs m in
	pprint_semiterm_decls decls <+>
	pprint_seq pprint_dictenv dictenvs <+>
	pprint_seq pprint_hiddendecl tempdecls <+>
	pprint_seq pprint_hiddendecl fwddecls <+>
	pprint_seq pprint_lamenv lamenvs <+>
	pprint_seq pprint_stmt stmts


(* ------------------------------------------------------------
 * Core Types
 * ------------------------------------------------------------ *)

and pprint_structkind kind = match kind with
	| SKStruct -> jkl_choice (new_key ()) [str "struct";str "tagged"] 
					(str "struct")
	| SKUnion -> str "union"
	| SKTagged -> twin "tagged" "struct"

and pprint_structname kind name = 
	let s = id_str name in
	match kind with
	| SKTagged -> twin s (name_tagged name)
	| _ -> str s

and pprint_struct_tyvars (m,tyvars,decls) =
	onlyjkl_block (pprint_typaram_vars tyvars)

and pprint_structdetails (m,tyvars,decls) =
	braceblock (
		pprint_semiterm_decls (decls : declaration list)
	)
	
and pprint_enumfield (id,exp_opt) = match exp_opt with
	| None -> pprint_id id
	| Some exp -> pprint_id id <++> str "=" <++> pprint_exp exp
	
and pprint_enumdetails enums =
	braceblock (pprint_list pprint_enumfield enums)

and pprint_real_struct kind name details =
	pprint_structkind kind <++>
	pprint_maybe pprint_struct_tyvars details <++>
	pprint_maybe (pprint_structname kind) name <++>
	pprint_maybe pprint_structdetails details
	
and pprint_struct kind name details = match maybe_tagged details,details with
	| Some fields,Some (m,tyvars,_) -> 
			pprint_real_struct SKTagged name (Some (m,tyvars,fields))
	| _ -> pprint_real_struct kind name details

and pprint_c_decl_tag (decl : declaration) = pprint_id (simpledecl_name decl)
	
and pprint_c_tagged_tag (decls : declaration list) =
	str "enum" <+> str "{" <+>
		pprint_list pprint_c_decl_tag decls <+>
	str "}" <++> str "_tag" <+> str ";"

and pprint_tagged_field decl = 
	let typ = simpledecl_typ decl in
	if typ = tyvoid then empty
	else
		pprint_decl decl <+> str ";"	

and pprint_tagged name (m,tyvars,decls : structdetails) =
	twin "tagged" "struct" <++>
	onlyjkl_block (pprint_typaram_vars tyvars) <+>
	pprint_maybe (pprint_structname SKTagged) name <+>
	braceblock (
		onlyc_block (pprint_c_tagged_tag decls) <++>
		onlyc "union" <++>
		onlyc "{" <+> indent ( 
			pprint_seq pprint_tagged_field decls
		) <+> onlyc "}" <++> onlyc "_body" <+> onlyc ";"
	)
	
and pprint_enum name details = 
	str "enum" <++>
	pprint_maybe pprint_id name <+>
	pprint_maybe pprint_enumdetails details

and pprint_tyof uscore = 
	if uscore then str "__typeof__"
	else str "typeof"

and pprint_coretyp coretyp = match coretyp with
	| TyBasic -> empty
	| TyWild id -> pprint_tyvar id
	| TyVoid -> decodejkl_var (new_namevar "void" "%") "void"
	| TyName id -> pprint_id id
	| TyStruct (SKTagged,name,Some details) ->
			pprint_tagged name details
	| TyStruct (kind,name,details) -> 
			pprint_struct kind name details
	| TyEnum (name,details) -> pprint_enum name details
	| TyTypeofExp (uscore,e) -> 
			pprint_tyof uscore <+> parens (pprint_exp e)
	| TyTypeofTyp (uscore,t) ->
			pprint_tyof uscore <+> parens (pprint_abstyp t)

and pprint_specifier pp spec = (match spec with
	| SCoreHere -> pp
	| SStatic -> str "static"
	| SInline -> str "inline"
	| SExtern -> str "extern"
	| SAuto -> str "auto"
	| SRegister -> str "register"
	| SChar -> str "char"
	| SShort -> str "short"
	| SInt -> str "int"
	| SLong -> str "long"
	| SFloat -> str "float"
	| SDouble -> str "double"
	| SSigned -> str "signed"
	| SUnsigned -> str "unsigned"
	| SConst -> str "const"
	| SVolatile -> str "volatile"
	| STaggedOnly -> raise (Failure "taggedonly not stripped")
	| STypeDef -> str "typedef")
	<+> space
	
and pprint_fatptr specs ptyp pp =
	pprint_seq (pprint_specifier empty) specs <+> 
	extract_jkl ptyp <+>
	twin "?" "fatptr" <+>
	onlyc "(" <+>
	extract_c ptyp <+>
	onlyc ")" <+>
	pp	
	
and pprint_real_decltyp declty pp = match declty with
	| [] -> pp
	| DFatPtr specs::xs -> pprint_fatptr specs (pprint_real_decltyp xs empty) pp
	| x::xs -> pprint_real_decltyp xs (pprint_declmod x pp)

and pprint_decltyp declty pp =
	if !Cmdargs.j2c || !noexpand then pprint_real_decltyp declty pp else
	let posargs = get_possible_tyargs () in
	let dflt = pprint_real_decltyp declty pp in
	let arg_pps = List.map (with_noexpand pprint_tyargs) posargs in
	let opt_pps = List.map (fun a -> a <+> dflt) arg_pps in
	jkl_choice (new_key ()) (dflt::opt_pps) dflt

and pprint_c_envarg needcomma kind = match kind with
	| Closure | IfaceMethod | DictMethod None -> 
		str "void" <+> str"*" <++> str name_envvar <+>
		if needcomma then break <+> str "," <+> space else empty
	| DictMethod (Some (iface,tyname)) -> 
		str "struct" <++> str (name_dictenv tyname iface) <+> str "*" <++>
		str name_envvar <+>
		if needcomma then break <+> str "," <+> space else empty
	| SimpleFun | VarArgs -> empty
	
and pprint_tyiface_arg (var,iface) =
	onlyc_prefix ["struct"] (onlyc_block forcespace <+> pprint_iface_name iface) <+> onlyc "*" <++> 
	pprint_dict_name iface var <++>
	onlyc_block (str "," <++> str "void" <+> str "*" <++> 
				 str (name_dictenv var iface)) 

and pprint_hidden_params tyifaces decls kind = 
	if tyifaces <> [] then 
		twin "<" "(" <+>
		onlyc_block (pprint_c_envarg true kind) <+>
		let pp_iface = pprint_list pprint_tyiface_arg tyifaces in
		if decls <> [] then
			pp_iface <+> onlyjkl ">" <+> twin "(" ","
		else
			pp_iface <+> str ">" <+> str "("
	else 
		str "(" <+> 
		onlyc_block (pprint_c_envarg (decls <> []) kind) 	

and pprint_arg decl = match decl with
	| m,(basetyp,[(DFun(args,Closure)::_,Some id),None])->
		pprint_decl decl <+> 
		onlyc_block (str "," <++> str "void" <+> str "*" <++>
			str (name_funenv id))
	| _ -> pprint_decl decl

and pprint_argsfull m tyifaces decls kind =
	let pp1 = pprint_hidden_params tyifaces decls kind in
	let pp2 = 
		if kind = Closure then onlyjkl "!" <+> pp1 
		else pp1 in
	let pp3 =
		if decls = [] then
			if (tyifaces <> [] || kind <> SimpleFun) then
				pp2 <+> onlyjkl "void"
			else
				pp2 <+> str "void"
		else
			pp2 <+> pprint_list pprint_arg decls in		
	pp3 <+> 
		(if kind = VarArgs then str "," <+> str "..." else empty) <+>
	str ")"	
	
and pprint_argdetails argdetails va = match argdetails with
	| ArgsFull (m,tyifaces,decls) -> pprint_argsfull m tyifaces decls va
	| ArgsNamed ids -> pprint_list pprint_id ids
	| ArgsNoinfo -> str "(" <+> str ")"	
	
and pprint_declmod declmod pp = match declmod with
	| DPtr specs ->
		str "*" <+> pprint_seq (pprint_specifier empty) specs <+> pp
	| DArray size_opt ->
		pp <+> str "[" <+> pprint_maybe pprint_exp size_opt <+> str "]"
	| DBitField bits ->
		pp <++> str ":" <++> pprint_exp bits
	| DParens ->
		parens pp 
	| DWithArgs args ->
		pprint_tyargs args <++> pp 
	| DFun (args,va) ->
		pp <+> pprint_argdetails args va
	| DFatPtr _ -> str "?"
	
and pprint_basetyp (specifiers,coretyp) =	
	pprint_seq (pprint_specifier (pprint_coretyp coretyp)) specifiers 
		
and pprint_typ (specifiers,coretyp,declty) pp =
	pprint_basetyp (specifiers,coretyp) <+>
	pprint_decltyp declty pp

and pprint_abstyp typ = pprint_typ typ empty
and pp_ty typ = pprint_abstyp typ

and pprint_tyargs typs = 
	onlyjkl_block (str "<" <+> pprint_list pprint_abstyp typs <+> str ">")

and pprint_maybe_tyargs typs = match typs with 
	| [] -> empty
	| _ -> pprint_tyargs typs


(* ------------------------------------------------------------
 * Declarations
 * ------------------------------------------------------------ *)

and pprint_initialiser initialiser = match initialiser with
	| IExp e -> pprint_exp e
	| IFields (inits,finalcomma) -> braceblock (
				pprint_list pprint_initialiser inits <+>
				(if finalcomma then str "," else empty)
	 		)

and pprint_declarator (declty,id_opt) =
	pprint_decltyp declty (pprint_maybe pprint_id id_opt) 

and pprint_init_declarator (declarator,init_opt) =
	pprint_declarator declarator <+>
	pprint_maybe (fun i -> space <+> str "=" <++> pprint_initialiser i) init_opt

and pprint_real_decl (m,(basetyp,init_declarators) : declaration) =
	pprint_basetyp basetyp <+>
	pprint_list pprint_init_declarator init_declarators 

and pprint_decl (m,((_,ctyp),_) as decl) = 
	let ppdecl = pprint_real_decl decl in
	match maybe_interface ctyp with
	| Some (id,mems) -> 
			let ppdecl = pprint_real_decl decl in
			jkl_choice (new_key ()) 
				[ppdecl;pprint_decoded_iface id mems] ppdecl 
	| _ -> ppdecl

and pprint_semiterm_decl (m,_ as decl) = 
	let ppdecl = pprint_decl decl <+> str ";" <+> newline in
	match get_maybehidden m with
	| Some k -> jkl_choice k [empty;ppdecl] ppdecl
	| None -> ppdecl 

and pprint_decl_list decls = pprint_list pprint_decl decls
and pprint_semiterm_decls decls = 
	pprint_seq pprint_semiterm_decl decls


(* ------------------------------------------------------------
 * Decoded interface
 * ------------------------------------------------------------ *)

and pprint_decoded_iface id methods =
	str "interface" <++> pprint_id id <++>
	onlyjkl_var (new_namevar "%a" "%") <++>
	braceblock (
		pprint_seq pprint_method_field methods
	)


(* ------------------------------------------------------------
 * Methods and functions
 * ------------------------------------------------------------ *)

and pprint_argdetails_sep argdetails = match argdetails with
	| ArgsNoinfo -> empty
	| _ -> str ","

and pprint_field_funsig prefix fundecl =
	let id,ret_ty,args,funkind = split_fundecl fundecl in
	pprint_abstyp ret_ty <++>
	onlyc_prefix ["(";"*"]  
		(twin (id_str id) (add_prefix prefix (id_str id))) <+>
	onlyc ")" <+>
	pprint_argdetails args funkind

and pprint_funsig prefix fundecl = 
	let id,ret_ty,args,funkind = split_fundecl fundecl in
	newline <+> pprint_abstyp ret_ty <++>
	twin (id_str id) (add_prefix prefix (id_str id)) <+>
	pprint_argdetails args funkind
	
and pprint_method prefix (m,(decl,krdecls,block)) =
	pprint_funsig prefix decl <+>
	pprint_semiterm_decls krdecls <+>
	pprint_block m block

and pprint_function m (decl,krdecls,block) =
	let lambdas = get_lambdas m in
	pprint_seq extract_c_lambda lambdas <+>
	pprint_funsig "" decl <+> 
	pprint_semiterm_decls krdecls <+>
	pprint_block m block
	
and pprint_method_field funsig =
	pprint_field_funsig "" funsig <+> str ";"	
	

(* ------------------------------------------------------------
 * Interface declarations
 * ------------------------------------------------------------ *)

let pprint_iface id (thisid,methods) =
	twin "interface" "struct" <++>
	pprint_iface_name id <+> pprint_thisvar thisid <+>
	braceblock (
		pprint_seq pprint_method_field methods
	) <+> str ";"
	

(* ------------------------------------------------------------
 * Dictionary Definitions
 * ------------------------------------------------------------ *)

let pprint_methodset tyname iface (fundecl,_ ,_: fundef) =
	let id,_,_,_ = split_fundecl fundecl in
	break <+> (str "&" <+> str (name_method tyname iface id))

let pprint_c_dict_structfill prefix tyname iface mmethods =
	let methods = List.map snd mmethods in
	let methodsigs = List.map fundef_decl methods in
	str "struct" <++> str "{" <+>
	indent (pprint_list_sep (fun () -> newline) pprint_method_field methodsigs) <+> str "}" <++>
	str (prefix ^ "_dict") <++>
	str "=" <++> str "{" <+> 
	indent (pprint_list (pprint_methodset tyname iface) methods) <+> 
	str "}" <+> str ";"

let pprint_typ_skind (specs,coretyp,declmods) = match coretyp with
	| TyStruct (SKTagged,_,_) -> twin "tagged" "struct"
	| TyStruct (kind,_,_) -> pprint_structkind kind
	| _ -> empty
	
let pprint_jkl_dictinfo iface typ = pprint_id iface <++> with_noexpand pprint_abstyp typ  

let pprint_dict_requirement (id,iface) =
	onlyc "struct" <++> pprint_iface_name iface <+> onlyc "*" <++>
	twin (name_jkl_tyvar id) (name_dict id iface) <+> onlyc ";" <++> onlyc "void" <+> 
	onlyc "*" <++> onlyc (name_dictenv id iface) <+> onlyc ";" <+> newline

let pprint_dict_requirements tyname iface requires =
	if requires <> [] then
		newline <+>
		onlyc "struct" <++> 
		onlyc (name_dictenv tyname iface) <+> 
		twin ":" "{" <+>
		indent (pprint_ojlist pprint_dict_requirement requires) <+>
		onlyc "}" <+> onlyc ";"
	else
		newline

let pprint_dict_funprotos_c protos tyname iface =
	indent (pprint_seq 
		(fun s -> pprint_funsig (id_str tyname) s <+> str ";") protos) 
	<+> newline 

let pprint_dict_proto_c tyname iface = 
	str "struct" <++> 
	pprint_iface_name iface <++>
	str (name_dict tyname iface) <+> str ";" <+> newline

let pprint_dictproto protos tyname iface requires =
	onlyc_block (pprint_dict_proto_c tyname iface) <+>
	pprint_dict_requirements tyname iface requires <+>
	onlyjkl ";" <+> newline <+>
	onlyc_block (pprint_dict_funprotos_c protos tyname iface) 
	
let pprint_method_funref_c tyname iface iproto =
	parens (str "void" <+> str "*") <+>
	str "&" <+> 
	str (name_method tyname iface (simpledecl_name iproto))
	
let pprint_dict_struct_c iprotos tyname iface methods =
	pprint_iface_name iface <++> 
	str (name_dict tyname iface) <++>
	str "=" <++> str "{" <+> indent (
		pprint_list (pprint_method_funref_c tyname iface) iprotos
	) <+> str "}" <+> str ";" <+> newline
		
let pprint_jkl_dictimpl_requirement (id,iface) = pprint_id iface <+> pprint_tyvar id	
		
let pprint_jkl_dictimpl_requirements requires =
	if requires <> [] then
		str ":" <+> newline <+>
		indent (pprint_jlist pprint_jkl_dictimpl_requirement requires)
	else
		empty		
		
let pprint_dictimpl iprotos tyname iface methods requires =
	onlyc_block (pprint_dict_struct_c iprotos tyname iface methods) <++>
	onlyjkl_block (pprint_jkl_dictimpl_requirements requires) <+>
	onlyjkl "{" <+>
	pprint_seq (pprint_method (id_str tyname)) methods <+>
	onlyjkl "}" 

let pprint_dictdef m (typ,iface,body) =
	let tyname = typ_name_nocheck typ in
	match body with
		| DictProto requirements -> 
			twin "implement" "extern" <+>
			onlyjkl_block (pprint_jkl_dictinfo iface typ) <++>
			pprint_dictproto (get_dict_protos m) tyname iface requirements
		| DictImpl (requirements,methods) -> 
			twin "implement" "struct" <+>
			onlyjkl_block (pprint_jkl_dictinfo iface typ) <++>
			pprint_dictimpl (get_iface_protos m) tyname iface 
				methods requirements	
	

(* ------------------------------------------------------------
 * #define
 * ------------------------------------------------------------ *)
	
let pprint_macrokind kind = match kind with
	| MDecl decl -> pprint_decl decl
	| MSilent ids -> str "silent" <++> pprint_list pprint_id ids
	| MSilentFun ids -> str "silent-fun" <++> pprint_list pprint_id ids	
	
let pprint_macrotype kind = 
	str "unsafe" <++>
	str "macrotype" <+>
	onlyc "(" <+>
	pprint_macrokind kind <+>
	onlyc ")" <+> str ";"


(* ------------------------------------------------------------
 * Stdtype declarations
 * ------------------------------------------------------------ *)

let pprint_stdtype decl =
	str "stdtype" <++> onlyc "(" <+> pprint_decl decl <+>
	onlyc ")" <+> str ";"


(* ------------------------------------------------------------
 * Used soley for error messages
 * ------------------------------------------------------------ *)

let pprint_funkind funkind = str (match funkind with
	| DictMethod _ -> "dictionary"
	| IfaceMethod -> "interface"
	| Closure -> "closure"
	| SimpleFun -> "simple"
	| VarArgs -> "varargs")


(* ------------------------------------------------------------
 * Jekyll Import
 * ------------------------------------------------------------ *)

let pprint_import id = 
	twin "import" "#include" <+>
	onlyc "\"" <+> pprint_id id <+> onlyc "\""


(* ------------------------------------------------------------
 * Top level declaration
 * ------------------------------------------------------------ *)

let rec pprint_real_extdecl (m,extdecl) = newline <+> match extdecl with
	| Include s -> twin ("#include "^s) ("#include "^Parseutils.process_c_include s) <+> newline
	| Interface (id,impl) -> pprint_iface id impl
	| Dict dictdef -> pprint_dictdef m dictdef
	| Decl decl -> pprint_decl decl <+> str ";"
	| Func fundef -> pprint_function m fundef
	| MacroType kind -> pprint_macrotype kind
	| DSemicolon -> str ";"
	| StdType decl -> pprint_stdtype decl
	| NonDet (key,opts,c) -> 
		jkl_choice key (List.map pprint_extdecl opts) (pprint_extdecl c)
	
and pprint_extdecl (m,extdecl) = 
	let ppextdecl = pprint_real_extdecl (m,extdecl) in
	match get_maybehidden m with
	| Some k -> jkl_choice k [empty;ppextdecl] ppextdecl
	| None -> ppextdecl
	
let pprint_program prog = pprint_seq pprint_extdecl prog



