
open Filename
open Location
open Syntax
open Syntaxutil
open Parser
open Prettyutil
open Prettyprint
open Typecheck
open Metadata
open Util
open Lexing
open Errors
open Cdecode
open Parseutils


let msg s = if !Cmdargs.verbose then print_endline s else ()

let fail s = print_endline s; exit 2


(* ------------------------------------------------------------
 * Filenames
 * ------------------------------------------------------------ *)

let output_extension fname = match !Cmdargs.j2c with
	| true when check_suffix fname "jkl" -> ".c"
	| true when check_suffix fname "jkh" -> ".h"
	| false when check_suffix fname "h" -> ".jkh"
	| false when check_suffix fname "c" -> ".jkl"
	| _ -> fail "Unrecognised file exetension"

let create_output_filename fname = match !Cmdargs.outfile with
	| Some f -> f
	| None when !Cmdargs.copy_output ->  
		chop_extension fname ^ "2" ^ output_extension fname
	| None -> chop_extension fname ^ output_extension fname

let create_prev_filename fname = match !Cmdargs.prevfile with
	| Some f -> f
	| None -> chop_extension fname ^ output_extension fname
		

(* ------------------------------------------------------------
 * Tokenize the old version of the file
 * ------------------------------------------------------------ *)

let rec read_tokens lexbuf = match Lexer.token lexbuf with
	| EOF _ -> ()
	| _ -> read_tokens lexbuf

let lines_for_old_version fname = if !Cmdargs.new_output then [] 
	else 
		let fname = create_prev_filename fname in
		msg ("old version = "^fname);
		reset_inputtokens ();
		let in_chan = open_in_bin fname in
		let lexbuf = Lexing.from_channel in_chan in
		Parseutils.set_current_file fname;
		lexstate_set_file_line lexbuf fname 1 0;	
		read_tokens lexbuf;
		close_in in_chan;
		Location.get_inputlines ()
	

(* ------------------------------------------------------------
 * Parse the program
 * ------------------------------------------------------------ *)

let pprint_incdir name = space <+> str "-I" <+> str name 

let pprint_include s = str "#include" <++> str s <+> newline

let parse_includes filename =
	let includes = find_includes filename in
	let includes = process_includes includes in
	let incdirs = pretty_to_string 
		(pprint_seq pprint_incdir !Cmdargs.includedirs) in
	let in_chan,out_chan = Unix.open_process 
		("cpp -dD -D__JEKYLL_H__" ^ !Cmdargs.cparams ^ " " ^ incdirs) in
	output_pretty out_chan (pprint_seq pprint_include includes);
	output_string out_chan "\n";
	close_out out_chan;
	let lexbuf = Lexing.from_channel in_chan in
	lexstate_set_file_line lexbuf "<<includes>>" 1 0;
	let decls = Parser.translation_unit Lexer.token lexbuf in
	close_in in_chan;
	decls

let parse_program fname =
	reset_inputtokens ();
	dos_endings := Parseutils.file_is_dos fname;
	let in_chan = open_in_bin fname in
	let lexbuf = Lexing.from_channel in_chan in
	set_current_file fname;
	lexstate_set_file_line lexbuf fname 1 0;
	let program = Parser.translation_unit Lexer.token lexbuf in
	close_in in_chan;
	program

let pretty_print_program fname program lextokens =
	let output = open_out_bin (create_output_filename fname) in
	msg "producing non-deterministic description";
	let pretty = pprint_program program in
	msg "resolving non-determinism";
	if !Cmdargs.j2c then 
		output_string output ("#include <jekyll_1.h>" ^ line_break());
	let oldlines = lines_for_old_version fname in
	output_preserving_whitespace output lextokens pretty oldlines; 
	output_string output (gather_whitespace ());
	close_out output
	

(* ------------------------------------------------------------
 * Translate the program
 * ------------------------------------------------------------ *)
		
let parse_with_includes fname =
	reset_lexnames ();
	msg "parsing includes";
	let includes = parse_includes fname in
	msg "parsing programs";
	let program = parse_program fname in
	includes,program		

let parse_oldversion fname = 
	if !Cmdargs.new_output then [] 
	else 
		let prevname = create_prev_filename fname in
		let includes,program = parse_with_includes prevname in
		includes @ program
		
let main () =
	Cmdargs.process_command_args ();
	let fname = Cmdargs.get_fname () in 
	let includes,program = parse_with_includes fname in
	let lextokens = get_inputlines () in
	msg "parsing previous version";
	let oldversion = parse_oldversion fname in
	Ctojekyll.examine_old_program oldversion;
	msg "decoding program";
	let program = if !Cmdargs.j2c then fixup_jekyll program 
			else decode_features program oldversion in
	if !Cmdargs.typecheck && !Cmdargs.j2c then 
		(msg "type checking"; check_program program includes); 
	msg "pretty printing";
	pretty_print_program fname program lextokens
	
	
(* ------------------------------------------------------------
 * Program Entry Point
 * ------------------------------------------------------------ *)

let _ = try main () with 
	| Parsing.Parse_error -> Parseutils.parse_error ()
	| Sys_error s -> print_endline ("fatal error: "^s); exit 2
