
open Location
open Lexing
open Util
open Cmdargs
open Parseutils
open Syntax

(* ------------------------------------------------------------
 * Display errors nicely
 * ------------------------------------------------------------ *)

let print_position_source p = if pos_is_nowhere p then () else begin
	let inp = open_in p.pos_fname in
	for i = 1 to (p.pos_lnum - 1) do ignore (input_line inp) done;
	let line = input_line inp in
	let line = strip_tabs line in
	let line = (if p.pos_fname <> get_fname () 
				then collapse_spaces line else line) in
	close_in inp;
	print_endline line
	end

let print_span_source s = try print_position_source s.pstart
	with
		Sys_error s -> print_endline s

let print_position_guide p = if p.pos_fname = "" then () else begin
	let col = p.pos_cnum - p.pos_bol in
	let spaces = String.create (col + 1)  in
	String.fill spaces 0 col ' ';
	String.set spaces col '^';
	print_string spaces
	end

let pos_same_line x y = 
	(x.pos_fname = y.pos_fname) && (x.pos_lnum = y.pos_lnum)

let print_span_guide s = 
	print_position_guide s.pstart;
	if pos_same_line s.pstart s.pend then
		let markcount = s.pend.pos_cnum - s.pstart.pos_cnum - 1 in
		print_endline (list_to_string (list_repeat '^' markcount))
	else if not (span_is_nowhere s) then
		print_endline " --->"
	
let pos_column p = p.pos_cnum - p.pos_bol	
	
let span_to_str s =
	if span_is_nowhere s then "" else 
	if pos_same_line s.pstart s.pend then
		"File " ^ s.pstart.pos_fname ^ " : " ^
		string_of_int s.pstart.pos_lnum ^ " characters " ^
		string_of_int (pos_column s.pstart) ^ " - " ^ 
		string_of_int (pos_column s.pend - 1)
	else
		"File " ^ s.pstart.pos_fname ^ " : " ^
		string_of_int s.pstart.pos_lnum ^ " - " ^ 
		string_of_int s.pend.pos_lnum
		
let print_hbar () = print_string 
		("------------------------------------------" ^ 		
		 "-------------------------------------\n")

let print_error_header msg s =
	print_newline (); print_hbar ();
	print_string (span_to_str s);
	print_newline ();
	print_string msg; print_newline ()

let print_error_trailer () =
	print_hbar ();
	print_newline ()

(* print an error, together with the source code that caused it *)
let print_verbose_error msg s = 
	print_error_header msg s;
	if not (span_is_nowhere s) then begin
		print_span_source s; 
		print_span_guide s
	end;
	print_error_trailer ()
	
let print_simple_error msg s = print_string (
		s.pstart.pos_fname ^ ":" ^ 
		string_of_int s.pstart.pos_lnum ^ ": " ^ msg)
	
let print_span_msg s msg = 
	if !Cmdargs.short_errors then
		print_simple_error msg s
	else
		print_verbose_error msg s;
	if not !Cmdargs.many_errors then exit 2


(* ------------------------------------------------------------
 * Error entry points
 * ------------------------------------------------------------ *)

let errors_enabled = ref true

let with_suspend_errors f =
	let old = !errors_enabled in
	errors_enabled := false;
	let ret = f () in
	errors_enabled := old;
	ret
	
let error s msg = if !errors_enabled then print_span_msg s ("error: "^msg)	
let warning s msg = if !errors_enabled then print_span_msg s ("warning: "^msg)
let fatal s msg = print_span_msg s ("fatal error: "^msg); exit 2
let intwarn msg = print_endline ("internal warning: "^msg)	
let intfatal msg = print_endline ("fatal internal error: "^msg); exit 2

let msg s = if !Cmdargs.verbose then print_endline s else ()



