@echo off
rem File Name: h4h5build.bat
rem This batch file is used to build H4H5 Tools.
rem There batch file takes the following options:
rem         vs9               Build using Visual Studio 2008
rem         nodebug           Note: Default is to build debug and release versions
rem         useenv            Build using variables set in the environment.
rem

rem This batch file makes the following assumptions:
rem    - The appropriate version of Visual Studio is installed and setup
rem    - The directory structure is setup from a fresh source copy
rem    - copy_hdf.bat has already been run from the ./windows directory
rem    - The directory structure for external libraries has been setup in the
rem      IDE, or..
rem    - If using useenv switch, environment variables INCLUDE and LIB have 
rem      been setup with jpeg, zlib, szip (unless disabled), hdf4 and hdf5 directories
rem    - jpeg, zlib, and szip (unless disabled) DLLs are already placed in an 
rem      accessible directory
rem    - hdf4_ext_jpeg, hdf4_ext_zlib, and hdf4_ext_szip (unless disabled) 
rem      should all be defined
rem    - The target platform architecture is specified in the environment
rem      variable PROCESSOR_ARCHITECTURE
rem    - The hdf4 and hdf5 libraries are already placed in an acessible directory

setlocal enabledelayedexpansion
pushd %~dp0

set nerrors=0
if "%1"=="/?" goto help
set blddebug=debug
set bldrelease=release
goto main

rem Print a help message
:help

    echo.Builds H4H5 Libraries and Tools.
    echo.
    echo.Usage: %~nx0 [OPTIONS]
    echo.
    echo.   /?                      Help information
    echo.   vs9                     Build using Visual Studio 2008
    echo.   nodebug                 Note: Default is to build debug and release versions
    echo.   useenv                  Build using variables set in the environment.
    echo.
    echo.Specifying useenv requires you to set the INCLUDE and LIB variables to the
    echo.directories of all external libraries.  This overrides any include or library
    echo.directories set in the IDE.

    exit /b 0


rem Parse through the parameters sent to file, and set appropriate variables
:parse_params

    for %%a in (%*) do (
        if "%%a"=="vs9" (
            rem Use VS2008 as our compiler
            set h4h5_use_vs2008=true
            
        ) else if "%%a"=="nodebug" (
            rem Enable Fortran
            set blddebug=
            
        ) else if "%%a"=="useenv" (
            rem Tell the compiler to use variable settings in the environment 
            set h4h5_useenv=true
            
        ) else (
            rem Set errorlevel if we receive a bad parameter 
            echo.Unknown option: %%a
            call :help
            exit /b 1
        )
    )
    
    exit /b 0


rem Setup our environment
:setup

    echo.Setting up environment
    
    rem Sanity checks
    rem Only VS 2008 are supported
    if not "%h4h5_use_vs2008%"=="true" (
        echo.Error: Only Visual Studio 2008 are supported on Windows.
        exit /b 1
    )
    rem Make sure PROCESSOR_ARCHITECURE is set to either x86 or AMD64
    if "%PROCESSOR_ARCHITECTURE%"=="x86" (
        set h4h5_platform=Win32
    ) else if "%PROCESSOR_ARCHITECTURE%"=="AMD64" (
        set h4h5_platform=x64
    ) else (
        echo.Error: Environment variable PROCESSOR_ARCHITECTURE should contain
        echo.either x86 or AMD64
        exit /b 1
    )
    rem Make sure we have environment variables specified for our compilers
    if defined h4h5_use_vs2008 (
        rem Assume Visual Studio 2008
        if not defined vs90comntools (
            echo.Error: Cannot setup Visual Studio 2008 environment.  Please
            echo.make sure VS90COMNTOOLS is defined in the environment.
            exit /b 1
        )
    )

    rem Figure out which solution file to use based on configuration, and
    rem setup Visual Studio environment
    if defined h4h5_use_vs2008 (
        rem Assume VS2008
        echo.Using Visual Studio 2008
		if %h4h5_platform%==Win32 (
			call "%vs90comntools%\..\..\VC\vcvarsall.bat" x86
		) else (
			call "%vs90comntools%\..\..\VC\vcvarsall.bat" x86_amd64
		)
		set h4h5_sln="%CD%\windows\proj\all.sln"
    )
            
    rem See if "useenv" was specified
    if defined h4h5_useenv (
        rem This will tell Visual Studio to use include, library, etc. paths
        rem defined by %INCLUDE% %LIB%, etc.  Assume the user has already
        rem added external library paths to these variables.
        echo.Using compiler settings from environment.
        set h4h5_cflags=/useenv
    )

    exit /b 0

rem Upgrade the project files to the latest format for Visual Studio
:upgrade
    
    echo.Upgrading project files
    devenv %h4h5_sln% /Upgrade /NoLogo
    
    exit /b

rem Build the h4h5 libraries.  By default, C libraries are built.
:all

    echo.*****************************************************************************
    echo.                        Build h4h5 Libraries and Tools
    echo.*****************************************************************************
    echo.
    

    rem Build both debug and release versions
    for %%a in (%blddebug% %bldrelease%) do (
        echo.Building %h4h5_platform% %%a libraries...
        rem Modify our command-line syntax slightly for each compiler, because they
        rem each expect different parameters
        rem Assume Visual Studio 2008
        devenv %h4h5_sln% %h4h5_cflags% /Build "%%a|%h4h5_platform%"
        if !errorlevel! neq 0 ( 
            exit /b
        )
    )

    exit /b 0


rem Handle errors
:error

    rem For now, our error handling just consists of setting nerrors and calling end
    echo.h4h5 build failed.
    set /a nerrors=%nerrors%+1
    goto end

    rem We'll never really get here, but we keep this line for consistency
    exit /b

 

rem This is where the magic happens
:main

    call :parse_params %*
    if %errorlevel% neq 0 (
        echo.Error parsing parameters!
        goto error
    )
    
    call :setup
    if %errorlevel% neq 0 (
        echo.Error setting up build environment.
        goto error
    )
    
    call :all
    if %errorlevel% neq 0 (
        echo.Error building h4h5 libraries!
        goto error
    )

    if "%nerrors%"=="0" (
        echo. All h4h5 libraries and tools build successfully!
    )

    rem Fall throught to end

:end
    popd
    endlocal & exit /b %nerrors%

