/*
 *  $Id: rprofile.c 21888 2019-02-15 14:45:12Z yeti-dn $
 *  Copyright (C) 2003-2018 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwymodule/gwymodule-tool.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/datafield.h>
#include <libprocess/stats.h>
#include <libprocess/linestats.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwynullstore.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwydgetutils.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>

#define GWY_TYPE_TOOL_RPROFILE            (gwy_tool_rprofile_get_type())
#define GWY_TOOL_RPROFILE(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj), GWY_TYPE_TOOL_RPROFILE, GwyToolRprofile))
#define GWY_IS_TOOL_RPROFILE(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj), GWY_TYPE_TOOL_RPROFILE))
#define GWY_TOOL_RPROFILE_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj), GWY_TYPE_TOOL_RPROFILE, GwyToolRprofileClass))

enum {
    NLINES = 1024,
    MIN_RESOLUTION = 4,
    MAX_RESOLUTION = 16384
};

enum {
    COLUMN_I, COLUMN_X1, COLUMN_Y1, COLUMN_X2, COLUMN_Y2, NCOLUMNS
};

typedef struct _GwyToolRprofile      GwyToolRprofile;
typedef struct _GwyToolRprofileClass GwyToolRprofileClass;

typedef struct {
    gboolean options_visible;
    gint resolution;
    gboolean fixres;
    GwyMaskingType masking;
    gboolean separate;
    gboolean number_lines;
    GwyAppDataId target;
} ToolArgs;

struct _GwyToolRprofile {
    GwyPlainTool parent_instance;

    ToolArgs args;

    GtkTreeView *treeview;
    GtkTreeModel *model;

    GwyDataLine *line;
    GtkWidget *graph;
    GwyGraphModel *gmodel;
    GdkPixbuf *colorpixbuf;

    GtkWidget *options;
    GtkWidget *improve;
    GtkWidget *improve_all;
    GtkObject *resolution;
    GtkWidget *fixres;
    GtkWidget *number_lines;
    GtkWidget *separate;
    GtkWidget *apply;
    GtkWidget *target_graph;
    GSList *masking;

    /* potential class data */
    GwySIValueFormat *pixel_format;
    GType layer_type_line;
};

struct _GwyToolRprofileClass {
    GwyPlainToolClass parent_class;
};

static gboolean module_register(void);

static GType    gwy_tool_rprofile_get_type              (void)                      G_GNUC_CONST;
static void     gwy_tool_rprofile_finalize              (GObject *object);
static void     gwy_tool_rprofile_init_dialog           (GwyToolRprofile *tool);
static void     gwy_tool_rprofile_data_switched         (GwyTool *gwytool,
                                                         GwyDataView *data_view);
static void     gwy_tool_rprofile_response              (GwyTool *tool,
                                                         gint response_id);
static void     gwy_tool_rprofile_data_changed          (GwyPlainTool *plain_tool);
static void     gwy_tool_rprofile_selection_changed     (GwyPlainTool *plain_tool,
                                                         gint hint);
static void     gwy_tool_rprofile_update_symm_sensitivty(GwyToolRprofile *tool);
static void     gwy_tool_rprofile_update_curve          (GwyToolRprofile *tool,
                                                         gint i);
static void     gwy_tool_rprofile_update_all_curves     (GwyToolRprofile *tool);
static void     gwy_tool_rprofile_improve_all           (GwyToolRprofile *tool);
static void     gwy_tool_rprofile_improve               (GwyToolRprofile *tool);
static void     gwy_tool_rprofile_symmetrize_profile    (GwyToolRprofile *tool,
                                                         gint id);
static void     gwy_tool_rprofile_render_cell           (GtkCellLayout *layout,
                                                         GtkCellRenderer *renderer,
                                                         GtkTreeModel *model,
                                                         GtkTreeIter *iter,
                                                         gpointer user_data);
static void     gwy_tool_rprofile_render_color          (GtkCellLayout *layout,
                                                         GtkCellRenderer *renderer,
                                                         GtkTreeModel *model,
                                                         GtkTreeIter *iter,
                                                         gpointer user_data);
static void     gwy_tool_rprofile_options_expanded      (GtkExpander *expander,
                                                         GParamSpec *pspec,
                                                         GwyToolRprofile *tool);
static void     gwy_tool_rprofile_resolution_changed    (GwyToolRprofile *tool,
                                                         GtkAdjustment *adj);
static void     gwy_tool_rprofile_fixres_changed        (GtkToggleButton *check,
                                                         GwyToolRprofile *tool);
static void     gwy_tool_rprofile_number_lines_changed  (GtkToggleButton *check,
                                                         GwyToolRprofile *tool);
static void     gwy_tool_rprofile_separate_changed      (GtkToggleButton *check,
                                                         GwyToolRprofile *tool);
static void     gwy_tool_rprofile_update_target_graphs  (GwyToolRprofile *tool);
static gboolean filter_target_graphs                    (GwyContainer *data,
                                                         gint id,
                                                         gpointer user_data);
static void     gwy_tool_rprofile_target_changed        (GwyToolRprofile *tool);
static void     gwy_tool_rprofile_masking_changed       (GtkWidget *button,
                                                         GwyToolRprofile *tool);
static void     gwy_tool_rprofile_apply                 (GwyToolRprofile *tool);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Creates angularly averaged profile graphs."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2018",
};

static const gchar fixres_key[]          = "/module/rprofile/fixres";
static const gchar masking_key[]         = "/module/rprofile/masking";
static const gchar number_lines_key[]    = "/module/rprofile/number_lines";
static const gchar options_visible_key[] = "/module/rprofile/options_visible";
static const gchar resolution_key[]      = "/module/rprofile/resolution";
static const gchar separate_key[]        = "/module/rprofile/separate";

static const ToolArgs default_args = {
    FALSE,
    120,
    FALSE,
    GWY_MASK_IGNORE,
    FALSE,
    TRUE,
    GWY_APP_DATA_ID_NONE,
};

GWY_MODULE_QUERY2(module_info, rprofile)

G_DEFINE_TYPE(GwyToolRprofile, gwy_tool_rprofile, GWY_TYPE_PLAIN_TOOL)

static gboolean
module_register(void)
{
    gwy_tool_func_register(GWY_TYPE_TOOL_RPROFILE);

    return TRUE;
}

static void
gwy_tool_rprofile_class_init(GwyToolRprofileClass *klass)
{
    GwyPlainToolClass *ptool_class = GWY_PLAIN_TOOL_CLASS(klass);
    GwyToolClass *tool_class = GWY_TOOL_CLASS(klass);
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->finalize = gwy_tool_rprofile_finalize;

    tool_class->stock_id = GWY_STOCK_RADIAL_PROFILE;
    tool_class->title = _("Radial Profiles");
    tool_class->tooltip = _("Extract angularly averaged profiles");
    tool_class->prefix = "/module/rprofile";
    tool_class->default_width = 640;
    tool_class->default_height = 400;
    tool_class->data_switched = gwy_tool_rprofile_data_switched;
    tool_class->response = gwy_tool_rprofile_response;

    ptool_class->data_changed = gwy_tool_rprofile_data_changed;
    ptool_class->selection_changed = gwy_tool_rprofile_selection_changed;
}

static void
gwy_tool_rprofile_finalize(GObject *object)
{
    GwyToolRprofile *tool;
    GwyContainer *settings;

    tool = GWY_TOOL_RPROFILE(object);

    settings = gwy_app_settings_get();
    gwy_container_set_boolean_by_name(settings, options_visible_key,
                                      tool->args.options_visible);
    gwy_container_set_int32_by_name(settings, resolution_key,
                                    tool->args.resolution);
    gwy_container_set_boolean_by_name(settings, fixres_key,
                                      tool->args.fixres);
    gwy_container_set_enum_by_name(settings, masking_key,
                                   tool->args.masking);
    gwy_container_set_boolean_by_name(settings, separate_key,
                                      tool->args.separate);
    gwy_container_set_boolean_by_name(settings, number_lines_key,
                                      tool->args.number_lines);

    GWY_OBJECT_UNREF(tool->line);
    if (tool->model) {
        gtk_tree_view_set_model(tool->treeview, NULL);
        GWY_OBJECT_UNREF(tool->model);
    }
    GWY_OBJECT_UNREF(tool->colorpixbuf);
    GWY_OBJECT_UNREF(tool->gmodel);
    GWY_SI_VALUE_FORMAT_FREE(tool->pixel_format);
    G_OBJECT_CLASS(gwy_tool_rprofile_parent_class)->finalize(object);
}

static void
gwy_tool_rprofile_init(GwyToolRprofile *tool)
{
    GwyPlainTool *plain_tool;
    GwyContainer *settings;
    gint width, height;

    plain_tool = GWY_PLAIN_TOOL(tool);
    tool->layer_type_line = gwy_plain_tool_check_layer_type(plain_tool,
                                                            "GwyLayerLine");
    if (!tool->layer_type_line)
        return;

    plain_tool->unit_style = GWY_SI_UNIT_FORMAT_MARKUP;
    plain_tool->lazy_updates = TRUE;

    settings = gwy_app_settings_get();
    tool->args = default_args;
    gwy_container_gis_boolean_by_name(settings, options_visible_key,
                                      &tool->args.options_visible);
    gwy_container_gis_int32_by_name(settings, resolution_key,
                                    &tool->args.resolution);
    gwy_container_gis_boolean_by_name(settings, fixres_key,
                                      &tool->args.fixres);
    gwy_container_gis_enum_by_name(settings, masking_key,
                                   &tool->args.masking);
    tool->args.masking = gwy_enum_sanitize_value(tool->args.masking,
                                                 GWY_TYPE_MASKING_TYPE);
    gwy_container_gis_boolean_by_name(settings, separate_key,
                                      &tool->args.separate);
    gwy_container_gis_boolean_by_name(settings, number_lines_key,
                                      &tool->args.number_lines);

    gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, &width, &height);
    height |= 1;
    tool->colorpixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8,
                                       height, height);

    tool->pixel_format = gwy_si_unit_value_format_new(1.0, 0, _("px"));
    gwy_plain_tool_connect_selection(plain_tool, tool->layer_type_line,
                                     "line");

    gwy_tool_rprofile_init_dialog(tool);
}

static void
gwy_tool_rprofile_init_dialog(GwyToolRprofile *tool)
{
    static const gchar *column_titles[] = {
        "<b>n</b>",
        "<b>x<sub>1</sub></b>",
        "<b>y<sub>1</sub></b>",
        "<b>x<sub>2</sub></b>",
        "<b>y<sub>2</sub></b>",
    };
    GtkTreeViewColumn *column;
    GtkTreeSelection *selection;
    GtkCellRenderer *renderer;
    GtkDialog *dialog;
    GtkWidget *scwin, *label, *hbox, *vbox, *hbox2;
    GtkTable *table;
    GwyNullStore *store;
    guint i, row;

    dialog = GTK_DIALOG(GWY_TOOL(tool)->dialog);

    hbox = gtk_hbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(dialog->vbox), hbox, TRUE, TRUE, 0);

    /* Left pane */
    vbox = gtk_vbox_new(FALSE, 8);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

    /* Line coordinates */
    store = gwy_null_store_new(0);
    tool->model = GTK_TREE_MODEL(store);
    tool->treeview = GTK_TREE_VIEW(gtk_tree_view_new_with_model(tool->model));
    gwy_plain_tool_enable_object_deletion(GWY_PLAIN_TOOL(tool), tool->treeview);

    selection = gtk_tree_view_get_selection(tool->treeview);
    g_signal_connect_swapped(selection, "changed",
                             G_CALLBACK(gwy_tool_rprofile_update_symm_sensitivty),
                             tool);

    for (i = 0; i < NCOLUMNS; i++) {
        column = gtk_tree_view_column_new();
        gtk_tree_view_column_set_expand(column, TRUE);
        gtk_tree_view_column_set_alignment(column, 0.5);
        g_object_set_data(G_OBJECT(column), "id", GUINT_TO_POINTER(i));
        renderer = gtk_cell_renderer_text_new();
        g_object_set(renderer, "xalign", 1.0, NULL);
        gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(column), renderer, TRUE);
        gtk_cell_layout_set_cell_data_func(GTK_CELL_LAYOUT(column), renderer,
                                           gwy_tool_rprofile_render_cell, tool,
                                           NULL);
        if (i == COLUMN_I) {
            renderer = gtk_cell_renderer_pixbuf_new();
            g_object_set(renderer, "pixbuf", tool->colorpixbuf, NULL);
            gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(column),
                                       renderer, FALSE);
            gtk_cell_layout_set_cell_data_func(GTK_CELL_LAYOUT(column),
                                               renderer,
                                               gwy_tool_rprofile_render_color,
                                               tool,
                                               NULL);
        }

        label = gtk_label_new(NULL);
        gtk_label_set_markup(GTK_LABEL(label), column_titles[i]);
        gtk_tree_view_column_set_widget(column, label);
        gtk_widget_show(label);
        gtk_tree_view_append_column(tool->treeview, column);
    }

    scwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin),
                                   GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(scwin), GTK_WIDGET(tool->treeview));
    gtk_box_pack_start(GTK_BOX(vbox), scwin, TRUE, TRUE, 0);

    /* Options */
    tool->options = gtk_expander_new(_("<b>Options</b>"));
    gtk_expander_set_use_markup(GTK_EXPANDER(tool->options), TRUE);
    gtk_expander_set_expanded(GTK_EXPANDER(tool->options),
                              tool->args.options_visible);
    g_signal_connect(tool->options, "notify::expanded",
                     G_CALLBACK(gwy_tool_rprofile_options_expanded), tool);
    gtk_box_pack_start(GTK_BOX(vbox), tool->options, FALSE, FALSE, 0);

    table = GTK_TABLE(gtk_table_new(9, 3, FALSE));
    gtk_table_set_col_spacings(table, 6);
    gtk_table_set_row_spacings(table, 2);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_container_add(GTK_CONTAINER(tool->options), GTK_WIDGET(table));
    row = 0;

    hbox2 = gtk_hbox_new(FALSE, 2);
    gtk_table_attach(table, hbox2, 0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    tool->improve_all = gtk_button_new_with_mnemonic(_("Symmetrize _All"));
    gtk_box_pack_end(GTK_BOX(hbox2), tool->improve_all, FALSE, FALSE, 0);
    g_signal_connect_swapped(tool->improve_all, "clicked",
                             G_CALLBACK(gwy_tool_rprofile_improve_all), tool);
    tool->improve = gtk_button_new_with_mnemonic(_("S_ymmetrize"));
    gtk_box_pack_end(GTK_BOX(hbox2), tool->improve, FALSE, FALSE, 0);
    g_signal_connect_swapped(tool->improve, "clicked",
                             G_CALLBACK(gwy_tool_rprofile_improve), tool);
    row++;

    tool->resolution = gtk_adjustment_new(tool->args.resolution,
                                          MIN_RESOLUTION, MAX_RESOLUTION,
                                          1, 10, 0);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row,
                            _("_Fixed resolution:"), NULL,
                            tool->resolution, GWY_HSCALE_CHECK);
    g_signal_connect_swapped(tool->resolution, "value-changed",
                             G_CALLBACK(gwy_tool_rprofile_resolution_changed),
                             tool);
    tool->fixres = gwy_table_hscale_get_check(tool->resolution);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tool->fixres),
                                 tool->args.fixres);
    g_signal_connect(tool->fixres, "toggled",
                     G_CALLBACK(gwy_tool_rprofile_fixres_changed), tool);
    row++;

    tool->number_lines
        = gtk_check_button_new_with_mnemonic(_("_Number lines"));
    gtk_table_attach(table, tool->number_lines,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tool->number_lines),
                                 tool->args.number_lines);
    g_signal_connect(tool->number_lines, "toggled",
                     G_CALLBACK(gwy_tool_rprofile_number_lines_changed), tool);
    row++;

    tool->separate
        = gtk_check_button_new_with_mnemonic(_("_Separate profiles"));
    gtk_table_attach(table, tool->separate,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tool->separate),
                                 tool->args.separate);
    g_signal_connect(tool->separate, "toggled",
                     G_CALLBACK(gwy_tool_rprofile_separate_changed), tool);
    row++;

    tool->target_graph = gwy_data_chooser_new_graphs();
    gwy_data_chooser_set_none(GWY_DATA_CHOOSER(tool->target_graph),
                              _("New graph"));
    gwy_data_chooser_set_active(GWY_DATA_CHOOSER(tool->target_graph), NULL, -1);
    gwy_data_chooser_set_filter(GWY_DATA_CHOOSER(tool->target_graph),
                                filter_target_graphs, tool, NULL);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row, _("Target _graph:"), NULL,
                            GTK_OBJECT(tool->target_graph),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    g_signal_connect_swapped(tool->target_graph, "changed",
                             G_CALLBACK(gwy_tool_rprofile_target_changed),
                             tool);
    row++;

    gtk_table_set_row_spacing(table, row-1, 8);
    label = gwy_label_new_header(_("Masking Mode"));
    gtk_table_attach(table, label,
                     0, 3, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    tool->masking
        = gwy_radio_buttons_create(gwy_masking_type_get_enum(), -1,
                                   G_CALLBACK(gwy_tool_rprofile_masking_changed),
                                   tool,
                                   tool->args.masking);
    row = gwy_radio_buttons_attach_to_table(tool->masking, table, 2, row);

    tool->gmodel = gwy_graph_model_new();
    g_object_set(tool->gmodel, "title", _("Radial profiles"), NULL);

    tool->graph = gwy_graph_new(tool->gmodel);
    gwy_graph_enable_user_input(GWY_GRAPH(tool->graph), FALSE);
    g_object_set(tool->gmodel, "label-visible", FALSE, NULL);
    gtk_box_pack_start(GTK_BOX(hbox), tool->graph, TRUE, TRUE, 2);

    gwy_plain_tool_add_clear_button(GWY_PLAIN_TOOL(tool));
    gwy_tool_add_hide_button(GWY_TOOL(tool), FALSE);
    tool->apply = gtk_dialog_add_button(dialog, GTK_STOCK_APPLY,
                                        GTK_RESPONSE_APPLY);
    gtk_dialog_set_default_response(dialog, GTK_RESPONSE_APPLY);
    gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_APPLY, FALSE);
    gwy_help_add_to_tool_dialog(dialog, GWY_TOOL(tool), GWY_HELP_NO_BUTTON);

    gtk_widget_show_all(dialog->vbox);
}

static void
gwy_tool_rprofile_data_switched(GwyTool *gwytool, GwyDataView *data_view)
{
    GwyPlainTool *plain_tool;
    GwyToolRprofile *tool;
    gboolean ignore;

    plain_tool = GWY_PLAIN_TOOL(gwytool);
    ignore = (data_view == plain_tool->data_view);

    GWY_TOOL_CLASS(gwy_tool_rprofile_parent_class)->data_switched(gwytool,
                                                                 data_view);

    if (ignore || plain_tool->init_failed)
        return;

    tool = GWY_TOOL_RPROFILE(gwytool);
    if (data_view) {
        gwy_object_set_or_reset(plain_tool->layer,
                                tool->layer_type_line,
                                "line-numbers", tool->args.number_lines,
                                "thickness", 1,
                                "center-tick", TRUE,
                                "editable", TRUE,
                                "focus", -1,
                                NULL);
        gwy_selection_set_max_objects(plain_tool->selection, NLINES);
    }

    gwy_graph_model_remove_all_curves(tool->gmodel);
    gwy_tool_rprofile_update_all_curves(tool);
    gwy_tool_rprofile_update_target_graphs(tool);
}

static void
gwy_tool_rprofile_response(GwyTool *tool,
                           gint response_id)
{
    GWY_TOOL_CLASS(gwy_tool_rprofile_parent_class)->response(tool, response_id);

    if (response_id == GTK_RESPONSE_APPLY)
        gwy_tool_rprofile_apply(GWY_TOOL_RPROFILE(tool));
}

static void
gwy_tool_rprofile_data_changed(GwyPlainTool *plain_tool)
{
    gwy_tool_rprofile_update_all_curves(GWY_TOOL_RPROFILE(plain_tool));
    gwy_tool_rprofile_update_target_graphs(GWY_TOOL_RPROFILE(plain_tool));
}

static void
gwy_tool_rprofile_selection_changed(GwyPlainTool *plain_tool,
                                    gint hint)
{
    GwyToolRprofile *tool = GWY_TOOL_RPROFILE(plain_tool);
    GtkDialog *dialog = GTK_DIALOG(GWY_TOOL(tool)->dialog);
    GwyNullStore *store;
    gint n;

    store = GWY_NULL_STORE(tool->model);
    n = gwy_null_store_get_n_rows(store);
    g_return_if_fail(hint <= n);

    if (hint < 0) {
        gtk_tree_view_set_model(tool->treeview, NULL);
        if (plain_tool->selection)
            n = gwy_selection_get_data(plain_tool->selection, NULL);
        else
            n = 0;
        gwy_null_store_set_n_rows(store, n);
        gtk_tree_view_set_model(tool->treeview, tool->model);
        gwy_graph_model_remove_all_curves(tool->gmodel);
        gwy_tool_rprofile_update_all_curves(tool);
    }
    else {
        GtkTreeSelection *selection;
        GtkTreePath *path;
        GtkTreeIter iter;

        if (hint < n)
            gwy_null_store_row_changed(store, hint);
        else
            gwy_null_store_set_n_rows(store, n+1);
        gwy_tool_rprofile_update_curve(tool, hint);
        n++;

        gtk_tree_model_iter_nth_child(tool->model, &iter, NULL, hint);
        path = gtk_tree_model_get_path(tool->model, &iter);
        selection = gtk_tree_view_get_selection(tool->treeview);
        gtk_tree_selection_select_iter(selection, &iter);
        gtk_tree_view_scroll_to_cell(tool->treeview, path, NULL,
                                     FALSE, 0.0, 0.0);
        gtk_tree_path_free(path);
    }

    gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_APPLY, n > 0);
}

static void
gwy_tool_rprofile_update_symm_sensitivty(GwyToolRprofile *tool)
{
    GtkTreeSelection *selection;
    gboolean is_selected, has_lines;
    GtkTreeModel *model;
    GtkTreeIter iter;

    selection = gtk_tree_view_get_selection(tool->treeview);
    is_selected = gtk_tree_selection_get_selected(selection, &model, &iter);
    has_lines = (model && gtk_tree_model_iter_n_children(model, NULL) > 0);

    gtk_widget_set_sensitive(tool->improve, is_selected);
    gtk_widget_set_sensitive(tool->improve_all, has_lines);
}

static void
gwy_tool_rprofile_update_curve(GwyToolRprofile *tool, gint i)
{
    GwyPlainTool *plain_tool;
    GwyGraphCurveModel *gcmodel;
    gdouble xc, yc, r, line[4];
    gint xl1, yl1, xl2, yl2;
    gint n, lineres;
    gchar *desc;
    const GwyRGBA *color;
    GwyDataField *data_field, *mask;

    plain_tool = GWY_PLAIN_TOOL(tool);
    g_return_if_fail(plain_tool->selection);
    g_return_if_fail(gwy_selection_get_object(plain_tool->selection, i, line));
    data_field = plain_tool->data_field;
    mask = plain_tool->mask_field;

    xl1 = floor(gwy_data_field_rtoj(data_field, line[0]));
    yl1 = floor(gwy_data_field_rtoi(data_field, line[1]));
    xl2 = floor(gwy_data_field_rtoj(data_field, line[2]));
    yl2 = floor(gwy_data_field_rtoi(data_field, line[3]));
    if (!tool->args.fixres) {
        lineres = GWY_ROUND(hypot(abs(xl1 - xl2) + 1, abs(yl1 - yl2) + 1));
        lineres = MAX(lineres, MIN_RESOLUTION);
    }
    else
        lineres = tool->args.resolution;

    xc = 0.5*(line[0] + line[2]) + data_field->xoff;
    yc = 0.5*(line[1] + line[3]) + data_field->yoff;
    r = 0.5*hypot(line[2] - line[0], line[3] - line[1]);

    /* Just create some line when there is none. */
    if (!tool->line)
        tool->line = gwy_data_line_new(1, 1.0, FALSE);
    r = MAX(r, hypot(gwy_data_field_get_dx(data_field),
                     gwy_data_field_get_dy(data_field)));
    gwy_data_field_angular_average(data_field, tool->line, mask,
                                   tool->args.masking,
                                   xc, yc, r, lineres);

    n = gwy_graph_model_get_n_curves(tool->gmodel);
    if (i < n) {
        gcmodel = gwy_graph_model_get_curve(tool->gmodel, i);
        gwy_graph_curve_model_set_data_from_dataline(gcmodel, tool->line, 0, 0);
    }
    else {
        gcmodel = gwy_graph_curve_model_new();
        desc = g_strdup_printf(_("Radial profile %d"), i+1);
        color = gwy_graph_get_preset_color(i);
        g_object_set(gcmodel,
                     "mode", GWY_GRAPH_CURVE_LINE,
                     "description", desc,
                     "color", color,
                     NULL);
        g_free(desc);
        gwy_graph_curve_model_set_data_from_dataline(gcmodel, tool->line, 0, 0);
        gwy_graph_model_add_curve(tool->gmodel, gcmodel);
        g_object_unref(gcmodel);

        if (i == 0) {
            GwySIUnit *xunit, *yunit;

            xunit = gwy_data_field_get_si_unit_xy(plain_tool->data_field);
            xunit = gwy_si_unit_duplicate(xunit);
            yunit = gwy_data_field_get_si_unit_z(plain_tool->data_field);
            yunit = gwy_si_unit_duplicate(yunit);
            g_object_set(tool->gmodel,
                         "si-unit-x", xunit,
                         "si-unit-y", yunit,
                         NULL);
            g_object_unref(xunit);
            g_object_unref(yunit);

            gwy_tool_rprofile_update_target_graphs(tool);
        }
    }
}

static void
gwy_tool_rprofile_improve(GwyToolRprofile *tool)
{
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    GtkTreePath *path;
    GtkTreeIter iter;
    const gint *indices;

    selection = gtk_tree_view_get_selection(tool->treeview);
    if (!gtk_tree_selection_get_selected(selection, &model, &iter))
        return;

    path = gtk_tree_model_get_path(model, &iter);
    indices = gtk_tree_path_get_indices(path);
    gwy_tool_rprofile_symmetrize_profile(tool, indices[0]);
    gtk_tree_path_free(path);
}

static void
gwy_tool_rprofile_improve_all(GwyToolRprofile *tool)
{
    GwyPlainTool *plain_tool;
    gint n, i;

    plain_tool = GWY_PLAIN_TOOL(tool);
    if (!plain_tool->selection
        || !(n = gwy_selection_get_data(plain_tool->selection, NULL)))
        return;

    for (i = 0; i < n; i++) {
        gwy_tool_rprofile_symmetrize_profile(tool, i);
    }
}

static void
gwy_tool_rprofile_update_all_curves(GwyToolRprofile *tool)
{
    GwyPlainTool *plain_tool;
    gint n, i;

    plain_tool = GWY_PLAIN_TOOL(tool);
    if (!plain_tool->selection
        || !(n = gwy_selection_get_data(plain_tool->selection, NULL))) {
        gwy_graph_model_remove_all_curves(tool->gmodel);
        return;
    }

    for (i = 0; i < n; i++)
        gwy_tool_rprofile_update_curve(tool, i);
}

static gdouble
estimate_angular_variation(GwyDataField *dfield, GwyDataLine *tmp,
                           const gdouble *line)
{
    gdouble xc, yc, r, dx, dy, h;
    gdouble *s, *s2;
    gdouble n, variation = 0.0;
    gint ir, ndirs, res, i, j;

    /* Ignore offsets here we do not call any function that uses them. */
    xc = 0.5*(line[0] + line[2]);
    yc = 0.5*(line[1] + line[3]);
    r = 0.5*hypot(line[2] - line[0], line[3] - line[1]);
    /* The profile cannot go outside the field anywhere. */
    r = MIN(r, MIN(xc, yc));
    r = MIN(r, MIN(dfield->xreal - xc, dfield->yreal - yc));

    dx = gwy_data_field_get_dx(dfield);
    dy = gwy_data_field_get_dy(dfield);
    h = 2.0*dx*dy/(dx + dy);
    ir = GWY_ROUND(r/h);

    if (ir < 1)
        return 0.0;

    res = 2*ir + 1;
    ndirs = 2*(GWY_ROUND(log(ir)) + 1);
    g_assert(ndirs >= 2);

    s = g_new0(gdouble, res);
    s2 = g_new0(gdouble, res);

    for (i = 0; i < ndirs; i++) {
        gdouble si = sin(G_PI*i/ndirs), ci = cos(G_PI*i/ndirs);
        gint xl1 = floor(gwy_data_field_rtoj(dfield, xc + ci*r));
        gint yl1 = floor(gwy_data_field_rtoi(dfield, yc + si*r));
        gint xl2 = floor(gwy_data_field_rtoj(dfield, xc - ci*r));
        gint yl2 = floor(gwy_data_field_rtoi(dfield, yc - si*r));

        xl1 = CLAMP(xl1, 0, dfield->xres-1);
        yl1 = CLAMP(yl1, 0, dfield->yres-1);
        xl2 = CLAMP(xl2, 0, dfield->xres-1);
        yl2 = CLAMP(yl2, 0, dfield->yres-1);
        gwy_data_field_get_profile(dfield, tmp, xl1, yl1, xl2, yl2,
                                   res, 1, GWY_INTERPOLATION_LINEAR);
        if (tmp->res != res) {
            g_warning("Cannot get profile of length exactly %d.", res);
            goto fail;
        }

        for (j = 0; j < res; j++) {
            gdouble v1 = tmp->data[j], v2 = tmp->data[res-1 - j];
            s[j] += v1 + v2;
            s2[j] += v1*v1 + v2*v2;
        }
    }

    n = 2.0*ndirs;
    for (j = 0; j < res; j++)
        variation += s2[j]/n - s[j]*s[j]/(n*n);
    variation /= res;

fail:
    g_free(s2);
    g_free(s);

    return variation;
}

static gboolean
symmetrize_at_scale(GwyDataField *dfield,
                    GwyDataLine *tmpline,
                    gdouble r,
                    gdouble *line)
{
    gdouble xreal = dfield->xreal, yreal = dfield->yreal;
    gint i, j, besti = 0, bestj = 0;
    gdouble dx = gwy_data_field_get_dx(dfield);
    gdouble dy = gwy_data_field_get_dy(dfield);
    gdouble allvar[25], z[9];
    gdouble var, bestvar = G_MAXDOUBLE;
    gboolean ok;

    for (i = -2; i <= 2; i++) {
        for (j = -2; j <= 2; j++) {
            gdouble offline[4] = {
                line[0] + j*r, line[1] + i*r,
                line[2] + j*r, line[3] + i*r,
            };

            allvar[5*(i + 2) + (j + 2)] = G_MAXDOUBLE;
            if (offline[0] < 0.0 || offline[2] > xreal
                || offline[1] < 0.0 || offline[3] > yreal)
                continue;

            var = estimate_angular_variation(dfield, tmpline, offline);
            allvar[5*(i + 2) + (j + 2)] = var;
            if (i*j + j*j <= 5 && var < bestvar) {
                besti = i;
                bestj = j;
                bestvar = var;
            }
        }
    }

    line[0] += bestj*r;
    line[1] += besti*r;
    line[2] += bestj*r;
    line[3] += besti*r;

    /* If steps are large, just choose the maximum and continue. */
    if (r > 0.7 * 2*dx*dy/(dx + dy)) {
        gwy_debug("symmetrize at scale %g: (%d,%d)", r, bestj, besti);
        return FALSE;
    }

    /* When we get to the smallest possible discrete scale, attempt subpixel
     * improvement if the maximum does not lie on a border. */
    ok = TRUE;
    for (i = -1; i <= 1; i++) {
        if (!ok || ABS(besti + i) > 2) {
            ok = FALSE;
            break;
        }
        for (j = -1; j <= 1; j++) {
            if (ABS(bestj + j) > 2) {
                ok = FALSE;
                break;
            }

            var = allvar[5*(besti + i + 2) + (bestj + j + 2)];
            if (var >= G_MAXDOUBLE) {
                ok = FALSE;
                break;
            }
            z[(i + 1)*3 + (j + 1)] = var;
        }
    }
    if (ok) {
        gdouble x, y;
        gwy_math_refine_maximum_2d(z, &x, &y);
        gwy_debug("subpixel refinement (%g,%g)", x, y);
        line[0] += x*r;
        line[1] += y*r;
        line[2] += x*r;
        line[3] += y*r;
    }

    return TRUE;
}

static void
gwy_tool_rprofile_symmetrize_profile(GwyToolRprofile *tool, gint id)
{
    GwyPlainTool *plain_tool;
    GwyDataField *dfield;
    GwyDataLine *tmpline;
    gdouble line[4];
    gdouble r, dx, dy;

    plain_tool = GWY_PLAIN_TOOL(tool);
    g_return_if_fail(plain_tool->selection);
    g_return_if_fail(gwy_selection_get_object(plain_tool->selection, id, line));
    dfield = plain_tool->data_field;
    dx = gwy_data_field_get_dx(dfield);
    dy = gwy_data_field_get_dy(dfield);

    /* Don't attempt to optimise very short lines. It would end up in tears. */
    if (hypot((line[2] - line[0])/dx, (line[3] - line[1])/dy) < 4.0)
        return;

    tmpline = gwy_data_line_new(1, 1.0, FALSE);
    r = 0.07*hypot(line[2] - line[0], line[3] - line[1]);
    while (!symmetrize_at_scale(dfield, tmpline, r, line))
        r *= 0.5;

    gwy_selection_set_object(plain_tool->selection, id, line);

    g_object_unref(tmpline);
}

static void
gwy_tool_rprofile_render_cell(GtkCellLayout *layout,
                              GtkCellRenderer *renderer,
                              GtkTreeModel *model,
                              GtkTreeIter *iter,
                              gpointer user_data)
{
    GwyToolRprofile *tool = (GwyToolRprofile*)user_data;
    GwyPlainTool *plain_tool;
    const GwySIValueFormat *vf;
    gchar buf[32];
    gdouble line[4];
    gdouble val;
    guint idx, id;

    id = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(layout), "id"));
    gtk_tree_model_get(model, iter, 0, &idx, -1);
    if (id == COLUMN_I) {
        g_snprintf(buf, sizeof(buf), "%d", idx + 1);
        g_object_set(renderer, "text", buf, NULL);
        return;
    }

    plain_tool = GWY_PLAIN_TOOL(tool);
    gwy_selection_get_object(plain_tool->selection, idx, line);

    vf = tool->pixel_format;
    switch (id) {
        case COLUMN_X1:
        val = floor(gwy_data_field_rtoj(plain_tool->data_field, line[0]));
        break;

        case COLUMN_Y1:
        val = floor(gwy_data_field_rtoi(plain_tool->data_field, line[1]));
        break;

        case COLUMN_X2:
        val = floor(gwy_data_field_rtoj(plain_tool->data_field, line[2]));
        break;

        case COLUMN_Y2:
        val = floor(gwy_data_field_rtoi(plain_tool->data_field, line[3]));
        break;

        default:
        g_return_if_reached();
        break;
    }

    if (vf)
        g_snprintf(buf, sizeof(buf), "%.*f", vf->precision, val/vf->magnitude);
    else
        g_snprintf(buf, sizeof(buf), "%.3g", val);

    g_object_set(renderer, "text", buf, NULL);
}

static void
gwy_tool_rprofile_render_color(G_GNUC_UNUSED GtkCellLayout *layout,
                               G_GNUC_UNUSED GtkCellRenderer *renderer,
                               GtkTreeModel *model,
                               GtkTreeIter *iter,
                               gpointer user_data)
{
    GwyToolRprofile *tool = (GwyToolRprofile*)user_data;
    GwyGraphCurveModel *gcmodel;
    GwyRGBA *rgba;
    guint idx, pixel;

    gtk_tree_model_get(model, iter, 0, &idx, -1);
    gcmodel = gwy_graph_model_get_curve(tool->gmodel, idx);
    g_object_get(gcmodel, "color", &rgba, NULL);
    pixel = 0xff | gwy_rgba_to_pixbuf_pixel(rgba);
    gwy_rgba_free(rgba);
    gdk_pixbuf_fill(tool->colorpixbuf, pixel);
}

static void
gwy_tool_rprofile_options_expanded(GtkExpander *expander,
                                   G_GNUC_UNUSED GParamSpec *pspec,
                                   GwyToolRprofile *tool)
{
    tool->args.options_visible = gtk_expander_get_expanded(expander);
}

static void
gwy_tool_rprofile_resolution_changed(GwyToolRprofile *tool,
                                     GtkAdjustment *adj)
{
    tool->args.resolution = gwy_adjustment_get_int(adj);
    /* Resolution can be changed only when fixres == TRUE */
    gwy_tool_rprofile_update_all_curves(tool);
}

static void
gwy_tool_rprofile_fixres_changed(GtkToggleButton *check,
                                 GwyToolRprofile *tool)
{
    tool->args.fixres = gtk_toggle_button_get_active(check);
    gwy_tool_rprofile_update_all_curves(tool);
}

static void
gwy_tool_rprofile_number_lines_changed(GtkToggleButton *check,
                                       GwyToolRprofile *tool)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);

    tool->args.number_lines = gtk_toggle_button_get_active(check);
    if (plain_tool->layer) {
        g_object_set(plain_tool->layer,
                     "line-numbers", tool->args.number_lines,
                     NULL);
    }
}

static void
gwy_tool_rprofile_separate_changed(GtkToggleButton *check,
                                   GwyToolRprofile *tool)
{
    tool->args.separate = gtk_toggle_button_get_active(check);
    gwy_table_hscale_set_sensitive(GTK_OBJECT(tool->target_graph),
                                   !tool->args.separate);
    if (tool->args.separate)
        gwy_data_chooser_set_active(GWY_DATA_CHOOSER(tool->target_graph),
                                    NULL, -1);
}

static void
gwy_tool_rprofile_update_target_graphs(GwyToolRprofile *tool)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(tool->target_graph);
    gwy_data_chooser_refilter(chooser);
}

static gboolean
filter_target_graphs(GwyContainer *data, gint id, gpointer user_data)
{
    GwyToolRprofile *tool = (GwyToolRprofile*)user_data;
    GwyGraphModel *gmodel, *targetgmodel;
    GQuark quark = gwy_app_get_graph_key_for_id(id);

    return ((gmodel = tool->gmodel)
            && gwy_container_gis_object(data, quark, (GObject**)&targetgmodel)
            && gwy_graph_model_units_are_compatible(gmodel, targetgmodel));
}

static void
gwy_tool_rprofile_target_changed(GwyToolRprofile *tool)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(tool->target_graph);
    gwy_data_chooser_get_active_id(chooser, &tool->args.target);
}

static void
gwy_tool_rprofile_masking_changed(GtkWidget *button,
                                  GwyToolRprofile *tool)
{
    GwyPlainTool *plain_tool;

    if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))
        return;

    plain_tool = GWY_PLAIN_TOOL(tool);
    tool->args.masking = gwy_radio_button_get_value(button);
    if (plain_tool->data_field && plain_tool->mask_field)
        gwy_tool_rprofile_update_all_curves(tool);
}

static void
gwy_tool_rprofile_apply(GwyToolRprofile *tool)
{
    GwyPlainTool *plain_tool;
    GwyGraphCurveModel *gcmodel;
    GwyGraphModel *gmodel;
    gchar *s;
    gint i, n;

    plain_tool = GWY_PLAIN_TOOL(tool);
    g_return_if_fail(plain_tool->selection);
    n = gwy_selection_get_data(plain_tool->selection, NULL);
    g_return_if_fail(n);

    if (tool->args.target.datano) {
        GwyContainer *data = gwy_app_data_browser_get(tool->args.target.datano);
        GQuark quark = gwy_app_get_graph_key_for_id(tool->args.target.id);
        gmodel = gwy_container_get_object(data, quark);
        g_return_if_fail(gmodel);
        gwy_graph_model_append_curves(gmodel, tool->gmodel, 1);
        return;
    }

    if (!tool->args.separate) {
        gmodel = gwy_graph_model_duplicate(tool->gmodel);
        g_object_set(gmodel, "label-visible", TRUE, NULL);
        gwy_app_data_browser_add_graph_model(gmodel, plain_tool->container,
                                             TRUE);
        g_object_unref(gmodel);
        return;
    }

    for (i = 0; i < n; i++) {
        gmodel = gwy_graph_model_new_alike(tool->gmodel);
        g_object_set(gmodel, "label-visible", TRUE, NULL);
        gcmodel = gwy_graph_model_get_curve(tool->gmodel, i);
        gcmodel = gwy_graph_curve_model_duplicate(gcmodel);

        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
        g_object_get(gcmodel, "description", &s, NULL);
        g_object_set(gmodel, "title", s, NULL);
        g_free(s);
        gwy_app_data_browser_add_graph_model(gmodel, plain_tool->container,
                                             TRUE);
        g_object_unref(gmodel);
    }
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
