/*
 *  $Id: facet_measure.c 21867 2019-01-31 17:14:19Z yeti-dn $
 *  Copyright (C) 2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

/**
 * Facet (angle) view uses a zoomed area-preserving projection of north
 * hemisphere normal.  Coordinates on hemisphere are labeled (theta, phi),
 * coordinates on the projection (x, y)
 **/

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libgwyddion/gwyddion.h>
#include <libprocess/stats.h>
#include <libprocess/level.h>
#include <libprocess/filters.h>
#include <libprocess/grains.h>
#include <libprocess/elliptic.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwynullstore.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>
#include "preview.h"

#define FACET_MEASURE_RUN_MODES (GWY_RUN_INTERACTIVE)

#define FVIEW_GRADIENT "DFit"

enum {
    MAX_PLANE_SIZE = 7,  /* this is actually half */
    FACETVIEW_SIZE = (PREVIEW_SMALL_SIZE + PREVIEW_HALF_SIZE)/2 | 1,
};

enum {
    FACET_COLUMN_N,
    FACET_COLUMN_NPOINTS,
    FACET_COLUMN_TOL,
    FACET_COLUMN_THETA,
    FACET_COLUMN_PHI,
    FACET_COLUMN_X,
    FACET_COLUMN_Y,
    FACET_COLUMN_Z,
    FACET_COLUMN_ERROR,
};

typedef struct {
    gdouble tolerance;
    gdouble theta;
    gdouble phi;
    GwyXYZ v;
    gdouble error;
    guint npoints;
} FacetMeasurement;

typedef struct {
    gdouble tolerance;
    gint kernel_size;
    gboolean combine;
    GwyMergeType combine_type;
    GwyResultsReportType report_style;
    gdouble theta0;
    gdouble phi0;
} FacetMeasureArgs;

typedef struct {
    FacetMeasureArgs *args;
    GtkWidget *dialogue;
    GtkWidget *view;
    GtkWidget *fview;
    GwySelection *fselection;
    GwySelection *fselection0;
    GwySelection *iselection;
    GwyNullStore *store;
    GArray *measured_data;
    GtkWidget *pointlist;
    GtkWidget *rexport;
    GtkWidget *delete;
    GtkWidget *theta_min_label;
    GtkWidget *theta_0_label;
    GtkWidget *theta_max_label;
    GtkObject *tolerance;
    GtkObject *kernel_size;
    GtkWidget *combine;
    GSList *combine_type;
    GtkWidget *color_button;
    GtkWidget *theta_label;
    GtkWidget *phi_label;
    GwyContainer *mydata;
    GwyContainer *fdata;
    gdouble q;
    gint selid;
    gboolean in_update;
} FacetMeasureControls;

static gboolean module_register                  (void);
static void     facet_measure                    (GwyContainer *data,
                                                  GwyRunType run);
static void     facet_measure_dialogue           (FacetMeasureArgs *args,
                                                  GwyContainer *data,
                                                  GwyDataField *dfield,
                                                  GwyDataField *mfield,
                                                  gint id,
                                                  GQuark mquark);
static void     create_point_list                (FacetMeasureControls *controls);
static void     point_list_selection_changed     (GtkTreeSelection *treesel,
                                                  FacetMeasureControls *controls);
static void     render_id                        (GtkCellLayout *layout,
                                                  GtkCellRenderer *renderer,
                                                  GtkTreeModel *model,
                                                  GtkTreeIter *iter,
                                                  gpointer user_data);
static void     render_npoints                   (GtkCellLayout *layout,
                                                  GtkCellRenderer *renderer,
                                                  GtkTreeModel *model,
                                                  GtkTreeIter *iter,
                                                  gpointer user_data);
static void     render_facet_angle               (GtkCellLayout *layout,
                                                  GtkCellRenderer *renderer,
                                                  GtkTreeModel *model,
                                                  GtkTreeIter *iter,
                                                  gpointer user_data);
static void     render_facet_coordinate          (GtkCellLayout *layout,
                                                  GtkCellRenderer *renderer,
                                                  GtkTreeModel *model,
                                                  GtkTreeIter *iter,
                                                  gpointer user_data);
static void     clear_measurements               (FacetMeasureControls *controls);
static void     delete_measurement               (FacetMeasureControls *controls);
static void     refine_facet                     (FacetMeasureControls *controls);
static void     mark_facet                       (FacetMeasureControls *controls);
static void     measure_facet                    (FacetMeasureControls *controls);
static gboolean point_list_key_pressed           (FacetMeasureControls *controls,
                                                  GdkEventKey *event);
static void     run_noninteractive               (FacetMeasureArgs *args,
                                                  GwyContainer *data,
                                                  GwyDataField *dtheta,
                                                  GwyDataField *dphi,
                                                  GwyDataField *dfield,
                                                  GwyDataField *mfield,
                                                  GQuark mquark,
                                                  gdouble theta,
                                                  gdouble phi);
static void     report_style_changed             (FacetMeasureControls *controls,
                                                  GwyResultsExport *rexport);
static void     copy_facet_table                 (FacetMeasureControls *controls);
static void     save_facet_table                 (FacetMeasureControls *controls);
static void     kernel_size_changed              (GtkAdjustment *adj,
                                                  FacetMeasureControls *controls);
static void     update_theta_range               (FacetMeasureControls *controls);
static void     facet_view_select_angle          (FacetMeasureControls *controls,
                                                  gdouble theta,
                                                  gdouble phi);
static void     facet_view_selection_updated     (GwySelection *selection,
                                                  gint hint,
                                                  FacetMeasureControls *controls);
static void     preview_selection_updated        (GwySelection *selection,
                                                  gint id,
                                                  FacetMeasureControls *controls);
static void     combine_changed                  (FacetMeasureControls *controls,
                                                  GtkToggleButton *toggle);
static void     combine_type_changed             (FacetMeasureControls *controls);
static void     gwy_data_field_mark_facets       (GwyDataField *dtheta,
                                                  GwyDataField *dphi,
                                                  gdouble theta0,
                                                  gdouble phi0,
                                                  gdouble tolerance,
                                                  GwyDataField *mask);
static void     calculate_average_angle          (GwyDataField *dtheta,
                                                  GwyDataField *dphi,
                                                  gdouble theta0,
                                                  gdouble phi0,
                                                  gdouble tolerance,
                                                  FacetMeasurement *fmeas);
static gdouble  gwy_data_field_facet_distribution(GwyDataField *dfield,
                                                  gint half_size,
                                                  GwyContainer *container);
static void     compute_slopes                   (GwyDataField *dfield,
                                                  gint kernel_size,
                                                  GwyDataField *xder,
                                                  GwyDataField *yder);
static void     tolerance_changed                (FacetMeasureControls *controls,
                                                  GtkAdjustment *adj);
static void     add_mask_field                   (GwyDataView *view,
                                                  const GwyRGBA *color);
static void     facet_measure_mark_fdata         (GwyDataField *mask,
                                                  gdouble q,
                                                  gdouble theta0,
                                                  gdouble phi0,
                                                  gdouble tolerance);
static void     facet_measure_load_args          (GwyContainer *container,
                                                  FacetMeasureArgs *args);
static void     facet_measure_save_args          (GwyContainer *container,
                                                  const FacetMeasureArgs *args);
static void     make_unit_vector                 (GwyXYZ *v,
                                                  gdouble theta,
                                                  gdouble phi);
static void     vector_angles                    (const GwyXYZ *v,
                                                  gdouble *theta,
                                                  gdouble *phi);

static const FacetMeasureArgs facet_measure_defaults = {
    3.0*G_PI/180.0,
    3,
    FALSE, GWY_MERGE_UNION,
    GWY_RESULTS_REPORT_TABSEP,
    0.0, 0.0,
};

static const GwyRGBA mask_color = { 0.56, 0.39, 0.07, 0.5 };

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Visualizes, marks and measures facet orientation."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2019",
};

GWY_MODULE_QUERY2(module_info, facet_measure)

static gboolean
module_register(void)
{
    gwy_process_func_register("facet_measure",
                              (GwyProcessFunc)&facet_measure,
                              N_("/_Statistics/Facet _Measurement..."),
                              NULL,
                              FACET_MEASURE_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Measure facet angles"));

    return TRUE;
}


static void
facet_measure(GwyContainer *data, GwyRunType run)
{
    FacetMeasureArgs args;
    GwyDataField *dfield, *mfield;
    GQuark mquark;
    gint id;

    g_return_if_fail(run & FACET_MEASURE_RUN_MODES);
    g_return_if_fail(g_type_from_name("GwyLayerPoint"));

    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_MASK_FIELD_KEY, &mquark,
                                     GWY_APP_MASK_FIELD, &mfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(dfield && mquark);

    if (!gwy_si_unit_equal(gwy_data_field_get_si_unit_xy(dfield),
                           gwy_data_field_get_si_unit_z(dfield))) {
        GtkWidget *dialogue;

        dialogue = gtk_message_dialog_new
                                 (gwy_app_find_window_for_channel(data, id),
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_ERROR,
                                  GTK_BUTTONS_OK,
                                  _("%s: Lateral dimensions and value must "
                                    "be the same physical quantity."),
                                  _("Measure Facets"));
        gtk_dialog_run(GTK_DIALOG(dialogue));
        gtk_widget_destroy(dialogue);
        return;
    }

    facet_measure_load_args(gwy_app_settings_get(), &args);
    facet_measure_dialogue(&args, data, dfield, mfield, id, mquark);
    facet_measure_save_args(gwy_app_settings_get(), &args);
}

static void
facet_measure_dialogue(FacetMeasureArgs *args,
                       GwyContainer *data,
                       GwyDataField *dfield,
                       GwyDataField *mfield,
                       gint id,
                       GQuark mquark)
{
    GtkWidget *dialogue, *hbox, *vbox, *label, *thetabox, *scwin,
              *button, *scale;
    GtkTable *table;
    FacetMeasureControls controls;
    GtkSizeGroup *sizegroup;
    GwyResultsExport *rexport;
    GwyDataField *dtheta, *dphi;
    gint row, response;
    GwySelection *selection;
    gchar *selkey;
    gboolean restored_selection = FALSE;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.selid = -1;
    dialogue = gtk_dialog_new_with_buttons(_("Measure Facets"),
                                           NULL,
                                           GTK_DIALOG_DESTROY_WITH_PARENT,
                                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                           GTK_STOCK_OK, GTK_RESPONSE_OK,
                                           NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialogue), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);
    controls.dialogue = dialogue;

    /* Shallow-copy stuff to temporary container */
    controls.fdata = gwy_container_new();
    controls.mydata = gwy_container_new();
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);
    gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                            GWY_DATA_ITEM_PALETTE,
                            GWY_DATA_ITEM_RANGE,
                            GWY_DATA_ITEM_MASK_COLOR,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.q = gwy_data_field_facet_distribution(dfield, args->kernel_size,
                                                   controls.fdata);

    /* First row: Image + Options **/
    hbox = gtk_hbox_new(FALSE, 4);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), hbox,
                       FALSE, FALSE, 2);

    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE, TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), controls.view, FALSE, FALSE, 4);
    selection = create_vector_layer(GWY_DATA_VIEW(controls.view), 0, "Point",
                                    TRUE);
    controls.iselection = selection;
    g_signal_connect(selection, "changed",
                     G_CALLBACK(preview_selection_updated), &controls);

    table = GTK_TABLE(gtk_table_new(3 + 1*(!!mfield), 3, FALSE));
    gtk_table_set_row_spacings(table, 2);
    gtk_table_set_col_spacings(table, 6);
    gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(table), TRUE, TRUE, 4);
    row = 0;

    controls.kernel_size = gtk_adjustment_new(args->kernel_size,
                                              0.0, MAX_PLANE_SIZE, 1.0, 1.0,
                                              0);
    gwy_table_attach_adjbar(GTK_WIDGET(table),
                            row++, _("Facet plane size:"), _("px"),
                            controls.kernel_size,
                            GWY_HSCALE_LINEAR | GWY_HSCALE_SNAP);
    g_signal_connect(controls.kernel_size, "value-changed",
                     G_CALLBACK(kernel_size_changed), &controls);

    controls.tolerance = gtk_adjustment_new(args->tolerance*180.0/G_PI,
                                             0.0, 30.0, 0.01, 0.1, 0);
    scale = gwy_table_attach_adjbar(GTK_WIDGET(table),
                                    row++, _("_Tolerance:"), _("deg"),
                                    controls.tolerance, GWY_HSCALE_SQRT);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(scale), 3);
    g_signal_connect_swapped(controls.tolerance, "value-changed",
                             G_CALLBACK(tolerance_changed), &controls);

    gtk_table_set_row_spacing(table, row-1, 8);
    label = gtk_label_new(_("Selected ϑ:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(table, label, 0, 1, row, row+1, GTK_FILL, 0, 0, 0);

    controls.theta_label = label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gtk_table_attach(table, label, 1, 2, row, row+1, GTK_FILL, 0, 0, 0);

    label = gtk_label_new(_("deg"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(table, label, 2, 3, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    label = gtk_label_new(_("Selected φ:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(table, label, 0, 1, row, row+1, GTK_FILL, 0, 0, 0);

    controls.phi_label = label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gtk_table_attach(table, label, 1, 2, row, row+1, GTK_FILL, 0, 0, 0);

    label = gtk_label_new(_("deg"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(table, label, 2, 3, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    gtk_table_set_row_spacing(table, row-1, 8);
    hbox = gtk_hbox_new(FALSE, 4);

    sizegroup = gtk_size_group_new(GTK_SIZE_GROUP_BOTH);
    button = gtk_button_new_with_mnemonic(_("_Refine"));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sizegroup, button);
    g_signal_connect_swapped(button, "clicked",
                             G_CALLBACK(refine_facet), &controls);

    button = gtk_button_new_with_mnemonic(_("_Mark"));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sizegroup, button);
    g_signal_connect_swapped(button, "clicked",
                             G_CALLBACK(mark_facet), &controls);

    button = gtk_button_new_with_mnemonic(_("Mea_sure"));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sizegroup, button);
    g_signal_connect_swapped(button, "clicked",
                             G_CALLBACK(measure_facet), &controls);
    g_object_unref(sizegroup);

    gtk_table_attach(table, hbox, 0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    gtk_table_set_row_spacing(table, row-1, 8);
    if (mfield) {
        gwy_container_set_object_by_name(controls.fdata, "/1/mask", mfield);
        create_mask_merge_buttons(GTK_WIDGET(table), row, NULL,
                                  args->combine,
                                  G_CALLBACK(combine_changed),
                                  args->combine_type,
                                  G_CALLBACK(combine_type_changed),
                                  &controls,
                                  &controls.combine, &controls.combine_type);
        row++;
    }

    controls.color_button = create_mask_color_button(controls.mydata,
                                                     controls.dialogue, 0);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row++, _("_Mask color:"), NULL,
                            GTK_OBJECT(controls.color_button),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    /* Second row: Facet view + Facet list */
    hbox = gtk_hbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), hbox,
                       FALSE, FALSE, 2);

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 4);

    controls.fview = create_preview(controls.fdata, 0, FACETVIEW_SIZE, TRUE);
    gtk_box_pack_start(GTK_BOX(vbox), controls.fview, FALSE, FALSE, 0);
    selection = create_vector_layer(GWY_DATA_VIEW(controls.fview), 0, "Point",
                                    TRUE);
    controls.fselection = selection;
    gwy_selection_set_max_objects(selection, 1);
    selkey = g_strdup_printf("/%d/select/_facets", id);
    if (gwy_container_gis_object_by_name(data, selkey, &selection)
        && gwy_selection_get_data(selection, NULL) > 0) {
        gdouble xy[2];

        gwy_selection_get_object(selection, 0, xy);
        gwy_selection_set_object(controls.fselection, 0, xy);
        selection = controls.fselection;
        restored_selection = TRUE;
    }
    controls.fselection0 = gwy_selection_duplicate(selection);
    g_signal_connect(selection, "changed",
                     G_CALLBACK(facet_view_selection_updated), &controls);

    thetabox = gtk_hbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), thetabox, FALSE, FALSE, 0);

    label = controls.theta_min_label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(thetabox), label, TRUE, TRUE, 0);

    label = controls.theta_0_label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
    gtk_box_pack_start(GTK_BOX(thetabox), label, TRUE, TRUE, 0);

    label = controls.theta_max_label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gtk_box_pack_start(GTK_BOX(thetabox), label, TRUE, TRUE, 0);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

    create_point_list(&controls);
    scwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin),
                                   GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(scwin), controls.pointlist);
    gtk_box_pack_start(GTK_BOX(vbox), scwin, TRUE, TRUE, 0);

    controls.rexport = gwy_results_export_new(args->report_style);
    rexport = GWY_RESULTS_EXPORT(controls.rexport);
    gwy_results_export_set_style(rexport, GWY_RESULTS_EXPORT_TABULAR_DATA);
    gtk_box_pack_start(GTK_BOX(vbox), controls.rexport, FALSE, FALSE, 0);
    g_signal_connect_swapped(rexport, "format-changed",
                             G_CALLBACK(report_style_changed), &controls);
    g_signal_connect_swapped(controls.rexport, "copy",
                             G_CALLBACK(copy_facet_table), &controls);
    g_signal_connect_swapped(controls.rexport, "save",
                             G_CALLBACK(save_facet_table), &controls);

    sizegroup = gtk_size_group_new(GTK_SIZE_GROUP_BOTH);
    button = gwy_stock_like_button_new(_("_Clear"), GTK_STOCK_CLEAR);
    gtk_box_pack_start(GTK_BOX(controls.rexport), button, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sizegroup, button);
    g_signal_connect_swapped(button, "clicked",
                             G_CALLBACK(clear_measurements), &controls);

    button = gwy_stock_like_button_new(_("_Delete"), GTK_STOCK_DELETE);
    controls.delete = button;
    gtk_box_pack_start(GTK_BOX(controls.rexport), button, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sizegroup, button);
    gtk_widget_set_sensitive(button, FALSE);
    g_signal_connect_swapped(button, "clicked",
                             G_CALLBACK(delete_measurement), &controls);
    g_object_unref(sizegroup);

    gtk_widget_show_all(dialogue);
    update_theta_range(&controls);

    if (restored_selection)
        facet_view_selection_updated(selection, -1, &controls);
    else
        facet_view_select_angle(&controls, args->theta0, args->phi0);

    do {
        response = gtk_dialog_run(GTK_DIALOG(dialogue));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialogue);
            case GTK_RESPONSE_NONE:
            g_object_unref(controls.mydata);
            g_object_unref(controls.fdata);
            g_object_unref(controls.fselection0);
            g_free(selkey);
            return;
            break;

            case GTK_RESPONSE_OK:
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gwy_app_sync_data_items(controls.mydata, data, 0, id, FALSE,
                            GWY_DATA_ITEM_MASK_COLOR,
                            0);
    gwy_container_set_object_by_name(data, selkey, controls.fselection);
    gtk_widget_destroy(dialogue);

    g_object_unref(controls.mydata);
    dtheta = gwy_container_get_object_by_name(controls.fdata, "/theta");
    dphi = gwy_container_get_object_by_name(controls.fdata, "/phi");
    run_noninteractive(args, data, dtheta, dphi, dfield, mfield, mquark,
                       args->theta0, args->phi0);
    g_object_unref(controls.fdata);
    g_object_unref(controls.fselection0);
    g_free(selkey);
    gwy_app_channel_log_add_proc(data, id, id);
}

static void
create_point_list_column(GtkTreeView *treeview, GtkCellRenderer *renderer,
                         FacetMeasureControls *controls,
                         const gchar *name, const gchar *units,
                         guint facet_column)
{
    GtkTreeViewColumn *column;
    GtkCellLayoutDataFunc cellfunc;
    GtkWidget *label;
    gchar *s;

    column = gtk_tree_view_column_new();
    gtk_tree_view_column_set_expand(column, TRUE);
    gtk_tree_view_column_set_alignment(column, 0.5);
    g_object_set_data(G_OBJECT(column), "id", GUINT_TO_POINTER(facet_column));

    gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(column), renderer, TRUE);
    if (facet_column == FACET_COLUMN_N)
        cellfunc = render_id;
    else if (facet_column == FACET_COLUMN_NPOINTS)
        cellfunc = render_npoints;
    else if (facet_column >= FACET_COLUMN_X && facet_column <= FACET_COLUMN_Z)
        cellfunc = render_facet_coordinate;
    else
        cellfunc = render_facet_angle;

    gtk_cell_layout_set_cell_data_func(GTK_CELL_LAYOUT(column), renderer,
                                       cellfunc, controls, NULL);

    label = gtk_label_new(NULL);
    if (units && strlen(units))
        s = g_strdup_printf("<b>%s</b> [%s]", name, units);
    else
        s = g_strdup_printf("<b>%s</b>", name);
    gtk_label_set_markup(GTK_LABEL(label), s);
    g_free(s);
    gtk_tree_view_column_set_widget(column, label);
    gtk_widget_show(label);
    gtk_tree_view_append_column(treeview, column);
}

static void
create_point_list(FacetMeasureControls *controls)
{
    GtkTreeView *treeview;
    GtkCellRenderer *renderer;
    GtkTreeSelection *treesel;

    controls->store = gwy_null_store_new(0);
    controls->measured_data = g_array_new(FALSE, FALSE,
                                          sizeof(FacetMeasurement));
    controls->pointlist
        = gtk_tree_view_new_with_model(GTK_TREE_MODEL(controls->store));
    treeview = GTK_TREE_VIEW(controls->pointlist);

    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "xalign", 1.0, NULL);

    create_point_list_column(treeview, renderer, controls,
                             "n", NULL, FACET_COLUMN_N);
    create_point_list_column(treeview, renderer, controls,
                             _("points"), NULL, FACET_COLUMN_NPOINTS);
    create_point_list_column(treeview, renderer, controls,
                             "t", _("deg"), FACET_COLUMN_TOL);
    create_point_list_column(treeview, renderer, controls,
                             "θ", _("deg"), FACET_COLUMN_THETA);
    create_point_list_column(treeview, renderer, controls,
                             "φ", _("deg"), FACET_COLUMN_PHI);
    create_point_list_column(treeview, renderer, controls,
                             "x", NULL, FACET_COLUMN_X);
    create_point_list_column(treeview, renderer, controls,
                             "y", NULL, FACET_COLUMN_Y);
    create_point_list_column(treeview, renderer, controls,
                             "z", NULL, FACET_COLUMN_Z);
    create_point_list_column(treeview, renderer, controls,
                             "δ", _("deg"), FACET_COLUMN_ERROR);

    treesel = gtk_tree_view_get_selection(treeview);
    gtk_tree_selection_set_mode(treesel, GTK_SELECTION_BROWSE);
    g_signal_connect(treesel, "changed",
                     G_CALLBACK(point_list_selection_changed), controls);

    g_signal_connect_swapped(treeview, "key-press-event",
                             G_CALLBACK(point_list_key_pressed), controls);

    g_object_unref(controls->store);
}

static void
point_list_selection_changed(GtkTreeSelection *treesel,
                             FacetMeasureControls *controls)
{
    GtkTreeModel *model;
    GtkTreeIter iter;
    gboolean sens;

    if ((sens = gtk_tree_selection_get_selected(treesel, &model, &iter)))
        gtk_tree_model_get(model, &iter, 0, &controls->selid, -1);
    else
        controls->selid = -1;

    gtk_widget_set_sensitive(controls->delete, sens);
}

static void
render_id(G_GNUC_UNUSED GtkCellLayout *layout,
          GtkCellRenderer *renderer,
          GtkTreeModel *model,
          GtkTreeIter *iter,
          G_GNUC_UNUSED gpointer user_data)
{
    gchar buf[16];
    guint i;

    gtk_tree_model_get(model, iter, 0, &i, -1);
    g_snprintf(buf, sizeof(buf), "%d", i + 1);
    g_object_set(renderer, "text", buf, NULL);
}

static void
render_npoints(G_GNUC_UNUSED GtkCellLayout *layout,
               GtkCellRenderer *renderer,
               GtkTreeModel *model,
               GtkTreeIter *iter,
               gpointer user_data)
{
    FacetMeasureControls *controls = (FacetMeasureControls*)user_data;
    FacetMeasurement *fmeas;
    gchar buf[16];
    guint i;

    gtk_tree_model_get(model, iter, 0, &i, -1);
    g_return_if_fail(i < controls->measured_data->len);
    fmeas = &g_array_index(controls->measured_data, FacetMeasurement, i);
    g_snprintf(buf, sizeof(buf), "%u", fmeas->npoints);
    g_object_set(renderer, "text", buf, NULL);
}

static inline void
slopes_to_angles(gdouble xder, gdouble yder,
                 gdouble *theta, gdouble *phi)
{
    *phi = atan2(yder, -xder);
    *theta = atan(hypot(xder, yder));
}

/* Transforms (ϑ, φ) to Cartesian selection coordinates [-q,q], which is
 * [-1,1] for the full range of angles. */
static inline void
angles_to_xy(gdouble theta, gdouble phi,
             gdouble *x, gdouble *y)
{
    gdouble rho = G_SQRT2*sin(theta/2.0);
    gdouble c = cos(phi), s = sin(phi);

    *x = rho*c;
    *y = -rho*s;
}

static inline void
xy_to_angles(gdouble x, gdouble y,
             gdouble *theta, gdouble *phi)
{
    gdouble s = hypot(x, y)/G_SQRT2;

    *phi = atan2(-y, x);
    if (s <= 1.0)
        *theta = 2.0*asin(s);
    else
        *theta = G_PI - 2.0*asin(2.0 - s);
}

static void
render_facet_angle(GtkCellLayout *layout,
                   GtkCellRenderer *renderer,
                   GtkTreeModel *model,
                   GtkTreeIter *iter,
                   gpointer user_data)
{
    FacetMeasureControls *controls = (FacetMeasureControls*)user_data;
    FacetMeasurement *fmeas;
    gchar buf[16];
    guint i, id;
    gdouble u;

    id = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(layout), "id"));
    gtk_tree_model_get(model, iter, 0, &i, -1);
    g_return_if_fail(i < controls->measured_data->len);
    fmeas = &g_array_index(controls->measured_data, FacetMeasurement, i);

    if (id == FACET_COLUMN_THETA)
        u = fmeas->theta;
    else if (id == FACET_COLUMN_PHI)
        u = fmeas->phi;
    else if (id == FACET_COLUMN_ERROR)
        u = fmeas->error;
    else if (id == FACET_COLUMN_TOL)
        u = fmeas->tolerance;
    else {
        g_assert_not_reached();
    }

    g_snprintf(buf, sizeof(buf), "%.3f", 180.0/G_PI*u);
    g_object_set(renderer, "text", buf, NULL);
}

static void
render_facet_coordinate(GtkCellLayout *layout,
                        GtkCellRenderer *renderer,
                        GtkTreeModel *model,
                        GtkTreeIter *iter,
                        gpointer user_data)
{
    FacetMeasureControls *controls = (FacetMeasureControls*)user_data;
    FacetMeasurement *fmeas;
    gchar buf[16];
    guint i, id;
    gdouble u;

    id = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(layout), "id"));
    gtk_tree_model_get(model, iter, 0, &i, -1);
    g_return_if_fail(i < controls->measured_data->len);
    fmeas = &g_array_index(controls->measured_data, FacetMeasurement, i);

    if (id == FACET_COLUMN_X)
        u = fmeas->v.x;
    else if (id == FACET_COLUMN_Y)
        u = fmeas->v.y;
    else if (id == FACET_COLUMN_Z)
        u = fmeas->v.z;
    else {
        g_assert_not_reached();
    }

    g_snprintf(buf, sizeof(buf), "%.3f", u);
    g_object_set(renderer, "text", buf, NULL);
}

static void
clear_measurements(FacetMeasureControls *controls)
{
    gwy_null_store_set_n_rows(controls->store, 0);
    g_array_set_size(controls->measured_data, 0);
}

static void
delete_measurement(FacetMeasureControls *controls)
{
    gint selid = controls->selid;
    gint i, n = controls->measured_data->len;

    if (selid < 0 || selid >= n)
        return;

    gwy_null_store_set_n_rows(controls->store, n-1);
    g_array_remove_index(controls->measured_data, selid);
    for (i = selid; i < n-1; i++)
        gwy_null_store_row_changed(controls->store, i);
}

static void
refine_facet(FacetMeasureControls *controls)
{
    GwyDataField *dist;
    gdouble xy[2], theta, phi, x, y, h;
    gint fres, range;

    gwy_selection_get_object(controls->fselection, 0, xy);
    xy_to_angles(xy[0] - controls->q, xy[1] - controls->q, &theta, &phi);

    dist = gwy_container_get_object_by_name(controls->fdata, "/0/data");
    fres = gwy_data_field_get_xres(dist);
    h = gwy_data_field_get_dx(dist);
    range = GWY_ROUND(fres/controls->q * 0.5/G_SQRT2 * cos(0.5*theta)
                      * controls->args->tolerance);
    x = xy[0]/h;
    y = xy[1]/h;
    gwy_data_field_local_maximum(dist, &x, &y, range, range);
    xy[0] = x*h;
    xy[1] = y*h;

    gwy_selection_set_object(controls->fselection, 0, xy);
}

static void
mark_facet(FacetMeasureControls *controls)
{
    FacetMeasureArgs *args = controls->args;
    GwyDataField *dtheta, *dphi, *mask, *mfield = NULL;
    GwyContainer *data, *fdata;

    data = controls->mydata;
    fdata = controls->fdata;

    add_mask_field(GWY_DATA_VIEW(controls->view), NULL);
    add_mask_field(GWY_DATA_VIEW(controls->fview), &mask_color);

    mask = gwy_container_get_object_by_name(data, "/0/mask");
    dtheta = gwy_container_get_object_by_name(fdata, "/theta");
    dphi = gwy_container_get_object_by_name(fdata, "/phi");
    gwy_container_gis_object_by_name(fdata, "/1/mask", (GObject**)&mfield);

    gwy_data_field_mark_facets(dtheta, dphi, args->theta0, args->phi0,
                               args->tolerance, mask);
    if (mfield && args->combine) {
        if (args->combine_type == GWY_MERGE_UNION)
            gwy_data_field_grains_add(mask, mfield);
        else if (args->combine_type == GWY_MERGE_INTERSECTION)
            gwy_data_field_grains_intersect(mask, mfield);
    }
    gwy_data_field_data_changed(mask);

    mask = gwy_container_get_object_by_name(fdata, "/0/mask");
    facet_measure_mark_fdata(mask, controls->q,
                             args->theta0, args->phi0, args->tolerance);
    gwy_data_field_data_changed(mask);
}

static void
measure_facet(FacetMeasureControls *controls)
{
    FacetMeasureArgs *args = controls->args;
    GwyNullStore *store = controls->store;
    GwyDataField *dtheta, *dphi;
    FacetMeasurement fmeas;

    dtheta = gwy_container_get_object_by_name(controls->fdata, "/theta");
    dphi = gwy_container_get_object_by_name(controls->fdata, "/phi");

    calculate_average_angle(dtheta, dphi,
                            args->theta0, args->phi0, args->tolerance, &fmeas);
    g_array_append_val(controls->measured_data, fmeas);
    gwy_null_store_set_n_rows(store, gwy_null_store_get_n_rows(store) + 1);
}

static gboolean
point_list_key_pressed(FacetMeasureControls *controls,
                       GdkEventKey *event)
{
    if (event->keyval == GDK_Delete) {
        delete_measurement(controls);
        return TRUE;
    }
    return FALSE;
}

static void
report_style_changed(FacetMeasureControls *controls, GwyResultsExport *rexport)
{
    controls->args->report_style = gwy_results_export_get_format(rexport);
}

static gchar*
format_facet_table(FacetMeasureControls *controls)
{
    GwyResultsReportType report_style = controls->args->report_style;
    FacetMeasurement *fmeas;
    GString *str;
    gdouble q = 1.0;
    guint n, i;

    if (!(n = controls->measured_data->len))
        return NULL;

    str = g_string_new(NULL);
    if (!(report_style & GWY_RESULTS_REPORT_MACHINE)) {
        gwy_format_result_table_strings(str, report_style, 8,
                                        "N", "t [deg]",
                                        "ϑ [deg]", "φ [deg]",
                                        "x", "y", "z",
                                        "δ");
        q = 180.0/G_PI;
    }
    else {
        gwy_format_result_table_strings(str, report_style, 8,
                                        "N", "t", "ϑ", "φ", "x", "y", "z", "δ");
    }

    for (i = 0; i < n; i++) {
        fmeas = &g_array_index(controls->measured_data, FacetMeasurement, i);
        gwy_format_result_table_row(str, report_style, 8,
                                    1.0*fmeas->npoints, fmeas->tolerance,
                                    q*fmeas->theta, q*fmeas->phi,
                                    fmeas->v.x, fmeas->v.y, fmeas->v.z,
                                    q*fmeas->error);
    }

    return g_string_free(str, FALSE);
}

static void
copy_facet_table(FacetMeasureControls *controls)
{
    GdkDisplay *display;
    GtkClipboard *clipboard;
    gchar *report;

    if (!(report = format_facet_table(controls)))
        return;
    display = gtk_widget_get_display(controls->dialogue);
    clipboard = gtk_clipboard_get_for_display(display,
                                              GDK_SELECTION_CLIPBOARD);
    gtk_clipboard_set_text(clipboard, report, -1);
    g_free(report);
}

static void
save_facet_table(FacetMeasureControls *controls)
{
    gchar *report;

    if (!(report = format_facet_table(controls)))
        return;
    gwy_save_auxiliary_data(_("Save Facet Vectors"),
                            GTK_WINDOW(controls->dialogue), -1, report);
    g_free(report);
}

static void
kernel_size_changed(GtkAdjustment *adj, FacetMeasureControls *controls)
{
    GwyDataField *dfield;
    GwySelection *selection;
    gdouble *xy;
    gdouble q;
    guint i, n;

    selection = controls->fselection;
    q = controls->q;
    n = gwy_selection_get_data(selection, NULL);
    xy = g_new(gdouble, 2*n);
    gwy_selection_get_data(selection, xy);
    for (i = 0; i < n; i++)
        xy_to_angles(xy[2*i] - q, xy[2*i+1] - q, xy + 2*i, xy + 2*i+1);

    controls->args->kernel_size = gwy_adjustment_get_int(adj);
    gwy_app_wait_cursor_start(GTK_WINDOW(controls->dialogue));
    dfield = gwy_container_get_object_by_name(controls->mydata, "/0/data");
    controls->q = gwy_data_field_facet_distribution(dfield,
                                                    controls->args->kernel_size,
                                                    controls->fdata);
    q = controls->q;

    /* TODO: Handle mask combining options to show the correct mask on the
     * image. */
    if (gwy_container_gis_object_by_name(controls->mydata, "/0/mask",
                                         &dfield)) {
        gwy_data_field_clear(dfield);
        gwy_data_field_data_changed(dfield);
    }
    if (gwy_container_gis_object_by_name(controls->fdata, "/0/mask",
                                         &dfield)) {
        gwy_data_field_clear(dfield);
        gwy_data_field_data_changed(dfield);
    }

    update_theta_range(controls);
    if (gwy_selection_get_data(controls->iselection, NULL))
        gwy_selection_clear(controls->iselection);

    for (i = 0; i < n; i++) {
        angles_to_xy(xy[2*i], xy[2*i+1], xy + 2*i, xy + 2*i+1);
        xy[2*i] += q;
        xy[2*i+1] += q;
    }
    gwy_selection_set_data(selection, n, xy);
    g_free(xy);
    gwy_app_wait_cursor_finish(GTK_WINDOW(controls->dialogue));
}

static void
update_theta_range(FacetMeasureControls *controls)
{
    gdouble x, y, theta, phi;
    gchar buf[32];

    x = controls->q;
    y = 0.0;
    xy_to_angles(x, y, &theta, &phi);
    g_snprintf(buf, sizeof(buf), "%.1f %s", -180.0/G_PI*theta, _("deg"));
    gtk_label_set_text(GTK_LABEL(controls->theta_min_label), buf);
    g_snprintf(buf, sizeof(buf), "0 %s", _("deg"));
    gtk_label_set_text(GTK_LABEL(controls->theta_0_label), buf);
    g_snprintf(buf, sizeof(buf), "%.1f %s", 180.0/G_PI*theta, _("deg"));
    gtk_label_set_text(GTK_LABEL(controls->theta_max_label), buf);
}

static void
facet_view_select_angle(FacetMeasureControls *controls,
                        gdouble theta,
                        gdouble phi)
{
    gdouble xy[2];

    controls->in_update = TRUE;
    angles_to_xy(theta, phi, xy+0, xy+1);
    xy[0] += controls->q;
    xy[1] += controls->q;
    gwy_selection_set_object(controls->fselection, 0, xy);
    controls->in_update = FALSE;
}

static void
facet_view_selection_updated(GwySelection *selection,
                             gint hint,
                             FacetMeasureControls *controls)
{
    gdouble xy[2], theta, phi;
    guchar buf[16];

    g_return_if_fail(hint == 0 || hint == -1);
    gwy_selection_get_object(selection, 0, xy);
    xy_to_angles(xy[0] - controls->q, xy[1] - controls->q, &theta, &phi);
    controls->args->theta0 = theta;
    controls->args->phi0 = phi;

    g_snprintf(buf, sizeof(buf), "%.2f", 180.0/G_PI*theta);
    gtk_label_set_text(GTK_LABEL(controls->theta_label), buf);

    g_snprintf(buf, sizeof(buf), "%.2f", 180.0/G_PI*phi);
    gtk_label_set_text(GTK_LABEL(controls->phi_label), buf);
}

static void
preview_selection_updated(GwySelection *selection,
                          gint hint,
                          FacetMeasureControls *controls)
{
    GwyDataField *dfield;
    gdouble theta, phi, xy[2];
    gint i, j;

    if (controls->in_update || hint != 0)
        return;

    dfield = gwy_container_get_object_by_name(controls->mydata, "/0/data");
    gwy_selection_get_object(selection, 0, xy);
    j = gwy_data_field_rtoj(dfield, xy[0]);
    i = gwy_data_field_rtoi(dfield, xy[1]);
    dfield = gwy_container_get_object_by_name(controls->fdata, "/theta");
    theta = gwy_data_field_get_val(dfield, j, i);
    dfield = gwy_container_get_object_by_name(controls->fdata, "/phi");
    phi = gwy_data_field_get_val(dfield, j, i);
    facet_view_select_angle(controls, theta, phi);
}

static void
run_noninteractive(FacetMeasureArgs *args,
                   GwyContainer *data,
                   GwyDataField *dtheta,
                   GwyDataField *dphi,
                   GwyDataField *dfield,
                   GwyDataField *mfield,
                   GQuark mquark,
                   gdouble theta,
                   gdouble phi)
{
    GwyDataField *mask;

    gwy_app_undo_qcheckpointv(data, 1, &mquark);
    mask = create_mask_field(dfield);

    gwy_data_field_mark_facets(dtheta, dphi, theta, phi, args->tolerance, mask);
    if (mfield && args->combine) {
        if (args->combine_type == GWY_MERGE_UNION)
            gwy_data_field_grains_add(mfield, mask);
        else if (args->combine_type == GWY_MERGE_INTERSECTION)
            gwy_data_field_grains_intersect(mfield, mask);
        gwy_data_field_data_changed(mfield);
    }
    else if (mfield) {
        gwy_data_field_copy(mask, mfield, FALSE);
        gwy_data_field_data_changed(mfield);
    }
    else {
        gwy_container_set_object(data, mquark, mask);
    }
    g_object_unref(mask);
}

static void
gwy_data_field_mark_facets(GwyDataField *dtheta,
                           GwyDataField *dphi,
                           gdouble theta0,
                           gdouble phi0,
                           gdouble tolerance,
                           GwyDataField *mask)
{
    gdouble ctol, cth0, sth0;
    const gdouble *td, *pd;
    gdouble *md;
    guint i, n;

    ctol = cos(tolerance);
    cth0 = cos(theta0);
    sth0 = sin(theta0);

    td = gwy_data_field_get_data_const(dtheta);
    pd = gwy_data_field_get_data_const(dphi);
    md = gwy_data_field_get_data(mask);
    n = gwy_data_field_get_xres(dtheta)*gwy_data_field_get_yres(dtheta);

#ifdef _OPENMP
#pragma omp parallel for if(gwy_threads_are_enabled()) default(none) \
            private(i) \
            shared(td,pd,md,n,cth0,sth0,phi0,ctol)
#endif
    for (i = 0; i < n; i++) {
        gdouble cro = cth0*cos(td[i]) + sth0*sin(td[i])*cos(pd[i] - phi0);
        md[i] = (cro >= ctol);
    }
}

static gdouble
gwy_data_field_facet_distribution(GwyDataField *dfield,
                                  gint half_size,
                                  GwyContainer *container)
{
    GwyDataField *dtheta, *dphi, *dist;
    gdouble *xd, *yd, *data;
    const gdouble *xdc, *ydc;
    gdouble q, x, y;
    gint hres, i, xres, yres, n, fres;

    if (gwy_container_gis_object_by_name(container, "/theta", &dtheta))
        g_object_ref(dtheta);
    else
        dtheta = gwy_data_field_new_alike(dfield, FALSE);

    if (gwy_container_gis_object_by_name(container, "/phi", &dphi))
        g_object_ref(dphi);
    else
        dphi = gwy_data_field_new_alike(dfield, FALSE);

    compute_slopes(dfield, 2*half_size + 1, dtheta, dphi);
    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    xd = gwy_data_field_get_data(dtheta);
    yd = gwy_data_field_get_data(dphi);
    n = xres*yres;

#ifdef _OPENMP
#pragma omp parallel for if(gwy_threads_are_enabled()) default(none) \
            private(i) \
            shared(xd,yd,n)
#endif
    for (i = 0; i < n; i++) {
        gdouble theta, phi;

        slopes_to_angles(xd[i], yd[i], &theta, &phi);
        xd[i] = theta;
        yd[i] = phi;
    }
    q = gwy_data_field_get_max(dtheta);
    q = MIN(q*1.05, 1.001*G_PI/2.0);
    q = G_SQRT2*sin(q/2.0);

    hres = GWY_ROUND(cbrt(3.49*n));
    fres = 2*hres + 1;
    dist = gwy_data_field_new(fres, fres, 2.0*q, 2.0*q, TRUE);
    gwy_data_field_set_xoffset(dist, -q);
    gwy_data_field_set_yoffset(dist, -q);
    gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_xy(dist), NULL);
    gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_z(dist), NULL);

    data = gwy_data_field_get_data(dist);
    xdc = gwy_data_field_get_data_const(dtheta);
    ydc = gwy_data_field_get_data_const(dphi);
    for (i = n; i; i--, xdc++, ydc++) {
        gint xx, yy;

        angles_to_xy(*xdc, *ydc, &x, &y);
        x = (x + q)/q*hres;
        y = (y + q)/q*hres;
        xx = (gint)floor(x - 0.5);
        yy = (gint)floor(y - 0.5);

        if (G_UNLIKELY(xx < 0)) {
            xx = 0;
            x = 0.0;
        }
        else if (G_UNLIKELY(xx >= fres-1)) {
            xx = fres-2;
            x = 1.0;
        }
        else
            x = x - (xx + 0.5);

        if (G_UNLIKELY(yy < 0)) {
            yy = 0;
            y = 0.0;
        }
        else if (G_UNLIKELY(yy >= fres-1)) {
            yy = fres-2;
            y = 1.0;
        }
        else
            y = y - (yy + 0.5);

        data[yy*fres + xx] += (1.0 - x)*(1.0 - y);
        data[yy*fres + xx+1] += x*(1.0 - y);
        data[yy*fres+fres + xx] += (1.0 - x)*y;
        data[yy*fres+fres + xx+1] += x*y;
    }

    /* Transform values for visualisation. */
    for (i = fres*fres; i; i--, data++)
        *data = cbrt(*data);

    gwy_container_set_object_by_name(container, "/0/data", dist);
    g_object_unref(dist);
    gwy_container_set_object_by_name(container, "/theta", dtheta);
    g_object_unref(dtheta);
    gwy_container_set_object_by_name(container, "/phi", dphi);
    g_object_unref(dphi);
    gwy_container_set_const_string_by_name(container, "/0/base/palette",
                                           FVIEW_GRADIENT);

    return q;
}

static void
compute_slopes(GwyDataField *dfield,
               gint kernel_size,
               GwyDataField *xder,
               GwyDataField *yder)
{
    gint xres, yres;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    if (kernel_size > 1) {
        GwyPlaneFitQuantity quantites[] = {
            GWY_PLANE_FIT_BX, GWY_PLANE_FIT_BY
        };
        GwyDataField *fields[2];

        fields[0] = xder;
        fields[1] = yder;
        gwy_data_field_fit_local_planes(dfield, kernel_size,
                                        2, quantites, fields);

        gwy_data_field_multiply(xder, xres/gwy_data_field_get_xreal(dfield));
        gwy_data_field_multiply(yder, yres/gwy_data_field_get_yreal(dfield));
    }
    else
        gwy_data_field_filter_slope(dfield, xder, yder);
}

static void
tolerance_changed(FacetMeasureControls *controls, GtkAdjustment *adj)
{
    FacetMeasureArgs *args = controls->args;
    args->tolerance = G_PI/180.0 * gtk_adjustment_get_value(adj);
}

static void
combine_changed(FacetMeasureControls *controls, GtkToggleButton *toggle)
{
    controls->args->combine = gtk_toggle_button_get_active(toggle);
}

static void
combine_type_changed(FacetMeasureControls *controls)
{
    FacetMeasureArgs *args = controls->args;
    args->combine_type = gwy_radio_buttons_get_current(controls->combine_type);
}

static void
add_mask_field(GwyDataView *view,
               const GwyRGBA *color)
{
    GwyContainer *data;
    GwyDataField *mfield, *dfield;

    data = gwy_data_view_get_data(view);
    if (gwy_container_gis_object_by_name(data, "/0/mask", &mfield))
        return;

    gwy_container_gis_object_by_name(data, "/0/data", &dfield);
    mfield = create_mask_field(dfield);
    gwy_container_set_object_by_name(data, "/0/mask", mfield);
    g_object_unref(mfield);
    if (color)
        gwy_rgba_store_to_container(color, data, "/0/mask");
}

static void
facet_measure_mark_fdata(GwyDataField *mask,
                         gdouble q,
                         gdouble theta0, gdouble phi0, gdouble tolerance)
{
    gdouble r, r2, cr, cro, cth0, sth0, cphi0, sphi0;
    gint fres, hres, i, j;
    gdouble *m;

    cr = cos(tolerance);
    cth0 = cos(theta0);
    sth0 = sin(theta0);
    cphi0 = cos(phi0);
    sphi0 = sin(phi0);
    fres = gwy_data_field_get_xres(mask);
    g_assert(gwy_data_field_get_yres(mask) == fres);
    hres = (fres - 1)/2;
    m = gwy_data_field_get_data(mask);

#ifdef _OPENMP
#pragma omp parallel for if(gwy_threads_are_enabled()) default(none) \
            private(i,j,r2,r,cro) \
            shared(m,fres,hres,cth0,sth0,cphi0,sphi0,q,cr)
#endif
    for (i = 0; i < fres; i++) {
        gdouble y = -q*(i/(gdouble)hres - 1.0);

        for (j = 0; j < fres; j++) {
            gdouble x = q*(j/(gdouble)hres - 1.0);

            /**
             * Orthodromic distance computed directly from x, y:
             * cos(theta) = 1 - r^2
             * sin(theta) = r*sqrt(1 - r^2/2)
             * cos(phi) = x/r
             * sin(phi) = y/r
             * where r = hypot(x, y)
             **/
            r2 = x*x + y*y;
            r = sqrt(r2);
            cro = cth0*(1.0 - r2)
                  + sth0*G_SQRT2*r*sqrt(1.0 - r2/2.0)*(x/r*cphi0 + y/r*sphi0);
            m[i*fres + j] = (cro >= cr);
        }
    }
}

static void
calculate_average_angle(GwyDataField *dtheta,
                        GwyDataField *dphi,
                        gdouble theta0, gdouble phi0,
                        gdouble tolerance,
                        FacetMeasurement *fmeas)
{
    gdouble s2, sx, sy, sz, cth0, sth0, ctol;
    const gdouble *td, *pd;
    gint i, n, count;
    GwyXYZ s;

    gwy_clear(fmeas, 1);
    fmeas->tolerance = tolerance;

    cth0 = cos(theta0);
    sth0 = sin(theta0);
    ctol = cos(tolerance);

    td = gwy_data_field_get_data_const(dtheta);
    pd = gwy_data_field_get_data_const(dphi);
    n = gwy_data_field_get_xres(dtheta)*gwy_data_field_get_yres(dtheta);
    count = 0;
    sx = sy = sz = 0.0;

#ifdef _OPENMP
#pragma omp parallel for if(gwy_threads_are_enabled()) default(none) \
            reduction(+:count,sx,sy,sz) \
            private(i) \
            shared(td,pd,n,cth0,sth0,phi0,ctol)
#endif
    for (i = 0; i < n; i++) {
        gdouble cro = cth0*cos(td[i]) + sth0*sin(td[i]) * cos(pd[i] - phi0);
        if (cro >= ctol) {
            GwyXYZ v;

            make_unit_vector(&v, td[i], pd[i]);
            sx += v.x;
            sy += v.y;
            sz += v.z;
            count++;
        }
    }
    s.x = sx;
    s.y = sy;
    s.z = sz;
    fmeas->npoints = count;

    if (!count)
        return;

    vector_angles(&s, &fmeas->theta, &fmeas->phi);
    make_unit_vector(&s, fmeas->theta, fmeas->phi);
    fmeas->v = s;
    if (count == 1)
        return;

    /* Since we calculate the mean direction as vector average, not point on
     * sphere with minimum square geodesic distance, do the same for the
     * dispersion estimate. */
    s2 = 0.0;

#ifdef _OPENMP
#pragma omp parallel for if(gwy_threads_are_enabled()) default(none) \
            reduction(+:s2) \
            private(i) \
            shared(td,pd,s,n,cth0,sth0,phi0,ctol)
#endif
    for (i = 0; i < n; i++) {
        gdouble cro = cth0*cos(td[i]) + sth0*sin(td[i]) * cos(pd[i] - phi0);
        if (cro >= ctol) {
            GwyXYZ v;

            make_unit_vector(&v, td[i], pd[i]);
            s2 += ((v.x - s.x)*(v.x - s.x)
                   + (v.y - s.y)*(v.y - s.y)
                   + (v.z - s.z)*(v.z - s.z));
        }
    }

    /* This is already in radians. */
    fmeas->error = sqrt(s2/(count - 1));
}

static inline void
make_unit_vector(GwyXYZ *v, gdouble theta, gdouble phi)
{
    v->x = sin(theta)*cos(phi);
    v->y = sin(theta)*sin(phi);
    v->z = cos(theta);
}

static inline void
vector_angles(const GwyXYZ *v, gdouble *theta, gdouble *phi)
{
    *theta = atan2(sqrt(v->x*v->x + v->y*v->y), v->z);
    *phi = atan2(v->y, v->x);
}

static const gchar combine_key[]       = "/module/facet_measure/combine";
static const gchar combine_type_key[]  = "/module/facet_measure/combine_type";
static const gchar kernel_size_key[]   = "/module/facet_measure/kernel-size";
static const gchar phi0_key[]          = "/module/facet_measure/phi0";
static const gchar report_style_key[]  = "/module/facet_measure/report_style";
static const gchar theta0_key[]        = "/module/facet_measure/theta0";
static const gchar tolerance_key[]     = "/module/facet_measure/tolerance";

static void
facet_measure_sanitize_args(FacetMeasureArgs *args)
{
    args->combine = !!args->combine;
    args->tolerance = CLAMP(args->tolerance, 0.0, 30.0*G_PI/180.0);
    args->phi0 = fmod(args->phi0, 2.0*G_PI);
    args->theta0 = CLAMP(args->theta0, 0.0, 0.5*G_PI);
    args->kernel_size = CLAMP(args->kernel_size, 0, MAX_PLANE_SIZE);
    args->combine_type = MIN(args->combine_type, GWY_MERGE_INTERSECTION);
}

static void
facet_measure_load_args(GwyContainer *container, FacetMeasureArgs *args)
{
    *args = facet_measure_defaults;

    gwy_container_gis_boolean_by_name(container, combine_key, &args->combine);
    gwy_container_gis_double_by_name(container, tolerance_key,
                                     &args->tolerance);
    gwy_container_gis_double_by_name(container, phi0_key, &args->phi0);
    gwy_container_gis_double_by_name(container, theta0_key, &args->theta0);
    gwy_container_gis_int32_by_name(container, kernel_size_key,
                                    &args->kernel_size);
    gwy_container_gis_enum_by_name(container, combine_type_key,
                                   &args->combine_type);
    gwy_container_gis_enum_by_name(container, report_style_key,
                                   &args->report_style);
    facet_measure_sanitize_args(args);
}

static void
facet_measure_save_args(GwyContainer *container, const FacetMeasureArgs *args)
{
    gwy_container_set_boolean_by_name(container, combine_key, args->combine);
    gwy_container_set_double_by_name(container, tolerance_key,
                                     args->tolerance);
    gwy_container_set_double_by_name(container, phi0_key, args->phi0);
    gwy_container_set_double_by_name(container, theta0_key, args->theta0);
    gwy_container_set_int32_by_name(container, kernel_size_key,
                                    args->kernel_size);
    gwy_container_set_enum_by_name(container, combine_type_key,
                                   args->combine_type);
    gwy_container_set_enum_by_name(container, report_style_key,
                                   args->report_style);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
