/*
 *  $Id: gwyddion.c 21684 2018-11-26 12:31:34Z yeti-dn $
 *  Copyright (C) 2003-2018 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <libgwyddion/gwyddion.h>
#include "gwyddioninternal.h"

static gboolean threads_enabled = FALSE;

/**
 * gwy_type_init:
 *
 * Makes libgwyddion types safe for deserialization and performs other
 * initialization.  You have to call this function from the main thread before
 * using objects from libgwyddion.
 *
 * It calls g_type_init() first to make sure GLib object system is initialized.
 *
 * It is safe to call this function more than once, subsequent calls are no-op.
 **/
void
gwy_type_init(void)
{
    static gboolean types_initialized = FALSE;

    if (types_initialized)
        return;

#if (!GLIB_CHECK_VERSION(2, 36, 0))
    g_type_init();
#endif

    g_type_class_peek(GWY_TYPE_SI_UNIT);
    g_type_class_peek(GWY_TYPE_CONTAINER);
    g_type_class_peek(GWY_TYPE_INVENTORY);
    g_type_class_peek(GWY_TYPE_RESOURCE);
    g_type_class_peek(GWY_TYPE_NLFIT_PRESET);
    g_type_class_peek(GWY_TYPE_FD_CURVE_PRESET);
    g_type_class_peek(GWY_TYPE_STRING_LIST);
    g_type_class_peek(GWY_TYPE_XY);
    g_type_class_peek(GWY_TYPE_XYZ);
    g_type_class_peek(GWY_TYPE_ENUM);
    g_type_class_peek(GWY_TYPE_SI_VALUE_FORMAT);
    types_initialized = TRUE;

    _gwy_nlfit_preset_class_setup_presets();
    _gwy_fd_curve_preset_class_setup_presets();
}

/**
 * gwy_threads_are_enabled:
 *
 * Obtains the state of internal multithread processing.
 *
 * Returns: %TRUE if multithread processing is enabled; %FALSE otherwise (this
 *          includes the case of Gwyddion not built with multithread processing
 *          support at all).
 *
 * Since: 2.53
 **/
gboolean
gwy_threads_are_enabled(void)
{
    return threads_enabled;
}

/**
 * gwy_threads_set_enabled:
 * @setting: %TRUE to enable multithread processing; %FALSE to disable it.
 *
 * Enables or disables internal multithread processing.
 *
 * Multithread processing, if enabled, is utilised internally and transparently
 * in Gwyddion functions.  No threads are exposed in the API.
 *
 * The default setting is %FALSE.  If you run programs or scripts based on
 * Gwyddion in parallel, for instance in a simulation or batch data processing,
 * it is recommended to keep multithread processing disabled.  For GUI programs
 * (like Gwyddion itself) or tasks run serially, it can be useful to enable it.
 *
 * If Gwyddion was not built with multithread processing support, enabling
 * threads does not do anything and gwy_threads_are_enabled() will continue to
 * return %FALSE.
 *
 * Since: 2.53
 **/
void
gwy_threads_set_enabled(G_GNUC_UNUSED gboolean setting)
{
#ifdef _OPENMP
    threads_enabled = setting;
#endif
}

/************************** Documentation ****************************/

/**
 * SECTION:gwyddion
 * @title: gwyddion
 * @short_description: Base functions, library initialization
 * @see_also: #GwySerializable
 *
 * Gwyddion classes has to be initialized before they can be safely
 * deserialized. The function gwy_type_init() performs this initialization.
 **/

/**
 * SECTION:gwyddionenums
 * @title: gwyddionenums
 * @short_description: Common libgwyddion enumerations
 **/

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
