package main

import (
	"fmt"
	"html/template"
	"math"
	"path"
	"reflect"
	"strings"

	"github.com/google/shlex"
)

// funcs returns functions for use in templates.
func (s *Site) funcs() map[string]interface{} {
	return map[string]interface{}{
		"exec":         executeString,
		"math":         func() _math { return _math{} },
		"path":         func() _path { return _path{} },
		"partial":      s.templates.ExecutePartial,
		"reverse":      reverse,
		"safeCSS":      func(s string) template.CSS { return template.CSS(s) },
		"safeHTML":     func(s string) template.HTML { return template.HTML(s) },
		"safeHTMLAttr": func(s string) template.HTMLAttr { return template.HTMLAttr(s) },
		"safeJS":       func(s string) template.JS { return template.JS(s) },
		"safeURL":      func(s string) template.URL { return template.URL(s) },
		"site":         func() *Site { return s },
		"strings":      func() _strings { return _strings{} },
	}
}

type _math struct{}

func (_math) Add(a, b interface{}) interface{} {
	switch a := get64(a).(type) {
	case int64:
		switch b := get64(b).(type) {
		case int64:
			return a + b
		case float64:
			return float64(a) + b
		}
	case float64:
		switch b := get64(b).(type) {
		case int64:
			return a + float64(b)
		case float64:
			return a + b
		}
	}
	panic("unreachable")
}

func (_math) Sub(a, b interface{}) interface{} {
	switch a := get64(a).(type) {
	case int64:
		switch b := get64(b).(type) {
		case int64:
			return a - b
		case float64:
			return float64(a) - b
		}
	case float64:
		switch b := get64(b).(type) {
		case int64:
			return a - float64(b)
		case float64:
			return a - b
		}
	}
	panic("unreachable")
}

func (_math) Mul(a, b interface{}) interface{} {
	switch a := get64(a).(type) {
	case int64:
		switch b := get64(b).(type) {
		case int64:
			return a * b
		case float64:
			return float64(a) * b
		}
	case float64:
		switch b := get64(b).(type) {
		case int64:
			return a * float64(b)
		case float64:
			return a * b
		}
	}
	panic("unreachable")
}

func (_math) Div(a, b interface{}) float64 {
	switch a := get64(a).(type) {
	case int64:
		switch b := get64(b).(type) {
		case int64:
			return float64(a) / float64(b)
		case float64:
			return float64(a) / b
		}
	case float64:
		switch b := get64(b).(type) {
		case int64:
			return a / float64(b)
		case float64:
			return a / b
		}
	}
	panic("unreachable")
}

func (_math) Mod(a, b interface{}) int64 {
	switch a := get64(a).(type) {
	case int64:
		switch b := get64(b).(type) {
		case int64:
			return a % b
		case float64:
			return int64(math.Mod(float64(a), b))
		}
	case float64:
		switch b := get64(b).(type) {
		case int64:
			return int64(math.Mod(a, float64(b)))
		case float64:
			return int64(math.Mod(a, b))
		}
	}
	panic("unreachable")
}

func (_math) Ceil(a interface{}) int64 {
	switch a := get64(a).(type) {
	case int64:
		return a
	case float64:
		return int64(math.Ceil(a))
	}
	panic("unreachable")
}

func (_math) Floor(a interface{}) int64 {
	switch a := get64(a).(type) {
	case int64:
		return a
	case float64:
		return int64(math.Floor(a))
	}
	panic("unreachable")
}

func (_math) Log(a interface{}) float64 {
	switch a := get64(a).(type) {
	case int64:
		return math.Log(float64(a))
	case float64:
		return math.Log(a)
	}
	panic("unreachable")
}

func (_math) Max(a, b interface{}) interface{} {
	switch a := get64(a).(type) {
	case int64:
		switch b := get64(b).(type) {
		case int64:
			if a >= b {
				return a
			} else {
				return b
			}
		case float64:
			if float64(a) >= b {
				return float64(a)
			} else {
				return b
			}
		}
	case float64:
		switch b := get64(b).(type) {
		case int64:
			if a >= float64(b) {
				return a
			} else {
				return float64(b)
			}
		case float64:
			if a >= b {
				return a
			} else {
				return b
			}
		}
	}
	panic("unreachable")
}

func (_math) Min(a, b interface{}) interface{} {
	switch a := get64(a).(type) {
	case int64:
		switch b := get64(b).(type) {
		case int64:
			if a <= b {
				return a
			} else {
				return b
			}
		case float64:
			if float64(a) <= b {
				return float64(a)
			} else {
				return b
			}
		}
	case float64:
		switch b := get64(b).(type) {
		case int64:
			if a <= float64(b) {
				return a
			} else {
				return float64(b)
			}
		case float64:
			if a <= b {
				return a
			} else {
				return b
			}
		}
	}
	panic("unreachable")
}

func (_math) Pow(a, b interface{}) float64 {
	switch a := get64(a).(type) {
	case int64:
		switch b := get64(b).(type) {
		case int64:
			return math.Pow(float64(a), float64(b))
		case float64:
			return math.Pow(float64(a), b)
		}
	case float64:
		switch b := get64(b).(type) {
		case int64:
			return math.Pow(a, float64(b))
		case float64:
			return math.Pow(a, b)
		}
	}
	panic("unreachable")
}

func (_math) Round(a interface{}) int64 {
	switch a := get64(a).(type) {
	case int64:
		return a
	case float64:
		return int64(math.Round(a))
	}
	panic("unreachable")
}

func (_math) Sqrt(a interface{}) float64 {
	switch a := get64(a).(type) {
	case int64:
		return math.Sqrt(float64(a))
	case float64:
		return math.Sqrt(a)
	}
	panic("unreachable")
}

func get64(x interface{}) interface{} {
	switch x := x.(type) {
	case uint8:
		return int64(x)
	case int8:
		return int64(x)
	case uint16:
		return int64(x)
	case int16:
		return int64(x)
	case uint32:
		return int64(x)
	case int32:
		return int64(x)
	case uint64:
		return int64(x)
	case int64:
		return int64(x)
	case int:
		return int64(x)
	case float32:
		return float64(x)
	case float64:
		return float64(x)
	default:
		panic(fmt.Errorf("unexpected type %T", x))
	}
}

type _path struct{}

func (_path) Base(s string) string            { return path.Base(s) }
func (_path) Clean(s string) string           { return path.Clean(s) }
func (_path) Dir(s string) string             { return path.Dir(s) }
func (_path) Ext(s string) string             { return path.Ext(s) }
func (_path) Join(elem ...string) string      { return path.Join(elem...) }
func (_path) Split(s string) (string, string) { return path.Split(s) }

type _strings struct{}

func (_strings) Count(a, b string) int                { return strings.Count(a, b) }
func (_strings) HasPrefix(a, b string) bool           { return strings.HasPrefix(a, b) }
func (_strings) HasSuffix(a, b string) bool           { return strings.HasSuffix(a, b) }
func (_strings) Join(elems []string, s string) string { return strings.Join(elems, s) }
func (_strings) Repeat(s string, i int) string        { return strings.Repeat(s, i) }
func (_strings) Replace(a, b, c string, n int) string { return strings.Replace(a, b, c, n) }
func (_strings) ReplaceAll(a, b, c string) string     { return strings.ReplaceAll(a, b, c) }
func (_strings) Split(a, b string) []string           { return strings.Split(a, b) }
func (_strings) Title(s string) string                { return strings.Title(s) }
func (_strings) ToLower(s string) string              { return strings.ToLower(s) }
func (_strings) ToUpper(s string) string              { return strings.ToUpper(s) }
func (_strings) Trim(a, b string) string              { return strings.Trim(a, b) }
func (_strings) TrimLeft(a, b string) string          { return strings.TrimLeft(a, b) }
func (_strings) TrimPrefix(a, b string) string        { return strings.TrimPrefix(a, b) }
func (_strings) TrimRight(a, b string) string         { return strings.TrimRight(a, b) }
func (_strings) TrimSpace(s string) string            { return strings.TrimSpace(s) }
func (_strings) TrimSuffix(a, b string) string        { return strings.TrimSuffix(a, b) }

func executeString(command, input string) (string, error) {
	var b strings.Builder
	cmd, err := shlex.Split(command)
	if err != nil {
		return "", err
	}
	if err := execute(cmd, strings.NewReader(input), &b); err != nil {
		return "", err
	}
	return b.String(), nil
}

func reverse(s interface{}) interface{} {
	v := reflect.ValueOf(s)
	n := v.Len()
	r := reflect.MakeSlice(v.Type(), n, n)
	reflect.Copy(r, v)
	swap := reflect.Swapper(r.Interface())
	for i, j := 0, n-1; i < j; i, j = i+1, j-1 {
		swap(i, j)
	}
	return r.Interface()
}
