/*
 * Gmsh - Copyright (C) 1997-2018 C. Geuzaine, J.-F. Remacle
 *
 * See the LICENSE.txt file for license information. Please report all
 * issues on https://gitlab.onelab.info/gmsh/gmsh/issues
 */

#ifndef _GMSHC_H_
#define _GMSHC_H_

/*
 * This file defines the Gmsh C API (v4.0).
 *
 * Do not edit it directly: it is automatically generated by `api/gen.py'.
 *
 * By design, the Gmsh C API is purely functional, and only uses elementary
 * types. See `demos/api' for examples.
 */

#include <stddef.h>

#define GMSH_API_VERSION "4.0"

#if defined(GMSH_DLL)
#if defined(GMSH_DLL_EXPORT)
#define GMSH_API __declspec(dllexport)
#else
#define GMSH_API __declspec(dllimport)
#endif
#else
#define GMSH_API
#endif

GMSH_API void gmshFree(void *p);
GMSH_API void *gmshMalloc(size_t n);

/* Initialize Gmsh. This must be called before any call to the other functions
 * in the API. If `argc' and `argv' (or just `argv' in Python or Julia) are
 * provided, they will be handled in the same way as the command line
 * arguments in the Gmsh app. If `readConfigFiles' is set, read system Gmsh
 * configuration files (gmshrc and gmsh-options). */
GMSH_API void gmshInitialize(int argc, char ** argv,
                             const int readConfigFiles,
                             int * ierr);

/* Finalize Gmsh. This must be called when you are done using the Gmsh API. */
GMSH_API void gmshFinalize(int * ierr);

/* Open a file. Equivalent to the `File->Open' menu in the Gmsh app. Handling
 * of the file depends on its extension and/or its contents. */
GMSH_API void gmshOpen(const char * fileName,
                       int * ierr);

/* Merge a file. Equivalent to the `File->Merge' menu in the Gmsh app.
 * Handling of the file depends on its extension and/or its contents. */
GMSH_API void gmshMerge(const char * fileName,
                        int * ierr);

/* Write a file. The export format is determined by the file extension. */
GMSH_API void gmshWrite(const char * fileName,
                        int * ierr);

/* Clear all loaded models and post-processing data, and add a new empty
 * model. */
GMSH_API void gmshClear(int * ierr);

/* Set a numerical option to `value'. `name' is of the form "category.option"
 * or "category[num].option". Available categories and options are listed in
 * the Gmsh reference manual. */
GMSH_API void gmshOptionSetNumber(const char * name,
                                  const double value,
                                  int * ierr);

/* Get the `value' of a numerical option. */
GMSH_API void gmshOptionGetNumber(const char * name,
                                  double * value,
                                  int * ierr);

/* Set a string option to `value'. */
GMSH_API void gmshOptionSetString(const char * name,
                                  const char * value,
                                  int * ierr);

/* Get the `value' of a string option. */
GMSH_API void gmshOptionGetString(const char * name,
                                  char ** value,
                                  int * ierr);

/* Add a new model, with name `name', and set it as the current model. */
GMSH_API void gmshModelAdd(const char * name,
                           int * ierr);

/* Remove the current model. */
GMSH_API void gmshModelRemove(int * ierr);

/* List the names of all models. */
GMSH_API void gmshModelList(char *** names, size_t * names_n,
                            int * ierr);

/* Set the current model to the model with name `name'. If several models have
 * the same name, select the one that was added first. */
GMSH_API void gmshModelSetCurrent(const char * name,
                                  int * ierr);

/* Get all the (elementary) geometrical entities in the current model. If
 * `dim' is >= 0, return only the entities of the specified dimension (e.g.
 * points if `dim' == 0). The entities are returned as a vector of (dim, tag)
 * integer pairs. */
GMSH_API void gmshModelGetEntities(int ** dimTags, size_t * dimTags_n,
                                   const int dim,
                                   int * ierr);

/* Get all the physical groups in the current model. If `dim' is >= 0, return
 * only the entities of the specified dimension (e.g. physical points if `dim'
 * == 0). The entities are returned as a vector of (dim, tag) integer pairs. */
GMSH_API void gmshModelGetPhysicalGroups(int ** dimTags, size_t * dimTags_n,
                                         const int dim,
                                         int * ierr);

/* Get the tags of the geometrical entities making up the physical group of
 * dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetEntitiesForPhysicalGroup(const int dim,
                                                   const int tag,
                                                   int ** tags, size_t * tags_n,
                                                   int * ierr);

/* Get the tags of the physical groups (if any) to which the geometrical
 * entity of dimension `dim' and tag `tag' belongs. */
GMSH_API void gmshModelGetPhysicalGroupsForEntity(const int dim,
                                                  const int tag,
                                                  int ** physicalTags, size_t * physicalTags_n,
                                                  int * ierr);

/* Add a physical group of dimension `dim', grouping the elementary entities
 * with tags `tags'. Return the tag of the physical group, equal to `tag' if
 * `tag' is positive, or a new tag if `tag' < 0. */
GMSH_API int gmshModelAddPhysicalGroup(const int dim,
                                       int * tags, size_t tags_n,
                                       const int tag,
                                       int * ierr);

/* Set the name of the physical group of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelSetPhysicalName(const int dim,
                                       const int tag,
                                       const char * name,
                                       int * ierr);

/* Get the name of the physical group of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetPhysicalName(const int dim,
                                       const int tag,
                                       char ** name,
                                       int * ierr);

/* Get the boundary of the geometrical entities `dimTags'. Return in
 * `outDimTags' the boundary of the individual entities (if `combined' is
 * false) or the boundary of the combined geometrical shape formed by all
 * input entities (if `combined' is true). Return tags multiplied by the sign
 * of the boundary entity if `oriented' is true. Apply the boundary operator
 * recursively down to dimension 0 (i.e. to points) if `recursive' is true. */
GMSH_API void gmshModelGetBoundary(int * dimTags, size_t dimTags_n,
                                   int ** outDimTags, size_t * outDimTags_n,
                                   const int combined,
                                   const int oriented,
                                   const int recursive,
                                   int * ierr);

/* Get the (elementary) geometrical entities in the bounding box defined by
 * the two points (`xmin', `ymin', `zmin') and (`xmax', `ymax', `zmax'). If
 * `dim' is >= 0, return only the entities of the specified dimension (e.g.
 * points if `dim' == 0). */
GMSH_API void gmshModelGetEntitiesInBoundingBox(const double xmin,
                                                const double ymin,
                                                const double zmin,
                                                const double xmax,
                                                const double ymax,
                                                const double zmax,
                                                int ** tags, size_t * tags_n,
                                                const int dim,
                                                int * ierr);

/* Get the bounding box (`xmin', `ymin', `zmin'), (`xmax', `ymax', `zmax') of
 * the geometrical entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetBoundingBox(const int dim,
                                      const int tag,
                                      double * xmin,
                                      double * ymin,
                                      double * zmin,
                                      double * xmax,
                                      double * ymax,
                                      double * zmax,
                                      int * ierr);

/* Get the geometrical dimension of the current model. */
GMSH_API int gmshModelGetDimension(int * ierr);

/* Add a discrete geometrical entity (defined by a mesh) of dimension `dim' in
 * the current model. Return the tag of the new discrete entity, equal to
 * `tag' if `tag' is positive, or a new tag if `tag' < 0. `boundary' specifies
 * the tags of the entities on the boundary of the discrete entity, if any.
 * Specyfing `boundary' allows Gmsh to construct the topology of the overall
 * model. */
GMSH_API int gmshModelAddDiscreteEntity(const int dim,
                                        const int tag,
                                        int * boundary, size_t boundary_n,
                                        int * ierr);

/* Remove the entities `dimTags' of the current model. If `recursive' is true,
 * remove all the entities on their boundaries, down to dimension 0. */
GMSH_API void gmshModelRemoveEntities(int * dimTags, size_t dimTags_n,
                                      const int recursive,
                                      int * ierr);

/* Remove the physical groups `dimTags' of the current model. If `dimTags' is
 * empty, remove all groups. */
GMSH_API void gmshModelRemovePhysicalGroups(int * dimTags, size_t dimTags_n,
                                            int * ierr);

/* Get the type of the entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetType(const int dim,
                               const int tag,
                               char ** entityType,
                               int * ierr);

/* In a partitioned model, get the parent of the entity of dimension `dim' and
 * tag `tag', i.e. from which the entity is a part of, if any. `parentDim' and
 * `parentTag' are set to -1 if the entity has no parent. */
GMSH_API void gmshModelGetParent(const int dim,
                                 const int tag,
                                 int * parentDim,
                                 int * parentTag,
                                 int * ierr);

/* Evaluate the parametrization of the entity of dimension `dim' and tag `tag'
 * at the parametric coordinates `parametricCoord'. Only valid for `dim' equal
 * to 0 (with empty `parametricCoord'), 1 (with `parametricCoord' containing
 * parametric coordinates on the curve) or 2 (with `parametricCoord'
 * containing pairs of u, v parametric coordinates on the surface,
 * concatenated: [p1u, p1v, p2u, ...]). Return triplets of x, y, z coordinates
 * in `points', concatenated: [p1x, p1y, p1z, p2x, ...]. */
GMSH_API void gmshModelGetValue(const int dim,
                                const int tag,
                                double * parametricCoord, size_t parametricCoord_n,
                                double ** points, size_t * points_n,
                                int * ierr);

/* Evaluate the derivative of the parametrization of the entity of dimension
 * `dim' and tag `tag' at the parametric coordinates `parametricCoord'. Only
 * valid for `dim' equal to 1 (with `parametricCoord' containing parametric
 * coordinates on the curve) or 2 (with `parametricCoord' containing pairs of
 * u, v parametric coordinates on the surface, concatenated: [p1u, p1v, p2u,
 * ...]). For `dim' equal to 1 return the x, y, z components of the derivative
 * with respect to u [d1ux, d1uy, d1uz, d2ux, ...]; for `dim' equal to 2
 * return the x, y, z components of the derivate with respect to u and v:
 * [d1ux, d1uy, d1uz, d1vx, d1vy, d1vz, d2ux, ...]. */
GMSH_API void gmshModelGetDerivative(const int dim,
                                     const int tag,
                                     double * parametricCoord, size_t parametricCoord_n,
                                     double ** derivatives, size_t * derivatives_n,
                                     int * ierr);

/* Evaluate the (maximum) curvature of the entity of dimension `dim' and tag
 * `tag' at the parametric coordinates `parametricCoord'. Only valid for `dim'
 * equal to 1 (with `parametricCoord' containing parametric coordinates on the
 * curve) or 2 (with `parametricCoord' containing pairs of u, v parametric
 * coordinates on the surface, concatenated: [p1u, p1v, p2u, ...]). */
GMSH_API void gmshModelGetCurvature(const int dim,
                                    const int tag,
                                    double * parametricCoord, size_t parametricCoord_n,
                                    double ** curvatures, size_t * curvatures_n,
                                    int * ierr);

/* Evaluate the principal curvatures of the surface with tag `tag' at the
 * parametric coordinates `parametricCoord', as well as their respective
 * directions. `parametricCoord' are given by pair of u and v coordinates,
 * concatenated: [p1u, p1v, p2u, ...]. */
GMSH_API void gmshModelGetPrincipalCurvatures(const int tag,
                                              double * parametricCoord, size_t parametricCoord_n,
                                              double ** curvatureMax, size_t * curvatureMax_n,
                                              double ** curvatureMin, size_t * curvatureMin_n,
                                              double ** directionMax, size_t * directionMax_n,
                                              double ** directionMin, size_t * directionMin_n,
                                              int * ierr);

/* Get the normal to the surface with tag `tag' at the parametric coordinates
 * `parametricCoord'. `parametricCoord' are given by pairs of u and v
 * coordinates, concatenated: [p1u, p1v, p2u, ...]. `normals' are returned as
 * triplets of x, y, z components, concatenated: [n1x, n1y, n1z, n2x, ...]. */
GMSH_API void gmshModelGetNormal(const int tag,
                                 double * parametricCoord, size_t parametricCoord_n,
                                 double ** normals, size_t * normals_n,
                                 int * ierr);

/* Set the visibility of the geometrical entities `dimTags' to `value'. Apply
 * the visibility setting recursively if `recursive' is true. */
GMSH_API void gmshModelSetVisibility(int * dimTags, size_t dimTags_n,
                                     const int value,
                                     const int recursive,
                                     int * ierr);

/* Get the visibility of the geometrical entity of dimension `dim' and tag
 * `tag'. */
GMSH_API void gmshModelGetVisibility(const int dim,
                                     const int tag,
                                     int * value,
                                     int * ierr);

/* Set the color of the geometrical entities `dimTags' to the RGBA value (`r',
 * `g', `b', `a'), where `r', `g', `b' and `a' should be integers between 0
 * and 255. Apply the color setting recursively if `recursive' is true. */
GMSH_API void gmshModelSetColor(int * dimTags, size_t dimTags_n,
                                const int r,
                                const int g,
                                const int b,
                                const int a,
                                const int recursive,
                                int * ierr);

/* Get the color of the geometrical entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetColor(const int dim,
                                const int tag,
                                int * r,
                                int * g,
                                int * b,
                                int * a,
                                int * ierr);

/* Generate a mesh of the current model, up to dimension `dim' (0, 1, 2 or 3). */
GMSH_API void gmshModelMeshGenerate(const int dim,
                                    int * ierr);

/* Partition the mesh of the current model into `numPart' partitions. */
GMSH_API void gmshModelMeshPartition(const int numPart,
                                     int * ierr);

/* Refine the mesh of the current model by uniformly splitting the elements. */
GMSH_API void gmshModelMeshRefine(int * ierr);

/* Set the order of the elements in the mesh of the current model to `order'. */
GMSH_API void gmshModelMeshSetOrder(const int order,
                                    int * ierr);

/* Get the last entities (if any) where a meshing error occurred. Currently
 * only populated by the new 3D meshing algorithms. */
GMSH_API void gmshModelMeshGetLastEntityError(int ** dimTags, size_t * dimTags_n,
                                              int * ierr);

/* Get the last nodes (if any) where a meshing error occurred. Currently only
 * populated by the new 3D meshing algorithms. */
GMSH_API void gmshModelMeshGetLastNodeError(int ** nodeTags, size_t * nodeTags_n,
                                            int * ierr);

/* Get the nodes classified on the entity of dimension `dim' and tag `tag'. If
 * `tag' < 0, get the nodes for all entities of dimension `dim'. If `dim' and
 * `tag' are negative, get all the nodes in the mesh. `nodeTags' contains the
 * node tags (their unique, strictly positive identification numbers). `coord'
 * is a vector of length 3 times the length of `nodeTags' that contains the x,
 * y, z coordinates of the nodes, concatenated: [n1x, n1y, n1z, n2x, ...]. If
 * `dim' >= 0, `parametricCoord' contains the parametric coordinates ([u1, u2,
 * ...] or [u1, v1, u2, ...]) of the nodes, if available. The length of
 * `parametricCoord' can be 0 or `dim' times the length of `nodeTags'. If
 * `includeBoundary' is set, also return the nodes classified on the boundary
 * of the entity (wich will be reparametrized on the entity if `dim' >= 0 in
 * order to compute their parametric coordinates). */
GMSH_API void gmshModelMeshGetNodes(int ** nodeTags, size_t * nodeTags_n,
                                    double ** coord, size_t * coord_n,
                                    double ** parametricCoord, size_t * parametricCoord_n,
                                    const int dim,
                                    const int tag,
                                    const int includeBoundary,
                                    int * ierr);

/* Get the coordinates and the parametric coordinates (if any) of the node
 * with tag `tag'. This is a sometimes useful but inefficient way of accessing
 * nodes, as it relies on a cache stored in the model. For large meshes all
 * the nodes in the model should be numbered in a continuous sequence of tags
 * from 1 to N to maintain reasonnable performance (in this case the internal
 * cache is based on a vector; otherwise it uses a map). */
GMSH_API void gmshModelMeshGetNode(const int nodeTag,
                                   double ** coord, size_t * coord_n,
                                   double ** parametricCoord, size_t * parametricCoord_n,
                                   int * ierr);

/* Rebuild the node cache. */
GMSH_API void gmshModelMeshRebuildNodeCache(const int onlyIfNecessary,
                                            int * ierr);

/* Get the nodes from all the elements belonging to the physical group of
 * dimension `dim' and tag `tag'. `nodeTags' contains the node tags; `coord'
 * is a vector of length 3 times the length of `nodeTags' that contains the x,
 * y, z coordinates of the nodes, concatenated: [n1x, n1y, n1z, n2x, ...]. */
GMSH_API void gmshModelMeshGetNodesForPhysicalGroup(const int dim,
                                                    const int tag,
                                                    int ** nodeTags, size_t * nodeTags_n,
                                                    double ** coord, size_t * coord_n,
                                                    int * ierr);

/* Set the nodes classified on the geometrical entity of dimension `dim' and
 * tag `tag'. `nodeTags' contains the node tags (their unique, strictly
 * positive identification numbers). `coord' is a vector of length 3 times the
 * length of `nodeTags' that contains the x, y, z coordinates of the nodes,
 * concatenated: [n1x, n1y, n1z, n2x, ...]. The optional `parametricCoord'
 * vector contains the parametric coordinates of the nodes, if any. The length
 * of `parametricCoord' can be 0 or `dim' times the length of `nodeTags'. */
GMSH_API void gmshModelMeshSetNodes(const int dim,
                                    const int tag,
                                    int * nodeTags, size_t nodeTags_n,
                                    double * coord, size_t coord_n,
                                    double * parametricCoord, size_t parametricCoord_n,
                                    int * ierr);

/* Reclassify all nodes on their associated geometrical entity, based on the
 * elements. Can be used when importing nodes in bulk (e.g. by associating
 * them all to a single volume), to reclassify them correctly on model
 * surfaces, curves, etc. after the elements have been set. */
GMSH_API void gmshModelMeshReclassifyNodes(int * ierr);

/* Get the elements classified on the entity of dimension `dim' and tag `tag'.
 * If `tag' < 0, get the elements for all entities of dimension `dim'. If
 * `dim' and `tag' are negative, get all the elements in the mesh.
 * `elementTypes' contains the MSH types of the elements (e.g. `2' for 3-node
 * triangles: see `getElementProperties' to obtain the properties for a given
 * element type). `elementTags' is a vector of the same length as
 * `elementTypes'; each entry is a vector containing the tags (unique,
 * strictly positive identifiers) of the elements of the corresponding type.
 * `nodeTags' is also a vector of the same length as `elementTypes'; each
 * entry is a vector of length equal to the number of elements of the given
 * type times the number N of nodes for this type of element, that contains
 * the node tags of all the elements of the given type, concatenated: [e1n1,
 * e1n2, ..., e1nN, e2n1, ...]. */
GMSH_API void gmshModelMeshGetElements(int ** elementTypes, size_t * elementTypes_n,
                                       int *** elementTags, size_t ** elementTags_n, size_t *elementTags_nn,
                                       int *** nodeTags, size_t ** nodeTags_n, size_t *nodeTags_nn,
                                       const int dim,
                                       const int tag,
                                       int * ierr);

/* Get the type and node tags of the element with tag `tag'. This is a
 * sometimes useful but inefficient way of accessing elements, as it relies on
 * a cache stored in the model. For large meshes all the elements in the model
 * should be numbered in a continuous sequence of tags from 1 to N to maintain
 * reasonnable performance (in this case the internal cache is based on a
 * vector; otherwise it uses a map). */
GMSH_API void gmshModelMeshGetElement(const int elementTag,
                                      int * elementType,
                                      int ** nodeTags, size_t * nodeTags_n,
                                      int * ierr);

/* Get the tag, type and node tags of the element located at coordinates (`x',
 * `y', `z'). This is a sometimes useful but inefficient way of accessing
 * elements, as it relies on a search in a spatial octree. */
GMSH_API void gmshModelMeshGetElementByCoordinates(const double x,
                                                   const double y,
                                                   const double z,
                                                   int * elementTag,
                                                   int * elementType,
                                                   int ** nodeTags, size_t * nodeTags_n,
                                                   int * ierr);

/* Set the elements of the entity of dimension `dim' and tag `tag'. `types'
 * contains the MSH types of the elements (e.g. `2' for 3-node triangles: see
 * the Gmsh reference manual). `elementTags' is a vector of the same length as
 * `types'; each entry is a vector containing the tags (unique, strictly
 * positive identifiers) of the elements of the corresponding type. `nodeTags'
 * is also a vector of the same length as `types'; each entry is a vector of
 * length equal to the number of elements of the given type times the number N
 * of nodes per element, that contains the node tags of all the elements of
 * the given type, concatenated: [e1n1, e1n2, ..., e1nN, e2n1, ...]. */
GMSH_API void gmshModelMeshSetElements(const int dim,
                                       const int tag,
                                       int * elementTypes, size_t elementTypes_n,
                                       const int ** elementTags, const size_t * elementTags_n, size_t elementTags_nn,
                                       const int ** nodeTags, const size_t * nodeTags_n, size_t nodeTags_nn,
                                       int * ierr);

/* Get the types of elements in the entity of dimension `dim' and tag `tag'.
 * If `tag' < 0, get the types for all entities of dimension `dim'. If `dim'
 * and `tag' are negative, get all the types in the mesh. */
GMSH_API void gmshModelMeshGetElementTypes(int ** elementTypes, size_t * elementTypes_n,
                                           const int dim,
                                           const int tag,
                                           int * ierr);

/* Get the properties of an element of type `elementType': its name
 * (`elementName'), dimension (`dim'), order (`order'), number of nodes
 * (`numNodes') and parametric node coordinates (`parametricCoord' vector, of
 * length `dim' times `numNodes'). */
GMSH_API void gmshModelMeshGetElementProperties(const int elementType,
                                                char ** elementName,
                                                int * dim,
                                                int * order,
                                                int * numNodes,
                                                double ** parametricCoord, size_t * parametricCoord_n,
                                                int * ierr);

/* Get the elements of type `elementType' classified on the entity of of tag
 * `tag'. If `tag' < 0, get the elements for all entities. `elementTags' is a
 * vector containing the tags (unique, strictly positive identifiers) of the
 * elements of the corresponding type. `nodeTags' is a vector of length equal
 * to the number of elements of the given type times the number N of nodes for
 * this type of element, that contains the node tags of all the elements of
 * the given type, concatenated: [e1n1, e1n2, ..., e1nN, e2n1, ...]. If
 * `numTasks' > 1, only compute and return the part of the data indexed by
 * `task'. */
GMSH_API void gmshModelMeshGetElementsByType(const int elementType,
                                             int ** elementTags, size_t * elementTags_n,
                                             int ** nodeTags, size_t * nodeTags_n,
                                             const int tag,
                                             const size_t task,
                                             const size_t numTasks,
                                             int * ierr);

/* Preallocate the data for `getElementsByType'. This is necessary only if
 * `getElementsByType' is called with `numTasks' > 1. */
GMSH_API void gmshModelMeshPreallocateElementsByType(const int elementType,
                                                     const int elementTag,
                                                     const int nodeTag,
                                                     int ** elementTags, size_t * elementTags_n,
                                                     int ** nodeTags, size_t * nodeTags_n,
                                                     const int tag,
                                                     int * ierr);

/* Get the Jacobians of all the elements of type `elementType' classified on
 * the entity of dimension `dim' and tag `tag', at the G integration points
 * required by the `integrationType' integration rule (e.g. "Gauss4"). Data is
 * returned by element, with elements in the same order as in `getElements'
 * and `getElementsByType'. `jacobians' contains for each element the 9
 * entries of a 3x3 Jacobian matrix (by row), for each integration point:
 * [e1g1Jxx, e1g1Jxy, e1g1Jxz, ... e1g1Jzz, e1g2Jxx, ..., e1gGJzz, e2g1Jxx,
 * ...]. `determinants' contains for each element the determinant of the
 * Jacobian matrix for each integration point: [e1g1, e1g2, ... e1gG, e2g1,
 * ...]. `points' contains for each element the x, y, z coordinates of the
 * integration points. If `tag' < 0, get the Jacobian data for all entities.
 * If `numTasks' > 1, only compute and return the part of the data indexed by
 * `task'. */
GMSH_API void gmshModelMeshGetJacobians(const int elementType,
                                        const char * integrationType,
                                        double ** jacobians, size_t * jacobians_n,
                                        double ** determinants, size_t * determinants_n,
                                        double ** points, size_t * points_n,
                                        const int tag,
                                        const size_t task,
                                        const size_t numTasks,
                                        int * ierr);

/* Preallocate the data required by `getJacobians'. This is necessary only if
 * `getJacobians' is called with `numTasks' > 1. */
GMSH_API void gmshModelMeshPreallocateJacobians(const int elementType,
                                                const char * integrationType,
                                                const int jacobian,
                                                const int determinant,
                                                const int point,
                                                double ** jacobians, size_t * jacobians_n,
                                                double ** determinants, size_t * determinants_n,
                                                double ** points, size_t * points_n,
                                                const int tag,
                                                int * ierr);

/* Get the basis functions of the element of type `elementType' for the given
 * `integrationType' integration rule (e.g. "Gauss4") and `functionSpaceType'
 * function space (e.g. "IsoParametric"). `integrationPoints' contains the
 * parametric coordinates u, v, w and the weight q for each integeration
 * point, concatenated: [g1u, g1v, g1w, g1q, g2u, ...]. `numComponents'
 * returns the number C of components of a basis function. `basisFunctions'
 * contains the evaluation of the basis functions at the integration points:
 * [g1f1, ..., g1fC, g2f1, ...]. */
GMSH_API void gmshModelMeshGetBasisFunctions(const int elementType,
                                             const char * integrationType,
                                             const char * functionSpaceType,
                                             double ** integrationPoints, size_t * integrationPoints_n,
                                             int * numComponents,
                                             double ** basisFunctions, size_t * basisFunctions_n,
                                             int * ierr);

/* Precomputes the basis functions corresponding to `elementType'. */
GMSH_API void gmshModelMeshPrecomputeBasisFunctions(const int elementType,
                                                    int * ierr);

/* Get the barycenters of all elements of type `elementType' classified on the
 * entity of tag `tag'. If `primary' is set, only the primary nodes of the
 * elements are taken into account for the barycenter calculation. If `fast'
 * is set, the function returns the sum of the primary node coordinates
 * (without normalizing by the number of nodes). If `numTasks' > 1, only
 * compute and return the part of the data indexed by `task'. */
GMSH_API void gmshModelMeshGetBarycenters(const int elementType,
                                          const int tag,
                                          const int fast,
                                          const int primary,
                                          double ** barycenters, size_t * barycenters_n,
                                          const size_t task,
                                          const size_t numTasks,
                                          int * ierr);

/* Preallocate the data required by `getBarycenters'. This is necessary only
 * if `getBarycenters' is called with `numTasks' > 1. */
GMSH_API void gmshModelMeshPreallocateBarycenters(const int elementType,
                                                  double ** barycenters, size_t * barycenters_n,
                                                  const int tag,
                                                  int * ierr);

/* Set a mesh size constraint on the geometrical entities `dimTags'. Currently
 * only entities of dimension 0 (points) are handled. */
GMSH_API void gmshModelMeshSetSize(int * dimTags, size_t dimTags_n,
                                   const double size,
                                   int * ierr);

/* Set a transfinite meshing constraint on the curve `tag', with `numNodes'
 * nodes distributed according to `meshType' and `coef'. Currently supported
 * types are "Progression" (geometrical progression with power `coef') and
 * "Bump" (refinement toward both extremities of the curve). */
GMSH_API void gmshModelMeshSetTransfiniteCurve(const int tag,
                                               const int numNodes,
                                               const char * meshType,
                                               const double coef,
                                               int * ierr);

/* Set a transfinite meshing constraint on the surface `tag'. `arrangement'
 * describes the arrangement of the triangles when the surface is not flagged
 * as recombined: currently supported values are "Left", "Right",
 * "AlternateLeft" and "AlternateRight". `cornerTags' can be used to specify
 * the (3 or 4) corners of the transfinite interpolation explicitly;
 * specifying the corners explicitly is mandatory if the surface has more that
 * 3 or 4 points on its boundary. */
GMSH_API void gmshModelMeshSetTransfiniteSurface(const int tag,
                                                 const char * arrangement,
                                                 int * cornerTags, size_t cornerTags_n,
                                                 int * ierr);

/* Set a transfinite meshing constraint on the surface `tag'. `cornerTags' can
 * be used to specify the (6 or 8) corners of the transfinite interpolation
 * explicitly. */
GMSH_API void gmshModelMeshSetTransfiniteVolume(const int tag,
                                                int * cornerTags, size_t cornerTags_n,
                                                int * ierr);

/* Set a recombination meshing constraint on the geometrical entity of
 * dimension `dim' and tag `tag'. Currently only entities of dimension 2 (to
 * recombine triangles into quadrangles) are supported. */
GMSH_API void gmshModelMeshSetRecombine(const int dim,
                                        const int tag,
                                        int * ierr);

/* Set a smoothing meshing constraint on the geometrical entity of dimension
 * `dim' and tag `tag'. `val' iterations of a Laplace smoother are applied. */
GMSH_API void gmshModelMeshSetSmoothing(const int dim,
                                        const int tag,
                                        const int val,
                                        int * ierr);

/* Set a reverse meshing constraint on the geometrical entity of dimension
 * `dim' and tag `tag'. If `val' is true, the mesh orientation will be
 * reversed with respect to the natural mesh orientation (i.e. the orientation
 * consistent with the orientation of the geometrical entity). If `val' is
 * false, the mesh is left as-is. */
GMSH_API void gmshModelMeshSetReverse(const int dim,
                                      const int tag,
                                      const int val,
                                      int * ierr);

/* Set meshing constraints on the bounding surfaces of the volume of tag `tag'
 * so that all surfaces are oriented with outward pointing normals. Currently
 * only available with the OpenCASCADE kernel, as it relies on the STL
 * triangulation. */
GMSH_API void gmshModelMeshSetOutwardOrientation(const int tag,
                                                 int * ierr);

/* Embed the geometrical entities of dimension `dim' and tags `tags' in the
 * (inDim, inTag) geometrical entity. `inDim' must be strictly greater than
 * `dim'. */
GMSH_API void gmshModelMeshEmbed(const int dim,
                                 int * tags, size_t tags_n,
                                 const int inDim,
                                 const int inTag,
                                 int * ierr);

/* Reorder the elements of type `elementType' classified on the entity of tag
 * `tag' according to `ordering'. */
GMSH_API void gmshModelMeshReorderElements(const int elementType,
                                           const int tag,
                                           int * ordering, size_t ordering_n,
                                           int * ierr);

/* Renumber the node tags in a contiunous sequence. */
GMSH_API void gmshModelMeshRenumberNodes(int * ierr);

/* Renumber the element tags in a contiunous sequence. */
GMSH_API void gmshModelMeshRenumberElements(int * ierr);

/* Set the meshes of the entities of dimension `dim' and tag `tags' as
 * periodic copies of the meshes of entities `tagsSource', using the affine
 * transformation specified in `affineTransformation' (16 entries of a 4x4
 * matrix, by row). Currently only available for `dim' == 1 and `dim' == 2. */
GMSH_API void gmshModelMeshSetPeriodic(const int dim,
                                       int * tags, size_t tags_n,
                                       int * tagsSource, size_t tagsSource_n,
                                       double * affineTransformation, size_t affineTransformation_n,
                                       int * ierr);

/* Get the master entity, periodic node pairs and affine transform for the
 * entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelMeshGetPeriodicNodes(const int dim,
                                            const int tag,
                                            int * tagMaster,
                                            int ** nodes, size_t * nodes_n,
                                            double ** affineTransform, size_t * affineTransform_n,
                                            int * ierr);

/* Remove duplicate nodes in the mesh of the current model. */
GMSH_API void gmshModelMeshRemoveDuplicateNodes(int * ierr);

/* Create a boundary representation from the mesh if the model does not have
 * one (e.g. when imported from mesh file formats with no BRep representation
 * of the underlying model). Warning: this is an experimental feature. */
GMSH_API void gmshModelMeshCreateTopology(int * ierr);

/* Create a parametrization for curves and surfaces that do not have one (i.e.
 * discrete curves and surfaces represented solely by meshes, without an
 * underlying CAD description). Warning: this is an experimental feature. */
GMSH_API void gmshModelMeshCreateGeometry(int * ierr);

/* Compute a basis representation for homology spaces after a mesh has been
 * generated. The computation domain is given in a list of physical group tags
 * `domainTags'; if empty, the whole mesh is the domain. The computation
 * subdomain for relative homology computation is given in a list of physical
 * group tags `subdomainTags'; if empty, absolute homology is computed. The
 * dimensions homology bases to be computed are given in the list `dim'; if
 * empty, all bases are computed. Resulting basis representation chains are
 * stored as physical groups in the mesh. */
GMSH_API void gmshModelMeshComputeHomology(int * domainTags, size_t domainTags_n,
                                           int * subdomainTags, size_t subdomainTags_n,
                                           int * dims, size_t dims_n,
                                           int * ierr);

/* Compute a basis representation for cohomology spaces after a mesh has been
 * generated. The computation domain is given in a list of physical group tags
 * `domainTags'; if empty, the whole mesh is the domain. The computation
 * subdomain for relative cohomology computation is given in a list of
 * physical group tags `subdomainTags'; if empty, absolute cohomology is
 * computed. The dimensions homology bases to be computed are given in the
 * list `dim'; if empty, all bases are computed. Resulting basis
 * representation cochains are stored as physical groups in the mesh. */
GMSH_API void gmshModelMeshComputeCohomology(int * domainTags, size_t domainTags_n,
                                             int * subdomainTags, size_t subdomainTags_n,
                                             int * dims, size_t dims_n,
                                             int * ierr);

/* Add a new mesh size field of type `fieldType'. If `tag' is positive, assign
 * the tag explcitly; otherwise a new tag is assigned automatically. Return
 * the field tag. */
GMSH_API int gmshModelMeshFieldAdd(const char * fieldType,
                                   const int tag,
                                   int * ierr);

/* Remove the field with tag `tag'. */
GMSH_API void gmshModelMeshFieldRemove(const int tag,
                                       int * ierr);

/* Set the numerical option `option' to value `value' for field `tag'. */
GMSH_API void gmshModelMeshFieldSetNumber(const int tag,
                                          const char * option,
                                          const double value,
                                          int * ierr);

/* Set the string option `option' to value `value' for field `tag'. */
GMSH_API void gmshModelMeshFieldSetString(const int tag,
                                          const char * option,
                                          const char * value,
                                          int * ierr);

/* Set the numerical list option `option' to value `value' for field `tag'. */
GMSH_API void gmshModelMeshFieldSetNumbers(const int tag,
                                           const char * option,
                                           double * value, size_t value_n,
                                           int * ierr);

/* Set the field `tag' as the background mesh size field. */
GMSH_API void gmshModelMeshFieldSetAsBackgroundMesh(const int tag,
                                                    int * ierr);

/* Set the field `tag' as a boundary layer size field. */
GMSH_API void gmshModelMeshFieldSetAsBoundaryLayer(const int tag,
                                                   int * ierr);

/* Add a geometrical point in the internal GEO CAD representation, at
 * coordinates (`x', `y', `z'). If `meshSize' is > 0, add a meshing constraint
 * at that point. If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. Return the tag of the point. (Note that
 * the point will be added in the current model only after `synchronize' is
 * called. This behavior holds for all the entities added in the geo module.) */
GMSH_API int gmshModelGeoAddPoint(const double x,
                                  const double y,
                                  const double z,
                                  const double meshSize,
                                  const int tag,
                                  int * ierr);

/* Add a straight line segment between the two points with tags `startTag' and
 * `endTag'. If `tag' is positive, set the tag explicitly; otherwise a new tag
 * is selected automatically. Return the tag of the line. */
GMSH_API int gmshModelGeoAddLine(const int startTag,
                                 const int endTag,
                                 const int tag,
                                 int * ierr);

/* Add a circle arc (stricly smaller than Pi) between the two points with tags
 * `startTag' and `endTag', with center `centertag'. If `tag' is positive, set
 * the tag explicitly; otherwise a new tag is selected automatically. If
 * (`nx', `ny', `nz') != (0,0,0), explicitely set the plane of the circle arc.
 * Return the tag of the circle arc. */
GMSH_API int gmshModelGeoAddCircleArc(const int startTag,
                                      const int centerTag,
                                      const int endTag,
                                      const int tag,
                                      const double nx,
                                      const double ny,
                                      const double nz,
                                      int * ierr);

/* Add an ellipse arc (stricly smaller than Pi) between the two points
 * `startTag' and `endTag', with center `centertag' and major axis point
 * `majorTag'. If `tag' is positive, set the tag explicitly; otherwise a new
 * tag is selected automatically. If (`nx', `ny', `nz') != (0,0,0),
 * explicitely set the plane of the circle arc. Return the tag of the ellipse
 * arc. */
GMSH_API int gmshModelGeoAddEllipseArc(const int startTag,
                                       const int centerTag,
                                       const int majorTag,
                                       const int endTag,
                                       const int tag,
                                       const double nx,
                                       const double ny,
                                       const double nz,
                                       int * ierr);

/* Add a spline (Catmull-Rom) curve going through the points `pointTags'. If
 * `tag' is positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. Create a periodic curve if the first and last points are the
 * same. Return the tag of the spline curve. */
GMSH_API int gmshModelGeoAddSpline(int * pointTags, size_t pointTags_n,
                                   const int tag,
                                   int * ierr);

/* Add a cubic b-spline curve with `pointTags' control points. If `tag' is
 * positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. Creates a periodic curve if the first and last points are
 * the same. Return the tag of the b-spline curve. */
GMSH_API int gmshModelGeoAddBSpline(int * pointTags, size_t pointTags_n,
                                    const int tag,
                                    int * ierr);

/* Add a Bezier curve with `pointTags' control points. If `tag' is positive,
 * set the tag explicitly; otherwise a new tag is selected automatically.
 * Return the tag of the Bezier curve. */
GMSH_API int gmshModelGeoAddBezier(int * pointTags, size_t pointTags_n,
                                   const int tag,
                                   int * ierr);

/* Add a curve loop (a closed wire) formed by the curves `curveTags'.
 * `curveTags' should contain (signed) tags of geometrical enties of dimension
 * 1 forming a closed loop: a negative tag signifies that the underlying curve
 * is considered with reversed orientation. If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. Return the tag
 * of the curve loop. */
GMSH_API int gmshModelGeoAddCurveLoop(int * curveTags, size_t curveTags_n,
                                      const int tag,
                                      int * ierr);

/* Add a plane surface defined by one or more curve loops `wireTags'. The
 * first curve loop defines the exterior contour; additional curve loop define
 * holes. If `tag' is positive, set the tag explicitly; otherwise a new tag is
 * selected automatically. Return the tag of the surface. */
GMSH_API int gmshModelGeoAddPlaneSurface(int * wireTags, size_t wireTags_n,
                                         const int tag,
                                         int * ierr);

/* Add a surface filling the curve loops in `wireTags'. Currently only a
 * single curve loop is supported; this curve loop should be composed by 3 or
 * 4 curves only. If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. Return the tag of the surface. */
GMSH_API int gmshModelGeoAddSurfaceFilling(int * wireTags, size_t wireTags_n,
                                           const int tag,
                                           const int sphereCenterTag,
                                           int * ierr);

/* Add a surface loop (a closed shell) formed by `surfaceTags'.  If `tag' is
 * positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. Return the tag of the shell. */
GMSH_API int gmshModelGeoAddSurfaceLoop(int * surfaceTags, size_t surfaceTags_n,
                                        const int tag,
                                        int * ierr);

/* Add a volume (a region) defined by one or more shells `shellTags'. The
 * first surface loop defines the exterior boundary; additional surface loop
 * define holes. If `tag' is positive, set the tag explicitly; otherwise a new
 * tag is selected automatically. Return the tag of the volume. */
GMSH_API int gmshModelGeoAddVolume(int * shellTags, size_t shellTags_n,
                                   const int tag,
                                   int * ierr);

/* Extrude the geometrical entities `dimTags' by translation along (`dx',
 * `dy', `dz'). Return extruded entities in `outDimTags'. If `numElements' is
 * not empty, also extrude the mesh: the entries in `numElements' give the
 * number of elements in each layer. If `height' is not empty, it provides the
 * (cummulative) height of the different layers, normalized to 1. */
GMSH_API void gmshModelGeoExtrude(int * dimTags, size_t dimTags_n,
                                  const double dx,
                                  const double dy,
                                  const double dz,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  int * numElements, size_t numElements_n,
                                  double * heights, size_t heights_n,
                                  const int recombine,
                                  int * ierr);

/* Extrude the geometrical entities `dimTags' by rotation of `angle' radians
 * around the axis of revolution defined by the point (`x', `y', `z') and the
 * direction (`ax', `ay', `az'). Return extruded entities in `outDimTags'. If
 * `numElements' is not empty, also extrude the mesh: the entries in
 * `numElements' give the number of elements in each layer. If `height' is not
 * empty, it provides the (cummulative) height of the different layers,
 * normalized to 1. */
GMSH_API void gmshModelGeoRevolve(int * dimTags, size_t dimTags_n,
                                  const double x,
                                  const double y,
                                  const double z,
                                  const double ax,
                                  const double ay,
                                  const double az,
                                  const double angle,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  int * numElements, size_t numElements_n,
                                  double * heights, size_t heights_n,
                                  const int recombine,
                                  int * ierr);

/* Extrude the geometrical entities `dimTags' by a combined translation and
 * rotation of `angle' radians, along (`dx', `dy', `dz') and around the axis
 * of revolution defined by the point (`x', `y', `z') and the direction (`ax',
 * `ay', `az'). Return extruded entities in `outDimTags'. If `numElements' is
 * not empty, also extrude the mesh: the entries in `numElements' give the
 * number of elements in each layer. If `height' is not empty, it provides the
 * (cummulative) height of the different layers, normalized to 1. */
GMSH_API void gmshModelGeoTwist(int * dimTags, size_t dimTags_n,
                                const double x,
                                const double y,
                                const double z,
                                const double dx,
                                const double dy,
                                const double dz,
                                const double ax,
                                const double ay,
                                const double az,
                                const double angle,
                                int ** outDimTags, size_t * outDimTags_n,
                                int * numElements, size_t numElements_n,
                                double * heights, size_t heights_n,
                                const int recombine,
                                int * ierr);

/* Translate the geometrical entities `dimTags' along (`dx', `dy', `dz'). */
GMSH_API void gmshModelGeoTranslate(int * dimTags, size_t dimTags_n,
                                    const double dx,
                                    const double dy,
                                    const double dz,
                                    int * ierr);

/* Rotate the geometrical entities `dimTags' of `angle' radians around the
 * axis of revolution defined by the point (`x', `y', `z') and the direction
 * (`ax', `ay', `az'). */
GMSH_API void gmshModelGeoRotate(int * dimTags, size_t dimTags_n,
                                 const double x,
                                 const double y,
                                 const double z,
                                 const double ax,
                                 const double ay,
                                 const double az,
                                 const double angle,
                                 int * ierr);

/* Scale the geometrical entities `dimTag' by factors `a', `b' and `c' along
 * the three coordinate axes; use (`x', `y', `z') as the center of the
 * homothetic transformation. */
GMSH_API void gmshModelGeoDilate(int * dimTags, size_t dimTags_n,
                                 const double x,
                                 const double y,
                                 const double z,
                                 const double a,
                                 const double b,
                                 const double c,
                                 int * ierr);

/* Apply a symmetry transformation to the geometrical entities `dimTag', with
 * respect to the plane of equation `a' * x + `b' * y + `c' * z + `d' = 0. */
GMSH_API void gmshModelGeoSymmetrize(int * dimTags, size_t dimTags_n,
                                     const double a,
                                     const double b,
                                     const double c,
                                     const double d,
                                     int * ierr);

/* Copy the entities `dimTags'; the new entities are returned in `outDimTags'. */
GMSH_API void gmshModelGeoCopy(int * dimTags, size_t dimTags_n,
                               int ** outDimTags, size_t * outDimTags_n,
                               int * ierr);

/* Remove the entities `dimTags'. If `recursive' is true, remove all the
 * entities on their boundaries, down to dimension 0. */
GMSH_API void gmshModelGeoRemove(int * dimTags, size_t dimTags_n,
                                 const int recursive,
                                 int * ierr);

/* Remove all duplicate entities (different entities at the same geometrical
 * location). */
GMSH_API void gmshModelGeoRemoveAllDuplicates(int * ierr);

/* Synchronize the internal GEO CAD representation with the current Gmsh
 * model. This can be called at any time, but since it involves a non trivial
 * amount of processing, the number of synchronization points should normally
 * be minimized. */
GMSH_API void gmshModelGeoSynchronize(int * ierr);

/* Set a mesh size constraint on the geometrical entities `dimTags'. Currently
 * only entities of dimension 0 (points) are handled. */
GMSH_API void gmshModelGeoMeshSetSize(int * dimTags, size_t dimTags_n,
                                      const double size,
                                      int * ierr);

/* Set a transfinite meshing constraint on the curve `tag', with `numNodes'
 * nodes distributed according to `meshType' and `coef'. Currently supported
 * types are "Progression" (geometrical progression with power `coef') and
 * "Bump" (refinement toward both extreminties of the curve). */
GMSH_API void gmshModelGeoMeshSetTransfiniteCurve(const int tag,
                                                  const int nPoints,
                                                  const char * meshType,
                                                  const double coef,
                                                  int * ierr);

/* Set a transfinite meshing constraint on the surface `tag'. `arrangement'
 * describes the arrangement of the triangles when the surface is not flagged
 * as recombined: currently supported values are "Left", "Right",
 * "AlternateLeft" and "AlternateRight". `cornerTags' can be used to specify
 * the (3 or 4) corners of the transfinite interpolation explicitly;
 * specifying the corners explicitly is mandatory if the surface has more that
 * 3 or 4 points on its boundary. */
GMSH_API void gmshModelGeoMeshSetTransfiniteSurface(const int tag,
                                                    const char * arrangement,
                                                    int * cornerTags, size_t cornerTags_n,
                                                    int * ierr);

/* Set a transfinite meshing constraint on the surface `tag'. `cornerTags' can
 * be used to specify the (6 or 8) corners of the transfinite interpolation
 * explicitly. */
GMSH_API void gmshModelGeoMeshSetTransfiniteVolume(const int tag,
                                                   int * cornerTags, size_t cornerTags_n,
                                                   int * ierr);

/* Set a recombination meshing constraint on the geometrical entity of
 * dimension `dim' and tag `tag'. Currently only entities of dimension 2 (to
 * recombine triangles into quadrangles) are supported. */
GMSH_API void gmshModelGeoMeshSetRecombine(const int dim,
                                           const int tag,
                                           const double angle,
                                           int * ierr);

/* Set a smoothing meshing constraint on the geometrical entity of dimension
 * `dim' and tag `tag'. `val' iterations of a Laplace smoother are applied. */
GMSH_API void gmshModelGeoMeshSetSmoothing(const int dim,
                                           const int tag,
                                           const int val,
                                           int * ierr);

/* Set a reverse meshing constraint on the geometrical entity of dimension
 * `dim' and tag `tag'. If `val' is true, the mesh orientation will be
 * reversed with respect to the natural mesh orientation (i.e. the orientation
 * consistent with the orientation of the geometrical entity). If `val' is
 * false, the mesh is left as-is. */
GMSH_API void gmshModelGeoMeshSetReverse(const int dim,
                                         const int tag,
                                         const int val,
                                         int * ierr);

/* Add a geometrical point in the internal OpenCASCADE CAD representation, at
 * coordinates (`x', `y', `z'). If `meshSize' is > 0, add a meshing constraint
 * at that point. If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. Return the tag of the point. (Note that
 * the point will be added in the current model only after `synchronize' is
 * called. This behavior holds for all the entities added in the occ module.) */
GMSH_API int gmshModelOccAddPoint(const double x,
                                  const double y,
                                  const double z,
                                  const double meshSize,
                                  const int tag,
                                  int * ierr);

/* Add a straight line segment between the two points with tags `startTag' and
 * `endTag'. If `tag' is positive, set the tag explicitly; otherwise a new tag
 * is selected automatically. Return the tag of the line. */
GMSH_API int gmshModelOccAddLine(const int startTag,
                                 const int endTag,
                                 const int tag,
                                 int * ierr);

/* Add a circle arc between the two points with tags `startTag' and `endTag',
 * with center `centerTag'. If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. Return the tag of the circle
 * arc. */
GMSH_API int gmshModelOccAddCircleArc(const int startTag,
                                      const int centerTag,
                                      const int endTag,
                                      const int tag,
                                      int * ierr);

/* Add a circle of center (`x', `y', `z') and radius `r'. If `tag' is
 * positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. If `angle1' and `angle2' are specified, create a circle arc
 * between the two angles. Return the tag of the circle. */
GMSH_API int gmshModelOccAddCircle(const double x,
                                   const double y,
                                   const double z,
                                   const double r,
                                   const int tag,
                                   const double angle1,
                                   const double angle2,
                                   int * ierr);

/* Add an ellipse arc between the two points with tags `startTag' and
 * `endTag', with center `centerTag'. If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. Return the tag
 * of the ellipse arc. */
GMSH_API int gmshModelOccAddEllipseArc(const int startTag,
                                       const int centerTag,
                                       const int endTag,
                                       const int tag,
                                       int * ierr);

/* Add an ellipse of center (`x', `y', `z') and radii `r1' and `r2' along the
 * x- and y-axes respectively. If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. If `angle1' and `angle2' are
 * specified, create an ellipse arc between the two angles. Return the tag of
 * the ellipse. */
GMSH_API int gmshModelOccAddEllipse(const double x,
                                    const double y,
                                    const double z,
                                    const double r1,
                                    const double r2,
                                    const int tag,
                                    const double angle1,
                                    const double angle2,
                                    int * ierr);

/* Add a spline (C2 b-spline) curve going through the points `pointTags'. If
 * `tag' is positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. Create a periodic curve if the first and last points are the
 * same. Return the tag of the spline curve. */
GMSH_API int gmshModelOccAddSpline(int * pointTags, size_t pointTags_n,
                                   const int tag,
                                   int * ierr);

/* Add a b-spline curve of degree `degree' with `pointTags' control points. If
 * `weights', `knots' or `multiplicities' are not provided, default parameters
 * are computed automatically. If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. Create a periodic curve if
 * the first and last points are the same. Return the tag of the b-spline
 * curve. */
GMSH_API int gmshModelOccAddBSpline(int * pointTags, size_t pointTags_n,
                                    const int tag,
                                    const int degree,
                                    double * weights, size_t weights_n,
                                    double * knots, size_t knots_n,
                                    int * multiplicities, size_t multiplicities_n,
                                    int * ierr);

/* Add a Bezier curve with `pointTags' control points. If `tag' is positive,
 * set the tag explicitly; otherwise a new tag is selected automatically.
 * Return the tag of the Bezier curve. */
GMSH_API int gmshModelOccAddBezier(int * pointTags, size_t pointTags_n,
                                   const int tag,
                                   int * ierr);

/* Add a wire (open or closed) formed by the curves `curveTags'. `curveTags'
 * should contain (signed) tags: a negative tag signifies that the underlying
 * curve is considered with reversed orientation. If `tag' is positive, set
 * the tag explicitly; otherwise a new tag is selected automatically. Return
 * the tag of the wire. */
GMSH_API int gmshModelOccAddWire(int * curveTags, size_t curveTags_n,
                                 const int tag,
                                 const int checkClosed,
                                 int * ierr);

/* Add a curve loop (a closed wire) formed by the curves `curveTags'.
 * `curveTags' should contain tags of curves forming a closed loop. If `tag'
 * is positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. Return the tag of the curve loop. */
GMSH_API int gmshModelOccAddCurveLoop(int * curveTags, size_t curveTags_n,
                                      const int tag,
                                      int * ierr);

/* Add a rectangle with lower left corner at (`x', `y', `z') and upper right
 * corner at (`x' + `dx', `y' + `dy', `z'). If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. Round the
 * corners if `roundedRadius' is nonzero. Return the tag of the rectangle. */
GMSH_API int gmshModelOccAddRectangle(const double x,
                                      const double y,
                                      const double z,
                                      const double dx,
                                      const double dy,
                                      const int tag,
                                      const double roundedRadius,
                                      int * ierr);

/* Add a disk with center (`xc', `yc', `zc') and radius `rx' along the x-axis
 * and `ry' along the y-axis. If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. Return the tag of the disk. */
GMSH_API int gmshModelOccAddDisk(const double xc,
                                 const double yc,
                                 const double zc,
                                 const double rx,
                                 const double ry,
                                 const int tag,
                                 int * ierr);

/* Add a plane surface defined by one or more curve loops (or closed wires)
 * `wireTags'. The first curve loop defines the exterior contour; additional
 * curve loop define holes. If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. Return the tag of the
 * surface. */
GMSH_API int gmshModelOccAddPlaneSurface(int * wireTags, size_t wireTags_n,
                                         const int tag,
                                         int * ierr);

/* Add a surface filling the curve loops in `wireTags'. If `tag' is positive,
 * set the tag explicitly; otherwise a new tag is selected automatically.
 * Return the tag of the surface. */
GMSH_API int gmshModelOccAddSurfaceFilling(const int wireTag,
                                           const int tag,
                                           int * ierr);

/* Add a surface loop (a closed shell) formed by `surfaceTags'.  If `tag' is
 * positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. Return the tag of the surface loop. */
GMSH_API int gmshModelOccAddSurfaceLoop(int * surfaceTags, size_t surfaceTags_n,
                                        const int tag,
                                        int * ierr);

/* Add a volume (a region) defined by one or more surface loops `shellTags'.
 * The first surface loop defines the exterior boundary; additional surface
 * loop define holes. If `tag' is positive, set the tag explicitly; otherwise
 * a new tag is selected automatically. Return the tag of the volume. */
GMSH_API int gmshModelOccAddVolume(int * shellTags, size_t shellTags_n,
                                   const int tag,
                                   int * ierr);

/* Add a sphere of center (`xc', `yc', `zc') and radius `r'. The optional
 * `angle1' and `angle2' arguments define the polar angle opening (from -Pi/2
 * to Pi/2). The optional `angle3' argument defines the azimuthal opening
 * (from 0 to 2*Pi). If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. Return the tag of the sphere. */
GMSH_API int gmshModelOccAddSphere(const double xc,
                                   const double yc,
                                   const double zc,
                                   const double radius,
                                   const int tag,
                                   const double angle1,
                                   const double angle2,
                                   const double angle3,
                                   int * ierr);

/* Add a parallelepipedic box defined by a point (`x', `y', `z') and the
 * extents along the x-, y- and z-axes. If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. Return the tag
 * of the box. */
GMSH_API int gmshModelOccAddBox(const double x,
                                const double y,
                                const double z,
                                const double dx,
                                const double dy,
                                const double dz,
                                const int tag,
                                int * ierr);

/* Add a cylinder, defined by the center (`x', `y', `z') of its first circular
 * face, the 3 components (`dx', `dy', `dz') of the vector defining its axis
 * and its radius `r'. The optional `angle' argument defines the angular
 * opening (from 0 to 2*Pi). If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. Return the tag of the
 * cylinder. */
GMSH_API int gmshModelOccAddCylinder(const double x,
                                     const double y,
                                     const double z,
                                     const double dx,
                                     const double dy,
                                     const double dz,
                                     const double r,
                                     const int tag,
                                     const double angle,
                                     int * ierr);

/* Add a cone, defined by the center (`x', `y', `z') of its first circular
 * face, the 3 components of the vector (`dx', `dy', `dz') defining its axis
 * and the two radii `r1' and `r2' of the faces (these radii can be zero). If
 * `tag' is positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. `angle' defines the optional angular opening (from 0 to
 * 2*Pi). Return the tag of the cone. */
GMSH_API int gmshModelOccAddCone(const double x,
                                 const double y,
                                 const double z,
                                 const double dx,
                                 const double dy,
                                 const double dz,
                                 const double r1,
                                 const double r2,
                                 const int tag,
                                 const double angle,
                                 int * ierr);

/* Add a right angular wedge, defined by the right-angle point (`x', `y', `z')
 * and the 3 extends along the x-, y- and z-axes (`dx', `dy', `dz'). If `tag'
 * is positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. The optional argument `ltx' defines the top extent along the
 * x-axis. Return the tag of the wedge. */
GMSH_API int gmshModelOccAddWedge(const double x,
                                  const double y,
                                  const double z,
                                  const double dx,
                                  const double dy,
                                  const double dz,
                                  const int tag,
                                  const double ltx,
                                  int * ierr);

/* Add a torus, defined by its center (`x', `y', `z') and its 2 radii `r' and
 * `r2'. If `tag' is positive, set the tag explicitly; otherwise a new tag is
 * selected automatically. The optional argument `angle' defines the angular
 * opening (from 0 to 2*Pi). Return the tag of the wedge. */
GMSH_API int gmshModelOccAddTorus(const double x,
                                  const double y,
                                  const double z,
                                  const double r1,
                                  const double r2,
                                  const int tag,
                                  const double angle,
                                  int * ierr);

/* Add a volume (if the optional argument `makeSolid' is set) or surfaces
 * defined through the open or closed wires `wireTags'. If `tag' is positive,
 * set the tag explicitly; otherwise a new tag is selected automatically. The
 * new entities are returned in `outDimTags'. If the optional argument
 * `makeRuled' is set, the surfaces created on the boundary are forced to be
 * ruled surfaces. */
GMSH_API void gmshModelOccAddThruSections(int * wireTags, size_t wireTags_n,
                                          int ** outDimTags, size_t * outDimTags_n,
                                          const int tag,
                                          const int makeSolid,
                                          const int makeRuled,
                                          int * ierr);

/* Add a hollowed volume built from an initial volume `volumeTag' and a set of
 * faces from this volume `excludeSurfaceTags', which are to be removed. The
 * remaining faces of the volume become the walls of the hollowed solid, with
 * thickness `offset'. If `tag' is positive, set the tag explicitly; otherwise
 * a new tag is selected automatically. */
GMSH_API void gmshModelOccAddThickSolid(const int volumeTag,
                                        int * excludeSurfaceTags, size_t excludeSurfaceTags_n,
                                        const double offset,
                                        int ** outDimTags, size_t * outDimTags_n,
                                        const int tag,
                                        int * ierr);

/* Extrude the geometrical entities `dimTags' by translation along (`dx',
 * `dy', `dz'). Return extruded entities in `outDimTags'. If `numElements' is
 * not empty, also extrude the mesh: the entries in `numElements' give the
 * number of elements in each layer. If `height' is not empty, it provides the
 * (cummulative) height of the different layers, normalized to 1. */
GMSH_API void gmshModelOccExtrude(int * dimTags, size_t dimTags_n,
                                  const double dx,
                                  const double dy,
                                  const double dz,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  int * numElements, size_t numElements_n,
                                  double * heights, size_t heights_n,
                                  const int recombine,
                                  int * ierr);

/* Extrude the geometrical entities `dimTags' by rotation of `angle' radians
 * around the axis of revolution defined by the point (`x', `y', `z') and the
 * direction (`ax', `ay', `az'). Return extruded entities in `outDimTags'. If
 * `numElements' is not empty, also extrude the mesh: the entries in
 * `numElements' give the number of elements in each layer. If `height' is not
 * empty, it provides the (cummulative) height of the different layers,
 * normalized to 1. */
GMSH_API void gmshModelOccRevolve(int * dimTags, size_t dimTags_n,
                                  const double x,
                                  const double y,
                                  const double z,
                                  const double ax,
                                  const double ay,
                                  const double az,
                                  const double angle,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  int * numElements, size_t numElements_n,
                                  double * heights, size_t heights_n,
                                  const int recombine,
                                  int * ierr);

/* Add a pipe by extruding the entities `dimTags' along the wire `wireTag'.
 * Return the pipe in `outDimTags'. */
GMSH_API void gmshModelOccAddPipe(int * dimTags, size_t dimTags_n,
                                  const int wireTag,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  int * ierr);

/* Fillet the volumes `volumeTags' on the curves `curveTags' with radii
 * `radii'. The `radii' vector can either contain a single radius, as many
 * radii as `curveTags', or twice as many as `curveTags' (in which case
 * different radii are provided for the begin and end points of the curves).
 * Return the filleted entities in `outDimTags'. Remove the original volume if
 * `removeVolume' is set. */
GMSH_API void gmshModelOccFillet(int * volumeTags, size_t volumeTags_n,
                                 int * curveTags, size_t curveTags_n,
                                 double * radii, size_t radii_n,
                                 int ** outDimTags, size_t * outDimTags_n,
                                 const int removeVolume,
                                 int * ierr);

/* Chamfer the volumes `volumeTags' on the curves `curveTags' with distances
 * `distances' measured on surfaces `surfaceTags'. The `distances' vector can
 * either contain a single distance, as many distances as `curveTags' and
 * `surfaceTags', or twice as many as `curveTags' and `surfaceTags' (in which
 * case the first in each pair is measured on the corresponding surface in
 * `surfaceTags', the other on the other adjacent surface). Return the
 * chamfered entities in `outDimTags'. Remove the original volume if
 * `removeVolume' is set. */
GMSH_API void gmshModelOccChamfer(int * volumeTags, size_t volumeTags_n,
                                  int * curveTags, size_t curveTags_n,
                                  int * surfaceTags, size_t surfaceTags_n,
                                  double * distances, size_t distances_n,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  const int removeVolume,
                                  int * ierr);

/* Compute the boolean union (the fusion) of the entities `objectDimTags' and
 * `toolDimTags'. Return the resulting entities in `outDimTags'. If `tag' is
 * positive, try to set the tag explicitly (ony valid if the boolean operation
 * results in a single entity). Remove the object if `removeObject' is set.
 * Remove the tool if `removeTool' is set. */
GMSH_API void gmshModelOccFuse(int * objectDimTags, size_t objectDimTags_n,
                               int * toolDimTags, size_t toolDimTags_n,
                               int ** outDimTags, size_t * outDimTags_n,
                               int *** outDimTagsMap, size_t ** outDimTagsMap_n, size_t *outDimTagsMap_nn,
                               const int tag,
                               const int removeObject,
                               const int removeTool,
                               int * ierr);

/* Compute the boolean intersection (the common parts) of the entities
 * `objectDimTags' and `toolDimTags'. Return the resulting entities in
 * `outDimTags'. If `tag' is positive, try to set the tag explicitly (ony
 * valid if the boolean operation results in a single entity). Remove the
 * object if `removeObject' is set. Remove the tool if `removeTool' is set. */
GMSH_API void gmshModelOccIntersect(int * objectDimTags, size_t objectDimTags_n,
                                    int * toolDimTags, size_t toolDimTags_n,
                                    int ** outDimTags, size_t * outDimTags_n,
                                    int *** outDimTagsMap, size_t ** outDimTagsMap_n, size_t *outDimTagsMap_nn,
                                    const int tag,
                                    const int removeObject,
                                    const int removeTool,
                                    int * ierr);

/* Compute the boolean difference between the entities `objectDimTags' and
 * `toolDimTags'. Return the resulting entities in `outDimTags'. If `tag' is
 * positive, try to set the tag explicitly (ony valid if the boolean operation
 * results in a single entity). Remove the object if `removeObject' is set.
 * Remove the tool if `removeTool' is set. */
GMSH_API void gmshModelOccCut(int * objectDimTags, size_t objectDimTags_n,
                              int * toolDimTags, size_t toolDimTags_n,
                              int ** outDimTags, size_t * outDimTags_n,
                              int *** outDimTagsMap, size_t ** outDimTagsMap_n, size_t *outDimTagsMap_nn,
                              const int tag,
                              const int removeObject,
                              const int removeTool,
                              int * ierr);

/* Compute the boolean fragments (general fuse) of the entities
 * `objectDimTags' and `toolDimTags'. Return the resulting entities in
 * `outDimTags'. If `tag' is positive, try to set the tag explicitly (ony
 * valid if the boolean operation results in a single entity). Remove the
 * object if `removeObject' is set. Remove the tool if `removeTool' is set. */
GMSH_API void gmshModelOccFragment(int * objectDimTags, size_t objectDimTags_n,
                                   int * toolDimTags, size_t toolDimTags_n,
                                   int ** outDimTags, size_t * outDimTags_n,
                                   int *** outDimTagsMap, size_t ** outDimTagsMap_n, size_t *outDimTagsMap_nn,
                                   const int tag,
                                   const int removeObject,
                                   const int removeTool,
                                   int * ierr);

/* Translate the geometrical entities `dimTags' along (`dx', `dy', `dz'). */
GMSH_API void gmshModelOccTranslate(int * dimTags, size_t dimTags_n,
                                    const double dx,
                                    const double dy,
                                    const double dz,
                                    int * ierr);

/* Rotate the geometrical entities `dimTags' of `angle' radians around the
 * axis of revolution defined by the point (`x', `y', `z') and the direction
 * (`ax', `ay', `az'). */
GMSH_API void gmshModelOccRotate(int * dimTags, size_t dimTags_n,
                                 const double x,
                                 const double y,
                                 const double z,
                                 const double ax,
                                 const double ay,
                                 const double az,
                                 const double angle,
                                 int * ierr);

/* Scale the geometrical entities `dimTag' by factors `a', `b' and `c' along
 * the three coordinate axes; use (`x', `y', `z') as the center of the
 * homothetic transformation. */
GMSH_API void gmshModelOccDilate(int * dimTags, size_t dimTags_n,
                                 const double x,
                                 const double y,
                                 const double z,
                                 const double a,
                                 const double b,
                                 const double c,
                                 int * ierr);

/* Apply a symmetry transformation to the geometrical entities `dimTag', with
 * respect to the plane of equation `a' * x + `b' * y + `c' * z + `d' = 0. */
GMSH_API void gmshModelOccSymmetrize(int * dimTags, size_t dimTags_n,
                                     const double a,
                                     const double b,
                                     const double c,
                                     const double d,
                                     int * ierr);

/* Copy the entities `dimTags'; the new entities are returned in `outDimTags'. */
GMSH_API void gmshModelOccCopy(int * dimTags, size_t dimTags_n,
                               int ** outDimTags, size_t * outDimTags_n,
                               int * ierr);

/* Remove the entities `dimTags'. If `recursive' is true, remove all the
 * entities on their boundaries, down to dimension 0. */
GMSH_API void gmshModelOccRemove(int * dimTags, size_t dimTags_n,
                                 const int recursive,
                                 int * ierr);

/* Remove all duplicate entities (different entities at the same geometrical
 * location) after intersecting (using boolean fragments) all highest
 * dimensional entities. */
GMSH_API void gmshModelOccRemoveAllDuplicates(int * ierr);

/* Import BREP, STEP or IGES shapes from the file `fileName'. The imported
 * entities are returned in `outDimTags'. If the optional argument
 * `highestDimOnly' is set, only import the highest dimensional entities in
 * the file. The optional argument `format' can be used to force the format of
 * the file (currently "brep", "step" or "iges"). */
GMSH_API void gmshModelOccImportShapes(const char * fileName,
                                       int ** outDimTags, size_t * outDimTags_n,
                                       const int highestDimOnly,
                                       const char * format,
                                       int * ierr);

/* Imports an OpenCASCADE `shape' by providing a pointer to a native
 * OpenCASCADE `TopoDS_Shape' object (passed as a pointer to void). The
 * imported entities are returned in `outDimTags'. If the optional argument
 * `highestDimOnly' is set, only import the highest dimensional entities in
 * `shape'. Warning: this function is unsafe, as providing an invalid pointer
 * will lead to undefined behavior. */
GMSH_API void gmshModelOccImportShapesNativePointer(const void * shape,
                                                    int ** outDimTags, size_t * outDimTags_n,
                                                    const int highestDimOnly,
                                                    int * ierr);

/* Set a mesh size constraint on the geometrical entities `dimTags'. Currently
 * only entities of dimension 0 (points) are handled. */
GMSH_API void gmshModelOccSetMeshSize(int * dimTags, size_t dimTags_n,
                                      const double size,
                                      int * ierr);

/* Synchronize the internal OpenCASCADE CAD representation with the current
 * Gmsh model. This can be called at any time, but since it involves a non
 * trivial amount of processing, the number of synchronization points should
 * normally be minimized. */
GMSH_API void gmshModelOccSynchronize(int * ierr);

/* Add a new post-processing view, with name `name'. If `tag' is positive use
 * it (and remove the view with that tag if it already exists), otherwise
 * associate a new tag. Return the view tag. */
GMSH_API int gmshViewAdd(const char * name,
                         const int tag,
                         int * ierr);

/* Remove the view with tag `tag'. */
GMSH_API void gmshViewRemove(const int tag,
                             int * ierr);

/* Get the index of the view with tag `tag' in the list of currently loaded
 * views. This dynamic index (it can change when views are removed) is used to
 * access view options. */
GMSH_API int gmshViewGetIndex(const int tag,
                              int * ierr);

/* Get the tags of all views. */
GMSH_API void gmshViewGetTags(int ** tags, size_t * tags_n,
                              int * ierr);

/* Add model-based post-processing data to the view with tag `tag'.
 * `modelName' identifies the model the data is attached to. `dataType'
 * specifies the type of data, currently either "NodeData", "ElementData" or
 * "ElementNodeData". `step' specifies the identifier (>= 0) of the data in a
 * sequence. `tags' gives the tags of the nodes or elements in the mesh to
 * which the data is associated. `data' is a vector of the same length as
 * `tags': each entry is the vector of double precision numbers representing
 * the data associated with the corresponding tag. The optional `time'
 * argument associate a time value with the data. `numComponents' gives the
 * number of data components (1 for scalar data, 3 for vector data, etc.) per
 * entity; if negative, it is automatically inferred (when possible) from the
 * input data. `partition' allows to specify data in several sub-sets. */
GMSH_API void gmshViewAddModelData(const int tag,
                                   const int step,
                                   const char * modelName,
                                   const char * dataType,
                                   int * tags, size_t tags_n,
                                   const double ** data, const size_t * data_n, size_t data_nn,
                                   const double time,
                                   const int numComponents,
                                   const int partition,
                                   int * ierr);

/* Get model-based post-processing data from the view with tag `tag' at step
 * `step'. Return the `data' associated to the nodes or the elements with tags
 * `tags', as well as the `dataType' and the number of components
 * `numComponents'. */
GMSH_API void gmshViewGetModelData(const int tag,
                                   const int step,
                                   char ** dataType,
                                   int ** tags, size_t * tags_n,
                                   double *** data, size_t ** data_n, size_t *data_nn,
                                   double * time,
                                   int * numComponents,
                                   int * ierr);

/* Add list-based post-processing data to the view with tag `tag'. `dataType'
 * identifies the data: "SP" for scalar points, "VP", for vector points, etc.
 * `numEle' gives the number of elements in the data. `data' contains the data
 * for the `numEle' elements. */
GMSH_API void gmshViewAddListData(const int tag,
                                  const char * dataType,
                                  const int numEle,
                                  double * data, size_t data_n,
                                  int * ierr);

/* Get list-based post-processing data from the view with tag `tag'. Return
 * the types `dataTypes', the number of elements `numElements' for each data
 * type and the `data' for each data type. */
GMSH_API void gmshViewGetListData(const int tag,
                                  char *** dataType, size_t * dataType_n,
                                  int ** numElements, size_t * numElements_n,
                                  double *** data, size_t ** data_n, size_t *data_nn,
                                  int * ierr);

/* Probe the view `tag' for its `value' at point (`x', `y', `z'). Return only
 * the value at step `step' is `step' is positive. Return only values with
 * `numComp' if `numComp' is positive. Return the gradient of the `value' if
 * `gradient' is set. Probes with a geometrical tolerance (in the reference
 * unit cube) of `tolerance' if `tolerance' is not zero. Return the result
 * from the element described by its coordinates if `xElementCoord',
 * `yElementCoord' and `zElementCoord' are provided. */
GMSH_API void gmshViewProbe(const int tag,
                            const double x,
                            const double y,
                            const double z,
                            double ** value, size_t * value_n,
                            const int step,
                            const int numComp,
                            const int gradient,
                            const double tolerance,
                            double * xElemCoord, size_t xElemCoord_n,
                            double * yElemCoord, size_t yElemCoord_n,
                            double * zElemCoord, size_t zElemCoord_n,
                            int * ierr);

/* Write the view to a file `fileName'. The export format is determined by the
 * file extension. Append to the file if `append' is set. */
GMSH_API void gmshViewWrite(const int tag,
                            const char * fileName,
                            const int append,
                            int * ierr);

/* Set the numerical option `option' to the value `value' for plugin `name'. */
GMSH_API void gmshPluginSetNumber(const char * name,
                                  const char * option,
                                  const double value,
                                  int * ierr);

/* Set the string option `option' to the value `value' for plugin `name'. */
GMSH_API void gmshPluginSetString(const char * name,
                                  const char * option,
                                  const char * value,
                                  int * ierr);

/* Run the plugin `name'. */
GMSH_API void gmshPluginRun(const char * name,
                            int * ierr);

/* Draw all the OpenGL scenes. */
GMSH_API void gmshGraphicsDraw(int * ierr);

/* Create the Fltk graphical user interface. */
GMSH_API void gmshFltkInitialize(int * ierr);

/* Wait at most `time' seconds for user interface events and return. If `time'
 * < 0, wait indefinitely. First automatically create the user interface if it
 * has not yet been initialized. */
GMSH_API void gmshFltkWait(const double time,
                           int * ierr);

/* Run the event loop of the Fltk graphical user interface, i.e. repeatedly
 * calls `wait'. First automatically create the user interface if it has not
 * yet been initialized. */
GMSH_API void gmshFltkRun(int * ierr);

/* Get `data' from the ONELAB server. */
GMSH_API void gmshOnelabGet(char ** data,
                            const char * format,
                            int * ierr);

/* Set `data' in the ONELAB server. */
GMSH_API void gmshOnelabSet(const char * data,
                            const char * format,
                            int * ierr);

/* Run a ONELAB client. If `name' is provided, create a new ONELAB client with
 * name `name' and executes `command'. If not, try to run a client that might
 * be linked to the processed input files. */
GMSH_API void gmshOnelabRun(const char * name,
                            const char * command,
                            int * ierr);

/* Start logging messages in `log'. */
GMSH_API void gmshLoggerStart(char *** log, size_t * log_n,
                              int * ierr);

/* Stop logging messages. */
GMSH_API void gmshLoggerStop(int * ierr);

#endif
