/*
 * Decompiled with CFR 0.152.
 */
package ghidra.framework.store.db;

import db.DBHandle;
import db.Database;
import db.buffers.BufferFile;
import db.buffers.BufferFileManager;
import db.buffers.LocalManagedBufferFile;
import generic.jar.ResourceFile;
import ghidra.framework.store.db.PackedDBHandle;
import ghidra.framework.store.db.PackedDatabaseCache;
import ghidra.framework.store.local.ItemDeserializer;
import ghidra.framework.store.local.ItemSerializer;
import ghidra.framework.store.local.LocalFileSystem;
import ghidra.framework.store.local.LockFile;
import ghidra.util.Msg;
import ghidra.util.ReadOnlyException;
import ghidra.util.StringUtilities;
import ghidra.util.datastruct.WeakDataStructureFactory;
import ghidra.util.datastruct.WeakSet;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateFileException;
import ghidra.util.exception.FileInUseException;
import ghidra.util.exception.IOCancelledException;
import ghidra.util.task.TaskMonitor;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.Random;
import utilities.util.FileUtilities;

public class PackedDatabase
extends Database {
    public static final String READ_ONLY_DIRECTORY_LOCK_FILE = ".dbDirLock";
    private static final Random RANDOM = new Random();
    private static final String TEMPDB_PREFIX = "tmp";
    private static final String TEMPDB_EXT = ".pdb";
    private static final String TEMPDB_DIR_PREFIX = LocalFileSystem.HIDDEN_DIR_PREFIX + "tmp";
    private static final String TEMPDB_DIR_EXT = ".pdb.db";
    private static final String UPDATE_LOCK_TYPE = "u";
    static final int LOCK_TIMEOUT = 30000;
    private static final long ONE_WEEK_MS = 604800000L;
    private static WeakSet<PackedDatabase> pdbInstances;
    private ResourceFile packedDbFile;
    private boolean isCached;
    private String itemName;
    private String contentType;
    private LockFile packedDbLock;
    private LockFile updateLock;
    private PackedDBHandle dbHandle;
    private long dbTime;
    private boolean isReadOnly = false;

    private PackedDatabase(ResourceFile packedDbFile) throws IOException {
        super(PackedDatabase.createDBDir(), null, true);
        this.packedDbFile = packedDbFile;
        this.bfMgr = new PDBBufferFileManager();
        boolean success = false;
        try {
            this.isReadOnly = PackedDatabase.isReadOnlyPDBDirectory(packedDbFile.getParentFile());
            if (!this.isReadOnly) {
                this.updateLock = PackedDatabase.getUpdateLock(packedDbFile.getFile(false));
                this.packedDbLock = PackedDatabase.getFileLock(packedDbFile.getFile(false));
            }
            this.readContentTypeAndName();
            PackedDatabase.addInstance(this);
            success = true;
        }
        finally {
            if (!success) {
                this.dispose();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    PackedDatabase(ResourceFile packedDbFile, LockFile packedDbLock, PackedDatabaseCache.CachedDB cachedDb, TaskMonitor monitor) throws CancelledException, IOException {
        super(cachedDb.dbDir, null, false);
        this.packedDbFile = packedDbFile;
        this.contentType = cachedDb.contentType;
        this.itemName = cachedDb.itemName;
        this.dbTime = cachedDb.getLastModified();
        this.isCached = true;
        this.bfMgr = new PDBBufferFileManager();
        boolean success = false;
        try {
            this.packedDbLock = packedDbLock;
            if (packedDbLock != null) {
                this.updateLock = PackedDatabase.getUpdateLock(packedDbFile.getFile(false));
            } else {
                this.isReadOnly = true;
            }
            if (cachedDb.refreshRequired()) {
                this.refreshUnpacking(monitor);
            }
            PackedDatabase.addInstance(this);
            success = true;
        }
        finally {
            if (!success) {
                this.dispose();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    PackedDatabase(PackedDBHandle dbHandle, ResourceFile packedDbFile, String itemName, Long newDatabaseId, TaskMonitor monitor) throws CancelledException, IOException {
        super(PackedDatabase.createDBDir(), null, true);
        this.bfMgr = new PDBBufferFileManager();
        boolean success = false;
        try {
            this.dbHandle = dbHandle;
            this.packedDbFile = packedDbFile;
            this.itemName = itemName;
            this.contentType = dbHandle.getContentType();
            if (PackedDatabase.isReadOnlyPDBDirectory(packedDbFile.getParentFile())) {
                throw new ReadOnlyException("Read-only DB directory lock, file update not allowed: " + packedDbFile);
            }
            this.updateLock = PackedDatabase.getUpdateLock(packedDbFile.getFile(false));
            this.packedDbLock = PackedDatabase.getFileLock(packedDbFile.getFile(false));
            if (packedDbFile.exists() || !this.updateLock.createLock(0, true)) {
                throw new DuplicateFileException(packedDbFile + " already exists");
            }
            LocalManagedBufferFile bfile = new LocalManagedBufferFile(dbHandle.getBufferSize(), this.bfMgr, -1L);
            dbHandle.saveAs((BufferFile)bfile, newDatabaseId, true, monitor);
            this.packDatabase(monitor);
            PackedDatabase.addInstance(this);
            success = true;
        }
        finally {
            if (!success) {
                this.dispose();
            }
        }
    }

    public boolean isReadOnly() {
        return this.isReadOnly;
    }

    private static synchronized void addInstance(PackedDatabase pdb) {
        if (pdbInstances == null) {
            pdbInstances = WeakDataStructureFactory.createCopyOnReadWeakSet();
            Thread cleanupThread = new Thread("Packed Database Disposer"){

                @Override
                public void run() {
                    for (PackedDatabase pdbInstance : pdbInstances) {
                        try {
                            if (pdbInstance.dbHandle != null) {
                                pdbInstance.dbHandle.close();
                            }
                            pdbInstance.dispose();
                        }
                        catch (Throwable throwable) {}
                    }
                }
            };
            Runtime.getRuntime().addShutdownHook(cleanupThread);
        }
        pdbInstances.add((Object)pdb);
    }

    private static synchronized void removeInstance(PackedDatabase pdb) {
        if (pdbInstances != null) {
            pdbInstances.remove((Object)pdb);
        }
    }

    public static PackedDatabase getPackedDatabase(File packedDbFile, TaskMonitor monitor) throws IOException, CancelledException {
        return PackedDatabase.getPackedDatabase(packedDbFile, false, monitor);
    }

    public static PackedDatabase getPackedDatabase(File packedDbFile, boolean neverCache, TaskMonitor monitor) throws IOException, CancelledException {
        return PackedDatabase.getPackedDatabase(new ResourceFile(packedDbFile), neverCache, monitor);
    }

    public static PackedDatabase getPackedDatabase(ResourceFile packedDbFile, boolean neverCache, TaskMonitor monitor) throws IOException, CancelledException {
        if (!neverCache && PackedDatabaseCache.isEnabled()) {
            try {
                return PackedDatabaseCache.getCache().getCachedDB(packedDbFile, monitor);
            }
            catch (IOException e) {
                Msg.warn(PackedDatabase.class, (Object)("PackedDatabase cache failure for: " + packedDbFile + ", " + e.getMessage()));
            }
        }
        return new PackedDatabase(packedDbFile);
    }

    public static boolean isReadOnlyPDBDirectory(ResourceFile directory) {
        File dir = directory.getFile(false);
        if (dir == null) {
            return true;
        }
        File readOnlyLockFile = new File(dir, READ_ONLY_DIRECTORY_LOCK_FILE);
        if (!readOnlyLockFile.isFile()) {
            try {
                ResourceFile parentFile = directory.getParentFile();
                if (parentFile == null) {
                    return false;
                }
                return PackedDatabase.isReadOnlyPDBDirectory(parentFile);
            }
            catch (SecurityException securityException) {
                // empty catch block
            }
        }
        return true;
    }

    protected void finalize() throws Throwable {
        this.dispose();
    }

    public void dispose() {
        if (!this.isCached && this.dbDir != null && this.dbDir.exists()) {
            File tmpDbDir = new File(this.dbDir.getParentFile(), this.dbDir.getName() + ".delete");
            if (!this.dbDir.renameTo(tmpDbDir)) {
                Msg.error((Object)((Object)this), (Object)("Failed to dispose PackedDatabase - it may still be in use!\n" + this.packedDbFile), (Throwable)new Exception());
                return;
            }
            PackedDatabase.deleteDir((File)tmpDbDir);
        }
        if (this.dbHandle != null) {
            this.dbHandle = null;
            if (this.updateLock != null && this.updateLock.haveLock(true)) {
                this.updateLock.removeLock();
            }
        }
        if (this.packedDbLock != null && this.packedDbLock.haveLock(true)) {
            this.packedDbLock.removeLock();
        }
        PackedDatabase.removeInstance(this);
    }

    static String getRandomString() {
        int num = RANDOM.nextInt();
        return StringUtilities.pad((String)Integer.toHexString(num).toUpperCase(), (char)'0', (int)8);
    }

    private static File createDBDir() throws IOException {
        File tmpDir = new File(System.getProperty("java.io.tmpdir"));
        int tries = 0;
        while (tries++ < 10) {
            File dir = new File(tmpDir, TEMPDB_DIR_PREFIX + PackedDatabase.getRandomString() + TEMPDB_DIR_EXT);
            if (!dir.mkdir()) continue;
            return dir;
        }
        throw new IOException("Unable to create temporary database");
    }

    private static LockFile getUpdateLock(File packedFile) {
        return new LockFile(packedFile.getParentFile(), packedFile.getName(), UPDATE_LOCK_TYPE);
    }

    static LockFile getFileLock(File packedFile) {
        return new LockFile(packedFile.getParentFile(), packedFile.getName());
    }

    public String getContentType() {
        return this.contentType;
    }

    public ResourceFile getPackedFile() {
        return this.packedDbFile;
    }

    public void delete() throws IOException {
        if (this.isReadOnly) {
            throw new ReadOnlyException("Read-only DB directory lock, file removal not allowed: " + this.packedDbFile);
        }
        this.dispose();
        PackedDatabase.lock(this.updateLock, false, false);
        try {
            if (this.packedDbFile.exists() && !this.packedDbFile.delete()) {
                throw new IOException("File is in use or write protected");
            }
        }
        finally {
            this.updateLock.removeLock();
        }
    }

    public static void delete(File packedDbFile) throws IOException {
        LockFile updateLock = PackedDatabase.getUpdateLock(packedDbFile);
        PackedDatabase.lock(updateLock, false, false);
        try {
            if (packedDbFile.exists() && !packedDbFile.delete()) {
                throw new IOException("File is in use or write protected");
            }
        }
        finally {
            updateLock.removeLock();
        }
    }

    static void lock(LockFile lockFile, boolean wait, boolean hold) throws FileInUseException {
        if (!lockFile.createLock(wait ? 30000 : 0, hold)) {
            String msg = "File is in use - '" + lockFile + "'";
            String user = lockFile.getLockOwner();
            if (user != null) {
                msg = msg + " by " + user;
            }
            throw new FileInUseException(msg);
        }
    }

    private void readContentTypeAndName() throws IOException {
        ItemDeserializer itemDeserializer = null;
        if (this.packedDbLock != null) {
            PackedDatabase.lock(this.packedDbLock, true, true);
        }
        try {
            itemDeserializer = new ItemDeserializer(this.packedDbFile);
            if (itemDeserializer.getFileType() != 0) {
                throw new IOException("Incorrect file type");
            }
            this.contentType = itemDeserializer.getContentType();
            this.itemName = itemDeserializer.getItemName();
        }
        finally {
            if (itemDeserializer != null) {
                itemDeserializer.dispose();
            }
            if (this.packedDbLock != null) {
                this.packedDbLock.removeLock();
            }
        }
    }

    public static void unpackDatabase(BufferFileManager bfMgr, long checkinId, File packedFile, TaskMonitor monitor) throws IOException, CancelledException {
        if (bfMgr.getCurrentVersion() != 0) {
            throw new IllegalStateException("Expected empty database");
        }
        PackedDatabase.refreshDatabase(bfMgr, checkinId, new ResourceFile(packedFile), monitor);
    }

    private static void refreshDatabase(BufferFileManager bfMgr, long checkinId, ResourceFile packedFile, TaskMonitor monitor) throws IOException, CancelledException {
        if (monitor == null) {
            monitor = TaskMonitor.DUMMY;
        }
        int version = bfMgr.getCurrentVersion() + 1;
        File file = bfMgr.getBufferFile(version);
        BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file));
        ItemDeserializer itemDeserializer = null;
        try {
            Msg.debug(PackedDatabase.class, (Object)("Unpacking database " + packedFile + " -> " + file));
            itemDeserializer = new ItemDeserializer(packedFile);
            itemDeserializer.saveItem(out, monitor);
            bfMgr.versionCreated(version, "Unpacked " + packedFile, checkinId);
        }
        catch (IOCancelledException e) {
            throw new CancelledException();
        }
        finally {
            if (itemDeserializer != null) {
                itemDeserializer.dispose();
            }
            try {
                ((OutputStream)out).close();
            }
            catch (IOException iOException) {}
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean refreshUnpacking(TaskMonitor monitor) throws CancelledException, IOException {
        monitor.setMessage("Waiting...");
        if (!this.dbDir.isDirectory()) {
            throw new IOException("PackedDatabase has been disposed");
        }
        if (this.packedDbLock != null) {
            PackedDatabase.lock(this.packedDbLock, true, true);
        }
        try {
            PackedDatabaseCache.CachedDB entry;
            if (!this.packedDbFile.isFile()) {
                throw new FileNotFoundException("File not found: " + this.packedDbFile);
            }
            long modTime = this.packedDbFile.lastModified();
            if (this.isCached && (entry = PackedDatabaseCache.getCache().getCachedDBEntry(this.packedDbFile)) != null && entry.getLastModified() == modTime) {
                boolean bl = true;
                return bl;
            }
            if (this.dbTime == modTime) {
                boolean bl = true;
                return bl;
            }
            monitor.setMessage("Unpacking file...");
            PackedDatabase.refreshDatabase(this.bfMgr, -1L, this.packedDbFile, monitor);
            this.dbTime = modTime;
            if (this.isCached) {
                PackedDatabaseCache.getCache().updateLastModified(this.packedDbFile, modTime);
            }
        }
        finally {
            if (this.packedDbLock != null) {
                this.packedDbLock.removeLock();
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void packDatabase(DBHandle dbh, String itemName, String contentType, File outputFile, TaskMonitor monitor) throws CancelledException, IOException {
        DBHandle dBHandle = dbh;
        synchronized (dBHandle) {
            if (PackedDatabase.isReadOnlyPDBDirectory(new ResourceFile(outputFile.getParentFile()))) {
                throw new ReadOnlyException("Read-only DB directory lock, file creation not allowed: " + outputFile);
            }
            if (outputFile.exists()) {
                throw new DuplicateFileException(outputFile + " already exists");
            }
            boolean success = false;
            InputStream itemIn = null;
            File tmpFile = null;
            try {
                tmpFile = File.createTempFile("pack", ".tmp");
                tmpFile.delete();
                dbh.saveAs(tmpFile, false, monitor);
                itemIn = new BufferedInputStream(new FileInputStream(tmpFile));
                try {
                    ItemSerializer.outputItem(itemName, contentType, 0, tmpFile.length(), itemIn, outputFile, monitor);
                }
                finally {
                    try {
                        itemIn.close();
                    }
                    catch (IOException iOException) {}
                }
                success = true;
            }
            finally {
                if (itemIn != null) {
                    try {
                        itemIn.close();
                    }
                    catch (IOException iOException) {}
                }
                tmpFile.delete();
                if (!success) {
                    outputFile.delete();
                }
            }
        }
    }

    private static void packDatabase(String name, String contentType, File dbFile, File outputFile, TaskMonitor monitor) throws IOException, CancelledException {
        if (monitor == null) {
            monitor = TaskMonitor.DUMMY;
        }
        monitor.setMessage("Packing file...");
        FileInputStream itemIn = new FileInputStream(dbFile);
        try {
            ItemSerializer.outputItem(name, contentType, 0, dbFile.length(), itemIn, outputFile, monitor);
        }
        catch (IOCancelledException e) {
            throw new CancelledException();
        }
        finally {
            try {
                ((InputStream)itemIn).close();
            }
            catch (IOException iOException) {}
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void packDatabase(TaskMonitor monitor) throws CancelledException, IOException {
        if (this.isReadOnly || this.dbHandle == null || this.bfMgr == null || this.bfMgr.getCurrentVersion() == 0 || !this.updateLock.haveLock()) {
            throw new IOException("Update not allowed");
        }
        if (monitor == null) {
            monitor = TaskMonitor.DUMMY;
        }
        monitor.setMessage("Waiting...");
        if (this.packedDbLock != null) {
            PackedDatabase.lock(this.packedDbLock, true, true);
        }
        try {
            File packedFile = this.packedDbFile.getFile(false);
            File dbFile = this.bfMgr.getBufferFile(this.bfMgr.getCurrentVersion());
            File parentFile = packedFile.getAbsoluteFile().getParentFile();
            File tmpFile = File.createTempFile(TEMPDB_PREFIX, TEMPDB_EXT, parentFile);
            Msg.debug(PackedDatabase.class, (Object)("Packing database " + dbFile + " -> " + packedFile));
            PackedDatabase.packDatabase(this.itemName, this.contentType, dbFile, tmpFile, monitor);
            File bakFile = new File(parentFile, packedFile.getName() + ".bak");
            bakFile.delete();
            long oldTime = packedFile.lastModified();
            packedFile.renameTo(bakFile);
            if (!tmpFile.renameTo(packedFile)) {
                bakFile.renameTo(packedFile);
                throw new IOException("Update failed for " + packedFile);
            }
            bakFile.delete();
            this.dbTime = packedFile.lastModified();
            if (oldTime == this.dbTime) {
                this.dbTime += 1000L;
                packedFile.setLastModified(this.dbTime);
            }
            if (this.isCached) {
                try {
                    PackedDatabaseCache.getCache().updateLastModified(this.packedDbFile, this.dbTime);
                }
                catch (IOException e) {
                    Msg.warn((Object)((Object)this), (Object)("cache update failed: " + e.getMessage()));
                }
            }
        }
        finally {
            if (this.packedDbLock != null) {
                this.packedDbLock.removeLock();
            }
        }
    }

    public synchronized DBHandle open(TaskMonitor monitor) throws CancelledException, IOException {
        if (this.dbHandle != null) {
            throw new IOException("Database is already open");
        }
        if (monitor == null) {
            monitor = TaskMonitor.DUMMY;
        }
        if (!this.refreshUnpacking(monitor)) {
            throw new IOException("Failed to unpack/refresh database - it may be in use");
        }
        LocalManagedBufferFile bfile = new LocalManagedBufferFile(this.bfMgr, false, -1, -1L);
        this.dbHandle = new PackedDBHandle(this, (BufferFile)bfile);
        return this.dbHandle;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized DBHandle openForUpdate(TaskMonitor monitor) throws CancelledException, IOException {
        PackedDBHandle dbh;
        if (this.dbHandle != null) {
            throw new IOException("Database is already open");
        }
        if (this.isReadOnly) {
            throw new ReadOnlyException("Read-only DB directory lock, file update not allowed: " + this.packedDbFile);
        }
        if (monitor == null) {
            monitor = TaskMonitor.DUMMY;
        }
        PackedDatabase.lock(this.updateLock, false, true);
        boolean success = false;
        try {
            if (!this.refreshUnpacking(monitor)) {
                throw new IOException("Failed to unpack/refresh database - it may be in use");
            }
            LocalManagedBufferFile bfile = new LocalManagedBufferFile(this.bfMgr, true, -1, -1L);
            this.dbHandle = dbh = new PackedDBHandle(this, (BufferFile)bfile);
            success = true;
        }
        finally {
            if (!success) {
                this.updateLock.removeLock();
            }
        }
        return dbh;
    }

    public static void cleanupOldTempDatabases() {
        File tmpDir = new File(System.getProperty("java.io.tmpdir"));
        File[] tempDbs = tmpDir.listFiles(file -> {
            String name = file.getName();
            return file.isDirectory() && name.indexOf(TEMPDB_DIR_PREFIX) == 0 && name.endsWith(TEMPDB_DIR_EXT);
        });
        if (tempDbs == null) {
            return;
        }
        long lastWeek = new Date().getTime() - 604800000L;
        for (File tempDb : tempDbs) {
            try {
                if (!tempDb.isDirectory() || tempDb.lastModified() > lastWeek || !FileUtilities.deleteDir((File)tempDb)) continue;
                Msg.info(PackedDatabase.class, (Object)("Removed temporary database: " + tempDb));
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    private class PDBBufferFileManager
    extends Database.DBBufferFileManager {
        private PDBBufferFileManager() {
            super((Database)PackedDatabase.this);
        }

        public void updateEnded(long checkinId) {
            PackedDatabase.this.dbHandle = null;
            if (PackedDatabase.this.updateLock != null && PackedDatabase.this.updateLock.haveLock(true)) {
                PackedDatabase.this.updateLock.removeLock();
            }
            super.updateEnded(checkinId);
        }
    }
}

