/*
 * Decompiled with CFR 0.152.
 */
package db;

import db.ChainedBuffer;
import db.DBBuffer;
import db.DBChangeSet;
import db.DBListener;
import db.DBParms;
import db.DBRollbackException;
import db.IndexTable;
import db.MasterTable;
import db.NoTransactionException;
import db.Schema;
import db.Table;
import db.TableRecord;
import db.TerminatedTransactionException;
import db.Transaction;
import db.buffers.BufferFile;
import db.buffers.BufferMgr;
import db.buffers.LocalBufferFile;
import db.util.ErrorHandler;
import ghidra.util.Msg;
import ghidra.util.UniversalIdGenerator;
import ghidra.util.datastruct.WeakDataStructureFactory;
import ghidra.util.datastruct.WeakSet;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.ClosedException;
import ghidra.util.exception.DuplicateFileException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.task.TaskMonitor;
import java.io.File;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Iterator;

public class DBHandle {
    protected BufferMgr bufferMgr;
    private DBParms dbParms;
    private MasterTable masterTable;
    private Hashtable<String, Table> tables;
    private long databaseId;
    private DBHandle scratchPad;
    private WeakSet<DBListener> listenerList = WeakDataStructureFactory.createCopyOnReadWeakSet();
    private long lastTransactionID;
    private boolean txStarted = false;
    private boolean waitingForNewTransaction = false;
    private boolean reloadInProgress = false;
    private long checkpointNum;
    private long lastRecoverySnapshotId;

    public DBHandle() throws IOException {
        this(16384, 0x400000L);
    }

    public DBHandle(int requestedBufferSize) throws IOException {
        this(requestedBufferSize, 0x400000L);
    }

    public DBHandle(int requestedBufferSize, long approxCacheSize) throws IOException {
        this.bufferMgr = new BufferMgr(requestedBufferSize, approxCacheSize, 10);
        this.dbParms = new DBParms(this.bufferMgr, true);
        this.dbParms.set(DBParms.MASTER_TABLE_ROOT_BUFFER_ID_PARM, -1);
        this.masterTable = new MasterTable(this);
        this.initDatabaseId();
        this.bufferMgr.clearCheckpoints();
        this.tables = new Hashtable();
    }

    public DBHandle(BufferFile bufferFile) throws IOException {
        this.bufferMgr = new BufferMgr(bufferFile);
        this.dbParms = new DBParms(this.bufferMgr, false);
        this.readDatabaseId();
        if (this.databaseId == 0L && this.bufferMgr.canSave()) {
            this.initDatabaseId();
            this.bufferMgr.clearCheckpoints();
        }
        this.masterTable = new MasterTable(this);
        this.loadTables();
    }

    public DBHandle(BufferFile bufferFile, boolean recover, TaskMonitor monitor) throws IOException, CancelledException {
        this.bufferMgr = new BufferMgr(bufferFile);
        if (this.bufferMgr.canSave()) {
            if (recover) {
                this.bufferMgr.recover(monitor);
            } else {
                this.bufferMgr.clearRecoveryFiles();
            }
        }
        this.dbParms = new DBParms(this.bufferMgr, false);
        this.readDatabaseId();
        if (this.databaseId == 0L && this.bufferMgr.canSave()) {
            this.initDatabaseId();
        }
        this.bufferMgr.clearCheckpoints();
        this.masterTable = new MasterTable(this);
        this.loadTables();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DBHandle(File file) throws IOException {
        LocalBufferFile bfile = new LocalBufferFile(file, true);
        boolean success = false;
        try {
            this.bufferMgr = new BufferMgr(bfile);
            this.dbParms = new DBParms(this.bufferMgr, false);
            this.readDatabaseId();
            this.masterTable = new MasterTable(this);
            this.loadTables();
            success = true;
        }
        finally {
            if (!success) {
                bfile.close();
            }
        }
    }

    public boolean isConsistent(TaskMonitor monitor) throws CancelledException {
        int consistentCount = 0;
        for (Table table : this.getTables()) {
            try {
                if (!table.isConsistent(monitor)) continue;
                ++consistentCount;
            }
            catch (IOException e) {
                Msg.error((Object)this, (Object)("Consistency check error while processing table: " + table.getName()), (Throwable)e);
            }
        }
        return consistentCount == this.tables.size();
    }

    public boolean rebuild(TaskMonitor monitor) throws CancelledException {
        for (Table table : this.getTables()) {
            try {
                table.rebuild(monitor);
            }
            catch (IOException e) {
                Msg.error((Object)this, (Object)("Rebuild failed while processing table: " + table.getName()), (Throwable)e);
                return false;
            }
        }
        return true;
    }

    public static void resetDatabaseId(File file) throws IOException {
        long databaseId = UniversalIdGenerator.nextID().getValue();
        DBParms.poke(file, DBParms.DATABASE_ID_HIGH_PARM, (int)(databaseId >> 32));
        DBParms.poke(file, DBParms.DATABASE_ID_LOW_PARM, (int)databaseId);
    }

    private void setDatabaseId(long id) throws IOException {
        this.databaseId = id;
        this.dbParms.set(DBParms.DATABASE_ID_HIGH_PARM, (int)(this.databaseId >> 32));
        this.dbParms.set(DBParms.DATABASE_ID_LOW_PARM, (int)this.databaseId);
    }

    private void initDatabaseId() throws IOException {
        this.setDatabaseId(UniversalIdGenerator.nextID().getValue());
    }

    private void readDatabaseId() throws IOException {
        try {
            this.databaseId = ((long)this.dbParms.get(DBParms.DATABASE_ID_HIGH_PARM) << 32) + ((long)this.dbParms.get(DBParms.DATABASE_ID_LOW_PARM) & 0xFFFFFFFFL);
        }
        catch (IndexOutOfBoundsException indexOutOfBoundsException) {
            // empty catch block
        }
    }

    public long getDatabaseId() {
        return this.databaseId;
    }

    public LocalBufferFile getRecoveryChangeSetFile() throws IOException {
        return this.bufferMgr.getRecoveryChangeSetFile();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean takeRecoverySnapshot(DBChangeSet changeSet, TaskMonitor monitor) throws CancelledException, IOException {
        long cpNum;
        DBHandle dBHandle = this;
        synchronized (dBHandle) {
            if (!this.bufferMgr.modifiedSinceSnapshot()) {
                return true;
            }
            if (this.txStarted) {
                return false;
            }
            if (this.lastRecoverySnapshotId == this.checkpointNum) {
                return true;
            }
            cpNum = this.checkpointNum;
        }
        if (!this.bufferMgr.takeRecoverySnapshot(changeSet, monitor)) {
            return false;
        }
        dBHandle = this;
        synchronized (dBHandle) {
            this.lastRecoverySnapshotId = cpNum;
        }
        return true;
    }

    public DBHandle getScratchPad() throws IOException {
        if (this.scratchPad == null) {
            this.scratchPad = new DBHandle();
            this.scratchPad.startTransaction();
        }
        return this.scratchPad;
    }

    public void closeScratchPad() {
        DBHandle scratchDbh = this.scratchPad;
        if (scratchDbh != null) {
            scratchDbh.close();
            this.scratchPad = null;
        }
    }

    public void addListener(DBListener listener) {
        this.listenerList.add((Object)listener);
    }

    private void notifyDbRestored() {
        for (DBListener listener : this.listenerList) {
            listener.dbRestored(this);
        }
    }

    private void notifyDbClosed() {
        for (DBListener listener : this.listenerList) {
            listener.dbClosed(this);
        }
    }

    private void notifyTableAdded(Table table) {
        if (this.reloadInProgress) {
            return;
        }
        for (DBListener listener : this.listenerList) {
            listener.tableAdded(this, table);
        }
    }

    void notifyTableDeleted(Table table) {
        if (this.reloadInProgress) {
            return;
        }
        for (DBListener listener : this.listenerList) {
            listener.tableDeleted(this, table);
        }
    }

    MasterTable getMasterTable() {
        return this.masterTable;
    }

    BufferMgr getBufferMgr() {
        return this.bufferMgr;
    }

    public void enablePreCache() {
        this.bufferMgr.enablePreCache();
    }

    DBParms getDBParms() {
        return this.dbParms;
    }

    public void checkTransaction() {
        if (!this.txStarted) {
            if (this.waitingForNewTransaction) {
                throw new TerminatedTransactionException();
            }
            throw new NoTransactionException();
        }
    }

    public void checkIsClosed() throws ClosedException {
        if (this.isClosed()) {
            throw new ClosedException();
        }
    }

    public boolean isTransactionActive() {
        return this.txStarted;
    }

    public Transaction openTransaction(final ErrorHandler errorHandler) throws IllegalStateException {
        return new Transaction(){
            long txId;
            {
                this.txId = DBHandle.this.startTransaction();
            }

            @Override
            protected boolean endTransaction(boolean commit) {
                try {
                    return DBHandle.this.endTransaction(this.txId, commit);
                }
                catch (IOException e) {
                    errorHandler.dbError(e);
                    return false;
                }
            }
        };
    }

    public synchronized long startTransaction() {
        if (this.isClosed()) {
            throw new IllegalStateException("Database is closed");
        }
        if (this.txStarted) {
            throw new IllegalStateException("Transaction already started");
        }
        this.waitingForNewTransaction = false;
        this.txStarted = true;
        return ++this.lastTransactionID;
    }

    public boolean endTransaction(long id, boolean commit) throws IOException {
        try {
            return this.doEndTransaction(id, commit);
        }
        catch (DBRollbackException e) {
            this.notifyDbRestored();
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized boolean doEndTransaction(long id, boolean commit) throws DBRollbackException, IOException {
        if (id != this.lastTransactionID) {
            throw new IllegalStateException("Transaction id is not active");
        }
        try {
            if (this.bufferMgr != null && !this.bufferMgr.atCheckpoint()) {
                if (commit) {
                    this.masterTable.flush();
                    if (this.bufferMgr.checkpoint()) {
                        ++this.checkpointNum;
                        boolean bl = true;
                        return bl;
                    }
                    boolean bl = false;
                    return bl;
                }
                this.bufferMgr.undo(false);
                this.reloadTables();
                throw new DBRollbackException();
            }
        }
        finally {
            this.txStarted = false;
        }
        return false;
    }

    public synchronized boolean hasUncommittedChanges() {
        return this.bufferMgr != null && !this.bufferMgr.atCheckpoint();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void terminateTransaction(long id, boolean commit) throws IOException {
        boolean rollback = false;
        DBHandle dBHandle = this;
        synchronized (dBHandle) {
            try {
                this.doEndTransaction(id, commit);
            }
            catch (DBRollbackException e) {
                rollback = true;
            }
            this.waitingForNewTransaction = true;
        }
        if (rollback) {
            this.notifyDbRestored();
        }
    }

    public boolean canUndo() {
        return !this.txStarted && this.bufferMgr != null && this.bufferMgr.hasUndoCheckpoints();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean undo() throws IOException {
        boolean success = false;
        DBHandle dBHandle = this;
        synchronized (dBHandle) {
            if (this.canUndo() && this.bufferMgr.undo(true)) {
                ++this.checkpointNum;
                this.reloadTables();
                success = true;
            }
        }
        if (success) {
            this.notifyDbRestored();
        }
        return success;
    }

    public int getAvailableUndoCount() {
        return this.bufferMgr != null ? this.bufferMgr.getAvailableUndoCount() : 0;
    }

    public int getAvailableRedoCount() {
        return this.bufferMgr != null ? this.bufferMgr.getAvailableRedoCount() : 0;
    }

    public boolean canRedo() {
        return !this.txStarted && this.bufferMgr != null && this.bufferMgr.hasRedoCheckpoints();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean redo() throws IOException {
        boolean success = false;
        DBHandle dBHandle = this;
        synchronized (dBHandle) {
            if (this.canRedo() && this.bufferMgr.redo()) {
                ++this.checkpointNum;
                this.reloadTables();
                success = true;
            }
        }
        if (success) {
            this.notifyDbRestored();
        }
        return success;
    }

    public synchronized void setMaxUndos(int maxUndos) {
        this.bufferMgr.setMaxUndos(maxUndos);
    }

    public int getTableCount() {
        return this.tables.size();
    }

    public void close() {
        this.close(false);
    }

    public void close(boolean keepRecoveryData) {
        this.closeScratchPad();
        BufferMgr mgr = this.bufferMgr;
        if (mgr != null) {
            this.notifyDbClosed();
            mgr.dispose(keepRecoveryData);
            this.bufferMgr = null;
        }
    }

    public synchronized boolean isChanged() {
        return this.bufferMgr != null && this.bufferMgr.isChanged();
    }

    public boolean isClosed() {
        return this.bufferMgr == null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void save(String comment, DBChangeSet changeSet, TaskMonitor monitor) throws IOException, CancelledException {
        if (this.txStarted) {
            throw new AssertException("Can't save during transaction");
        }
        long txId = this.startTransaction();
        try {
            this.masterTable.flush();
        }
        finally {
            this.endTransaction(txId, true);
        }
        this.bufferMgr.save(comment, changeSet, monitor);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void saveAs(BufferFile outFile, boolean associateWithNewFile, TaskMonitor monitor) throws IOException, CancelledException {
        if (this.txStarted) {
            throw new AssertException("Can't save during transaction");
        }
        long txId = this.startTransaction();
        boolean addedTx = false;
        try {
            if (this.bufferMgr.getSourceFile() != null) {
                this.initDatabaseId();
            }
            this.masterTable.flush();
        }
        finally {
            addedTx = this.endTransaction(txId, true);
        }
        this.bufferMgr.saveAs(outFile, associateWithNewFile, monitor);
        if (addedTx && !associateWithNewFile) {
            this.undo();
            this.readDatabaseId();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected synchronized void saveAs(BufferFile outFile, Long newDatabaseId, boolean associateWithNewFile, TaskMonitor monitor) throws IOException, CancelledException {
        if (this.txStarted) {
            throw new IllegalStateException("Can't save during transaction");
        }
        long txId = this.startTransaction();
        try {
            if (newDatabaseId == null) {
                this.initDatabaseId();
            } else if (this.databaseId != newDatabaseId) {
                this.setDatabaseId(newDatabaseId);
            }
            this.masterTable.flush();
        }
        finally {
            this.endTransaction(txId, true);
        }
        this.bufferMgr.saveAs(outFile, associateWithNewFile, monitor);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void saveAs(File file, boolean associateWithNewFile, TaskMonitor monitor) throws IOException, CancelledException {
        this.checkIsClosed();
        if (file.exists()) {
            throw new DuplicateFileException("File already exists: " + file);
        }
        LocalBufferFile outFile = new LocalBufferFile(file, this.bufferMgr.getBufferSize());
        boolean success = false;
        try {
            this.saveAs(outFile, associateWithNewFile, monitor);
            success = true;
        }
        finally {
            if (!success) {
                outFile.delete();
            } else if (!associateWithNewFile) {
                outFile.dispose();
            }
        }
    }

    public DBBuffer createBuffer(int length) throws IOException {
        this.checkTransaction();
        return new DBBuffer(this, new ChainedBuffer(length, true, this.bufferMgr));
    }

    public DBBuffer createBuffer(DBBuffer shadowBuffer) throws IOException {
        this.checkTransaction();
        return new DBBuffer(this, new ChainedBuffer(shadowBuffer.length(), true, shadowBuffer.buf, 0, this.bufferMgr));
    }

    public DBBuffer getBuffer(int id) throws IOException {
        this.checkIsClosed();
        return new DBBuffer(this, new ChainedBuffer(this.bufferMgr, id));
    }

    public DBBuffer getBuffer(int id, DBBuffer shadowBuffer) throws IOException {
        this.checkIsClosed();
        return new DBBuffer(this, new ChainedBuffer(this.bufferMgr, id, shadowBuffer.buf, 0));
    }

    public boolean canUpdate() {
        try {
            return this.bufferMgr != null && this.bufferMgr.canSave();
        }
        catch (IOException e) {
            return false;
        }
    }

    private void loadTables() throws IOException {
        TableRecord[] tableRecords;
        this.tables = new Hashtable();
        for (TableRecord tableRecord : tableRecords = this.masterTable.getTableRecords()) {
            if (tableRecord.getIndexedColumn() < 0) {
                Table table = new Table(this, tableRecord);
                this.tables.put(table.getName(), table);
                continue;
            }
            IndexTable.getIndexTable(this, tableRecord);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void reloadTables() throws IOException {
        this.reloadInProgress = true;
        try {
            TableRecord[] tableRecords;
            this.dbParms.refresh();
            Hashtable<String, Table> oldTables = this.tables;
            this.tables = new Hashtable();
            for (TableRecord tableRecord : tableRecords = this.masterTable.refreshTableRecords()) {
                String tableName = tableRecord.getName();
                if (tableRecord.getIndexedColumn() < 0) {
                    Table t = oldTables.get(tableName);
                    if (t == null || t.isInvalid()) {
                        oldTables.remove(tableName);
                        t = new Table(this, tableRecord);
                        this.notifyTableAdded(t);
                    }
                    this.tables.put(tableName, t);
                    continue;
                }
                if (oldTables.containsKey(tableName)) continue;
                IndexTable.getIndexTable(this, tableRecord);
            }
        }
        finally {
            this.reloadInProgress = false;
        }
    }

    public Table getTable(String name) {
        return this.tables.get(name);
    }

    public Table[] getTables() {
        Table[] t = new Table[this.tables.size()];
        Iterator<Table> it = this.tables.values().iterator();
        int i = 0;
        while (it.hasNext()) {
            t[i++] = it.next();
        }
        return t;
    }

    public Table createTable(String name, Schema schema) throws IOException {
        return this.createTable(name, schema, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Table createTable(String name, Schema schema, int[] indexedColumns) throws IOException {
        Table table;
        DBHandle dBHandle = this;
        synchronized (dBHandle) {
            if (this.tables.containsKey(name)) {
                throw new IOException("Table already exists: " + name);
            }
            this.checkTransaction();
            table = new Table(this, this.masterTable.createTableRecord(name, schema, -1));
            this.tables.put(name, table);
            if (indexedColumns != null) {
                for (int indexedColumn : indexedColumns) {
                    IndexTable.createIndexTable(table, indexedColumn);
                }
            }
        }
        this.notifyTableAdded(table);
        return table;
    }

    public synchronized boolean setTableName(String oldName, String newName) throws DuplicateNameException {
        if (!this.tables.containsKey(oldName)) {
            return false;
        }
        this.checkTransaction();
        if (this.tables.containsKey(newName)) {
            throw new DuplicateNameException("Table already exists: " + newName);
        }
        Table table = this.tables.remove(oldName);
        if (table == null) {
            return false;
        }
        this.masterTable.changeTableName(oldName, newName);
        this.tables.put(newName, table);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteTable(String name) throws IOException {
        Table table;
        DBHandle dBHandle = this;
        synchronized (dBHandle) {
            int[] indexedColumns;
            table = this.tables.get(name);
            if (table == null) {
                return;
            }
            this.checkTransaction();
            for (int indexedColumn : indexedColumns = table.getIndexedColumns()) {
                table.removeIndex(indexedColumn);
            }
            table.deleteAll();
            this.masterTable.deleteTableRecord(table.getTableNum());
            this.tables.remove(name);
        }
        this.notifyTableDeleted(table);
    }

    public long getCacheHits() {
        if (this.bufferMgr == null) {
            throw new IllegalStateException("Database is closed");
        }
        return this.bufferMgr.getCacheHits();
    }

    public long getCacheMisses() {
        if (this.bufferMgr == null) {
            throw new IllegalStateException("Database is closed");
        }
        return this.bufferMgr.getCacheMisses();
    }

    public int getLowBufferCount() {
        if (this.bufferMgr == null) {
            throw new IllegalStateException("Database is closed");
        }
        return this.bufferMgr.getLowBufferCount();
    }

    protected void finalize() throws Throwable {
        this.close(true);
    }

    public int getBufferSize() {
        if (this.bufferMgr == null) {
            throw new IllegalStateException("Database is closed");
        }
        return this.bufferMgr.getBufferSize();
    }
}

