/**
 * @license
 * Copyright 2026 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { vi, describe, it, expect, beforeEach, afterEach } from 'vitest';
import { format } from 'node:util';
import { handleEnable, enableCommand } from './enable.js';
import { loadSettings, SettingScope, } from '../../config/settings.js';
const emitConsoleLog = vi.hoisted(() => vi.fn());
const debugLogger = vi.hoisted(() => ({
    log: vi.fn((message, ...args) => {
        emitConsoleLog('log', format(message, ...args));
    }),
}));
vi.mock('@google/gemini-cli-core', async (importOriginal) => {
    const actual = await importOriginal();
    return {
        ...actual,
        debugLogger,
    };
});
vi.mock('../../config/settings.js', async (importOriginal) => {
    const actual = await importOriginal();
    return {
        ...actual,
        loadSettings: vi.fn(),
    };
});
vi.mock('../utils.js', () => ({
    exitCli: vi.fn(),
}));
describe('skills enable command', () => {
    const mockLoadSettings = vi.mocked(loadSettings);
    beforeEach(() => {
        vi.clearAllMocks();
    });
    afterEach(() => {
        vi.restoreAllMocks();
    });
    describe('handleEnable', () => {
        it('should enable a disabled skill in user scope', async () => {
            const mockSettings = {
                forScope: vi.fn().mockReturnValue({
                    settings: { skills: { disabled: ['skill1'] } },
                }),
                setValue: vi.fn(),
            };
            mockLoadSettings.mockReturnValue(mockSettings);
            await handleEnable({
                name: 'skill1',
                scope: SettingScope.User,
            });
            expect(mockSettings.setValue).toHaveBeenCalledWith(SettingScope.User, 'skills.disabled', []);
            expect(emitConsoleLog).toHaveBeenCalledWith('log', 'Skill "skill1" successfully enabled in scope "User".');
        });
        it('should log a message if the skill is already enabled', async () => {
            const mockSettings = {
                forScope: vi.fn().mockReturnValue({
                    settings: { skills: { disabled: [] } },
                }),
                setValue: vi.fn(),
            };
            mockLoadSettings.mockReturnValue(mockSettings);
            await handleEnable({
                name: 'skill1',
                scope: SettingScope.User,
            });
            expect(mockSettings.setValue).not.toHaveBeenCalled();
            expect(emitConsoleLog).toHaveBeenCalledWith('log', 'Skill "skill1" is already enabled in scope "User".');
        });
    });
    describe('enableCommand', () => {
        it('should have correct command and describe', () => {
            expect(enableCommand.command).toBe('enable <name>');
            expect(enableCommand.describe).toBe('Enables an agent skill.');
        });
    });
});
//# sourceMappingURL=enable.test.js.map